"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
// eslint-disable-next-line import/no-extraneous-dependencies
const aws_sdk_1 = require("aws-sdk");
async function handler(event) {
    const resourceProps = event.ResourceProperties;
    switch (event.RequestType) {
        case 'Create':
        case 'Update':
            return cfnEventHandler(resourceProps, false);
        case 'Delete':
            return cfnEventHandler(resourceProps, true);
    }
}
exports.handler = handler;
async function cfnEventHandler(props, isDeleteEvent) {
    const { AssumeRoleArn, ParentZoneId, ParentZoneName, DelegatedZoneName, DelegatedZoneNameServers, TTL } = props;
    if (!ParentZoneId && !ParentZoneName) {
        throw Error('One of ParentZoneId or ParentZoneName must be specified');
    }
    const credentials = await getCrossAccountCredentials(AssumeRoleArn);
    const route53 = new aws_sdk_1.Route53({ credentials });
    const parentZoneId = ParentZoneId !== null && ParentZoneId !== void 0 ? ParentZoneId : await getHostedZoneIdByName(ParentZoneName, route53);
    await route53.changeResourceRecordSets({
        HostedZoneId: parentZoneId,
        ChangeBatch: {
            Changes: [{
                    Action: isDeleteEvent ? 'DELETE' : 'UPSERT',
                    ResourceRecordSet: {
                        Name: DelegatedZoneName,
                        Type: 'NS',
                        TTL,
                        ResourceRecords: DelegatedZoneNameServers.map(ns => ({ Value: ns })),
                    },
                }],
        },
    }).promise();
}
async function getCrossAccountCredentials(roleArn) {
    const sts = new aws_sdk_1.STS();
    const timestamp = (new Date()).getTime();
    const { Credentials: assumedCredentials } = await sts
        .assumeRole({
        RoleArn: roleArn,
        RoleSessionName: `cross-account-zone-delegation-${timestamp}`,
    })
        .promise();
    if (!assumedCredentials) {
        throw Error('Error getting assume role credentials');
    }
    return new aws_sdk_1.Credentials({
        accessKeyId: assumedCredentials.AccessKeyId,
        secretAccessKey: assumedCredentials.SecretAccessKey,
        sessionToken: assumedCredentials.SessionToken,
    });
}
async function getHostedZoneIdByName(name, route53) {
    const zones = await route53.listHostedZonesByName({ DNSName: name }).promise();
    const matchedZones = zones.HostedZones.filter(zone => zone.Name === `${name}.`);
    if (matchedZones.length !== 1) {
        throw Error(`Expected one hosted zone to match the given name but found ${matchedZones.length}`);
    }
    return matchedZones[0].Id;
}
//# sourceMappingURL=data:application/json;base64,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