"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.renderUnless = exports.helperRemovalPolicy = exports.renderCredentials = exports.defaultDeletionProtection = exports.engineDescription = exports.setupS3ImportExport = exports.DEFAULT_PASSWORD_EXCLUDE_CHARS = void 0;
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const database_secret_1 = require("../database-secret");
const props_1 = require("../props");
/**
 * The default set of characters we exclude from generated passwords for database users.
 * It's a combination of characters that have a tendency to cause problems in shell scripts,
 * some engine-specific characters (for example, Oracle doesn't like '@' in its passwords),
 * and some that trip up other services, like DMS.
 *
 * This constant is private to the RDS module.
 */
exports.DEFAULT_PASSWORD_EXCLUDE_CHARS = " %+~`#$&*()|[]{}:;<>?!'/@\"\\";
/**
 * Validates the S3 import/export props and returns the import/export roles, if any.
 * If `combineRoles` is true, will reuse the import role for export (or vice versa) if possible.
 *
 * Notably, `combineRoles` is (by default) set to true for instances, but false for clusters.
 * This is because the `combineRoles` functionality is most applicable to instances and didn't exist
 * for the initial clusters implementation. To maintain backwards compatibility, it is set to false for clusters.
 */
function setupS3ImportExport(scope, props, combineRoles) {
    let s3ImportRole = props.s3ImportRole;
    let s3ExportRole = props.s3ExportRole;
    if (props.s3ImportBuckets && props.s3ImportBuckets.length > 0) {
        if (props.s3ImportRole) {
            throw new Error('Only one of s3ImportRole or s3ImportBuckets must be specified, not both.');
        }
        s3ImportRole = (combineRoles && s3ExportRole) ? s3ExportRole : new iam.Role(scope, 'S3ImportRole', {
            assumedBy: new iam.ServicePrincipal('rds.amazonaws.com'),
        });
        for (const bucket of props.s3ImportBuckets) {
            bucket.grantRead(s3ImportRole);
        }
    }
    if (props.s3ExportBuckets && props.s3ExportBuckets.length > 0) {
        if (props.s3ExportRole) {
            throw new Error('Only one of s3ExportRole or s3ExportBuckets must be specified, not both.');
        }
        s3ExportRole = (combineRoles && s3ImportRole) ? s3ImportRole : new iam.Role(scope, 'S3ExportRole', {
            assumedBy: new iam.ServicePrincipal('rds.amazonaws.com'),
        });
        for (const bucket of props.s3ExportBuckets) {
            bucket.grantReadWrite(s3ExportRole);
        }
    }
    return { s3ImportRole, s3ExportRole };
}
exports.setupS3ImportExport = setupS3ImportExport;
function engineDescription(engine) {
    var _a;
    return engine.engineType + (((_a = engine.engineVersion) === null || _a === void 0 ? void 0 : _a.fullVersion) ? `-${engine.engineVersion.fullVersion}` : '');
}
exports.engineDescription = engineDescription;
/**
 * By default, deletion protection is disabled.
 * Enable if explicitly provided or if the RemovalPolicy has been set to RETAIN
 */
function defaultDeletionProtection(deletionProtection, removalPolicy) {
    return deletionProtection !== null && deletionProtection !== void 0 ? deletionProtection : (removalPolicy === core_1.RemovalPolicy.RETAIN ? true : undefined);
}
exports.defaultDeletionProtection = defaultDeletionProtection;
/**
 * Renders the credentials for an instance or cluster
 */
function renderCredentials(scope, engine, credentials) {
    var _a;
    let renderedCredentials = credentials !== null && credentials !== void 0 ? credentials : props_1.Credentials.fromUsername((_a = engine.defaultUsername) !== null && _a !== void 0 ? _a : 'admin'); // For backwards compatibilty
    if (!renderedCredentials.secret && !renderedCredentials.password) {
        renderedCredentials = props_1.Credentials.fromSecret(new database_secret_1.DatabaseSecret(scope, 'Secret', {
            username: renderedCredentials.username,
            secretName: renderedCredentials.secretName,
            encryptionKey: renderedCredentials.encryptionKey,
            excludeCharacters: renderedCredentials.excludeCharacters,
            // if username must be referenced as a string we can safely replace the
            // secret when customization options are changed without risking a replacement
            replaceOnPasswordCriteriaChanges: credentials === null || credentials === void 0 ? void 0 : credentials.usernameAsString,
        }), 
        // pass username if it must be referenced as a string
        (credentials === null || credentials === void 0 ? void 0 : credentials.usernameAsString) ? renderedCredentials.username : undefined);
    }
    return renderedCredentials;
}
exports.renderCredentials = renderCredentials;
/**
 * The RemovalPolicy that should be applied to a "helper" resource, if the base resource has the given removal policy
 *
 * - For Clusters, this determines the RemovalPolicy for Instances/SubnetGroups.
 * - For Instances, this determines the RemovalPolicy for SubnetGroups.
 *
 * If the basePolicy is:
 *
 *  DESTROY or SNAPSHOT -> DESTROY (snapshot is good enough to recreate)
 *  RETAIN              -> RETAIN  (anything else will lose data or fail to deploy)
 *  (undefined)         -> DESTROY (base policy is assumed to be SNAPSHOT)
 */
function helperRemovalPolicy(basePolicy) {
    return basePolicy === core_1.RemovalPolicy.RETAIN
        ? core_1.RemovalPolicy.RETAIN
        : core_1.RemovalPolicy.DESTROY;
}
exports.helperRemovalPolicy = helperRemovalPolicy;
/**
 * Return a given value unless it's the same as another value
 */
function renderUnless(value, suppressValue) {
    return value === suppressValue ? undefined : value;
}
exports.renderUnless = renderUnless;
//# sourceMappingURL=data:application/json;base64,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