"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseInstanceReadReplica = exports.DatabaseInstanceFromSnapshot = exports.DatabaseInstance = exports.StorageType = exports.LicenseModel = exports.DatabaseInstanceBase = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const events = require("../../aws-events"); // Automatically re-written from '@aws-cdk/aws-events'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const logs = require("../../aws-logs"); // Automatically re-written from '@aws-cdk/aws-logs'
const secretsmanager = require("../../aws-secretsmanager"); // Automatically re-written from '@aws-cdk/aws-secretsmanager'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cxapi = require("../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const database_secret_1 = require("./database-secret");
const endpoint_1 = require("./endpoint");
const util_1 = require("./private/util");
const props_1 = require("./props");
const proxy_1 = require("./proxy");
const rds_generated_1 = require("./rds.generated");
const subnet_group_1 = require("./subnet-group");
/**
 * (experimental) A new or imported database instance.
 *
 * @experimental
 */
class DatabaseInstanceBase extends core_1.Resource {
    /**
     * (experimental) Import an existing database instance.
     *
     * @experimental
     */
    static fromDatabaseInstanceAttributes(scope, id, attrs) {
        class Import extends DatabaseInstanceBase {
            constructor() {
                super(...arguments);
                this.defaultPort = ec2.Port.tcp(attrs.port);
                this.connections = new ec2.Connections({
                    securityGroups: attrs.securityGroups,
                    defaultPort: this.defaultPort,
                });
                this.instanceIdentifier = attrs.instanceIdentifier;
                this.dbInstanceEndpointAddress = attrs.instanceEndpointAddress;
                this.dbInstanceEndpointPort = attrs.port.toString();
                this.instanceEndpoint = new endpoint_1.Endpoint(attrs.instanceEndpointAddress, attrs.port);
                this.engine = attrs.engine;
                this.enableIamAuthentication = true;
            }
        }
        return new Import(scope, id);
    }
    /**
     * (experimental) Add a new db proxy to this instance.
     *
     * @experimental
     */
    addProxy(id, options) {
        return new proxy_1.DatabaseProxy(this, id, {
            proxyTarget: proxy_1.ProxyTarget.fromInstance(this),
            ...options,
        });
    }
    /**
     * (experimental) Grant the given identity connection access to the database.
     *
     * @experimental
     */
    grantConnect(grantee) {
        if (this.enableIamAuthentication === false) {
            throw new Error('Cannot grant connect when IAM authentication is disabled');
        }
        this.enableIamAuthentication = true;
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['rds-db:connect'],
            resourceArns: [this.instanceArn],
        });
    }
    /**
     * (experimental) Defines a CloudWatch event rule which triggers for instance events.
     *
     * Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     *
     * @experimental
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addEventPattern({
            source: ['aws.rds'],
            resources: [this.instanceArn],
        });
        rule.addTarget(options.target);
        return rule;
    }
    /**
     * (experimental) The instance arn.
     *
     * @experimental
     */
    get instanceArn() {
        const commonAnComponents = {
            service: 'rds',
            resource: 'db',
            sep: ':',
        };
        const localArn = core_1.Stack.of(this).formatArn({
            ...commonAnComponents,
            resourceName: this.instanceIdentifier,
        });
        return this.getResourceArnAttribute(localArn, {
            ...commonAnComponents,
            resourceName: this.physicalName,
        });
    }
    /**
     * (experimental) Renders the secret attachment target specifications.
     *
     * @experimental
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.instanceIdentifier,
            targetType: secretsmanager.AttachmentTargetType.RDS_DB_INSTANCE,
        };
    }
}
exports.DatabaseInstanceBase = DatabaseInstanceBase;
_a = JSII_RTTI_SYMBOL_1;
DatabaseInstanceBase[_a] = { fqn: "monocdk.aws_rds.DatabaseInstanceBase", version: "1.106.1" };
/**
 * (experimental) The license model.
 *
 * @experimental
 */
var LicenseModel;
(function (LicenseModel) {
    LicenseModel["LICENSE_INCLUDED"] = "license-included";
    LicenseModel["BRING_YOUR_OWN_LICENSE"] = "bring-your-own-license";
    LicenseModel["GENERAL_PUBLIC_LICENSE"] = "general-public-license";
})(LicenseModel = exports.LicenseModel || (exports.LicenseModel = {}));
/**
 * (experimental) The type of storage.
 *
 * @experimental
 */
var StorageType;
(function (StorageType) {
    StorageType["STANDARD"] = "standard";
    StorageType["GP2"] = "gp2";
    StorageType["IO1"] = "io1";
})(StorageType = exports.StorageType || (exports.StorageType = {}));
/**
 * A new database instance.
 */
class DatabaseInstanceNew extends DatabaseInstanceBase {
    constructor(scope, id, props) {
        var _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r;
        // RDS always lower-cases the ID of the database, so use that for the physical name
        // (which is the name used for cross-environment access, so it needs to be correct,
        // regardless of the feature flag that changes it in the template for the L1)
        const instancePhysicalName = core_1.Token.isUnresolved(props.instanceIdentifier)
            ? props.instanceIdentifier
            : (_e = props.instanceIdentifier) === null || _e === void 0 ? void 0 : _e.toLowerCase();
        super(scope, id, {
            physicalName: instancePhysicalName,
        });
        this.vpc = props.vpc;
        if (props.vpcSubnets && props.vpcPlacement) {
            throw new Error('Only one of `vpcSubnets` or `vpcPlacement` can be specified');
        }
        this.vpcPlacement = (_f = props.vpcSubnets) !== null && _f !== void 0 ? _f : props.vpcPlacement;
        const subnetGroup = (_g = props.subnetGroup) !== null && _g !== void 0 ? _g : new subnet_group_1.SubnetGroup(this, 'SubnetGroup', {
            description: `Subnet group for ${this.node.id} database`,
            vpc: this.vpc,
            vpcSubnets: this.vpcPlacement,
            removalPolicy: util_1.renderUnless(util_1.helperRemovalPolicy(props.removalPolicy), core_1.RemovalPolicy.DESTROY),
        });
        const securityGroups = props.securityGroups || [new ec2.SecurityGroup(this, 'SecurityGroup', {
                description: `Security group for ${this.node.id} database`,
                vpc: props.vpc,
            })];
        this.connections = new ec2.Connections({
            securityGroups,
            defaultPort: ec2.Port.tcp(core_1.Lazy.number({ produce: () => this.instanceEndpoint.port })),
        });
        let monitoringRole;
        if (props.monitoringInterval && props.monitoringInterval.toSeconds()) {
            monitoringRole = props.monitoringRole || new iam.Role(this, 'MonitoringRole', {
                assumedBy: new iam.ServicePrincipal('monitoring.rds.amazonaws.com'),
                managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonRDSEnhancedMonitoringRole')],
            });
        }
        const storageType = props.storageType || StorageType.GP2;
        const iops = storageType === StorageType.IO1 ? (props.iops || 1000) : undefined;
        this.cloudwatchLogsExports = props.cloudwatchLogsExports;
        this.cloudwatchLogsRetention = props.cloudwatchLogsRetention;
        this.cloudwatchLogsRetentionRole = props.cloudwatchLogsRetentionRole;
        this.enableIamAuthentication = props.iamAuthentication;
        const enablePerformanceInsights = props.enablePerformanceInsights
            || props.performanceInsightRetention !== undefined || props.performanceInsightEncryptionKey !== undefined;
        if (enablePerformanceInsights && props.enablePerformanceInsights === false) {
            throw new Error('`enablePerformanceInsights` disabled, but `performanceInsightRetention` or `performanceInsightEncryptionKey` was set');
        }
        if (props.domain) {
            this.domainId = props.domain;
            this.domainRole = props.domainRole || new iam.Role(this, 'RDSDirectoryServiceRole', {
                assumedBy: new iam.ServicePrincipal('rds.amazonaws.com'),
                managedPolicies: [
                    iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonRDSDirectoryServiceAccess'),
                ],
            });
        }
        const maybeLowercasedInstanceId = core_1.FeatureFlags.of(this).isEnabled(cxapi.RDS_LOWERCASE_DB_IDENTIFIER)
            ? (_h = props.instanceIdentifier) === null || _h === void 0 ? void 0 : _h.toLowerCase() : props.instanceIdentifier;
        this.newCfnProps = {
            autoMinorVersionUpgrade: props.autoMinorVersionUpgrade,
            availabilityZone: props.multiAz ? undefined : props.availabilityZone,
            backupRetentionPeriod: (_j = props.backupRetention) === null || _j === void 0 ? void 0 : _j.toDays(),
            copyTagsToSnapshot: (_k = props.copyTagsToSnapshot) !== null && _k !== void 0 ? _k : true,
            dbInstanceClass: core_1.Lazy.string({ produce: () => `db.${this.instanceType}` }),
            dbInstanceIdentifier: core_1.Token.isUnresolved(props.instanceIdentifier)
                // if the passed identifier is a Token,
                // we need to use the physicalName of the database
                // (we cannot change its case anyway),
                // as it might be used in a cross-environment fashion
                ? this.physicalName
                : maybeLowercasedInstanceId,
            dbSubnetGroupName: subnetGroup.subnetGroupName,
            deleteAutomatedBackups: props.deleteAutomatedBackups,
            deletionProtection: util_1.defaultDeletionProtection(props.deletionProtection, props.removalPolicy),
            enableCloudwatchLogsExports: this.cloudwatchLogsExports,
            enableIamDatabaseAuthentication: core_1.Lazy.any({ produce: () => this.enableIamAuthentication }),
            enablePerformanceInsights: enablePerformanceInsights || props.enablePerformanceInsights,
            iops,
            monitoringInterval: (_l = props.monitoringInterval) === null || _l === void 0 ? void 0 : _l.toSeconds(),
            monitoringRoleArn: monitoringRole === null || monitoringRole === void 0 ? void 0 : monitoringRole.roleArn,
            multiAz: props.multiAz,
            optionGroupName: (_m = props.optionGroup) === null || _m === void 0 ? void 0 : _m.optionGroupName,
            performanceInsightsKmsKeyId: (_o = props.performanceInsightEncryptionKey) === null || _o === void 0 ? void 0 : _o.keyArn,
            performanceInsightsRetentionPeriod: enablePerformanceInsights
                ? (props.performanceInsightRetention || props_1.PerformanceInsightRetention.DEFAULT)
                : undefined,
            port: (_p = props.port) === null || _p === void 0 ? void 0 : _p.toString(),
            preferredBackupWindow: props.preferredBackupWindow,
            preferredMaintenanceWindow: props.preferredMaintenanceWindow,
            processorFeatures: props.processorFeatures && renderProcessorFeatures(props.processorFeatures),
            publiclyAccessible: (_q = props.publiclyAccessible) !== null && _q !== void 0 ? _q : (this.vpcPlacement && this.vpcPlacement.subnetType === ec2.SubnetType.PUBLIC),
            storageType,
            vpcSecurityGroups: securityGroups.map(s => s.securityGroupId),
            maxAllocatedStorage: props.maxAllocatedStorage,
            domain: this.domainId,
            domainIamRoleName: (_r = this.domainRole) === null || _r === void 0 ? void 0 : _r.roleName,
        };
    }
    /**
     * @experimental
     */
    setLogRetention() {
        if (this.cloudwatchLogsExports && this.cloudwatchLogsRetention) {
            for (const log of this.cloudwatchLogsExports) {
                new logs.LogRetention(this, `LogRetention${log}`, {
                    logGroupName: `/aws/rds/instance/${this.instanceIdentifier}/${log}`,
                    retention: this.cloudwatchLogsRetention,
                    role: this.cloudwatchLogsRetentionRole,
                });
            }
        }
    }
}
/**
 * A new source database instance (not a read replica)
 */
class DatabaseInstanceSource extends DatabaseInstanceNew {
    constructor(scope, id, props) {
        var _e, _f, _g, _h, _j, _k;
        super(scope, id, props);
        this.singleUserRotationApplication = props.engine.singleUserRotationApplication;
        this.multiUserRotationApplication = props.engine.multiUserRotationApplication;
        this.engine = props.engine;
        let { s3ImportRole, s3ExportRole } = util_1.setupS3ImportExport(this, props, true);
        const engineConfig = props.engine.bindToInstance(this, {
            ...props,
            s3ImportRole,
            s3ExportRole,
        });
        const instanceAssociatedRoles = [];
        const engineFeatures = engineConfig.features;
        if (s3ImportRole) {
            if (!(engineFeatures === null || engineFeatures === void 0 ? void 0 : engineFeatures.s3Import)) {
                throw new Error(`Engine '${util_1.engineDescription(props.engine)}' does not support S3 import`);
            }
            instanceAssociatedRoles.push({ roleArn: s3ImportRole.roleArn, featureName: engineFeatures === null || engineFeatures === void 0 ? void 0 : engineFeatures.s3Import });
        }
        if (s3ExportRole) {
            if (!(engineFeatures === null || engineFeatures === void 0 ? void 0 : engineFeatures.s3Export)) {
                throw new Error(`Engine '${util_1.engineDescription(props.engine)}' does not support S3 export`);
            }
            // Only add the export role and feature if they are different from the import role & feature.
            if (s3ImportRole !== s3ExportRole || engineFeatures.s3Import !== (engineFeatures === null || engineFeatures === void 0 ? void 0 : engineFeatures.s3Export)) {
                instanceAssociatedRoles.push({ roleArn: s3ExportRole.roleArn, featureName: engineFeatures === null || engineFeatures === void 0 ? void 0 : engineFeatures.s3Export });
            }
        }
        this.instanceType = (_e = props.instanceType) !== null && _e !== void 0 ? _e : ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.LARGE);
        const instanceParameterGroupConfig = (_f = props.parameterGroup) === null || _f === void 0 ? void 0 : _f.bindToInstance({});
        this.sourceCfnProps = {
            ...this.newCfnProps,
            associatedRoles: instanceAssociatedRoles.length > 0 ? instanceAssociatedRoles : undefined,
            optionGroupName: (_g = engineConfig.optionGroup) === null || _g === void 0 ? void 0 : _g.optionGroupName,
            allocatedStorage: (_j = (_h = props.allocatedStorage) === null || _h === void 0 ? void 0 : _h.toString()) !== null && _j !== void 0 ? _j : '100',
            allowMajorVersionUpgrade: props.allowMajorVersionUpgrade,
            dbName: props.databaseName,
            dbParameterGroupName: instanceParameterGroupConfig === null || instanceParameterGroupConfig === void 0 ? void 0 : instanceParameterGroupConfig.parameterGroupName,
            engine: props.engine.engineType,
            engineVersion: (_k = props.engine.engineVersion) === null || _k === void 0 ? void 0 : _k.fullVersion,
            licenseModel: props.licenseModel,
            timezone: props.timezone,
        };
    }
    /**
     * (experimental) Adds the single user rotation of the master password to this instance.
     *
     * @param options the options for the rotation, if you want to override the defaults.
     * @experimental
     */
    addRotationSingleUser(options = {}) {
        var _e;
        if (!this.secret) {
            throw new Error('Cannot add single user rotation for an instance without secret.');
        }
        const id = 'RotationSingleUser';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('A single user rotation was already added to this instance.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: this.secret,
            application: this.singleUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcPlacement,
            target: this,
            ...options,
            excludeCharacters: (_e = options.excludeCharacters) !== null && _e !== void 0 ? _e : util_1.DEFAULT_PASSWORD_EXCLUDE_CHARS,
        });
    }
    /**
     * (experimental) Adds the multi user rotation to this instance.
     *
     * @experimental
     */
    addRotationMultiUser(id, options) {
        var _e;
        if (!this.secret) {
            throw new Error('Cannot add multi user rotation for an instance without secret.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            ...options,
            excludeCharacters: (_e = options.excludeCharacters) !== null && _e !== void 0 ? _e : util_1.DEFAULT_PASSWORD_EXCLUDE_CHARS,
            masterSecret: this.secret,
            application: this.multiUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcPlacement,
            target: this,
        });
    }
}
/**
 * (experimental) A database instance.
 *
 * @experimental
 * @resource AWS::RDS::DBInstance
 */
class DatabaseInstance extends DatabaseInstanceSource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _e, _f;
        super(scope, id, props);
        const credentials = util_1.renderCredentials(this, props.engine, props.credentials);
        const secret = credentials.secret;
        const instance = new rds_generated_1.CfnDBInstance(this, 'Resource', {
            ...this.sourceCfnProps,
            characterSetName: props.characterSetName,
            kmsKeyId: props.storageEncryptionKey && props.storageEncryptionKey.keyArn,
            masterUsername: credentials.username,
            masterUserPassword: (_e = credentials.password) === null || _e === void 0 ? void 0 : _e.toString(),
            storageEncrypted: props.storageEncryptionKey ? true : props.storageEncrypted,
        });
        this.instanceIdentifier = this.getResourceNameAttribute(instance.ref);
        this.dbInstanceEndpointAddress = instance.attrEndpointAddress;
        this.dbInstanceEndpointPort = instance.attrEndpointPort;
        // create a number token that represents the port of the instance
        const portAttribute = core_1.Token.asNumber(instance.attrEndpointPort);
        this.instanceEndpoint = new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute);
        instance.applyRemovalPolicy((_f = props.removalPolicy) !== null && _f !== void 0 ? _f : core_1.RemovalPolicy.SNAPSHOT);
        if (secret) {
            this.secret = secret.attach(this);
        }
        this.setLogRetention();
    }
}
exports.DatabaseInstance = DatabaseInstance;
_b = JSII_RTTI_SYMBOL_1;
DatabaseInstance[_b] = { fqn: "monocdk.aws_rds.DatabaseInstance", version: "1.106.1" };
/**
 * (experimental) A database instance restored from a snapshot.
 *
 * @experimental
 * @resource AWS::RDS::DBInstance
 */
class DatabaseInstanceFromSnapshot extends DatabaseInstanceSource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _e, _f, _g, _h;
        super(scope, id, props);
        let credentials = props.credentials;
        let secret = credentials === null || credentials === void 0 ? void 0 : credentials.secret;
        if (!secret && (credentials === null || credentials === void 0 ? void 0 : credentials.generatePassword)) {
            if (!credentials.username) {
                throw new Error('`credentials` `username` must be specified when `generatePassword` is set to true');
            }
            secret = new database_secret_1.DatabaseSecret(this, 'Secret', {
                username: credentials.username,
                encryptionKey: credentials.encryptionKey,
                excludeCharacters: credentials.excludeCharacters,
                replaceOnPasswordCriteriaChanges: credentials.replaceOnPasswordCriteriaChanges,
            });
        }
        const instance = new rds_generated_1.CfnDBInstance(this, 'Resource', {
            ...this.sourceCfnProps,
            dbSnapshotIdentifier: props.snapshotIdentifier,
            masterUserPassword: (_f = (_e = secret === null || secret === void 0 ? void 0 : secret.secretValueFromJson('password')) === null || _e === void 0 ? void 0 : _e.toString()) !== null && _f !== void 0 ? _f : (_g = credentials === null || credentials === void 0 ? void 0 : credentials.password) === null || _g === void 0 ? void 0 : _g.toString(),
        });
        this.instanceIdentifier = instance.ref;
        this.dbInstanceEndpointAddress = instance.attrEndpointAddress;
        this.dbInstanceEndpointPort = instance.attrEndpointPort;
        // create a number token that represents the port of the instance
        const portAttribute = core_1.Token.asNumber(instance.attrEndpointPort);
        this.instanceEndpoint = new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute);
        instance.applyRemovalPolicy((_h = props.removalPolicy) !== null && _h !== void 0 ? _h : core_1.RemovalPolicy.SNAPSHOT);
        if (secret) {
            this.secret = secret.attach(this);
        }
        this.setLogRetention();
    }
}
exports.DatabaseInstanceFromSnapshot = DatabaseInstanceFromSnapshot;
_c = JSII_RTTI_SYMBOL_1;
DatabaseInstanceFromSnapshot[_c] = { fqn: "monocdk.aws_rds.DatabaseInstanceFromSnapshot", version: "1.106.1" };
/**
 * (experimental) A read replica database instance.
 *
 * @experimental
 * @resource AWS::RDS::DBInstance
 */
class DatabaseInstanceReadReplica extends DatabaseInstanceNew {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _e, _f;
        super(scope, id, props);
        /**
         * (experimental) The engine of this database Instance.
         *
         * May be not known for imported Instances if it wasn't provided explicitly,
         * or for read replicas.
         *
         * @experimental
         */
        this.engine = undefined;
        const instance = new rds_generated_1.CfnDBInstance(this, 'Resource', {
            ...this.newCfnProps,
            // this must be ARN, not ID, because of https://github.com/terraform-providers/terraform-provider-aws/issues/528#issuecomment-391169012
            sourceDbInstanceIdentifier: props.sourceDatabaseInstance.instanceArn,
            kmsKeyId: (_e = props.storageEncryptionKey) === null || _e === void 0 ? void 0 : _e.keyArn,
            storageEncrypted: props.storageEncryptionKey ? true : props.storageEncrypted,
        });
        this.instanceType = props.instanceType;
        this.instanceIdentifier = instance.ref;
        this.dbInstanceEndpointAddress = instance.attrEndpointAddress;
        this.dbInstanceEndpointPort = instance.attrEndpointPort;
        // create a number token that represents the port of the instance
        const portAttribute = core_1.Token.asNumber(instance.attrEndpointPort);
        this.instanceEndpoint = new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute);
        instance.applyRemovalPolicy((_f = props.removalPolicy) !== null && _f !== void 0 ? _f : core_1.RemovalPolicy.SNAPSHOT);
        this.setLogRetention();
    }
}
exports.DatabaseInstanceReadReplica = DatabaseInstanceReadReplica;
_d = JSII_RTTI_SYMBOL_1;
DatabaseInstanceReadReplica[_d] = { fqn: "monocdk.aws_rds.DatabaseInstanceReadReplica", version: "1.106.1" };
/**
 * Renders the processor features specifications
 *
 * @param features the processor features
 */
function renderProcessorFeatures(features) {
    const featuresList = Object.entries(features).map(([name, value]) => ({ name, value: value.toString() }));
    return featuresList.length === 0 ? undefined : featuresList;
}
//# sourceMappingURL=data:application/json;base64,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