"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseClusterFromSnapshot = exports.DatabaseCluster = exports.DatabaseClusterBase = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const aws_iam_1 = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const logs = require("../../aws-logs"); // Automatically re-written from '@aws-cdk/aws-logs'
const secretsmanager = require("../../aws-secretsmanager"); // Automatically re-written from '@aws-cdk/aws-secretsmanager'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cxapi = require("../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const endpoint_1 = require("./endpoint");
const util_1 = require("./private/util");
const props_1 = require("./props");
const proxy_1 = require("./proxy");
const rds_generated_1 = require("./rds.generated");
const subnet_group_1 = require("./subnet-group");
/**
 * (experimental) A new or imported clustered database.
 *
 * @experimental
 */
class DatabaseClusterBase extends core_1.Resource {
    /**
     * (experimental) Add a new db proxy to this cluster.
     *
     * @experimental
     */
    addProxy(id, options) {
        return new proxy_1.DatabaseProxy(this, id, {
            proxyTarget: proxy_1.ProxyTarget.fromCluster(this),
            ...options,
        });
    }
    /**
     * (experimental) Renders the secret attachment target specifications.
     *
     * @experimental
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.clusterIdentifier,
            targetType: secretsmanager.AttachmentTargetType.RDS_DB_CLUSTER,
        };
    }
}
exports.DatabaseClusterBase = DatabaseClusterBase;
_a = JSII_RTTI_SYMBOL_1;
DatabaseClusterBase[_a] = { fqn: "monocdk.aws_rds.DatabaseClusterBase", version: "1.106.1" };
/**
 * Abstract base for ``DatabaseCluster`` and ``DatabaseClusterFromSnapshot``
 */
class DatabaseClusterNew extends DatabaseClusterBase {
    constructor(scope, id, props) {
        var _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p;
        super(scope, id);
        const { subnetIds } = props.instanceProps.vpc.selectSubnets(props.instanceProps.vpcSubnets);
        // Cannot test whether the subnets are in different AZs, but at least we can test the amount.
        if (subnetIds.length < 2) {
            core_1.Annotations.of(this).addError(`Cluster requires at least 2 subnets, got ${subnetIds.length}`);
        }
        this.subnetGroup = (_d = props.subnetGroup) !== null && _d !== void 0 ? _d : new subnet_group_1.SubnetGroup(this, 'Subnets', {
            description: `Subnets for ${id} database`,
            vpc: props.instanceProps.vpc,
            vpcSubnets: props.instanceProps.vpcSubnets,
            removalPolicy: util_1.renderUnless(util_1.helperRemovalPolicy(props.removalPolicy), core_1.RemovalPolicy.DESTROY),
        });
        this.securityGroups = (_e = props.instanceProps.securityGroups) !== null && _e !== void 0 ? _e : [
            new ec2.SecurityGroup(this, 'SecurityGroup', {
                description: 'RDS security group',
                vpc: props.instanceProps.vpc,
            }),
        ];
        let { s3ImportRole, s3ExportRole } = util_1.setupS3ImportExport(this, props);
        // bind the engine to the Cluster
        const clusterEngineBindConfig = props.engine.bindToCluster(this, {
            s3ImportRole,
            s3ExportRole,
            parameterGroup: props.parameterGroup,
        });
        const clusterAssociatedRoles = [];
        if (s3ImportRole) {
            clusterAssociatedRoles.push({ roleArn: s3ImportRole.roleArn, featureName: (_f = clusterEngineBindConfig.features) === null || _f === void 0 ? void 0 : _f.s3Import });
        }
        if (s3ExportRole) {
            clusterAssociatedRoles.push({ roleArn: s3ExportRole.roleArn, featureName: (_g = clusterEngineBindConfig.features) === null || _g === void 0 ? void 0 : _g.s3Export });
        }
        const clusterParameterGroup = (_h = props.parameterGroup) !== null && _h !== void 0 ? _h : clusterEngineBindConfig.parameterGroup;
        const clusterParameterGroupConfig = clusterParameterGroup === null || clusterParameterGroup === void 0 ? void 0 : clusterParameterGroup.bindToCluster({});
        this.engine = props.engine;
        const clusterIdentifier = core_1.FeatureFlags.of(this).isEnabled(cxapi.RDS_LOWERCASE_DB_IDENTIFIER)
            ? (_j = props.clusterIdentifier) === null || _j === void 0 ? void 0 : _j.toLowerCase() : props.clusterIdentifier;
        this.newCfnProps = {
            // Basic
            engine: props.engine.engineType,
            engineVersion: (_k = props.engine.engineVersion) === null || _k === void 0 ? void 0 : _k.fullVersion,
            dbClusterIdentifier: clusterIdentifier,
            dbSubnetGroupName: this.subnetGroup.subnetGroupName,
            vpcSecurityGroupIds: this.securityGroups.map(sg => sg.securityGroupId),
            port: (_l = props.port) !== null && _l !== void 0 ? _l : clusterEngineBindConfig.port,
            dbClusterParameterGroupName: clusterParameterGroupConfig === null || clusterParameterGroupConfig === void 0 ? void 0 : clusterParameterGroupConfig.parameterGroupName,
            associatedRoles: clusterAssociatedRoles.length > 0 ? clusterAssociatedRoles : undefined,
            deletionProtection: util_1.defaultDeletionProtection(props.deletionProtection, props.removalPolicy),
            enableIamDatabaseAuthentication: props.iamAuthentication,
            // Admin
            backupRetentionPeriod: (_o = (_m = props.backup) === null || _m === void 0 ? void 0 : _m.retention) === null || _o === void 0 ? void 0 : _o.toDays(),
            preferredBackupWindow: (_p = props.backup) === null || _p === void 0 ? void 0 : _p.preferredWindow,
            preferredMaintenanceWindow: props.preferredMaintenanceWindow,
            databaseName: props.defaultDatabaseName,
            enableCloudwatchLogsExports: props.cloudwatchLogsExports,
        };
    }
}
/**
 * Represents an imported database cluster.
 */
class ImportedDatabaseCluster extends DatabaseClusterBase {
    constructor(scope, id, attrs) {
        super(scope, id);
        this.clusterIdentifier = attrs.clusterIdentifier;
        const defaultPort = attrs.port ? ec2.Port.tcp(attrs.port) : undefined;
        this.connections = new ec2.Connections({
            securityGroups: attrs.securityGroups,
            defaultPort,
        });
        this.engine = attrs.engine;
        this._clusterEndpoint = (attrs.clusterEndpointAddress && attrs.port) ? new endpoint_1.Endpoint(attrs.clusterEndpointAddress, attrs.port) : undefined;
        this._clusterReadEndpoint = (attrs.readerEndpointAddress && attrs.port) ? new endpoint_1.Endpoint(attrs.readerEndpointAddress, attrs.port) : undefined;
        this._instanceIdentifiers = attrs.instanceIdentifiers;
        this._instanceEndpoints = (attrs.instanceEndpointAddresses && attrs.port)
            ? attrs.instanceEndpointAddresses.map(addr => new endpoint_1.Endpoint(addr, attrs.port))
            : undefined;
    }
    get clusterEndpoint() {
        if (!this._clusterEndpoint) {
            throw new Error('Cannot access `clusterEndpoint` of an imported cluster without an endpoint address and port');
        }
        return this._clusterEndpoint;
    }
    get clusterReadEndpoint() {
        if (!this._clusterReadEndpoint) {
            throw new Error('Cannot access `clusterReadEndpoint` of an imported cluster without a readerEndpointAddress and port');
        }
        return this._clusterReadEndpoint;
    }
    get instanceIdentifiers() {
        if (!this._instanceIdentifiers) {
            throw new Error('Cannot access `instanceIdentifiers` of an imported cluster without provided instanceIdentifiers');
        }
        return this._instanceIdentifiers;
    }
    get instanceEndpoints() {
        if (!this._instanceEndpoints) {
            throw new Error('Cannot access `instanceEndpoints` of an imported cluster without instanceEndpointAddresses and port');
        }
        return this._instanceEndpoints;
    }
}
/**
 * (experimental) Create a clustered database with a given number of instances.
 *
 * @experimental
 * @resource AWS::RDS::DBCluster
 */
class DatabaseCluster extends DatabaseClusterNew {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _d, _e, _f;
        super(scope, id, props);
        this.vpc = props.instanceProps.vpc;
        this.vpcSubnets = props.instanceProps.vpcSubnets;
        this.singleUserRotationApplication = props.engine.singleUserRotationApplication;
        this.multiUserRotationApplication = props.engine.multiUserRotationApplication;
        const credentials = util_1.renderCredentials(this, props.engine, props.credentials);
        const secret = credentials.secret;
        const cluster = new rds_generated_1.CfnDBCluster(this, 'Resource', {
            ...this.newCfnProps,
            // Admin
            masterUsername: credentials.username,
            masterUserPassword: (_d = credentials.password) === null || _d === void 0 ? void 0 : _d.toString(),
            // Encryption
            kmsKeyId: (_e = props.storageEncryptionKey) === null || _e === void 0 ? void 0 : _e.keyArn,
            storageEncrypted: props.storageEncryptionKey ? true : props.storageEncrypted,
        });
        this.clusterIdentifier = cluster.ref;
        // create a number token that represents the port of the cluster
        const portAttribute = core_1.Token.asNumber(cluster.attrEndpointPort);
        this.clusterEndpoint = new endpoint_1.Endpoint(cluster.attrEndpointAddress, portAttribute);
        this.clusterReadEndpoint = new endpoint_1.Endpoint(cluster.attrReadEndpointAddress, portAttribute);
        this.connections = new ec2.Connections({
            securityGroups: this.securityGroups,
            defaultPort: ec2.Port.tcp(this.clusterEndpoint.port),
        });
        cluster.applyRemovalPolicy((_f = props.removalPolicy) !== null && _f !== void 0 ? _f : core_1.RemovalPolicy.SNAPSHOT);
        if (secret) {
            this.secret = secret.attach(this);
        }
        setLogRetention(this, props);
        const createdInstances = createInstances(this, props, this.subnetGroup);
        this.instanceIdentifiers = createdInstances.instanceIdentifiers;
        this.instanceEndpoints = createdInstances.instanceEndpoints;
    }
    /**
     * (experimental) Import an existing DatabaseCluster from properties.
     *
     * @experimental
     */
    static fromDatabaseClusterAttributes(scope, id, attrs) {
        return new ImportedDatabaseCluster(scope, id, attrs);
    }
    /**
     * (experimental) Adds the single user rotation of the master password to this cluster.
     *
     * @experimental
     */
    addRotationSingleUser(options = {}) {
        var _d;
        if (!this.secret) {
            throw new Error('Cannot add single user rotation for a cluster without secret.');
        }
        const id = 'RotationSingleUser';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('A single user rotation was already added to this cluster.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: this.secret,
            application: this.singleUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
            ...options,
            excludeCharacters: (_d = options.excludeCharacters) !== null && _d !== void 0 ? _d : util_1.DEFAULT_PASSWORD_EXCLUDE_CHARS,
        });
    }
    /**
     * (experimental) Adds the multi user rotation to this cluster.
     *
     * @experimental
     */
    addRotationMultiUser(id, options) {
        var _d;
        if (!this.secret) {
            throw new Error('Cannot add multi user rotation for a cluster without secret.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            ...options,
            excludeCharacters: (_d = options.excludeCharacters) !== null && _d !== void 0 ? _d : util_1.DEFAULT_PASSWORD_EXCLUDE_CHARS,
            masterSecret: this.secret,
            application: this.multiUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
}
exports.DatabaseCluster = DatabaseCluster;
_b = JSII_RTTI_SYMBOL_1;
DatabaseCluster[_b] = { fqn: "monocdk.aws_rds.DatabaseCluster", version: "1.106.1" };
/**
 * (experimental) A database cluster restored from a snapshot.
 *
 * @experimental
 * @resource AWS::RDS::DBInstance
 */
class DatabaseClusterFromSnapshot extends DatabaseClusterNew {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _d;
        super(scope, id, props);
        const cluster = new rds_generated_1.CfnDBCluster(this, 'Resource', {
            ...this.newCfnProps,
            snapshotIdentifier: props.snapshotIdentifier,
        });
        this.clusterIdentifier = cluster.ref;
        // create a number token that represents the port of the cluster
        const portAttribute = core_1.Token.asNumber(cluster.attrEndpointPort);
        this.clusterEndpoint = new endpoint_1.Endpoint(cluster.attrEndpointAddress, portAttribute);
        this.clusterReadEndpoint = new endpoint_1.Endpoint(cluster.attrReadEndpointAddress, portAttribute);
        this.connections = new ec2.Connections({
            securityGroups: this.securityGroups,
            defaultPort: ec2.Port.tcp(this.clusterEndpoint.port),
        });
        cluster.applyRemovalPolicy((_d = props.removalPolicy) !== null && _d !== void 0 ? _d : core_1.RemovalPolicy.SNAPSHOT);
        setLogRetention(this, props);
        const createdInstances = createInstances(this, props, this.subnetGroup);
        this.instanceIdentifiers = createdInstances.instanceIdentifiers;
        this.instanceEndpoints = createdInstances.instanceEndpoints;
    }
}
exports.DatabaseClusterFromSnapshot = DatabaseClusterFromSnapshot;
_c = JSII_RTTI_SYMBOL_1;
DatabaseClusterFromSnapshot[_c] = { fqn: "monocdk.aws_rds.DatabaseClusterFromSnapshot", version: "1.106.1" };
/**
 * Sets up CloudWatch log retention if configured.
 * A function rather than protected member to prevent exposing ``DatabaseClusterBaseProps``.
 */
function setLogRetention(cluster, props) {
    if (props.cloudwatchLogsExports) {
        const unsupportedLogTypes = props.cloudwatchLogsExports.filter(logType => !props.engine.supportedLogTypes.includes(logType));
        if (unsupportedLogTypes.length > 0) {
            throw new Error(`Unsupported logs for the current engine type: ${unsupportedLogTypes.join(',')}`);
        }
        if (props.cloudwatchLogsRetention) {
            for (const log of props.cloudwatchLogsExports) {
                new logs.LogRetention(cluster, `LogRetention${log}`, {
                    logGroupName: `/aws/rds/cluster/${cluster.clusterIdentifier}/${log}`,
                    retention: props.cloudwatchLogsRetention,
                    role: props.cloudwatchLogsRetentionRole,
                });
            }
        }
    }
}
/**
 * Creates the instances for the cluster.
 * A function rather than a protected method on ``DatabaseClusterNew`` to avoid exposing
 * ``DatabaseClusterNew`` and ``DatabaseClusterBaseProps`` in the API.
 */
function createInstances(cluster, props, subnetGroup) {
    var _d, _e, _f, _g, _h;
    const instanceCount = props.instances != null ? props.instances : 2;
    if (core_1.Token.isUnresolved(instanceCount)) {
        throw new Error('The number of instances an RDS Cluster consists of cannot be provided as a deploy-time only value!');
    }
    if (instanceCount < 1) {
        throw new Error('At least one instance is required');
    }
    const instanceIdentifiers = [];
    const instanceEndpoints = [];
    const portAttribute = cluster.clusterEndpoint.port;
    const instanceProps = props.instanceProps;
    // Get the actual subnet objects so we can depend on internet connectivity.
    const internetConnected = instanceProps.vpc.selectSubnets(instanceProps.vpcSubnets).internetConnectivityEstablished;
    let monitoringRole;
    if (props.monitoringInterval && props.monitoringInterval.toSeconds()) {
        monitoringRole = props.monitoringRole || new aws_iam_1.Role(cluster, 'MonitoringRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('monitoring.rds.amazonaws.com'),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonRDSEnhancedMonitoringRole'),
            ],
        });
    }
    const enablePerformanceInsights = instanceProps.enablePerformanceInsights
        || instanceProps.performanceInsightRetention !== undefined || instanceProps.performanceInsightEncryptionKey !== undefined;
    if (enablePerformanceInsights && instanceProps.enablePerformanceInsights === false) {
        throw new Error('`enablePerformanceInsights` disabled, but `performanceInsightRetention` or `performanceInsightEncryptionKey` was set');
    }
    const instanceType = (_d = instanceProps.instanceType) !== null && _d !== void 0 ? _d : ec2.InstanceType.of(ec2.InstanceClass.T3, ec2.InstanceSize.MEDIUM);
    const instanceParameterGroupConfig = (_e = instanceProps.parameterGroup) === null || _e === void 0 ? void 0 : _e.bindToInstance({});
    for (let i = 0; i < instanceCount; i++) {
        const instanceIndex = i + 1;
        const instanceIdentifier = props.instanceIdentifierBase != null ? `${props.instanceIdentifierBase}${instanceIndex}` :
            props.clusterIdentifier != null ? `${props.clusterIdentifier}instance${instanceIndex}` :
                undefined;
        const instance = new rds_generated_1.CfnDBInstance(cluster, `Instance${instanceIndex}`, {
            // Link to cluster
            engine: props.engine.engineType,
            engineVersion: (_f = props.engine.engineVersion) === null || _f === void 0 ? void 0 : _f.fullVersion,
            dbClusterIdentifier: cluster.clusterIdentifier,
            dbInstanceIdentifier: instanceIdentifier,
            // Instance properties
            dbInstanceClass: databaseInstanceType(instanceType),
            publiclyAccessible: (_g = instanceProps.publiclyAccessible) !== null && _g !== void 0 ? _g : (instanceProps.vpcSubnets && instanceProps.vpcSubnets.subnetType === ec2.SubnetType.PUBLIC),
            enablePerformanceInsights: enablePerformanceInsights || instanceProps.enablePerformanceInsights,
            performanceInsightsKmsKeyId: (_h = instanceProps.performanceInsightEncryptionKey) === null || _h === void 0 ? void 0 : _h.keyArn,
            performanceInsightsRetentionPeriod: enablePerformanceInsights
                ? (instanceProps.performanceInsightRetention || props_1.PerformanceInsightRetention.DEFAULT)
                : undefined,
            // This is already set on the Cluster. Unclear to me whether it should be repeated or not. Better yes.
            dbSubnetGroupName: subnetGroup.subnetGroupName,
            dbParameterGroupName: instanceParameterGroupConfig === null || instanceParameterGroupConfig === void 0 ? void 0 : instanceParameterGroupConfig.parameterGroupName,
            monitoringInterval: props.monitoringInterval && props.monitoringInterval.toSeconds(),
            monitoringRoleArn: monitoringRole && monitoringRole.roleArn,
            autoMinorVersionUpgrade: props.instanceProps.autoMinorVersionUpgrade,
            allowMajorVersionUpgrade: props.instanceProps.allowMajorVersionUpgrade,
            deleteAutomatedBackups: props.instanceProps.deleteAutomatedBackups,
        });
        // For instances that are part of a cluster:
        //
        //  Cluster DESTROY or SNAPSHOT -> DESTROY (snapshot is good enough to recreate)
        //  Cluster RETAIN              -> RETAIN (otherwise cluster state will disappear)
        instance.applyRemovalPolicy(util_1.helperRemovalPolicy(props.removalPolicy));
        // We must have a dependency on the NAT gateway provider here to create
        // things in the right order.
        instance.node.addDependency(internetConnected);
        instanceIdentifiers.push(instance.ref);
        instanceEndpoints.push(new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute));
    }
    return { instanceEndpoints, instanceIdentifiers };
}
/**
 * Turn a regular instance type into a database instance type
 */
function databaseInstanceType(instanceType) {
    return 'db.' + instanceType.toString();
}
//# sourceMappingURL=data:application/json;base64,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