import * as ec2 from "../../aws-ec2";
import { IRole } from "../../aws-iam";
import * as kms from "../../aws-kms";
import * as logs from "../../aws-logs";
import * as s3 from "../../aws-s3";
import * as secretsmanager from "../../aws-secretsmanager";
import { Duration, RemovalPolicy, Resource } from "../../core";
import { Construct } from 'constructs';
import { IClusterEngine } from './cluster-engine';
import { DatabaseClusterAttributes, IDatabaseCluster } from './cluster-ref';
import { Endpoint } from './endpoint';
import { IParameterGroup } from './parameter-group';
import { BackupProps, Credentials, InstanceProps, RotationSingleUserOptions, RotationMultiUserOptions } from './props';
import { DatabaseProxy, DatabaseProxyOptions } from './proxy';
import { CfnDBClusterProps } from './rds.generated';
import { ISubnetGroup } from './subnet-group';
/**
 * Common properties for a new database cluster or cluster from snapshot.
 */
interface DatabaseClusterBaseProps {
    /**
     * (experimental) What kind of database to start.
     *
     * @experimental
     */
    readonly engine: IClusterEngine;
    /**
     * (experimental) How many replicas/instances to create.
     *
     * Has to be at least 1.
     *
     * @default 2
     * @experimental
     */
    readonly instances?: number;
    /**
     * (experimental) Settings for the individual instances that are launched.
     *
     * @experimental
     */
    readonly instanceProps: InstanceProps;
    /**
     * (experimental) Backup settings.
     *
     * @default - Backup retention period for automated backups is 1 day.
     * Backup preferred window is set to a 30-minute window selected at random from an
     * 8-hour block of time for each AWS Region, occurring on a random day of the week.
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html#USER_WorkingWithAutomatedBackups.BackupWindow
     * @experimental
     */
    readonly backup?: BackupProps;
    /**
     * (experimental) What port to listen on.
     *
     * @default - The default for the engine is used.
     * @experimental
     */
    readonly port?: number;
    /**
     * (experimental) An optional identifier for the cluster.
     *
     * @default - A name is automatically generated.
     * @experimental
     */
    readonly clusterIdentifier?: string;
    /**
     * (experimental) Base identifier for instances.
     *
     * Every replica is named by appending the replica number to this string, 1-based.
     *
     * @default - clusterIdentifier is used with the word "Instance" appended.
     * If clusterIdentifier is not provided, the identifier is automatically generated.
     * @experimental
     */
    readonly instanceIdentifierBase?: string;
    /**
     * (experimental) Name of a database which is automatically created inside the cluster.
     *
     * @default - Database is not created in cluster.
     * @experimental
     */
    readonly defaultDatabaseName?: string;
    /**
     * (experimental) Indicates whether the DB cluster should have deletion protection enabled.
     *
     * @default - true if ``removalPolicy`` is RETAIN, false otherwise
     * @experimental
     */
    readonly deletionProtection?: boolean;
    /**
     * (experimental) A preferred maintenance window day/time range. Should be specified as a range ddd:hh24:mi-ddd:hh24:mi (24H Clock UTC).
     *
     * Example: 'Sun:23:45-Mon:00:15'
     *
     * @default - 30-minute window selected at random from an 8-hour block of time for
     * each AWS Region, occurring on a random day of the week.
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_UpgradeDBInstance.Maintenance.html#Concepts.DBMaintenance
     * @experimental
     */
    readonly preferredMaintenanceWindow?: string;
    /**
     * (experimental) Additional parameters to pass to the database engine.
     *
     * @default - No parameter group.
     * @experimental
     */
    readonly parameterGroup?: IParameterGroup;
    /**
     * (experimental) The removal policy to apply when the cluster and its instances are removed from the stack or replaced during an update.
     *
     * @default - RemovalPolicy.SNAPSHOT (remove the cluster and instances, but retain a snapshot of the data)
     * @experimental
     */
    readonly removalPolicy?: RemovalPolicy;
    /**
     * (experimental) The list of log types that need to be enabled for exporting to CloudWatch Logs.
     *
     * @default - no log exports
     * @experimental
     */
    readonly cloudwatchLogsExports?: string[];
    /**
     * (experimental) The number of days log events are kept in CloudWatch Logs.
     *
     * When updating
     * this property, unsetting it doesn't remove the log retention policy. To
     * remove the retention policy, set the value to `Infinity`.
     *
     * @default - logs never expire
     * @experimental
     */
    readonly cloudwatchLogsRetention?: logs.RetentionDays;
    /**
     * (experimental) The IAM role for the Lambda function associated with the custom resource that sets the retention policy.
     *
     * @default - a new role is created.
     * @experimental
     */
    readonly cloudwatchLogsRetentionRole?: IRole;
    /**
     * (experimental) The interval, in seconds, between points when Amazon RDS collects enhanced monitoring metrics for the DB instances.
     *
     * @default no enhanced monitoring
     * @experimental
     */
    readonly monitoringInterval?: Duration;
    /**
     * (experimental) Role that will be used to manage DB instances monitoring.
     *
     * @default - A role is automatically created for you
     * @experimental
     */
    readonly monitoringRole?: IRole;
    /**
     * (experimental) Role that will be associated with this DB cluster to enable S3 import.
     *
     * This feature is only supported by the Aurora database engine.
     *
     * This property must not be used if `s3ImportBuckets` is used.
     *
     * For MySQL:
     *
     * @default - New role is created if `s3ImportBuckets` is set, no role is defined otherwise
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraPostgreSQL.Migrating.html
     * @experimental
     */
    readonly s3ImportRole?: IRole;
    /**
     * (experimental) S3 buckets that you want to load data from. This feature is only supported by the Aurora database engine.
     *
     * This property must not be used if `s3ImportRole` is used.
     *
     * For MySQL:
     *
     * @default - None
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraPostgreSQL.Migrating.html
     * @experimental
     */
    readonly s3ImportBuckets?: s3.IBucket[];
    /**
     * (experimental) Role that will be associated with this DB cluster to enable S3 export.
     *
     * This feature is only supported by the Aurora database engine.
     *
     * This property must not be used if `s3ExportBuckets` is used.
     *
     * For MySQL:
     *
     * @default - New role is created if `s3ExportBuckets` is set, no role is defined otherwise
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/postgresql-s3-export.html
     * @experimental
     */
    readonly s3ExportRole?: IRole;
    /**
     * (experimental) S3 buckets that you want to load data into. This feature is only supported by the Aurora database engine.
     *
     * This property must not be used if `s3ExportRole` is used.
     *
     * For MySQL:
     *
     * @default - None
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/postgresql-s3-export.html
     * @experimental
     */
    readonly s3ExportBuckets?: s3.IBucket[];
    /**
     * (experimental) Existing subnet group for the cluster.
     *
     * @default - a new subnet group will be created.
     * @experimental
     */
    readonly subnetGroup?: ISubnetGroup;
    /**
     * (experimental) Whether to enable mapping of AWS Identity and Access Management (IAM) accounts to database accounts.
     *
     * @default false
     * @experimental
     */
    readonly iamAuthentication?: boolean;
}
/**
 * (experimental) A new or imported clustered database.
 *
 * @experimental
 */
export declare abstract class DatabaseClusterBase extends Resource implements IDatabaseCluster {
    /**
     * (experimental) The engine of this Cluster.
     *
     * May be not known for imported Clusters if it wasn't provided explicitly.
     *
     * @experimental
     */
    abstract readonly engine?: IClusterEngine;
    /**
     * (experimental) Identifier of the cluster.
     *
     * @experimental
     */
    abstract readonly clusterIdentifier: string;
    /**
     * (experimental) Identifiers of the replicas.
     *
     * @experimental
     */
    abstract readonly instanceIdentifiers: string[];
    /**
     * (experimental) The endpoint to use for read/write operations.
     *
     * @experimental
     */
    abstract readonly clusterEndpoint: Endpoint;
    /**
     * (experimental) Endpoint to use for load-balanced read-only operations.
     *
     * @experimental
     */
    abstract readonly clusterReadEndpoint: Endpoint;
    /**
     * (experimental) Endpoints which address each individual replica.
     *
     * @experimental
     */
    abstract readonly instanceEndpoints: Endpoint[];
    /**
     * (experimental) Access to the network connections.
     *
     * @experimental
     */
    abstract readonly connections: ec2.Connections;
    /**
     * (experimental) Add a new db proxy to this cluster.
     *
     * @experimental
     */
    addProxy(id: string, options: DatabaseProxyOptions): DatabaseProxy;
    /**
     * (experimental) Renders the secret attachment target specifications.
     *
     * @experimental
     */
    asSecretAttachmentTarget(): secretsmanager.SecretAttachmentTargetProps;
}
/**
 * Abstract base for ``DatabaseCluster`` and ``DatabaseClusterFromSnapshot``
 */
declare abstract class DatabaseClusterNew extends DatabaseClusterBase {
    /**
     * (experimental) The engine for this Cluster.
     *
     * Never undefined.
     *
     * @experimental
     */
    readonly engine?: IClusterEngine;
    /**
     * @experimental
     */
    protected readonly newCfnProps: CfnDBClusterProps;
    /**
     * @experimental
     */
    protected readonly securityGroups: ec2.ISecurityGroup[];
    /**
     * @experimental
     */
    protected readonly subnetGroup: ISubnetGroup;
    constructor(scope: Construct, id: string, props: DatabaseClusterBaseProps);
}
/**
 * (experimental) Properties for a new database cluster.
 *
 * @experimental
 */
export interface DatabaseClusterProps extends DatabaseClusterBaseProps {
    /**
     * (experimental) Credentials for the administrative user.
     *
     * @default - A username of 'admin' (or 'postgres' for PostgreSQL) and SecretsManager-generated password
     * @experimental
     */
    readonly credentials?: Credentials;
    /**
     * (experimental) Whether to enable storage encryption.
     *
     * @default - true if storageEncryptionKey is provided, false otherwise
     * @experimental
     */
    readonly storageEncrypted?: boolean;
    /**
     * (experimental) The KMS key for storage encryption.
     *
     * If specified, {@link storageEncrypted} will be set to `true`.
     *
     * @default - if storageEncrypted is true then the default master key, no key otherwise
     * @experimental
     */
    readonly storageEncryptionKey?: kms.IKey;
}
/**
 * (experimental) Create a clustered database with a given number of instances.
 *
 * @experimental
 * @resource AWS::RDS::DBCluster
 */
export declare class DatabaseCluster extends DatabaseClusterNew {
    /**
     * (experimental) Import an existing DatabaseCluster from properties.
     *
     * @experimental
     */
    static fromDatabaseClusterAttributes(scope: Construct, id: string, attrs: DatabaseClusterAttributes): IDatabaseCluster;
    /**
     * (experimental) Identifier of the cluster.
     *
     * @experimental
     */
    readonly clusterIdentifier: string;
    /**
     * (experimental) The endpoint to use for read/write operations.
     *
     * @experimental
     */
    readonly clusterEndpoint: Endpoint;
    /**
     * (experimental) Endpoint to use for load-balanced read-only operations.
     *
     * @experimental
     */
    readonly clusterReadEndpoint: Endpoint;
    /**
     * (experimental) Access to the network connections.
     *
     * @experimental
     */
    readonly connections: ec2.Connections;
    /**
     * (experimental) Identifiers of the replicas.
     *
     * @experimental
     */
    readonly instanceIdentifiers: string[];
    /**
     * (experimental) Endpoints which address each individual replica.
     *
     * @experimental
     */
    readonly instanceEndpoints: Endpoint[];
    /**
     * (experimental) The secret attached to this cluster.
     *
     * @experimental
     */
    readonly secret?: secretsmanager.ISecret;
    private readonly vpc;
    private readonly vpcSubnets?;
    private readonly singleUserRotationApplication;
    private readonly multiUserRotationApplication;
    /**
     * @experimental
     */
    constructor(scope: Construct, id: string, props: DatabaseClusterProps);
    /**
     * (experimental) Adds the single user rotation of the master password to this cluster.
     *
     * @experimental
     */
    addRotationSingleUser(options?: RotationSingleUserOptions): secretsmanager.SecretRotation;
    /**
     * (experimental) Adds the multi user rotation to this cluster.
     *
     * @experimental
     */
    addRotationMultiUser(id: string, options: RotationMultiUserOptions): secretsmanager.SecretRotation;
}
/**
 * (experimental) Properties for ``DatabaseClusterFromSnapshot``.
 *
 * @experimental
 */
export interface DatabaseClusterFromSnapshotProps extends DatabaseClusterBaseProps {
    /**
     * (experimental) The identifier for the DB instance snapshot or DB cluster snapshot to restore from.
     *
     * You can use either the name or the Amazon Resource Name (ARN) to specify a DB cluster snapshot.
     * However, you can use only the ARN to specify a DB instance snapshot.
     *
     * @experimental
     */
    readonly snapshotIdentifier: string;
}
/**
 * (experimental) A database cluster restored from a snapshot.
 *
 * @experimental
 * @resource AWS::RDS::DBInstance
 */
export declare class DatabaseClusterFromSnapshot extends DatabaseClusterNew {
    /**
     * (experimental) Identifier of the cluster.
     *
     * @experimental
     */
    readonly clusterIdentifier: string;
    /**
     * (experimental) The endpoint to use for read/write operations.
     *
     * @experimental
     */
    readonly clusterEndpoint: Endpoint;
    /**
     * (experimental) Endpoint to use for load-balanced read-only operations.
     *
     * @experimental
     */
    readonly clusterReadEndpoint: Endpoint;
    /**
     * (experimental) Access to the network connections.
     *
     * @experimental
     */
    readonly connections: ec2.Connections;
    /**
     * (experimental) Identifiers of the replicas.
     *
     * @experimental
     */
    readonly instanceIdentifiers: string[];
    /**
     * (experimental) Endpoints which address each individual replica.
     *
     * @experimental
     */
    readonly instanceEndpoints: Endpoint[];
    /**
     * @experimental
     */
    constructor(scope: Construct, id: string, props: DatabaseClusterFromSnapshotProps);
}
export {};
