"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.trimFromStart = exports.calculateFunctionHash = void 0;
const crypto = require("crypto");
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cx_api_1 = require("../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const function_1 = require("./function");
function calculateFunctionHash(fn) {
    const stack = core_1.Stack.of(fn);
    const functionResource = fn.node.defaultChild;
    // render the cloudformation resource from this function
    const config = stack.resolve(functionResource._toCloudFormation());
    // config is of the shape: { Resources: { LogicalId: { Type: 'Function', Properties: { ... } }}}
    const resources = config.Resources;
    const resourceKeys = Object.keys(resources);
    if (resourceKeys.length !== 1) {
        throw new Error(`Expected one rendered CloudFormation resource but found ${resourceKeys.length}`);
    }
    const logicalId = resourceKeys[0];
    const properties = resources[logicalId].Properties;
    let stringifiedConfig;
    if (core_1.FeatureFlags.of(fn).isEnabled(cx_api_1.LAMBDA_RECOGNIZE_VERSION_PROPS)) {
        const updatedProps = sortProperties(filterUsefulKeys(properties));
        stringifiedConfig = JSON.stringify(updatedProps);
    }
    else {
        const sorted = sortProperties(properties);
        config.Resources[logicalId].Properties = sorted;
        stringifiedConfig = JSON.stringify(config);
    }
    const hash = crypto.createHash('md5');
    hash.update(stringifiedConfig);
    return hash.digest('hex');
}
exports.calculateFunctionHash = calculateFunctionHash;
function trimFromStart(s, maxLength) {
    const desiredLength = Math.min(maxLength, s.length);
    const newStart = s.length - desiredLength;
    return s.substring(newStart);
}
exports.trimFromStart = trimFromStart;
/*
 * The list of properties found in CfnFunction (or AWS::Lambda::Function).
 * They are classified as "locked" to a Function Version or not.
 * When a property is locked, any change to that property will not take effect on previously created Versions.
 * Instead, a new Version must be generated for the change to take effect.
 * Similarly, if a property that's not locked to a Version is modified, a new Version
 * must not be generated.
 *
 * Adding a new property to this list - If the property is part of the UpdateFunctionConfiguration
 * API, then it must be classified as true, otherwise false.
 * See https://docs.aws.amazon.com/lambda/latest/dg/API_UpdateFunctionConfiguration.html
 */
const VERSION_LOCKED = {
    // locked to the version
    Code: true,
    DeadLetterConfig: true,
    Description: true,
    Environment: true,
    FileSystemConfigs: true,
    FunctionName: true,
    Handler: true,
    ImageConfig: true,
    KmsKeyArn: true,
    Layers: true,
    MemorySize: true,
    PackageType: true,
    Role: true,
    Runtime: true,
    Timeout: true,
    TracingConfig: true,
    VpcConfig: true,
    // not locked to the version
    CodeSigningConfigArn: false,
    ReservedConcurrentExecutions: false,
    Tags: false,
};
function filterUsefulKeys(properties) {
    const versionProps = { ...VERSION_LOCKED, ...function_1.Function._VER_PROPS };
    const unclassified = Object.entries(properties)
        .filter(([k, v]) => v != null && !Object.keys(versionProps).includes(k))
        .map(([k, _]) => k);
    if (unclassified.length > 0) {
        throw new Error(`The following properties are not recognized as version properties: [${unclassified}].`
            + ' See the README of the aws-lambda module to learn more about this and to fix it.');
    }
    const notLocked = Object.entries(versionProps).filter(([_, v]) => !v).map(([k, _]) => k);
    notLocked.forEach(p => delete properties[p]);
    const ret = {};
    Object.entries(properties).filter(([k, _]) => versionProps[k]).forEach(([k, v]) => ret[k] = v);
    return ret;
}
function sortProperties(properties) {
    const ret = {};
    // We take all required properties in the order that they were historically,
    // to make sure the hash we calculate is stable.
    // There cannot be more required properties added in the future,
    // as that would be a backwards-incompatible change.
    const requiredProperties = ['Code', 'Handler', 'Role', 'Runtime'];
    for (const requiredProperty of requiredProperties) {
        ret[requiredProperty] = properties[requiredProperty];
    }
    // then, add all of the non-required properties,
    // in the original order
    for (const property of Object.keys(properties)) {
        if (requiredProperties.indexOf(property) === -1) {
            ret[property] = properties[property];
        }
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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