"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.stageDependencies = exports.bundle = exports.BUNDLER_DEPENDENCIES_CACHE = exports.DEPENDENCY_EXCLUDES = void 0;
const fs = require("fs");
const path = require("path");
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * Dependency files to exclude from the asset hash.
 */
exports.DEPENDENCY_EXCLUDES = ['*.pyc'];
/**
 * The location in the image that the bundler image caches dependencies.
 */
exports.BUNDLER_DEPENDENCIES_CACHE = '/var/dependencies';
/**
 * Produce bundled Lambda asset code
 */
function bundle(options) {
    const { entry, runtime, outputPathSuffix } = options;
    const stagedir = cdk.FileSystem.mkdtemp('python-bundling-');
    const hasDeps = stageDependencies(entry, stagedir);
    const depsCommand = chain([
        hasDeps ? `rsync -r ${exports.BUNDLER_DEPENDENCIES_CACHE}/. ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/${outputPathSuffix}` : '',
        `rsync -r . ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/${outputPathSuffix}`,
    ]);
    // Determine which dockerfile to use. When dependencies are present, we use a
    // Dockerfile that can create a cacheable layer. We can't use this Dockerfile
    // if there aren't dependencies or the Dockerfile will complain about missing
    // sources.
    const dockerfile = hasDeps
        ? 'Dockerfile.dependencies'
        : 'Dockerfile';
    // copy Dockerfile to workdir
    fs.copyFileSync(path.join(__dirname, dockerfile), path.join(stagedir, dockerfile));
    const image = cdk.DockerImage.fromBuild(stagedir, {
        buildArgs: {
            IMAGE: runtime.bundlingDockerImage.image,
        },
        file: dockerfile,
    });
    return lambda.Code.fromAsset(entry, {
        assetHashType: options.assetHashType,
        assetHash: options.assetHash,
        exclude: exports.DEPENDENCY_EXCLUDES,
        bundling: {
            image,
            command: ['bash', '-c', depsCommand],
        },
    });
}
exports.bundle = bundle;
/**
 * Checks to see if the `entry` directory contains a type of dependency that
 * we know how to install.
 */
function stageDependencies(entry, stagedir) {
    const prefixes = [
        'Pipfile',
        'pyproject',
        'poetry',
        'requirements.txt',
    ];
    let found = false;
    for (const file of fs.readdirSync(entry)) {
        for (const prefix of prefixes) {
            if (file.startsWith(prefix)) {
                fs.copyFileSync(path.join(entry, file), path.join(stagedir, file));
                found = true;
            }
        }
    }
    return found;
}
exports.stageDependencies = stageDependencies;
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,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