"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractDependencies = exports.tryGetModuleVersion = exports.exec = exports.findUp = exports.callsites = void 0;
const child_process_1 = require("child_process");
const fs = require("fs");
const path = require("path");
/**
 * Get callsites from the V8 stack trace API
 *
 * https://github.com/sindresorhus/callsites
 */
function callsites() {
    var _a;
    const _prepareStackTrace = Error.prepareStackTrace;
    Error.prepareStackTrace = (_, stack) => stack;
    const stack = (_a = new Error().stack) === null || _a === void 0 ? void 0 : _a.slice(1);
    Error.prepareStackTrace = _prepareStackTrace;
    return stack;
}
exports.callsites = callsites;
/**
 * Find a file by walking up parent directories
 */
function findUp(name, directory = process.cwd()) {
    const absoluteDirectory = path.resolve(directory);
    const file = path.join(directory, name);
    if (fs.existsSync(file)) {
        return file;
    }
    const { root } = path.parse(absoluteDirectory);
    if (absoluteDirectory === root) {
        return undefined;
    }
    return findUp(name, path.dirname(absoluteDirectory));
}
exports.findUp = findUp;
/**
 * Spawn sync with error handling
 */
function exec(cmd, args, options) {
    var _a, _b;
    const proc = child_process_1.spawnSync(cmd, args, options);
    if (proc.error) {
        throw proc.error;
    }
    if (proc.status !== 0) {
        if (proc.stdout || proc.stderr) {
            throw new Error(`[Status ${proc.status}] stdout: ${(_a = proc.stdout) === null || _a === void 0 ? void 0 : _a.toString().trim()}\n\n\nstderr: ${(_b = proc.stderr) === null || _b === void 0 ? void 0 : _b.toString().trim()}`);
        }
        throw new Error(`${cmd} exited with status ${proc.status}`);
    }
    return proc;
}
exports.exec = exec;
/**
 * Returns a module version by requiring its package.json file
 */
function tryGetModuleVersion(mod) {
    try {
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        return require(`${mod}/package.json`).version;
    }
    catch (err) {
        return undefined;
    }
}
exports.tryGetModuleVersion = tryGetModuleVersion;
/**
 * Extract versions for a list of modules.
 *
 * First lookup the version in the package.json and then fallback to requiring
 * the module's package.json. The fallback is needed for transitive dependencies.
 */
function extractDependencies(pkgPath, modules) {
    var _a, _b, _c, _d;
    const dependencies = {};
    // Use require for cache
    const pkgJson = require(pkgPath); // eslint-disable-line @typescript-eslint/no-require-imports
    const pkgDependencies = {
        ...(_a = pkgJson.dependencies) !== null && _a !== void 0 ? _a : {},
        ...(_b = pkgJson.devDependencies) !== null && _b !== void 0 ? _b : {},
        ...(_c = pkgJson.peerDependencies) !== null && _c !== void 0 ? _c : {},
    };
    for (const mod of modules) {
        const version = (_d = pkgDependencies[mod]) !== null && _d !== void 0 ? _d : tryGetModuleVersion(mod);
        if (!version) {
            throw new Error(`Cannot extract version for module '${mod}'. Check that it's referenced in your package.json or installed.`);
        }
        dependencies[mod] = version;
    }
    return dependencies;
}
exports.extractDependencies = extractDependencies;
//# sourceMappingURL=data:application/json;base64,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