"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const os = require("os");
const path = require("path");
const aws_lambda_1 = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const esbuild_installation_1 = require("./esbuild-installation");
const package_manager_1 = require("./package-manager");
const util_1 = require("./util");
const ESBUILD_MAJOR_VERSION = '0';
/**
 * Bundling with esbuild
 */
class Bundling {
    constructor(props) {
        var _a, _b, _c, _d, _e, _f;
        this.props = props;
        this.packageManager = package_manager_1.PackageManager.fromLockFile(props.depsLockFilePath);
        Bundling.esbuildInstallation = (_a = Bundling.esbuildInstallation) !== null && _a !== void 0 ? _a : esbuild_installation_1.EsbuildInstallation.detect();
        this.projectRoot = path.dirname(props.depsLockFilePath);
        this.relativeEntryPath = path.relative(this.projectRoot, path.resolve(props.entry));
        if (props.tsconfig) {
            this.relativeTsconfigPath = path.relative(this.projectRoot, path.resolve(props.tsconfig));
        }
        this.externals = [
            ...(_b = props.externalModules) !== null && _b !== void 0 ? _b : ['aws-sdk'],
            ...(_c = props.nodeModules) !== null && _c !== void 0 ? _c : [],
        ];
        // Docker bundling
        const shouldBuildImage = props.forceDockerBundling || !Bundling.esbuildInstallation;
        this.image = shouldBuildImage
            ? (_d = props.dockerImage) !== null && _d !== void 0 ? _d : cdk.DockerImage.fromBuild(path.join(__dirname, '../lib'), {
                buildArgs: {
                    ...(_e = props.buildArgs) !== null && _e !== void 0 ? _e : {},
                    IMAGE: props.runtime.bundlingImage.image,
                    ESBUILD_VERSION: (_f = props.esbuildVersion) !== null && _f !== void 0 ? _f : ESBUILD_MAJOR_VERSION,
                },
            }) : cdk.DockerImage.fromRegistry('dummy'); // Do not build if we don't need to
        const bundlingCommand = this.createBundlingCommand({
            inputDir: cdk.AssetStaging.BUNDLING_INPUT_DIR,
            outputDir: cdk.AssetStaging.BUNDLING_OUTPUT_DIR,
            esbuildRunner: 'esbuild',
            osPlatform: 'linux',
        });
        this.command = ['bash', '-c', bundlingCommand];
        this.environment = props.environment;
        // Bundling sets the working directory to cdk.AssetStaging.BUNDLING_INPUT_DIR
        // and we want to force npx to use the globally installed esbuild.
        this.workingDirectory = '/';
        // Local bundling
        if (!props.forceDockerBundling) { // only if Docker is not forced
            this.local = this.getLocalBundlingProvider();
        }
    }
    /**
     * esbuild bundled Lambda asset code
     */
    static bundle(options) {
        return aws_lambda_1.Code.fromAsset(path.dirname(options.depsLockFilePath), {
            assetHashType: cdk.AssetHashType.OUTPUT,
            bundling: new Bundling(options),
        });
    }
    static clearEsbuildInstallationCache() {
        this.esbuildInstallation = undefined;
    }
    createBundlingCommand(options) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j;
        const pathJoin = osPathJoin(options.osPlatform);
        const loaders = Object.entries((_a = this.props.loader) !== null && _a !== void 0 ? _a : {});
        const defines = Object.entries((_b = this.props.define) !== null && _b !== void 0 ? _b : {});
        const esbuildCommand = [
            options.esbuildRunner,
            '--bundle', `"${pathJoin(options.inputDir, this.relativeEntryPath)}"`,
            `--target=${(_c = this.props.target) !== null && _c !== void 0 ? _c : toTarget(this.props.runtime)}`,
            '--platform=node',
            `--outfile="${pathJoin(options.outputDir, 'index.js')}"`,
            ...this.props.minify ? ['--minify'] : [],
            ...this.props.sourceMap ? ['--sourcemap'] : [],
            ...this.externals.map(external => `--external:${external}`),
            ...loaders.map(([ext, name]) => `--loader:${ext}=${name}`),
            ...defines.map(([key, value]) => `--define:${key}=${JSON.stringify(value)}`),
            ...this.props.logLevel ? [`--log-level=${this.props.logLevel}`] : [],
            ...this.props.keepNames ? ['--keep-names'] : [],
            ...this.relativeTsconfigPath ? [`--tsconfig=${pathJoin(options.inputDir, this.relativeTsconfigPath)}`] : [],
            ...this.props.metafile ? [`--metafile=${pathJoin(options.outputDir, 'index.meta.json')}`] : [],
            ...this.props.banner ? [`--banner:js=${JSON.stringify(this.props.banner)}`] : [],
            ...this.props.footer ? [`--footer:js=${JSON.stringify(this.props.footer)}`] : [],
        ];
        let depsCommand = '';
        if (this.props.nodeModules) {
            // Find 'package.json' closest to entry folder, we are going to extract the
            // modules versions from it.
            const pkgPath = util_1.findUp('package.json', path.dirname(this.props.entry));
            if (!pkgPath) {
                throw new Error('Cannot find a `package.json` in this project. Using `nodeModules` requires a `package.json`.');
            }
            // Determine dependencies versions, lock file and installer
            const dependencies = util_1.extractDependencies(pkgPath, this.props.nodeModules);
            const osCommand = new OsCommand(options.osPlatform);
            // Create dummy package.json, copy lock file if any and then install
            depsCommand = chain([
                osCommand.writeJson(pathJoin(options.outputDir, 'package.json'), { dependencies }),
                osCommand.copy(pathJoin(options.inputDir, this.packageManager.lockFile), pathJoin(options.outputDir, this.packageManager.lockFile)),
                osCommand.changeDirectory(options.outputDir),
                this.packageManager.installCommand.join(' '),
            ]);
        }
        return chain([
            ...(_e = (_d = this.props.commandHooks) === null || _d === void 0 ? void 0 : _d.beforeBundling(options.inputDir, options.outputDir)) !== null && _e !== void 0 ? _e : [],
            esbuildCommand.join(' '),
            ...(_g = (this.props.nodeModules && ((_f = this.props.commandHooks) === null || _f === void 0 ? void 0 : _f.beforeInstall(options.inputDir, options.outputDir)))) !== null && _g !== void 0 ? _g : [],
            depsCommand,
            ...(_j = (_h = this.props.commandHooks) === null || _h === void 0 ? void 0 : _h.afterBundling(options.inputDir, options.outputDir)) !== null && _j !== void 0 ? _j : [],
        ]);
    }
    getLocalBundlingProvider() {
        var _a;
        const osPlatform = os.platform();
        const createLocalCommand = (outputDir, esbuild) => this.createBundlingCommand({
            inputDir: this.projectRoot,
            outputDir,
            esbuildRunner: esbuild.isLocal ? this.packageManager.runBinCommand('esbuild') : 'esbuild',
            osPlatform,
        });
        const environment = (_a = this.props.environment) !== null && _a !== void 0 ? _a : {};
        const cwd = path.dirname(this.props.entry);
        return {
            tryBundle(outputDir) {
                if (!Bundling.esbuildInstallation) {
                    process.stderr.write('esbuild cannot run locally. Switching to Docker bundling.\n');
                    return false;
                }
                if (!Bundling.esbuildInstallation.version.startsWith(`${ESBUILD_MAJOR_VERSION}.`)) {
                    throw new Error(`Expected esbuild version ${ESBUILD_MAJOR_VERSION}.x but got ${Bundling.esbuildInstallation.version}`);
                }
                const localCommand = createLocalCommand(outputDir, Bundling.esbuildInstallation);
                util_1.exec(osPlatform === 'win32' ? 'cmd' : 'bash', [
                    osPlatform === 'win32' ? '/c' : '-c',
                    localCommand,
                ], {
                    env: { ...process.env, ...environment },
                    stdio: [
                        'ignore',
                        process.stderr,
                        'inherit',
                    ],
                    cwd,
                    windowsVerbatimArguments: osPlatform === 'win32',
                });
                return true;
            },
        };
    }
}
exports.Bundling = Bundling;
/**
 * OS agnostic command
 */
class OsCommand {
    constructor(osPlatform) {
        this.osPlatform = osPlatform;
    }
    writeJson(filePath, data) {
        const stringifiedData = JSON.stringify(data);
        if (this.osPlatform === 'win32') {
            return `echo ^${stringifiedData}^ > ${filePath}`;
        }
        return `echo '${stringifiedData}' > ${filePath}`;
    }
    copy(src, dest) {
        if (this.osPlatform === 'win32') {
            return `copy ${src} ${dest}`;
        }
        return `cp ${src} ${dest}`;
    }
    changeDirectory(dir) {
        return `cd ${dir}`;
    }
}
/**
 * Chain commands
 */
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
/**
 * Platform specific path join
 */
function osPathJoin(platform) {
    return function (...paths) {
        const joined = path.join(...paths);
        // If we are on win32 but need posix style paths
        if (os.platform() === 'win32' && platform !== 'win32') {
            return joined.replace(/\\/g, '/');
        }
        return joined;
    };
}
/**
 * Converts a runtime to an esbuild node target
 */
function toTarget(runtime) {
    const match = runtime.name.match(/nodejs(\d+)/);
    if (!match) {
        throw new Error('Cannot extract version from runtime.');
    }
    return `node${match[1]}`;
}
//# sourceMappingURL=data:application/json;base64,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