"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const os = require("os");
const path = require("path");
const aws_lambda_1 = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const util_1 = require("./util");
/**
 * Bundling
 */
class Bundling {
    constructor(props) {
        var _a, _b, _c, _d;
        this.props = props;
        Bundling.runsLocally = (_b = (_a = Bundling.runsLocally) !== null && _a !== void 0 ? _a : util_1.getGoBuildVersion()) !== null && _b !== void 0 ? _b : false;
        const projectRoot = path.dirname(props.moduleDir);
        this.relativeEntryPath = `./${path.relative(projectRoot, path.resolve(props.entry))}`;
        const cgoEnabled = props.cgoEnabled ? '1' : '0';
        const environment = {
            CGO_ENABLED: cgoEnabled,
            GO111MODULE: 'on',
            GOARCH: 'amd64',
            GOOS: 'linux',
            ...props.environment,
        };
        // Docker bundling
        const shouldBuildImage = props.forcedDockerBundling || !Bundling.runsLocally;
        this.image = shouldBuildImage
            ? (_c = props.dockerImage) !== null && _c !== void 0 ? _c : cdk.DockerImage.fromBuild(path.join(__dirname, '../lib'), {
                buildArgs: {
                    ...(_d = props.buildArgs) !== null && _d !== void 0 ? _d : {},
                    IMAGE: aws_lambda_1.Runtime.GO_1_X.bundlingImage.image,
                },
            }) : cdk.DockerImage.fromRegistry('dummy'); // Do not build if we don't need to
        const bundlingCommand = this.createBundlingCommand(cdk.AssetStaging.BUNDLING_INPUT_DIR, cdk.AssetStaging.BUNDLING_OUTPUT_DIR);
        this.command = ['bash', '-c', bundlingCommand];
        this.environment = environment;
        // Local bundling
        if (!props.forcedDockerBundling) { // only if Docker is not forced
            const osPlatform = os.platform();
            const createLocalCommand = (outputDir) => this.createBundlingCommand(projectRoot, outputDir, osPlatform);
            this.local = {
                tryBundle(outputDir) {
                    if (Bundling.runsLocally == false) {
                        process.stderr.write('go build cannot run locally. Switching to Docker bundling.\n');
                        return false;
                    }
                    const localCommand = createLocalCommand(outputDir);
                    util_1.exec(osPlatform === 'win32' ? 'cmd' : 'bash', [
                        osPlatform === 'win32' ? '/c' : '-c',
                        localCommand,
                    ], {
                        env: { ...process.env, ...environment !== null && environment !== void 0 ? environment : {} },
                        stdio: [
                            'ignore',
                            process.stderr,
                            'inherit',
                        ],
                        cwd: path.dirname(props.moduleDir),
                        windowsVerbatimArguments: osPlatform === 'win32',
                    });
                    return true;
                },
            };
        }
    }
    static bundle(options) {
        var _a;
        const bundling = new Bundling(options);
        return aws_lambda_1.Code.fromAsset(path.dirname(options.moduleDir), {
            assetHashType: (_a = options.assetHashType) !== null && _a !== void 0 ? _a : cdk.AssetHashType.OUTPUT,
            assetHash: options.assetHash,
            bundling: {
                image: bundling.image,
                command: bundling.command,
                environment: bundling.environment,
                local: bundling.local,
            },
        });
    }
    static clearRunsLocallyCache() {
        this.runsLocally = undefined;
    }
    createBundlingCommand(inputDir, outputDir, osPlatform = 'linux') {
        var _a, _b, _c, _d;
        const pathJoin = osPathJoin(osPlatform);
        const hasVendor = util_1.findUp('vendor', path.dirname(this.props.entry));
        const goBuildCommand = [
            'go', 'build',
            hasVendor ? '-mod=vendor' : '',
            '-o', `${pathJoin(outputDir, 'bootstrap')}`,
            `${this.props.goBuildFlags ? this.props.goBuildFlags.join(' ') : ''}`,
            `${this.relativeEntryPath.replace(/\\/g, '/')}`,
        ].filter(c => !!c).join(' ');
        return chain([
            ...(_b = (_a = this.props.commandHooks) === null || _a === void 0 ? void 0 : _a.beforeBundling(inputDir, outputDir)) !== null && _b !== void 0 ? _b : [],
            goBuildCommand,
            ...(_d = (_c = this.props.commandHooks) === null || _c === void 0 ? void 0 : _c.afterBundling(inputDir, outputDir)) !== null && _d !== void 0 ? _d : [],
        ]);
    }
}
exports.Bundling = Bundling;
/**
 * Platform specific path join
 */
function osPathJoin(platform) {
    return function (...paths) {
        const joined = path.join(...paths);
        // If we are on win32 but need posix style paths
        if (os.platform() === 'win32' && platform !== 'win32') {
            return joined.replace(/\\/g, '/');
        }
        return joined;
    };
}
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,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