"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Application = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const ka = require("../../aws-kinesisanalytics"); // Automatically re-written from '@aws-cdk/aws-kinesisanalytics'
const logs = require("../../aws-logs"); // Automatically re-written from '@aws-cdk/aws-logs'
const core = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const environment_properties_1 = require("./private/environment-properties");
const flink_application_configuration_1 = require("./private/flink-application-configuration");
const validation_1 = require("./private/validation");
/**
 * Implements the functionality shared between CDK created and imported
 * IApplications.
 */
class ApplicationBase extends core.Resource {
    /**
     * (experimental) Implement the convenience {@link IApplication.addToPrincipalPolicy} method.
     *
     * @experimental
     */
    addToRolePolicy(policyStatement) {
        if (this.role) {
            this.role.addToPrincipalPolicy(policyStatement);
            return true;
        }
        return false;
    }
}
/**
 * An imported Flink application.
 */
class Import extends ApplicationBase {
    constructor(scope, id, attrs) {
        super(scope, id);
        // Imported applications have no associated role or grantPrincipal
        this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
        this.role = undefined;
        this.applicationArn = attrs.applicationArn;
        this.applicationName = attrs.applicationName;
    }
}
/**
 * (experimental) The L2 construct for Flink Kinesis Data Applications.
 *
 * @experimental
 * @resource AWS::KinesisAnalyticsV2::Application
 */
class Application extends ApplicationBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _b, _c, _d;
        super(scope, id, { physicalName: props.applicationName });
        validation_1.validateFlinkApplicationProps(props);
        this.role = (_b = props.role) !== null && _b !== void 0 ? _b : new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('kinesisanalytics.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        // Permit metric publishing to CloudWatch
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['cloudwatch:PutMetricData'],
            resources: ['*'],
        }));
        const code = props.code.bind(this);
        code.bucket.grantRead(this);
        const resource = new ka.CfnApplicationV2(this, 'Resource', {
            runtimeEnvironment: props.runtime.value,
            serviceExecutionRole: this.role.roleArn,
            applicationConfiguration: {
                ...code.applicationCodeConfigurationProperty,
                environmentProperties: environment_properties_1.environmentProperties(props.propertyGroups),
                flinkApplicationConfiguration: flink_application_configuration_1.flinkApplicationConfiguration({
                    checkpointingEnabled: props.checkpointingEnabled,
                    checkpointInterval: props.checkpointInterval,
                    minPauseBetweenCheckpoints: props.minPauseBetweenCheckpoints,
                    logLevel: props.logLevel,
                    metricsLevel: props.metricsLevel,
                    autoScalingEnabled: props.autoScalingEnabled,
                    parallelism: props.parallelism,
                    parallelismPerKpu: props.parallelismPerKpu,
                }),
                applicationSnapshotConfiguration: {
                    snapshotsEnabled: (_c = props.snapshotsEnabled) !== null && _c !== void 0 ? _c : true,
                },
            },
        });
        resource.node.addDependency(this.role);
        const logGroup = (_d = props.logGroup) !== null && _d !== void 0 ? _d : new logs.LogGroup(this, 'LogGroup');
        const logStream = new logs.LogStream(this, 'LogStream', { logGroup });
        /* Permit logging */
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['logs:DescribeLogGroups'],
            resources: [
                core.Stack.of(this).formatArn({
                    service: 'logs',
                    resource: 'log-group',
                    sep: ':',
                    resourceName: '*',
                }),
            ],
        }));
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['logs:DescribeLogStreams'],
            resources: [logGroup.logGroupArn],
        }));
        const logStreamArn = `arn:${core.Aws.PARTITION}:logs:${core.Aws.REGION}:${core.Aws.ACCOUNT_ID}:log-group:${logGroup.logGroupName}:log-stream:${logStream.logStreamName}`;
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['logs:PutLogEvents'],
            resources: [logStreamArn],
        }));
        new ka.CfnApplicationCloudWatchLoggingOptionV2(this, 'LoggingOption', {
            applicationName: resource.ref,
            cloudWatchLoggingOption: {
                logStreamArn,
            },
        });
        this.applicationName = this.getResourceNameAttribute(resource.ref);
        this.applicationArn = this.getResourceArnAttribute(core.Stack.of(this).formatArn(applicationArnComponents(resource.ref)), applicationArnComponents(this.physicalName));
        resource.applyRemovalPolicy(props.removalPolicy, {
            default: core.RemovalPolicy.DESTROY,
        });
    }
    /**
     * (experimental) Import an existing Flink application defined outside of CDK code by applicationName.
     *
     * @experimental
     */
    static fromApplicationName(scope, id, applicationName) {
        const applicationArn = core.Stack.of(scope).formatArn(applicationArnComponents(applicationName));
        return new Import(scope, id, { applicationArn, applicationName });
    }
    /**
     * (experimental) Import an existing application defined outside of CDK code by applicationArn.
     *
     * @experimental
     */
    static fromApplicationArn(scope, id, applicationArn) {
        const applicationName = core.Stack.of(scope).parseArn(applicationArn).resourceName;
        if (!applicationName) {
            throw new Error(`applicationArn for fromApplicationArn (${applicationArn}) must include resource name`);
        }
        return new Import(scope, id, { applicationArn, applicationName });
    }
}
exports.Application = Application;
_a = JSII_RTTI_SYMBOL_1;
Application[_a] = { fqn: "monocdk.aws_kinesisanalytics_flink.Application", version: "1.106.1" };
function applicationArnComponents(resourceName) {
    return {
        service: 'kinesisanalytics',
        resource: 'application',
        resourceName,
    };
}
//# sourceMappingURL=data:application/json;base64,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