"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UniqueStringSet = exports.mergePrincipal = exports.AttachedPolicies = exports.generatePolicyName = exports.undefinedIfEmpty = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const MAX_POLICY_NAME_LEN = 128;
function undefinedIfEmpty(f) {
    return core_1.Lazy.list({
        produce: () => {
            const array = f();
            return (array && array.length > 0) ? array : undefined;
        },
    });
}
exports.undefinedIfEmpty = undefinedIfEmpty;
/**
 * Used to generate a unique policy name based on the policy resource construct.
 * The logical ID of the resource is a great candidate as long as it doesn't exceed
 * 128 characters, so we take the last 128 characters (in order to make sure the hash
 * is there).
 */
function generatePolicyName(scope, logicalId) {
    // as logicalId is itself a Token, resolve it first
    const resolvedLogicalId = core_1.Tokenization.resolve(logicalId, {
        scope,
        resolver: new core_1.DefaultTokenResolver(new core_1.StringConcat()),
    });
    return lastNCharacters(resolvedLogicalId, MAX_POLICY_NAME_LEN);
}
exports.generatePolicyName = generatePolicyName;
/**
 * Returns a string composed of the last n characters of str.
 * If str is shorter than n, returns str.
 *
 * @param str the string to return the last n characters of
 * @param n how many characters to return
 */
function lastNCharacters(str, n) {
    const startIndex = Math.max(str.length - n, 0);
    return str.substring(startIndex, str.length);
}
/**
 * Helper class that maintains the set of attached policies for a principal.
 */
class AttachedPolicies {
    constructor() {
        this.policies = new Array();
    }
    /**
     * Adds a policy to the list of attached policies.
     *
     * If this policy is already, attached, returns false.
     * If there is another policy attached with the same name, throws an exception.
     */
    attach(policy) {
        if (this.policies.find(p => p === policy)) {
            return; // already attached
        }
        if (this.policies.find(p => p.policyName === policy.policyName)) {
            throw new Error(`A policy named "${policy.policyName}" is already attached`);
        }
        this.policies.push(policy);
    }
}
exports.AttachedPolicies = AttachedPolicies;
/**
 * Merge two dictionaries that represent IAM principals
 */
function mergePrincipal(target, source) {
    for (const key of Object.keys(source)) {
        target[key] = target[key] || [];
        let value = source[key];
        if (!Array.isArray(value)) {
            value = [value];
        }
        target[key].push(...value);
    }
    return target;
}
exports.mergePrincipal = mergePrincipal;
/**
 * Lazy string set token that dedupes entries
 *
 * Needs to operate post-resolve, because the inputs could be
 * `[ '${Token[TOKEN.9]}', '${Token[TOKEN.10]}', '${Token[TOKEN.20]}' ]`, which
 * still all resolve to the same string value.
 *
 * Needs to JSON.stringify() results because strings could resolve to literal
 * strings but could also resolve to `{ Fn::Join: [...] }`.
 */
class UniqueStringSet {
    constructor(fn) {
        this.fn = fn;
        this.creationStack = core_1.captureStackTrace();
    }
    static from(fn) {
        return core_1.Token.asList(new UniqueStringSet(fn));
    }
    resolve(context) {
        context.registerPostProcessor(this);
        return this.fn();
    }
    postProcess(input, _context) {
        if (!Array.isArray(input)) {
            return input;
        }
        if (input.length === 0) {
            return undefined;
        }
        const uniq = {};
        for (const el of input) {
            uniq[JSON.stringify(el)] = el;
        }
        return Object.values(uniq);
    }
    toString() {
        return core_1.Token.asString(this);
    }
}
exports.UniqueStringSet = UniqueStringSet;
//# sourceMappingURL=data:application/json;base64,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