"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Table = exports.TableEncryption = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const kms = require("../../aws-kms"); // Automatically re-written from '@aws-cdk/aws-kms'
const s3 = require("../../aws-s3"); // Automatically re-written from '@aws-cdk/aws-s3'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const glue_generated_1 = require("./glue.generated");
/**
 * (experimental) Encryption options for a Table.
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/encryption.html
 * @experimental
 */
var TableEncryption;
(function (TableEncryption) {
    TableEncryption["UNENCRYPTED"] = "Unencrypted";
    TableEncryption["S3_MANAGED"] = "SSE-S3";
    TableEncryption["KMS"] = "SSE-KMS";
    TableEncryption["KMS_MANAGED"] = "SSE-KMS-MANAGED";
    TableEncryption["CLIENT_SIDE_KMS"] = "CSE-KMS";
})(TableEncryption = exports.TableEncryption || (exports.TableEncryption = {}));
/**
 * (experimental) A Glue table.
 *
 * @experimental
 */
class Table extends core_1.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e;
        super(scope, id, {
            physicalName: props.tableName,
        });
        this.database = props.database;
        this.dataFormat = props.dataFormat;
        this.s3Prefix = (_b = props.s3Prefix) !== null && _b !== void 0 ? _b : '';
        validateSchema(props.columns, props.partitionKeys);
        this.columns = props.columns;
        this.partitionKeys = props.partitionKeys;
        this.compressed = (_c = props.compressed) !== null && _c !== void 0 ? _c : false;
        const { bucket, encryption, encryptionKey } = createBucket(this, props);
        this.bucket = bucket;
        this.encryption = encryption;
        this.encryptionKey = encryptionKey;
        const tableResource = new glue_generated_1.CfnTable(this, 'Table', {
            catalogId: props.database.catalogId,
            databaseName: props.database.databaseName,
            tableInput: {
                name: this.physicalName,
                description: props.description || `${props.tableName} generated by CDK`,
                partitionKeys: renderColumns(props.partitionKeys),
                parameters: {
                    classification: (_d = props.dataFormat.classificationString) === null || _d === void 0 ? void 0 : _d.value,
                    has_encrypted_data: this.encryption !== TableEncryption.UNENCRYPTED,
                },
                storageDescriptor: {
                    location: `s3://${this.bucket.bucketName}/${this.s3Prefix}`,
                    compressed: this.compressed,
                    storedAsSubDirectories: (_e = props.storedAsSubDirectories) !== null && _e !== void 0 ? _e : false,
                    columns: renderColumns(props.columns),
                    inputFormat: props.dataFormat.inputFormat.className,
                    outputFormat: props.dataFormat.outputFormat.className,
                    serdeInfo: {
                        serializationLibrary: props.dataFormat.serializationLibrary.className,
                    },
                },
                tableType: 'EXTERNAL_TABLE',
            },
        });
        this.tableName = this.getResourceNameAttribute(tableResource.ref);
        this.tableArn = this.stack.formatArn({
            service: 'glue',
            resource: 'table',
            resourceName: `${this.database.databaseName}/${this.tableName}`,
        });
        this.node.defaultChild = tableResource;
    }
    /**
     * @experimental
     */
    static fromTableArn(scope, id, tableArn) {
        const tableName = core_1.Fn.select(1, core_1.Fn.split('/', core_1.Stack.of(scope).parseArn(tableArn).resourceName));
        return Table.fromTableAttributes(scope, id, {
            tableArn,
            tableName,
        });
    }
    /**
     * (experimental) Creates a Table construct that represents an external table.
     *
     * @param scope The scope creating construct (usually `this`).
     * @param id The construct's id.
     * @param attrs Import attributes.
     * @experimental
     */
    static fromTableAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.tableArn = attrs.tableArn;
                this.tableName = attrs.tableName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * (experimental) Grant read permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param grantee the principal.
     * @experimental
     */
    grantRead(grantee) {
        const ret = this.grant(grantee, readPermissions);
        if (this.encryptionKey && this.encryption === TableEncryption.CLIENT_SIDE_KMS) {
            this.encryptionKey.grantDecrypt(grantee);
        }
        this.bucket.grantRead(grantee, this.getS3PrefixForGrant());
        return ret;
    }
    /**
     * (experimental) Grant write permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param grantee the principal.
     * @experimental
     */
    grantWrite(grantee) {
        const ret = this.grant(grantee, writePermissions);
        if (this.encryptionKey && this.encryption === TableEncryption.CLIENT_SIDE_KMS) {
            this.encryptionKey.grantEncrypt(grantee);
        }
        this.bucket.grantWrite(grantee, this.getS3PrefixForGrant());
        return ret;
    }
    /**
     * (experimental) Grant read and write permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param grantee the principal.
     * @experimental
     */
    grantReadWrite(grantee) {
        const ret = this.grant(grantee, [...readPermissions, ...writePermissions]);
        if (this.encryptionKey && this.encryption === TableEncryption.CLIENT_SIDE_KMS) {
            this.encryptionKey.grantEncryptDecrypt(grantee);
        }
        this.bucket.grantReadWrite(grantee, this.getS3PrefixForGrant());
        return ret;
    }
    grant(grantee, actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            resourceArns: [this.tableArn],
            actions,
        });
    }
    getS3PrefixForGrant() {
        return this.s3Prefix + '*';
    }
}
exports.Table = Table;
_a = JSII_RTTI_SYMBOL_1;
Table[_a] = { fqn: "monocdk.aws_glue.Table", version: "1.106.1" };
function validateSchema(columns, partitionKeys) {
    if (columns.length === 0) {
        throw new Error('you must specify at least one column for the table');
    }
    // Check there is at least one column and no duplicated column names or partition keys.
    const names = new Set();
    (columns.concat(partitionKeys || [])).forEach(column => {
        if (names.has(column.name)) {
            throw new Error(`column names and partition keys must be unique, but \'${column.name}\' is duplicated`);
        }
        names.add(column.name);
    });
}
// map TableEncryption to bucket's SSE configuration (s3.BucketEncryption)
const encryptionMappings = {
    [TableEncryption.S3_MANAGED]: s3.BucketEncryption.S3_MANAGED,
    [TableEncryption.KMS_MANAGED]: s3.BucketEncryption.KMS_MANAGED,
    [TableEncryption.KMS]: s3.BucketEncryption.KMS,
    [TableEncryption.CLIENT_SIDE_KMS]: s3.BucketEncryption.UNENCRYPTED,
    [TableEncryption.UNENCRYPTED]: s3.BucketEncryption.UNENCRYPTED,
};
// create the bucket to store a table's data depending on the `encryption` and `encryptionKey` properties.
function createBucket(table, props) {
    const encryption = props.encryption || TableEncryption.UNENCRYPTED;
    let bucket = props.bucket;
    if (bucket && (encryption !== TableEncryption.UNENCRYPTED && encryption !== TableEncryption.CLIENT_SIDE_KMS)) {
        throw new Error('you can not specify encryption settings if you also provide a bucket');
    }
    let encryptionKey;
    if (encryption === TableEncryption.CLIENT_SIDE_KMS && props.encryptionKey === undefined) {
        // CSE-KMS should behave the same as SSE-KMS - use the provided key or create one automatically
        // Since Bucket only knows about SSE, we repeat the logic for CSE-KMS at the Table level.
        encryptionKey = new kms.Key(table, 'Key');
    }
    else {
        encryptionKey = props.encryptionKey;
    }
    // create the bucket if none was provided
    if (!bucket) {
        if (encryption === TableEncryption.CLIENT_SIDE_KMS) {
            bucket = new s3.Bucket(table, 'Bucket');
        }
        else {
            bucket = new s3.Bucket(table, 'Bucket', {
                encryption: encryptionMappings[encryption],
                encryptionKey,
            });
            encryptionKey = bucket.encryptionKey;
        }
    }
    return {
        bucket,
        encryption,
        encryptionKey,
    };
}
const readPermissions = [
    'glue:BatchDeletePartition',
    'glue:BatchGetPartition',
    'glue:GetPartition',
    'glue:GetPartitions',
    'glue:GetTable',
    'glue:GetTables',
    'glue:GetTableVersion',
    'glue:GetTableVersions',
];
const writePermissions = [
    'glue:BatchCreatePartition',
    'glue:BatchDeletePartition',
    'glue:CreatePartition',
    'glue:DeletePartition',
    'glue:UpdatePartition',
];
function renderColumns(columns) {
    if (columns === undefined) {
        return undefined;
    }
    return columns.map(column => {
        return {
            name: column.name,
            type: column.type.inputString,
            comment: column.comment,
        };
    });
}
//# sourceMappingURL=data:application/json;base64,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