"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AcceleratorSecurityGroupPeer = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const custom_resources_1 = require("../../custom-resources"); // Automatically re-written from '@aws-cdk/custom-resources'
/**
 * The security group used by a Global Accelerator to send traffic to resources in a VPC.
 */
class AcceleratorSecurityGroupPeer {
    constructor(securityGroupId) {
        this.securityGroupId = securityGroupId;
        this.canInlineRule = false;
        this.connections = new ec2.Connections({ peer: this });
        this.uniqueId = 'GlobalAcceleratorGroup';
    }
    /**
     * Lookup the Global Accelerator security group at CloudFormation deployment time.
     *
     * As of this writing, Global Accelerators (AGA) create a single security group per VPC. AGA security groups are shared
     * by all AGAs in an account. Additionally, there is no CloudFormation mechanism to reference the AGA security groups.
     *
     * This makes creating security group rules which allow traffic from an AGA complicated in CDK. This lookup will identify
     * the AGA security group for a given VPC at CloudFormation deployment time, and lets you create rules for traffic from AGA
     * to other resources created by CDK.
     */
    static fromVpc(scope, id, vpc, endpointGroup) {
        // The security group name is always 'GlobalAccelerator'
        const globalAcceleratorSGName = 'GlobalAccelerator';
        // How to reference the security group name in the response from EC2
        const ec2ResponseSGIdField = 'SecurityGroups.0.GroupId';
        // The AWS Custom Resource that make a call to EC2 to get the security group ID, for the given VPC
        const lookupAcceleratorSGCustomResource = new custom_resources_1.AwsCustomResource(scope, id + 'CustomResource', {
            onCreate: {
                service: 'EC2',
                action: 'describeSecurityGroups',
                parameters: {
                    Filters: [
                        {
                            Name: 'group-name',
                            Values: [
                                globalAcceleratorSGName,
                            ],
                        },
                        {
                            Name: 'vpc-id',
                            Values: [
                                vpc.vpcId,
                            ],
                        },
                    ],
                },
                // We get back a list of responses, but the list should be of length 0 or 1
                // Getting no response means no resources have been linked to the AGA
                physicalResourceId: custom_resources_1.PhysicalResourceId.fromResponse(ec2ResponseSGIdField),
            },
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: custom_resources_1.AwsCustomResourcePolicy.ANY_RESOURCE,
            }),
        });
        // We add a dependency on the endpoint group, guaranteeing that CloudFormation won't
        // try and look up the SG before AGA creates it. The SG is created when a VPC resource
        // is associated with an AGA
        lookupAcceleratorSGCustomResource.node.addDependency(endpointGroup.node.defaultChild);
        // Look up the security group ID
        return new AcceleratorSecurityGroupPeer(lookupAcceleratorSGCustomResource.getResponseField(ec2ResponseSGIdField));
    }
    toIngressRuleConfig() {
        return { sourceSecurityGroupId: this.securityGroupId };
    }
    toEgressRuleConfig() {
        return { destinationSecurityGroupId: this.securityGroupId };
    }
}
exports.AcceleratorSecurityGroupPeer = AcceleratorSecurityGroupPeer;
//# sourceMappingURL=data:application/json;base64,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