"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LustreFileSystem = exports.LustreDeploymentType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ec2_1 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const file_system_1 = require("./file-system");
const fsx_generated_1 = require("./fsx.generated");
/**
 * (experimental) The different kinds of file system deployments used by Lustre.
 *
 * @experimental
 */
var LustreDeploymentType;
(function (LustreDeploymentType) {
    LustreDeploymentType["SCRATCH_1"] = "SCRATCH_1";
    LustreDeploymentType["SCRATCH_2"] = "SCRATCH_2";
    LustreDeploymentType["PERSISTENT_1"] = "PERSISTENT_1";
})(LustreDeploymentType = exports.LustreDeploymentType || (exports.LustreDeploymentType = {}));
/**
 * (experimental) The FSx for Lustre File System implementation of IFileSystem.
 *
 * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-fsx-filesystem.html
 * @experimental
 * @resource AWS::FSx::FileSystem
 */
class LustreFileSystem extends file_system_1.FileSystemBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        this.validateProps(props);
        const updatedLustureProps = {
            importedFileChunkSize: props.lustreConfiguration.importedFileChunkSizeMiB,
            weeklyMaintenanceStartTime: (_b = props.lustreConfiguration.weeklyMaintenanceStartTime) === null || _b === void 0 ? void 0 : _b.toTimestamp(),
        };
        const lustreConfiguration = Object.assign({}, props.lustreConfiguration, updatedLustureProps);
        const securityGroup = (props.securityGroup || new aws_ec2_1.SecurityGroup(this, 'FsxLustreSecurityGroup', {
            vpc: props.vpc,
        }));
        securityGroup.addIngressRule(securityGroup, aws_ec2_1.Port.tcpRange(LustreFileSystem.DEFAULT_PORT_RANGE.startPort, LustreFileSystem.DEFAULT_PORT_RANGE.endPort));
        this.connections = LustreFileSystem.configureConnections(securityGroup);
        this.fileSystem = new fsx_generated_1.CfnFileSystem(this, 'Resource', {
            fileSystemType: LustreFileSystem.DEFAULT_FILE_SYSTEM_TYPE,
            subnetIds: [props.vpcSubnet.subnetId],
            backupId: props.backupId,
            kmsKeyId: (props.kmsKey ? props.kmsKey.keyId : undefined),
            lustreConfiguration,
            securityGroupIds: [securityGroup.securityGroupId],
            storageCapacity: props.storageCapacityGiB,
        });
        this.fileSystem.applyRemovalPolicy(props.removalPolicy);
        this.fileSystemId = this.fileSystem.ref;
        this.dnsName = `${this.fileSystemId}.fsx.${this.stack.region}.${core_1.Aws.URL_SUFFIX}`;
        this.mountName = this.fileSystem.attrLustreMountName;
    }
    /**
     * (experimental) Import an existing FSx for Lustre file system from the given properties.
     *
     * @experimental
     */
    static fromLustreFileSystemAttributes(scope, id, attrs) {
        class Import extends file_system_1.FileSystemBase {
            constructor() {
                super(...arguments);
                this.dnsName = attrs.dnsName;
                this.fileSystemId = attrs.fileSystemId;
                this.connections = LustreFileSystem.configureConnections(attrs.securityGroup);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Configures a Connections object with all the ports required by FSx for Lustre
     */
    static configureConnections(securityGroup) {
        const connections = new aws_ec2_1.Connections({
            securityGroups: [securityGroup],
            defaultPort: aws_ec2_1.Port.tcpRange(LustreFileSystem.DEFAULT_PORT_RANGE.startPort, LustreFileSystem.DEFAULT_PORT_RANGE.endPort),
        });
        return connections;
    }
    /**
     * Validates the props provided for a new FSx for Lustre file system.
     */
    validateProps(props) {
        const lustreConfiguration = props.lustreConfiguration;
        const deploymentType = lustreConfiguration.deploymentType;
        // Make sure the import path is valid before validating the export path
        this.validateImportPath(lustreConfiguration.importPath);
        this.validateExportPath(lustreConfiguration.exportPath, lustreConfiguration.importPath);
        this.validateImportedFileChunkSize(lustreConfiguration.importedFileChunkSizeMiB);
        this.validatePerUnitStorageThroughput(deploymentType, lustreConfiguration.perUnitStorageThroughput);
        this.validateStorageCapacity(deploymentType, props.storageCapacityGiB);
    }
    /**
     * Validates the export path is in the correct format and matches the import path.
     */
    validateExportPath(exportPath, importPath) {
        if (exportPath === undefined) {
            return;
        }
        if (importPath === undefined) {
            throw new Error('Cannot define an export path without also defining an import path');
        }
        if (core_1.Token.isUnresolved(exportPath) && core_1.Token.isUnresolved(importPath)) {
            return;
        }
        if (core_1.Token.isUnresolved(importPath) !== core_1.Token.isUnresolved(exportPath)) {
            throw new Error('The importPath and exportPath must each be Tokens or not Tokens, you cannot use a mix');
        }
        if (!exportPath.startsWith(importPath)) {
            throw new Error(`The export path "${exportPath}" is invalid. Expecting the format: s3://{IMPORT_PATH}/optional-prefix`);
        }
        if (exportPath.length > 900) {
            throw new Error(`The export path "${exportPath}" exceeds the maximum length of 900 characters`);
        }
    }
    /**
     * Validates the importedFileChunkSize is in the correct range.
     */
    validateImportedFileChunkSize(importedFileChunkSize) {
        if (importedFileChunkSize === undefined) {
            return;
        }
        if (importedFileChunkSize < 1 || importedFileChunkSize > 512000) {
            throw new Error(`importedFileChunkSize cannot be ${importedFileChunkSize} MiB. It must be a value from 1 to 512,000 MiB`);
        }
    }
    /**
     * Validates the import path is the correct format.
     */
    validateImportPath(importPath) {
        if (importPath === undefined || core_1.Token.isUnresolved(importPath)) {
            return;
        }
        const regexp = /^s3:\/\//;
        if (importPath.search(regexp) === -1) {
            throw new Error(`The import path "${importPath}" is invalid. Expecting the format: s3://{BUCKET_NAME}/optional-prefix`);
        }
        if (importPath.length > 900) {
            throw new Error(`The import path "${importPath}" exceeds the maximum length of 900 characters`);
        }
    }
    /**
     * Validates the perUnitStorageThroughput is defined correctly for the given deploymentType.
     */
    validatePerUnitStorageThroughput(deploymentType, perUnitStorageThroughput) {
        if (perUnitStorageThroughput === undefined) {
            return;
        }
        if (deploymentType !== LustreDeploymentType.PERSISTENT_1) {
            throw new Error('perUnitStorageThroughput can only be set for the PERSISTENT_1 deployment type');
        }
        if (![50, 100, 200].includes(perUnitStorageThroughput)) {
            throw new Error('perUnitStorageThroughput must be 50, 100, or 200 MB/s/TiB');
        }
    }
    /**
     * Validates the storage capacity is an acceptable value for the deployment type.
     */
    validateStorageCapacity(deploymentType, storageCapacity) {
        if (deploymentType === LustreDeploymentType.SCRATCH_1) {
            if (![1200, 2400, 3600].includes(storageCapacity) && storageCapacity % 3600 !== 0) {
                throw new Error('storageCapacity must be 1,200, 2,400, 3,600, or a multiple of 3,600');
            }
        }
        else {
            if (![1200, 2400].includes(storageCapacity) && storageCapacity % 2400 !== 0) {
                throw new Error('storageCapacity must be 1,200, 2,400, or a multiple of 2,400');
            }
        }
    }
}
exports.LustreFileSystem = LustreFileSystem;
_a = JSII_RTTI_SYMBOL_1;
LustreFileSystem[_a] = { fqn: "monocdk.aws_fsx.LustreFileSystem", version: "1.106.1" };
/**
 * The default FSx file system type used by FSx for Lustre.
 */
LustreFileSystem.DEFAULT_FILE_SYSTEM_TYPE = 'LUSTRE';
/**
 * The default ports the file system listens on. Actual port list is: [988, 1021, 1022, 1023]
 */
LustreFileSystem.DEFAULT_PORT_RANGE = { startPort: 988, endPort: 1023 };
//# sourceMappingURL=data:application/json;base64,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