"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Rule = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const constructs_1 = require("constructs");
const events_generated_1 = require("./events.generated");
const schedule_1 = require("./schedule");
const util_1 = require("./util");
/**
 * (experimental) Defines an EventBridge Rule in this stack.
 *
 * @experimental
 * @resource AWS::Events::Rule
 */
class Rule extends core_1.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.ruleName,
        });
        this.targets = new Array();
        this.eventPattern = {};
        this.accountEventBusTargets = {};
        if (props.eventBus && props.schedule) {
            throw new Error('Cannot associate rule with \'eventBus\' when using \'schedule\'');
        }
        this.description = props.description;
        this.scheduleExpression = props.schedule && props.schedule.expressionString;
        const resource = new events_generated_1.CfnRule(this, 'Resource', {
            name: this.physicalName,
            description: this.description,
            state: props.enabled == null ? 'ENABLED' : (props.enabled ? 'ENABLED' : 'DISABLED'),
            scheduleExpression: this.scheduleExpression,
            eventPattern: core_1.Lazy.any({ produce: () => this._renderEventPattern() }),
            targets: core_1.Lazy.any({ produce: () => this.renderTargets() }),
            eventBusName: props.eventBus && props.eventBus.eventBusName,
        });
        this.ruleArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'events',
            resource: 'rule',
            resourceName: this.physicalName,
        });
        this.ruleName = this.getResourceNameAttribute(resource.ref);
        this.addEventPattern(props.eventPattern);
        for (const target of props.targets || []) {
            this.addTarget(target);
        }
    }
    /**
     * (experimental) Import an existing EventBridge Rule provided an ARN.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param eventRuleArn Event Rule ARN (i.e. arn:aws:events:<region>:<account-id>:rule/MyScheduledRule).
     * @experimental
     */
    static fromEventRuleArn(scope, id, eventRuleArn) {
        const parts = core_1.Stack.of(scope).parseArn(eventRuleArn);
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.ruleArn = eventRuleArn;
                this.ruleName = parts.resourceName || '';
            }
        }
        return new Import(scope, id);
    }
    /**
     * (experimental) Adds a target to the rule. The abstract class RuleTarget can be extended to define new targets.
     *
     * No-op if target is undefined.
     *
     * @experimental
     */
    addTarget(target) {
        var _b;
        if (!target) {
            return;
        }
        // Simply increment id for each `addTarget` call. This is guaranteed to be unique.
        const autoGeneratedId = `Target${this.targets.length}`;
        const targetProps = target.bind(this, autoGeneratedId);
        const inputProps = targetProps.input && targetProps.input.bind(this);
        const roleArn = (_b = targetProps.role) === null || _b === void 0 ? void 0 : _b.roleArn;
        const id = targetProps.id || autoGeneratedId;
        if (targetProps.targetResource) {
            const targetStack = core_1.Stack.of(targetProps.targetResource);
            const targetAccount = targetStack.account;
            const targetRegion = targetStack.region;
            const sourceStack = core_1.Stack.of(this);
            const sourceAccount = sourceStack.account;
            const sourceRegion = sourceStack.region;
            if (targetRegion !== sourceRegion) {
                throw new Error('Rule and target must be in the same region');
            }
            if (targetAccount !== sourceAccount) {
                // cross-account event - strap in, this works differently than regular events!
                // based on:
                // https://docs.aws.amazon.com/eventbridge/latest/userguide/eventbridge-cross-account-event-delivery.html
                // for cross-account events, we require concrete accounts
                if (core_1.Token.isUnresolved(targetAccount)) {
                    throw new Error('You need to provide a concrete account for the target stack when using cross-account events');
                }
                if (core_1.Token.isUnresolved(sourceAccount)) {
                    throw new Error('You need to provide a concrete account for the source stack when using cross-account events');
                }
                // and the target region has to be concrete as well
                if (core_1.Token.isUnresolved(targetRegion)) {
                    throw new Error('You need to provide a concrete region for the target stack when using cross-account events');
                }
                // the _actual_ target is just the event bus of the target's account
                // make sure we only add it once per account
                const exists = this.accountEventBusTargets[targetAccount];
                if (!exists) {
                    this.accountEventBusTargets[targetAccount] = true;
                    this.targets.push({
                        id,
                        arn: targetStack.formatArn({
                            service: 'events',
                            resource: 'event-bus',
                            resourceName: 'default',
                            region: targetRegion,
                            account: targetAccount,
                        }),
                    });
                }
                // Grant the source account permissions to publish events to the event bus of the target account.
                // Do it in a separate stack instead of the target stack (which seems like the obvious place to put it),
                // because it needs to be deployed before the rule containing the above event-bus target in the source stack
                // (EventBridge verifies whether you have permissions to the targets on rule creation),
                // but it's common for the target stack to depend on the source stack
                // (that's the case with CodePipeline, for example)
                const sourceApp = this.node.root;
                if (!sourceApp || !core_1.App.isApp(sourceApp)) {
                    throw new Error('Event stack which uses cross-account targets must be part of a CDK app');
                }
                const targetApp = constructs_1.Node.of(targetProps.targetResource).root;
                if (!targetApp || !core_1.App.isApp(targetApp)) {
                    throw new Error('Target stack which uses cross-account event targets must be part of a CDK app');
                }
                if (sourceApp !== targetApp) {
                    throw new Error('Event stack and target stack must belong to the same CDK app');
                }
                const stackId = `EventBusPolicy-${sourceAccount}-${targetRegion}-${targetAccount}`;
                let eventBusPolicyStack = sourceApp.node.tryFindChild(stackId);
                if (!eventBusPolicyStack) {
                    eventBusPolicyStack = new core_1.Stack(sourceApp, stackId, {
                        env: {
                            account: targetAccount,
                            region: targetRegion,
                        },
                        stackName: `${targetStack.stackName}-EventBusPolicy-support-${targetRegion}-${sourceAccount}`,
                    });
                    new events_generated_1.CfnEventBusPolicy(eventBusPolicyStack, 'GivePermToOtherAccount', {
                        action: 'events:PutEvents',
                        statementId: `Allow-account-${sourceAccount}`,
                        principal: sourceAccount,
                    });
                }
                // deploy the event bus permissions before the source stack
                sourceStack.addDependency(eventBusPolicyStack);
                // The actual rule lives in the target stack.
                // Other than the account, it's identical to this one
                // eventPattern is mutable through addEventPattern(), so we need to lazy evaluate it
                // but only Tokens can be lazy in the framework, so make a subclass instead
                const self = this;
                class CopyRule extends Rule {
                    _renderEventPattern() {
                        return self._renderEventPattern();
                    }
                    // we need to override validate(), as it uses the
                    // value of the eventPattern field,
                    // which might be empty in the case of the copied rule
                    // (as the patterns in the original might be added through addEventPattern(),
                    // not passed through the constructor).
                    // Anyway, even if the original rule is invalid,
                    // we would get duplicate errors if we didn't override this,
                    // which is probably a bad idea in and of itself
                    validate() {
                        return [];
                    }
                }
                new CopyRule(targetStack, `${core_1.Names.uniqueId(this)}-${id}`, {
                    targets: [target],
                    eventPattern: this.eventPattern,
                    schedule: this.scheduleExpression ? schedule_1.Schedule.expression(this.scheduleExpression) : undefined,
                    description: this.description,
                });
                return;
            }
        }
        this.targets.push({
            id,
            arn: targetProps.arn,
            roleArn,
            ecsParameters: targetProps.ecsParameters,
            kinesisParameters: targetProps.kinesisParameters,
            runCommandParameters: targetProps.runCommandParameters,
            batchParameters: targetProps.batchParameters,
            deadLetterConfig: targetProps.deadLetterConfig,
            retryPolicy: targetProps.retryPolicy,
            sqsParameters: targetProps.sqsParameters,
            httpParameters: targetProps.httpParameters,
            input: inputProps && inputProps.input,
            inputPath: inputProps && inputProps.inputPath,
            inputTransformer: (inputProps === null || inputProps === void 0 ? void 0 : inputProps.inputTemplate) !== undefined ? {
                inputTemplate: inputProps.inputTemplate,
                inputPathsMap: inputProps.inputPathsMap,
            } : undefined,
        });
    }
    /**
     * (experimental) Adds an event pattern filter to this rule.
     *
     * If a pattern was already specified,
     * these values are merged into the existing pattern.
     *
     * For example, if the rule already contains the pattern:
     *
     *     {
     *       "resources": [ "r1" ],
     *       "detail": {
     *         "hello": [ 1 ]
     *       }
     *     }
     *
     * And `addEventPattern` is called with the pattern:
     *
     *     {
     *       "resources": [ "r2" ],
     *       "detail": {
     *         "foo": [ "bar" ]
     *       }
     *     }
     *
     * The resulting event pattern will be:
     *
     *     {
     *       "resources": [ "r1", "r2" ],
     *       "detail": {
     *         "hello": [ 1 ],
     *         "foo": [ "bar" ]
     *       }
     *     }
     *
     * @experimental
     */
    addEventPattern(eventPattern) {
        if (!eventPattern) {
            return;
        }
        util_1.mergeEventPattern(this.eventPattern, eventPattern);
    }
    /**
     * Not private only to be overrideen in CopyRule.
     *
     * @internal
     */
    _renderEventPattern() {
        const eventPattern = this.eventPattern;
        if (Object.keys(eventPattern).length === 0) {
            return undefined;
        }
        // rename 'detailType' to 'detail-type'
        const out = {};
        for (let key of Object.keys(eventPattern)) {
            const value = eventPattern[key];
            if (key === 'detailType') {
                key = 'detail-type';
            }
            out[key] = value;
        }
        return out;
    }
    /**
     * (experimental) Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @experimental
     */
    validate() {
        if (Object.keys(this.eventPattern).length === 0 && !this.scheduleExpression) {
            return ['Either \'eventPattern\' or \'schedule\' must be defined'];
        }
        return [];
    }
    renderTargets() {
        if (this.targets.length === 0) {
            return undefined;
        }
        return this.targets;
    }
}
exports.Rule = Rule;
_a = JSII_RTTI_SYMBOL_1;
Rule[_a] = { fqn: "monocdk.aws_events.Rule", version: "1.106.1" };
//# sourceMappingURL=data:application/json;base64,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