"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.addToDeadLetterQueueResourcePolicy = exports.addLambdaPermission = exports.singletonEventRole = exports.bindBaseTargetConfig = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_2 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * Bind props to base rule target config.
 * @internal
 */
function bindBaseTargetConfig(props) {
    let { deadLetterQueue, retryAttempts, maxEventAge } = props;
    return {
        deadLetterConfig: deadLetterQueue ? { arn: deadLetterQueue === null || deadLetterQueue === void 0 ? void 0 : deadLetterQueue.queueArn } : undefined,
        retryPolicy: retryAttempts || maxEventAge
            ? {
                maximumRetryAttempts: retryAttempts,
                maximumEventAgeInSeconds: maxEventAge === null || maxEventAge === void 0 ? void 0 : maxEventAge.toSeconds({ integral: true }),
            }
            : undefined,
    };
}
exports.bindBaseTargetConfig = bindBaseTargetConfig;
/**
 * Obtain the Role for the EventBridge event
 *
 * If a role already exists, it will be returned. This ensures that if multiple
 * events have the same target, they will share a role.
 * @internal
 */
function singletonEventRole(scope, policyStatements) {
    const id = 'EventsRole';
    const existing = scope.node.tryFindChild(id);
    if (existing) {
        return existing;
    }
    const role = new iam.Role(scope, id, {
        assumedBy: new iam.ServicePrincipal('events.amazonaws.com'),
    });
    policyStatements.forEach(role.addToPolicy.bind(role));
    return role;
}
exports.singletonEventRole = singletonEventRole;
/**
 * Allows a Lambda function to be called from a rule
 * @internal
 */
function addLambdaPermission(rule, handler) {
    let scope;
    let node = handler.permissionsNode;
    let permissionId = `AllowEventRule${core_1.Names.nodeUniqueId(rule.node)}`;
    if (rule instanceof core_2.Construct) {
        // Place the Permission resource in the same stack as Rule rather than the Function
        // This is to reduce circular dependency when the lambda handler and the rule are across stacks.
        scope = rule;
        node = rule.node;
        permissionId = `AllowEventRule${core_1.Names.nodeUniqueId(handler.node)}`;
    }
    if (!node.tryFindChild(permissionId)) {
        handler.addPermission(permissionId, {
            scope,
            action: 'lambda:InvokeFunction',
            principal: new iam.ServicePrincipal('events.amazonaws.com'),
            sourceArn: rule.ruleArn,
        });
    }
}
exports.addLambdaPermission = addLambdaPermission;
/**
 * Allow a rule to send events with failed invocation to an Amazon SQS queue.
 * @internal
 */
function addToDeadLetterQueueResourcePolicy(rule, queue) {
    if (!sameEnvDimension(rule.env.region, queue.env.region)) {
        throw new Error(`Cannot assign Dead Letter Queue in region ${queue.env.region} to the rule ${core_1.Names.nodeUniqueId(rule.node)} in region ${rule.env.region}. Both the queue and the rule must be in the same region.`);
    }
    // Skip Resource Policy creation if the Queue is not in the same account.
    // There is no way to add a target onto an imported rule, so we can assume we will run the following code only
    // in the account where the rule is created.
    if (sameEnvDimension(rule.env.account, queue.env.account)) {
        const policyStatementId = `AllowEventRule${core_1.Names.nodeUniqueId(rule.node)}`;
        queue.addToResourcePolicy(new iam.PolicyStatement({
            sid: policyStatementId,
            principals: [new iam.ServicePrincipal('events.amazonaws.com')],
            effect: iam.Effect.ALLOW,
            actions: ['sqs:SendMessage'],
            resources: [queue.queueArn],
            conditions: {
                ArnEquals: {
                    'aws:SourceArn': rule.ruleArn,
                },
            },
        }));
    }
    else {
        core_1.Annotations.of(rule).addWarning(`Cannot add a resource policy to your dead letter queue associated with rule ${rule.ruleName} because the queue is in a different account. You must add the resource policy manually to the dead letter queue in account ${queue.env.account}.`);
    }
}
exports.addToDeadLetterQueueResourcePolicy = addToDeadLetterQueueResourcePolicy;
/**
 * Whether two string probably contain the same environment dimension (region or account)
 *
 * Used to compare either accounts or regions, and also returns true if both
 * are unresolved (in which case both are expted to be "current region" or "current account").
 * @internal
 */
function sameEnvDimension(dim1, dim2) {
    return [core_1.TokenComparison.SAME, core_1.TokenComparison.BOTH_UNRESOLVED].includes(core_1.Token.compareStrings(dim1, dim2));
}
//# sourceMappingURL=data:application/json;base64,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