"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.mapTagMapToCxschema = exports.validateNetworkProtocol = exports.ifUndefined = exports.determineProtocolAndPort = exports.defaultProtocolForPort = exports.defaultPortForProtocol = exports.renderAttributes = void 0;
const enums_1 = require("./enums");
/**
 * Render an attribute dict to a list of { key, value } pairs
 */
function renderAttributes(attributes) {
    const ret = [];
    for (const [key, value] of Object.entries(attributes)) {
        if (value !== undefined) {
            ret.push({ key, value });
        }
    }
    return ret;
}
exports.renderAttributes = renderAttributes;
/**
 * Return the appropriate default port for a given protocol
 */
function defaultPortForProtocol(proto) {
    switch (proto) {
        case enums_1.ApplicationProtocol.HTTP: return 80;
        case enums_1.ApplicationProtocol.HTTPS: return 443;
        default:
            throw new Error(`Unrecognized protocol: ${proto}`);
    }
}
exports.defaultPortForProtocol = defaultPortForProtocol;
/**
 * Return the appropriate default protocol for a given port
 */
function defaultProtocolForPort(port) {
    switch (port) {
        case 80:
        case 8000:
        case 8008:
        case 8080:
            return enums_1.ApplicationProtocol.HTTP;
        case 443:
        case 8443:
            return enums_1.ApplicationProtocol.HTTPS;
        default:
            throw new Error(`Don't know default protocol for port: ${port}; please supply a protocol`);
    }
}
exports.defaultProtocolForPort = defaultProtocolForPort;
/**
 * Given a protocol and a port, try to guess the other one if it's undefined
 */
// eslint-disable-next-line max-len
function determineProtocolAndPort(protocol, port) {
    if (protocol === undefined && port === undefined) {
        return [undefined, undefined];
    }
    if (protocol === undefined) {
        protocol = defaultProtocolForPort(port);
    }
    if (port === undefined) {
        port = defaultPortForProtocol(protocol);
    }
    return [protocol, port];
}
exports.determineProtocolAndPort = determineProtocolAndPort;
/**
 * Helper function to default undefined input props
 */
function ifUndefined(x, def) {
    return x !== null && x !== void 0 ? x : def;
}
exports.ifUndefined = ifUndefined;
/**
 * Helper function for ensuring network listeners and target groups only accept valid
 * protocols.
 */
function validateNetworkProtocol(protocol) {
    const NLB_PROTOCOLS = [enums_1.Protocol.TCP, enums_1.Protocol.TLS, enums_1.Protocol.UDP, enums_1.Protocol.TCP_UDP];
    if (NLB_PROTOCOLS.indexOf(protocol) === -1) {
        throw new Error(`The protocol must be one of ${NLB_PROTOCOLS.join(', ')}. Found ${protocol}`);
    }
}
exports.validateNetworkProtocol = validateNetworkProtocol;
/**
 * Helper to map a map of tags to cxschema tag format.
 * @internal
 */
function mapTagMapToCxschema(tagMap) {
    return Object.entries(tagMap)
        .map(([key, value]) => ({ key, value }));
}
exports.mapTagMapToCxschema = mapTagMapToCxschema;
//# sourceMappingURL=data:application/json;base64,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