"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NetworkTargetGroup = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("../../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const base_target_group_1 = require("../shared/base-target-group");
const enums_1 = require("../shared/enums");
const imported_1 = require("../shared/imported");
const util_1 = require("../shared/util");
/**
 * (experimental) Define a Network Target Group.
 *
 * @experimental
 */
class NetworkTargetGroup extends base_target_group_1.TargetGroupBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        const proto = props.protocol || enums_1.Protocol.TCP;
        util_1.validateNetworkProtocol(proto);
        super(scope, id, props, {
            protocol: proto,
            port: props.port,
        });
        this.listeners = [];
        if (props.proxyProtocolV2 != null) {
            this.setAttribute('proxy_protocol_v2.enabled', props.proxyProtocolV2 ? 'true' : 'false');
        }
        if (props.preserveClientIp !== undefined) {
            this.setAttribute('preserve_client_ip.enabled', props.preserveClientIp ? 'true' : 'false');
        }
        this.addTarget(...(props.targets || []));
    }
    /**
     * (experimental) Import an existing target group.
     *
     * @experimental
     */
    static fromTargetGroupAttributes(scope, id, attrs) {
        return new ImportedNetworkTargetGroup(scope, id, attrs);
    }
    /**
     * (deprecated) Import an existing listener.
     *
     * @deprecated Use `fromTargetGroupAttributes` instead
     */
    static import(scope, id, props) {
        return NetworkTargetGroup.fromTargetGroupAttributes(scope, id, props);
    }
    /**
     * (experimental) Add a load balancing target to this target group.
     *
     * @experimental
     */
    addTarget(...targets) {
        for (const target of targets) {
            const result = target.attachToNetworkTargetGroup(this);
            this.addLoadBalancerTarget(result);
        }
    }
    /**
     * (experimental) Register a listener that is load balancing to this target group.
     *
     * Don't call this directly. It will be called by listeners.
     *
     * @experimental
     */
    registerListener(listener) {
        this.loadBalancerAttachedDependencies.add(listener);
        this.listeners.push(listener);
    }
    /**
     * (experimental) The number of targets that are considered healthy.
     *
     * @default Average over 5 minutes
     * @experimental
     */
    metricHealthyHostCount(props) {
        return this.metric('HealthyHostCount', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * (experimental) The number of targets that are considered unhealthy.
     *
     * @default Average over 5 minutes
     * @experimental
     */
    metricUnHealthyHostCount(props) {
        return this.metric('UnHealthyHostCount', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * (experimental) Full name of first load balancer.
     *
     * @experimental
     */
    get firstLoadBalancerFullName() {
        if (this.listeners.length === 0) {
            throw new Error('The TargetGroup needs to be attached to a LoadBalancer before you can call this method');
        }
        return base_target_group_1.loadBalancerNameFromListenerArn(this.listeners[0].listenerArn);
    }
    /**
     * (experimental) Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @experimental
     */
    validate() {
        const ret = super.validate();
        const healthCheck = this.healthCheck || {};
        const allowedIntervals = [10, 30];
        if (healthCheck.interval) {
            const seconds = healthCheck.interval.toSeconds();
            if (!cdk.Token.isUnresolved(seconds) && !allowedIntervals.includes(seconds)) {
                ret.push(`Health check interval '${seconds}' not supported. Must be one of the following values '${allowedIntervals.join(',')}'.`);
            }
        }
        if (healthCheck.healthyThresholdCount) {
            const thresholdCount = healthCheck.healthyThresholdCount;
            if (thresholdCount < 2 || thresholdCount > 10) {
                ret.push(`Healthy Threshold Count '${thresholdCount}' not supported. Must be a number between 2 and 10.`);
            }
        }
        if (healthCheck.unhealthyThresholdCount) {
            const thresholdCount = healthCheck.unhealthyThresholdCount;
            if (thresholdCount < 2 || thresholdCount > 10) {
                ret.push(`Unhealthy Threshold Count '${thresholdCount}' not supported. Must be a number between 2 and 10.`);
            }
        }
        if (healthCheck.healthyThresholdCount && healthCheck.unhealthyThresholdCount &&
            healthCheck.healthyThresholdCount !== healthCheck.unhealthyThresholdCount) {
            ret.push([
                `Healthy and Unhealthy Threshold Counts must be the same: ${healthCheck.healthyThresholdCount}`,
                `is not equal to ${healthCheck.unhealthyThresholdCount}.`,
            ].join(' '));
        }
        if (!healthCheck.protocol) {
            return ret;
        }
        if (!NLB_HEALTH_CHECK_PROTOCOLS.includes(healthCheck.protocol)) {
            ret.push(`Health check protocol '${healthCheck.protocol}' is not supported. Must be one of [${NLB_HEALTH_CHECK_PROTOCOLS.join(', ')}]`);
        }
        if (healthCheck.path && !NLB_PATH_HEALTH_CHECK_PROTOCOLS.includes(healthCheck.protocol)) {
            ret.push([
                `'${healthCheck.protocol}' health checks do not support the path property.`,
                `Must be one of [${NLB_PATH_HEALTH_CHECK_PROTOCOLS.join(', ')}]`,
            ].join(' '));
        }
        if (healthCheck.timeout && healthCheck.timeout.toSeconds() !== NLB_HEALTH_CHECK_TIMEOUTS[healthCheck.protocol]) {
            ret.push([
                'Custom health check timeouts are not supported for Network Load Balancer health checks.',
                `Expected ${NLB_HEALTH_CHECK_TIMEOUTS[healthCheck.protocol]} seconds for ${healthCheck.protocol}, got ${healthCheck.timeout.toSeconds()}`,
            ].join(' '));
        }
        return ret;
    }
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/NetworkELB',
            metricName,
            dimensions: { LoadBalancer: this.firstLoadBalancerFullName, TargetGroup: this.targetGroupFullName },
            ...props,
        }).attachTo(this);
    }
}
exports.NetworkTargetGroup = NetworkTargetGroup;
_a = JSII_RTTI_SYMBOL_1;
NetworkTargetGroup[_a] = { fqn: "monocdk.aws_elasticloadbalancingv2.NetworkTargetGroup", version: "1.106.1" };
/**
 * An imported network target group
 */
class ImportedNetworkTargetGroup extends imported_1.ImportedTargetGroupBase {
    registerListener(_listener) {
        // Nothing to do, we know nothing of our members
    }
    addTarget(...targets) {
        for (const target of targets) {
            const result = target.attachToNetworkTargetGroup(this);
            if (result.targetJson !== undefined) {
                throw new Error('Cannot add a non-self registering target to an imported TargetGroup. Create a new TargetGroup instead.');
            }
        }
    }
}
const NLB_HEALTH_CHECK_PROTOCOLS = [enums_1.Protocol.HTTP, enums_1.Protocol.HTTPS, enums_1.Protocol.TCP];
const NLB_PATH_HEALTH_CHECK_PROTOCOLS = [enums_1.Protocol.HTTP, enums_1.Protocol.HTTPS];
const NLB_HEALTH_CHECK_TIMEOUTS = {
    [enums_1.Protocol.HTTP]: 6,
    [enums_1.Protocol.HTTPS]: 10,
    [enums_1.Protocol.TCP]: 10,
};
//# sourceMappingURL=data:application/json;base64,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