"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApplicationListener = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("../../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const cxschema = require("../../../cloud-assembly-schema"); // Automatically re-written from '@aws-cdk/cloud-assembly-schema'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const base_listener_1 = require("../shared/base-listener");
const enums_1 = require("../shared/enums");
const listener_certificate_1 = require("../shared/listener-certificate");
const util_1 = require("../shared/util");
const application_listener_action_1 = require("./application-listener-action");
const application_listener_certificate_1 = require("./application-listener-certificate");
const application_listener_rule_1 = require("./application-listener-rule");
const application_target_group_1 = require("./application-target-group");
/**
 * (experimental) Define an ApplicationListener.
 *
 * @experimental
 * @resource AWS::ElasticLoadBalancingV2::Listener
 */
class ApplicationListener extends base_listener_1.BaseListener {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        const [protocol, port] = util_1.determineProtocolAndPort(props.protocol, props.port);
        if (protocol === undefined || port === undefined) {
            throw new Error('At least one of \'port\' or \'protocol\' is required');
        }
        super(scope, id, {
            loadBalancerArn: props.loadBalancer.loadBalancerArn,
            certificates: core_1.Lazy.any({ produce: () => this.certificateArns.map(certificateArn => ({ certificateArn })) }, { omitEmptyArray: true }),
            protocol,
            port,
            sslPolicy: props.sslPolicy,
        });
        this.loadBalancer = props.loadBalancer;
        this.protocol = protocol;
        this.certificateArns = [];
        // Attach certificates
        if (props.certificateArns && props.certificateArns.length > 0) {
            this.addCertificateArns('ListenerCertificate', props.certificateArns);
        }
        if (props.certificates && props.certificates.length > 0) {
            this.addCertificates('DefaultCertificates', props.certificates);
        }
        // This listener edits the securitygroup of the load balancer,
        // but adds its own default port.
        this.connections = new ec2.Connections({
            securityGroups: props.loadBalancer.connections.securityGroups,
            defaultPort: ec2.Port.tcp(port),
        });
        if (props.defaultAction && props.defaultTargetGroups) {
            throw new Error('Specify at most one of \'defaultAction\' and \'defaultTargetGroups\'');
        }
        if (props.defaultAction) {
            this.setDefaultAction(props.defaultAction);
        }
        if (props.defaultTargetGroups) {
            this.setDefaultAction(application_listener_action_1.ListenerAction.forward(props.defaultTargetGroups));
        }
        if (props.open !== false) {
            this.connections.allowDefaultPortFrom(ec2.Peer.anyIpv4(), `Allow from anyone on port ${port}`);
            if (this.loadBalancer.ipAddressType === enums_1.IpAddressType.DUAL_STACK) {
                this.connections.allowDefaultPortFrom(ec2.Peer.anyIpv6(), `Allow from anyone on port ${port}`);
            }
        }
    }
    /**
     * (experimental) Look up an ApplicationListener.
     *
     * @experimental
     */
    static fromLookup(scope, id, options) {
        if (core_1.Token.isUnresolved(options.listenerArn)) {
            throw new Error('All arguments to look up a load balancer listener must be concrete (no Tokens)');
        }
        let listenerProtocol;
        switch (options.listenerProtocol) {
            case enums_1.ApplicationProtocol.HTTP:
                listenerProtocol = cxschema.LoadBalancerListenerProtocol.HTTP;
                break;
            case enums_1.ApplicationProtocol.HTTPS:
                listenerProtocol = cxschema.LoadBalancerListenerProtocol.HTTPS;
                break;
        }
        const props = base_listener_1.BaseListener._queryContextProvider(scope, {
            userOptions: options,
            loadBalancerType: cxschema.LoadBalancerType.APPLICATION,
            listenerArn: options.listenerArn,
            listenerProtocol,
        });
        return new LookedUpApplicationListener(scope, id, props);
    }
    /**
     * (experimental) Import an existing listener.
     *
     * @experimental
     */
    static fromApplicationListenerAttributes(scope, id, attrs) {
        return new ImportedApplicationListener(scope, id, attrs);
    }
    /**
     * (deprecated) Add one or more certificates to this listener.
     *
     * After the first certificate, this creates ApplicationListenerCertificates
     * resources since cloudformation requires the certificates array on the
     * listener resource to have a length of 1.
     *
     * @deprecated Use `addCertificates` instead.
     */
    addCertificateArns(id, arns) {
        this.addCertificates(id, arns.map(listener_certificate_1.ListenerCertificate.fromArn));
    }
    /**
     * (experimental) Add one or more certificates to this listener.
     *
     * After the first certificate, this creates ApplicationListenerCertificates
     * resources since cloudformation requires the certificates array on the
     * listener resource to have a length of 1.
     *
     * @experimental
     */
    addCertificates(id, certificates) {
        const additionalCerts = [...certificates];
        if (this.certificateArns.length === 0 && additionalCerts.length > 0) {
            const first = additionalCerts.splice(0, 1)[0];
            this.certificateArns.push(first.certificateArn);
        }
        // Only one certificate can be specified per resource, even though
        // `certificates` is of type Array
        for (let i = 0; i < additionalCerts.length; i++) {
            new application_listener_certificate_1.ApplicationListenerCertificate(this, `${id}${i + 1}`, {
                listener: this,
                certificates: [additionalCerts[i]],
            });
        }
    }
    /**
     * (experimental) Perform the given default action on incoming requests.
     *
     * This allows full control of the default action of the load balancer,
     * including Action chaining, fixed responses and redirect responses. See
     * the `ListenerAction` class for all options.
     *
     * It's possible to add routing conditions to the Action added in this way.
     * At least one Action must be added without conditions (which becomes the
     * default Action).
     *
     * @experimental
     */
    addAction(id, props) {
        checkAddRuleProps(props);
        if (props.priority !== undefined) {
            // New rule
            //
            // TargetGroup.registerListener is called inside ApplicationListenerRule.
            new application_listener_rule_1.ApplicationListenerRule(this, id + 'Rule', {
                listener: this,
                conditions: props.conditions,
                hostHeader: props.hostHeader,
                pathPattern: props.pathPattern,
                pathPatterns: props.pathPatterns,
                priority: props.priority,
                action: props.action,
            });
        }
        else {
            // New default target with these targetgroups
            this.setDefaultAction(props.action);
        }
    }
    /**
     * (experimental) Load balance incoming requests to the given target groups.
     *
     * All target groups will be load balanced to with equal weight and without
     * stickiness. For a more complex configuration than that, use `addAction()`.
     *
     * It's possible to add routing conditions to the TargetGroups added in this
     * way. At least one TargetGroup must be added without conditions (which will
     * become the default Action for this listener).
     *
     * @experimental
     */
    addTargetGroups(id, props) {
        checkAddRuleProps(props);
        if (props.priority !== undefined) {
            // New rule
            //
            // TargetGroup.registerListener is called inside ApplicationListenerRule.
            new application_listener_rule_1.ApplicationListenerRule(this, id + 'Rule', {
                listener: this,
                conditions: props.conditions,
                hostHeader: props.hostHeader,
                pathPattern: props.pathPattern,
                pathPatterns: props.pathPatterns,
                priority: props.priority,
                targetGroups: props.targetGroups,
            });
        }
        else {
            // New default target with these targetgroups
            this.setDefaultAction(application_listener_action_1.ListenerAction.forward(props.targetGroups));
        }
    }
    /**
     * (experimental) Load balance incoming requests to the given load balancing targets.
     *
     * This method implicitly creates an ApplicationTargetGroup for the targets
     * involved, and a 'forward' action to route traffic to the given TargetGroup.
     *
     * If you want more control over the precise setup, create the TargetGroup
     * and use `addAction` yourself.
     *
     * It's possible to add conditions to the targets added in this way. At least
     * one set of targets must be added without conditions.
     *
     * @returns The newly created target group
     * @experimental
     */
    addTargets(id, props) {
        if (!this.loadBalancer.vpc) {
            // eslint-disable-next-line max-len
            throw new Error('Can only call addTargets() when using a constructed Load Balancer or an imported Load Balancer with specified vpc; construct a new TargetGroup and use addTargetGroup');
        }
        const group = new application_target_group_1.ApplicationTargetGroup(this, id + 'Group', {
            deregistrationDelay: props.deregistrationDelay,
            healthCheck: props.healthCheck,
            port: props.port,
            protocol: props.protocol,
            protocolVersion: props.protocolVersion,
            slowStart: props.slowStart,
            stickinessCookieDuration: props.stickinessCookieDuration,
            stickinessCookieName: props.stickinessCookieName,
            targetGroupName: props.targetGroupName,
            targets: props.targets,
            vpc: this.loadBalancer.vpc,
        });
        this.addTargetGroups(id, {
            conditions: props.conditions,
            hostHeader: props.hostHeader,
            pathPattern: props.pathPattern,
            pathPatterns: props.pathPatterns,
            priority: props.priority,
            targetGroups: [group],
        });
        return group;
    }
    /**
     * (deprecated) Add a fixed response.
     *
     * @deprecated Use `addAction()` instead
     */
    addFixedResponse(id, props) {
        checkAddRuleProps(props);
        const fixedResponse = {
            statusCode: props.statusCode,
            contentType: props.contentType,
            messageBody: props.messageBody,
        };
        application_listener_rule_1.validateFixedResponse(fixedResponse);
        if (props.priority) {
            new application_listener_rule_1.ApplicationListenerRule(this, id + 'Rule', {
                listener: this,
                priority: props.priority,
                fixedResponse,
                ...props,
            });
        }
        else {
            this.setDefaultAction(application_listener_action_1.ListenerAction.fixedResponse(core_1.Token.asNumber(props.statusCode), {
                contentType: props.contentType,
                messageBody: props.messageBody,
            }));
        }
    }
    /**
     * (deprecated) Add a redirect response.
     *
     * @deprecated Use `addAction()` instead
     */
    addRedirectResponse(id, props) {
        checkAddRuleProps(props);
        const redirectResponse = {
            host: props.host,
            path: props.path,
            port: props.port,
            protocol: props.protocol,
            query: props.query,
            statusCode: props.statusCode,
        };
        application_listener_rule_1.validateRedirectResponse(redirectResponse);
        if (props.priority) {
            new application_listener_rule_1.ApplicationListenerRule(this, id + 'Rule', {
                listener: this,
                priority: props.priority,
                redirectResponse,
                ...props,
            });
        }
        else {
            this.setDefaultAction(application_listener_action_1.ListenerAction.redirect({
                host: props.host,
                path: props.path,
                port: props.port,
                protocol: props.protocol,
                query: props.query,
                permanent: props.statusCode === 'HTTP_301',
            }));
        }
    }
    /**
     * (experimental) Register that a connectable that has been added to this load balancer.
     *
     * Don't call this directly. It is called by ApplicationTargetGroup.
     *
     * @experimental
     */
    registerConnectable(connectable, portRange) {
        connectable.connections.allowFrom(this.loadBalancer, portRange, 'Load balancer to target');
    }
    /**
     * (experimental) Validate this listener.
     *
     * @experimental
     */
    validate() {
        const errors = super.validate();
        if (this.protocol === enums_1.ApplicationProtocol.HTTPS && this.certificateArns.length === 0) {
            errors.push('HTTPS Listener needs at least one certificate (call addCertificates)');
        }
        return errors;
    }
    /**
     * Wrapper for _setDefaultAction which does a type-safe bind
     */
    setDefaultAction(action) {
        action.bind(this, this);
        this._setDefaultAction(action);
    }
}
exports.ApplicationListener = ApplicationListener;
_a = JSII_RTTI_SYMBOL_1;
ApplicationListener[_a] = { fqn: "monocdk.aws_elasticloadbalancingv2.ApplicationListener", version: "1.106.1" };
class ExternalApplicationListener extends core_1.Resource {
    constructor(scope, id) {
        super(scope, id);
    }
    /**
     * Register that a connectable that has been added to this load balancer.
     *
     * Don't call this directly. It is called by ApplicationTargetGroup.
     */
    registerConnectable(connectable, portRange) {
        this.connections.allowTo(connectable, portRange, 'Load balancer to target');
    }
    /**
     * Add one or more certificates to this listener.
     * @deprecated use `addCertificates()`
     */
    addCertificateArns(id, arns) {
        this.addCertificates(id, arns.map(listener_certificate_1.ListenerCertificate.fromArn));
    }
    /**
     * Add one or more certificates to this listener.
     */
    addCertificates(id, certificates) {
        const arns = certificates.map(c => c.certificateArn);
        new application_listener_certificate_1.ApplicationListenerCertificate(this, id, {
            listener: this,
            certificateArns: arns,
        });
    }
    /**
     * Load balance incoming requests to the given target groups.
     *
     * It's possible to add conditions to the TargetGroups added in this way.
     * At least one TargetGroup must be added without conditions.
     */
    addTargetGroups(id, props) {
        checkAddRuleProps(props);
        if (props.priority !== undefined) {
            // New rule
            new application_listener_rule_1.ApplicationListenerRule(this, id, {
                listener: this,
                conditions: props.conditions,
                hostHeader: props.hostHeader,
                pathPattern: props.pathPattern,
                pathPatterns: props.pathPatterns,
                priority: props.priority,
                targetGroups: props.targetGroups,
            });
        }
        else {
            throw new Error('Cannot add default Target Groups to imported ApplicationListener');
        }
    }
    /**
     * Load balance incoming requests to the given load balancing targets.
     *
     * This method implicitly creates an ApplicationTargetGroup for the targets
     * involved.
     *
     * It's possible to add conditions to the targets added in this way. At least
     * one set of targets must be added without conditions.
     *
     * @returns The newly created target group
     */
    addTargets(_id, _props) {
        // eslint-disable-next-line max-len
        throw new Error('Can only call addTargets() when using a constructed ApplicationListener; construct a new TargetGroup and use addTargetGroup.');
    }
}
/**
 * An imported application listener.
 */
class ImportedApplicationListener extends ExternalApplicationListener {
    constructor(scope, id, props) {
        super(scope, id);
        this.listenerArn = props.listenerArn;
        const defaultPort = props.defaultPort !== undefined ? ec2.Port.tcp(props.defaultPort) : undefined;
        let securityGroup;
        if (props.securityGroup) {
            securityGroup = props.securityGroup;
        }
        else if (props.securityGroupId) {
            securityGroup = ec2.SecurityGroup.fromSecurityGroupId(this, 'SecurityGroup', props.securityGroupId, {
                allowAllOutbound: props.securityGroupAllowsAllOutbound,
            });
        }
        else {
            throw new Error('Either `securityGroup` or `securityGroupId` must be specified to import an application listener.');
        }
        this.connections = new ec2.Connections({
            securityGroups: [securityGroup],
            defaultPort,
        });
    }
}
class LookedUpApplicationListener extends ExternalApplicationListener {
    constructor(scope, id, props) {
        super(scope, id);
        this.listenerArn = props.listenerArn;
        this.connections = new ec2.Connections({
            defaultPort: ec2.Port.tcp(props.listenerPort),
        });
        for (const securityGroupId of props.securityGroupIds) {
            const securityGroup = ec2.SecurityGroup.fromLookup(this, `SecurityGroup-${securityGroupId}`, securityGroupId);
            this.connections.addSecurityGroup(securityGroup);
        }
    }
}
function checkAddRuleProps(props) {
    var _b;
    const conditionsCount = ((_b = props.conditions) === null || _b === void 0 ? void 0 : _b.length) || 0;
    const hasAnyConditions = conditionsCount !== 0 ||
        props.hostHeader !== undefined || props.pathPattern !== undefined || props.pathPatterns !== undefined;
    const hasPriority = props.priority !== undefined;
    if (hasAnyConditions !== hasPriority) {
        throw new Error('Setting \'conditions\', \'pathPattern\' or \'hostHeader\' also requires \'priority\', and vice versa');
    }
}
//# sourceMappingURL=data:application/json;base64,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