"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UnauthenticatedAction = exports.ListenerAction = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * (experimental) What to do when a client makes a request to a listener.
 *
 * Some actions can be combined with other ones (specifically,
 * you can perform authentication before serving the request).
 *
 * Multiple actions form a linked chain; the chain must always terminate in a
 * *(weighted)forward*, *fixedResponse* or *redirect* action.
 *
 * If an action supports chaining, the next action can be indicated
 * by passing it in the `next` property.
 *
 * (Called `ListenerAction` instead of the more strictly correct
 * `ListenerAction` because this is the class most users interact
 * with, and we want to make it not too visually overwhelming).
 *
 * @experimental
 */
class ListenerAction {
    /**
     * (experimental) Create an instance of ListenerAction.
     *
     * The default class should be good enough for most cases and
     * should be created by using one of the static factory functions,
     * but allow overriding to make sure we allow flexibility for the future.
     *
     * @experimental
     */
    constructor(actionJson, next) {
        this.actionJson = actionJson;
        this.next = next;
    }
    /**
     * (experimental) Authenticate using an identity provider (IdP) that is compliant with OpenID Connect (OIDC).
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/listener-authenticate-users.html#oidc-requirements
     * @experimental
     */
    static authenticateOidc(options) {
        var _b;
        return new ListenerAction({
            type: 'authenticate-oidc',
            authenticateOidcConfig: {
                authorizationEndpoint: options.authorizationEndpoint,
                clientId: options.clientId,
                clientSecret: options.clientSecret.toString(),
                issuer: options.issuer,
                tokenEndpoint: options.tokenEndpoint,
                userInfoEndpoint: options.userInfoEndpoint,
                authenticationRequestExtraParams: options.authenticationRequestExtraParams,
                onUnauthenticatedRequest: options.onUnauthenticatedRequest,
                scope: options.scope,
                sessionCookieName: options.sessionCookieName,
                sessionTimeout: (_b = options.sessionTimeout) === null || _b === void 0 ? void 0 : _b.toSeconds().toString(),
            },
        }, options.next);
    }
    /**
     * (experimental) Forward to one or more Target Groups.
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-listeners.html#forward-actions
     * @experimental
     */
    static forward(targetGroups, options = {}) {
        if (targetGroups.length === 0) {
            throw new Error('Need at least one targetGroup in a ListenerAction.forward()');
        }
        if (targetGroups.length === 1 && options.stickinessDuration === undefined) {
            // Render a "simple" action for backwards compatibility with old templates
            return new TargetGroupListenerAction(targetGroups, {
                type: 'forward',
                targetGroupArn: targetGroups[0].targetGroupArn,
            });
        }
        return new TargetGroupListenerAction(targetGroups, {
            type: 'forward',
            forwardConfig: {
                targetGroups: targetGroups.map(g => ({ targetGroupArn: g.targetGroupArn })),
                targetGroupStickinessConfig: options.stickinessDuration ? {
                    durationSeconds: options.stickinessDuration.toSeconds(),
                    enabled: true,
                } : undefined,
            },
        });
    }
    /**
     * (experimental) Forward to one or more Target Groups which are weighted differently.
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-listeners.html#forward-actions
     * @experimental
     */
    static weightedForward(targetGroups, options = {}) {
        if (targetGroups.length === 0) {
            throw new Error('Need at least one targetGroup in a ListenerAction.weightedForward()');
        }
        return new TargetGroupListenerAction(targetGroups.map(g => g.targetGroup), {
            type: 'forward',
            forwardConfig: {
                targetGroups: targetGroups.map(g => ({ targetGroupArn: g.targetGroup.targetGroupArn, weight: g.weight })),
                targetGroupStickinessConfig: options.stickinessDuration ? {
                    durationSeconds: options.stickinessDuration.toSeconds(),
                    enabled: true,
                } : undefined,
            },
        });
    }
    /**
     * (experimental) Return a fixed response.
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-listeners.html#fixed-response-actions
     * @experimental
     */
    static fixedResponse(statusCode, options = {}) {
        return new ListenerAction({
            type: 'fixed-response',
            fixedResponseConfig: {
                statusCode: core_1.Tokenization.stringifyNumber(statusCode),
                contentType: options.contentType,
                messageBody: options.messageBody,
            },
        });
    }
    /**
     * (experimental) Redirect to a different URI.
     *
     * A URI consists of the following components:
     * protocol://hostname:port/path?query. You must modify at least one of the
     * following components to avoid a redirect loop: protocol, hostname, port, or
     * path. Any components that you do not modify retain their original values.
     *
     * You can reuse URI components using the following reserved keywords:
     *
     * - `#{protocol}`
     * - `#{host}`
     * - `#{port}`
     * - `#{path}` (the leading "/" is removed)
     * - `#{query}`
     *
     * For example, you can change the path to "/new/#{path}", the hostname to
     * "example.#{host}", or the query to "#{query}&value=xyz".
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-listeners.html#redirect-actions
     * @experimental
     */
    static redirect(options) {
        if ([options.host, options.path, options.port, options.protocol, options.query].findIndex(x => x !== undefined) === -1) {
            throw new Error('To prevent redirect loops, set at least one of \'protocol\', \'host\', \'port\', \'path\', or \'query\'.');
        }
        return new ListenerAction({
            type: 'redirect',
            redirectConfig: {
                statusCode: options.permanent ? 'HTTP_301' : 'HTTP_302',
                host: options.host,
                path: options.path,
                port: options.port,
                protocol: options.protocol,
                query: options.query,
            },
        });
    }
    /**
     * (experimental) Render the actions in this chain.
     *
     * @experimental
     */
    renderActions() {
        var _b, _c;
        return this.renumber([this.actionJson, ...(_c = (_b = this.next) === null || _b === void 0 ? void 0 : _b.renderActions()) !== null && _c !== void 0 ? _c : []]);
    }
    /**
     * (experimental) Called when the action is being used in a listener.
     *
     * @experimental
     */
    bind(scope, listener, associatingConstruct) {
        // Empty on purpose
        Array.isArray(scope);
        Array.isArray(listener);
        Array.isArray(associatingConstruct);
    }
    /**
     * (experimental) Renumber the "order" fields in the actions array.
     *
     * We don't number for 0 or 1 elements, but otherwise number them 1...#actions
     * so ELB knows about the right order.
     *
     * Do this in `ListenerAction` instead of in `Listener` so that we give
     * users the opportunity to override by subclassing and overriding `renderActions`.
     *
     * @experimental
     */
    renumber(actions) {
        if (actions.length < 2) {
            return actions;
        }
        return actions.map((action, i) => ({ ...action, order: i + 1 }));
    }
}
exports.ListenerAction = ListenerAction;
_a = JSII_RTTI_SYMBOL_1;
ListenerAction[_a] = { fqn: "monocdk.aws_elasticloadbalancingv2.ListenerAction", version: "1.106.1" };
/**
 * (experimental) What to do with unauthenticated requests.
 *
 * @experimental
 */
var UnauthenticatedAction;
(function (UnauthenticatedAction) {
    UnauthenticatedAction["DENY"] = "deny";
    UnauthenticatedAction["ALLOW"] = "allow";
    UnauthenticatedAction["AUTHENTICATE"] = "authenticate";
})(UnauthenticatedAction = exports.UnauthenticatedAction || (exports.UnauthenticatedAction = {}));
/**
 * Listener Action that calls "registerListener" on TargetGroups
 */
class TargetGroupListenerAction extends ListenerAction {
    constructor(targetGroups, actionJson) {
        super(actionJson);
        this.targetGroups = targetGroups;
    }
    bind(_scope, listener, associatingConstruct) {
        for (const tg of this.targetGroups) {
            tg.registerListener(listener, associatingConstruct);
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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