"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ListenerPort = exports.LoadBalancer = exports.LoadBalancingProtocol = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ec2_1 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const elasticloadbalancing_generated_1 = require("./elasticloadbalancing.generated");
/**
 * @experimental
 */
var LoadBalancingProtocol;
(function (LoadBalancingProtocol) {
    LoadBalancingProtocol["TCP"] = "tcp";
    LoadBalancingProtocol["SSL"] = "ssl";
    LoadBalancingProtocol["HTTP"] = "http";
    LoadBalancingProtocol["HTTPS"] = "https";
})(LoadBalancingProtocol = exports.LoadBalancingProtocol || (exports.LoadBalancingProtocol = {}));
/**
 * (experimental) A load balancer with a single listener.
 *
 * Routes to a fleet of of instances in a VPC.
 *
 * @experimental
 */
class LoadBalancer extends core_1.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _c;
        super(scope, id);
        /**
         * (experimental) An object controlling specifically the connections for each listener added to this load balancer.
         *
         * @experimental
         */
        this.listenerPorts = [];
        this.listeners = [];
        this.instancePorts = [];
        this.targets = [];
        this.securityGroup = new aws_ec2_1.SecurityGroup(this, 'SecurityGroup', { vpc: props.vpc, allowAllOutbound: false });
        this.connections = new aws_ec2_1.Connections({ securityGroups: [this.securityGroup] });
        // Depending on whether the ELB has public or internal IPs, pick the right backend subnets
        const selectedSubnets = loadBalancerSubnets(props);
        this.elb = new elasticloadbalancing_generated_1.CfnLoadBalancer(this, 'Resource', {
            securityGroups: [this.securityGroup.securityGroupId],
            subnets: selectedSubnets.subnetIds,
            listeners: core_1.Lazy.any({ produce: () => this.listeners }),
            scheme: props.internetFacing ? 'internet-facing' : 'internal',
            healthCheck: props.healthCheck && healthCheckToJSON(props.healthCheck),
            crossZone: (_c = props.crossZone) !== null && _c !== void 0 ? _c : true,
        });
        if (props.internetFacing) {
            this.elb.node.addDependency(selectedSubnets.internetConnectivityEstablished);
        }
        ifUndefined(props.listeners, []).forEach(b => this.addListener(b));
        ifUndefined(props.targets, []).forEach(t => this.addTarget(t));
    }
    /**
     * (experimental) Add a backend to the load balancer.
     *
     * @returns A ListenerPort object that controls connections to the listener port
     * @experimental
     */
    addListener(listener) {
        if (listener.sslCertificateArn && listener.sslCertificateId) {
            throw new Error('"sslCertificateId" is deprecated, please use "sslCertificateArn" only.');
        }
        const protocol = ifUndefinedLazy(listener.externalProtocol, () => wellKnownProtocol(listener.externalPort));
        const instancePort = listener.internalPort || listener.externalPort;
        const sslCertificateArn = listener.sslCertificateArn || listener.sslCertificateId;
        const instanceProtocol = ifUndefined(listener.internalProtocol, ifUndefined(tryWellKnownProtocol(instancePort), isHttpProtocol(protocol) ? LoadBalancingProtocol.HTTP : LoadBalancingProtocol.TCP));
        this.listeners.push({
            loadBalancerPort: listener.externalPort.toString(),
            protocol,
            instancePort: instancePort.toString(),
            instanceProtocol,
            sslCertificateId: sslCertificateArn,
            policyNames: listener.policyNames,
        });
        const port = new ListenerPort(this.securityGroup, aws_ec2_1.Port.tcp(listener.externalPort));
        // Allow connections on the public port for all supplied peers (default: everyone)
        ifUndefined(listener.allowConnectionsFrom, [aws_ec2_1.Peer.anyIpv4()]).forEach(peer => {
            port.connections.allowDefaultPortFrom(peer, `Default rule allow on ${listener.externalPort}`);
        });
        this.newInstancePort(instancePort);
        // Keep track using array so user can get to them even if they were all supplied in the constructor
        this.listenerPorts.push(port);
        return port;
    }
    /**
     * @experimental
     */
    addTarget(target) {
        target.attachToClassicLB(this);
        this.newTarget(target);
    }
    /**
     * @experimental
     * @attribute true
     */
    get loadBalancerName() {
        return this.elb.ref;
    }
    /**
     * @experimental
     * @attribute true
     */
    get loadBalancerCanonicalHostedZoneNameId() {
        return this.elb.attrCanonicalHostedZoneNameId;
    }
    /**
     * @experimental
     * @attribute true
     */
    get loadBalancerCanonicalHostedZoneName() {
        return this.elb.attrCanonicalHostedZoneName;
    }
    /**
     * @experimental
     * @attribute true
     */
    get loadBalancerDnsName() {
        return this.elb.attrDnsName;
    }
    /**
     * @experimental
     * @attribute true
     */
    get loadBalancerSourceSecurityGroupGroupName() {
        return this.elb.attrSourceSecurityGroupGroupName;
    }
    /**
     * @experimental
     * @attribute true
     */
    get loadBalancerSourceSecurityGroupOwnerAlias() {
        return this.elb.attrSourceSecurityGroupOwnerAlias;
    }
    /**
     * Allow connections to all existing targets on new instance port
     */
    newInstancePort(instancePort) {
        this.targets.forEach(t => this.allowTargetConnection(instancePort, t));
        // Keep track of port for future targets
        this.instancePorts.push(instancePort);
    }
    /**
     * Allow connections to target on all existing instance ports
     */
    newTarget(target) {
        this.instancePorts.forEach(p => this.allowTargetConnection(p, target));
        // Keep track of target for future listeners.
        this.targets.push(target);
    }
    /**
     * Allow connections for a single (port, target) pair
     */
    allowTargetConnection(instancePort, target) {
        this.connections.allowTo(target, aws_ec2_1.Port.tcp(instancePort), `Port ${instancePort} LB to fleet`);
    }
}
exports.LoadBalancer = LoadBalancer;
_a = JSII_RTTI_SYMBOL_1;
LoadBalancer[_a] = { fqn: "monocdk.aws_elasticloadbalancing.LoadBalancer", version: "1.106.1" };
/**
 * (experimental) Reference to a listener's port just created.
 *
 * This implements IConnectable with a default port (the port that an ELB
 * listener was just created on) for a given security group so that it can be
 * conveniently used just like any Connectable. E.g:
 *
 *     const listener = elb.addListener(...);
 *
 *     listener.connections.allowDefaultPortFromAnyIPv4();
 *     // or
 *     instance.connections.allowToDefaultPort(listener);
 *
 * @experimental
 */
class ListenerPort {
    /**
     * @experimental
     */
    constructor(securityGroup, defaultPort) {
        this.connections = new aws_ec2_1.Connections({ securityGroups: [securityGroup], defaultPort });
    }
}
exports.ListenerPort = ListenerPort;
_b = JSII_RTTI_SYMBOL_1;
ListenerPort[_b] = { fqn: "monocdk.aws_elasticloadbalancing.ListenerPort", version: "1.106.1" };
function wellKnownProtocol(port) {
    const proto = tryWellKnownProtocol(port);
    if (!proto) {
        throw new Error(`Please supply protocol to go with port ${port}`);
    }
    return proto;
}
function tryWellKnownProtocol(port) {
    if (port === 80) {
        return LoadBalancingProtocol.HTTP;
    }
    if (port === 443) {
        return LoadBalancingProtocol.HTTPS;
    }
    return undefined;
}
function isHttpProtocol(proto) {
    return proto === LoadBalancingProtocol.HTTPS || proto === LoadBalancingProtocol.HTTP;
}
function ifUndefined(x, def) {
    return x != null ? x : def;
}
function ifUndefinedLazy(x, def) {
    return x != null ? x : def();
}
/**
 * Turn health check parameters into a parameter blob for the LB
 */
function healthCheckToJSON(healthCheck) {
    const protocol = ifUndefined(healthCheck.protocol, ifUndefined(tryWellKnownProtocol(healthCheck.port), LoadBalancingProtocol.TCP));
    const path = protocol === LoadBalancingProtocol.HTTP || protocol === LoadBalancingProtocol.HTTPS ? ifUndefined(healthCheck.path, '/') : '';
    const target = `${protocol.toUpperCase()}:${healthCheck.port}${path}`;
    return {
        healthyThreshold: ifUndefined(healthCheck.healthyThreshold, 2).toString(),
        interval: (healthCheck.interval || core_1.Duration.seconds(30)).toSeconds().toString(),
        target,
        timeout: (healthCheck.timeout || core_1.Duration.seconds(5)).toSeconds().toString(),
        unhealthyThreshold: ifUndefined(healthCheck.unhealthyThreshold, 5).toString(),
    };
}
function loadBalancerSubnets(props) {
    if (props.subnetSelection !== undefined) {
        return props.vpc.selectSubnets(props.subnetSelection);
    }
    else if (props.internetFacing) {
        return props.vpc.selectSubnets({
            subnetType: aws_ec2_1.SubnetType.PUBLIC,
        });
    }
    else {
        return props.vpc.selectSubnets({
            subnetType: aws_ec2_1.SubnetType.PRIVATE,
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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