"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KubectlProvider = void 0;
const path = require("path");
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cr = require("../../custom-resources"); // Automatically re-written from '@aws-cdk/custom-resources'
const lambda_layer_awscli_1 = require("../../lambda-layer-awscli"); // Automatically re-written from '@aws-cdk/lambda-layer-awscli'
const lambda_layer_kubectl_1 = require("../../lambda-layer-kubectl"); // Automatically re-written from '@aws-cdk/lambda-layer-kubectl'
const cluster_1 = require("./cluster");
class KubectlProvider extends core_1.NestedStack {
    constructor(scope, id, props) {
        super(scope, id);
        const cluster = props.cluster;
        if (!cluster.kubectlRole) {
            throw new Error('"kubectlRole" is not defined, cannot issue kubectl commands against this cluster');
        }
        if (cluster.kubectlPrivateSubnets && !cluster.kubectlSecurityGroup) {
            throw new Error('"kubectlSecurityGroup" is required if "kubectlSubnets" is specified');
        }
        const memorySize = cluster.kubectlMemory ? cluster.kubectlMemory.toMebibytes() : 1024;
        const handler = new lambda.Function(this, 'Handler', {
            code: lambda.Code.fromAsset(path.join(__dirname, 'kubectl-handler')),
            runtime: lambda.Runtime.PYTHON_3_7,
            handler: 'index.handler',
            timeout: core_1.Duration.minutes(15),
            description: 'onEvent handler for EKS kubectl resource provider',
            memorySize,
            environment: cluster.kubectlEnvironment,
            // defined only when using private access
            vpc: cluster.kubectlPrivateSubnets ? cluster.vpc : undefined,
            securityGroups: cluster.kubectlSecurityGroup ? [cluster.kubectlSecurityGroup] : undefined,
            vpcSubnets: cluster.kubectlPrivateSubnets ? { subnets: cluster.kubectlPrivateSubnets } : undefined,
        });
        // allow user to customize the layer
        if (!props.cluster.kubectlLayer) {
            handler.addLayers(new lambda_layer_awscli_1.AwsCliLayer(this, 'AwsCliLayer'));
            handler.addLayers(new lambda_layer_kubectl_1.KubectlLayer(this, 'KubectlLayer'));
        }
        else {
            handler.addLayers(props.cluster.kubectlLayer);
        }
        this.handlerRole = handler.role;
        this.handlerRole.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['eks:DescribeCluster'],
            resources: [cluster.clusterArn],
        }));
        // allow this handler to assume the kubectl role
        cluster.kubectlRole.grant(this.handlerRole, 'sts:AssumeRole');
        const provider = new cr.Provider(this, 'Provider', {
            onEventHandler: handler,
            vpc: cluster.kubectlPrivateSubnets ? cluster.vpc : undefined,
            vpcSubnets: cluster.kubectlPrivateSubnets ? { subnets: cluster.kubectlPrivateSubnets } : undefined,
            securityGroups: cluster.kubectlSecurityGroup ? [cluster.kubectlSecurityGroup] : undefined,
        });
        this.serviceToken = provider.serviceToken;
        this.roleArn = cluster.kubectlRole.roleArn;
    }
    static getOrCreate(scope, cluster) {
        // if this is an "owned" cluster, it has a provider associated with it
        if (cluster instanceof cluster_1.Cluster) {
            return cluster._attachKubectlResourceScope(scope);
        }
        // if this is an imported cluster, we need to provision a custom resource provider in this stack
        // we will define one per stack for each cluster based on the cluster uniqueid
        const uid = `${core_1.Names.nodeUniqueId(cluster.node)}-KubectlProvider`;
        const stack = core_1.Stack.of(scope);
        let provider = stack.node.tryFindChild(uid);
        if (!provider) {
            provider = new KubectlProvider(stack, uid, { cluster });
        }
        return provider;
    }
}
exports.KubectlProvider = KubectlProvider;
//# sourceMappingURL=data:application/json;base64,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