"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ClusterResource = void 0;
const path = require("path");
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const kubectl_layer_1 = require("./kubectl-layer");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_2 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * A low-level CFN resource Amazon EKS cluster implemented through a custom
 * resource.
 *
 * Implements EKS create/update/delete through a CloudFormation custom resource
 * in order to allow us to control the IAM role which creates the cluster. This
 * is required in order to be able to allow CloudFormation to interact with the
 * cluster via `kubectl` to enable Kubernetes management capabilities like apply
 * manifest and IAM role/user RBAC mapping.
 */
class ClusterResource extends core_2.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // each cluster resource will have it's own lambda handler since permissions
        // are scoped to this cluster and related resources like it's role
        const handler = new lambda.Function(this, 'ResourceHandler', {
            code: lambda.Code.fromAsset(path.join(__dirname, 'cluster-resource')),
            runtime: lambda.Runtime.PYTHON_3_7,
            handler: 'index.handler',
            timeout: core_1.Duration.minutes(15),
            memorySize: 512,
            layers: [kubectl_layer_1.KubectlLayer.getOrCreate(this)],
        });
        if (!props.roleArn) {
            throw new Error('"roleArn" is required');
        }
        // since we don't know the cluster name at this point, we must give this role star resource permissions
        handler.addToRolePolicy(new iam.PolicyStatement({
            actions: ['eks:CreateCluster', 'eks:DescribeCluster', 'eks:DeleteCluster', 'eks:UpdateClusterVersion'],
            resources: ['*'],
        }));
        // the CreateCluster API will allow the cluster to assume this role, so we
        // need to allow the lambda execution role to pass it.
        handler.addToRolePolicy(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: [props.roleArn],
        }));
        const resource = new core_1.CustomResource(this, 'Resource', {
            resourceType: ClusterResource.RESOURCE_TYPE,
            serviceToken: handler.functionArn,
            properties: {
                Config: props,
            },
        });
        this.ref = resource.ref;
        this.attrEndpoint = core_1.Token.asString(resource.getAtt('Endpoint'));
        this.attrArn = core_1.Token.asString(resource.getAtt('Arn'));
        this.attrCertificateAuthorityData = core_1.Token.asString(resource.getAtt('CertificateAuthorityData'));
        this.creationRole = handler.role;
    }
}
exports.ClusterResource = ClusterResource;
/**
 * The AWS CloudFormation resource type used for this resource.
 */
ClusterResource.RESOURCE_TYPE = 'Custom::AWSCDK-EKS-Cluster';
//# sourceMappingURL=data:application/json;base64,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