"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FirelensLogRouter = exports.obtainDefaultFluentBitECRImage = exports.FirelensConfigFileType = exports.FirelensLogRouterType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const ssm = require("../../aws-ssm"); // Automatically re-written from '@aws-cdk/aws-ssm'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const container_definition_1 = require("./container-definition");
const container_image_1 = require("./container-image");
/**
 * (experimental) Firelens log router type, fluentbit or fluentd.
 *
 * https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html
 *
 * @experimental
 */
var FirelensLogRouterType;
(function (FirelensLogRouterType) {
    FirelensLogRouterType["FLUENTBIT"] = "fluentbit";
    FirelensLogRouterType["FLUENTD"] = "fluentd";
})(FirelensLogRouterType = exports.FirelensLogRouterType || (exports.FirelensLogRouterType = {}));
/**
 * (experimental) Firelens configuration file type, s3 or file path.
 *
 * https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html#firelens-taskdef-customconfig
 *
 * @experimental
 */
var FirelensConfigFileType;
(function (FirelensConfigFileType) {
    FirelensConfigFileType["S3"] = "s3";
    FirelensConfigFileType["FILE"] = "file";
})(FirelensConfigFileType = exports.FirelensConfigFileType || (exports.FirelensConfigFileType = {}));
/**
 * Render to CfnTaskDefinition.FirelensConfigurationProperty from FirelensConfig
 */
function renderFirelensConfig(firelensConfig) {
    if (!firelensConfig.options) {
        return { type: firelensConfig.type };
    }
    else {
        // firelensConfig.options.configFileType has been filled with s3 or file type in constructor.
        return {
            type: firelensConfig.type,
            options: {
                'enable-ecs-log-metadata': firelensConfig.options.enableECSLogMetadata ? 'true' : 'false',
                'config-file-type': firelensConfig.options.configFileType,
                'config-file-value': firelensConfig.options.configFileValue,
            },
        };
    }
}
/**
 * SSM parameters for latest fluent bit docker image in ECR
 * https://github.com/aws/aws-for-fluent-bit#using-ssm-to-find-available-versions
 */
const fluentBitImageSSMPath = '/aws/service/aws-for-fluent-bit';
/**
 * Obtain Fluent Bit image in Amazon ECR and setup corresponding IAM permissions.
 * ECR image pull permissions will be granted in task execution role.
 * Cloudwatch logs, Kinesis data stream or firehose permissions will be grant by check options in logDriverConfig.
 * https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html#firelens-using-fluentbit
 */
function obtainDefaultFluentBitECRImage(task, logDriverConfig, imageTag) {
    // grant ECR image pull permissions to executor role
    task.addToExecutionRolePolicy(new iam.PolicyStatement({
        actions: [
            'ecr:GetAuthorizationToken',
            'ecr:BatchCheckLayerAvailability',
            'ecr:GetDownloadUrlForLayer',
            'ecr:BatchGetImage',
        ],
        resources: ['*'],
    }));
    // grant cloudwatch or firehose permissions to task role
    const logName = logDriverConfig && logDriverConfig.logDriver === 'awsfirelens'
        && logDriverConfig.options && logDriverConfig.options.Name;
    if (logName === 'cloudwatch') {
        task.addToTaskRolePolicy(new iam.PolicyStatement({
            actions: [
                'logs:CreateLogGroup',
                'logs:CreateLogStream',
                'logs:DescribeLogStreams',
                'logs:PutLogEvents',
            ],
            resources: ['*'],
        }));
    }
    else if (logName === 'firehose') {
        task.addToTaskRolePolicy(new iam.PolicyStatement({
            actions: [
                'firehose:PutRecordBatch',
            ],
            resources: ['*'],
        }));
    }
    else if (logName === 'kinesis') {
        task.addToTaskRolePolicy(new iam.PolicyStatement({
            actions: [
                'kinesis:PutRecords',
            ],
            resources: ['*'],
        }));
    }
    const fluentBitImageTag = imageTag || 'latest';
    const fluentBitImage = `${fluentBitImageSSMPath}/${fluentBitImageTag}`;
    // Not use ContainerImage.fromEcrRepository since it's not support parsing ECR repo URI,
    // use repo ARN might result in complex Fn:: functions in cloudformation template.
    return container_image_1.ContainerImage.fromRegistry(ssm.StringParameter.valueForStringParameter(task, fluentBitImage));
}
exports.obtainDefaultFluentBitECRImage = obtainDefaultFluentBitECRImage;
/**
 * (experimental) Firelens log router.
 *
 * @experimental
 */
class FirelensLogRouter extends container_definition_1.ContainerDefinition {
    /**
     * (experimental) Constructs a new instance of the FirelensLogRouter class.
     *
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        const options = props.firelensConfig.options;
        if (options) {
            const enableECSLogMetadata = options.enableECSLogMetadata || options.enableECSLogMetadata === undefined;
            const configFileType = (options.configFileType === undefined || options.configFileType === FirelensConfigFileType.S3) &&
                (cdk.Token.isUnresolved(options.configFileValue) || /arn:aws[a-zA-Z-]*:s3:::.+/.test(options.configFileValue))
                ? FirelensConfigFileType.S3 : FirelensConfigFileType.FILE;
            this.firelensConfig = {
                type: props.firelensConfig.type,
                options: {
                    enableECSLogMetadata,
                    configFileType,
                    configFileValue: options.configFileValue,
                },
            };
            // grant s3 access permissions
            if (configFileType === FirelensConfigFileType.S3) {
                props.taskDefinition.addToExecutionRolePolicy(new iam.PolicyStatement({
                    actions: [
                        's3:GetObject',
                    ],
                    resources: [options.configFileValue],
                }));
                props.taskDefinition.addToExecutionRolePolicy(new iam.PolicyStatement({
                    actions: [
                        's3:GetBucketLocation',
                    ],
                    resources: [options.configFileValue.split('/')[0]],
                }));
            }
        }
        else {
            this.firelensConfig = props.firelensConfig;
        }
    }
    /**
     * (experimental) Render this container definition to a CloudFormation object.
     *
     * @experimental
     */
    renderContainerDefinition(_taskDefinition) {
        return {
            ...(super.renderContainerDefinition()),
            firelensConfiguration: this.firelensConfig && renderFirelensConfig(this.firelensConfig),
        };
    }
}
exports.FirelensLogRouter = FirelensLogRouter;
_a = JSII_RTTI_SYMBOL_1;
FirelensLogRouter[_a] = { fqn: "monocdk.aws_ecs.FirelensLogRouter", version: "1.106.1" };
//# sourceMappingURL=data:application/json;base64,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