"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.isFargateCompatible = exports.isEc2Compatible = exports.Compatibility = exports.Scope = exports.PidMode = exports.IpcMode = exports.NetworkMode = exports.TaskDefinition = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("../../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const container_definition_1 = require("../container-definition");
const ecs_generated_1 = require("../ecs.generated");
const firelens_log_router_1 = require("../firelens-log-router");
const aws_log_driver_1 = require("../log-drivers/aws-log-driver");
const _imported_task_definition_1 = require("./_imported-task-definition");
class TaskDefinitionBase extends core_1.Resource {
    /**
     * (experimental) Return true if the task definition can be run on an EC2 cluster.
     *
     * @experimental
     */
    get isEc2Compatible() {
        return isEc2Compatible(this.compatibility);
    }
    /**
     * (experimental) Return true if the task definition can be run on a Fargate cluster.
     *
     * @experimental
     */
    get isFargateCompatible() {
        return isFargateCompatible(this.compatibility);
    }
}
/**
 * (experimental) The base class for all task definitions.
 *
 * @experimental
 */
class TaskDefinition extends TaskDefinitionBase {
    /**
     * (experimental) Constructs a new instance of the TaskDefinition class.
     *
     * @experimental
     */
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        /**
         * (experimental) The container definitions.
         *
         * @experimental
         */
        this.containers = new Array();
        /**
         * All volumes
         */
        this.volumes = [];
        /**
         * Placement constraints for task instances
         */
        this.placementConstraints = new Array();
        /**
         * Inference accelerators for task instances
         */
        this._inferenceAccelerators = [];
        this.family = props.family || core_1.Names.uniqueId(this);
        this.compatibility = props.compatibility;
        if (props.volumes) {
            props.volumes.forEach(v => this.addVolume(v));
        }
        this.networkMode = (_b = props.networkMode) !== null && _b !== void 0 ? _b : (this.isFargateCompatible ? NetworkMode.AWS_VPC : NetworkMode.BRIDGE);
        if (this.isFargateCompatible && this.networkMode !== NetworkMode.AWS_VPC) {
            throw new Error(`Fargate tasks can only have AwsVpc network mode, got: ${this.networkMode}`);
        }
        if (props.proxyConfiguration && this.networkMode !== NetworkMode.AWS_VPC) {
            throw new Error(`ProxyConfiguration can only be used with AwsVpc network mode, got: ${this.networkMode}`);
        }
        if (props.placementConstraints && props.placementConstraints.length > 0 && this.isFargateCompatible) {
            throw new Error('Cannot set placement constraints on tasks that run on Fargate');
        }
        if (this.isFargateCompatible && (!props.cpu || !props.memoryMiB)) {
            throw new Error(`Fargate-compatible tasks require both CPU (${props.cpu}) and memory (${props.memoryMiB}) specifications`);
        }
        if (props.inferenceAccelerators && props.inferenceAccelerators.length > 0 && this.isFargateCompatible) {
            throw new Error('Cannot use inference accelerators on tasks that run on Fargate');
        }
        this._executionRole = props.executionRole;
        this.taskRole = props.taskRole || new iam.Role(this, 'TaskRole', {
            assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
        });
        if (props.inferenceAccelerators) {
            props.inferenceAccelerators.forEach(ia => this.addInferenceAccelerator(ia));
        }
        const taskDef = new ecs_generated_1.CfnTaskDefinition(this, 'Resource', {
            containerDefinitions: core_1.Lazy.any({ produce: () => this.renderContainers() }, { omitEmptyArray: true }),
            volumes: core_1.Lazy.any({ produce: () => this.renderVolumes() }, { omitEmptyArray: true }),
            executionRoleArn: core_1.Lazy.string({ produce: () => this.executionRole && this.executionRole.roleArn }),
            family: this.family,
            taskRoleArn: this.taskRole.roleArn,
            requiresCompatibilities: [
                ...(isEc2Compatible(props.compatibility) ? ['EC2'] : []),
                ...(isFargateCompatible(props.compatibility) ? ['FARGATE'] : []),
            ],
            networkMode: this.renderNetworkMode(this.networkMode),
            placementConstraints: core_1.Lazy.any({
                produce: () => !isFargateCompatible(this.compatibility) ? this.placementConstraints : undefined,
            }, { omitEmptyArray: true }),
            proxyConfiguration: props.proxyConfiguration ? props.proxyConfiguration.bind(this.stack, this) : undefined,
            cpu: props.cpu,
            memory: props.memoryMiB,
            ipcMode: props.ipcMode,
            pidMode: props.pidMode,
            inferenceAccelerators: core_1.Lazy.any({
                produce: () => !isFargateCompatible(this.compatibility) ? this.renderInferenceAccelerators() : undefined,
            }, { omitEmptyArray: true }),
        });
        if (props.placementConstraints) {
            props.placementConstraints.forEach(pc => this.addPlacementConstraint(pc));
        }
        this.taskDefinitionArn = taskDef.ref;
    }
    /**
     * (experimental) Imports a task definition from the specified task definition ARN.
     *
     * The task will have a compatibility of EC2+Fargate.
     *
     * @experimental
     */
    static fromTaskDefinitionArn(scope, id, taskDefinitionArn) {
        return new _imported_task_definition_1.ImportedTaskDefinition(scope, id, { taskDefinitionArn: taskDefinitionArn });
    }
    /**
     * (experimental) Create a task definition from a task definition reference.
     *
     * @experimental
     */
    static fromTaskDefinitionAttributes(scope, id, attrs) {
        return new _imported_task_definition_1.ImportedTaskDefinition(scope, id, {
            taskDefinitionArn: attrs.taskDefinitionArn,
            compatibility: attrs.compatibility,
            networkMode: attrs.networkMode,
            taskRole: attrs.taskRole,
        });
    }
    /**
     * (experimental) Execution role for this task definition.
     *
     * @experimental
     */
    get executionRole() {
        return this._executionRole;
    }
    /**
     * (experimental) Public getter method to access list of inference accelerators attached to the instance.
     *
     * @experimental
     */
    get inferenceAccelerators() {
        return this._inferenceAccelerators;
    }
    renderVolumes() {
        return this.volumes.map(renderVolume);
        function renderVolume(spec) {
            return {
                host: spec.host,
                name: spec.name,
                dockerVolumeConfiguration: spec.dockerVolumeConfiguration && {
                    autoprovision: spec.dockerVolumeConfiguration.autoprovision,
                    driver: spec.dockerVolumeConfiguration.driver,
                    driverOpts: spec.dockerVolumeConfiguration.driverOpts,
                    labels: spec.dockerVolumeConfiguration.labels,
                    scope: spec.dockerVolumeConfiguration.scope,
                },
                efsVolumeConfiguration: spec.efsVolumeConfiguration && {
                    fileSystemId: spec.efsVolumeConfiguration.fileSystemId,
                    authorizationConfig: spec.efsVolumeConfiguration.authorizationConfig,
                    rootDirectory: spec.efsVolumeConfiguration.rootDirectory,
                    transitEncryption: spec.efsVolumeConfiguration.transitEncryption,
                    transitEncryptionPort: spec.efsVolumeConfiguration.transitEncryptionPort,
                },
            };
        }
    }
    renderInferenceAccelerators() {
        return this._inferenceAccelerators.map(renderInferenceAccelerator);
        function renderInferenceAccelerator(inferenceAccelerator) {
            return {
                deviceName: inferenceAccelerator.deviceName,
                deviceType: inferenceAccelerator.deviceType,
            };
        }
    }
    /**
     * Validate the existence of the input target and set default values.
     *
     * @internal
     */
    _validateTarget(options) {
        const targetContainer = this.findContainer(options.containerName);
        if (targetContainer === undefined) {
            throw new Error(`No container named '${options.containerName}'. Did you call "addContainer()"?`);
        }
        const targetProtocol = options.protocol || container_definition_1.Protocol.TCP;
        const targetContainerPort = options.containerPort || targetContainer.containerPort;
        const portMapping = targetContainer.findPortMapping(targetContainerPort, targetProtocol);
        if (portMapping === undefined) {
            // eslint-disable-next-line max-len
            throw new Error(`Container '${targetContainer}' has no mapping for port ${options.containerPort} and protocol ${targetProtocol}. Did you call "container.addPortMappings()"?`);
        }
        return {
            containerName: options.containerName,
            portMapping,
        };
    }
    /**
     * Returns the port range to be opened that match the provided container name and container port.
     *
     * @internal
     */
    _portRangeFromPortMapping(portMapping) {
        if (portMapping.hostPort !== undefined && portMapping.hostPort !== 0) {
            return portMapping.protocol === container_definition_1.Protocol.UDP ? ec2.Port.udp(portMapping.hostPort) : ec2.Port.tcp(portMapping.hostPort);
        }
        if (this.networkMode === NetworkMode.BRIDGE || this.networkMode === NetworkMode.NAT) {
            return EPHEMERAL_PORT_RANGE;
        }
        return portMapping.protocol === container_definition_1.Protocol.UDP ? ec2.Port.udp(portMapping.containerPort) : ec2.Port.tcp(portMapping.containerPort);
    }
    /**
     * (experimental) Adds a policy statement to the task IAM role.
     *
     * @experimental
     */
    addToTaskRolePolicy(statement) {
        this.taskRole.addToPrincipalPolicy(statement);
    }
    /**
     * (experimental) Adds a policy statement to the task execution IAM role.
     *
     * @experimental
     */
    addToExecutionRolePolicy(statement) {
        this.obtainExecutionRole().addToPrincipalPolicy(statement);
    }
    /**
     * (experimental) Adds a new container to the task definition.
     *
     * @experimental
     */
    addContainer(id, props) {
        return new container_definition_1.ContainerDefinition(this, id, { taskDefinition: this, ...props });
    }
    /**
     * (experimental) Adds a firelens log router to the task definition.
     *
     * @experimental
     */
    addFirelensLogRouter(id, props) {
        // only one firelens log router is allowed in each task.
        if (this.containers.find(x => x instanceof firelens_log_router_1.FirelensLogRouter)) {
            throw new Error('Firelens log router is already added in this task.');
        }
        return new firelens_log_router_1.FirelensLogRouter(this, id, { taskDefinition: this, ...props });
    }
    /**
     * Links a container to this task definition.
     * @internal
     */
    _linkContainer(container) {
        this.containers.push(container);
        if (this.defaultContainer === undefined && container.essential) {
            this.defaultContainer = container;
        }
        if (container.referencesSecretJsonField) {
            this._referencesSecretJsonField = true;
        }
    }
    /**
     * (experimental) Adds a volume to the task definition.
     *
     * @experimental
     */
    addVolume(volume) {
        this.volumes.push(volume);
    }
    /**
     * (experimental) Adds the specified placement constraint to the task definition.
     *
     * @experimental
     */
    addPlacementConstraint(constraint) {
        if (isFargateCompatible(this.compatibility)) {
            throw new Error('Cannot set placement constraints on tasks that run on Fargate');
        }
        this.placementConstraints.push(...constraint.toJson());
    }
    /**
     * (experimental) Adds the specified extention to the task definition.
     *
     * Extension can be used to apply a packaged modification to
     * a task definition.
     *
     * @experimental
     */
    addExtension(extension) {
        extension.extend(this);
    }
    /**
     * (experimental) Adds an inference accelerator to the task definition.
     *
     * @experimental
     */
    addInferenceAccelerator(inferenceAccelerator) {
        if (isFargateCompatible(this.compatibility)) {
            throw new Error('Cannot use inference accelerators on tasks that run on Fargate');
        }
        this._inferenceAccelerators.push(inferenceAccelerator);
    }
    /**
     * (experimental) Creates the task execution IAM role if it doesn't already exist.
     *
     * @experimental
     */
    obtainExecutionRole() {
        if (!this._executionRole) {
            this._executionRole = new iam.Role(this, 'ExecutionRole', {
                assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
            });
        }
        return this._executionRole;
    }
    /**
     * (experimental) Whether this task definition has at least a container that references a specific JSON field of a secret stored in Secrets Manager.
     *
     * @experimental
     */
    get referencesSecretJsonField() {
        return this._referencesSecretJsonField;
    }
    /**
     * (experimental) Validates the task definition.
     *
     * @experimental
     */
    validate() {
        const ret = super.validate();
        if (isEc2Compatible(this.compatibility)) {
            // EC2 mode validations
            // Container sizes
            for (const container of this.containers) {
                if (!container.memoryLimitSpecified) {
                    ret.push(`ECS Container ${container.containerName} must have at least one of 'memoryLimitMiB' or 'memoryReservationMiB' specified`);
                }
            }
        }
        return ret;
    }
    /**
     * Returns the container that match the provided containerName.
     */
    findContainer(containerName) {
        return this.containers.find(c => c.containerName === containerName);
    }
    renderNetworkMode(networkMode) {
        return (networkMode === NetworkMode.NAT) ? undefined : networkMode;
    }
    renderContainers() {
        // add firelens log router container if any application container is using firelens log driver,
        // also check if already created log router container
        for (const container of this.containers) {
            if (container.logDriverConfig && container.logDriverConfig.logDriver === 'awsfirelens'
                && !this.containers.find(x => x instanceof firelens_log_router_1.FirelensLogRouter)) {
                this.addFirelensLogRouter('log-router', {
                    image: firelens_log_router_1.obtainDefaultFluentBitECRImage(this, container.logDriverConfig),
                    firelensConfig: {
                        type: firelens_log_router_1.FirelensLogRouterType.FLUENTBIT,
                    },
                    logging: new aws_log_driver_1.AwsLogDriver({ streamPrefix: 'firelens' }),
                    memoryReservationMiB: 50,
                });
                break;
            }
        }
        return this.containers.map(x => x.renderContainerDefinition());
    }
}
exports.TaskDefinition = TaskDefinition;
_a = JSII_RTTI_SYMBOL_1;
TaskDefinition[_a] = { fqn: "monocdk.aws_ecs.TaskDefinition", version: "1.106.1" };
/**
 * The port range to open up for dynamic port mapping
 */
const EPHEMERAL_PORT_RANGE = ec2.Port.tcpRange(32768, 65535);
/**
 * (experimental) The networking mode to use for the containers in the task.
 *
 * @experimental
 */
var NetworkMode;
(function (NetworkMode) {
    NetworkMode["NONE"] = "none";
    NetworkMode["BRIDGE"] = "bridge";
    NetworkMode["AWS_VPC"] = "awsvpc";
    NetworkMode["HOST"] = "host";
    NetworkMode["NAT"] = "nat";
})(NetworkMode = exports.NetworkMode || (exports.NetworkMode = {}));
/**
 * (experimental) The IPC resource namespace to use for the containers in the task.
 *
 * @experimental
 */
var IpcMode;
(function (IpcMode) {
    IpcMode["NONE"] = "none";
    IpcMode["HOST"] = "host";
    IpcMode["TASK"] = "task";
})(IpcMode = exports.IpcMode || (exports.IpcMode = {}));
/**
 * (experimental) The process namespace to use for the containers in the task.
 *
 * @experimental
 */
var PidMode;
(function (PidMode) {
    PidMode["HOST"] = "host";
    PidMode["TASK"] = "task";
})(PidMode = exports.PidMode || (exports.PidMode = {}));
/**
 * (experimental) The scope for the Docker volume that determines its lifecycle.
 *
 * Docker volumes that are scoped to a task are automatically provisioned when the task starts and destroyed when the task stops.
 * Docker volumes that are scoped as shared persist after the task stops.
 *
 * @experimental
 */
var Scope;
(function (Scope) {
    Scope["TASK"] = "task";
    Scope["SHARED"] = "shared";
})(Scope = exports.Scope || (exports.Scope = {}));
/**
 * (experimental) The task launch type compatibility requirement.
 *
 * @experimental
 */
var Compatibility;
(function (Compatibility) {
    Compatibility[Compatibility["EC2"] = 0] = "EC2";
    Compatibility[Compatibility["FARGATE"] = 1] = "FARGATE";
    Compatibility[Compatibility["EC2_AND_FARGATE"] = 2] = "EC2_AND_FARGATE";
})(Compatibility = exports.Compatibility || (exports.Compatibility = {}));
/**
 * Return true if the given task definition can be run on an EC2 cluster
 */
function isEc2Compatible(compatibility) {
    return [Compatibility.EC2, Compatibility.EC2_AND_FARGATE].includes(compatibility);
}
exports.isEc2Compatible = isEc2Compatible;
/**
 * Return true if the given task definition can be run on a Fargate cluster
 */
function isFargateCompatible(compatibility) {
    return [Compatibility.FARGATE, Compatibility.EC2_AND_FARGATE].includes(compatibility);
}
exports.isFargateCompatible = isFargateCompatible;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGFzay1kZWZpbml0aW9uLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidGFzay1kZWZpbml0aW9uLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEsd0NBQXdDLENBQUMsbURBQW1EO0FBQzVGLHdDQUF3QyxDQUFDLG1EQUFtRDtBQUM1Rix3Q0FBaUUsQ0FBQyxnREFBZ0Q7QUFFbEgsa0VBQWlIO0FBQ2pILG9EQUFxRDtBQUNyRCxnRUFBc0o7QUFDdEosa0VBQTZEO0FBRzdELDJFQUFxRTtBQXNNckUsTUFBZSxrQkFBbUIsU0FBUSxlQUFROzs7Ozs7SUFTOUMsSUFBVyxlQUFlO1FBQ3RCLE9BQU8sZUFBZSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUMvQyxDQUFDOzs7Ozs7SUFJRCxJQUFXLG1CQUFtQjtRQUMxQixPQUFPLG1CQUFtQixDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUNuRCxDQUFDO0NBQ0o7Ozs7OztBQUlELE1BQWEsY0FBZSxTQUFRLGtCQUFrQjs7Ozs7O0lBdUVsRCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQTBCOztRQUNoRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDOzs7Ozs7UUFuQkYsZUFBVSxHQUFHLElBQUksS0FBSyxFQUF1QixDQUFDO1FBQ2pFOztXQUVHO1FBQ2MsWUFBTyxHQUFhLEVBQUUsQ0FBQztRQUN4Qzs7V0FFRztRQUNjLHlCQUFvQixHQUFHLElBQUksS0FBSyxFQUErRCxDQUFDO1FBQ2pIOztXQUVHO1FBQ2MsMkJBQXNCLEdBQTJCLEVBQUUsQ0FBQztRQVFqRSxJQUFJLENBQUMsTUFBTSxHQUFHLEtBQUssQ0FBQyxNQUFNLElBQUksWUFBSyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNuRCxJQUFJLENBQUMsYUFBYSxHQUFHLEtBQUssQ0FBQyxhQUFhLENBQUM7UUFDekMsSUFBSSxLQUFLLENBQUMsT0FBTyxFQUFFO1lBQ2YsS0FBSyxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7U0FDakQ7UUFDRCxJQUFJLENBQUMsV0FBVyxTQUFHLEtBQUssQ0FBQyxXQUFXLG1DQUFJLENBQUMsSUFBSSxDQUFDLG1CQUFtQixDQUFDLENBQUMsQ0FBQyxXQUFXLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxXQUFXLENBQUMsTUFBTSxDQUFDLENBQUM7UUFDOUcsSUFBSSxJQUFJLENBQUMsbUJBQW1CLElBQUksSUFBSSxDQUFDLFdBQVcsS0FBSyxXQUFXLENBQUMsT0FBTyxFQUFFO1lBQ3RFLE1BQU0sSUFBSSxLQUFLLENBQUMseURBQXlELElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQyxDQUFDO1NBQ2hHO1FBQ0QsSUFBSSxLQUFLLENBQUMsa0JBQWtCLElBQUksSUFBSSxDQUFDLFdBQVcsS0FBSyxXQUFXLENBQUMsT0FBTyxFQUFFO1lBQ3RFLE1BQU0sSUFBSSxLQUFLLENBQUMsc0VBQXNFLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQyxDQUFDO1NBQzdHO1FBQ0QsSUFBSSxLQUFLLENBQUMsb0JBQW9CLElBQUksS0FBSyxDQUFDLG9CQUFvQixDQUFDLE1BQU0sR0FBRyxDQUFDLElBQUksSUFBSSxDQUFDLG1CQUFtQixFQUFFO1lBQ2pHLE1BQU0sSUFBSSxLQUFLLENBQUMsK0RBQStELENBQUMsQ0FBQztTQUNwRjtRQUNELElBQUksSUFBSSxDQUFDLG1CQUFtQixJQUFJLENBQUMsQ0FBQyxLQUFLLENBQUMsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxFQUFFO1lBQzlELE1BQU0sSUFBSSxLQUFLLENBQUMsOENBQThDLEtBQUssQ0FBQyxHQUFHLGlCQUFpQixLQUFLLENBQUMsU0FBUyxrQkFBa0IsQ0FBQyxDQUFDO1NBQzlIO1FBQ0QsSUFBSSxLQUFLLENBQUMscUJBQXFCLElBQUksS0FBSyxDQUFDLHFCQUFxQixDQUFDLE1BQU0sR0FBRyxDQUFDLElBQUksSUFBSSxDQUFDLG1CQUFtQixFQUFFO1lBQ25HLE1BQU0sSUFBSSxLQUFLLENBQUMsZ0VBQWdFLENBQUMsQ0FBQztTQUNyRjtRQUNELElBQUksQ0FBQyxjQUFjLEdBQUcsS0FBSyxDQUFDLGFBQWEsQ0FBQztRQUMxQyxJQUFJLENBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQyxRQUFRLElBQUksSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDN0QsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHlCQUF5QixDQUFDO1NBQ2pFLENBQUMsQ0FBQztRQUNILElBQUksS0FBSyxDQUFDLHFCQUFxQixFQUFFO1lBQzdCLEtBQUssQ0FBQyxxQkFBcUIsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsdUJBQXVCLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQztTQUMvRTtRQUNELE1BQU0sT0FBTyxHQUFHLElBQUksaUNBQWlCLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUNwRCxvQkFBb0IsRUFBRSxXQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsRUFBRSxFQUFFLEVBQUUsRUFBRSxjQUFjLEVBQUUsSUFBSSxFQUFFLENBQUM7WUFDcEcsT0FBTyxFQUFFLFdBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLGFBQWEsRUFBRSxFQUFFLEVBQUUsRUFBRSxjQUFjLEVBQUUsSUFBSSxFQUFFLENBQUM7WUFDcEYsZ0JBQWdCLEVBQUUsV0FBSSxDQUFDLE1BQU0sQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsYUFBYSxJQUFJLElBQUksQ0FBQyxhQUFhLENBQUMsT0FBTyxFQUFFLENBQUM7WUFDbEcsTUFBTSxFQUFFLElBQUksQ0FBQyxNQUFNO1lBQ25CLFdBQVcsRUFBRSxJQUFJLENBQUMsUUFBUSxDQUFDLE9BQU87WUFDbEMsdUJBQXVCLEVBQUU7Z0JBQ3JCLEdBQUcsQ0FBQyxlQUFlLENBQUMsS0FBSyxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7Z0JBQ3hELEdBQUcsQ0FBQyxtQkFBbUIsQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQzthQUNuRTtZQUNELFdBQVcsRUFBRSxJQUFJLENBQUMsaUJBQWlCLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQztZQUNyRCxvQkFBb0IsRUFBRSxXQUFJLENBQUMsR0FBRyxDQUFDO2dCQUMzQixPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDLENBQUMsU0FBUzthQUNsRyxFQUFFLEVBQUUsY0FBYyxFQUFFLElBQUksRUFBRSxDQUFDO1lBQzVCLGtCQUFrQixFQUFFLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLGtCQUFrQixDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTO1lBQzFHLEdBQUcsRUFBRSxLQUFLLENBQUMsR0FBRztZQUNkLE1BQU0sRUFBRSxLQUFLLENBQUMsU0FBUztZQUN2QixPQUFPLEVBQUUsS0FBSyxDQUFDLE9BQU87WUFDdEIsT0FBTyxFQUFFLEtBQUssQ0FBQyxPQUFPO1lBQ3RCLHFCQUFxQixFQUFFLFdBQUksQ0FBQyxHQUFHLENBQUM7Z0JBQzVCLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDLG1CQUFtQixDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLDJCQUEyQixFQUFFLENBQUMsQ0FBQyxDQUFDLFNBQVM7YUFDM0csRUFBRSxFQUFFLGNBQWMsRUFBRSxJQUFJLEVBQUUsQ0FBQztTQUMvQixDQUFDLENBQUM7UUFDSCxJQUFJLEtBQUssQ0FBQyxvQkFBb0IsRUFBRTtZQUM1QixLQUFLLENBQUMsb0JBQW9CLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLHNCQUFzQixDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUM7U0FDN0U7UUFDRCxJQUFJLENBQUMsaUJBQWlCLEdBQUcsT0FBTyxDQUFDLEdBQUcsQ0FBQztJQUN6QyxDQUFDOzs7Ozs7OztJQTFITSxNQUFNLENBQUMscUJBQXFCLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsaUJBQXlCO1FBQ3ZGLE9BQU8sSUFBSSxrREFBc0IsQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEVBQUUsaUJBQWlCLEVBQUUsaUJBQWlCLEVBQUUsQ0FBQyxDQUFDO0lBQzNGLENBQUM7Ozs7OztJQUlNLE1BQU0sQ0FBQyw0QkFBNEIsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUErQjtRQUNwRyxPQUFPLElBQUksa0RBQXNCLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRTtZQUN6QyxpQkFBaUIsRUFBRSxLQUFLLENBQUMsaUJBQWlCO1lBQzFDLGFBQWEsRUFBRSxLQUFLLENBQUMsYUFBYTtZQUNsQyxXQUFXLEVBQUUsS0FBSyxDQUFDLFdBQVc7WUFDOUIsUUFBUSxFQUFFLEtBQUssQ0FBQyxRQUFRO1NBQzNCLENBQUMsQ0FBQztJQUNQLENBQUM7Ozs7OztJQThHRCxJQUFXLGFBQWE7UUFDcEIsT0FBTyxJQUFJLENBQUMsY0FBYyxDQUFDO0lBQy9CLENBQUM7Ozs7OztJQUlELElBQVcscUJBQXFCO1FBQzVCLE9BQU8sSUFBSSxDQUFDLHNCQUFzQixDQUFDO0lBQ3ZDLENBQUM7SUFDTyxhQUFhO1FBQ2pCLE9BQU8sSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsWUFBWSxDQUFDLENBQUM7UUFDdEMsU0FBUyxZQUFZLENBQUMsSUFBWTtZQUM5QixPQUFPO2dCQUNILElBQUksRUFBRSxJQUFJLENBQUMsSUFBSTtnQkFDZixJQUFJLEVBQUUsSUFBSSxDQUFDLElBQUk7Z0JBQ2YseUJBQXlCLEVBQUUsSUFBSSxDQUFDLHlCQUF5QixJQUFJO29CQUN6RCxhQUFhLEVBQUUsSUFBSSxDQUFDLHlCQUF5QixDQUFDLGFBQWE7b0JBQzNELE1BQU0sRUFBRSxJQUFJLENBQUMseUJBQXlCLENBQUMsTUFBTTtvQkFDN0MsVUFBVSxFQUFFLElBQUksQ0FBQyx5QkFBeUIsQ0FBQyxVQUFVO29CQUNyRCxNQUFNLEVBQUUsSUFBSSxDQUFDLHlCQUF5QixDQUFDLE1BQU07b0JBQzdDLEtBQUssRUFBRSxJQUFJLENBQUMseUJBQXlCLENBQUMsS0FBSztpQkFDOUM7Z0JBQ0Qsc0JBQXNCLEVBQUUsSUFBSSxDQUFDLHNCQUFzQixJQUFJO29CQUNuRCxZQUFZLEVBQUUsSUFBSSxDQUFDLHNCQUFzQixDQUFDLFlBQVk7b0JBQ3RELG1CQUFtQixFQUFFLElBQUksQ0FBQyxzQkFBc0IsQ0FBQyxtQkFBbUI7b0JBQ3BFLGFBQWEsRUFBRSxJQUFJLENBQUMsc0JBQXNCLENBQUMsYUFBYTtvQkFDeEQsaUJBQWlCLEVBQUUsSUFBSSxDQUFDLHNCQUFzQixDQUFDLGlCQUFpQjtvQkFDaEUscUJBQXFCLEVBQUUsSUFBSSxDQUFDLHNCQUFzQixDQUFDLHFCQUFxQjtpQkFDM0U7YUFDSixDQUFDO1FBQ04sQ0FBQztJQUNMLENBQUM7SUFDTywyQkFBMkI7UUFDL0IsT0FBTyxJQUFJLENBQUMsc0JBQXNCLENBQUMsR0FBRyxDQUFDLDBCQUEwQixDQUFDLENBQUM7UUFDbkUsU0FBUywwQkFBMEIsQ0FBQyxvQkFBMEM7WUFDMUUsT0FBTztnQkFDSCxVQUFVLEVBQUUsb0JBQW9CLENBQUMsVUFBVTtnQkFDM0MsVUFBVSxFQUFFLG9CQUFvQixDQUFDLFVBQVU7YUFDOUMsQ0FBQztRQUNOLENBQUM7SUFDTCxDQUFDO0lBQ0Q7Ozs7T0FJRztJQUNJLGVBQWUsQ0FBQyxPQUFrQztRQUNyRCxNQUFNLGVBQWUsR0FBRyxJQUFJLENBQUMsYUFBYSxDQUFDLE9BQU8sQ0FBQyxhQUFhLENBQUMsQ0FBQztRQUNsRSxJQUFJLGVBQWUsS0FBSyxTQUFTLEVBQUU7WUFDL0IsTUFBTSxJQUFJLEtBQUssQ0FBQyx1QkFBdUIsT0FBTyxDQUFDLGFBQWEsbUNBQW1DLENBQUMsQ0FBQztTQUNwRztRQUNELE1BQU0sY0FBYyxHQUFHLE9BQU8sQ0FBQyxRQUFRLElBQUksK0JBQVEsQ0FBQyxHQUFHLENBQUM7UUFDeEQsTUFBTSxtQkFBbUIsR0FBRyxPQUFPLENBQUMsYUFBYSxJQUFJLGVBQWUsQ0FBQyxhQUFhLENBQUM7UUFDbkYsTUFBTSxXQUFXLEdBQUcsZUFBZSxDQUFDLGVBQWUsQ0FBQyxtQkFBbUIsRUFBRSxjQUFjLENBQUMsQ0FBQztRQUN6RixJQUFJLFdBQVcsS0FBSyxTQUFTLEVBQUU7WUFDM0IsbUNBQW1DO1lBQ25DLE1BQU0sSUFBSSxLQUFLLENBQUMsY0FBYyxlQUFlLDZCQUE2QixPQUFPLENBQUMsYUFBYSxpQkFBaUIsY0FBYywrQ0FBK0MsQ0FBQyxDQUFDO1NBQ2xMO1FBQ0QsT0FBTztZQUNILGFBQWEsRUFBRSxPQUFPLENBQUMsYUFBYTtZQUNwQyxXQUFXO1NBQ2QsQ0FBQztJQUNOLENBQUM7SUFDRDs7OztPQUlHO0lBQ0kseUJBQXlCLENBQUMsV0FBd0I7UUFDckQsSUFBSSxXQUFXLENBQUMsUUFBUSxLQUFLLFNBQVMsSUFBSSxXQUFXLENBQUMsUUFBUSxLQUFLLENBQUMsRUFBRTtZQUNsRSxPQUFPLFdBQVcsQ0FBQyxRQUFRLEtBQUssK0JBQVEsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLFdBQVcsQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1NBQzFIO1FBQ0QsSUFBSSxJQUFJLENBQUMsV0FBVyxLQUFLLFdBQVcsQ0FBQyxNQUFNLElBQUksSUFBSSxDQUFDLFdBQVcsS0FBSyxXQUFXLENBQUMsR0FBRyxFQUFFO1lBQ2pGLE9BQU8sb0JBQW9CLENBQUM7U0FDL0I7UUFDRCxPQUFPLFdBQVcsQ0FBQyxRQUFRLEtBQUssK0JBQVEsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLFdBQVcsQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsV0FBVyxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBQ3JJLENBQUM7Ozs7OztJQUlNLG1CQUFtQixDQUFDLFNBQThCO1FBQ3JELElBQUksQ0FBQyxRQUFRLENBQUMsb0JBQW9CLENBQUMsU0FBUyxDQUFDLENBQUM7SUFDbEQsQ0FBQzs7Ozs7O0lBSU0sd0JBQXdCLENBQUMsU0FBOEI7UUFDMUQsSUFBSSxDQUFDLG1CQUFtQixFQUFFLENBQUMsb0JBQW9CLENBQUMsU0FBUyxDQUFDLENBQUM7SUFDL0QsQ0FBQzs7Ozs7O0lBSU0sWUFBWSxDQUFDLEVBQVUsRUFBRSxLQUFpQztRQUM3RCxPQUFPLElBQUksMENBQW1CLENBQUMsSUFBSSxFQUFFLEVBQUUsRUFBRSxFQUFFLGNBQWMsRUFBRSxJQUFJLEVBQUUsR0FBRyxLQUFLLEVBQUUsQ0FBQyxDQUFDO0lBQ2pGLENBQUM7Ozs7OztJQUlNLG9CQUFvQixDQUFDLEVBQVUsRUFBRSxLQUF5QztRQUM3RSx3REFBd0Q7UUFDeEQsSUFBSSxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsWUFBWSx1Q0FBaUIsQ0FBQyxFQUFFO1lBQzNELE1BQU0sSUFBSSxLQUFLLENBQUMsb0RBQW9ELENBQUMsQ0FBQztTQUN6RTtRQUNELE9BQU8sSUFBSSx1Q0FBaUIsQ0FBQyxJQUFJLEVBQUUsRUFBRSxFQUFFLEVBQUUsY0FBYyxFQUFFLElBQUksRUFBRSxHQUFHLEtBQUssRUFBRSxDQUFDLENBQUM7SUFDL0UsQ0FBQztJQUNEOzs7T0FHRztJQUNJLGNBQWMsQ0FBQyxTQUE4QjtRQUNoRCxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUNoQyxJQUFJLElBQUksQ0FBQyxnQkFBZ0IsS0FBSyxTQUFTLElBQUksU0FBUyxDQUFDLFNBQVMsRUFBRTtZQUM1RCxJQUFJLENBQUMsZ0JBQWdCLEdBQUcsU0FBUyxDQUFDO1NBQ3JDO1FBQ0QsSUFBSSxTQUFTLENBQUMseUJBQXlCLEVBQUU7WUFDckMsSUFBSSxDQUFDLDBCQUEwQixHQUFHLElBQUksQ0FBQztTQUMxQztJQUNMLENBQUM7Ozs7OztJQUlNLFNBQVMsQ0FBQyxNQUFjO1FBQzNCLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDO0lBQzlCLENBQUM7Ozs7OztJQUlNLHNCQUFzQixDQUFDLFVBQStCO1FBQ3pELElBQUksbUJBQW1CLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxFQUFFO1lBQ3pDLE1BQU0sSUFBSSxLQUFLLENBQUMsK0RBQStELENBQUMsQ0FBQztTQUNwRjtRQUNELElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxJQUFJLENBQUMsR0FBRyxVQUFVLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQztJQUMzRCxDQUFDOzs7Ozs7Ozs7SUFPTSxZQUFZLENBQUMsU0FBbUM7UUFDbkQsU0FBUyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUMzQixDQUFDOzs7Ozs7SUFJTSx1QkFBdUIsQ0FBQyxvQkFBMEM7UUFDckUsSUFBSSxtQkFBbUIsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLEVBQUU7WUFDekMsTUFBTSxJQUFJLEtBQUssQ0FBQyxnRUFBZ0UsQ0FBQyxDQUFDO1NBQ3JGO1FBQ0QsSUFBSSxDQUFDLHNCQUFzQixDQUFDLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO0lBQzNELENBQUM7Ozs7OztJQUlNLG1CQUFtQjtRQUN0QixJQUFJLENBQUMsSUFBSSxDQUFDLGNBQWMsRUFBRTtZQUN0QixJQUFJLENBQUMsY0FBYyxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsZUFBZSxFQUFFO2dCQUN0RCxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMseUJBQXlCLENBQUM7YUFDakUsQ0FBQyxDQUFDO1NBQ047UUFDRCxPQUFPLElBQUksQ0FBQyxjQUFjLENBQUM7SUFDL0IsQ0FBQzs7Ozs7O0lBS0QsSUFBVyx5QkFBeUI7UUFDaEMsT0FBTyxJQUFJLENBQUMsMEJBQTBCLENBQUM7SUFDM0MsQ0FBQzs7Ozs7O0lBSVMsUUFBUTtRQUNkLE1BQU0sR0FBRyxHQUFHLEtBQUssQ0FBQyxRQUFRLEVBQUUsQ0FBQztRQUM3QixJQUFJLGVBQWUsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLEVBQUU7WUFDckMsdUJBQXVCO1lBQ3ZCLGtCQUFrQjtZQUNsQixLQUFLLE1BQU0sU0FBUyxJQUFJLElBQUksQ0FBQyxVQUFVLEVBQUU7Z0JBQ3JDLElBQUksQ0FBQyxTQUFTLENBQUMsb0JBQW9CLEVBQUU7b0JBQ2pDLEdBQUcsQ0FBQyxJQUFJLENBQUMsaUJBQWlCLFNBQVMsQ0FBQyxhQUFhLGlGQUFpRixDQUFDLENBQUM7aUJBQ3ZJO2FBQ0o7U0FDSjtRQUNELE9BQU8sR0FBRyxDQUFDO0lBQ2YsQ0FBQztJQUNEOztPQUVHO0lBQ0ssYUFBYSxDQUFDLGFBQXFCO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsYUFBYSxLQUFLLGFBQWEsQ0FBQyxDQUFDO0lBQ3hFLENBQUM7SUFDTyxpQkFBaUIsQ0FBQyxXQUF3QjtRQUM5QyxPQUFPLENBQUMsV0FBVyxLQUFLLFdBQVcsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxXQUFXLENBQUM7SUFDdkUsQ0FBQztJQUNPLGdCQUFnQjtRQUNwQiwrRkFBK0Y7UUFDL0YscURBQXFEO1FBQ3JELEtBQUssTUFBTSxTQUFTLElBQUksSUFBSSxDQUFDLFVBQVUsRUFBRTtZQUNyQyxJQUFJLFNBQVMsQ0FBQyxlQUFlLElBQUksU0FBUyxDQUFDLGVBQWUsQ0FBQyxTQUFTLEtBQUssYUFBYTttQkFDL0UsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsWUFBWSx1Q0FBaUIsQ0FBQyxFQUFFO2dCQUMvRCxJQUFJLENBQUMsb0JBQW9CLENBQUMsWUFBWSxFQUFFO29CQUNwQyxLQUFLLEVBQUUsb0RBQThCLENBQUMsSUFBSSxFQUFFLFNBQVMsQ0FBQyxlQUFlLENBQUM7b0JBQ3RFLGNBQWMsRUFBRTt3QkFDWixJQUFJLEVBQUUsMkNBQXFCLENBQUMsU0FBUztxQkFDeEM7b0JBQ0QsT0FBTyxFQUFFLElBQUksNkJBQVksQ0FBQyxFQUFFLFlBQVksRUFBRSxVQUFVLEVBQUUsQ0FBQztvQkFDdkQsb0JBQW9CLEVBQUUsRUFBRTtpQkFDM0IsQ0FBQyxDQUFDO2dCQUNILE1BQU07YUFDVDtTQUNKO1FBQ0QsT0FBTyxJQUFJLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyx5QkFBeUIsRUFBRSxDQUFDLENBQUM7SUFDbkUsQ0FBQzs7QUFyVkwsd0NBc1ZDOzs7QUFDRDs7R0FFRztBQUNILE1BQU0sb0JBQW9CLEdBQUcsR0FBRyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFJN0QsSUFBWSxXQTJCWDtBQTNCRCxXQUFZLFdBQVc7SUFJbkIsNEJBQWEsQ0FBQTtJQUliLGdDQUFpQixDQUFBO0lBSWpCLGlDQUFrQixDQUFBO0lBT2xCLDRCQUFhLENBQUE7SUFPYiwwQkFBVyxDQUFBO0FBQ2YsQ0FBQyxFQTNCVyxXQUFXLEdBQVgsbUJBQVcsS0FBWCxtQkFBVyxRQTJCdEI7Ozs7OztBQUlELElBQVksT0FlWDtBQWZELFdBQVksT0FBTztJQUtmLHdCQUFhLENBQUE7SUFLYix3QkFBYSxDQUFBO0lBSWIsd0JBQWEsQ0FBQTtBQUNqQixDQUFDLEVBZlcsT0FBTyxHQUFQLGVBQU8sS0FBUCxlQUFPLFFBZWxCOzs7Ozs7QUFJRCxJQUFZLE9BVVg7QUFWRCxXQUFZLE9BQU87SUFLZix3QkFBYSxDQUFBO0lBSWIsd0JBQWEsQ0FBQTtBQUNqQixDQUFDLEVBVlcsT0FBTyxHQUFQLGVBQU8sS0FBUCxlQUFPLFFBVWxCOzs7Ozs7Ozs7QUF5TkQsSUFBWSxLQVNYO0FBVEQsV0FBWSxLQUFLO0lBSWIsc0JBQWEsQ0FBQTtJQUliLDBCQUFpQixDQUFBO0FBQ3JCLENBQUMsRUFUVyxLQUFLLEdBQUwsYUFBSyxLQUFMLGFBQUssUUFTaEI7Ozs7OztBQUlELElBQVksYUFhWDtBQWJELFdBQVksYUFBYTtJQUlyQiwrQ0FBRyxDQUFBO0lBSUgsdURBQU8sQ0FBQTtJQUlQLHVFQUFlLENBQUE7QUFDbkIsQ0FBQyxFQWJXLGFBQWEsR0FBYixxQkFBYSxLQUFiLHFCQUFhLFFBYXhCO0FBa0JEOztHQUVHO0FBQ0gsU0FBZ0IsZUFBZSxDQUFDLGFBQTRCO0lBQ3hELE9BQU8sQ0FBQyxhQUFhLENBQUMsR0FBRyxFQUFFLGFBQWEsQ0FBQyxlQUFlLENBQUMsQ0FBQyxRQUFRLENBQUMsYUFBYSxDQUFDLENBQUM7QUFDdEYsQ0FBQztBQUZELDBDQUVDO0FBQ0Q7O0dBRUc7QUFDSCxTQUFnQixtQkFBbUIsQ0FBQyxhQUE0QjtJQUM1RCxPQUFPLENBQUMsYUFBYSxDQUFDLE9BQU8sRUFBRSxhQUFhLENBQUMsZUFBZSxDQUFDLENBQUMsUUFBUSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0FBQzFGLENBQUM7QUFGRCxrREFFQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGVjMiBmcm9tIFwiLi4vLi4vLi4vYXdzLWVjMlwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWVjMidcbmltcG9ydCAqIGFzIGlhbSBmcm9tIFwiLi4vLi4vLi4vYXdzLWlhbVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSdcbmltcG9ydCB7IElSZXNvdXJjZSwgTGF6eSwgTmFtZXMsIFJlc291cmNlIH0gZnJvbSBcIi4uLy4uLy4uL2NvcmVcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2NvcmUnXG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IENvbnRhaW5lckRlZmluaXRpb24sIENvbnRhaW5lckRlZmluaXRpb25PcHRpb25zLCBQb3J0TWFwcGluZywgUHJvdG9jb2wgfSBmcm9tICcuLi9jb250YWluZXItZGVmaW5pdGlvbic7XG5pbXBvcnQgeyBDZm5UYXNrRGVmaW5pdGlvbiB9IGZyb20gJy4uL2Vjcy5nZW5lcmF0ZWQnO1xuaW1wb3J0IHsgRmlyZWxlbnNMb2dSb3V0ZXIsIEZpcmVsZW5zTG9nUm91dGVyRGVmaW5pdGlvbk9wdGlvbnMsIEZpcmVsZW5zTG9nUm91dGVyVHlwZSwgb2J0YWluRGVmYXVsdEZsdWVudEJpdEVDUkltYWdlIH0gZnJvbSAnLi4vZmlyZWxlbnMtbG9nLXJvdXRlcic7XG5pbXBvcnQgeyBBd3NMb2dEcml2ZXIgfSBmcm9tICcuLi9sb2ctZHJpdmVycy9hd3MtbG9nLWRyaXZlcic7XG5pbXBvcnQgeyBQbGFjZW1lbnRDb25zdHJhaW50IH0gZnJvbSAnLi4vcGxhY2VtZW50JztcbmltcG9ydCB7IFByb3h5Q29uZmlndXJhdGlvbiB9IGZyb20gJy4uL3Byb3h5LWNvbmZpZ3VyYXRpb24vcHJveHktY29uZmlndXJhdGlvbic7XG5pbXBvcnQgeyBJbXBvcnRlZFRhc2tEZWZpbml0aW9uIH0gZnJvbSAnLi9faW1wb3J0ZWQtdGFzay1kZWZpbml0aW9uJztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIElUYXNrRGVmaW5pdGlvbiBleHRlbmRzIElSZXNvdXJjZSB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IHRhc2tEZWZpbml0aW9uQXJuOiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IGV4ZWN1dGlvblJvbGU/OiBpYW0uSVJvbGU7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgY29tcGF0aWJpbGl0eTogQ29tcGF0aWJpbGl0eTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgaXNFYzJDb21wYXRpYmxlOiBib29sZWFuO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBpc0ZhcmdhdGVDb21wYXRpYmxlOiBib29sZWFuO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgbmV0d29ya01vZGU6IE5ldHdvcmtNb2RlO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgdGFza1JvbGU6IGlhbS5JUm9sZTtcbn1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBDb21tb25UYXNrRGVmaW5pdGlvblByb3BzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IGZhbWlseT86IHN0cmluZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBleGVjdXRpb25Sb2xlPzogaWFtLklSb2xlO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSB0YXNrUm9sZT86IGlhbS5JUm9sZTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBwcm94eUNvbmZpZ3VyYXRpb24/OiBQcm94eUNvbmZpZ3VyYXRpb247XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IHZvbHVtZXM/OiBWb2x1bWVbXTtcbn1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIFRhc2tEZWZpbml0aW9uUHJvcHMgZXh0ZW5kcyBDb21tb25UYXNrRGVmaW5pdGlvblByb3BzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBuZXR3b3JrTW9kZT86IE5ldHdvcmtNb2RlO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgcGxhY2VtZW50Q29uc3RyYWludHM/OiBQbGFjZW1lbnRDb25zdHJhaW50W107XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBjb21wYXRpYmlsaXR5OiBDb21wYXRpYmlsaXR5O1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgY3B1Pzogc3RyaW5nO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBtZW1vcnlNaUI/OiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IGlwY01vZGU/OiBJcGNNb2RlO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgcGlkTW9kZT86IFBpZE1vZGU7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBpbmZlcmVuY2VBY2NlbGVyYXRvcnM/OiBJbmZlcmVuY2VBY2NlbGVyYXRvcltdO1xufVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBDb21tb25UYXNrRGVmaW5pdGlvbkF0dHJpYnV0ZXMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSB0YXNrRGVmaW5pdGlvbkFybjogc3RyaW5nO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBuZXR3b3JrTW9kZT86IE5ldHdvcmtNb2RlO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgdGFza1JvbGU/OiBpYW0uSVJvbGU7XG59XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgVGFza0RlZmluaXRpb25BdHRyaWJ1dGVzIGV4dGVuZHMgQ29tbW9uVGFza0RlZmluaXRpb25BdHRyaWJ1dGVzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IGNvbXBhdGliaWxpdHk/OiBDb21wYXRpYmlsaXR5O1xufVxuYWJzdHJhY3QgY2xhc3MgVGFza0RlZmluaXRpb25CYXNlIGV4dGVuZHMgUmVzb3VyY2UgaW1wbGVtZW50cyBJVGFza0RlZmluaXRpb24ge1xuICAgIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBjb21wYXRpYmlsaXR5OiBDb21wYXRpYmlsaXR5O1xuICAgIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBuZXR3b3JrTW9kZTogTmV0d29ya01vZGU7XG4gICAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IHRhc2tEZWZpbml0aW9uQXJuOiBzdHJpbmc7XG4gICAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IHRhc2tSb2xlOiBpYW0uSVJvbGU7XG4gICAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IGV4ZWN1dGlvblJvbGU/OiBpYW0uSVJvbGU7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBnZXQgaXNFYzJDb21wYXRpYmxlKCk6IGJvb2xlYW4ge1xuICAgICAgICByZXR1cm4gaXNFYzJDb21wYXRpYmxlKHRoaXMuY29tcGF0aWJpbGl0eSk7XG4gICAgfVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgZ2V0IGlzRmFyZ2F0ZUNvbXBhdGlibGUoKTogYm9vbGVhbiB7XG4gICAgICAgIHJldHVybiBpc0ZhcmdhdGVDb21wYXRpYmxlKHRoaXMuY29tcGF0aWJpbGl0eSk7XG4gICAgfVxufVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgVGFza0RlZmluaXRpb24gZXh0ZW5kcyBUYXNrRGVmaW5pdGlvbkJhc2Uge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyBmcm9tVGFza0RlZmluaXRpb25Bcm4oc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgdGFza0RlZmluaXRpb25Bcm46IHN0cmluZyk6IElUYXNrRGVmaW5pdGlvbiB7XG4gICAgICAgIHJldHVybiBuZXcgSW1wb3J0ZWRUYXNrRGVmaW5pdGlvbihzY29wZSwgaWQsIHsgdGFza0RlZmluaXRpb25Bcm46IHRhc2tEZWZpbml0aW9uQXJuIH0pO1xuICAgIH1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyBmcm9tVGFza0RlZmluaXRpb25BdHRyaWJ1dGVzKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIGF0dHJzOiBUYXNrRGVmaW5pdGlvbkF0dHJpYnV0ZXMpOiBJVGFza0RlZmluaXRpb24ge1xuICAgICAgICByZXR1cm4gbmV3IEltcG9ydGVkVGFza0RlZmluaXRpb24oc2NvcGUsIGlkLCB7XG4gICAgICAgICAgICB0YXNrRGVmaW5pdGlvbkFybjogYXR0cnMudGFza0RlZmluaXRpb25Bcm4sXG4gICAgICAgICAgICBjb21wYXRpYmlsaXR5OiBhdHRycy5jb21wYXRpYmlsaXR5LFxuICAgICAgICAgICAgbmV0d29ya01vZGU6IGF0dHJzLm5ldHdvcmtNb2RlLFxuICAgICAgICAgICAgdGFza1JvbGU6IGF0dHJzLnRhc2tSb2xlLFxuICAgICAgICB9KTtcbiAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgcmVhZG9ubHkgZmFtaWx5OiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHJlYWRvbmx5IHRhc2tEZWZpbml0aW9uQXJuOiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgcmVhZG9ubHkgdGFza1JvbGU6IGlhbS5JUm9sZTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyByZWFkb25seSBuZXR3b3JrTW9kZTogTmV0d29ya01vZGU7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIGRlZmF1bHRDb250YWluZXI/OiBDb250YWluZXJEZWZpbml0aW9uO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyByZWFkb25seSBjb21wYXRpYmlsaXR5OiBDb21wYXRpYmlsaXR5O1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHByb3RlY3RlZCByZWFkb25seSBjb250YWluZXJzID0gbmV3IEFycmF5PENvbnRhaW5lckRlZmluaXRpb24+KCk7XG4gICAgLyoqXG4gICAgICogQWxsIHZvbHVtZXNcbiAgICAgKi9cbiAgICBwcml2YXRlIHJlYWRvbmx5IHZvbHVtZXM6IFZvbHVtZVtdID0gW107XG4gICAgLyoqXG4gICAgICogUGxhY2VtZW50IGNvbnN0cmFpbnRzIGZvciB0YXNrIGluc3RhbmNlc1xuICAgICAqL1xuICAgIHByaXZhdGUgcmVhZG9ubHkgcGxhY2VtZW50Q29uc3RyYWludHMgPSBuZXcgQXJyYXk8Q2ZuVGFza0RlZmluaXRpb24uVGFza0RlZmluaXRpb25QbGFjZW1lbnRDb25zdHJhaW50UHJvcGVydHk+KCk7XG4gICAgLyoqXG4gICAgICogSW5mZXJlbmNlIGFjY2VsZXJhdG9ycyBmb3IgdGFzayBpbnN0YW5jZXNcbiAgICAgKi9cbiAgICBwcml2YXRlIHJlYWRvbmx5IF9pbmZlcmVuY2VBY2NlbGVyYXRvcnM6IEluZmVyZW5jZUFjY2VsZXJhdG9yW10gPSBbXTtcbiAgICBwcml2YXRlIF9leGVjdXRpb25Sb2xlPzogaWFtLklSb2xlO1xuICAgIHByaXZhdGUgX3JlZmVyZW5jZXNTZWNyZXRKc29uRmllbGQ/OiBib29sZWFuO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogVGFza0RlZmluaXRpb25Qcm9wcykge1xuICAgICAgICBzdXBlcihzY29wZSwgaWQpO1xuICAgICAgICB0aGlzLmZhbWlseSA9IHByb3BzLmZhbWlseSB8fCBOYW1lcy51bmlxdWVJZCh0aGlzKTtcbiAgICAgICAgdGhpcy5jb21wYXRpYmlsaXR5ID0gcHJvcHMuY29tcGF0aWJpbGl0eTtcbiAgICAgICAgaWYgKHByb3BzLnZvbHVtZXMpIHtcbiAgICAgICAgICAgIHByb3BzLnZvbHVtZXMuZm9yRWFjaCh2ID0+IHRoaXMuYWRkVm9sdW1lKHYpKTtcbiAgICAgICAgfVxuICAgICAgICB0aGlzLm5ldHdvcmtNb2RlID0gcHJvcHMubmV0d29ya01vZGUgPz8gKHRoaXMuaXNGYXJnYXRlQ29tcGF0aWJsZSA/IE5ldHdvcmtNb2RlLkFXU19WUEMgOiBOZXR3b3JrTW9kZS5CUklER0UpO1xuICAgICAgICBpZiAodGhpcy5pc0ZhcmdhdGVDb21wYXRpYmxlICYmIHRoaXMubmV0d29ya01vZGUgIT09IE5ldHdvcmtNb2RlLkFXU19WUEMpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgRmFyZ2F0ZSB0YXNrcyBjYW4gb25seSBoYXZlIEF3c1ZwYyBuZXR3b3JrIG1vZGUsIGdvdDogJHt0aGlzLm5ldHdvcmtNb2RlfWApO1xuICAgICAgICB9XG4gICAgICAgIGlmIChwcm9wcy5wcm94eUNvbmZpZ3VyYXRpb24gJiYgdGhpcy5uZXR3b3JrTW9kZSAhPT0gTmV0d29ya01vZGUuQVdTX1ZQQykge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBQcm94eUNvbmZpZ3VyYXRpb24gY2FuIG9ubHkgYmUgdXNlZCB3aXRoIEF3c1ZwYyBuZXR3b3JrIG1vZGUsIGdvdDogJHt0aGlzLm5ldHdvcmtNb2RlfWApO1xuICAgICAgICB9XG4gICAgICAgIGlmIChwcm9wcy5wbGFjZW1lbnRDb25zdHJhaW50cyAmJiBwcm9wcy5wbGFjZW1lbnRDb25zdHJhaW50cy5sZW5ndGggPiAwICYmIHRoaXMuaXNGYXJnYXRlQ29tcGF0aWJsZSkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdDYW5ub3Qgc2V0IHBsYWNlbWVudCBjb25zdHJhaW50cyBvbiB0YXNrcyB0aGF0IHJ1biBvbiBGYXJnYXRlJyk7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKHRoaXMuaXNGYXJnYXRlQ29tcGF0aWJsZSAmJiAoIXByb3BzLmNwdSB8fCAhcHJvcHMubWVtb3J5TWlCKSkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBGYXJnYXRlLWNvbXBhdGlibGUgdGFza3MgcmVxdWlyZSBib3RoIENQVSAoJHtwcm9wcy5jcHV9KSBhbmQgbWVtb3J5ICgke3Byb3BzLm1lbW9yeU1pQn0pIHNwZWNpZmljYXRpb25zYCk7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKHByb3BzLmluZmVyZW5jZUFjY2VsZXJhdG9ycyAmJiBwcm9wcy5pbmZlcmVuY2VBY2NlbGVyYXRvcnMubGVuZ3RoID4gMCAmJiB0aGlzLmlzRmFyZ2F0ZUNvbXBhdGlibGUpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignQ2Fubm90IHVzZSBpbmZlcmVuY2UgYWNjZWxlcmF0b3JzIG9uIHRhc2tzIHRoYXQgcnVuIG9uIEZhcmdhdGUnKTtcbiAgICAgICAgfVxuICAgICAgICB0aGlzLl9leGVjdXRpb25Sb2xlID0gcHJvcHMuZXhlY3V0aW9uUm9sZTtcbiAgICAgICAgdGhpcy50YXNrUm9sZSA9IHByb3BzLnRhc2tSb2xlIHx8IG5ldyBpYW0uUm9sZSh0aGlzLCAnVGFza1JvbGUnLCB7XG4gICAgICAgICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnZWNzLXRhc2tzLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgICAgfSk7XG4gICAgICAgIGlmIChwcm9wcy5pbmZlcmVuY2VBY2NlbGVyYXRvcnMpIHtcbiAgICAgICAgICAgIHByb3BzLmluZmVyZW5jZUFjY2VsZXJhdG9ycy5mb3JFYWNoKGlhID0+IHRoaXMuYWRkSW5mZXJlbmNlQWNjZWxlcmF0b3IoaWEpKTtcbiAgICAgICAgfVxuICAgICAgICBjb25zdCB0YXNrRGVmID0gbmV3IENmblRhc2tEZWZpbml0aW9uKHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgICAgICAgIGNvbnRhaW5lckRlZmluaXRpb25zOiBMYXp5LmFueSh7IHByb2R1Y2U6ICgpID0+IHRoaXMucmVuZGVyQ29udGFpbmVycygpIH0sIHsgb21pdEVtcHR5QXJyYXk6IHRydWUgfSksXG4gICAgICAgICAgICB2b2x1bWVzOiBMYXp5LmFueSh7IHByb2R1Y2U6ICgpID0+IHRoaXMucmVuZGVyVm9sdW1lcygpIH0sIHsgb21pdEVtcHR5QXJyYXk6IHRydWUgfSksXG4gICAgICAgICAgICBleGVjdXRpb25Sb2xlQXJuOiBMYXp5LnN0cmluZyh7IHByb2R1Y2U6ICgpID0+IHRoaXMuZXhlY3V0aW9uUm9sZSAmJiB0aGlzLmV4ZWN1dGlvblJvbGUucm9sZUFybiB9KSxcbiAgICAgICAgICAgIGZhbWlseTogdGhpcy5mYW1pbHksXG4gICAgICAgICAgICB0YXNrUm9sZUFybjogdGhpcy50YXNrUm9sZS5yb2xlQXJuLFxuICAgICAgICAgICAgcmVxdWlyZXNDb21wYXRpYmlsaXRpZXM6IFtcbiAgICAgICAgICAgICAgICAuLi4oaXNFYzJDb21wYXRpYmxlKHByb3BzLmNvbXBhdGliaWxpdHkpID8gWydFQzInXSA6IFtdKSxcbiAgICAgICAgICAgICAgICAuLi4oaXNGYXJnYXRlQ29tcGF0aWJsZShwcm9wcy5jb21wYXRpYmlsaXR5KSA/IFsnRkFSR0FURSddIDogW10pLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIG5ldHdvcmtNb2RlOiB0aGlzLnJlbmRlck5ldHdvcmtNb2RlKHRoaXMubmV0d29ya01vZGUpLFxuICAgICAgICAgICAgcGxhY2VtZW50Q29uc3RyYWludHM6IExhenkuYW55KHtcbiAgICAgICAgICAgICAgICBwcm9kdWNlOiAoKSA9PiAhaXNGYXJnYXRlQ29tcGF0aWJsZSh0aGlzLmNvbXBhdGliaWxpdHkpID8gdGhpcy5wbGFjZW1lbnRDb25zdHJhaW50cyA6IHVuZGVmaW5lZCxcbiAgICAgICAgICAgIH0sIHsgb21pdEVtcHR5QXJyYXk6IHRydWUgfSksXG4gICAgICAgICAgICBwcm94eUNvbmZpZ3VyYXRpb246IHByb3BzLnByb3h5Q29uZmlndXJhdGlvbiA/IHByb3BzLnByb3h5Q29uZmlndXJhdGlvbi5iaW5kKHRoaXMuc3RhY2ssIHRoaXMpIDogdW5kZWZpbmVkLFxuICAgICAgICAgICAgY3B1OiBwcm9wcy5jcHUsXG4gICAgICAgICAgICBtZW1vcnk6IHByb3BzLm1lbW9yeU1pQixcbiAgICAgICAgICAgIGlwY01vZGU6IHByb3BzLmlwY01vZGUsXG4gICAgICAgICAgICBwaWRNb2RlOiBwcm9wcy5waWRNb2RlLFxuICAgICAgICAgICAgaW5mZXJlbmNlQWNjZWxlcmF0b3JzOiBMYXp5LmFueSh7XG4gICAgICAgICAgICAgICAgcHJvZHVjZTogKCkgPT4gIWlzRmFyZ2F0ZUNvbXBhdGlibGUodGhpcy5jb21wYXRpYmlsaXR5KSA/IHRoaXMucmVuZGVySW5mZXJlbmNlQWNjZWxlcmF0b3JzKCkgOiB1bmRlZmluZWQsXG4gICAgICAgICAgICB9LCB7IG9taXRFbXB0eUFycmF5OiB0cnVlIH0pLFxuICAgICAgICB9KTtcbiAgICAgICAgaWYgKHByb3BzLnBsYWNlbWVudENvbnN0cmFpbnRzKSB7XG4gICAgICAgICAgICBwcm9wcy5wbGFjZW1lbnRDb25zdHJhaW50cy5mb3JFYWNoKHBjID0+IHRoaXMuYWRkUGxhY2VtZW50Q29uc3RyYWludChwYykpO1xuICAgICAgICB9XG4gICAgICAgIHRoaXMudGFza0RlZmluaXRpb25Bcm4gPSB0YXNrRGVmLnJlZjtcbiAgICB9XG4gICAgcHVibGljIGdldCBleGVjdXRpb25Sb2xlKCk6IGlhbS5JUm9sZSB8IHVuZGVmaW5lZCB7XG4gICAgICAgIHJldHVybiB0aGlzLl9leGVjdXRpb25Sb2xlO1xuICAgIH1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIGdldCBpbmZlcmVuY2VBY2NlbGVyYXRvcnMoKTogSW5mZXJlbmNlQWNjZWxlcmF0b3JbXSB7XG4gICAgICAgIHJldHVybiB0aGlzLl9pbmZlcmVuY2VBY2NlbGVyYXRvcnM7XG4gICAgfVxuICAgIHByaXZhdGUgcmVuZGVyVm9sdW1lcygpOiBDZm5UYXNrRGVmaW5pdGlvbi5Wb2x1bWVQcm9wZXJ0eVtdIHtcbiAgICAgICAgcmV0dXJuIHRoaXMudm9sdW1lcy5tYXAocmVuZGVyVm9sdW1lKTtcbiAgICAgICAgZnVuY3Rpb24gcmVuZGVyVm9sdW1lKHNwZWM6IFZvbHVtZSk6IENmblRhc2tEZWZpbml0aW9uLlZvbHVtZVByb3BlcnR5IHtcbiAgICAgICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICAgICAgaG9zdDogc3BlYy5ob3N0LFxuICAgICAgICAgICAgICAgIG5hbWU6IHNwZWMubmFtZSxcbiAgICAgICAgICAgICAgICBkb2NrZXJWb2x1bWVDb25maWd1cmF0aW9uOiBzcGVjLmRvY2tlclZvbHVtZUNvbmZpZ3VyYXRpb24gJiYge1xuICAgICAgICAgICAgICAgICAgICBhdXRvcHJvdmlzaW9uOiBzcGVjLmRvY2tlclZvbHVtZUNvbmZpZ3VyYXRpb24uYXV0b3Byb3Zpc2lvbixcbiAgICAgICAgICAgICAgICAgICAgZHJpdmVyOiBzcGVjLmRvY2tlclZvbHVtZUNvbmZpZ3VyYXRpb24uZHJpdmVyLFxuICAgICAgICAgICAgICAgICAgICBkcml2ZXJPcHRzOiBzcGVjLmRvY2tlclZvbHVtZUNvbmZpZ3VyYXRpb24uZHJpdmVyT3B0cyxcbiAgICAgICAgICAgICAgICAgICAgbGFiZWxzOiBzcGVjLmRvY2tlclZvbHVtZUNvbmZpZ3VyYXRpb24ubGFiZWxzLFxuICAgICAgICAgICAgICAgICAgICBzY29wZTogc3BlYy5kb2NrZXJWb2x1bWVDb25maWd1cmF0aW9uLnNjb3BlLFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgZWZzVm9sdW1lQ29uZmlndXJhdGlvbjogc3BlYy5lZnNWb2x1bWVDb25maWd1cmF0aW9uICYmIHtcbiAgICAgICAgICAgICAgICAgICAgZmlsZVN5c3RlbUlkOiBzcGVjLmVmc1ZvbHVtZUNvbmZpZ3VyYXRpb24uZmlsZVN5c3RlbUlkLFxuICAgICAgICAgICAgICAgICAgICBhdXRob3JpemF0aW9uQ29uZmlnOiBzcGVjLmVmc1ZvbHVtZUNvbmZpZ3VyYXRpb24uYXV0aG9yaXphdGlvbkNvbmZpZyxcbiAgICAgICAgICAgICAgICAgICAgcm9vdERpcmVjdG9yeTogc3BlYy5lZnNWb2x1bWVDb25maWd1cmF0aW9uLnJvb3REaXJlY3RvcnksXG4gICAgICAgICAgICAgICAgICAgIHRyYW5zaXRFbmNyeXB0aW9uOiBzcGVjLmVmc1ZvbHVtZUNvbmZpZ3VyYXRpb24udHJhbnNpdEVuY3J5cHRpb24sXG4gICAgICAgICAgICAgICAgICAgIHRyYW5zaXRFbmNyeXB0aW9uUG9ydDogc3BlYy5lZnNWb2x1bWVDb25maWd1cmF0aW9uLnRyYW5zaXRFbmNyeXB0aW9uUG9ydCxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgfTtcbiAgICAgICAgfVxuICAgIH1cbiAgICBwcml2YXRlIHJlbmRlckluZmVyZW5jZUFjY2VsZXJhdG9ycygpOiBDZm5UYXNrRGVmaW5pdGlvbi5JbmZlcmVuY2VBY2NlbGVyYXRvclByb3BlcnR5W10ge1xuICAgICAgICByZXR1cm4gdGhpcy5faW5mZXJlbmNlQWNjZWxlcmF0b3JzLm1hcChyZW5kZXJJbmZlcmVuY2VBY2NlbGVyYXRvcik7XG4gICAgICAgIGZ1bmN0aW9uIHJlbmRlckluZmVyZW5jZUFjY2VsZXJhdG9yKGluZmVyZW5jZUFjY2VsZXJhdG9yOiBJbmZlcmVuY2VBY2NlbGVyYXRvcik6IENmblRhc2tEZWZpbml0aW9uLkluZmVyZW5jZUFjY2VsZXJhdG9yUHJvcGVydHkge1xuICAgICAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgICAgICBkZXZpY2VOYW1lOiBpbmZlcmVuY2VBY2NlbGVyYXRvci5kZXZpY2VOYW1lLFxuICAgICAgICAgICAgICAgIGRldmljZVR5cGU6IGluZmVyZW5jZUFjY2VsZXJhdG9yLmRldmljZVR5cGUsXG4gICAgICAgICAgICB9O1xuICAgICAgICB9XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFZhbGlkYXRlIHRoZSBleGlzdGVuY2Ugb2YgdGhlIGlucHV0IHRhcmdldCBhbmQgc2V0IGRlZmF1bHQgdmFsdWVzLlxuICAgICAqXG4gICAgICogQGludGVybmFsXG4gICAgICovXG4gICAgcHVibGljIF92YWxpZGF0ZVRhcmdldChvcHRpb25zOiBMb2FkQmFsYW5jZXJUYXJnZXRPcHRpb25zKTogTG9hZEJhbGFuY2VyVGFyZ2V0IHtcbiAgICAgICAgY29uc3QgdGFyZ2V0Q29udGFpbmVyID0gdGhpcy5maW5kQ29udGFpbmVyKG9wdGlvbnMuY29udGFpbmVyTmFtZSk7XG4gICAgICAgIGlmICh0YXJnZXRDb250YWluZXIgPT09IHVuZGVmaW5lZCkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBObyBjb250YWluZXIgbmFtZWQgJyR7b3B0aW9ucy5jb250YWluZXJOYW1lfScuIERpZCB5b3UgY2FsbCBcImFkZENvbnRhaW5lcigpXCI/YCk7XG4gICAgICAgIH1cbiAgICAgICAgY29uc3QgdGFyZ2V0UHJvdG9jb2wgPSBvcHRpb25zLnByb3RvY29sIHx8IFByb3RvY29sLlRDUDtcbiAgICAgICAgY29uc3QgdGFyZ2V0Q29udGFpbmVyUG9ydCA9IG9wdGlvbnMuY29udGFpbmVyUG9ydCB8fCB0YXJnZXRDb250YWluZXIuY29udGFpbmVyUG9ydDtcbiAgICAgICAgY29uc3QgcG9ydE1hcHBpbmcgPSB0YXJnZXRDb250YWluZXIuZmluZFBvcnRNYXBwaW5nKHRhcmdldENvbnRhaW5lclBvcnQsIHRhcmdldFByb3RvY29sKTtcbiAgICAgICAgaWYgKHBvcnRNYXBwaW5nID09PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICAgIC8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBtYXgtbGVuXG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYENvbnRhaW5lciAnJHt0YXJnZXRDb250YWluZXJ9JyBoYXMgbm8gbWFwcGluZyBmb3IgcG9ydCAke29wdGlvbnMuY29udGFpbmVyUG9ydH0gYW5kIHByb3RvY29sICR7dGFyZ2V0UHJvdG9jb2x9LiBEaWQgeW91IGNhbGwgXCJjb250YWluZXIuYWRkUG9ydE1hcHBpbmdzKClcIj9gKTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgICAgY29udGFpbmVyTmFtZTogb3B0aW9ucy5jb250YWluZXJOYW1lLFxuICAgICAgICAgICAgcG9ydE1hcHBpbmcsXG4gICAgICAgIH07XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFJldHVybnMgdGhlIHBvcnQgcmFuZ2UgdG8gYmUgb3BlbmVkIHRoYXQgbWF0Y2ggdGhlIHByb3ZpZGVkIGNvbnRhaW5lciBuYW1lIGFuZCBjb250YWluZXIgcG9ydC5cbiAgICAgKlxuICAgICAqIEBpbnRlcm5hbFxuICAgICAqL1xuICAgIHB1YmxpYyBfcG9ydFJhbmdlRnJvbVBvcnRNYXBwaW5nKHBvcnRNYXBwaW5nOiBQb3J0TWFwcGluZyk6IGVjMi5Qb3J0IHtcbiAgICAgICAgaWYgKHBvcnRNYXBwaW5nLmhvc3RQb3J0ICE9PSB1bmRlZmluZWQgJiYgcG9ydE1hcHBpbmcuaG9zdFBvcnQgIT09IDApIHtcbiAgICAgICAgICAgIHJldHVybiBwb3J0TWFwcGluZy5wcm90b2NvbCA9PT0gUHJvdG9jb2wuVURQID8gZWMyLlBvcnQudWRwKHBvcnRNYXBwaW5nLmhvc3RQb3J0KSA6IGVjMi5Qb3J0LnRjcChwb3J0TWFwcGluZy5ob3N0UG9ydCk7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKHRoaXMubmV0d29ya01vZGUgPT09IE5ldHdvcmtNb2RlLkJSSURHRSB8fCB0aGlzLm5ldHdvcmtNb2RlID09PSBOZXR3b3JrTW9kZS5OQVQpIHtcbiAgICAgICAgICAgIHJldHVybiBFUEhFTUVSQUxfUE9SVF9SQU5HRTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gcG9ydE1hcHBpbmcucHJvdG9jb2wgPT09IFByb3RvY29sLlVEUCA/IGVjMi5Qb3J0LnVkcChwb3J0TWFwcGluZy5jb250YWluZXJQb3J0KSA6IGVjMi5Qb3J0LnRjcChwb3J0TWFwcGluZy5jb250YWluZXJQb3J0KTtcbiAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBhZGRUb1Rhc2tSb2xlUG9saWN5KHN0YXRlbWVudDogaWFtLlBvbGljeVN0YXRlbWVudCkge1xuICAgICAgICB0aGlzLnRhc2tSb2xlLmFkZFRvUHJpbmNpcGFsUG9saWN5KHN0YXRlbWVudCk7XG4gICAgfVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIGFkZFRvRXhlY3V0aW9uUm9sZVBvbGljeShzdGF0ZW1lbnQ6IGlhbS5Qb2xpY3lTdGF0ZW1lbnQpIHtcbiAgICAgICAgdGhpcy5vYnRhaW5FeGVjdXRpb25Sb2xlKCkuYWRkVG9QcmluY2lwYWxQb2xpY3koc3RhdGVtZW50KTtcbiAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIGFkZENvbnRhaW5lcihpZDogc3RyaW5nLCBwcm9wczogQ29udGFpbmVyRGVmaW5pdGlvbk9wdGlvbnMpIHtcbiAgICAgICAgcmV0dXJuIG5ldyBDb250YWluZXJEZWZpbml0aW9uKHRoaXMsIGlkLCB7IHRhc2tEZWZpbml0aW9uOiB0aGlzLCAuLi5wcm9wcyB9KTtcbiAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIGFkZEZpcmVsZW5zTG9nUm91dGVyKGlkOiBzdHJpbmcsIHByb3BzOiBGaXJlbGVuc0xvZ1JvdXRlckRlZmluaXRpb25PcHRpb25zKSB7XG4gICAgICAgIC8vIG9ubHkgb25lIGZpcmVsZW5zIGxvZyByb3V0ZXIgaXMgYWxsb3dlZCBpbiBlYWNoIHRhc2suXG4gICAgICAgIGlmICh0aGlzLmNvbnRhaW5lcnMuZmluZCh4ID0+IHggaW5zdGFuY2VvZiBGaXJlbGVuc0xvZ1JvdXRlcikpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignRmlyZWxlbnMgbG9nIHJvdXRlciBpcyBhbHJlYWR5IGFkZGVkIGluIHRoaXMgdGFzay4nKTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gbmV3IEZpcmVsZW5zTG9nUm91dGVyKHRoaXMsIGlkLCB7IHRhc2tEZWZpbml0aW9uOiB0aGlzLCAuLi5wcm9wcyB9KTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogTGlua3MgYSBjb250YWluZXIgdG8gdGhpcyB0YXNrIGRlZmluaXRpb24uXG4gICAgICogQGludGVybmFsXG4gICAgICovXG4gICAgcHVibGljIF9saW5rQ29udGFpbmVyKGNvbnRhaW5lcjogQ29udGFpbmVyRGVmaW5pdGlvbikge1xuICAgICAgICB0aGlzLmNvbnRhaW5lcnMucHVzaChjb250YWluZXIpO1xuICAgICAgICBpZiAodGhpcy5kZWZhdWx0Q29udGFpbmVyID09PSB1bmRlZmluZWQgJiYgY29udGFpbmVyLmVzc2VudGlhbCkge1xuICAgICAgICAgICAgdGhpcy5kZWZhdWx0Q29udGFpbmVyID0gY29udGFpbmVyO1xuICAgICAgICB9XG4gICAgICAgIGlmIChjb250YWluZXIucmVmZXJlbmNlc1NlY3JldEpzb25GaWVsZCkge1xuICAgICAgICAgICAgdGhpcy5fcmVmZXJlbmNlc1NlY3JldEpzb25GaWVsZCA9IHRydWU7XG4gICAgICAgIH1cbiAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgYWRkVm9sdW1lKHZvbHVtZTogVm9sdW1lKSB7XG4gICAgICAgIHRoaXMudm9sdW1lcy5wdXNoKHZvbHVtZSk7XG4gICAgfVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgYWRkUGxhY2VtZW50Q29uc3RyYWludChjb25zdHJhaW50OiBQbGFjZW1lbnRDb25zdHJhaW50KSB7XG4gICAgICAgIGlmIChpc0ZhcmdhdGVDb21wYXRpYmxlKHRoaXMuY29tcGF0aWJpbGl0eSkpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignQ2Fubm90IHNldCBwbGFjZW1lbnQgY29uc3RyYWludHMgb24gdGFza3MgdGhhdCBydW4gb24gRmFyZ2F0ZScpO1xuICAgICAgICB9XG4gICAgICAgIHRoaXMucGxhY2VtZW50Q29uc3RyYWludHMucHVzaCguLi5jb25zdHJhaW50LnRvSnNvbigpKTtcbiAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBhZGRFeHRlbnNpb24oZXh0ZW5zaW9uOiBJVGFza0RlZmluaXRpb25FeHRlbnNpb24pIHtcbiAgICAgICAgZXh0ZW5zaW9uLmV4dGVuZCh0aGlzKTtcbiAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIGFkZEluZmVyZW5jZUFjY2VsZXJhdG9yKGluZmVyZW5jZUFjY2VsZXJhdG9yOiBJbmZlcmVuY2VBY2NlbGVyYXRvcikge1xuICAgICAgICBpZiAoaXNGYXJnYXRlQ29tcGF0aWJsZSh0aGlzLmNvbXBhdGliaWxpdHkpKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0Nhbm5vdCB1c2UgaW5mZXJlbmNlIGFjY2VsZXJhdG9ycyBvbiB0YXNrcyB0aGF0IHJ1biBvbiBGYXJnYXRlJyk7XG4gICAgICAgIH1cbiAgICAgICAgdGhpcy5faW5mZXJlbmNlQWNjZWxlcmF0b3JzLnB1c2goaW5mZXJlbmNlQWNjZWxlcmF0b3IpO1xuICAgIH1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBvYnRhaW5FeGVjdXRpb25Sb2xlKCk6IGlhbS5JUm9sZSB7XG4gICAgICAgIGlmICghdGhpcy5fZXhlY3V0aW9uUm9sZSkge1xuICAgICAgICAgICAgdGhpcy5fZXhlY3V0aW9uUm9sZSA9IG5ldyBpYW0uUm9sZSh0aGlzLCAnRXhlY3V0aW9uUm9sZScsIHtcbiAgICAgICAgICAgICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnZWNzLXRhc2tzLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgICAgICAgIH0pO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiB0aGlzLl9leGVjdXRpb25Sb2xlO1xuICAgIH1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgZ2V0IHJlZmVyZW5jZXNTZWNyZXRKc29uRmllbGQoKTogYm9vbGVhbiB8IHVuZGVmaW5lZCB7XG4gICAgICAgIHJldHVybiB0aGlzLl9yZWZlcmVuY2VzU2VjcmV0SnNvbkZpZWxkO1xuICAgIH1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHJvdGVjdGVkIHZhbGlkYXRlKCk6IHN0cmluZ1tdIHtcbiAgICAgICAgY29uc3QgcmV0ID0gc3VwZXIudmFsaWRhdGUoKTtcbiAgICAgICAgaWYgKGlzRWMyQ29tcGF0aWJsZSh0aGlzLmNvbXBhdGliaWxpdHkpKSB7XG4gICAgICAgICAgICAvLyBFQzIgbW9kZSB2YWxpZGF0aW9uc1xuICAgICAgICAgICAgLy8gQ29udGFpbmVyIHNpemVzXG4gICAgICAgICAgICBmb3IgKGNvbnN0IGNvbnRhaW5lciBvZiB0aGlzLmNvbnRhaW5lcnMpIHtcbiAgICAgICAgICAgICAgICBpZiAoIWNvbnRhaW5lci5tZW1vcnlMaW1pdFNwZWNpZmllZCkge1xuICAgICAgICAgICAgICAgICAgICByZXQucHVzaChgRUNTIENvbnRhaW5lciAke2NvbnRhaW5lci5jb250YWluZXJOYW1lfSBtdXN0IGhhdmUgYXQgbGVhc3Qgb25lIG9mICdtZW1vcnlMaW1pdE1pQicgb3IgJ21lbW9yeVJlc2VydmF0aW9uTWlCJyBzcGVjaWZpZWRgKTtcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHJldDtcbiAgICB9XG4gICAgLyoqXG4gICAgICogUmV0dXJucyB0aGUgY29udGFpbmVyIHRoYXQgbWF0Y2ggdGhlIHByb3ZpZGVkIGNvbnRhaW5lck5hbWUuXG4gICAgICovXG4gICAgcHJpdmF0ZSBmaW5kQ29udGFpbmVyKGNvbnRhaW5lck5hbWU6IHN0cmluZyk6IENvbnRhaW5lckRlZmluaXRpb24gfCB1bmRlZmluZWQge1xuICAgICAgICByZXR1cm4gdGhpcy5jb250YWluZXJzLmZpbmQoYyA9PiBjLmNvbnRhaW5lck5hbWUgPT09IGNvbnRhaW5lck5hbWUpO1xuICAgIH1cbiAgICBwcml2YXRlIHJlbmRlck5ldHdvcmtNb2RlKG5ldHdvcmtNb2RlOiBOZXR3b3JrTW9kZSk6IHN0cmluZyB8IHVuZGVmaW5lZCB7XG4gICAgICAgIHJldHVybiAobmV0d29ya01vZGUgPT09IE5ldHdvcmtNb2RlLk5BVCkgPyB1bmRlZmluZWQgOiBuZXR3b3JrTW9kZTtcbiAgICB9XG4gICAgcHJpdmF0ZSByZW5kZXJDb250YWluZXJzKCkge1xuICAgICAgICAvLyBhZGQgZmlyZWxlbnMgbG9nIHJvdXRlciBjb250YWluZXIgaWYgYW55IGFwcGxpY2F0aW9uIGNvbnRhaW5lciBpcyB1c2luZyBmaXJlbGVucyBsb2cgZHJpdmVyLFxuICAgICAgICAvLyBhbHNvIGNoZWNrIGlmIGFscmVhZHkgY3JlYXRlZCBsb2cgcm91dGVyIGNvbnRhaW5lclxuICAgICAgICBmb3IgKGNvbnN0IGNvbnRhaW5lciBvZiB0aGlzLmNvbnRhaW5lcnMpIHtcbiAgICAgICAgICAgIGlmIChjb250YWluZXIubG9nRHJpdmVyQ29uZmlnICYmIGNvbnRhaW5lci5sb2dEcml2ZXJDb25maWcubG9nRHJpdmVyID09PSAnYXdzZmlyZWxlbnMnXG4gICAgICAgICAgICAgICAgJiYgIXRoaXMuY29udGFpbmVycy5maW5kKHggPT4geCBpbnN0YW5jZW9mIEZpcmVsZW5zTG9nUm91dGVyKSkge1xuICAgICAgICAgICAgICAgIHRoaXMuYWRkRmlyZWxlbnNMb2dSb3V0ZXIoJ2xvZy1yb3V0ZXInLCB7XG4gICAgICAgICAgICAgICAgICAgIGltYWdlOiBvYnRhaW5EZWZhdWx0Rmx1ZW50Qml0RUNSSW1hZ2UodGhpcywgY29udGFpbmVyLmxvZ0RyaXZlckNvbmZpZyksXG4gICAgICAgICAgICAgICAgICAgIGZpcmVsZW5zQ29uZmlnOiB7XG4gICAgICAgICAgICAgICAgICAgICAgICB0eXBlOiBGaXJlbGVuc0xvZ1JvdXRlclR5cGUuRkxVRU5UQklULFxuICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgICBsb2dnaW5nOiBuZXcgQXdzTG9nRHJpdmVyKHsgc3RyZWFtUHJlZml4OiAnZmlyZWxlbnMnIH0pLFxuICAgICAgICAgICAgICAgICAgICBtZW1vcnlSZXNlcnZhdGlvbk1pQjogNTAsXG4gICAgICAgICAgICAgICAgfSk7XG4gICAgICAgICAgICAgICAgYnJlYWs7XG4gICAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHRoaXMuY29udGFpbmVycy5tYXAoeCA9PiB4LnJlbmRlckNvbnRhaW5lckRlZmluaXRpb24oKSk7XG4gICAgfVxufVxuLyoqXG4gKiBUaGUgcG9ydCByYW5nZSB0byBvcGVuIHVwIGZvciBkeW5hbWljIHBvcnQgbWFwcGluZ1xuICovXG5jb25zdCBFUEhFTUVSQUxfUE9SVF9SQU5HRSA9IGVjMi5Qb3J0LnRjcFJhbmdlKDMyNzY4LCA2NTUzNSk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBlbnVtIE5ldHdvcmtNb2RlIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIE5PTkUgPSAnbm9uZScsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBCUklER0UgPSAnYnJpZGdlJyxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgQVdTX1ZQQyA9ICdhd3N2cGMnLFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBIT1NUID0gJ2hvc3QnLFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgTkFUID0gJ25hdCdcbn1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBlbnVtIElwY01vZGUge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIE5PTkUgPSAnbm9uZScsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBIT1NUID0gJ2hvc3QnLFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIFRBU0sgPSAndGFzaydcbn1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgZW51bSBQaWRNb2RlIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBIT1NUID0gJ2hvc3QnLFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBUQVNLID0gJ3Rhc2snXG59XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIEluZmVyZW5jZUFjY2VsZXJhdG9yIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IGRldmljZU5hbWU/OiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IGRldmljZVR5cGU/OiBzdHJpbmc7XG59XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgVm9sdW1lIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgaG9zdD86IEhvc3Q7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBuYW1lOiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBkb2NrZXJWb2x1bWVDb25maWd1cmF0aW9uPzogRG9ja2VyVm9sdW1lQ29uZmlndXJhdGlvbjtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgZWZzVm9sdW1lQ29uZmlndXJhdGlvbj86IEVmc1ZvbHVtZUNvbmZpZ3VyYXRpb247XG59XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIEhvc3Qge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgc291cmNlUGF0aD86IHN0cmluZztcbn1cbi8qKlxuICogUHJvcGVydGllcyBmb3IgYW4gRUNTIHRhcmdldC5cbiAqXG4gKiBAaW50ZXJuYWxcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBMb2FkQmFsYW5jZXJUYXJnZXQge1xuICAgIC8qKlxuICAgICAqIFRoZSBuYW1lIG9mIHRoZSBjb250YWluZXIuXG4gICAgICovXG4gICAgcmVhZG9ubHkgY29udGFpbmVyTmFtZTogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSBwb3J0IG1hcHBpbmcgb2YgdGhlIHRhcmdldC5cbiAgICAgKi9cbiAgICByZWFkb25seSBwb3J0TWFwcGluZzogUG9ydE1hcHBpbmc7XG59XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIExvYWRCYWxhbmNlclRhcmdldE9wdGlvbnMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IGNvbnRhaW5lck5hbWU6IHN0cmluZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IGNvbnRhaW5lclBvcnQ/OiBudW1iZXI7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBwcm90b2NvbD86IFByb3RvY29sO1xufVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIERvY2tlclZvbHVtZUNvbmZpZ3VyYXRpb24ge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgYXV0b3Byb3Zpc2lvbj86IGJvb2xlYW47XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgZHJpdmVyOiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgZHJpdmVyT3B0cz86IHtcbiAgICAgICAgW2tleTogc3RyaW5nXTogc3RyaW5nO1xuICAgIH07XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IGxhYmVscz86IHtcbiAgICAgICAgW2tleTogc3RyaW5nXTogc3RyaW5nO1xuICAgIH07XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgc2NvcGU6IFNjb3BlO1xufVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBBdXRob3JpemF0aW9uQ29uZmlnIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBhY2Nlc3NQb2ludElkPzogc3RyaW5nO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgaWFtPzogc3RyaW5nO1xufVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgRWZzVm9sdW1lQ29uZmlndXJhdGlvbiB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBmaWxlU3lzdGVtSWQ6IHN0cmluZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IHJvb3REaXJlY3Rvcnk/OiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgdHJhbnNpdEVuY3J5cHRpb24/OiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSB0cmFuc2l0RW5jcnlwdGlvblBvcnQ/OiBudW1iZXI7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBhdXRob3JpemF0aW9uQ29uZmlnPzogQXV0aG9yaXphdGlvbkNvbmZpZztcbn1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBlbnVtIFNjb3BlIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgVEFTSyA9ICd0YXNrJyxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIFNIQVJFRCA9ICdzaGFyZWQnXG59XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgZW51bSBDb21wYXRpYmlsaXR5IHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBFQzIsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIEZBUkdBVEUsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIEVDMl9BTkRfRkFSR0FURVxufVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBJVGFza0RlZmluaXRpb25FeHRlbnNpb24ge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIGV4dGVuZCh0YXNrRGVmaW5pdGlvbjogVGFza0RlZmluaXRpb24pOiB2b2lkO1xufVxuLyoqXG4gKiBSZXR1cm4gdHJ1ZSBpZiB0aGUgZ2l2ZW4gdGFzayBkZWZpbml0aW9uIGNhbiBiZSBydW4gb24gYW4gRUMyIGNsdXN0ZXJcbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGlzRWMyQ29tcGF0aWJsZShjb21wYXRpYmlsaXR5OiBDb21wYXRpYmlsaXR5KTogYm9vbGVhbiB7XG4gICAgcmV0dXJuIFtDb21wYXRpYmlsaXR5LkVDMiwgQ29tcGF0aWJpbGl0eS5FQzJfQU5EX0ZBUkdBVEVdLmluY2x1ZGVzKGNvbXBhdGliaWxpdHkpO1xufVxuLyoqXG4gKiBSZXR1cm4gdHJ1ZSBpZiB0aGUgZ2l2ZW4gdGFzayBkZWZpbml0aW9uIGNhbiBiZSBydW4gb24gYSBGYXJnYXRlIGNsdXN0ZXJcbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGlzRmFyZ2F0ZUNvbXBhdGlibGUoY29tcGF0aWJpbGl0eTogQ29tcGF0aWJpbGl0eSk6IGJvb2xlYW4ge1xuICAgIHJldHVybiBbQ29tcGF0aWJpbGl0eS5GQVJHQVRFLCBDb21wYXRpYmlsaXR5LkVDMl9BTkRfRkFSR0FURV0uaW5jbHVkZXMoY29tcGF0aWJpbGl0eSk7XG59XG4iXX0=