"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Volume = exports.EbsDeviceVolumeType = exports.BlockDeviceVolume = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const aws_iam_1 = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const aws_kms_1 = require("../../aws-kms"); // Automatically re-written from '@aws-cdk/aws-kms'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const ec2_generated_1 = require("./ec2.generated");
/**
 * (experimental) Describes a block device mapping for an EC2 instance or Auto Scaling group.
 *
 * @experimental
 */
class BlockDeviceVolume {
    /**
     * @param ebsDevice EBS device info.
     * @param virtualName Virtual device name.
     * @experimental
     */
    constructor(ebsDevice, virtualName) {
        this.ebsDevice = ebsDevice;
        this.virtualName = virtualName;
    }
    /**
     * (experimental) Creates a new Elastic Block Storage device.
     *
     * @param volumeSize The volume size, in Gibibytes (GiB).
     * @param options additional device options.
     * @experimental
     */
    static ebs(volumeSize, options = {}) {
        return new this({ ...options, volumeSize });
    }
    /**
     * (experimental) Creates a new Elastic Block Storage device from an existing snapshot.
     *
     * @param snapshotId The snapshot ID of the volume to use.
     * @param options additional device options.
     * @experimental
     */
    static ebsFromSnapshot(snapshotId, options = {}) {
        return new this({ ...options, snapshotId });
    }
    /**
     * (experimental) Creates a virtual, ephemeral device.
     *
     * The name will be in the form ephemeral{volumeIndex}.
     *
     * @param volumeIndex the volume index.
     * @experimental
     */
    static ephemeral(volumeIndex) {
        if (volumeIndex < 0) {
            throw new Error(`volumeIndex must be a number starting from 0, got "${volumeIndex}"`);
        }
        return new this(undefined, `ephemeral${volumeIndex}`);
    }
}
exports.BlockDeviceVolume = BlockDeviceVolume;
_a = JSII_RTTI_SYMBOL_1;
BlockDeviceVolume[_a] = { fqn: "monocdk.aws_ec2.BlockDeviceVolume", version: "1.106.1" };
/**
 * (experimental) Supported EBS volume types for blockDevices.
 *
 * @experimental
 */
var EbsDeviceVolumeType;
(function (EbsDeviceVolumeType) {
    EbsDeviceVolumeType["STANDARD"] = "standard";
    EbsDeviceVolumeType["IO1"] = "io1";
    EbsDeviceVolumeType["IO2"] = "io2";
    EbsDeviceVolumeType["GP2"] = "gp2";
    EbsDeviceVolumeType["GP3"] = "gp3";
    EbsDeviceVolumeType["ST1"] = "st1";
    EbsDeviceVolumeType["SC1"] = "sc1";
    /**
     * General purpose SSD volume (GP2) that balances price and performance for a wide variety of workloads.
     */
    EbsDeviceVolumeType["GENERAL_PURPOSE_SSD"] = "gp2";
    /**
     * General purpose SSD volume (GP3) that balances price and performance for a wide variety of workloads.
     */
    EbsDeviceVolumeType["GENERAL_PURPOSE_SSD_GP3"] = "gp3";
    /**
     * Highest-performance SSD volume (IO1) for mission-critical low-latency or high-throughput workloads.
     */
    EbsDeviceVolumeType["PROVISIONED_IOPS_SSD"] = "io1";
    /**
     * Highest-performance SSD volume (IO2) for mission-critical low-latency or high-throughput workloads.
     */
    EbsDeviceVolumeType["PROVISIONED_IOPS_SSD_IO2"] = "io2";
    /**
     * Low-cost HDD volume designed for frequently accessed, throughput-intensive workloads.
     */
    EbsDeviceVolumeType["THROUGHPUT_OPTIMIZED_HDD"] = "st1";
    /**
     * Lowest cost HDD volume designed for less frequently accessed workloads.
     */
    EbsDeviceVolumeType["COLD_HDD"] = "sc1";
    /**
     * Magnetic volumes are backed by magnetic drives and are suited for workloads where data is accessed infrequently, and scenarios where low-cost
     * storage for small volume sizes is important.
     */
    EbsDeviceVolumeType["MAGNETIC"] = "standard";
})(EbsDeviceVolumeType = exports.EbsDeviceVolumeType || (exports.EbsDeviceVolumeType = {}));
/**
 * Common behavior of Volumes. Users should not use this class directly, and instead use ``Volume``.
 */
class VolumeBase extends core_1.Resource {
    /**
     * (experimental) Grants permission to attach this Volume to an instance.
     *
     * CAUTION: Granting an instance permission to attach to itself using this method will lead to
     * an unresolvable circular reference between the instance role and the instance.
     * Use {@link IVolume.grantAttachVolumeToSelf} to grant an instance permission to attach this
     * volume to itself.
     *
     * @experimental
     */
    grantAttachVolume(grantee, instances) {
        const result = aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: ['ec2:AttachVolume'],
            resourceArns: this.collectGrantResourceArns(instances),
        });
        if (this.encryptionKey) {
            // When attaching a volume, the EC2 Service will need to grant to itself permission
            // to be able to decrypt the encryption key. We restrict the CreateGrant for principle
            // of least privilege, in accordance with best practices.
            // See: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html#ebs-encryption-permissions
            const kmsGrant = this.encryptionKey.grant(grantee, 'kms:CreateGrant');
            kmsGrant.principalStatement.addConditions({
                Bool: { 'kms:GrantIsForAWSResource': true },
                StringEquals: {
                    'kms:ViaService': `ec2.${core_1.Stack.of(this).region}.amazonaws.com`,
                    'kms:GrantConstraintType': 'EncryptionContextSubset',
                },
            });
        }
        return result;
    }
    /**
     * (experimental) Grants permission to attach the Volume by a ResourceTag condition.
     *
     * If you are looking to
     * grant an Instance, AutoScalingGroup, EC2-Fleet, SpotFleet, ECS host, etc the ability to attach
     * this volume to **itself** then this is the method you want to use.
     *
     * This is implemented by adding a Tag with key `VolumeGrantAttach-<suffix>` to the given
     * constructs and this Volume, and then conditioning the Grant such that the grantee is only
     * given the ability to AttachVolume if both the Volume and the destination Instance have that
     * tag applied to them.
     *
     * @experimental
     */
    grantAttachVolumeByResourceTag(grantee, constructs, tagKeySuffix) {
        const tagValue = this.calculateResourceTagValue([this, ...constructs]);
        const tagKey = `VolumeGrantAttach-${tagKeySuffix !== null && tagKeySuffix !== void 0 ? tagKeySuffix : tagValue.slice(0, 10).toUpperCase()}`;
        const grantCondition = {};
        grantCondition[`ec2:ResourceTag/${tagKey}`] = tagValue;
        const result = this.grantAttachVolume(grantee);
        result.principalStatement.addCondition('ForAnyValue:StringEquals', grantCondition);
        // The ResourceTag condition requires that all resources involved in the operation have
        // the given tag, so we tag this and all constructs given.
        core_1.Tags.of(this).add(tagKey, tagValue);
        constructs.forEach(construct => core_1.Tags.of(construct).add(tagKey, tagValue));
        return result;
    }
    /**
     * (experimental) Grants permission to detach this Volume from an instance CAUTION: Granting an instance permission to detach from itself using this method will lead to an unresolvable circular reference between the instance role and the instance.
     *
     * Use {@link IVolume.grantDetachVolumeFromSelf} to grant an instance permission to detach this
     * volume from itself.
     *
     * @experimental
     */
    grantDetachVolume(grantee, instances) {
        const result = aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: ['ec2:DetachVolume'],
            resourceArns: this.collectGrantResourceArns(instances),
        });
        // Note: No encryption key permissions are required to detach an encrypted volume.
        return result;
    }
    /**
     * (experimental) Grants permission to detach the Volume by a ResourceTag condition.
     *
     * This is implemented via the same mechanism as {@link IVolume.grantAttachVolumeByResourceTag},
     * and is subject to the same conditions.
     *
     * @experimental
     */
    grantDetachVolumeByResourceTag(grantee, constructs, tagKeySuffix) {
        const tagValue = this.calculateResourceTagValue([this, ...constructs]);
        const tagKey = `VolumeGrantDetach-${tagKeySuffix !== null && tagKeySuffix !== void 0 ? tagKeySuffix : tagValue.slice(0, 10).toUpperCase()}`;
        const grantCondition = {};
        grantCondition[`ec2:ResourceTag/${tagKey}`] = tagValue;
        const result = this.grantDetachVolume(grantee);
        result.principalStatement.addCondition('ForAnyValue:StringEquals', grantCondition);
        // The ResourceTag condition requires that all resources involved in the operation have
        // the given tag, so we tag this and all constructs given.
        core_1.Tags.of(this).add(tagKey, tagValue);
        constructs.forEach(construct => core_1.Tags.of(construct).add(tagKey, tagValue));
        return result;
    }
    collectGrantResourceArns(instances) {
        const stack = core_1.Stack.of(this);
        const resourceArns = [
            `arn:${stack.partition}:ec2:${stack.region}:${stack.account}:volume/${this.volumeId}`,
        ];
        const instanceArnPrefix = `arn:${stack.partition}:ec2:${stack.region}:${stack.account}:instance`;
        if (instances) {
            instances.forEach(instance => resourceArns.push(`${instanceArnPrefix}/${instance === null || instance === void 0 ? void 0 : instance.instanceId}`));
        }
        else {
            resourceArns.push(`${instanceArnPrefix}/*`);
        }
        return resourceArns;
    }
    calculateResourceTagValue(constructs) {
        const md5 = crypto.createHash('md5');
        constructs.forEach(construct => md5.update(core_1.Names.uniqueId(construct)));
        return md5.digest('hex');
    }
}
/**
 * (experimental) Creates a new EBS Volume in AWS EC2.
 *
 * @experimental
 */
class Volume extends VolumeBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _c, _d, _e, _f, _g;
        super(scope, id, {
            physicalName: props.volumeName,
        });
        this.validateProps(props);
        const resource = new ec2_generated_1.CfnVolume(this, 'Resource', {
            availabilityZone: props.availabilityZone,
            autoEnableIo: props.autoEnableIo,
            encrypted: props.encrypted,
            kmsKeyId: (_c = props.encryptionKey) === null || _c === void 0 ? void 0 : _c.keyArn,
            iops: props.iops,
            multiAttachEnabled: (_d = props.enableMultiAttach) !== null && _d !== void 0 ? _d : false,
            size: (_e = props.size) === null || _e === void 0 ? void 0 : _e.toGibibytes({ rounding: core_1.SizeRoundingBehavior.FAIL }),
            snapshotId: props.snapshotId,
            volumeType: (_f = props.volumeType) !== null && _f !== void 0 ? _f : EbsDeviceVolumeType.GENERAL_PURPOSE_SSD,
        });
        resource.applyRemovalPolicy(props.removalPolicy);
        this.volumeId = resource.ref;
        this.availabilityZone = props.availabilityZone;
        this.encryptionKey = props.encryptionKey;
        if (this.encryptionKey) {
            // Per: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html#ebs-encryption-requirements
            const principal = new aws_kms_1.ViaServicePrincipal(`ec2.${core_1.Stack.of(this).region}.amazonaws.com`, new aws_iam_1.AccountRootPrincipal()).withConditions({
                StringEquals: {
                    'kms:CallerAccount': core_1.Stack.of(this).account,
                },
            });
            const grant = this.encryptionKey.grant(principal, 
            // Describe & Generate are required to be able to create the CMK-encrypted Volume.
            'kms:DescribeKey', 'kms:GenerateDataKeyWithoutPlainText');
            if (props.snapshotId) {
                // ReEncrypt is required for when re-encrypting from an encrypted snapshot.
                (_g = grant.principalStatement) === null || _g === void 0 ? void 0 : _g.addActions('kms:ReEncrypt*');
            }
        }
    }
    /**
     * (experimental) Import an existing EBS Volume into the Stack.
     *
     * @param scope the scope of the import.
     * @param id the ID of the imported Volume in the construct tree.
     * @param attrs the attributes of the imported Volume.
     * @experimental
     */
    static fromVolumeAttributes(scope, id, attrs) {
        class Import extends VolumeBase {
            constructor() {
                super(...arguments);
                this.volumeId = attrs.volumeId;
                this.availabilityZone = attrs.availabilityZone;
                this.encryptionKey = attrs.encryptionKey;
            }
        }
        // Check that the provided volumeId looks like it could be valid.
        if (!core_1.Token.isUnresolved(attrs.volumeId) && !/^vol-[0-9a-fA-F]+$/.test(attrs.volumeId)) {
            throw new Error('`volumeId` does not match expected pattern. Expected `vol-<hexadecmial value>` (ex: `vol-05abe246af`) or a Token');
        }
        return new Import(scope, id);
    }
    /**
     * @experimental
     */
    validateProps(props) {
        var _c, _d, _e;
        if (!(props.size || props.snapshotId)) {
            throw new Error('Must provide at least one of `size` or `snapshotId`');
        }
        if (props.snapshotId && !core_1.Token.isUnresolved(props.snapshotId) && !/^snap-[0-9a-fA-F]+$/.test(props.snapshotId)) {
            throw new Error('`snapshotId` does match expected pattern. Expected `snap-<hexadecmial value>` (ex: `snap-05abe246af`) or Token');
        }
        if (props.encryptionKey && !props.encrypted) {
            throw new Error('`encrypted` must be true when providing an `encryptionKey`.');
        }
        if (props.volumeType &&
            [
                EbsDeviceVolumeType.PROVISIONED_IOPS_SSD,
                EbsDeviceVolumeType.PROVISIONED_IOPS_SSD_IO2,
            ].includes(props.volumeType) &&
            !props.iops) {
            throw new Error('`iops` must be specified if the `volumeType` is `PROVISIONED_IOPS_SSD` or `PROVISIONED_IOPS_SSD_IO2`.');
        }
        if (props.iops) {
            const volumeType = (_c = props.volumeType) !== null && _c !== void 0 ? _c : EbsDeviceVolumeType.GENERAL_PURPOSE_SSD;
            if (![
                EbsDeviceVolumeType.PROVISIONED_IOPS_SSD,
                EbsDeviceVolumeType.PROVISIONED_IOPS_SSD_IO2,
                EbsDeviceVolumeType.GENERAL_PURPOSE_SSD_GP3,
            ].includes(volumeType)) {
                throw new Error('`iops` may only be specified if the `volumeType` is `PROVISIONED_IOPS_SSD`, `PROVISIONED_IOPS_SSD_IO2` or `GENERAL_PURPOSE_SSD_GP3`.');
            }
            // Enforce minimum & maximum IOPS:
            // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ebs-volume.html
            const iopsRanges = {};
            iopsRanges[EbsDeviceVolumeType.GENERAL_PURPOSE_SSD_GP3] = { Min: 3000, Max: 16000 };
            iopsRanges[EbsDeviceVolumeType.PROVISIONED_IOPS_SSD] = { Min: 100, Max: 64000 };
            iopsRanges[EbsDeviceVolumeType.PROVISIONED_IOPS_SSD_IO2] = { Min: 100, Max: 64000 };
            const { Min, Max } = iopsRanges[volumeType];
            if (props.iops < Min || props.iops > Max) {
                throw new Error(`\`${volumeType}\` volumes iops must be between ${Min} and ${Max}.`);
            }
            // Enforce maximum ratio of IOPS/GiB:
            // https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-volume-types.html
            const maximumRatios = {};
            maximumRatios[EbsDeviceVolumeType.GENERAL_PURPOSE_SSD_GP3] = 500;
            maximumRatios[EbsDeviceVolumeType.PROVISIONED_IOPS_SSD] = 50;
            maximumRatios[EbsDeviceVolumeType.PROVISIONED_IOPS_SSD_IO2] = 500;
            const maximumRatio = maximumRatios[volumeType];
            if (props.size && (props.iops > maximumRatio * props.size.toGibibytes({ rounding: core_1.SizeRoundingBehavior.FAIL }))) {
                throw new Error(`\`${volumeType}\` volumes iops has a maximum ratio of ${maximumRatio} IOPS/GiB.`);
            }
        }
        if (props.enableMultiAttach) {
            const volumeType = (_d = props.volumeType) !== null && _d !== void 0 ? _d : EbsDeviceVolumeType.GENERAL_PURPOSE_SSD;
            if (![
                EbsDeviceVolumeType.PROVISIONED_IOPS_SSD,
                EbsDeviceVolumeType.PROVISIONED_IOPS_SSD_IO2,
            ].includes(volumeType)) {
                throw new Error('multi-attach is supported exclusively on `PROVISIONED_IOPS_SSD` and `PROVISIONED_IOPS_SSD_IO2` volumes.');
            }
        }
        if (props.size) {
            const size = props.size.toGibibytes({ rounding: core_1.SizeRoundingBehavior.FAIL });
            // Enforce minimum & maximum volume size:
            // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ebs-volume.html
            const sizeRanges = {};
            sizeRanges[EbsDeviceVolumeType.GENERAL_PURPOSE_SSD] = { Min: 1, Max: 16384 };
            sizeRanges[EbsDeviceVolumeType.GENERAL_PURPOSE_SSD_GP3] = { Min: 1, Max: 16384 };
            sizeRanges[EbsDeviceVolumeType.PROVISIONED_IOPS_SSD] = { Min: 4, Max: 16384 };
            sizeRanges[EbsDeviceVolumeType.PROVISIONED_IOPS_SSD_IO2] = { Min: 4, Max: 16384 };
            sizeRanges[EbsDeviceVolumeType.THROUGHPUT_OPTIMIZED_HDD] = { Min: 125, Max: 16384 };
            sizeRanges[EbsDeviceVolumeType.COLD_HDD] = { Min: 125, Max: 16384 };
            sizeRanges[EbsDeviceVolumeType.MAGNETIC] = { Min: 1, Max: 1024 };
            const volumeType = (_e = props.volumeType) !== null && _e !== void 0 ? _e : EbsDeviceVolumeType.GENERAL_PURPOSE_SSD;
            const { Min, Max } = sizeRanges[volumeType];
            if (size < Min || size > Max) {
                throw new Error(`\`${volumeType}\` volumes must be between ${Min} GiB and ${Max} GiB in size.`);
            }
        }
    }
}
exports.Volume = Volume;
_b = JSII_RTTI_SYMBOL_1;
Volume[_b] = { fqn: "monocdk.aws_ec2.Volume", version: "1.106.1" };
//# sourceMappingURL=data:application/json;base64,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