"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MultipartUserData = exports.MultipartBody = exports.UserData = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const machine_image_1 = require("./machine-image");
/**
 * (experimental) Instance User Data.
 *
 * @experimental
 */
class UserData {
    /**
     * (experimental) Create a userdata object for Linux hosts.
     *
     * @experimental
     */
    static forLinux(options = {}) {
        return new LinuxUserData(options);
    }
    /**
     * (experimental) Create a userdata object for Windows hosts.
     *
     * @experimental
     */
    static forWindows() {
        return new WindowsUserData();
    }
    /**
     * (experimental) Create a userdata object with custom content.
     *
     * @experimental
     */
    static custom(content) {
        const userData = new CustomUserData();
        userData.addCommands(content);
        return userData;
    }
    /**
     * @experimental
     */
    static forOperatingSystem(os) {
        switch (os) {
            case machine_image_1.OperatingSystemType.LINUX: return UserData.forLinux();
            case machine_image_1.OperatingSystemType.WINDOWS: return UserData.forWindows();
            case machine_image_1.OperatingSystemType.UNKNOWN: throw new Error('Cannot determine UserData for unknown operating system type');
        }
    }
}
exports.UserData = UserData;
_a = JSII_RTTI_SYMBOL_1;
UserData[_a] = { fqn: "monocdk.aws_ec2.UserData", version: "1.106.1" };
/**
 * Linux Instance User Data
 */
class LinuxUserData extends UserData {
    constructor(props = {}) {
        super();
        this.props = props;
        this.lines = [];
        this.onExitLines = [];
    }
    addCommands(...commands) {
        this.lines.push(...commands);
    }
    addOnExitCommands(...commands) {
        this.onExitLines.push(...commands);
    }
    render() {
        var _d;
        const shebang = (_d = this.props.shebang) !== null && _d !== void 0 ? _d : '#!/bin/bash';
        return [shebang, ...(this.renderOnExitLines()), ...this.lines].join('\n');
    }
    addS3DownloadCommand(params) {
        const s3Path = `s3://${params.bucket.bucketName}/${params.bucketKey}`;
        const localPath = (params.localFile && params.localFile.length !== 0) ? params.localFile : `/tmp/${params.bucketKey}`;
        this.addCommands(`mkdir -p $(dirname '${localPath}')`, `aws s3 cp '${s3Path}' '${localPath}'`);
        return localPath;
    }
    addExecuteFileCommand(params) {
        var _d;
        this.addCommands('set -e', `chmod +x '${params.filePath}'`, `'${params.filePath}' ${(_d = params.arguments) !== null && _d !== void 0 ? _d : ''}`.trim());
    }
    addSignalOnExitCommand(resource) {
        const stack = core_1.Stack.of(resource);
        const resourceID = stack.getLogicalId(resource.node.defaultChild);
        this.addOnExitCommands(`/opt/aws/bin/cfn-signal --stack ${stack.stackName} --resource ${resourceID} --region ${stack.region} -e $exitCode || echo 'Failed to send Cloudformation Signal'`);
    }
    renderOnExitLines() {
        if (this.onExitLines.length > 0) {
            return ['function exitTrap(){', 'exitCode=$?', ...this.onExitLines, '}', 'trap exitTrap EXIT'];
        }
        return [];
    }
}
/**
 * Windows Instance User Data
 */
class WindowsUserData extends UserData {
    constructor() {
        super();
        this.lines = [];
        this.onExitLines = [];
    }
    addCommands(...commands) {
        this.lines.push(...commands);
    }
    addOnExitCommands(...commands) {
        this.onExitLines.push(...commands);
    }
    render() {
        return `<powershell>${[...(this.renderOnExitLines()),
            ...this.lines,
            ...(this.onExitLines.length > 0 ? ['throw "Success"'] : [])].join('\n')}</powershell>`;
    }
    addS3DownloadCommand(params) {
        const localPath = (params.localFile && params.localFile.length !== 0) ? params.localFile : `C:/temp/${params.bucketKey}`;
        this.addCommands(`mkdir (Split-Path -Path '${localPath}' ) -ea 0`, `Read-S3Object -BucketName '${params.bucket.bucketName}' -key '${params.bucketKey}' -file '${localPath}' -ErrorAction Stop`);
        return localPath;
    }
    addExecuteFileCommand(params) {
        var _d;
        this.addCommands(`&'${params.filePath}' ${(_d = params.arguments) !== null && _d !== void 0 ? _d : ''}`.trim(), `if (!$?) { Write-Error 'Failed to execute the file "${params.filePath}"' -ErrorAction Stop }`);
    }
    addSignalOnExitCommand(resource) {
        const stack = core_1.Stack.of(resource);
        const resourceID = stack.getLogicalId(resource.node.defaultChild);
        this.addOnExitCommands(`cfn-signal --stack ${stack.stackName} --resource ${resourceID} --region ${stack.region} --success ($success.ToString().ToLower())`);
    }
    renderOnExitLines() {
        if (this.onExitLines.length > 0) {
            return ['trap {', '$success=($PSItem.Exception.Message -eq "Success")', ...this.onExitLines, 'break', '}'];
        }
        return [];
    }
}
/**
 * Custom Instance User Data
 */
class CustomUserData extends UserData {
    constructor() {
        super();
        this.lines = [];
    }
    addCommands(...commands) {
        this.lines.push(...commands);
    }
    addOnExitCommands() {
        throw new Error('CustomUserData does not support addOnExitCommands, use UserData.forLinux() or UserData.forWindows() instead.');
    }
    render() {
        return this.lines.join('\n');
    }
    addS3DownloadCommand() {
        throw new Error('CustomUserData does not support addS3DownloadCommand, use UserData.forLinux() or UserData.forWindows() instead.');
    }
    addExecuteFileCommand() {
        throw new Error('CustomUserData does not support addExecuteFileCommand, use UserData.forLinux() or UserData.forWindows() instead.');
    }
    addSignalOnExitCommand() {
        throw new Error('CustomUserData does not support addSignalOnExitCommand, use UserData.forLinux() or UserData.forWindows() instead.');
    }
}
/**
 * (experimental) The base class for all classes which can be used as {@link MultipartUserData}.
 *
 * @experimental
 */
class MultipartBody {
    /**
     * @experimental
     */
    constructor() {
    }
    /**
     * (experimental) Constructs the new `MultipartBody` wrapping existing `UserData`. Modification to `UserData` are reflected in subsequent renders of the part.
     *
     * For more information about content types see {@link MultipartBodyOptions.contentType}.
     *
     * @param userData user data to wrap into body part.
     * @param contentType optional content type, if default one should not be used.
     * @experimental
     */
    static fromUserData(userData, contentType) {
        return new MultipartBodyUserDataWrapper(userData, contentType);
    }
    /**
     * (experimental) Constructs the raw `MultipartBody` using specified body, content type and transfer encoding.
     *
     * When transfer encoding is specified (typically as Base64), it's caller responsibility to convert body to
     * Base64 either by wrapping with `Fn.base64` or by converting it by other converters.
     *
     * @experimental
     */
    static fromRawBody(opts) {
        return new MultipartBodyRaw(opts);
    }
}
exports.MultipartBody = MultipartBody;
_b = JSII_RTTI_SYMBOL_1;
MultipartBody[_b] = { fqn: "monocdk.aws_ec2.MultipartBody", version: "1.106.1" };
/**
 * (experimental) Content type for shell scripts.
 *
 * @experimental
 */
MultipartBody.SHELL_SCRIPT = 'text/x-shellscript; charset="utf-8"';
/**
 * (experimental) Content type for boot hooks.
 *
 * @experimental
 */
MultipartBody.CLOUD_BOOTHOOK = 'text/cloud-boothook; charset="utf-8"';
/**
 * The raw part of multi-part user data, which can be added to {@link MultipartUserData}.
 */
class MultipartBodyRaw extends MultipartBody {
    constructor(props) {
        super();
        this.props = props;
    }
    /**
     * Render body part as the string.
     */
    renderBodyPart() {
        const result = [];
        result.push(`Content-Type: ${this.props.contentType}`);
        if (this.props.transferEncoding != null) {
            result.push(`Content-Transfer-Encoding: ${this.props.transferEncoding}`);
        }
        // One line free after separator
        result.push('');
        if (this.props.body != null) {
            result.push(this.props.body);
            // The new line added after join will be consumed by encapsulating or closing boundary
        }
        return result;
    }
}
/**
 * Wrapper for `UserData`.
 */
class MultipartBodyUserDataWrapper extends MultipartBody {
    constructor(userData, contentType) {
        super();
        this.userData = userData;
        this.contentType = contentType || MultipartBody.SHELL_SCRIPT;
    }
    /**
     * Render body part as the string.
     */
    renderBodyPart() {
        const result = [];
        result.push(`Content-Type: ${this.contentType}`);
        result.push('Content-Transfer-Encoding: base64');
        result.push('');
        result.push(core_1.Fn.base64(this.userData.render()));
        return result;
    }
}
/**
 * (experimental) Mime multipart user data.
 *
 * This class represents MIME multipart user data, as described in.
 * [Specifying Multiple User Data Blocks Using a MIME Multi Part Archive](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/bootstrap_container_instance.html#multi-part_user_data)
 *
 * @experimental
 */
class MultipartUserData extends UserData {
    /**
     * @experimental
     */
    constructor(opts) {
        super();
        this.parts = [];
        let partsSeparator;
        // Validate separator
        if ((opts === null || opts === void 0 ? void 0 : opts.partsSeparator) != null) {
            if (new RegExp(MultipartUserData.BOUNDRY_PATTERN).test(opts.partsSeparator)) {
                throw new Error(`Invalid characters in separator. Separator has to match pattern ${MultipartUserData.BOUNDRY_PATTERN}`);
            }
            else {
                partsSeparator = opts.partsSeparator;
            }
        }
        else {
            partsSeparator = '+AWS+CDK+User+Data+Separator==';
        }
        this.opts = {
            partsSeparator: partsSeparator,
        };
    }
    /**
     * (experimental) Adds a part to the list of parts.
     *
     * @experimental
     */
    addPart(part) {
        this.parts.push(part);
    }
    /**
     * (experimental) Adds a multipart part based on a UserData object.
     *
     * This is the same as calling:
     *
     * ```ts
     * multiPart.addPart(MultipartBody.fromUserData(userData, contentType));
     * ```
     *
     * @experimental
     */
    addUserDataPart(userData, contentType) {
        this.addPart(MultipartBody.fromUserData(userData, contentType));
    }
    /**
     * (experimental) Render the UserData for use in a construct.
     *
     * @experimental
     */
    render() {
        const boundary = this.opts.partsSeparator;
        // Now build final MIME archive - there are few changes from MIME message which are accepted by cloud-init:
        // - MIME RFC uses CRLF to separate lines - cloud-init is fine with LF \n only
        // Note: new lines matters, matters a lot.
        var resultArchive = new Array();
        resultArchive.push(`Content-Type: multipart/mixed; boundary="${boundary}"`);
        resultArchive.push('MIME-Version: 1.0');
        // Add new line, the next one will be boundary (encapsulating or closing)
        // so this line will count into it.
        resultArchive.push('');
        // Add parts - each part starts with boundary
        this.parts.forEach(part => {
            resultArchive.push(`--${boundary}`);
            resultArchive.push(...part.renderBodyPart());
        });
        // Add closing boundary
        resultArchive.push(`--${boundary}--`);
        resultArchive.push(''); // Force new line at the end
        return resultArchive.join('\n');
    }
    /**
     * (experimental) Adds commands to download a file from S3.
     *
     * @experimental
     */
    addS3DownloadCommand(_params) {
        throw new Error(MultipartUserData.USE_PART_ERROR);
    }
    /**
     * (experimental) Adds commands to execute a file.
     *
     * @experimental
     */
    addExecuteFileCommand(_params) {
        throw new Error(MultipartUserData.USE_PART_ERROR);
    }
    /**
     * (experimental) Adds a command which will send a cfn-signal when the user data script ends.
     *
     * @experimental
     */
    addSignalOnExitCommand(_resource) {
        throw new Error(MultipartUserData.USE_PART_ERROR);
    }
    /**
     * (experimental) Add one or more commands to the user data.
     *
     * @experimental
     */
    addCommands(..._commands) {
        throw new Error(MultipartUserData.USE_PART_ERROR);
    }
    /**
     * (experimental) Add one or more commands to the user data that will run when the script exits.
     *
     * @experimental
     */
    addOnExitCommands(..._commands) {
        throw new Error(MultipartUserData.USE_PART_ERROR);
    }
}
exports.MultipartUserData = MultipartUserData;
_c = JSII_RTTI_SYMBOL_1;
MultipartUserData[_c] = { fqn: "monocdk.aws_ec2.MultipartUserData", version: "1.106.1" };
MultipartUserData.USE_PART_ERROR = 'MultipartUserData does not support this operation. Please add part using addPart.';
MultipartUserData.BOUNDRY_PATTERN = '[^a-zA-Z0-9()+,-./:=?]';
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidXNlci1kYXRhLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidXNlci1kYXRhLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQ0EscUNBQTZELENBQUMsZ0RBQWdEO0FBQzlHLG1EQUFzRDs7Ozs7O0FBa0R0RCxNQUFzQixRQUFROzs7Ozs7SUFJbkIsTUFBTSxDQUFDLFFBQVEsQ0FBQyxVQUFnQyxFQUFFO1FBQ3JELE9BQU8sSUFBSSxhQUFhLENBQUMsT0FBTyxDQUFDLENBQUM7SUFDdEMsQ0FBQzs7Ozs7O0lBSU0sTUFBTSxDQUFDLFVBQVU7UUFDcEIsT0FBTyxJQUFJLGVBQWUsRUFBRSxDQUFDO0lBQ2pDLENBQUM7Ozs7OztJQUlNLE1BQU0sQ0FBQyxNQUFNLENBQUMsT0FBZTtRQUNoQyxNQUFNLFFBQVEsR0FBRyxJQUFJLGNBQWMsRUFBRSxDQUFDO1FBQ3RDLFFBQVEsQ0FBQyxXQUFXLENBQUMsT0FBTyxDQUFDLENBQUM7UUFDOUIsT0FBTyxRQUFRLENBQUM7SUFDcEIsQ0FBQzs7OztJQUNNLE1BQU0sQ0FBQyxrQkFBa0IsQ0FBQyxFQUF1QjtRQUNwRCxRQUFRLEVBQUUsRUFBRTtZQUNSLEtBQUssbUNBQW1CLENBQUMsS0FBSyxDQUFDLENBQUMsT0FBTyxRQUFRLENBQUMsUUFBUSxFQUFFLENBQUM7WUFDM0QsS0FBSyxtQ0FBbUIsQ0FBQyxPQUFPLENBQUMsQ0FBQyxPQUFPLFFBQVEsQ0FBQyxVQUFVLEVBQUUsQ0FBQztZQUMvRCxLQUFLLG1DQUFtQixDQUFDLE9BQU8sQ0FBQyxDQUFDLE1BQU0sSUFBSSxLQUFLLENBQUMsNkRBQTZELENBQUMsQ0FBQztTQUNwSDtJQUNMLENBQUM7O0FBM0JMLDRCQXNEQzs7O0FBQ0Q7O0dBRUc7QUFDSCxNQUFNLGFBQWMsU0FBUSxRQUFRO0lBR2hDLFlBQTZCLFFBQThCLEVBQUU7UUFDekQsS0FBSyxFQUFFLENBQUM7UUFEaUIsVUFBSyxHQUFMLEtBQUssQ0FBMkI7UUFGNUMsVUFBSyxHQUFhLEVBQUUsQ0FBQztRQUNyQixnQkFBVyxHQUFhLEVBQUUsQ0FBQztJQUc1QyxDQUFDO0lBQ00sV0FBVyxDQUFDLEdBQUcsUUFBa0I7UUFDcEMsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsR0FBRyxRQUFRLENBQUMsQ0FBQztJQUNqQyxDQUFDO0lBQ00saUJBQWlCLENBQUMsR0FBRyxRQUFrQjtRQUMxQyxJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxHQUFHLFFBQVEsQ0FBQyxDQUFDO0lBQ3ZDLENBQUM7SUFDTSxNQUFNOztRQUNULE1BQU0sT0FBTyxTQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxtQ0FBSSxhQUFhLENBQUM7UUFDcEQsT0FBTyxDQUFDLE9BQU8sRUFBRSxHQUFHLENBQUMsSUFBSSxDQUFDLGlCQUFpQixFQUFFLENBQUMsRUFBRSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDOUUsQ0FBQztJQUNNLG9CQUFvQixDQUFDLE1BQXlCO1FBQ2pELE1BQU0sTUFBTSxHQUFHLFFBQVEsTUFBTSxDQUFDLE1BQU0sQ0FBQyxVQUFVLElBQUksTUFBTSxDQUFDLFNBQVMsRUFBRSxDQUFDO1FBQ3RFLE1BQU0sU0FBUyxHQUFHLENBQUMsTUFBTSxDQUFDLFNBQVMsSUFBSSxNQUFNLENBQUMsU0FBUyxDQUFDLE1BQU0sS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsUUFBUSxNQUFNLENBQUMsU0FBUyxFQUFFLENBQUM7UUFDdEgsSUFBSSxDQUFDLFdBQVcsQ0FBQyx1QkFBdUIsU0FBUyxJQUFJLEVBQUUsY0FBYyxNQUFNLE1BQU0sU0FBUyxHQUFHLENBQUMsQ0FBQztRQUMvRixPQUFPLFNBQVMsQ0FBQztJQUNyQixDQUFDO0lBQ00scUJBQXFCLENBQUMsTUFBMEI7O1FBQ25ELElBQUksQ0FBQyxXQUFXLENBQUMsUUFBUSxFQUFFLGFBQWEsTUFBTSxDQUFDLFFBQVEsR0FBRyxFQUFFLElBQUksTUFBTSxDQUFDLFFBQVEsS0FBSyxNQUFBLE1BQU0sQ0FBQyxTQUFTLG1DQUFJLEVBQUUsRUFBRSxDQUFDLElBQUksRUFBRSxDQUFDLENBQUM7SUFDekgsQ0FBQztJQUNNLHNCQUFzQixDQUFDLFFBQWtCO1FBQzVDLE1BQU0sS0FBSyxHQUFHLFlBQUssQ0FBQyxFQUFFLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDakMsTUFBTSxVQUFVLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLFlBQTBCLENBQUMsQ0FBQztRQUNoRixJQUFJLENBQUMsaUJBQWlCLENBQUMsbUNBQW1DLEtBQUssQ0FBQyxTQUFTLGVBQWUsVUFBVSxhQUFhLEtBQUssQ0FBQyxNQUFNLDhEQUE4RCxDQUFDLENBQUM7SUFDL0wsQ0FBQztJQUNPLGlCQUFpQjtRQUNyQixJQUFJLElBQUksQ0FBQyxXQUFXLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUM3QixPQUFPLENBQUMsc0JBQXNCLEVBQUUsYUFBYSxFQUFFLEdBQUcsSUFBSSxDQUFDLFdBQVcsRUFBRSxHQUFHLEVBQUUsb0JBQW9CLENBQUMsQ0FBQztTQUNsRztRQUNELE9BQU8sRUFBRSxDQUFDO0lBQ2QsQ0FBQztDQUNKO0FBQ0Q7O0dBRUc7QUFDSCxNQUFNLGVBQWdCLFNBQVEsUUFBUTtJQUdsQztRQUNJLEtBQUssRUFBRSxDQUFDO1FBSEssVUFBSyxHQUFhLEVBQUUsQ0FBQztRQUNyQixnQkFBVyxHQUFhLEVBQUUsQ0FBQztJQUc1QyxDQUFDO0lBQ00sV0FBVyxDQUFDLEdBQUcsUUFBa0I7UUFDcEMsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsR0FBRyxRQUFRLENBQUMsQ0FBQztJQUNqQyxDQUFDO0lBQ00saUJBQWlCLENBQUMsR0FBRyxRQUFrQjtRQUMxQyxJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxHQUFHLFFBQVEsQ0FBQyxDQUFDO0lBQ3ZDLENBQUM7SUFDTSxNQUFNO1FBQ1QsT0FBTyxlQUFlLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxpQkFBaUIsRUFBRSxDQUFDO1lBQ2hELEdBQUcsSUFBSSxDQUFDLEtBQUs7WUFDYixHQUFHLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLGlCQUFpQixDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxlQUFlLENBQUM7SUFDL0YsQ0FBQztJQUNNLG9CQUFvQixDQUFDLE1BQXlCO1FBQ2pELE1BQU0sU0FBUyxHQUFHLENBQUMsTUFBTSxDQUFDLFNBQVMsSUFBSSxNQUFNLENBQUMsU0FBUyxDQUFDLE1BQU0sS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsV0FBVyxNQUFNLENBQUMsU0FBUyxFQUFFLENBQUM7UUFDekgsSUFBSSxDQUFDLFdBQVcsQ0FBQyw0QkFBNEIsU0FBUyxXQUFXLEVBQUUsOEJBQThCLE1BQU0sQ0FBQyxNQUFNLENBQUMsVUFBVSxXQUFXLE1BQU0sQ0FBQyxTQUFTLFlBQVksU0FBUyxxQkFBcUIsQ0FBQyxDQUFDO1FBQ2hNLE9BQU8sU0FBUyxDQUFDO0lBQ3JCLENBQUM7SUFDTSxxQkFBcUIsQ0FBQyxNQUEwQjs7UUFDbkQsSUFBSSxDQUFDLFdBQVcsQ0FBQyxLQUFLLE1BQU0sQ0FBQyxRQUFRLEtBQUssTUFBQSxNQUFNLENBQUMsU0FBUyxtQ0FBSSxFQUFFLEVBQUUsQ0FBQyxJQUFJLEVBQUUsRUFBRSx1REFBdUQsTUFBTSxDQUFDLFFBQVEsd0JBQXdCLENBQUMsQ0FBQztJQUMvSyxDQUFDO0lBQ00sc0JBQXNCLENBQUMsUUFBa0I7UUFDNUMsTUFBTSxLQUFLLEdBQUcsWUFBSyxDQUFDLEVBQUUsQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUNqQyxNQUFNLFVBQVUsR0FBRyxLQUFLLENBQUMsWUFBWSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsWUFBMEIsQ0FBQyxDQUFDO1FBQ2hGLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxzQkFBc0IsS0FBSyxDQUFDLFNBQVMsZUFBZSxVQUFVLGFBQWEsS0FBSyxDQUFDLE1BQU0sNENBQTRDLENBQUMsQ0FBQztJQUNoSyxDQUFDO0lBQ08saUJBQWlCO1FBQ3JCLElBQUksSUFBSSxDQUFDLFdBQVcsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQzdCLE9BQU8sQ0FBQyxRQUFRLEVBQUUsb0RBQW9ELEVBQUUsR0FBRyxJQUFJLENBQUMsV0FBVyxFQUFFLE9BQU8sRUFBRSxHQUFHLENBQUMsQ0FBQztTQUM5RztRQUNELE9BQU8sRUFBRSxDQUFDO0lBQ2QsQ0FBQztDQUNKO0FBQ0Q7O0dBRUc7QUFDSCxNQUFNLGNBQWUsU0FBUSxRQUFRO0lBRWpDO1FBQ0ksS0FBSyxFQUFFLENBQUM7UUFGSyxVQUFLLEdBQWEsRUFBRSxDQUFDO0lBR3RDLENBQUM7SUFDTSxXQUFXLENBQUMsR0FBRyxRQUFrQjtRQUNwQyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxHQUFHLFFBQVEsQ0FBQyxDQUFDO0lBQ2pDLENBQUM7SUFDTSxpQkFBaUI7UUFDcEIsTUFBTSxJQUFJLEtBQUssQ0FBQyw4R0FBOEcsQ0FBQyxDQUFDO0lBQ3BJLENBQUM7SUFDTSxNQUFNO1FBQ1QsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUNqQyxDQUFDO0lBQ00sb0JBQW9CO1FBQ3ZCLE1BQU0sSUFBSSxLQUFLLENBQUMsaUhBQWlILENBQUMsQ0FBQztJQUN2SSxDQUFDO0lBQ00scUJBQXFCO1FBQ3hCLE1BQU0sSUFBSSxLQUFLLENBQUMsa0hBQWtILENBQUMsQ0FBQztJQUN4SSxDQUFDO0lBQ00sc0JBQXNCO1FBQ3pCLE1BQU0sSUFBSSxLQUFLLENBQUMsbUhBQW1ILENBQUMsQ0FBQztJQUN6SSxDQUFDO0NBQ0o7Ozs7OztBQStCRCxNQUFzQixhQUFhOzs7O0lBOEIvQjtJQUNBLENBQUM7Ozs7Ozs7Ozs7SUFiTSxNQUFNLENBQUMsWUFBWSxDQUFDLFFBQWtCLEVBQUUsV0FBb0I7UUFDL0QsT0FBTyxJQUFJLDRCQUE0QixDQUFDLFFBQVEsRUFBRSxXQUFXLENBQUMsQ0FBQztJQUNuRSxDQUFDOzs7Ozs7Ozs7SUFPTSxNQUFNLENBQUMsV0FBVyxDQUFDLElBQTBCO1FBQ2hELE9BQU8sSUFBSSxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUN0QyxDQUFDOztBQTdCTCxzQ0FzQ0M7Ozs7Ozs7O0FBbEMwQiwwQkFBWSxHQUFHLHFDQUFxQyxDQUFDOzs7Ozs7QUFJckQsNEJBQWMsR0FBRyxzQ0FBc0MsQ0FBQztBQStCbkY7O0dBRUc7QUFDSCxNQUFNLGdCQUFpQixTQUFRLGFBQWE7SUFDeEMsWUFBb0MsS0FBMkI7UUFDM0QsS0FBSyxFQUFFLENBQUM7UUFEd0IsVUFBSyxHQUFMLEtBQUssQ0FBc0I7SUFFL0QsQ0FBQztJQUNEOztPQUVHO0lBQ0ksY0FBYztRQUNqQixNQUFNLE1BQU0sR0FBYSxFQUFFLENBQUM7UUFDNUIsTUFBTSxDQUFDLElBQUksQ0FBQyxpQkFBaUIsSUFBSSxDQUFDLEtBQUssQ0FBQyxXQUFXLEVBQUUsQ0FBQyxDQUFDO1FBQ3ZELElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxnQkFBZ0IsSUFBSSxJQUFJLEVBQUU7WUFDckMsTUFBTSxDQUFDLElBQUksQ0FBQyw4QkFBOEIsSUFBSSxDQUFDLEtBQUssQ0FBQyxnQkFBZ0IsRUFBRSxDQUFDLENBQUM7U0FDNUU7UUFDRCxnQ0FBZ0M7UUFDaEMsTUFBTSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQztRQUNoQixJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxJQUFJLElBQUksRUFBRTtZQUN6QixNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUM7WUFDN0Isc0ZBQXNGO1NBQ3pGO1FBQ0QsT0FBTyxNQUFNLENBQUM7SUFDbEIsQ0FBQztDQUNKO0FBQ0Q7O0dBRUc7QUFDSCxNQUFNLDRCQUE2QixTQUFRLGFBQWE7SUFFcEQsWUFBb0MsUUFBa0IsRUFBRSxXQUFvQjtRQUN4RSxLQUFLLEVBQUUsQ0FBQztRQUR3QixhQUFRLEdBQVIsUUFBUSxDQUFVO1FBRWxELElBQUksQ0FBQyxXQUFXLEdBQUcsV0FBVyxJQUFJLGFBQWEsQ0FBQyxZQUFZLENBQUM7SUFDakUsQ0FBQztJQUNEOztPQUVHO0lBQ0ksY0FBYztRQUNqQixNQUFNLE1BQU0sR0FBYSxFQUFFLENBQUM7UUFDNUIsTUFBTSxDQUFDLElBQUksQ0FBQyxpQkFBaUIsSUFBSSxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUM7UUFDakQsTUFBTSxDQUFDLElBQUksQ0FBQyxtQ0FBbUMsQ0FBQyxDQUFDO1FBQ2pELE1BQU0sQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUM7UUFDaEIsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFFLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBQy9DLE9BQU8sTUFBTSxDQUFDO0lBQ2xCLENBQUM7Q0FDSjs7Ozs7Ozs7O0FBcUJELE1BQWEsaUJBQWtCLFNBQVEsUUFBUTs7OztJQUszQyxZQUFZLElBQStCO1FBQ3ZDLEtBQUssRUFBRSxDQUFDO1FBSEosVUFBSyxHQUFvQixFQUFFLENBQUM7UUFJaEMsSUFBSSxjQUFzQixDQUFDO1FBQzNCLHFCQUFxQjtRQUNyQixJQUFJLENBQUEsSUFBSSxhQUFKLElBQUksdUJBQUosSUFBSSxDQUFFLGNBQWMsS0FBSSxJQUFJLEVBQUU7WUFDOUIsSUFBSSxJQUFJLE1BQU0sQ0FBQyxpQkFBaUIsQ0FBQyxlQUFlLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSyxDQUFDLGNBQWMsQ0FBQyxFQUFFO2dCQUMxRSxNQUFNLElBQUksS0FBSyxDQUFDLG1FQUFtRSxpQkFBaUIsQ0FBQyxlQUFlLEVBQUUsQ0FBQyxDQUFDO2FBQzNIO2lCQUNJO2dCQUNELGNBQWMsR0FBRyxJQUFLLENBQUMsY0FBYyxDQUFDO2FBQ3pDO1NBQ0o7YUFDSTtZQUNELGNBQWMsR0FBRyxnQ0FBZ0MsQ0FBQztTQUNyRDtRQUNELElBQUksQ0FBQyxJQUFJLEdBQUc7WUFDUixjQUFjLEVBQUUsY0FBYztTQUNqQyxDQUFDO0lBQ04sQ0FBQzs7Ozs7O0lBSU0sT0FBTyxDQUFDLElBQW1CO1FBQzlCLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQzFCLENBQUM7Ozs7Ozs7Ozs7OztJQVVNLGVBQWUsQ0FBQyxRQUFrQixFQUFFLFdBQW9CO1FBQzNELElBQUksQ0FBQyxPQUFPLENBQUMsYUFBYSxDQUFDLFlBQVksQ0FBQyxRQUFRLEVBQUUsV0FBVyxDQUFDLENBQUMsQ0FBQztJQUNwRSxDQUFDOzs7Ozs7SUFDTSxNQUFNO1FBQ1QsTUFBTSxRQUFRLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUM7UUFDMUMsMkdBQTJHO1FBQzNHLDhFQUE4RTtRQUM5RSwwQ0FBMEM7UUFDMUMsSUFBSSxhQUFhLEdBQUcsSUFBSSxLQUFLLEVBQVUsQ0FBQztRQUN4QyxhQUFhLENBQUMsSUFBSSxDQUFDLDRDQUE0QyxRQUFRLEdBQUcsQ0FBQyxDQUFDO1FBQzVFLGFBQWEsQ0FBQyxJQUFJLENBQUMsbUJBQW1CLENBQUMsQ0FBQztRQUN4Qyx5RUFBeUU7UUFDekUsbUNBQW1DO1FBQ25DLGFBQWEsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUM7UUFDdkIsNkNBQTZDO1FBQzdDLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxFQUFFO1lBQ3RCLGFBQWEsQ0FBQyxJQUFJLENBQUMsS0FBSyxRQUFRLEVBQUUsQ0FBQyxDQUFDO1lBQ3BDLGFBQWEsQ0FBQyxJQUFJLENBQUMsR0FBRyxJQUFJLENBQUMsY0FBYyxFQUFFLENBQUMsQ0FBQztRQUNqRCxDQUFDLENBQUMsQ0FBQztRQUNILHVCQUF1QjtRQUN2QixhQUFhLENBQUMsSUFBSSxDQUFDLEtBQUssUUFBUSxJQUFJLENBQUMsQ0FBQztRQUN0QyxhQUFhLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsNEJBQTRCO1FBQ3BELE9BQU8sYUFBYSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUNwQyxDQUFDOzs7Ozs7SUFDTSxvQkFBb0IsQ0FBQyxPQUEwQjtRQUNsRCxNQUFNLElBQUksS0FBSyxDQUFDLGlCQUFpQixDQUFDLGNBQWMsQ0FBQyxDQUFDO0lBQ3RELENBQUM7Ozs7OztJQUNNLHFCQUFxQixDQUFDLE9BQTJCO1FBQ3BELE1BQU0sSUFBSSxLQUFLLENBQUMsaUJBQWlCLENBQUMsY0FBYyxDQUFDLENBQUM7SUFDdEQsQ0FBQzs7Ozs7O0lBQ00sc0JBQXNCLENBQUMsU0FBbUI7UUFDN0MsTUFBTSxJQUFJLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxjQUFjLENBQUMsQ0FBQztJQUN0RCxDQUFDOzs7Ozs7SUFDTSxXQUFXLENBQUMsR0FBRyxTQUFtQjtRQUNyQyxNQUFNLElBQUksS0FBSyxDQUFDLGlCQUFpQixDQUFDLGNBQWMsQ0FBQyxDQUFDO0lBQ3RELENBQUM7Ozs7OztJQUNNLGlCQUFpQixDQUFDLEdBQUcsU0FBbUI7UUFDM0MsTUFBTSxJQUFJLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxjQUFjLENBQUMsQ0FBQztJQUN0RCxDQUFDOztBQTdFTCw4Q0E4RUM7OztBQTdFMkIsZ0NBQWMsR0FBRyxtRkFBbUYsQ0FBQztBQUNyRyxpQ0FBZSxHQUFHLHdCQUF3QixDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgSUJ1Y2tldCB9IGZyb20gXCIuLi8uLi9hd3MtczNcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1zMydcbmltcG9ydCB7IENmbkVsZW1lbnQsIEZuLCBSZXNvdXJjZSwgU3RhY2sgfSBmcm9tIFwiLi4vLi4vY29yZVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvY29yZSdcbmltcG9ydCB7IE9wZXJhdGluZ1N5c3RlbVR5cGUgfSBmcm9tICcuL21hY2hpbmUtaW1hZ2UnO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBMaW51eFVzZXJEYXRhT3B0aW9ucyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgc2hlYmFuZz86IHN0cmluZztcbn1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgUzNEb3dubG9hZE9wdGlvbnMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IGJ1Y2tldDogSUJ1Y2tldDtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IGJ1Y2tldEtleTogc3RyaW5nO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgbG9jYWxGaWxlPzogc3RyaW5nO1xufVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgRXhlY3V0ZUZpbGVPcHRpb25zIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgZmlsZVBhdGg6IHN0cmluZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgYXJndW1lbnRzPzogc3RyaW5nO1xufVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBhYnN0cmFjdCBjbGFzcyBVc2VyRGF0YSB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIGZvckxpbnV4KG9wdGlvbnM6IExpbnV4VXNlckRhdGFPcHRpb25zID0ge30pOiBVc2VyRGF0YSB7XG4gICAgICAgIHJldHVybiBuZXcgTGludXhVc2VyRGF0YShvcHRpb25zKTtcbiAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgZm9yV2luZG93cygpOiBVc2VyRGF0YSB7XG4gICAgICAgIHJldHVybiBuZXcgV2luZG93c1VzZXJEYXRhKCk7XG4gICAgfVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgY3VzdG9tKGNvbnRlbnQ6IHN0cmluZyk6IFVzZXJEYXRhIHtcbiAgICAgICAgY29uc3QgdXNlckRhdGEgPSBuZXcgQ3VzdG9tVXNlckRhdGEoKTtcbiAgICAgICAgdXNlckRhdGEuYWRkQ29tbWFuZHMoY29udGVudCk7XG4gICAgICAgIHJldHVybiB1c2VyRGF0YTtcbiAgICB9XG4gICAgcHVibGljIHN0YXRpYyBmb3JPcGVyYXRpbmdTeXN0ZW0ob3M6IE9wZXJhdGluZ1N5c3RlbVR5cGUpOiBVc2VyRGF0YSB7XG4gICAgICAgIHN3aXRjaCAob3MpIHtcbiAgICAgICAgICAgIGNhc2UgT3BlcmF0aW5nU3lzdGVtVHlwZS5MSU5VWDogcmV0dXJuIFVzZXJEYXRhLmZvckxpbnV4KCk7XG4gICAgICAgICAgICBjYXNlIE9wZXJhdGluZ1N5c3RlbVR5cGUuV0lORE9XUzogcmV0dXJuIFVzZXJEYXRhLmZvcldpbmRvd3MoKTtcbiAgICAgICAgICAgIGNhc2UgT3BlcmF0aW5nU3lzdGVtVHlwZS5VTktOT1dOOiB0aHJvdyBuZXcgRXJyb3IoJ0Nhbm5vdCBkZXRlcm1pbmUgVXNlckRhdGEgZm9yIHVua25vd24gb3BlcmF0aW5nIHN5c3RlbSB0eXBlJyk7XG4gICAgICAgIH1cbiAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIGFic3RyYWN0IGFkZENvbW1hbmRzKC4uLmNvbW1hbmRzOiBzdHJpbmdbXSk6IHZvaWQ7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBhYnN0cmFjdCBhZGRPbkV4aXRDb21tYW5kcyguLi5jb21tYW5kczogc3RyaW5nW10pOiB2b2lkO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgYWJzdHJhY3QgcmVuZGVyKCk6IHN0cmluZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBhYnN0cmFjdCBhZGRTM0Rvd25sb2FkQ29tbWFuZChwYXJhbXM6IFMzRG93bmxvYWRPcHRpb25zKTogc3RyaW5nO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIGFic3RyYWN0IGFkZEV4ZWN1dGVGaWxlQ29tbWFuZChwYXJhbXM6IEV4ZWN1dGVGaWxlT3B0aW9ucyk6IHZvaWQ7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIGFic3RyYWN0IGFkZFNpZ25hbE9uRXhpdENvbW1hbmQocmVzb3VyY2U6IFJlc291cmNlKTogdm9pZDtcbn1cbi8qKlxuICogTGludXggSW5zdGFuY2UgVXNlciBEYXRhXG4gKi9cbmNsYXNzIExpbnV4VXNlckRhdGEgZXh0ZW5kcyBVc2VyRGF0YSB7XG4gICAgcHJpdmF0ZSByZWFkb25seSBsaW5lczogc3RyaW5nW10gPSBbXTtcbiAgICBwcml2YXRlIHJlYWRvbmx5IG9uRXhpdExpbmVzOiBzdHJpbmdbXSA9IFtdO1xuICAgIGNvbnN0cnVjdG9yKHByaXZhdGUgcmVhZG9ubHkgcHJvcHM6IExpbnV4VXNlckRhdGFPcHRpb25zID0ge30pIHtcbiAgICAgICAgc3VwZXIoKTtcbiAgICB9XG4gICAgcHVibGljIGFkZENvbW1hbmRzKC4uLmNvbW1hbmRzOiBzdHJpbmdbXSkge1xuICAgICAgICB0aGlzLmxpbmVzLnB1c2goLi4uY29tbWFuZHMpO1xuICAgIH1cbiAgICBwdWJsaWMgYWRkT25FeGl0Q29tbWFuZHMoLi4uY29tbWFuZHM6IHN0cmluZ1tdKSB7XG4gICAgICAgIHRoaXMub25FeGl0TGluZXMucHVzaCguLi5jb21tYW5kcyk7XG4gICAgfVxuICAgIHB1YmxpYyByZW5kZXIoKTogc3RyaW5nIHtcbiAgICAgICAgY29uc3Qgc2hlYmFuZyA9IHRoaXMucHJvcHMuc2hlYmFuZyA/PyAnIyEvYmluL2Jhc2gnO1xuICAgICAgICByZXR1cm4gW3NoZWJhbmcsIC4uLih0aGlzLnJlbmRlck9uRXhpdExpbmVzKCkpLCAuLi50aGlzLmxpbmVzXS5qb2luKCdcXG4nKTtcbiAgICB9XG4gICAgcHVibGljIGFkZFMzRG93bmxvYWRDb21tYW5kKHBhcmFtczogUzNEb3dubG9hZE9wdGlvbnMpOiBzdHJpbmcge1xuICAgICAgICBjb25zdCBzM1BhdGggPSBgczM6Ly8ke3BhcmFtcy5idWNrZXQuYnVja2V0TmFtZX0vJHtwYXJhbXMuYnVja2V0S2V5fWA7XG4gICAgICAgIGNvbnN0IGxvY2FsUGF0aCA9IChwYXJhbXMubG9jYWxGaWxlICYmIHBhcmFtcy5sb2NhbEZpbGUubGVuZ3RoICE9PSAwKSA/IHBhcmFtcy5sb2NhbEZpbGUgOiBgL3RtcC8ke3BhcmFtcy5idWNrZXRLZXl9YDtcbiAgICAgICAgdGhpcy5hZGRDb21tYW5kcyhgbWtkaXIgLXAgJChkaXJuYW1lICcke2xvY2FsUGF0aH0nKWAsIGBhd3MgczMgY3AgJyR7czNQYXRofScgJyR7bG9jYWxQYXRofSdgKTtcbiAgICAgICAgcmV0dXJuIGxvY2FsUGF0aDtcbiAgICB9XG4gICAgcHVibGljIGFkZEV4ZWN1dGVGaWxlQ29tbWFuZChwYXJhbXM6IEV4ZWN1dGVGaWxlT3B0aW9ucyk6IHZvaWQge1xuICAgICAgICB0aGlzLmFkZENvbW1hbmRzKCdzZXQgLWUnLCBgY2htb2QgK3ggJyR7cGFyYW1zLmZpbGVQYXRofSdgLCBgJyR7cGFyYW1zLmZpbGVQYXRofScgJHtwYXJhbXMuYXJndW1lbnRzID8/ICcnfWAudHJpbSgpKTtcbiAgICB9XG4gICAgcHVibGljIGFkZFNpZ25hbE9uRXhpdENvbW1hbmQocmVzb3VyY2U6IFJlc291cmNlKTogdm9pZCB7XG4gICAgICAgIGNvbnN0IHN0YWNrID0gU3RhY2sub2YocmVzb3VyY2UpO1xuICAgICAgICBjb25zdCByZXNvdXJjZUlEID0gc3RhY2suZ2V0TG9naWNhbElkKHJlc291cmNlLm5vZGUuZGVmYXVsdENoaWxkIGFzIENmbkVsZW1lbnQpO1xuICAgICAgICB0aGlzLmFkZE9uRXhpdENvbW1hbmRzKGAvb3B0L2F3cy9iaW4vY2ZuLXNpZ25hbCAtLXN0YWNrICR7c3RhY2suc3RhY2tOYW1lfSAtLXJlc291cmNlICR7cmVzb3VyY2VJRH0gLS1yZWdpb24gJHtzdGFjay5yZWdpb259IC1lICRleGl0Q29kZSB8fCBlY2hvICdGYWlsZWQgdG8gc2VuZCBDbG91ZGZvcm1hdGlvbiBTaWduYWwnYCk7XG4gICAgfVxuICAgIHByaXZhdGUgcmVuZGVyT25FeGl0TGluZXMoKTogc3RyaW5nW10ge1xuICAgICAgICBpZiAodGhpcy5vbkV4aXRMaW5lcy5sZW5ndGggPiAwKSB7XG4gICAgICAgICAgICByZXR1cm4gWydmdW5jdGlvbiBleGl0VHJhcCgpeycsICdleGl0Q29kZT0kPycsIC4uLnRoaXMub25FeGl0TGluZXMsICd9JywgJ3RyYXAgZXhpdFRyYXAgRVhJVCddO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiBbXTtcbiAgICB9XG59XG4vKipcbiAqIFdpbmRvd3MgSW5zdGFuY2UgVXNlciBEYXRhXG4gKi9cbmNsYXNzIFdpbmRvd3NVc2VyRGF0YSBleHRlbmRzIFVzZXJEYXRhIHtcbiAgICBwcml2YXRlIHJlYWRvbmx5IGxpbmVzOiBzdHJpbmdbXSA9IFtdO1xuICAgIHByaXZhdGUgcmVhZG9ubHkgb25FeGl0TGluZXM6IHN0cmluZ1tdID0gW107XG4gICAgY29uc3RydWN0b3IoKSB7XG4gICAgICAgIHN1cGVyKCk7XG4gICAgfVxuICAgIHB1YmxpYyBhZGRDb21tYW5kcyguLi5jb21tYW5kczogc3RyaW5nW10pIHtcbiAgICAgICAgdGhpcy5saW5lcy5wdXNoKC4uLmNvbW1hbmRzKTtcbiAgICB9XG4gICAgcHVibGljIGFkZE9uRXhpdENvbW1hbmRzKC4uLmNvbW1hbmRzOiBzdHJpbmdbXSkge1xuICAgICAgICB0aGlzLm9uRXhpdExpbmVzLnB1c2goLi4uY29tbWFuZHMpO1xuICAgIH1cbiAgICBwdWJsaWMgcmVuZGVyKCk6IHN0cmluZyB7XG4gICAgICAgIHJldHVybiBgPHBvd2Vyc2hlbGw+JHtbLi4uKHRoaXMucmVuZGVyT25FeGl0TGluZXMoKSksXG4gICAgICAgICAgICAuLi50aGlzLmxpbmVzLFxuICAgICAgICAgICAgLi4uKHRoaXMub25FeGl0TGluZXMubGVuZ3RoID4gMCA/IFsndGhyb3cgXCJTdWNjZXNzXCInXSA6IFtdKV0uam9pbignXFxuJyl9PC9wb3dlcnNoZWxsPmA7XG4gICAgfVxuICAgIHB1YmxpYyBhZGRTM0Rvd25sb2FkQ29tbWFuZChwYXJhbXM6IFMzRG93bmxvYWRPcHRpb25zKTogc3RyaW5nIHtcbiAgICAgICAgY29uc3QgbG9jYWxQYXRoID0gKHBhcmFtcy5sb2NhbEZpbGUgJiYgcGFyYW1zLmxvY2FsRmlsZS5sZW5ndGggIT09IDApID8gcGFyYW1zLmxvY2FsRmlsZSA6IGBDOi90ZW1wLyR7cGFyYW1zLmJ1Y2tldEtleX1gO1xuICAgICAgICB0aGlzLmFkZENvbW1hbmRzKGBta2RpciAoU3BsaXQtUGF0aCAtUGF0aCAnJHtsb2NhbFBhdGh9JyApIC1lYSAwYCwgYFJlYWQtUzNPYmplY3QgLUJ1Y2tldE5hbWUgJyR7cGFyYW1zLmJ1Y2tldC5idWNrZXROYW1lfScgLWtleSAnJHtwYXJhbXMuYnVja2V0S2V5fScgLWZpbGUgJyR7bG9jYWxQYXRofScgLUVycm9yQWN0aW9uIFN0b3BgKTtcbiAgICAgICAgcmV0dXJuIGxvY2FsUGF0aDtcbiAgICB9XG4gICAgcHVibGljIGFkZEV4ZWN1dGVGaWxlQ29tbWFuZChwYXJhbXM6IEV4ZWN1dGVGaWxlT3B0aW9ucyk6IHZvaWQge1xuICAgICAgICB0aGlzLmFkZENvbW1hbmRzKGAmJyR7cGFyYW1zLmZpbGVQYXRofScgJHtwYXJhbXMuYXJndW1lbnRzID8/ICcnfWAudHJpbSgpLCBgaWYgKCEkPykgeyBXcml0ZS1FcnJvciAnRmFpbGVkIHRvIGV4ZWN1dGUgdGhlIGZpbGUgXCIke3BhcmFtcy5maWxlUGF0aH1cIicgLUVycm9yQWN0aW9uIFN0b3AgfWApO1xuICAgIH1cbiAgICBwdWJsaWMgYWRkU2lnbmFsT25FeGl0Q29tbWFuZChyZXNvdXJjZTogUmVzb3VyY2UpOiB2b2lkIHtcbiAgICAgICAgY29uc3Qgc3RhY2sgPSBTdGFjay5vZihyZXNvdXJjZSk7XG4gICAgICAgIGNvbnN0IHJlc291cmNlSUQgPSBzdGFjay5nZXRMb2dpY2FsSWQocmVzb3VyY2Uubm9kZS5kZWZhdWx0Q2hpbGQgYXMgQ2ZuRWxlbWVudCk7XG4gICAgICAgIHRoaXMuYWRkT25FeGl0Q29tbWFuZHMoYGNmbi1zaWduYWwgLS1zdGFjayAke3N0YWNrLnN0YWNrTmFtZX0gLS1yZXNvdXJjZSAke3Jlc291cmNlSUR9IC0tcmVnaW9uICR7c3RhY2sucmVnaW9ufSAtLXN1Y2Nlc3MgKCRzdWNjZXNzLlRvU3RyaW5nKCkuVG9Mb3dlcigpKWApO1xuICAgIH1cbiAgICBwcml2YXRlIHJlbmRlck9uRXhpdExpbmVzKCk6IHN0cmluZ1tdIHtcbiAgICAgICAgaWYgKHRoaXMub25FeGl0TGluZXMubGVuZ3RoID4gMCkge1xuICAgICAgICAgICAgcmV0dXJuIFsndHJhcCB7JywgJyRzdWNjZXNzPSgkUFNJdGVtLkV4Y2VwdGlvbi5NZXNzYWdlIC1lcSBcIlN1Y2Nlc3NcIiknLCAuLi50aGlzLm9uRXhpdExpbmVzLCAnYnJlYWsnLCAnfSddO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiBbXTtcbiAgICB9XG59XG4vKipcbiAqIEN1c3RvbSBJbnN0YW5jZSBVc2VyIERhdGFcbiAqL1xuY2xhc3MgQ3VzdG9tVXNlckRhdGEgZXh0ZW5kcyBVc2VyRGF0YSB7XG4gICAgcHJpdmF0ZSByZWFkb25seSBsaW5lczogc3RyaW5nW10gPSBbXTtcbiAgICBjb25zdHJ1Y3RvcigpIHtcbiAgICAgICAgc3VwZXIoKTtcbiAgICB9XG4gICAgcHVibGljIGFkZENvbW1hbmRzKC4uLmNvbW1hbmRzOiBzdHJpbmdbXSkge1xuICAgICAgICB0aGlzLmxpbmVzLnB1c2goLi4uY29tbWFuZHMpO1xuICAgIH1cbiAgICBwdWJsaWMgYWRkT25FeGl0Q29tbWFuZHMoKTogdm9pZCB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcignQ3VzdG9tVXNlckRhdGEgZG9lcyBub3Qgc3VwcG9ydCBhZGRPbkV4aXRDb21tYW5kcywgdXNlIFVzZXJEYXRhLmZvckxpbnV4KCkgb3IgVXNlckRhdGEuZm9yV2luZG93cygpIGluc3RlYWQuJyk7XG4gICAgfVxuICAgIHB1YmxpYyByZW5kZXIoKTogc3RyaW5nIHtcbiAgICAgICAgcmV0dXJuIHRoaXMubGluZXMuam9pbignXFxuJyk7XG4gICAgfVxuICAgIHB1YmxpYyBhZGRTM0Rvd25sb2FkQ29tbWFuZCgpOiBzdHJpbmcge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0N1c3RvbVVzZXJEYXRhIGRvZXMgbm90IHN1cHBvcnQgYWRkUzNEb3dubG9hZENvbW1hbmQsIHVzZSBVc2VyRGF0YS5mb3JMaW51eCgpIG9yIFVzZXJEYXRhLmZvcldpbmRvd3MoKSBpbnN0ZWFkLicpO1xuICAgIH1cbiAgICBwdWJsaWMgYWRkRXhlY3V0ZUZpbGVDb21tYW5kKCk6IHZvaWQge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0N1c3RvbVVzZXJEYXRhIGRvZXMgbm90IHN1cHBvcnQgYWRkRXhlY3V0ZUZpbGVDb21tYW5kLCB1c2UgVXNlckRhdGEuZm9yTGludXgoKSBvciBVc2VyRGF0YS5mb3JXaW5kb3dzKCkgaW5zdGVhZC4nKTtcbiAgICB9XG4gICAgcHVibGljIGFkZFNpZ25hbE9uRXhpdENvbW1hbmQoKTogdm9pZCB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcignQ3VzdG9tVXNlckRhdGEgZG9lcyBub3Qgc3VwcG9ydCBhZGRTaWduYWxPbkV4aXRDb21tYW5kLCB1c2UgVXNlckRhdGEuZm9yTGludXgoKSBvciBVc2VyRGF0YS5mb3JXaW5kb3dzKCkgaW5zdGVhZC4nKTtcbiAgICB9XG59XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIE11bHRpcGFydEJvZHlPcHRpb25zIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgY29udGVudFR5cGU6IHN0cmluZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IHRyYW5zZmVyRW5jb2Rpbmc/OiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgYm9keT86IHN0cmluZztcbn1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgYWJzdHJhY3QgY2xhc3MgTXVsdGlwYXJ0Qm9keSB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgU0hFTExfU0NSSVBUID0gJ3RleHQveC1zaGVsbHNjcmlwdDsgY2hhcnNldD1cInV0Zi04XCInO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IENMT1VEX0JPT1RIT09LID0gJ3RleHQvY2xvdWQtYm9vdGhvb2s7IGNoYXJzZXQ9XCJ1dGYtOFwiJztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgZnJvbVVzZXJEYXRhKHVzZXJEYXRhOiBVc2VyRGF0YSwgY29udGVudFR5cGU/OiBzdHJpbmcpOiBNdWx0aXBhcnRCb2R5IHtcbiAgICAgICAgcmV0dXJuIG5ldyBNdWx0aXBhcnRCb2R5VXNlckRhdGFXcmFwcGVyKHVzZXJEYXRhLCBjb250ZW50VHlwZSk7XG4gICAgfVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgZnJvbVJhd0JvZHkob3B0czogTXVsdGlwYXJ0Qm9keU9wdGlvbnMpOiBNdWx0aXBhcnRCb2R5IHtcbiAgICAgICAgcmV0dXJuIG5ldyBNdWx0aXBhcnRCb2R5UmF3KG9wdHMpO1xuICAgIH1cbiAgICBwdWJsaWMgY29uc3RydWN0b3IoKSB7XG4gICAgfVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgYWJzdHJhY3QgcmVuZGVyQm9keVBhcnQoKTogc3RyaW5nW107XG59XG4vKipcbiAqIFRoZSByYXcgcGFydCBvZiBtdWx0aS1wYXJ0IHVzZXIgZGF0YSwgd2hpY2ggY2FuIGJlIGFkZGVkIHRvIHtAbGluayBNdWx0aXBhcnRVc2VyRGF0YX0uXG4gKi9cbmNsYXNzIE11bHRpcGFydEJvZHlSYXcgZXh0ZW5kcyBNdWx0aXBhcnRCb2R5IHtcbiAgICBwdWJsaWMgY29uc3RydWN0b3IocHJpdmF0ZSByZWFkb25seSBwcm9wczogTXVsdGlwYXJ0Qm9keU9wdGlvbnMpIHtcbiAgICAgICAgc3VwZXIoKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogUmVuZGVyIGJvZHkgcGFydCBhcyB0aGUgc3RyaW5nLlxuICAgICAqL1xuICAgIHB1YmxpYyByZW5kZXJCb2R5UGFydCgpOiBzdHJpbmdbXSB7XG4gICAgICAgIGNvbnN0IHJlc3VsdDogc3RyaW5nW10gPSBbXTtcbiAgICAgICAgcmVzdWx0LnB1c2goYENvbnRlbnQtVHlwZTogJHt0aGlzLnByb3BzLmNvbnRlbnRUeXBlfWApO1xuICAgICAgICBpZiAodGhpcy5wcm9wcy50cmFuc2ZlckVuY29kaW5nICE9IG51bGwpIHtcbiAgICAgICAgICAgIHJlc3VsdC5wdXNoKGBDb250ZW50LVRyYW5zZmVyLUVuY29kaW5nOiAke3RoaXMucHJvcHMudHJhbnNmZXJFbmNvZGluZ31gKTtcbiAgICAgICAgfVxuICAgICAgICAvLyBPbmUgbGluZSBmcmVlIGFmdGVyIHNlcGFyYXRvclxuICAgICAgICByZXN1bHQucHVzaCgnJyk7XG4gICAgICAgIGlmICh0aGlzLnByb3BzLmJvZHkgIT0gbnVsbCkge1xuICAgICAgICAgICAgcmVzdWx0LnB1c2godGhpcy5wcm9wcy5ib2R5KTtcbiAgICAgICAgICAgIC8vIFRoZSBuZXcgbGluZSBhZGRlZCBhZnRlciBqb2luIHdpbGwgYmUgY29uc3VtZWQgYnkgZW5jYXBzdWxhdGluZyBvciBjbG9zaW5nIGJvdW5kYXJ5XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHJlc3VsdDtcbiAgICB9XG59XG4vKipcbiAqIFdyYXBwZXIgZm9yIGBVc2VyRGF0YWAuXG4gKi9cbmNsYXNzIE11bHRpcGFydEJvZHlVc2VyRGF0YVdyYXBwZXIgZXh0ZW5kcyBNdWx0aXBhcnRCb2R5IHtcbiAgICBwcml2YXRlIHJlYWRvbmx5IGNvbnRlbnRUeXBlOiBzdHJpbmc7XG4gICAgcHVibGljIGNvbnN0cnVjdG9yKHByaXZhdGUgcmVhZG9ubHkgdXNlckRhdGE6IFVzZXJEYXRhLCBjb250ZW50VHlwZT86IHN0cmluZykge1xuICAgICAgICBzdXBlcigpO1xuICAgICAgICB0aGlzLmNvbnRlbnRUeXBlID0gY29udGVudFR5cGUgfHwgTXVsdGlwYXJ0Qm9keS5TSEVMTF9TQ1JJUFQ7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFJlbmRlciBib2R5IHBhcnQgYXMgdGhlIHN0cmluZy5cbiAgICAgKi9cbiAgICBwdWJsaWMgcmVuZGVyQm9keVBhcnQoKTogc3RyaW5nW10ge1xuICAgICAgICBjb25zdCByZXN1bHQ6IHN0cmluZ1tdID0gW107XG4gICAgICAgIHJlc3VsdC5wdXNoKGBDb250ZW50LVR5cGU6ICR7dGhpcy5jb250ZW50VHlwZX1gKTtcbiAgICAgICAgcmVzdWx0LnB1c2goJ0NvbnRlbnQtVHJhbnNmZXItRW5jb2Rpbmc6IGJhc2U2NCcpO1xuICAgICAgICByZXN1bHQucHVzaCgnJyk7XG4gICAgICAgIHJlc3VsdC5wdXNoKEZuLmJhc2U2NCh0aGlzLnVzZXJEYXRhLnJlbmRlcigpKSk7XG4gICAgICAgIHJldHVybiByZXN1bHQ7XG4gICAgfVxufVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIE11bHRpcGFydFVzZXJEYXRhT3B0aW9ucyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IHBhcnRzU2VwYXJhdG9yPzogc3RyaW5nO1xufVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgTXVsdGlwYXJ0VXNlckRhdGEgZXh0ZW5kcyBVc2VyRGF0YSB7XG4gICAgcHJpdmF0ZSBzdGF0aWMgcmVhZG9ubHkgVVNFX1BBUlRfRVJST1IgPSAnTXVsdGlwYXJ0VXNlckRhdGEgZG9lcyBub3Qgc3VwcG9ydCB0aGlzIG9wZXJhdGlvbi4gUGxlYXNlIGFkZCBwYXJ0IHVzaW5nIGFkZFBhcnQuJztcbiAgICBwcml2YXRlIHN0YXRpYyByZWFkb25seSBCT1VORFJZX1BBVFRFUk4gPSAnW15hLXpBLVowLTkoKSssLS4vOj0/XSc7XG4gICAgcHJpdmF0ZSBwYXJ0czogTXVsdGlwYXJ0Qm9keVtdID0gW107XG4gICAgcHJpdmF0ZSBvcHRzOiBNdWx0aXBhcnRVc2VyRGF0YU9wdGlvbnM7XG4gICAgY29uc3RydWN0b3Iob3B0cz86IE11bHRpcGFydFVzZXJEYXRhT3B0aW9ucykge1xuICAgICAgICBzdXBlcigpO1xuICAgICAgICBsZXQgcGFydHNTZXBhcmF0b3I6IHN0cmluZztcbiAgICAgICAgLy8gVmFsaWRhdGUgc2VwYXJhdG9yXG4gICAgICAgIGlmIChvcHRzPy5wYXJ0c1NlcGFyYXRvciAhPSBudWxsKSB7XG4gICAgICAgICAgICBpZiAobmV3IFJlZ0V4cChNdWx0aXBhcnRVc2VyRGF0YS5CT1VORFJZX1BBVFRFUk4pLnRlc3Qob3B0cyEucGFydHNTZXBhcmF0b3IpKSB7XG4gICAgICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBJbnZhbGlkIGNoYXJhY3RlcnMgaW4gc2VwYXJhdG9yLiBTZXBhcmF0b3IgaGFzIHRvIG1hdGNoIHBhdHRlcm4gJHtNdWx0aXBhcnRVc2VyRGF0YS5CT1VORFJZX1BBVFRFUk59YCk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBlbHNlIHtcbiAgICAgICAgICAgICAgICBwYXJ0c1NlcGFyYXRvciA9IG9wdHMhLnBhcnRzU2VwYXJhdG9yO1xuICAgICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICAgIGVsc2Uge1xuICAgICAgICAgICAgcGFydHNTZXBhcmF0b3IgPSAnK0FXUytDREsrVXNlcitEYXRhK1NlcGFyYXRvcj09JztcbiAgICAgICAgfVxuICAgICAgICB0aGlzLm9wdHMgPSB7XG4gICAgICAgICAgICBwYXJ0c1NlcGFyYXRvcjogcGFydHNTZXBhcmF0b3IsXG4gICAgICAgIH07XG4gICAgfVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgYWRkUGFydChwYXJ0OiBNdWx0aXBhcnRCb2R5KSB7XG4gICAgICAgIHRoaXMucGFydHMucHVzaChwYXJ0KTtcbiAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgYWRkVXNlckRhdGFQYXJ0KHVzZXJEYXRhOiBVc2VyRGF0YSwgY29udGVudFR5cGU/OiBzdHJpbmcpIHtcbiAgICAgICAgdGhpcy5hZGRQYXJ0KE11bHRpcGFydEJvZHkuZnJvbVVzZXJEYXRhKHVzZXJEYXRhLCBjb250ZW50VHlwZSkpO1xuICAgIH1cbiAgICBwdWJsaWMgcmVuZGVyKCk6IHN0cmluZyB7XG4gICAgICAgIGNvbnN0IGJvdW5kYXJ5ID0gdGhpcy5vcHRzLnBhcnRzU2VwYXJhdG9yO1xuICAgICAgICAvLyBOb3cgYnVpbGQgZmluYWwgTUlNRSBhcmNoaXZlIC0gdGhlcmUgYXJlIGZldyBjaGFuZ2VzIGZyb20gTUlNRSBtZXNzYWdlIHdoaWNoIGFyZSBhY2NlcHRlZCBieSBjbG91ZC1pbml0OlxuICAgICAgICAvLyAtIE1JTUUgUkZDIHVzZXMgQ1JMRiB0byBzZXBhcmF0ZSBsaW5lcyAtIGNsb3VkLWluaXQgaXMgZmluZSB3aXRoIExGIFxcbiBvbmx5XG4gICAgICAgIC8vIE5vdGU6IG5ldyBsaW5lcyBtYXR0ZXJzLCBtYXR0ZXJzIGEgbG90LlxuICAgICAgICB2YXIgcmVzdWx0QXJjaGl2ZSA9IG5ldyBBcnJheTxzdHJpbmc+KCk7XG4gICAgICAgIHJlc3VsdEFyY2hpdmUucHVzaChgQ29udGVudC1UeXBlOiBtdWx0aXBhcnQvbWl4ZWQ7IGJvdW5kYXJ5PVwiJHtib3VuZGFyeX1cImApO1xuICAgICAgICByZXN1bHRBcmNoaXZlLnB1c2goJ01JTUUtVmVyc2lvbjogMS4wJyk7XG4gICAgICAgIC8vIEFkZCBuZXcgbGluZSwgdGhlIG5leHQgb25lIHdpbGwgYmUgYm91bmRhcnkgKGVuY2Fwc3VsYXRpbmcgb3IgY2xvc2luZylcbiAgICAgICAgLy8gc28gdGhpcyBsaW5lIHdpbGwgY291bnQgaW50byBpdC5cbiAgICAgICAgcmVzdWx0QXJjaGl2ZS5wdXNoKCcnKTtcbiAgICAgICAgLy8gQWRkIHBhcnRzIC0gZWFjaCBwYXJ0IHN0YXJ0cyB3aXRoIGJvdW5kYXJ5XG4gICAgICAgIHRoaXMucGFydHMuZm9yRWFjaChwYXJ0ID0+IHtcbiAgICAgICAgICAgIHJlc3VsdEFyY2hpdmUucHVzaChgLS0ke2JvdW5kYXJ5fWApO1xuICAgICAgICAgICAgcmVzdWx0QXJjaGl2ZS5wdXNoKC4uLnBhcnQucmVuZGVyQm9keVBhcnQoKSk7XG4gICAgICAgIH0pO1xuICAgICAgICAvLyBBZGQgY2xvc2luZyBib3VuZGFyeVxuICAgICAgICByZXN1bHRBcmNoaXZlLnB1c2goYC0tJHtib3VuZGFyeX0tLWApO1xuICAgICAgICByZXN1bHRBcmNoaXZlLnB1c2goJycpOyAvLyBGb3JjZSBuZXcgbGluZSBhdCB0aGUgZW5kXG4gICAgICAgIHJldHVybiByZXN1bHRBcmNoaXZlLmpvaW4oJ1xcbicpO1xuICAgIH1cbiAgICBwdWJsaWMgYWRkUzNEb3dubG9hZENvbW1hbmQoX3BhcmFtczogUzNEb3dubG9hZE9wdGlvbnMpOiBzdHJpbmcge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoTXVsdGlwYXJ0VXNlckRhdGEuVVNFX1BBUlRfRVJST1IpO1xuICAgIH1cbiAgICBwdWJsaWMgYWRkRXhlY3V0ZUZpbGVDb21tYW5kKF9wYXJhbXM6IEV4ZWN1dGVGaWxlT3B0aW9ucyk6IHZvaWQge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoTXVsdGlwYXJ0VXNlckRhdGEuVVNFX1BBUlRfRVJST1IpO1xuICAgIH1cbiAgICBwdWJsaWMgYWRkU2lnbmFsT25FeGl0Q29tbWFuZChfcmVzb3VyY2U6IFJlc291cmNlKTogdm9pZCB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihNdWx0aXBhcnRVc2VyRGF0YS5VU0VfUEFSVF9FUlJPUik7XG4gICAgfVxuICAgIHB1YmxpYyBhZGRDb21tYW5kcyguLi5fY29tbWFuZHM6IHN0cmluZ1tdKTogdm9pZCB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihNdWx0aXBhcnRVc2VyRGF0YS5VU0VfUEFSVF9FUlJPUik7XG4gICAgfVxuICAgIHB1YmxpYyBhZGRPbkV4aXRDb21tYW5kcyguLi5fY29tbWFuZHM6IHN0cmluZ1tdKTogdm9pZCB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihNdWx0aXBhcnRVc2VyRGF0YS5VU0VfUEFSVF9FUlJPUik7XG4gICAgfVxufVxuIl19