"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecurityGroup = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cxschema = require("../../cloud-assembly-schema"); // Automatically re-written from '@aws-cdk/cloud-assembly-schema'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const connections_1 = require("./connections");
const ec2_generated_1 = require("./ec2.generated");
const peer_1 = require("./peer");
const port_1 = require("./port");
const SECURITY_GROUP_SYMBOL = Symbol.for('@aws-cdk/iam.SecurityGroup');
const SECURITY_GROUP_DISABLE_INLINE_RULES_CONTEXT_KEY = '@aws-cdk/aws-ec2.securityGroupDisableInlineRules';
/**
 * A SecurityGroup that is not created in this template
 */
class SecurityGroupBase extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, props);
        /**
         * (experimental) Whether the rule can be inlined into a SecurityGroup or not.
         *
         * @experimental
         */
        this.canInlineRule = false;
        /**
         * @experimental
         */
        this.connections = new connections_1.Connections({ securityGroups: [this] });
        Object.defineProperty(this, SECURITY_GROUP_SYMBOL, { value: true });
    }
    /**
     * (experimental) Return whether the indicated object is a security group.
     *
     * @experimental
     */
    static isSecurityGroup(x) {
        return SECURITY_GROUP_SYMBOL in x;
    }
    /**
     * (experimental) A unique identifier for this connection peer.
     *
     * @experimental
     */
    get uniqueId() {
        return core_1.Names.nodeUniqueId(this.node);
    }
    addIngressRule(peer, connection, description, remoteRule) {
        if (description === undefined) {
            description = `from ${peer.uniqueId}:${connection}`;
        }
        const [scope, id] = determineRuleScope(this, peer, connection, 'from', remoteRule);
        // Skip duplicates
        if (scope.node.tryFindChild(id) === undefined) {
            new ec2_generated_1.CfnSecurityGroupIngress(scope, id, {
                groupId: this.securityGroupId,
                ...peer.toIngressRuleConfig(),
                ...connection.toRuleJson(),
                description,
            });
        }
    }
    addEgressRule(peer, connection, description, remoteRule) {
        if (description === undefined) {
            description = `to ${peer.uniqueId}:${connection}`;
        }
        const [scope, id] = determineRuleScope(this, peer, connection, 'to', remoteRule);
        // Skip duplicates
        if (scope.node.tryFindChild(id) === undefined) {
            new ec2_generated_1.CfnSecurityGroupEgress(scope, id, {
                groupId: this.securityGroupId,
                ...peer.toEgressRuleConfig(),
                ...connection.toRuleJson(),
                description,
            });
        }
    }
    /**
     * (experimental) Produce the ingress rule JSON for the given connection.
     *
     * @experimental
     */
    toIngressRuleConfig() {
        return { sourceSecurityGroupId: this.securityGroupId };
    }
    /**
     * (experimental) Produce the egress rule JSON for the given connection.
     *
     * @experimental
     */
    toEgressRuleConfig() {
        return { destinationSecurityGroupId: this.securityGroupId };
    }
}
/**
 * Determine where to parent a new ingress/egress rule
 *
 * A SecurityGroup rule is parented under the group it's related to, UNLESS
 * we're in a cross-stack scenario with another Security Group. In that case,
 * we respect the 'remoteRule' flag and will parent under the other security
 * group.
 *
 * This is necessary to avoid cyclic dependencies between stacks, since both
 * ingress and egress rules will reference both security groups, and a naive
 * parenting will lead to the following situation:
 *
 *   ╔════════════════════╗         ╔════════════════════╗
 *   ║  ┌───────────┐     ║         ║    ┌───────────┐   ║
 *   ║  │  GroupA   │◀────╬─┐   ┌───╬───▶│  GroupB   │   ║
 *   ║  └───────────┘     ║ │   │   ║    └───────────┘   ║
 *   ║        ▲           ║ │   │   ║          ▲         ║
 *   ║        │           ║ │   │   ║          │         ║
 *   ║        │           ║ │   │   ║          │         ║
 *   ║  ┌───────────┐     ║ └───┼───╬────┌───────────┐   ║
 *   ║  │  EgressA  │─────╬─────┘   ║    │ IngressB  │   ║
 *   ║  └───────────┘     ║         ║    └───────────┘   ║
 *   ║                    ║         ║                    ║
 *   ╚════════════════════╝         ╚════════════════════╝
 *
 * By having the ability to switch the parent, we avoid the cyclic reference by
 * keeping all rules in a single stack.
 *
 * If this happens, we also have to change the construct ID, because
 * otherwise we might have two objects with the same ID if we have
 * multiple reversed security group relationships.
 *
 *   ╔═══════════════════════════════════╗
 *   ║┌───────────┐                      ║
 *   ║│  GroupB   │                      ║
 *   ║└───────────┘                      ║
 *   ║      ▲                            ║
 *   ║      │              ┌───────────┐ ║
 *   ║      ├────"from A"──│ IngressB  │ ║
 *   ║      │              └───────────┘ ║
 *   ║      │              ┌───────────┐ ║
 *   ║      ├─────"to B"───│  EgressA  │ ║
 *   ║      │              └───────────┘ ║
 *   ║      │              ┌───────────┐ ║
 *   ║      └─────"to B"───│  EgressC  │ ║  <-- oops
 *   ║                     └───────────┘ ║
 *   ╚═══════════════════════════════════╝
 */
function determineRuleScope(group, peer, connection, fromTo, remoteRule) {
    if (remoteRule && SecurityGroupBase.isSecurityGroup(peer) && differentStacks(group, peer)) {
        // Reversed
        const reversedFromTo = fromTo === 'from' ? 'to' : 'from';
        return [peer, `${group.uniqueId}:${connection} ${reversedFromTo}`];
    }
    else {
        // Regular (do old ID escaping to in order to not disturb existing deployments)
        return [group, `${fromTo} ${renderPeer(peer)}:${connection}`.replace('/', '_')];
    }
}
function renderPeer(peer) {
    return core_1.Token.isUnresolved(peer.uniqueId) ? '{IndirectPeer}' : peer.uniqueId;
}
function differentStacks(group1, group2) {
    return core_1.Stack.of(group1) !== core_1.Stack.of(group2);
}
/**
 * (experimental) Creates an Amazon EC2 security group within a VPC.
 *
 * Security Groups act like a firewall with a set of rules, and are associated
 * with any AWS resource that has or creates Elastic Network Interfaces (ENIs).
 * A typical example of a resource that has a security group is an Instance (or
 * Auto Scaling Group of instances)
 *
 * If you are defining new infrastructure in CDK, there is a good chance you
 * won't have to interact with this class at all. Like IAM Roles, Security
 * Groups need to exist to control access between AWS resources, but CDK will
 * automatically generate and populate them with least-privilege permissions
 * for you so you can concentrate on your business logic.
 *
 * All Constructs that require Security Groups will create one for you if you
 * don't specify one at construction. After construction, you can selectively
 * allow connections to and between constructs via--for example-- the `instance.connections`
 * object. Think of it as "allowing connections to your instance", rather than
 * "adding ingress rules a security group". See the [Allowing
 * Connections](https://docs.aws.amazon.com/cdk/api/latest/docs/aws-ec2-readme.html#allowing-connections)
 * section in the library documentation for examples.
 *
 * Direct manipulation of the Security Group through `addIngressRule` and
 * `addEgressRule` is possible, but mutation through the `.connections` object
 * is recommended. If you peer two constructs with security groups this way,
 * appropriate rules will be created in both.
 *
 * If you have an existing security group you want to use in your CDK application,
 * you would import it like this:
 *
 * ```ts
 * const securityGroup = SecurityGroup.fromSecurityGroupId(this, 'SG', 'sg-12345', {
 *    mutable: false
 * });
 * ```
 *
 * @experimental
 */
class SecurityGroup extends SecurityGroupBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.securityGroupName,
        });
        this.directIngressRules = [];
        this.directEgressRules = [];
        const groupDescription = props.description || this.node.path;
        this.allowAllOutbound = props.allowAllOutbound !== false;
        this.disableInlineRules = props.disableInlineRules !== undefined ?
            !!props.disableInlineRules :
            !!this.node.tryGetContext(SECURITY_GROUP_DISABLE_INLINE_RULES_CONTEXT_KEY);
        this.securityGroup = new ec2_generated_1.CfnSecurityGroup(this, 'Resource', {
            groupName: this.physicalName,
            groupDescription,
            securityGroupIngress: core_1.Lazy.any({ produce: () => this.directIngressRules }, { omitEmptyArray: true }),
            securityGroupEgress: core_1.Lazy.any({ produce: () => this.directEgressRules }, { omitEmptyArray: true }),
            vpcId: props.vpc.vpcId,
        });
        this.securityGroupId = this.securityGroup.attrGroupId;
        this.securityGroupVpcId = this.securityGroup.attrVpcId;
        this.securityGroupName = this.securityGroup.ref;
        this.addDefaultEgressRule();
    }
    /**
     * (experimental) Look up a security group by id.
     *
     * @experimental
     */
    static fromLookup(scope, id, securityGroupId) {
        if (core_1.Token.isUnresolved(securityGroupId)) {
            throw new Error('All arguments to look up a security group must be concrete (no Tokens)');
        }
        const attributes = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.SECURITY_GROUP_PROVIDER,
            props: { securityGroupId },
            dummyValue: {
                securityGroupId: 'sg-12345',
                allowAllOutbound: true,
            },
        }).value;
        return SecurityGroup.fromSecurityGroupId(scope, id, attributes.securityGroupId, {
            allowAllOutbound: attributes.allowAllOutbound,
            mutable: true,
        });
    }
    /**
     * (experimental) Import an existing security group into this app.
     *
     * This method will assume that the Security Group has a rule in it which allows
     * all outbound traffic, and so will not add egress rules to the imported Security
     * Group (only ingress rules).
     *
     * If your existing Security Group needs to have egress rules added, pass the
     * `allowAllOutbound: false` option on import.
     *
     * @experimental
     */
    static fromSecurityGroupId(scope, id, securityGroupId, options = {}) {
        class MutableImport extends SecurityGroupBase {
            constructor() {
                var _b;
                super(...arguments);
                this.securityGroupId = securityGroupId;
                this.allowAllOutbound = (_b = options.allowAllOutbound) !== null && _b !== void 0 ? _b : true;
            }
            addEgressRule(peer, connection, description, remoteRule) {
                // Only if allowAllOutbound has been disabled
                if (options.allowAllOutbound === false) {
                    super.addEgressRule(peer, connection, description, remoteRule);
                }
            }
        }
        class ImmutableImport extends SecurityGroupBase {
            constructor() {
                var _b;
                super(...arguments);
                this.securityGroupId = securityGroupId;
                this.allowAllOutbound = (_b = options.allowAllOutbound) !== null && _b !== void 0 ? _b : true;
            }
            addEgressRule(_peer, _connection, _description, _remoteRule) {
                // do nothing
            }
            addIngressRule(_peer, _connection, _description, _remoteRule) {
                // do nothing
            }
        }
        return options.mutable !== false
            ? new MutableImport(scope, id)
            : new ImmutableImport(scope, id);
    }
    /**
     * (experimental) Add an ingress rule for the current security group.
     *
     * `remoteRule` controls where the Rule object is created if the peer is also a
     * securityGroup and they are in different stack. If false (default) the
     * rule object is created under the current SecurityGroup object. If true and the
     * peer is also a SecurityGroup, the rule object is created under the remote
     * SecurityGroup object.
     *
     * @experimental
     */
    addIngressRule(peer, connection, description, remoteRule) {
        if (!peer.canInlineRule || !connection.canInlineRule || this.disableInlineRules) {
            super.addIngressRule(peer, connection, description, remoteRule);
            return;
        }
        if (description === undefined) {
            description = `from ${peer.uniqueId}:${connection}`;
        }
        this.addDirectIngressRule({
            ...peer.toIngressRuleConfig(),
            ...connection.toRuleJson(),
            description,
        });
    }
    /**
     * (experimental) Add an egress rule for the current security group.
     *
     * `remoteRule` controls where the Rule object is created if the peer is also a
     * securityGroup and they are in different stack. If false (default) the
     * rule object is created under the current SecurityGroup object. If true and the
     * peer is also a SecurityGroup, the rule object is created under the remote
     * SecurityGroup object.
     *
     * @experimental
     */
    addEgressRule(peer, connection, description, remoteRule) {
        if (this.allowAllOutbound) {
            // In the case of "allowAllOutbound", we don't add any more rules. There
            // is only one rule which allows all traffic and that subsumes any other
            // rule.
            core_1.Annotations.of(this).addWarning('Ignoring Egress rule since \'allowAllOutbound\' is set to true; To add customize rules, set allowAllOutbound=false on the SecurityGroup');
            return;
        }
        else {
            // Otherwise, if the bogus rule exists we can now remove it because the
            // presence of any other rule will get rid of EC2's implicit "all
            // outbound" rule anyway.
            this.removeNoTrafficRule();
        }
        if (!peer.canInlineRule || !connection.canInlineRule || this.disableInlineRules) {
            super.addEgressRule(peer, connection, description, remoteRule);
            return;
        }
        if (description === undefined) {
            description = `from ${peer.uniqueId}:${connection}`;
        }
        const rule = {
            ...peer.toEgressRuleConfig(),
            ...connection.toRuleJson(),
            description,
        };
        if (isAllTrafficRule(rule)) {
            // We cannot allow this; if someone adds the rule in this way, it will be
            // removed again if they add other rules. We also can't automatically switch
            // to "allOutbound=true" mode, because we might have already emitted
            // EgressRule objects (which count as rules added later) and there's no way
            // to recall those. Better to prevent this for now.
            throw new Error('Cannot add an "all traffic" egress rule in this way; set allowAllOutbound=true on the SecurityGroup instead.');
        }
        this.addDirectEgressRule(rule);
    }
    /**
     * Add a direct ingress rule
     */
    addDirectIngressRule(rule) {
        if (!this.hasIngressRule(rule)) {
            this.directIngressRules.push(rule);
        }
    }
    /**
     * Return whether the given ingress rule exists on the group
     */
    hasIngressRule(rule) {
        return this.directIngressRules.findIndex(r => ingressRulesEqual(r, rule)) > -1;
    }
    /**
     * Add a direct egress rule
     */
    addDirectEgressRule(rule) {
        if (!this.hasEgressRule(rule)) {
            this.directEgressRules.push(rule);
        }
    }
    /**
     * Return whether the given egress rule exists on the group
     */
    hasEgressRule(rule) {
        return this.directEgressRules.findIndex(r => egressRulesEqual(r, rule)) > -1;
    }
    /**
     * Add the default egress rule to the securityGroup
     *
     * This depends on allowAllOutbound:
     *
     * - If allowAllOutbound is true, we *TECHNICALLY* don't need to do anything, because
     *   EC2 is going to create this default rule anyway. But, for maximum readability
     *   of the template, we will add one anyway.
     * - If allowAllOutbound is false, we add a bogus rule that matches no traffic in
     *   order to get rid of the default "all outbound" rule that EC2 creates by default.
     *   If other rules happen to get added later, we remove the bogus rule again so
     *   that it doesn't clutter up the template too much (even though that's not
     *   strictly necessary).
     */
    addDefaultEgressRule() {
        if (this.disableInlineRules) {
            const peer = this.allowAllOutbound ? ALL_TRAFFIC_PEER : NO_TRAFFIC_PEER;
            const port = this.allowAllOutbound ? ALL_TRAFFIC_PORT : NO_TRAFFIC_PORT;
            const description = this.allowAllOutbound ? ALLOW_ALL_RULE.description : MATCH_NO_TRAFFIC.description;
            super.addEgressRule(peer, port, description, false);
        }
        else {
            const rule = this.allowAllOutbound ? ALLOW_ALL_RULE : MATCH_NO_TRAFFIC;
            this.directEgressRules.push(rule);
        }
    }
    /**
     * Remove the bogus rule if it exists
     */
    removeNoTrafficRule() {
        if (this.disableInlineRules) {
            const [scope, id] = determineRuleScope(this, NO_TRAFFIC_PEER, NO_TRAFFIC_PORT, 'to', false);
            scope.node.tryRemoveChild(id);
        }
        else {
            const i = this.directEgressRules.findIndex(r => egressRulesEqual(r, MATCH_NO_TRAFFIC));
            if (i > -1) {
                this.directEgressRules.splice(i, 1);
            }
        }
    }
}
exports.SecurityGroup = SecurityGroup;
_a = JSII_RTTI_SYMBOL_1;
SecurityGroup[_a] = { fqn: "monocdk.aws_ec2.SecurityGroup", version: "1.106.1" };
/**
 * Egress rule that purposely matches no traffic
 *
 * This is used in order to disable the "all traffic" default of Security Groups.
 *
 * No machine can ever actually have the 255.255.255.255 IP address, but
 * in order to lock it down even more we'll restrict to a nonexistent
 * ICMP traffic type.
 */
const MATCH_NO_TRAFFIC = {
    cidrIp: '255.255.255.255/32',
    description: 'Disallow all traffic',
    ipProtocol: 'icmp',
    fromPort: 252,
    toPort: 86,
};
const NO_TRAFFIC_PEER = peer_1.Peer.ipv4(MATCH_NO_TRAFFIC.cidrIp);
const NO_TRAFFIC_PORT = port_1.Port.icmpTypeAndCode(MATCH_NO_TRAFFIC.fromPort, MATCH_NO_TRAFFIC.toPort);
/**
 * Egress rule that matches all traffic
 */
const ALLOW_ALL_RULE = {
    cidrIp: '0.0.0.0/0',
    description: 'Allow all outbound traffic by default',
    ipProtocol: '-1',
};
const ALL_TRAFFIC_PEER = peer_1.Peer.anyIpv4();
const ALL_TRAFFIC_PORT = port_1.Port.allTraffic();
/**
 * Compare two ingress rules for equality the same way CloudFormation would (discarding description)
 */
function ingressRulesEqual(a, b) {
    return a.cidrIp === b.cidrIp
        && a.cidrIpv6 === b.cidrIpv6
        && a.fromPort === b.fromPort
        && a.toPort === b.toPort
        && a.ipProtocol === b.ipProtocol
        && a.sourceSecurityGroupId === b.sourceSecurityGroupId
        && a.sourceSecurityGroupName === b.sourceSecurityGroupName
        && a.sourceSecurityGroupOwnerId === b.sourceSecurityGroupOwnerId;
}
/**
 * Compare two egress rules for equality the same way CloudFormation would (discarding description)
 */
function egressRulesEqual(a, b) {
    return a.cidrIp === b.cidrIp
        && a.cidrIpv6 === b.cidrIpv6
        && a.fromPort === b.fromPort
        && a.toPort === b.toPort
        && a.ipProtocol === b.ipProtocol
        && a.destinationPrefixListId === b.destinationPrefixListId
        && a.destinationSecurityGroupId === b.destinationSecurityGroupId;
}
/**
 * Whether this rule refers to all traffic
 */
function isAllTrafficRule(rule) {
    return rule.cidrIp === '0.0.0.0/0' && rule.ipProtocol === '-1';
}
//# sourceMappingURL=data:application/json;base64,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