"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NatInstanceImage = exports.NatInstanceProvider = exports.NatProvider = exports.NatTrafficDirection = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const connections_1 = require("./connections");
const instance_1 = require("./instance");
const machine_image_1 = require("./machine-image");
const port_1 = require("./port");
const security_group_1 = require("./security-group");
const vpc_1 = require("./vpc");
/**
 * (experimental) Direction of traffic to allow all by default.
 *
 * @experimental
 */
var NatTrafficDirection;
(function (NatTrafficDirection) {
    NatTrafficDirection["OUTBOUND_ONLY"] = "OUTBOUND_ONLY";
    NatTrafficDirection["INBOUND_AND_OUTBOUND"] = "INBOUND_AND_OUTBOUND";
    NatTrafficDirection["NONE"] = "NONE";
})(NatTrafficDirection = exports.NatTrafficDirection || (exports.NatTrafficDirection = {}));
/**
 * (experimental) NAT providers.
 *
 * Determines what type of NAT provider to create, either NAT gateways or NAT
 * instance.
 *
 * @experimental
 */
class NatProvider {
    /**
     * (experimental) Use NAT Gateways to provide NAT services for your VPC.
     *
     * NAT gateways are managed by AWS.
     *
     * @see https://docs.aws.amazon.com/vpc/latest/userguide/vpc-nat-gateway.html
     * @experimental
     */
    static gateway(props = {}) {
        return new NatGatewayProvider(props);
    }
    /**
     * (experimental) Use NAT instances to provide NAT services for your VPC.
     *
     * NAT instances are managed by you, but in return allow more configuration.
     *
     * Be aware that instances created using this provider will not be
     * automatically replaced if they are stopped for any reason. You should implement
     * your own NatProvider based on AutoScaling groups if you need that.
     *
     * @see https://docs.aws.amazon.com/vpc/latest/userguide/VPC_NAT_Instance.html
     * @experimental
     */
    static instance(props) {
        return new NatInstanceProvider(props);
    }
}
exports.NatProvider = NatProvider;
_a = JSII_RTTI_SYMBOL_1;
NatProvider[_a] = { fqn: "monocdk.aws_ec2.NatProvider", version: "1.106.1" };
/**
 * Provider for NAT Gateways
 */
class NatGatewayProvider extends NatProvider {
    constructor(props = {}) {
        super();
        this.props = props;
        this.gateways = new PrefSet();
    }
    configureNat(options) {
        // Create the NAT gateways
        let i = 0;
        for (const sub of options.natSubnets) {
            const gateway = sub.addNatGateway();
            if (this.props.eipAllocationIds) {
                gateway.allocationId = pickN(i, this.props.eipAllocationIds);
            }
            this.gateways.add(sub.availabilityZone, gateway.ref);
            i++;
        }
        // Add routes to them in the private subnets
        for (const sub of options.privateSubnets) {
            this.configureSubnet(sub);
        }
    }
    configureSubnet(subnet) {
        const az = subnet.availabilityZone;
        const gatewayId = this.gateways.pick(az);
        subnet.addRoute('DefaultRoute', {
            routerType: vpc_1.RouterType.NAT_GATEWAY,
            routerId: gatewayId,
            enablesInternetConnectivity: true,
        });
    }
    get configuredGateways() {
        return this.gateways.values().map(x => ({ az: x[0], gatewayId: x[1] }));
    }
}
/**
 * (experimental) NAT provider which uses NAT Instances.
 *
 * @experimental
 */
class NatInstanceProvider extends NatProvider {
    /**
     * @experimental
     */
    constructor(props) {
        super();
        this.props = props;
        this.gateways = new PrefSet();
        if (props.defaultAllowedTraffic !== undefined && props.allowAllTraffic !== undefined) {
            throw new Error('Can not specify both of \'defaultAllowedTraffic\' and \'defaultAllowedTraffic\'; prefer \'defaultAllowedTraffic\'');
        }
    }
    /**
     * (experimental) Called by the VPC to configure NAT.
     *
     * Don't call this directly, the VPC will call it automatically.
     *
     * @experimental
     */
    configureNat(options) {
        var _d, _e, _f;
        const defaultDirection = (_d = this.props.defaultAllowedTraffic) !== null && _d !== void 0 ? _d : (((_e = this.props.allowAllTraffic) !== null && _e !== void 0 ? _e : true) ? NatTrafficDirection.INBOUND_AND_OUTBOUND : NatTrafficDirection.OUTBOUND_ONLY);
        // Create the NAT instances. They can share a security group and a Role.
        const machineImage = this.props.machineImage || new NatInstanceImage();
        this._securityGroup = (_f = this.props.securityGroup) !== null && _f !== void 0 ? _f : new security_group_1.SecurityGroup(options.vpc, 'NatSecurityGroup', {
            vpc: options.vpc,
            description: 'Security Group for NAT instances',
            allowAllOutbound: isOutboundAllowed(defaultDirection),
        });
        this._connections = new connections_1.Connections({ securityGroups: [this._securityGroup] });
        if (isInboundAllowed(defaultDirection)) {
            this.connections.allowFromAnyIpv4(port_1.Port.allTraffic());
        }
        // FIXME: Ideally, NAT instances don't have a role at all, but
        // 'Instance' does not allow that right now.
        const role = new iam.Role(options.vpc, 'NatRole', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
        });
        for (const sub of options.natSubnets) {
            const natInstance = new instance_1.Instance(sub, 'NatInstance', {
                instanceType: this.props.instanceType,
                machineImage,
                sourceDestCheck: false,
                vpc: options.vpc,
                vpcSubnets: { subnets: [sub] },
                securityGroup: this._securityGroup,
                role,
                keyName: this.props.keyName,
            });
            // NAT instance routes all traffic, both ways
            this.gateways.add(sub.availabilityZone, natInstance);
        }
        // Add routes to them in the private subnets
        for (const sub of options.privateSubnets) {
            this.configureSubnet(sub);
        }
    }
    /**
     * (experimental) The Security Group associated with the NAT instances.
     *
     * @experimental
     */
    get securityGroup() {
        if (!this._securityGroup) {
            throw new Error('Pass the NatInstanceProvider to a Vpc before accessing \'securityGroup\'');
        }
        return this._securityGroup;
    }
    /**
     * (experimental) Manage the Security Groups associated with the NAT instances.
     *
     * @experimental
     */
    get connections() {
        if (!this._connections) {
            throw new Error('Pass the NatInstanceProvider to a Vpc before accessing \'connections\'');
        }
        return this._connections;
    }
    /**
     * (experimental) Return list of gateways spawned by the provider.
     *
     * @experimental
     */
    get configuredGateways() {
        return this.gateways.values().map(x => ({ az: x[0], gatewayId: x[1].instanceId }));
    }
    /**
     * (experimental) Configures subnet with the gateway.
     *
     * Don't call this directly, the VPC will call it automatically.
     *
     * @experimental
     */
    configureSubnet(subnet) {
        const az = subnet.availabilityZone;
        const gatewayId = this.gateways.pick(az).instanceId;
        subnet.addRoute('DefaultRoute', {
            routerType: vpc_1.RouterType.INSTANCE,
            routerId: gatewayId,
            enablesInternetConnectivity: true,
        });
    }
}
exports.NatInstanceProvider = NatInstanceProvider;
_b = JSII_RTTI_SYMBOL_1;
NatInstanceProvider[_b] = { fqn: "monocdk.aws_ec2.NatInstanceProvider", version: "1.106.1" };
/**
 * Preferential set
 *
 * Picks the value with the given key if available, otherwise distributes
 * evenly among the available options.
 */
class PrefSet {
    constructor() {
        this.map = {};
        this.vals = new Array();
        this.next = 0;
    }
    add(pref, value) {
        this.map[pref] = value;
        this.vals.push([pref, value]);
    }
    pick(pref) {
        if (this.vals.length === 0) {
            throw new Error('Cannot pick, set is empty');
        }
        if (pref in this.map) {
            return this.map[pref];
        }
        return this.vals[this.next++ % this.vals.length][1];
    }
    values() {
        return this.vals;
    }
}
/**
 * (experimental) Machine image representing the latest NAT instance image.
 *
 * @experimental
 */
class NatInstanceImage extends machine_image_1.LookupMachineImage {
    /**
     * @experimental
     */
    constructor() {
        super({
            name: 'amzn-ami-vpc-nat-*',
            owners: ['amazon'],
        });
    }
}
exports.NatInstanceImage = NatInstanceImage;
_c = JSII_RTTI_SYMBOL_1;
NatInstanceImage[_c] = { fqn: "monocdk.aws_ec2.NatInstanceImage", version: "1.106.1" };
function isOutboundAllowed(direction) {
    return direction === NatTrafficDirection.INBOUND_AND_OUTBOUND ||
        direction === NatTrafficDirection.OUTBOUND_ONLY;
}
function isInboundAllowed(direction) {
    return direction === NatTrafficDirection.INBOUND_AND_OUTBOUND;
}
/**
 * Token-aware pick index function
 */
function pickN(i, xs) {
    if (core_1.Token.isUnresolved(xs)) {
        return core_1.Fn.select(i, xs);
    }
    if (i >= xs.length) {
        throw new Error(`Cannot get element ${i} from ${xs}`);
    }
    return xs[i];
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibmF0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsibmF0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEscUNBQXFDLENBQUMsbURBQW1EO0FBQ3pGLHFDQUF1QyxDQUFDLGdEQUFnRDtBQUN4RiwrQ0FBMEQ7QUFDMUQseUNBQXNDO0FBRXRDLG1EQUFvRTtBQUNwRSxpQ0FBOEI7QUFDOUIscURBQWlFO0FBQ2pFLCtCQUFxRTs7Ozs7O0FBSXJFLElBQVksbUJBYVg7QUFiRCxXQUFZLG1CQUFtQjtJQUkzQixzREFBK0IsQ0FBQTtJQUkvQixvRUFBNkMsQ0FBQTtJQUk3QyxvQ0FBYSxDQUFBO0FBQ2pCLENBQUMsRUFiVyxtQkFBbUIsR0FBbkIsMkJBQW1CLEtBQW5CLDJCQUFtQixRQWE5Qjs7Ozs7Ozs7O0FBc0JELE1BQXNCLFdBQVc7Ozs7Ozs7OztJQVF0QixNQUFNLENBQUMsT0FBTyxDQUFDLFFBQXlCLEVBQUU7UUFDN0MsT0FBTyxJQUFJLGtCQUFrQixDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ3pDLENBQUM7Ozs7Ozs7Ozs7Ozs7SUFZTSxNQUFNLENBQUMsUUFBUSxDQUFDLEtBQXVCO1FBQzFDLE9BQU8sSUFBSSxtQkFBbUIsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUMxQyxDQUFDOztBQXhCTCxrQ0F5Q0M7OztBQXVHRDs7R0FFRztBQUNILE1BQU0sa0JBQW1CLFNBQVEsV0FBVztJQUV4QyxZQUE2QixRQUF5QixFQUFFO1FBQ3BELEtBQUssRUFBRSxDQUFDO1FBRGlCLFVBQUssR0FBTCxLQUFLLENBQXNCO1FBRGhELGFBQVEsR0FBb0IsSUFBSSxPQUFPLEVBQVUsQ0FBQztJQUcxRCxDQUFDO0lBQ00sWUFBWSxDQUFDLE9BQTRCO1FBQzVDLDBCQUEwQjtRQUMxQixJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDVixLQUFLLE1BQU0sR0FBRyxJQUFJLE9BQU8sQ0FBQyxVQUFVLEVBQUU7WUFDbEMsTUFBTSxPQUFPLEdBQUcsR0FBRyxDQUFDLGFBQWEsRUFBRSxDQUFDO1lBQ3BDLElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxnQkFBZ0IsRUFBRTtnQkFDN0IsT0FBTyxDQUFDLFlBQVksR0FBRyxLQUFLLENBQUMsQ0FBQyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsZ0JBQWdCLENBQUMsQ0FBQzthQUNoRTtZQUNELElBQUksQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxnQkFBZ0IsRUFBRSxPQUFPLENBQUMsR0FBRyxDQUFDLENBQUM7WUFDckQsQ0FBQyxFQUFFLENBQUM7U0FDUDtRQUNELDRDQUE0QztRQUM1QyxLQUFLLE1BQU0sR0FBRyxJQUFJLE9BQU8sQ0FBQyxjQUFjLEVBQUU7WUFDdEMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxHQUFHLENBQUMsQ0FBQztTQUM3QjtJQUNMLENBQUM7SUFDTSxlQUFlLENBQUMsTUFBcUI7UUFDeEMsTUFBTSxFQUFFLEdBQUcsTUFBTSxDQUFDLGdCQUFnQixDQUFDO1FBQ25DLE1BQU0sU0FBUyxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDO1FBQ3pDLE1BQU0sQ0FBQyxRQUFRLENBQUMsY0FBYyxFQUFFO1lBQzVCLFVBQVUsRUFBRSxnQkFBVSxDQUFDLFdBQVc7WUFDbEMsUUFBUSxFQUFFLFNBQVM7WUFDbkIsMkJBQTJCLEVBQUUsSUFBSTtTQUNwQyxDQUFDLENBQUM7SUFDUCxDQUFDO0lBQ0QsSUFBVyxrQkFBa0I7UUFDekIsT0FBTyxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsRUFBRSxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLFNBQVMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUM7SUFDNUUsQ0FBQztDQUNKOzs7Ozs7QUFJRCxNQUFhLG1CQUFvQixTQUFRLFdBQVc7Ozs7SUFJaEQsWUFBNkIsS0FBdUI7UUFDaEQsS0FBSyxFQUFFLENBQUM7UUFEaUIsVUFBSyxHQUFMLEtBQUssQ0FBa0I7UUFINUMsYUFBUSxHQUFzQixJQUFJLE9BQU8sRUFBWSxDQUFDO1FBSzFELElBQUksS0FBSyxDQUFDLHFCQUFxQixLQUFLLFNBQVMsSUFBSSxLQUFLLENBQUMsZUFBZSxLQUFLLFNBQVMsRUFBRTtZQUNsRixNQUFNLElBQUksS0FBSyxDQUFDLG1IQUFtSCxDQUFDLENBQUM7U0FDeEk7SUFDTCxDQUFDOzs7Ozs7OztJQUNNLFlBQVksQ0FBQyxPQUE0Qjs7UUFDNUMsTUFBTSxnQkFBZ0IsU0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLHFCQUFxQixtQ0FDckQsQ0FBQyxPQUFBLElBQUksQ0FBQyxLQUFLLENBQUMsZUFBZSxtQ0FBSSxJQUFJLEVBQUMsQ0FBQyxDQUFDLG1CQUFtQixDQUFDLG9CQUFvQixDQUFDLENBQUMsQ0FBQyxtQkFBbUIsQ0FBQyxhQUFhLENBQUMsQ0FBQztRQUN4SCx3RUFBd0U7UUFDeEUsTUFBTSxZQUFZLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxZQUFZLElBQUksSUFBSSxnQkFBZ0IsRUFBRSxDQUFDO1FBQ3ZFLElBQUksQ0FBQyxjQUFjLFNBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxhQUFhLG1DQUFJLElBQUksOEJBQWEsQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLGtCQUFrQixFQUFFO1lBQ2pHLEdBQUcsRUFBRSxPQUFPLENBQUMsR0FBRztZQUNoQixXQUFXLEVBQUUsa0NBQWtDO1lBQy9DLGdCQUFnQixFQUFFLGlCQUFpQixDQUFDLGdCQUFnQixDQUFDO1NBQ3hELENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxZQUFZLEdBQUcsSUFBSSx5QkFBVyxDQUFDLEVBQUUsY0FBYyxFQUFFLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxFQUFFLENBQUMsQ0FBQztRQUMvRSxJQUFJLGdCQUFnQixDQUFDLGdCQUFnQixDQUFDLEVBQUU7WUFDcEMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxnQkFBZ0IsQ0FBQyxXQUFJLENBQUMsVUFBVSxFQUFFLENBQUMsQ0FBQztTQUN4RDtRQUNELDhEQUE4RDtRQUM5RCw0Q0FBNEM7UUFDNUMsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHLEVBQUUsU0FBUyxFQUFFO1lBQzlDLFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxtQkFBbUIsQ0FBQztTQUMzRCxDQUFDLENBQUM7UUFDSCxLQUFLLE1BQU0sR0FBRyxJQUFJLE9BQU8sQ0FBQyxVQUFVLEVBQUU7WUFDbEMsTUFBTSxXQUFXLEdBQUcsSUFBSSxtQkFBUSxDQUFDLEdBQUcsRUFBRSxhQUFhLEVBQUU7Z0JBQ2pELFlBQVksRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFlBQVk7Z0JBQ3JDLFlBQVk7Z0JBQ1osZUFBZSxFQUFFLEtBQUs7Z0JBQ3RCLEdBQUcsRUFBRSxPQUFPLENBQUMsR0FBRztnQkFDaEIsVUFBVSxFQUFFLEVBQUUsT0FBTyxFQUFFLENBQUMsR0FBRyxDQUFDLEVBQUU7Z0JBQzlCLGFBQWEsRUFBRSxJQUFJLENBQUMsY0FBYztnQkFDbEMsSUFBSTtnQkFDSixPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPO2FBQzlCLENBQUMsQ0FBQztZQUNILDZDQUE2QztZQUM3QyxJQUFJLENBQUMsUUFBUSxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsZ0JBQWdCLEVBQUUsV0FBVyxDQUFDLENBQUM7U0FDeEQ7UUFDRCw0Q0FBNEM7UUFDNUMsS0FBSyxNQUFNLEdBQUcsSUFBSSxPQUFPLENBQUMsY0FBYyxFQUFFO1lBQ3RDLElBQUksQ0FBQyxlQUFlLENBQUMsR0FBRyxDQUFDLENBQUM7U0FDN0I7SUFDTCxDQUFDOzs7Ozs7SUFJRCxJQUFXLGFBQWE7UUFDcEIsSUFBSSxDQUFDLElBQUksQ0FBQyxjQUFjLEVBQUU7WUFDdEIsTUFBTSxJQUFJLEtBQUssQ0FBQywwRUFBMEUsQ0FBQyxDQUFDO1NBQy9GO1FBQ0QsT0FBTyxJQUFJLENBQUMsY0FBYyxDQUFDO0lBQy9CLENBQUM7Ozs7OztJQUlELElBQVcsV0FBVztRQUNsQixJQUFJLENBQUMsSUFBSSxDQUFDLFlBQVksRUFBRTtZQUNwQixNQUFNLElBQUksS0FBSyxDQUFDLHdFQUF3RSxDQUFDLENBQUM7U0FDN0Y7UUFDRCxPQUFPLElBQUksQ0FBQyxZQUFZLENBQUM7SUFDN0IsQ0FBQzs7Ozs7O0lBQ0QsSUFBVyxrQkFBa0I7UUFDekIsT0FBTyxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsRUFBRSxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLFNBQVMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsVUFBVSxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBQ3ZGLENBQUM7Ozs7Ozs7O0lBQ00sZUFBZSxDQUFDLE1BQXFCO1FBQ3hDLE1BQU0sRUFBRSxHQUFHLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQztRQUNuQyxNQUFNLFNBQVMsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQyxVQUFVLENBQUM7UUFDcEQsTUFBTSxDQUFDLFFBQVEsQ0FBQyxjQUFjLEVBQUU7WUFDNUIsVUFBVSxFQUFFLGdCQUFVLENBQUMsUUFBUTtZQUMvQixRQUFRLEVBQUUsU0FBUztZQUNuQiwyQkFBMkIsRUFBRSxJQUFJO1NBQ3BDLENBQUMsQ0FBQztJQUNQLENBQUM7O0FBN0VMLGtEQThFQzs7O0FBQ0Q7Ozs7O0dBS0c7QUFDSCxNQUFNLE9BQU87SUFBYjtRQUNxQixRQUFHLEdBQXNCLEVBQUUsQ0FBQztRQUM1QixTQUFJLEdBQUcsSUFBSSxLQUFLLEVBQWUsQ0FBQztRQUN6QyxTQUFJLEdBQVcsQ0FBQyxDQUFDO0lBaUI3QixDQUFDO0lBaEJVLEdBQUcsQ0FBQyxJQUFZLEVBQUUsS0FBUTtRQUM3QixJQUFJLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxHQUFHLEtBQUssQ0FBQztRQUN2QixJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLElBQUksRUFBRSxLQUFLLENBQUMsQ0FBQyxDQUFDO0lBQ2xDLENBQUM7SUFDTSxJQUFJLENBQUMsSUFBWTtRQUNwQixJQUFJLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUN4QixNQUFNLElBQUksS0FBSyxDQUFDLDJCQUEyQixDQUFDLENBQUM7U0FDaEQ7UUFDRCxJQUFJLElBQUksSUFBSSxJQUFJLENBQUMsR0FBRyxFQUFFO1lBQ2xCLE9BQU8sSUFBSSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsQ0FBQztTQUN6QjtRQUNELE9BQU8sSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUN4RCxDQUFDO0lBQ00sTUFBTTtRQUNULE9BQU8sSUFBSSxDQUFDLElBQUksQ0FBQztJQUNyQixDQUFDO0NBQ0o7Ozs7OztBQU1ELE1BQWEsZ0JBQWlCLFNBQVEsa0NBQWtCOzs7O0lBQ3BEO1FBQ0ksS0FBSyxDQUFDO1lBQ0YsSUFBSSxFQUFFLG9CQUFvQjtZQUMxQixNQUFNLEVBQUUsQ0FBQyxRQUFRLENBQUM7U0FDckIsQ0FBQyxDQUFDO0lBQ1AsQ0FBQzs7QUFOTCw0Q0FPQzs7O0FBQ0QsU0FBUyxpQkFBaUIsQ0FBQyxTQUE4QjtJQUNyRCxPQUFPLFNBQVMsS0FBSyxtQkFBbUIsQ0FBQyxvQkFBb0I7UUFDekQsU0FBUyxLQUFLLG1CQUFtQixDQUFDLGFBQWEsQ0FBQztBQUN4RCxDQUFDO0FBQ0QsU0FBUyxnQkFBZ0IsQ0FBQyxTQUE4QjtJQUNwRCxPQUFPLFNBQVMsS0FBSyxtQkFBbUIsQ0FBQyxvQkFBb0IsQ0FBQztBQUNsRSxDQUFDO0FBQ0Q7O0dBRUc7QUFDSCxTQUFTLEtBQUssQ0FBQyxDQUFTLEVBQUUsRUFBWTtJQUNsQyxJQUFJLFlBQUssQ0FBQyxZQUFZLENBQUMsRUFBRSxDQUFDLEVBQUU7UUFDeEIsT0FBTyxTQUFFLENBQUMsTUFBTSxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQztLQUMzQjtJQUNELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQyxNQUFNLEVBQUU7UUFDaEIsTUFBTSxJQUFJLEtBQUssQ0FBQyxzQkFBc0IsQ0FBQyxTQUFTLEVBQUUsRUFBRSxDQUFDLENBQUM7S0FDekQ7SUFDRCxPQUFPLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQztBQUNqQixDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgaWFtIGZyb20gXCIuLi8uLi9hd3MtaWFtXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9hd3MtaWFtJ1xuaW1wb3J0IHsgRm4sIFRva2VuIH0gZnJvbSBcIi4uLy4uL2NvcmVcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2NvcmUnXG5pbXBvcnQgeyBDb25uZWN0aW9ucywgSUNvbm5lY3RhYmxlIH0gZnJvbSAnLi9jb25uZWN0aW9ucyc7XG5pbXBvcnQgeyBJbnN0YW5jZSB9IGZyb20gJy4vaW5zdGFuY2UnO1xuaW1wb3J0IHsgSW5zdGFuY2VUeXBlIH0gZnJvbSAnLi9pbnN0YW5jZS10eXBlcyc7XG5pbXBvcnQgeyBJTWFjaGluZUltYWdlLCBMb29rdXBNYWNoaW5lSW1hZ2UgfSBmcm9tICcuL21hY2hpbmUtaW1hZ2UnO1xuaW1wb3J0IHsgUG9ydCB9IGZyb20gJy4vcG9ydCc7XG5pbXBvcnQgeyBJU2VjdXJpdHlHcm91cCwgU2VjdXJpdHlHcm91cCB9IGZyb20gJy4vc2VjdXJpdHktZ3JvdXAnO1xuaW1wb3J0IHsgUHJpdmF0ZVN1Ym5ldCwgUHVibGljU3VibmV0LCBSb3V0ZXJUeXBlLCBWcGMgfSBmcm9tICcuL3ZwYyc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGVudW0gTmF0VHJhZmZpY0RpcmVjdGlvbiB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIE9VVEJPVU5EX09OTFkgPSAnT1VUQk9VTkRfT05MWScsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIElOQk9VTkRfQU5EX09VVEJPVU5EID0gJ0lOQk9VTkRfQU5EX09VVEJPVU5EJyxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgTk9ORSA9ICdOT05FJ1xufVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIEdhdGV3YXlDb25maWcge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IGF6OiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBnYXRld2F5SWQ6IHN0cmluZztcbn1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBhYnN0cmFjdCBjbGFzcyBOYXRQcm92aWRlciB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgZ2F0ZXdheShwcm9wczogTmF0R2F0ZXdheVByb3BzID0ge30pOiBOYXRQcm92aWRlciB7XG4gICAgICAgIHJldHVybiBuZXcgTmF0R2F0ZXdheVByb3ZpZGVyKHByb3BzKTtcbiAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgaW5zdGFuY2UocHJvcHM6IE5hdEluc3RhbmNlUHJvcHMpOiBOYXRJbnN0YW5jZVByb3ZpZGVyIHtcbiAgICAgICAgcmV0dXJuIG5ldyBOYXRJbnN0YW5jZVByb3ZpZGVyKHByb3BzKTtcbiAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IGNvbmZpZ3VyZWRHYXRld2F5czogR2F0ZXdheUNvbmZpZ1tdO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBhYnN0cmFjdCBjb25maWd1cmVOYXQob3B0aW9uczogQ29uZmlndXJlTmF0T3B0aW9ucyk6IHZvaWQ7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIGFic3RyYWN0IGNvbmZpZ3VyZVN1Ym5ldChzdWJuZXQ6IFByaXZhdGVTdWJuZXQpOiB2b2lkO1xufVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgQ29uZmlndXJlTmF0T3B0aW9ucyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IHZwYzogVnBjO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBuYXRTdWJuZXRzOiBQdWJsaWNTdWJuZXRbXTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgcHJpdmF0ZVN1Ym5ldHM6IFByaXZhdGVTdWJuZXRbXTtcbn1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBOYXRHYXRld2F5UHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgZWlwQWxsb2NhdGlvbklkcz86IHN0cmluZ1tdO1xufVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBOYXRJbnN0YW5jZVByb3BzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBtYWNoaW5lSW1hZ2U/OiBJTWFjaGluZUltYWdlO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBpbnN0YW5jZVR5cGU6IEluc3RhbmNlVHlwZTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBrZXlOYW1lPzogc3RyaW5nO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgc2VjdXJpdHlHcm91cD86IElTZWN1cml0eUdyb3VwO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgYWxsb3dBbGxUcmFmZmljPzogYm9vbGVhbjtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgZGVmYXVsdEFsbG93ZWRUcmFmZmljPzogTmF0VHJhZmZpY0RpcmVjdGlvbjtcbn1cbi8qKlxuICogUHJvdmlkZXIgZm9yIE5BVCBHYXRld2F5c1xuICovXG5jbGFzcyBOYXRHYXRld2F5UHJvdmlkZXIgZXh0ZW5kcyBOYXRQcm92aWRlciB7XG4gICAgcHJpdmF0ZSBnYXRld2F5czogUHJlZlNldDxzdHJpbmc+ID0gbmV3IFByZWZTZXQ8c3RyaW5nPigpO1xuICAgIGNvbnN0cnVjdG9yKHByaXZhdGUgcmVhZG9ubHkgcHJvcHM6IE5hdEdhdGV3YXlQcm9wcyA9IHt9KSB7XG4gICAgICAgIHN1cGVyKCk7XG4gICAgfVxuICAgIHB1YmxpYyBjb25maWd1cmVOYXQob3B0aW9uczogQ29uZmlndXJlTmF0T3B0aW9ucykge1xuICAgICAgICAvLyBDcmVhdGUgdGhlIE5BVCBnYXRld2F5c1xuICAgICAgICBsZXQgaSA9IDA7XG4gICAgICAgIGZvciAoY29uc3Qgc3ViIG9mIG9wdGlvbnMubmF0U3VibmV0cykge1xuICAgICAgICAgICAgY29uc3QgZ2F0ZXdheSA9IHN1Yi5hZGROYXRHYXRld2F5KCk7XG4gICAgICAgICAgICBpZiAodGhpcy5wcm9wcy5laXBBbGxvY2F0aW9uSWRzKSB7XG4gICAgICAgICAgICAgICAgZ2F0ZXdheS5hbGxvY2F0aW9uSWQgPSBwaWNrTihpLCB0aGlzLnByb3BzLmVpcEFsbG9jYXRpb25JZHMpO1xuICAgICAgICAgICAgfVxuICAgICAgICAgICAgdGhpcy5nYXRld2F5cy5hZGQoc3ViLmF2YWlsYWJpbGl0eVpvbmUsIGdhdGV3YXkucmVmKTtcbiAgICAgICAgICAgIGkrKztcbiAgICAgICAgfVxuICAgICAgICAvLyBBZGQgcm91dGVzIHRvIHRoZW0gaW4gdGhlIHByaXZhdGUgc3VibmV0c1xuICAgICAgICBmb3IgKGNvbnN0IHN1YiBvZiBvcHRpb25zLnByaXZhdGVTdWJuZXRzKSB7XG4gICAgICAgICAgICB0aGlzLmNvbmZpZ3VyZVN1Ym5ldChzdWIpO1xuICAgICAgICB9XG4gICAgfVxuICAgIHB1YmxpYyBjb25maWd1cmVTdWJuZXQoc3VibmV0OiBQcml2YXRlU3VibmV0KSB7XG4gICAgICAgIGNvbnN0IGF6ID0gc3VibmV0LmF2YWlsYWJpbGl0eVpvbmU7XG4gICAgICAgIGNvbnN0IGdhdGV3YXlJZCA9IHRoaXMuZ2F0ZXdheXMucGljayhheik7XG4gICAgICAgIHN1Ym5ldC5hZGRSb3V0ZSgnRGVmYXVsdFJvdXRlJywge1xuICAgICAgICAgICAgcm91dGVyVHlwZTogUm91dGVyVHlwZS5OQVRfR0FURVdBWSxcbiAgICAgICAgICAgIHJvdXRlcklkOiBnYXRld2F5SWQsXG4gICAgICAgICAgICBlbmFibGVzSW50ZXJuZXRDb25uZWN0aXZpdHk6IHRydWUsXG4gICAgICAgIH0pO1xuICAgIH1cbiAgICBwdWJsaWMgZ2V0IGNvbmZpZ3VyZWRHYXRld2F5cygpOiBHYXRld2F5Q29uZmlnW10ge1xuICAgICAgICByZXR1cm4gdGhpcy5nYXRld2F5cy52YWx1ZXMoKS5tYXAoeCA9PiAoeyBhejogeFswXSwgZ2F0ZXdheUlkOiB4WzFdIH0pKTtcbiAgICB9XG59XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBOYXRJbnN0YW5jZVByb3ZpZGVyIGV4dGVuZHMgTmF0UHJvdmlkZXIgaW1wbGVtZW50cyBJQ29ubmVjdGFibGUge1xuICAgIHByaXZhdGUgZ2F0ZXdheXM6IFByZWZTZXQ8SW5zdGFuY2U+ID0gbmV3IFByZWZTZXQ8SW5zdGFuY2U+KCk7XG4gICAgcHJpdmF0ZSBfc2VjdXJpdHlHcm91cD86IElTZWN1cml0eUdyb3VwO1xuICAgIHByaXZhdGUgX2Nvbm5lY3Rpb25zPzogQ29ubmVjdGlvbnM7XG4gICAgY29uc3RydWN0b3IocHJpdmF0ZSByZWFkb25seSBwcm9wczogTmF0SW5zdGFuY2VQcm9wcykge1xuICAgICAgICBzdXBlcigpO1xuICAgICAgICBpZiAocHJvcHMuZGVmYXVsdEFsbG93ZWRUcmFmZmljICE9PSB1bmRlZmluZWQgJiYgcHJvcHMuYWxsb3dBbGxUcmFmZmljICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignQ2FuIG5vdCBzcGVjaWZ5IGJvdGggb2YgXFwnZGVmYXVsdEFsbG93ZWRUcmFmZmljXFwnIGFuZCBcXCdkZWZhdWx0QWxsb3dlZFRyYWZmaWNcXCc7IHByZWZlciBcXCdkZWZhdWx0QWxsb3dlZFRyYWZmaWNcXCcnKTtcbiAgICAgICAgfVxuICAgIH1cbiAgICBwdWJsaWMgY29uZmlndXJlTmF0KG9wdGlvbnM6IENvbmZpZ3VyZU5hdE9wdGlvbnMpIHtcbiAgICAgICAgY29uc3QgZGVmYXVsdERpcmVjdGlvbiA9IHRoaXMucHJvcHMuZGVmYXVsdEFsbG93ZWRUcmFmZmljID8/XG4gICAgICAgICAgICAodGhpcy5wcm9wcy5hbGxvd0FsbFRyYWZmaWMgPz8gdHJ1ZSA/IE5hdFRyYWZmaWNEaXJlY3Rpb24uSU5CT1VORF9BTkRfT1VUQk9VTkQgOiBOYXRUcmFmZmljRGlyZWN0aW9uLk9VVEJPVU5EX09OTFkpO1xuICAgICAgICAvLyBDcmVhdGUgdGhlIE5BVCBpbnN0YW5jZXMuIFRoZXkgY2FuIHNoYXJlIGEgc2VjdXJpdHkgZ3JvdXAgYW5kIGEgUm9sZS5cbiAgICAgICAgY29uc3QgbWFjaGluZUltYWdlID0gdGhpcy5wcm9wcy5tYWNoaW5lSW1hZ2UgfHwgbmV3IE5hdEluc3RhbmNlSW1hZ2UoKTtcbiAgICAgICAgdGhpcy5fc2VjdXJpdHlHcm91cCA9IHRoaXMucHJvcHMuc2VjdXJpdHlHcm91cCA/PyBuZXcgU2VjdXJpdHlHcm91cChvcHRpb25zLnZwYywgJ05hdFNlY3VyaXR5R3JvdXAnLCB7XG4gICAgICAgICAgICB2cGM6IG9wdGlvbnMudnBjLFxuICAgICAgICAgICAgZGVzY3JpcHRpb246ICdTZWN1cml0eSBHcm91cCBmb3IgTkFUIGluc3RhbmNlcycsXG4gICAgICAgICAgICBhbGxvd0FsbE91dGJvdW5kOiBpc091dGJvdW5kQWxsb3dlZChkZWZhdWx0RGlyZWN0aW9uKSxcbiAgICAgICAgfSk7XG4gICAgICAgIHRoaXMuX2Nvbm5lY3Rpb25zID0gbmV3IENvbm5lY3Rpb25zKHsgc2VjdXJpdHlHcm91cHM6IFt0aGlzLl9zZWN1cml0eUdyb3VwXSB9KTtcbiAgICAgICAgaWYgKGlzSW5ib3VuZEFsbG93ZWQoZGVmYXVsdERpcmVjdGlvbikpIHtcbiAgICAgICAgICAgIHRoaXMuY29ubmVjdGlvbnMuYWxsb3dGcm9tQW55SXB2NChQb3J0LmFsbFRyYWZmaWMoKSk7XG4gICAgICAgIH1cbiAgICAgICAgLy8gRklYTUU6IElkZWFsbHksIE5BVCBpbnN0YW5jZXMgZG9uJ3QgaGF2ZSBhIHJvbGUgYXQgYWxsLCBidXRcbiAgICAgICAgLy8gJ0luc3RhbmNlJyBkb2VzIG5vdCBhbGxvdyB0aGF0IHJpZ2h0IG5vdy5cbiAgICAgICAgY29uc3Qgcm9sZSA9IG5ldyBpYW0uUm9sZShvcHRpb25zLnZwYywgJ05hdFJvbGUnLCB7XG4gICAgICAgICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnZWMyLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgICAgfSk7XG4gICAgICAgIGZvciAoY29uc3Qgc3ViIG9mIG9wdGlvbnMubmF0U3VibmV0cykge1xuICAgICAgICAgICAgY29uc3QgbmF0SW5zdGFuY2UgPSBuZXcgSW5zdGFuY2Uoc3ViLCAnTmF0SW5zdGFuY2UnLCB7XG4gICAgICAgICAgICAgICAgaW5zdGFuY2VUeXBlOiB0aGlzLnByb3BzLmluc3RhbmNlVHlwZSxcbiAgICAgICAgICAgICAgICBtYWNoaW5lSW1hZ2UsXG4gICAgICAgICAgICAgICAgc291cmNlRGVzdENoZWNrOiBmYWxzZSxcbiAgICAgICAgICAgICAgICB2cGM6IG9wdGlvbnMudnBjLFxuICAgICAgICAgICAgICAgIHZwY1N1Ym5ldHM6IHsgc3VibmV0czogW3N1Yl0gfSxcbiAgICAgICAgICAgICAgICBzZWN1cml0eUdyb3VwOiB0aGlzLl9zZWN1cml0eUdyb3VwLFxuICAgICAgICAgICAgICAgIHJvbGUsXG4gICAgICAgICAgICAgICAga2V5TmFtZTogdGhpcy5wcm9wcy5rZXlOYW1lLFxuICAgICAgICAgICAgfSk7XG4gICAgICAgICAgICAvLyBOQVQgaW5zdGFuY2Ugcm91dGVzIGFsbCB0cmFmZmljLCBib3RoIHdheXNcbiAgICAgICAgICAgIHRoaXMuZ2F0ZXdheXMuYWRkKHN1Yi5hdmFpbGFiaWxpdHlab25lLCBuYXRJbnN0YW5jZSk7XG4gICAgICAgIH1cbiAgICAgICAgLy8gQWRkIHJvdXRlcyB0byB0aGVtIGluIHRoZSBwcml2YXRlIHN1Ym5ldHNcbiAgICAgICAgZm9yIChjb25zdCBzdWIgb2Ygb3B0aW9ucy5wcml2YXRlU3VibmV0cykge1xuICAgICAgICAgICAgdGhpcy5jb25maWd1cmVTdWJuZXQoc3ViKTtcbiAgICAgICAgfVxuICAgIH1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBnZXQgc2VjdXJpdHlHcm91cCgpOiBJU2VjdXJpdHlHcm91cCB7XG4gICAgICAgIGlmICghdGhpcy5fc2VjdXJpdHlHcm91cCkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdQYXNzIHRoZSBOYXRJbnN0YW5jZVByb3ZpZGVyIHRvIGEgVnBjIGJlZm9yZSBhY2Nlc3NpbmcgXFwnc2VjdXJpdHlHcm91cFxcJycpO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiB0aGlzLl9zZWN1cml0eUdyb3VwO1xuICAgIH1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIGdldCBjb25uZWN0aW9ucygpOiBDb25uZWN0aW9ucyB7XG4gICAgICAgIGlmICghdGhpcy5fY29ubmVjdGlvbnMpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignUGFzcyB0aGUgTmF0SW5zdGFuY2VQcm92aWRlciB0byBhIFZwYyBiZWZvcmUgYWNjZXNzaW5nIFxcJ2Nvbm5lY3Rpb25zXFwnJyk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHRoaXMuX2Nvbm5lY3Rpb25zO1xuICAgIH1cbiAgICBwdWJsaWMgZ2V0IGNvbmZpZ3VyZWRHYXRld2F5cygpOiBHYXRld2F5Q29uZmlnW10ge1xuICAgICAgICByZXR1cm4gdGhpcy5nYXRld2F5cy52YWx1ZXMoKS5tYXAoeCA9PiAoeyBhejogeFswXSwgZ2F0ZXdheUlkOiB4WzFdLmluc3RhbmNlSWQgfSkpO1xuICAgIH1cbiAgICBwdWJsaWMgY29uZmlndXJlU3VibmV0KHN1Ym5ldDogUHJpdmF0ZVN1Ym5ldCkge1xuICAgICAgICBjb25zdCBheiA9IHN1Ym5ldC5hdmFpbGFiaWxpdHlab25lO1xuICAgICAgICBjb25zdCBnYXRld2F5SWQgPSB0aGlzLmdhdGV3YXlzLnBpY2soYXopLmluc3RhbmNlSWQ7XG4gICAgICAgIHN1Ym5ldC5hZGRSb3V0ZSgnRGVmYXVsdFJvdXRlJywge1xuICAgICAgICAgICAgcm91dGVyVHlwZTogUm91dGVyVHlwZS5JTlNUQU5DRSxcbiAgICAgICAgICAgIHJvdXRlcklkOiBnYXRld2F5SWQsXG4gICAgICAgICAgICBlbmFibGVzSW50ZXJuZXRDb25uZWN0aXZpdHk6IHRydWUsXG4gICAgICAgIH0pO1xuICAgIH1cbn1cbi8qKlxuICogUHJlZmVyZW50aWFsIHNldFxuICpcbiAqIFBpY2tzIHRoZSB2YWx1ZSB3aXRoIHRoZSBnaXZlbiBrZXkgaWYgYXZhaWxhYmxlLCBvdGhlcndpc2UgZGlzdHJpYnV0ZXNcbiAqIGV2ZW5seSBhbW9uZyB0aGUgYXZhaWxhYmxlIG9wdGlvbnMuXG4gKi9cbmNsYXNzIFByZWZTZXQ8QT4ge1xuICAgIHByaXZhdGUgcmVhZG9ubHkgbWFwOiBSZWNvcmQ8c3RyaW5nLCBBPiA9IHt9O1xuICAgIHByaXZhdGUgcmVhZG9ubHkgdmFscyA9IG5ldyBBcnJheTxbc3RyaW5nLCBBXT4oKTtcbiAgICBwcml2YXRlIG5leHQ6IG51bWJlciA9IDA7XG4gICAgcHVibGljIGFkZChwcmVmOiBzdHJpbmcsIHZhbHVlOiBBKSB7XG4gICAgICAgIHRoaXMubWFwW3ByZWZdID0gdmFsdWU7XG4gICAgICAgIHRoaXMudmFscy5wdXNoKFtwcmVmLCB2YWx1ZV0pO1xuICAgIH1cbiAgICBwdWJsaWMgcGljayhwcmVmOiBzdHJpbmcpOiBBIHtcbiAgICAgICAgaWYgKHRoaXMudmFscy5sZW5ndGggPT09IDApIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignQ2Fubm90IHBpY2ssIHNldCBpcyBlbXB0eScpO1xuICAgICAgICB9XG4gICAgICAgIGlmIChwcmVmIGluIHRoaXMubWFwKSB7XG4gICAgICAgICAgICByZXR1cm4gdGhpcy5tYXBbcHJlZl07XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHRoaXMudmFsc1t0aGlzLm5leHQrKyAlIHRoaXMudmFscy5sZW5ndGhdWzFdO1xuICAgIH1cbiAgICBwdWJsaWMgdmFsdWVzKCk6IEFycmF5PFtzdHJpbmcsIEFdPiB7XG4gICAgICAgIHJldHVybiB0aGlzLnZhbHM7XG4gICAgfVxufVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIE5hdEluc3RhbmNlSW1hZ2UgZXh0ZW5kcyBMb29rdXBNYWNoaW5lSW1hZ2Uge1xuICAgIGNvbnN0cnVjdG9yKCkge1xuICAgICAgICBzdXBlcih7XG4gICAgICAgICAgICBuYW1lOiAnYW16bi1hbWktdnBjLW5hdC0qJyxcbiAgICAgICAgICAgIG93bmVyczogWydhbWF6b24nXSxcbiAgICAgICAgfSk7XG4gICAgfVxufVxuZnVuY3Rpb24gaXNPdXRib3VuZEFsbG93ZWQoZGlyZWN0aW9uOiBOYXRUcmFmZmljRGlyZWN0aW9uKSB7XG4gICAgcmV0dXJuIGRpcmVjdGlvbiA9PT0gTmF0VHJhZmZpY0RpcmVjdGlvbi5JTkJPVU5EX0FORF9PVVRCT1VORCB8fFxuICAgICAgICBkaXJlY3Rpb24gPT09IE5hdFRyYWZmaWNEaXJlY3Rpb24uT1VUQk9VTkRfT05MWTtcbn1cbmZ1bmN0aW9uIGlzSW5ib3VuZEFsbG93ZWQoZGlyZWN0aW9uOiBOYXRUcmFmZmljRGlyZWN0aW9uKSB7XG4gICAgcmV0dXJuIGRpcmVjdGlvbiA9PT0gTmF0VHJhZmZpY0RpcmVjdGlvbi5JTkJPVU5EX0FORF9PVVRCT1VORDtcbn1cbi8qKlxuICogVG9rZW4tYXdhcmUgcGljayBpbmRleCBmdW5jdGlvblxuICovXG5mdW5jdGlvbiBwaWNrTihpOiBudW1iZXIsIHhzOiBzdHJpbmdbXSkge1xuICAgIGlmIChUb2tlbi5pc1VucmVzb2x2ZWQoeHMpKSB7XG4gICAgICAgIHJldHVybiBGbi5zZWxlY3QoaSwgeHMpO1xuICAgIH1cbiAgICBpZiAoaSA+PSB4cy5sZW5ndGgpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBDYW5ub3QgZ2V0IGVsZW1lbnQgJHtpfSBmcm9tICR7eHN9YCk7XG4gICAgfVxuICAgIHJldHVybiB4c1tpXTtcbn1cbiJdfQ==