"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Connections = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const peer_1 = require("./peer");
/**
 * (experimental) Manage the allowed network connections for constructs with Security Groups.
 *
 * Security Groups can be thought of as a firewall for network-connected
 * devices. This class makes it easy to allow network connections to and
 * from security groups, and between security groups individually. When
 * establishing connectivity between security groups, it will automatically
 * add rules in both security groups
 *
 * This object can manage one or more security groups.
 *
 * @experimental
 */
class Connections {
    /**
     * @experimental
     */
    constructor(props = {}) {
        /**
         * Underlying securityGroup for this Connections object, if present
         *
         * May be empty if this Connections object is not managing a SecurityGroup,
         * but simply representing a Connectable peer.
         */
        this._securityGroups = new ReactiveList();
        /**
         * The rule that defines how to represent this peer in a security group
         */
        this._securityGroupRules = new ReactiveList();
        /**
         * When doing bidirectional grants between Connections, make sure we don't recursive infinitely
         */
        this.skip = false;
        /**
         * When doing bidirectional grants between Security Groups in different stacks, put the rule on the other SG
         */
        this.remoteRule = false;
        this.connections = this;
        this._securityGroups.push(...(props.securityGroups || []));
        this._securityGroupRules.push(...this._securityGroups.asArray());
        if (props.peer) {
            this._securityGroupRules.push(props.peer);
        }
        this.defaultPort = props.defaultPort;
    }
    /**
     * @experimental
     */
    get securityGroups() {
        return this._securityGroups.asArray();
    }
    /**
     * (experimental) Add a security group to the list of security groups managed by this object.
     *
     * @experimental
     */
    addSecurityGroup(...securityGroups) {
        for (const securityGroup of securityGroups) {
            this._securityGroups.push(securityGroup);
            this._securityGroupRules.push(securityGroup);
        }
    }
    /**
     * (experimental) Allow connections to the peer on the given port.
     *
     * @experimental
     */
    allowTo(other, portRange, description) {
        if (this.skip) {
            return;
        }
        const remoteRule = this.remoteRule; // Capture current value into local for callback to close over
        this._securityGroups.forEachAndForever(securityGroup => {
            other.connections._securityGroupRules.forEachAndForever(rule => {
                securityGroup.addEgressRule(rule, portRange, description, remoteRule);
            });
        });
        this.skip = true;
        other.connections.remoteRule = true;
        try {
            other.connections.allowFrom(this, portRange, description);
        }
        finally {
            this.skip = false;
            other.connections.remoteRule = false;
        }
    }
    /**
     * (experimental) Allow connections from the peer on the given port.
     *
     * @experimental
     */
    allowFrom(other, portRange, description) {
        if (this.skip) {
            return;
        }
        const remoteRule = this.remoteRule; // Capture current value into local for callback to close over
        this._securityGroups.forEachAndForever(securityGroup => {
            other.connections._securityGroupRules.forEachAndForever(rule => {
                securityGroup.addIngressRule(rule, portRange, description, remoteRule);
            });
        });
        this.skip = true;
        other.connections.remoteRule = true;
        try {
            other.connections.allowTo(this, portRange, description);
        }
        finally {
            this.skip = false;
            other.connections.remoteRule = false;
        }
    }
    /**
     * (experimental) Allow hosts inside the security group to connect to each other on the given port.
     *
     * @experimental
     */
    allowInternally(portRange, description) {
        this._securityGroups.forEachAndForever(securityGroup => {
            this._securityGroupRules.forEachAndForever(rule => {
                securityGroup.addIngressRule(rule, portRange, description);
                securityGroup.addEgressRule(rule, portRange, description);
            });
        });
    }
    /**
     * (experimental) Allow to all IPv4 ranges.
     *
     * @experimental
     */
    allowToAnyIpv4(portRange, description) {
        this.allowTo(peer_1.Peer.anyIpv4(), portRange, description);
    }
    /**
     * (experimental) Allow from any IPv4 ranges.
     *
     * @experimental
     */
    allowFromAnyIpv4(portRange, description) {
        this.allowFrom(peer_1.Peer.anyIpv4(), portRange, description);
    }
    /**
     * (experimental) Allow connections from the peer on our default port.
     *
     * Even if the peer has a default port, we will always use our default port.
     *
     * @experimental
     */
    allowDefaultPortFrom(other, description) {
        if (!this.defaultPort) {
            throw new Error('Cannot call allowDefaultPortFrom(): this resource has no default port');
        }
        this.allowFrom(other, this.defaultPort, description);
    }
    /**
     * (experimental) Allow hosts inside the security group to connect to each other.
     *
     * @experimental
     */
    allowDefaultPortInternally(description) {
        if (!this.defaultPort) {
            throw new Error('Cannot call allowDefaultPortInternally(): this resource has no default port');
        }
        this.allowInternally(this.defaultPort, description);
    }
    /**
     * (experimental) Allow default connections from all IPv4 ranges.
     *
     * @experimental
     */
    allowDefaultPortFromAnyIpv4(description) {
        if (!this.defaultPort) {
            throw new Error('Cannot call allowDefaultPortFromAnyIpv4(): this resource has no default port');
        }
        this.allowFromAnyIpv4(this.defaultPort, description);
    }
    /**
     * (experimental) Allow connections to the security group on their default port.
     *
     * @experimental
     */
    allowToDefaultPort(other, description) {
        if (other.connections.defaultPort === undefined) {
            throw new Error('Cannot call allowToDefaultPort(): other resource has no default port');
        }
        this.allowTo(other, other.connections.defaultPort, description);
    }
    /**
     * (experimental) Allow connections from the peer on our default port.
     *
     * Even if the peer has a default port, we will always use our default port.
     *
     * @experimental
     */
    allowDefaultPortTo(other, description) {
        if (!this.defaultPort) {
            throw new Error('Cannot call allowDefaultPortTo(): this resource has no default port');
        }
        this.allowTo(other, this.defaultPort, description);
    }
}
exports.Connections = Connections;
_a = JSII_RTTI_SYMBOL_1;
Connections[_a] = { fqn: "monocdk.aws_ec2.Connections", version: "1.106.1" };
class ReactiveList {
    constructor() {
        this.elements = new Array();
        this.listeners = new Array();
    }
    push(...xs) {
        this.elements.push(...xs);
        for (const listener of this.listeners) {
            for (const x of xs) {
                listener(x);
            }
        }
    }
    forEachAndForever(listener) {
        for (const element of this.elements) {
            listener(element);
        }
        this.listeners.push(listener);
    }
    asArray() {
        return this.elements.slice();
    }
    get length() {
        return this.elements.length;
    }
}
//# sourceMappingURL=data:application/json;base64,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