"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.InitConfig = exports.CloudFormationInit = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const machine_image_1 = require("./machine-image");
const cfn_init_internal_1 = require("./private/cfn-init-internal");
/**
 * (experimental) A CloudFormation-init configuration.
 *
 * @experimental
 */
class CloudFormationInit {
    constructor(configSets, configs) {
        this._configSets = {};
        this._configs = {};
        Object.assign(this._configSets, configSets);
        Object.assign(this._configs, configs);
    }
    /**
     * (experimental) Build a new config from a set of Init Elements.
     *
     * @experimental
     */
    static fromElements(...elements) {
        return CloudFormationInit.fromConfig(new InitConfig(elements));
    }
    /**
     * (experimental) Use an existing InitConfig object as the default and only config.
     *
     * @experimental
     */
    static fromConfig(config) {
        return CloudFormationInit.fromConfigSets({
            configSets: {
                default: ['config'],
            },
            configs: { config },
        });
    }
    /**
     * (experimental) Build a CloudFormationInit from config sets.
     *
     * @experimental
     */
    static fromConfigSets(props) {
        return new CloudFormationInit(props.configSets, props.configs);
    }
    /**
     * (experimental) Add a config with the given name to this CloudFormationInit object.
     *
     * @experimental
     */
    addConfig(configName, config) {
        if (this._configs[configName]) {
            throw new Error(`CloudFormationInit already contains a config named '${configName}'`);
        }
        this._configs[configName] = config;
    }
    /**
     * (experimental) Add a config set with the given name to this CloudFormationInit object.
     *
     * The new configset will reference the given configs in the given order.
     *
     * @experimental
     */
    addConfigSet(configSetName, configNames = []) {
        if (this._configSets[configSetName]) {
            throw new Error(`CloudFormationInit already contains a configSet named '${configSetName}'`);
        }
        const unk = configNames.filter(c => !this._configs[c]);
        if (unk.length > 0) {
            throw new Error(`Unknown configs referenced in definition of '${configSetName}': ${unk}`);
        }
        this._configSets[configSetName] = [...configNames];
    }
    /**
     * (experimental) Attach the CloudFormation Init config to the given resource.
     *
     * As an app builder, use `instance.applyCloudFormationInit()` or
     * `autoScalingGroup.applyCloudFormationInit()` to trigger this method.
     *
     * This method does the following:
     *
     * - Renders the `AWS::CloudFormation::Init` object to the given resource's
     *    metadata, potentially adding a `AWS::CloudFormation::Authentication` object
     *    next to it if required.
     * - Updates the instance role policy to be able to call the APIs required for
     *    `cfn-init` and `cfn-signal` to work, and potentially add permissions to download
     *    referenced asset and bucket resources.
     * - Updates the given UserData with commands to execute the `cfn-init` script.
     *
     * @experimental
     */
    attach(attachedResource, attachOptions) {
        var _c, _d, _e;
        if (attachOptions.platform === machine_image_1.OperatingSystemType.UNKNOWN) {
            throw new Error('Cannot attach CloudFormationInit to an unknown OS type');
        }
        const CFN_INIT_METADATA_KEY = 'AWS::CloudFormation::Init';
        if (attachedResource.getMetadata(CFN_INIT_METADATA_KEY) !== undefined) {
            throw new Error(`Cannot bind CfnInit: resource '${attachedResource.node.path}' already has '${CFN_INIT_METADATA_KEY}' attached`);
        }
        // Note: This will not reflect mutations made after attaching.
        const bindResult = this.bind(attachedResource.stack, attachOptions);
        attachedResource.addMetadata(CFN_INIT_METADATA_KEY, bindResult.configData);
        // Need to resolve the various tokens from assets in the config,
        // as well as include any asset hashes provided so the fingerprint is accurate.
        const resolvedConfig = attachedResource.stack.resolve(bindResult.configData);
        const fingerprintInput = { config: resolvedConfig, assetHash: bindResult.assetHash };
        const fingerprint = contentHash(JSON.stringify(fingerprintInput)).substr(0, 16);
        attachOptions.instanceRole.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['cloudformation:DescribeStackResource', 'cloudformation:SignalResource'],
            resources: [core_1.Aws.STACK_ID],
        }));
        if (bindResult.authData) {
            attachedResource.addMetadata('AWS::CloudFormation::Authentication', bindResult.authData);
        }
        // To identify the resources that have the metadata and where the signal
        // needs to be sent, we need { region, stackName, logicalId }
        const resourceLocator = `--region ${core_1.Aws.REGION} --stack ${core_1.Aws.STACK_NAME} --resource ${attachedResource.logicalId}`;
        const configSets = ((_c = attachOptions.configSets) !== null && _c !== void 0 ? _c : ['default']).join(',');
        const printLog = (_d = attachOptions.printLog) !== null && _d !== void 0 ? _d : true;
        if ((_e = attachOptions.embedFingerprint) !== null && _e !== void 0 ? _e : true) {
            // It just so happens that the comment char is '#' for both bash and PowerShell
            attachOptions.userData.addCommands(`# fingerprint: ${fingerprint}`);
        }
        if (attachOptions.platform === machine_image_1.OperatingSystemType.WINDOWS) {
            const errCode = attachOptions.ignoreFailures ? '0' : '$LASTEXITCODE';
            attachOptions.userData.addCommands(...[
                `cfn-init.exe -v ${resourceLocator} -c ${configSets}`,
                `cfn-signal.exe -e ${errCode} ${resourceLocator}`,
                ...printLog ? ['type C:\\cfn\\log\\cfn-init.log'] : [],
            ]);
        }
        else {
            const errCode = attachOptions.ignoreFailures ? '0' : '$?';
            attachOptions.userData.addCommands(...[
                // Run a subshell without 'errexit', so we can signal using the exit code of cfn-init
                '(',
                '  set +e',
                `  /opt/aws/bin/cfn-init -v ${resourceLocator} -c ${configSets}`,
                `  /opt/aws/bin/cfn-signal -e ${errCode} ${resourceLocator}`,
                ...printLog ? ['  cat /var/log/cfn-init.log >&2'] : [],
                ')',
            ]);
        }
    }
    bind(scope, options) {
        const nonEmptyConfigs = mapValues(this._configs, c => c.isEmpty() ? undefined : c);
        const configNameToBindResult = mapValues(nonEmptyConfigs, c => c._bind(scope, options));
        return {
            configData: {
                configSets: mapValues(this._configSets, configNames => configNames.filter(name => nonEmptyConfigs[name] !== undefined)),
                ...mapValues(configNameToBindResult, c => c.config),
            },
            authData: Object.values(configNameToBindResult).map(c => c.authentication).reduce(deepMerge, undefined),
            assetHash: combineAssetHashesOrUndefined(Object.values(configNameToBindResult).map(c => c.assetHash)),
        };
    }
}
exports.CloudFormationInit = CloudFormationInit;
_a = JSII_RTTI_SYMBOL_1;
CloudFormationInit[_a] = { fqn: "monocdk.aws_ec2.CloudFormationInit", version: "1.106.1" };
/**
 * (experimental) A collection of configuration elements.
 *
 * @experimental
 */
class InitConfig {
    /**
     * @experimental
     */
    constructor(elements) {
        this.elements = new Array();
        this.add(...elements);
    }
    /**
     * (experimental) Whether this configset has elements or not.
     *
     * @experimental
     */
    isEmpty() {
        return this.elements.length === 0;
    }
    /**
     * (experimental) Add one or more elements to the config.
     *
     * @experimental
     */
    add(...elements) {
        this.elements.push(...elements);
    }
    /**
     * Called when the config is applied to an instance.
     * Creates the CloudFormation representation of the Init config and handles any permissions and assets.
     * @internal
     */
    _bind(scope, options) {
        const bindOptions = {
            instanceRole: options.instanceRole,
            platform: this.initPlatformFromOSType(options.platform),
            scope,
        };
        const packageConfig = this.bindForType(cfn_init_internal_1.InitElementType.PACKAGE, bindOptions);
        const groupsConfig = this.bindForType(cfn_init_internal_1.InitElementType.GROUP, bindOptions);
        const usersConfig = this.bindForType(cfn_init_internal_1.InitElementType.USER, bindOptions);
        const sourcesConfig = this.bindForType(cfn_init_internal_1.InitElementType.SOURCE, bindOptions);
        const filesConfig = this.bindForType(cfn_init_internal_1.InitElementType.FILE, bindOptions);
        const commandsConfig = this.bindForType(cfn_init_internal_1.InitElementType.COMMAND, bindOptions);
        // Must be last!
        const servicesConfig = this.bindForType(cfn_init_internal_1.InitElementType.SERVICE, bindOptions);
        const allConfig = [packageConfig, groupsConfig, usersConfig, sourcesConfig, filesConfig, commandsConfig, servicesConfig];
        const authentication = allConfig.map(c => c === null || c === void 0 ? void 0 : c.authentication).reduce(deepMerge, undefined);
        const assetHash = combineAssetHashesOrUndefined(allConfig.map(c => c === null || c === void 0 ? void 0 : c.assetHash));
        return {
            config: {
                packages: packageConfig === null || packageConfig === void 0 ? void 0 : packageConfig.config,
                groups: groupsConfig === null || groupsConfig === void 0 ? void 0 : groupsConfig.config,
                users: usersConfig === null || usersConfig === void 0 ? void 0 : usersConfig.config,
                sources: sourcesConfig === null || sourcesConfig === void 0 ? void 0 : sourcesConfig.config,
                files: filesConfig === null || filesConfig === void 0 ? void 0 : filesConfig.config,
                commands: commandsConfig === null || commandsConfig === void 0 ? void 0 : commandsConfig.config,
                services: servicesConfig === null || servicesConfig === void 0 ? void 0 : servicesConfig.config,
            },
            authentication,
            assetHash,
        };
    }
    bindForType(elementType, renderOptions) {
        var _c;
        const elements = this.elements.filter(elem => elem.elementType === elementType);
        if (elements.length === 0) {
            return undefined;
        }
        const bindResults = elements.map((e, index) => e._bind({ index, ...renderOptions }));
        return {
            config: (_c = bindResults.map(r => r.config).reduce(deepMerge, undefined)) !== null && _c !== void 0 ? _c : {},
            authentication: bindResults.map(r => r.authentication).reduce(deepMerge, undefined),
            assetHash: combineAssetHashesOrUndefined(bindResults.map(r => r.assetHash)),
        };
    }
    initPlatformFromOSType(osType) {
        switch (osType) {
            case machine_image_1.OperatingSystemType.LINUX: {
                return cfn_init_internal_1.InitPlatform.LINUX;
            }
            case machine_image_1.OperatingSystemType.WINDOWS: {
                return cfn_init_internal_1.InitPlatform.WINDOWS;
            }
            default: {
                throw new Error('Cannot attach CloudFormationInit to an unknown OS type');
            }
        }
    }
}
exports.InitConfig = InitConfig;
_b = JSII_RTTI_SYMBOL_1;
InitConfig[_b] = { fqn: "monocdk.aws_ec2.InitConfig", version: "1.106.1" };
/**
 * Deep-merge objects and arrays
 *
 * Treat arrays as sets, removing duplicates. This is acceptable for rendering
 * cfn-inits, not applicable elsewhere.
 */
function deepMerge(target, src) {
    var _c, _d;
    if (target == null) {
        return src;
    }
    if (src == null) {
        return target;
    }
    for (const [key, value] of Object.entries(src)) {
        if (Array.isArray(value)) {
            if (target[key] && !Array.isArray(target[key])) {
                throw new Error(`Trying to merge array [${value}] into a non-array '${target[key]}'`);
            }
            target[key] = Array.from(new Set([
                ...(_c = target[key]) !== null && _c !== void 0 ? _c : [],
                ...value,
            ]));
            continue;
        }
        if (typeof value === 'object' && value) {
            target[key] = deepMerge((_d = target[key]) !== null && _d !== void 0 ? _d : {}, value);
            continue;
        }
        if (value !== undefined) {
            target[key] = value;
        }
    }
    return target;
}
/**
 * Map a function over values of an object
 *
 * If the mapping function returns undefined, remove the key
 */
function mapValues(xs, fn) {
    const ret = {};
    for (const [k, v] of Object.entries(xs)) {
        const mapped = fn(v);
        if (mapped !== undefined) {
            ret[k] = mapped;
        }
    }
    return ret;
}
// Combines all input asset hashes into one, or if no hashes are present, returns undefined.
function combineAssetHashesOrUndefined(hashes) {
    const hashArray = hashes.filter((x) => x !== undefined);
    return hashArray.length > 0 ? hashArray.join('') : undefined;
}
function contentHash(content) {
    return crypto.createHash('sha256').update(content).digest('hex');
}
//# sourceMappingURL=data:application/json;base64,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