"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ScalableTableAttribute = void 0;
const appscaling = require("../../aws-applicationautoscaling"); // Automatically re-written from '@aws-cdk/aws-applicationautoscaling'
/**
 * A scalable table attribute
 */
class ScalableTableAttribute extends appscaling.BaseScalableAttribute {
    constructor() {
        super(...arguments);
        this.scalingPolicyCreated = false;
    }
    /**
     * Scale out or in based on time
     */
    scaleOnSchedule(id, action) {
        this.scalingPolicyCreated = true;
        super.doScaleOnSchedule(id, action);
    }
    /**
     * Scale out or in to keep utilization at a given level
     */
    scaleOnUtilization(props) {
        if (props.targetUtilizationPercent < 10 || props.targetUtilizationPercent > 90) {
            // eslint-disable-next-line max-len
            throw new RangeError(`targetUtilizationPercent for DynamoDB scaling must be between 10 and 90 percent, got: ${props.targetUtilizationPercent}`);
        }
        this.scalingPolicyCreated = true;
        const predefinedMetric = this.props.dimension.indexOf('ReadCapacity') === -1
            ? appscaling.PredefinedMetric.DYANMODB_WRITE_CAPACITY_UTILIZATION
            : appscaling.PredefinedMetric.DYNAMODB_READ_CAPACITY_UTILIZATION;
        super.doScaleToTrackMetric('Tracking', {
            policyName: props.policyName,
            disableScaleIn: props.disableScaleIn,
            scaleInCooldown: props.scaleInCooldown,
            scaleOutCooldown: props.scaleOutCooldown,
            targetValue: props.targetUtilizationPercent,
            predefinedMetric,
        });
    }
    /** @internal */
    get _scalingPolicyCreated() {
        return this.scalingPolicyCreated;
    }
}
exports.ScalableTableAttribute = ScalableTableAttribute;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2NhbGFibGUtdGFibGUtYXR0cmlidXRlLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsic2NhbGFibGUtdGFibGUtYXR0cmlidXRlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLCtEQUErRCxDQUFDLHNFQUFzRTtBQUV0STs7R0FFRztBQUNILE1BQWEsc0JBQXVCLFNBQVEsVUFBVSxDQUFDLHFCQUFxQjtJQUE1RTs7UUFDWSx5QkFBb0IsR0FBRyxLQUFLLENBQUM7SUFpQ3pDLENBQUM7SUFoQ0c7O09BRUc7SUFDSSxlQUFlLENBQUMsRUFBVSxFQUFFLE1BQWtDO1FBQ2pFLElBQUksQ0FBQyxvQkFBb0IsR0FBRyxJQUFJLENBQUM7UUFDakMsS0FBSyxDQUFDLGlCQUFpQixDQUFDLEVBQUUsRUFBRSxNQUFNLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBQ0Q7O09BRUc7SUFDSSxrQkFBa0IsQ0FBQyxLQUE4QjtRQUNwRCxJQUFJLEtBQUssQ0FBQyx3QkFBd0IsR0FBRyxFQUFFLElBQUksS0FBSyxDQUFDLHdCQUF3QixHQUFHLEVBQUUsRUFBRTtZQUM1RSxtQ0FBbUM7WUFDbkMsTUFBTSxJQUFJLFVBQVUsQ0FBQyx5RkFBeUYsS0FBSyxDQUFDLHdCQUF3QixFQUFFLENBQUMsQ0FBQztTQUNuSjtRQUNELElBQUksQ0FBQyxvQkFBb0IsR0FBRyxJQUFJLENBQUM7UUFDakMsTUFBTSxnQkFBZ0IsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsY0FBYyxDQUFDLEtBQUssQ0FBQyxDQUFDO1lBQ3hFLENBQUMsQ0FBQyxVQUFVLENBQUMsZ0JBQWdCLENBQUMsbUNBQW1DO1lBQ2pFLENBQUMsQ0FBQyxVQUFVLENBQUMsZ0JBQWdCLENBQUMsa0NBQWtDLENBQUM7UUFDckUsS0FBSyxDQUFDLG9CQUFvQixDQUFDLFVBQVUsRUFBRTtZQUNuQyxVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVU7WUFDNUIsY0FBYyxFQUFFLEtBQUssQ0FBQyxjQUFjO1lBQ3BDLGVBQWUsRUFBRSxLQUFLLENBQUMsZUFBZTtZQUN0QyxnQkFBZ0IsRUFBRSxLQUFLLENBQUMsZ0JBQWdCO1lBQ3hDLFdBQVcsRUFBRSxLQUFLLENBQUMsd0JBQXdCO1lBQzNDLGdCQUFnQjtTQUNuQixDQUFDLENBQUM7SUFDUCxDQUFDO0lBQ0QsZ0JBQWdCO0lBQ2hCLElBQVcscUJBQXFCO1FBQzVCLE9BQU8sSUFBSSxDQUFDLG9CQUFvQixDQUFDO0lBQ3JDLENBQUM7Q0FDSjtBQWxDRCx3REFrQ0MiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBhcHBzY2FsaW5nIGZyb20gXCIuLi8uLi9hd3MtYXBwbGljYXRpb25hdXRvc2NhbGluZ1wiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWFwcGxpY2F0aW9uYXV0b3NjYWxpbmcnXG5pbXBvcnQgeyBVdGlsaXphdGlvblNjYWxpbmdQcm9wcyB9IGZyb20gJy4vc2NhbGFibGUtYXR0cmlidXRlLWFwaSc7XG4vKipcbiAqIEEgc2NhbGFibGUgdGFibGUgYXR0cmlidXRlXG4gKi9cbmV4cG9ydCBjbGFzcyBTY2FsYWJsZVRhYmxlQXR0cmlidXRlIGV4dGVuZHMgYXBwc2NhbGluZy5CYXNlU2NhbGFibGVBdHRyaWJ1dGUge1xuICAgIHByaXZhdGUgc2NhbGluZ1BvbGljeUNyZWF0ZWQgPSBmYWxzZTtcbiAgICAvKipcbiAgICAgKiBTY2FsZSBvdXQgb3IgaW4gYmFzZWQgb24gdGltZVxuICAgICAqL1xuICAgIHB1YmxpYyBzY2FsZU9uU2NoZWR1bGUoaWQ6IHN0cmluZywgYWN0aW9uOiBhcHBzY2FsaW5nLlNjYWxpbmdTY2hlZHVsZSkge1xuICAgICAgICB0aGlzLnNjYWxpbmdQb2xpY3lDcmVhdGVkID0gdHJ1ZTtcbiAgICAgICAgc3VwZXIuZG9TY2FsZU9uU2NoZWR1bGUoaWQsIGFjdGlvbik7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFNjYWxlIG91dCBvciBpbiB0byBrZWVwIHV0aWxpemF0aW9uIGF0IGEgZ2l2ZW4gbGV2ZWxcbiAgICAgKi9cbiAgICBwdWJsaWMgc2NhbGVPblV0aWxpemF0aW9uKHByb3BzOiBVdGlsaXphdGlvblNjYWxpbmdQcm9wcykge1xuICAgICAgICBpZiAocHJvcHMudGFyZ2V0VXRpbGl6YXRpb25QZXJjZW50IDwgMTAgfHwgcHJvcHMudGFyZ2V0VXRpbGl6YXRpb25QZXJjZW50ID4gOTApIHtcbiAgICAgICAgICAgIC8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBtYXgtbGVuXG4gICAgICAgICAgICB0aHJvdyBuZXcgUmFuZ2VFcnJvcihgdGFyZ2V0VXRpbGl6YXRpb25QZXJjZW50IGZvciBEeW5hbW9EQiBzY2FsaW5nIG11c3QgYmUgYmV0d2VlbiAxMCBhbmQgOTAgcGVyY2VudCwgZ290OiAke3Byb3BzLnRhcmdldFV0aWxpemF0aW9uUGVyY2VudH1gKTtcbiAgICAgICAgfVxuICAgICAgICB0aGlzLnNjYWxpbmdQb2xpY3lDcmVhdGVkID0gdHJ1ZTtcbiAgICAgICAgY29uc3QgcHJlZGVmaW5lZE1ldHJpYyA9IHRoaXMucHJvcHMuZGltZW5zaW9uLmluZGV4T2YoJ1JlYWRDYXBhY2l0eScpID09PSAtMVxuICAgICAgICAgICAgPyBhcHBzY2FsaW5nLlByZWRlZmluZWRNZXRyaWMuRFlBTk1PREJfV1JJVEVfQ0FQQUNJVFlfVVRJTElaQVRJT05cbiAgICAgICAgICAgIDogYXBwc2NhbGluZy5QcmVkZWZpbmVkTWV0cmljLkRZTkFNT0RCX1JFQURfQ0FQQUNJVFlfVVRJTElaQVRJT047XG4gICAgICAgIHN1cGVyLmRvU2NhbGVUb1RyYWNrTWV0cmljKCdUcmFja2luZycsIHtcbiAgICAgICAgICAgIHBvbGljeU5hbWU6IHByb3BzLnBvbGljeU5hbWUsXG4gICAgICAgICAgICBkaXNhYmxlU2NhbGVJbjogcHJvcHMuZGlzYWJsZVNjYWxlSW4sXG4gICAgICAgICAgICBzY2FsZUluQ29vbGRvd246IHByb3BzLnNjYWxlSW5Db29sZG93bixcbiAgICAgICAgICAgIHNjYWxlT3V0Q29vbGRvd246IHByb3BzLnNjYWxlT3V0Q29vbGRvd24sXG4gICAgICAgICAgICB0YXJnZXRWYWx1ZTogcHJvcHMudGFyZ2V0VXRpbGl6YXRpb25QZXJjZW50LFxuICAgICAgICAgICAgcHJlZGVmaW5lZE1ldHJpYyxcbiAgICAgICAgfSk7XG4gICAgfVxuICAgIC8qKiBAaW50ZXJuYWwgKi9cbiAgICBwdWJsaWMgZ2V0IF9zY2FsaW5nUG9saWN5Q3JlYXRlZCgpOiBib29sZWFuIHtcbiAgICAgICAgcmV0dXJuIHRoaXMuc2NhbGluZ1BvbGljeUNyZWF0ZWQ7XG4gICAgfVxufVxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBlbmFibGluZyBEeW5hbW9EQiBjYXBhY2l0eSBzY2FsaW5nXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRW5hYmxlU2NhbGluZ1Byb3BzIHtcbiAgICAvKipcbiAgICAgKiBNaW5pbXVtIGNhcGFjaXR5IHRvIHNjYWxlIHRvXG4gICAgICovXG4gICAgbWluQ2FwYWNpdHk6IG51bWJlcjtcbiAgICAvKipcbiAgICAgKiBNYXhpbXVtIGNhcGFjaXR5IHRvIHNjYWxlIHRvXG4gICAgICovXG4gICAgbWF4Q2FwYWNpdHk6IG51bWJlcjtcbn1cbiJdfQ==