"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ReplicaProvider = void 0;
const path = require("path");
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cr = require("../../custom-resources"); // Automatically re-written from '@aws-cdk/custom-resources'
class ReplicaProvider extends core_1.NestedStack {
    constructor(scope, id, props = {}) {
        super(scope, id);
        const code = lambda.Code.fromAsset(path.join(__dirname, 'replica-handler'));
        // Issues UpdateTable API calls
        this.onEventHandler = new lambda.Function(this, 'OnEventHandler', {
            code,
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.onEventHandler',
            timeout: core_1.Duration.minutes(5),
        });
        // Checks if table is back to `ACTIVE` state
        this.isCompleteHandler = new lambda.Function(this, 'IsCompleteHandler', {
            code,
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.isCompleteHandler',
            timeout: core_1.Duration.seconds(30),
        });
        // Allows the creation of the `AWSServiceRoleForDynamoDBReplication` service linked role
        this.onEventHandler.addToRolePolicy(new iam.PolicyStatement({
            actions: ['iam:CreateServiceLinkedRole'],
            resources: [core_1.Stack.of(this).formatArn({
                    service: 'iam',
                    region: '',
                    resource: 'role',
                    resourceName: 'aws-service-role/replication.dynamodb.amazonaws.com/AWSServiceRoleForDynamoDBReplication',
                })],
        }));
        // Required for replica table creation
        this.onEventHandler.addToRolePolicy(new iam.PolicyStatement({
            actions: ['dynamodb:DescribeLimits'],
            resources: ['*'],
        }));
        this.provider = new cr.Provider(this, 'Provider', {
            onEventHandler: this.onEventHandler,
            isCompleteHandler: this.isCompleteHandler,
            queryInterval: core_1.Duration.seconds(10),
            totalTimeout: props.timeout,
        });
    }
    /**
     * Creates a stack-singleton resource provider nested stack.
     */
    static getOrCreate(scope, props = {}) {
        var _a;
        const stack = core_1.Stack.of(scope);
        this.checkManagedPoliciesLimit(stack);
        const uid = '@aws-cdk/aws-dynamodb.ReplicaProvider';
        return (_a = stack.node.tryFindChild(uid)) !== null && _a !== void 0 ? _a : new ReplicaProvider(stack, uid, props);
    }
    static checkManagedPoliciesLimit(stack) {
        var _a;
        // The custom resource implementation uses IAM managed policies. There's
        // a limit of 10 managed policies per role in IAM. Throw if we reach this
        // limit.
        const calls = (_a = this.getOrCreateCalls.get(stack.stackName)) !== null && _a !== void 0 ? _a : 0;
        if (calls >= 10) {
            throw new Error('You currently cannot have more than 10 global tables in a single stack. Consider splitting your tables across multiple stacks.');
        }
        this.getOrCreateCalls.set(stack.stackName, calls + 1);
    }
}
exports.ReplicaProvider = ReplicaProvider;
// Map of getOrCreate() calls per stack
ReplicaProvider.getOrCreateCalls = new Map();
//# sourceMappingURL=data:application/json;base64,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