"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseCluster = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const secretsmanager = require("../../aws-secretsmanager"); // Automatically re-written from '@aws-cdk/aws-secretsmanager'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const database_secret_1 = require("./database-secret");
const docdb_generated_1 = require("./docdb.generated");
const endpoint_1 = require("./endpoint");
/**
 * A new or imported clustered database.
 */
class DatabaseClusterBase extends core_1.Resource {
    /**
     * (experimental) Renders the secret attachment target specifications.
     *
     * @experimental
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.clusterIdentifier,
            targetType: secretsmanager.AttachmentTargetType.DOCDB_DB_CLUSTER,
        };
    }
}
/**
 * (experimental) Create a clustered database with a given number of instances.
 *
 * @experimental
 * @resource AWS::DocDB::DBCluster
 */
class DatabaseCluster extends DatabaseClusterBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h;
        super(scope, id);
        /**
         * (experimental) Identifiers of the replicas.
         *
         * @experimental
         */
        this.instanceIdentifiers = [];
        /**
         * (experimental) Endpoints which address each individual replica.
         *
         * @experimental
         */
        this.instanceEndpoints = [];
        this.vpc = props.vpc;
        this.vpcSubnets = props.vpcSubnets;
        // Determine the subnet(s) to deploy the DocDB cluster to
        const { subnetIds, internetConnectivityEstablished } = this.vpc.selectSubnets(this.vpcSubnets);
        // DocDB clusters require a subnet group with subnets from at least two AZs.
        // We cannot test whether the subnets are in different AZs, but at least we can test the amount.
        // See https://docs.aws.amazon.com/documentdb/latest/developerguide/replication.html#replication.high-availability
        if (subnetIds.length < 2) {
            throw new Error(`Cluster requires at least 2 subnets, got ${subnetIds.length}`);
        }
        const subnetGroup = new docdb_generated_1.CfnDBSubnetGroup(this, 'Subnets', {
            dbSubnetGroupDescription: `Subnets for ${id} database`,
            subnetIds,
        });
        // Create the security group for the DB cluster
        let securityGroup;
        if (props.securityGroup) {
            securityGroup = props.securityGroup;
        }
        else {
            securityGroup = new ec2.SecurityGroup(this, 'SecurityGroup', {
                description: 'DocumentDB security group',
                vpc: this.vpc,
            });
            // HACK: Use an escape-hatch to apply a consistent removal policy to the
            // security group so we don't get errors when trying to delete the stack
            securityGroup.node.defaultChild.applyRemovalPolicy(props.removalPolicy, {
                applyToUpdateReplacePolicy: true,
            });
        }
        this.securityGroupId = securityGroup.securityGroupId;
        // Create the secret manager secret if no password is specified
        let secret;
        if (!props.masterUser.password) {
            secret = new database_secret_1.DatabaseSecret(this, 'Secret', {
                username: props.masterUser.username,
                encryptionKey: props.masterUser.kmsKey,
            });
        }
        // Default to encrypted storage
        const storageEncrypted = (_b = props.storageEncrypted) !== null && _b !== void 0 ? _b : true;
        if (props.kmsKey && !storageEncrypted) {
            throw new Error('KMS key supplied but storageEncrypted is false');
        }
        // Create the DocDB cluster
        this.cluster = new docdb_generated_1.CfnDBCluster(this, 'Resource', {
            // Basic
            engineVersion: props.engineVersion,
            dbClusterIdentifier: props.dbClusterName,
            dbSubnetGroupName: subnetGroup.ref,
            port: props.port,
            vpcSecurityGroupIds: [this.securityGroupId],
            dbClusterParameterGroupName: (_c = props.parameterGroup) === null || _c === void 0 ? void 0 : _c.parameterGroupName,
            // Admin
            masterUsername: secret ? secret.secretValueFromJson('username').toString() : props.masterUser.username,
            masterUserPassword: secret
                ? secret.secretValueFromJson('password').toString()
                : props.masterUser.password.toString(),
            // Backup
            backupRetentionPeriod: (_e = (_d = props.backup) === null || _d === void 0 ? void 0 : _d.retention) === null || _e === void 0 ? void 0 : _e.toDays(),
            preferredBackupWindow: (_f = props.backup) === null || _f === void 0 ? void 0 : _f.preferredWindow,
            preferredMaintenanceWindow: props.preferredMaintenanceWindow,
            // Encryption
            kmsKeyId: (_g = props.kmsKey) === null || _g === void 0 ? void 0 : _g.keyArn,
            storageEncrypted,
        });
        this.cluster.applyRemovalPolicy(props.removalPolicy, {
            applyToUpdateReplacePolicy: true,
        });
        this.clusterIdentifier = this.cluster.ref;
        this.clusterResourceIdentifier = this.cluster.attrClusterResourceId;
        const port = core_1.Token.asNumber(this.cluster.attrPort);
        this.clusterEndpoint = new endpoint_1.Endpoint(this.cluster.attrEndpoint, port);
        this.clusterReadEndpoint = new endpoint_1.Endpoint(this.cluster.attrReadEndpoint, port);
        if (secret) {
            this.secret = secret.attach(this);
        }
        // Create the instances
        const instanceCount = (_h = props.instances) !== null && _h !== void 0 ? _h : DatabaseCluster.DEFAULT_NUM_INSTANCES;
        if (instanceCount < 1) {
            throw new Error('At least one instance is required');
        }
        for (let i = 0; i < instanceCount; i++) {
            const instanceIndex = i + 1;
            const instanceIdentifier = props.instanceIdentifierBase != null ? `${props.instanceIdentifierBase}${instanceIndex}`
                : props.dbClusterName != null ? `${props.dbClusterName}instance${instanceIndex}` : undefined;
            const instance = new docdb_generated_1.CfnDBInstance(this, `Instance${instanceIndex}`, {
                // Link to cluster
                dbClusterIdentifier: this.cluster.ref,
                dbInstanceIdentifier: instanceIdentifier,
                // Instance properties
                dbInstanceClass: databaseInstanceType(props.instanceType),
            });
            instance.applyRemovalPolicy(props.removalPolicy, {
                applyToUpdateReplacePolicy: true,
            });
            // We must have a dependency on the NAT gateway provider here to create
            // things in the right order.
            instance.node.addDependency(internetConnectivityEstablished);
            this.instanceIdentifiers.push(instance.ref);
            this.instanceEndpoints.push(new endpoint_1.Endpoint(instance.attrEndpoint, port));
        }
        this.connections = new ec2.Connections({
            defaultPort: ec2.Port.tcp(port),
            securityGroups: [securityGroup],
        });
    }
    /**
     * (experimental) Import an existing DatabaseCluster from properties.
     *
     * @experimental
     */
    static fromDatabaseClusterAttributes(scope, id, attrs) {
        class Import extends DatabaseClusterBase {
            constructor() {
                super(...arguments);
                this.defaultPort = ec2.Port.tcp(attrs.port);
                this.connections = new ec2.Connections({
                    securityGroups: [attrs.securityGroup],
                    defaultPort: this.defaultPort,
                });
                this.clusterIdentifier = attrs.clusterIdentifier;
                this.instanceIdentifiers = attrs.instanceIdentifiers;
                this.clusterEndpoint = new endpoint_1.Endpoint(attrs.clusterEndpointAddress, attrs.port);
                this.clusterReadEndpoint = new endpoint_1.Endpoint(attrs.readerEndpointAddress, attrs.port);
                this.instanceEndpoints = attrs.instanceEndpointAddresses.map(a => new endpoint_1.Endpoint(a, attrs.port));
                this.securityGroupId = attrs.securityGroup.securityGroupId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * (experimental) Adds the single user rotation of the master password to this cluster.
     *
     * @param automaticallyAfter Specifies the number of days after the previous rotation before Secrets Manager triggers the next automatic rotation.
     * @experimental
     */
    addRotationSingleUser(automaticallyAfter) {
        if (!this.secret) {
            throw new Error('Cannot add single user rotation for a cluster without secret.');
        }
        const id = 'RotationSingleUser';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('A single user rotation was already added to this cluster.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: this.secret,
            automaticallyAfter,
            application: DatabaseCluster.SINGLE_USER_ROTATION_APPLICATION,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
    /**
     * (experimental) Adds the multi user rotation to this cluster.
     *
     * @experimental
     */
    addRotationMultiUser(id, options) {
        if (!this.secret) {
            throw new Error('Cannot add multi user rotation for a cluster without secret.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: options.secret,
            masterSecret: this.secret,
            automaticallyAfter: options.automaticallyAfter,
            application: DatabaseCluster.MULTI_USER_ROTATION_APPLICATION,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
    /**
     * (experimental) Adds security groups to this cluster.
     *
     * @param securityGroups The security groups to add.
     * @experimental
     */
    addSecurityGroups(...securityGroups) {
        if (this.cluster.vpcSecurityGroupIds === undefined) {
            this.cluster.vpcSecurityGroupIds = [];
        }
        this.cluster.vpcSecurityGroupIds.push(...securityGroups.map(sg => sg.securityGroupId));
    }
}
exports.DatabaseCluster = DatabaseCluster;
_a = JSII_RTTI_SYMBOL_1;
DatabaseCluster[_a] = { fqn: "monocdk.aws_docdb.DatabaseCluster", version: "1.106.1" };
/**
 * (experimental) The default number of instances in the DocDB cluster if none are specified.
 *
 * @experimental
 */
DatabaseCluster.DEFAULT_NUM_INSTANCES = 1;
/**
 * (experimental) The default port Document DB listens on.
 *
 * @experimental
 */
DatabaseCluster.DEFAULT_PORT = 27017;
/**
 * The single user secret rotation application.
 */
DatabaseCluster.SINGLE_USER_ROTATION_APPLICATION = secretsmanager.SecretRotationApplication.MONGODB_ROTATION_SINGLE_USER;
/**
 * The multi user secret rotation application.
 */
DatabaseCluster.MULTI_USER_ROTATION_APPLICATION = secretsmanager.SecretRotationApplication.MONGODB_ROTATION_MULTI_USER;
/**
 * Turn a regular instance type into a database instance type
 */
function databaseInstanceType(instanceType) {
    return 'db.' + instanceType.toString();
}
//# sourceMappingURL=data:application/json;base64,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