"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RichAction = void 0;
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * Helper routines to work with Actions
 *
 * Can't put these on Action themselves since we only have an interface
 * and every library would need to reimplement everything (there is no
 * `ActionBase`).
 *
 * So here go the members that should have gone onto the Action class
 * but can't.
 *
 * It was probably my own idea but I don't want it anymore:
 * https://github.com/aws/aws-cdk/issues/10393
 */
class RichAction {
    constructor(action, pipeline) {
        this.action = action;
        this.pipeline = pipeline;
        this.actionProperties = action.actionProperties;
    }
    bind(scope, stage, options) {
        return this.action.bind(scope, stage, options);
    }
    onStateChange(name, target, options) {
        return this.action.onStateChange(name, target, options);
    }
    get isCrossRegion() {
        return !actionDimensionSameAsPipelineDimension(this.effectiveRegion, this.pipeline.env.region);
    }
    get isCrossAccount() {
        return !actionDimensionSameAsPipelineDimension(this.effectiveAccount, this.pipeline.env.account);
    }
    /**
     * Returns the Stack of the resource backing this action
     * if they belong to the same environment.
     * Returns `undefined` if either this action is not backed by a resource,
     * or if the resource does not belong to the same env as its Stack
     * (which can happen for imported resources).
     */
    get resourceStack() {
        const actionResource = this.actionProperties.resource;
        if (!actionResource) {
            return undefined;
        }
        const actionResourceStack = core_1.Stack.of(actionResource);
        const actionResourceStackEnv = {
            region: actionResourceStack.region,
            account: actionResourceStack.account,
        };
        return sameEnv(actionResource.env, actionResourceStackEnv) ? actionResourceStack : undefined;
    }
    /**
     * The region this action wants to execute in.
     * `undefined` means it wants to execute in the same region as the pipeline.
     */
    get effectiveRegion() {
        var _a, _b;
        return (_b = (_a = this.action.actionProperties.resource) === null || _a === void 0 ? void 0 : _a.env.region) !== null && _b !== void 0 ? _b : this.action.actionProperties.region;
    }
    /**
     * The account this action wants to execute in.
     * `undefined` means it wants to execute in the same account as the pipeline.
     */
    get effectiveAccount() {
        var _a, _b, _c, _d, _e;
        return (_e = (_b = (_a = this.action.actionProperties.role) === null || _a === void 0 ? void 0 : _a.env.account) !== null && _b !== void 0 ? _b : (_d = (_c = this.action.actionProperties) === null || _c === void 0 ? void 0 : _c.resource) === null || _d === void 0 ? void 0 : _d.env.account) !== null && _e !== void 0 ? _e : this.action.actionProperties.account;
    }
}
exports.RichAction = RichAction;
function actionDimensionSameAsPipelineDimension(actionDim, pipelineDim) {
    // if the action's dimension is `undefined`,
    // it means it is in the same region/account as the pipeline
    if (!actionDim) {
        return true;
    }
    // if the action's region/account is AWS::Region/AWS::AccountId,
    // we assume it's also in the same region/account as the pipeline
    if (core_1.Token.isUnresolved(actionDim)) {
        return true;
    }
    // here, we know the action's dimension is explicitly set;
    // in this case, it must be equal to the pipeline's dimension
    // for the action to be considered in the same region/account
    return core_1.Token.compareStrings(actionDim, pipelineDim) === core_1.TokenComparison.SAME;
}
/**
 * Whether the two envs represent the same environment
 */
function sameEnv(env1, env2) {
    return sameEnvDimension(env1.region, env2.region)
        && sameEnvDimension(env1.account, env2.account);
}
/**
 * Whether two string probably contain the same environment dimension (region or account)
 *
 * Used to compare either accounts or regions, and also returns true if both
 * are unresolved (in which case both are expted to be "current region" or "current account").
 */
function sameEnvDimension(dim1, dim2) {
    return [core_1.TokenComparison.SAME, core_1.TokenComparison.BOTH_UNRESOLVED].includes(core_1.Token.compareStrings(dim1, dim2));
}
//# sourceMappingURL=data:application/json;base64,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