"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaDeploymentGroup = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const constructs_1 = require("constructs");
const codedeploy_generated_1 = require("../codedeploy.generated");
const utils_1 = require("../utils");
const application_1 = require("./application");
const deployment_config_1 = require("./deployment-config");
/**
 * @experimental
 * @resource AWS::CodeDeploy::DeploymentGroup
 */
class LambdaDeploymentGroup extends cdk.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.deploymentGroupName,
        });
        this.application = props.application || new application_1.LambdaApplication(this, 'Application');
        this.alarms = props.alarms || [];
        this.role = props.role || new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('codedeploy.amazonaws.com'),
        });
        this.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSCodeDeployRoleForLambdaLimited'));
        this.deploymentConfig = props.deploymentConfig || deployment_config_1.LambdaDeploymentConfig.CANARY_10PERCENT_5MINUTES;
        const resource = new codedeploy_generated_1.CfnDeploymentGroup(this, 'Resource', {
            applicationName: this.application.applicationName,
            serviceRoleArn: this.role.roleArn,
            deploymentGroupName: this.physicalName,
            deploymentConfigName: this.deploymentConfig.deploymentConfigName,
            deploymentStyle: {
                deploymentType: 'BLUE_GREEN',
                deploymentOption: 'WITH_TRAFFIC_CONTROL',
            },
            alarmConfiguration: cdk.Lazy.any({ produce: () => utils_1.renderAlarmConfiguration(this.alarms, props.ignorePollAlarmsFailure) }),
            autoRollbackConfiguration: cdk.Lazy.any({ produce: () => utils_1.renderAutoRollbackConfiguration(this.alarms, props.autoRollback) }),
        });
        this.deploymentGroupName = this.getResourceNameAttribute(resource.ref);
        this.deploymentGroupArn = this.getResourceArnAttribute(utils_1.arnForDeploymentGroup(this.application.applicationName, resource.ref), {
            service: 'codedeploy',
            resource: 'deploymentgroup',
            resourceName: `${this.application.applicationName}/${this.physicalName}`,
            sep: ':',
        });
        if (props.preHook) {
            this.addPreHook(props.preHook);
        }
        if (props.postHook) {
            this.addPostHook(props.postHook);
        }
        props.alias.node.defaultChild.cfnOptions.updatePolicy = {
            codeDeployLambdaAliasUpdate: {
                applicationName: this.application.applicationName,
                deploymentGroupName: resource.ref,
                beforeAllowTrafficHook: cdk.Lazy.string({ produce: () => this.preHook && this.preHook.functionName }),
                afterAllowTrafficHook: cdk.Lazy.string({ produce: () => this.postHook && this.postHook.functionName }),
            },
        };
        // If the deployment config is a construct, add a dependency to ensure the deployment config
        // is created before the deployment group is.
        if (this.deploymentConfig instanceof constructs_1.Construct) {
            this.node.addDependency(this.deploymentConfig);
        }
    }
    /**
     * (experimental) Import an Lambda Deployment Group defined either outside the CDK app, or in a different AWS region.
     *
     * @param scope the parent Construct for this new Construct.
     * @param id the logical ID of this new Construct.
     * @param attrs the properties of the referenced Deployment Group.
     * @returns a Construct representing a reference to an existing Deployment Group
     * @experimental
     */
    static fromLambdaDeploymentGroupAttributes(scope, id, attrs) {
        return new ImportedLambdaDeploymentGroup(scope, id, attrs);
    }
    /**
     * (experimental) Associates an additional alarm with this Deployment Group.
     *
     * @param alarm the alarm to associate with this Deployment Group.
     * @experimental
     */
    addAlarm(alarm) {
        this.alarms.push(alarm);
    }
    /**
     * (experimental) Associate a function to run before deployment begins.
     *
     * @param preHook function to run before deployment beings.
     * @experimental
     * @throws an error if a pre-hook function is already configured
     */
    addPreHook(preHook) {
        if (this.preHook !== undefined) {
            throw new Error('A pre-hook function is already defined for this deployment group');
        }
        this.preHook = preHook;
        this.grantPutLifecycleEventHookExecutionStatus(this.preHook);
        this.preHook.grantInvoke(this.role);
    }
    /**
     * (experimental) Associate a function to run after deployment completes.
     *
     * @param postHook function to run after deployment completes.
     * @experimental
     * @throws an error if a post-hook function is already configured
     */
    addPostHook(postHook) {
        if (this.postHook !== undefined) {
            throw new Error('A post-hook function is already defined for this deployment group');
        }
        this.postHook = postHook;
        this.grantPutLifecycleEventHookExecutionStatus(this.postHook);
        this.postHook.grantInvoke(this.role);
    }
    /**
     * (experimental) Grant a principal permission to codedeploy:PutLifecycleEventHookExecutionStatus on this deployment group resource.
     *
     * @param grantee to grant permission to.
     * @experimental
     */
    grantPutLifecycleEventHookExecutionStatus(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            resourceArns: [this.deploymentGroupArn],
            actions: ['codedeploy:PutLifecycleEventHookExecutionStatus'],
        });
    }
}
exports.LambdaDeploymentGroup = LambdaDeploymentGroup;
_a = JSII_RTTI_SYMBOL_1;
LambdaDeploymentGroup[_a] = { fqn: "monocdk.aws_codedeploy.LambdaDeploymentGroup", version: "1.106.1" };
class ImportedLambdaDeploymentGroup extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.application = props.application;
        this.deploymentGroupName = props.deploymentGroupName;
        this.deploymentGroupArn = utils_1.arnForDeploymentGroup(props.application.applicationName, props.deploymentGroupName);
        this.deploymentConfig = props.deploymentConfig || deployment_config_1.LambdaDeploymentConfig.CANARY_10PERCENT_5MINUTES;
    }
}
//# sourceMappingURL=data:application/json;base64,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