"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BuildEnvironmentVariableType = exports.WindowsBuildImage = exports.WindowsImageType = exports.LinuxBuildImage = exports.ImagePullPrincipalType = exports.ComputeType = exports.Project = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const aws_ecr_assets_1 = require("../../aws-ecr-assets"); // Automatically re-written from '@aws-cdk/aws-ecr-assets'
const events = require("../../aws-events"); // Automatically re-written from '@aws-cdk/aws-events'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const kms = require("../../aws-kms"); // Automatically re-written from '@aws-cdk/aws-kms'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const build_spec_1 = require("./build-spec");
const cache_1 = require("./cache");
const codebuild_canned_metrics_generated_1 = require("./codebuild-canned-metrics.generated");
const codebuild_generated_1 = require("./codebuild.generated");
const codepipeline_artifacts_1 = require("./codepipeline-artifacts");
const no_artifacts_1 = require("./no-artifacts");
const no_source_1 = require("./no-source");
const run_script_linux_build_spec_1 = require("./private/run-script-linux-build-spec");
const report_group_utils_1 = require("./report-group-utils");
const source_types_1 = require("./source-types");
/**
 * Represents a reference to a CodeBuild Project.
 *
 * If you're managing the Project alongside the rest of your CDK resources,
 * use the {@link Project} class.
 *
 * If you want to reference an already existing Project
 * (or one defined in a different CDK Stack),
 * use the {@link import} method.
 */
class ProjectBase extends core_1.Resource {
    /**
     * (experimental) Access the Connections object.
     *
     * Will fail if this Project does not have a VPC set.
     *
     * @experimental
     */
    get connections() {
        if (!this._connections) {
            throw new Error('Only VPC-associated Projects have security groups to manage. Supply the "vpc" parameter when creating the Project');
        }
        return this._connections;
    }
    enableBatchBuilds() {
        return undefined;
    }
    /**
     * (experimental) Add a permission only if there's a policy attached.
     *
     * @param statement The permissions statement to add.
     * @experimental
     */
    addToRolePolicy(statement) {
        if (this.role) {
            this.role.addToPrincipalPolicy(statement);
        }
    }
    /**
     * (experimental) Defines a CloudWatch event rule triggered when something happens with this project.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     * @experimental
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.codebuild'],
            detail: {
                'project-name': [this.projectName],
            },
        });
        return rule;
    }
    /**
     * (experimental) Defines a CloudWatch event rule triggered when the build project state changes.
     *
     * You can filter specific build status events using an event
     * pattern filter on the `build-status` detail field:
     *
     *     const rule = project.onStateChange('OnBuildStarted', { target });
     *     rule.addEventPattern({
     *       detail: {
     *         'build-status': [
     *           "IN_PROGRESS",
     *           "SUCCEEDED",
     *           "FAILED",
     *           "STOPPED"
     *         ]
     *       }
     *     });
     *
     * You can also use the methods `onBuildFailed` and `onBuildSucceeded` to define rules for
     * these specific state changes.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     * @experimental
     */
    onStateChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['CodeBuild Build State Change'],
        });
        return rule;
    }
    /**
     * (experimental) Defines a CloudWatch event rule that triggers upon phase change of this build project.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     * @experimental
     */
    onPhaseChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['CodeBuild Build Phase Change'],
        });
        return rule;
    }
    /**
     * (experimental) Defines an event rule which triggers when a build starts.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     *
     * @experimental
     */
    onBuildStarted(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({
            detail: {
                'build-status': ['IN_PROGRESS'],
            },
        });
        return rule;
    }
    /**
     * (experimental) Defines an event rule which triggers when a build fails.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     *
     * @experimental
     */
    onBuildFailed(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({
            detail: {
                'build-status': ['FAILED'],
            },
        });
        return rule;
    }
    /**
     * (experimental) Defines an event rule which triggers when a build completes successfully.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     *
     * @experimental
     */
    onBuildSucceeded(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({
            detail: {
                'build-status': ['SUCCEEDED'],
            },
        });
        return rule;
    }
    /**
     * @param metricName The name of the metric.
     * @param props Customization properties.
     * @returns a CloudWatch metric associated with this build project.
     * @experimental
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/CodeBuild',
            metricName,
            dimensions: { ProjectName: this.projectName },
            ...props,
        }).attachTo(this);
    }
    /**
     * (experimental) Measures the number of builds triggered.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     * @experimental
     */
    metricBuilds(props) {
        return this.cannedMetric(codebuild_canned_metrics_generated_1.CodeBuildMetrics.buildsSum, props);
    }
    /**
     * (experimental) Measures the duration of all builds over time.
     *
     * Units: Seconds
     *
     * Valid CloudWatch statistics: Average (recommended), Maximum, Minimum
     *
     * @default average over 5 minutes
     * @experimental
     */
    metricDuration(props) {
        return this.cannedMetric(codebuild_canned_metrics_generated_1.CodeBuildMetrics.durationAverage, props);
    }
    /**
     * (experimental) Measures the number of successful builds.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     * @experimental
     */
    metricSucceededBuilds(props) {
        return this.cannedMetric(codebuild_canned_metrics_generated_1.CodeBuildMetrics.succeededBuildsSum, props);
    }
    /**
     * (experimental) Measures the number of builds that failed because of client error or because of a timeout.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     * @experimental
     */
    metricFailedBuilds(props) {
        return this.cannedMetric(codebuild_canned_metrics_generated_1.CodeBuildMetrics.failedBuildsSum, props);
    }
    cannedMetric(fn, props) {
        return new cloudwatch.Metric({
            ...fn({ ProjectName: this.projectName }),
            ...props,
        }).attachTo(this);
    }
}
/**
 * (experimental) A representation of a CodeBuild Project.
 *
 * @experimental
 */
class Project extends ProjectBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.projectName,
        });
        this.role = props.role || new iam.Role(this, 'Role', {
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: new iam.ServicePrincipal('codebuild.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        this.buildImage = (props.environment && props.environment.buildImage) || LinuxBuildImage.STANDARD_1_0;
        // let source "bind" to the project. this usually involves granting permissions
        // for the code build role to interact with the source.
        this.source = props.source || new no_source_1.NoSource();
        const sourceConfig = this.source.bind(this, this);
        if (props.badge && !this.source.badgeSupported) {
            throw new Error(`Badge is not supported for source type ${this.source.type}`);
        }
        const artifacts = props.artifacts
            ? props.artifacts
            : (this.source.type === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE
                ? new codepipeline_artifacts_1.CodePipelineArtifacts()
                : new no_artifacts_1.NoArtifacts());
        const artifactsConfig = artifacts.bind(this, this);
        const cache = props.cache || cache_1.Cache.none();
        // give the caching strategy the option to grant permissions to any required resources
        cache._bind(this);
        // Inject download commands for asset if requested
        const environmentVariables = props.environmentVariables || {};
        const buildSpec = props.buildSpec;
        if (this.source.type === source_types_1.NO_SOURCE_TYPE && (buildSpec === undefined || !buildSpec.isImmediate)) {
            throw new Error("If the Project's source is NoSource, you need to provide a concrete buildSpec");
        }
        this._secondarySources = [];
        this._secondarySourceVersions = [];
        this._fileSystemLocations = [];
        for (const secondarySource of props.secondarySources || []) {
            this.addSecondarySource(secondarySource);
        }
        this._secondaryArtifacts = [];
        for (const secondaryArtifact of props.secondaryArtifacts || []) {
            this.addSecondaryArtifact(secondaryArtifact);
        }
        this.validateCodePipelineSettings(artifacts);
        for (const fileSystemLocation of props.fileSystemLocations || []) {
            this.addFileSystemLocation(fileSystemLocation);
        }
        const resource = new codebuild_generated_1.CfnProject(this, 'Resource', {
            description: props.description,
            source: {
                ...sourceConfig.sourceProperty,
                buildSpec: buildSpec && buildSpec.toBuildSpec(),
            },
            artifacts: artifactsConfig.artifactsProperty,
            serviceRole: this.role.roleArn,
            environment: this.renderEnvironment(props, environmentVariables),
            fileSystemLocations: core_1.Lazy.any({ produce: () => this.renderFileSystemLocations() }),
            // lazy, because we have a setter for it in setEncryptionKey
            // The 'alias/aws/s3' default is necessary because leaving the `encryptionKey` field
            // empty will not remove existing encryptionKeys during an update (ref. t/D17810523)
            encryptionKey: core_1.Lazy.string({ produce: () => this._encryptionKey ? this._encryptionKey.keyArn : 'alias/aws/s3' }),
            badgeEnabled: props.badge,
            cache: cache._toCloudFormation(),
            name: this.physicalName,
            timeoutInMinutes: props.timeout && props.timeout.toMinutes(),
            queuedTimeoutInMinutes: props.queuedTimeout && props.queuedTimeout.toMinutes(),
            concurrentBuildLimit: props.concurrentBuildLimit,
            secondarySources: core_1.Lazy.any({ produce: () => this.renderSecondarySources() }),
            secondarySourceVersions: core_1.Lazy.any({ produce: () => this.renderSecondarySourceVersions() }),
            secondaryArtifacts: core_1.Lazy.any({ produce: () => this.renderSecondaryArtifacts() }),
            triggers: sourceConfig.buildTriggers,
            sourceVersion: sourceConfig.sourceVersion,
            vpcConfig: this.configureVpc(props),
            logsConfig: this.renderLoggingConfiguration(props.logging),
            buildBatchConfig: core_1.Lazy.any({
                produce: () => {
                    const config = this._batchServiceRole ? {
                        serviceRole: this._batchServiceRole.roleArn,
                    } : undefined;
                    return config;
                },
            }),
        });
        this.addVpcRequiredPermissions(props, resource);
        this.projectArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'codebuild',
            resource: 'project',
            resourceName: this.physicalName,
        });
        this.projectName = this.getResourceNameAttribute(resource.ref);
        this.addToRolePolicy(this.createLoggingPermission());
        // add permissions to create and use test report groups
        // with names starting with the project's name,
        // unless the customer explicitly opts out of it
        if (props.grantReportGroupPermissions !== false) {
            this.addToRolePolicy(new iam.PolicyStatement({
                actions: [
                    'codebuild:CreateReportGroup',
                    'codebuild:CreateReport',
                    'codebuild:UpdateReport',
                    'codebuild:BatchPutTestCases',
                    'codebuild:BatchPutCodeCoverages',
                ],
                resources: [report_group_utils_1.renderReportGroupArn(this, `${this.projectName}-*`)],
            }));
        }
        if (props.encryptionKey) {
            this.encryptionKey = props.encryptionKey;
        }
        // bind
        const bindFunction = this.buildImage.bind;
        if (bindFunction) {
            bindFunction.call(this.buildImage, this, this, {});
        }
    }
    /**
     * @experimental
     */
    static fromProjectArn(scope, id, projectArn) {
        const parsedArn = core_1.Stack.of(scope).parseArn(projectArn);
        class Import extends ProjectBase {
            constructor(s, i) {
                super(s, i, {
                    account: parsedArn.account,
                    region: parsedArn.region,
                });
                this.projectArn = projectArn;
                this.projectName = parsedArn.resourceName;
                this.role = undefined;
                this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
            }
        }
        return new Import(scope, id);
    }
    /**
     * (experimental) Import a Project defined either outside the CDK, or in a different CDK Stack (and exported using the {@link export} method).
     *
     * @param scope the parent Construct for this Construct.
     * @param id the logical name of this Construct.
     * @param projectName the name of the project to import.
     * @returns a reference to the existing Project
     * @experimental
     * @note if you're importing a CodeBuild Project for use
     * in a CodePipeline, make sure the existing Project
     * has permissions to access the S3 Bucket of that Pipeline -
     * otherwise, builds in that Pipeline will always fail.
     */
    static fromProjectName(scope, id, projectName) {
        class Import extends ProjectBase {
            constructor(s, i) {
                super(s, i);
                this.role = undefined;
                this.projectArn = core_1.Stack.of(this).formatArn({
                    service: 'codebuild',
                    resource: 'project',
                    resourceName: projectName,
                });
                this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
                this.projectName = projectName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * (experimental) Convert the environment variables map of string to {@link BuildEnvironmentVariable}, which is the customer-facing type, to a list of {@link CfnProject.EnvironmentVariableProperty}, which is the representation of environment variables in CloudFormation.
     *
     * @param environmentVariables the map of string to environment variables.
     * @param validateNoPlainTextSecrets whether to throw an exception if any of the plain text environment variables contain secrets, defaults to 'false'.
     * @returns an array of {@link CfnProject.EnvironmentVariableProperty} instances
     * @experimental
     */
    static serializeEnvVariables(environmentVariables, validateNoPlainTextSecrets = false, principal) {
        var _d;
        const ret = new Array();
        const ssmIamResources = new Array();
        const secretsManagerIamResources = new Set();
        const kmsIamResources = new Set();
        for (const [name, envVariable] of Object.entries(environmentVariables)) {
            const envVariableValue = (_d = envVariable.value) === null || _d === void 0 ? void 0 : _d.toString();
            const cfnEnvVariable = {
                name,
                type: envVariable.type || BuildEnvironmentVariableType.PLAINTEXT,
                value: envVariableValue,
            };
            ret.push(cfnEnvVariable);
            // validate that the plain-text environment variables don't contain any secrets in them
            if (validateNoPlainTextSecrets && cfnEnvVariable.type === BuildEnvironmentVariableType.PLAINTEXT) {
                const fragments = core_1.Tokenization.reverseString(cfnEnvVariable.value);
                for (const token of fragments.tokens) {
                    if (token instanceof core_1.SecretValue) {
                        throw new Error(`Plaintext environment variable '${name}' contains a secret value! ` +
                            'This means the value of this variable will be visible in plain text in the AWS Console. ' +
                            "Please consider using CodeBuild's SecretsManager environment variables feature instead. " +
                            "If you'd like to continue with having this secret in the plaintext environment variables, " +
                            'please set the checkSecretsInPlainTextEnvVariables property to false');
                    }
                }
            }
            if (principal) {
                const stack = core_1.Stack.of(principal);
                // save the SSM env variables
                if (envVariable.type === BuildEnvironmentVariableType.PARAMETER_STORE) {
                    ssmIamResources.push(stack.formatArn({
                        service: 'ssm',
                        resource: 'parameter',
                        // If the parameter name starts with / the resource name is not separated with a double '/'
                        // arn:aws:ssm:region:1111111111:parameter/PARAM_NAME
                        resourceName: envVariableValue.startsWith('/')
                            ? envVariableValue.substr(1)
                            : envVariableValue,
                    }));
                }
                // save SecretsManager env variables
                if (envVariable.type === BuildEnvironmentVariableType.SECRETS_MANAGER) {
                    if (core_1.Token.isUnresolved(envVariableValue)) {
                        // the value of the property can be a complex string, separated by ':';
                        // see https://docs.aws.amazon.com/codebuild/latest/userguide/build-spec-ref.html#build-spec.env.secrets-manager
                        const secretArn = envVariableValue.split(':')[0];
                        // if we are passed a Token, we should assume it's the ARN of the Secret
                        // (as the name would not work anyway, because it would be the full name, which CodeBuild does not support)
                        secretsManagerIamResources.add(secretArn);
                    }
                    else {
                        // check if the provided value is a full ARN of the Secret
                        let parsedArn;
                        try {
                            parsedArn = stack.parseArn(envVariableValue, ':');
                        }
                        catch (e) { }
                        const secretSpecifier = parsedArn ? parsedArn.resourceName : envVariableValue;
                        // the value of the property can be a complex string, separated by ':';
                        // see https://docs.aws.amazon.com/codebuild/latest/userguide/build-spec-ref.html#build-spec.env.secrets-manager
                        const secretName = secretSpecifier.split(':')[0];
                        const secretIamResourceName = parsedArn
                            // If we were given an ARN, we don't' know whether the name is full, or partial,
                            // as CodeBuild supports both ARN forms.
                            // Because of that, follow the name with a '*', which works for both
                            ? `${secretName}*`
                            // If we were given just a name, it must be partial, as CodeBuild doesn't support providing full names.
                            // In this case, we need to accommodate for the generated suffix in the IAM resource name
                            : `${secretName}-??????`;
                        secretsManagerIamResources.add(stack.formatArn({
                            service: 'secretsmanager',
                            resource: 'secret',
                            resourceName: secretIamResourceName,
                            sep: ':',
                            // if we were given an ARN, we need to use the provided partition/account/region
                            partition: parsedArn === null || parsedArn === void 0 ? void 0 : parsedArn.partition,
                            account: parsedArn === null || parsedArn === void 0 ? void 0 : parsedArn.account,
                            region: parsedArn === null || parsedArn === void 0 ? void 0 : parsedArn.region,
                        }));
                        // if secret comes from another account, SecretsManager will need to access
                        // KMS on the other account as well to be able to get the secret
                        if (parsedArn && parsedArn.account && core_1.Token.compareStrings(parsedArn.account, stack.account) === core_1.TokenComparison.DIFFERENT) {
                            kmsIamResources.add(stack.formatArn({
                                service: 'kms',
                                resource: 'key',
                                // We do not know the ID of the key, but since this is a cross-account access,
                                // the key policies have to allow this access, so a wildcard is safe here
                                resourceName: '*',
                                sep: '/',
                                // if we were given an ARN, we need to use the provided partition/account/region
                                partition: parsedArn.partition,
                                account: parsedArn.account,
                                region: parsedArn.region,
                            }));
                        }
                    }
                }
            }
        }
        if (ssmIamResources.length !== 0) {
            principal === null || principal === void 0 ? void 0 : principal.grantPrincipal.addToPrincipalPolicy(new iam.PolicyStatement({
                actions: ['ssm:GetParameters'],
                resources: ssmIamResources,
            }));
        }
        if (secretsManagerIamResources.size !== 0) {
            principal === null || principal === void 0 ? void 0 : principal.grantPrincipal.addToPrincipalPolicy(new iam.PolicyStatement({
                actions: ['secretsmanager:GetSecretValue'],
                resources: Array.from(secretsManagerIamResources),
            }));
        }
        if (kmsIamResources.size !== 0) {
            principal === null || principal === void 0 ? void 0 : principal.grantPrincipal.addToPrincipalPolicy(new iam.PolicyStatement({
                actions: ['kms:Decrypt'],
                resources: Array.from(kmsIamResources),
            }));
        }
        return ret;
    }
    /**
     * (experimental) Enable batch builds.
     *
     * Returns an object contining the batch service role if batch builds
     * could be enabled.
     *
     * @experimental
     */
    enableBatchBuilds() {
        if (!this._batchServiceRole) {
            this._batchServiceRole = new iam.Role(this, 'BatchServiceRole', {
                assumedBy: new iam.ServicePrincipal('codebuild.amazonaws.com'),
            });
            this._batchServiceRole.addToPrincipalPolicy(new iam.PolicyStatement({
                resources: [core_1.Lazy.string({
                        produce: () => this.projectArn,
                    })],
                actions: [
                    'codebuild:StartBuild',
                    'codebuild:StopBuild',
                    'codebuild:RetryBuild',
                ],
            }));
        }
        return {
            role: this._batchServiceRole,
        };
    }
    /**
     * (experimental) Adds a secondary source to the Project.
     *
     * @param secondarySource the source to add as a secondary source.
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-multi-in-out.html
     * @experimental
     */
    addSecondarySource(secondarySource) {
        if (!secondarySource.identifier) {
            throw new Error('The identifier attribute is mandatory for secondary sources');
        }
        const secondarySourceConfig = secondarySource.bind(this, this);
        this._secondarySources.push(secondarySourceConfig.sourceProperty);
        if (secondarySourceConfig.sourceVersion) {
            this._secondarySourceVersions.push({
                sourceIdentifier: secondarySource.identifier,
                sourceVersion: secondarySourceConfig.sourceVersion,
            });
        }
    }
    /**
     * (experimental) Adds a fileSystemLocation to the Project.
     *
     * @param fileSystemLocation the fileSystemLocation to add.
     * @experimental
     */
    addFileSystemLocation(fileSystemLocation) {
        const fileSystemConfig = fileSystemLocation.bind(this, this);
        this._fileSystemLocations.push(fileSystemConfig.location);
    }
    /**
     * (experimental) Adds a secondary artifact to the Project.
     *
     * @param secondaryArtifact the artifact to add as a secondary artifact.
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-multi-in-out.html
     * @experimental
     */
    addSecondaryArtifact(secondaryArtifact) {
        if (!secondaryArtifact.identifier) {
            throw new Error('The identifier attribute is mandatory for secondary artifacts');
        }
        this._secondaryArtifacts.push(secondaryArtifact.bind(this, this).artifactsProperty);
    }
    /**
     * (experimental) A callback invoked when the given project is added to a CodePipeline.
     *
     * @param _scope the construct the binding is taking place in.
     * @param options additional options for the binding.
     * @experimental
     */
    bindToCodePipeline(_scope, options) {
        // work around a bug in CodeBuild: it ignores the KMS key set on the pipeline,
        // and always uses its own, project-level key
        if (options.artifactBucket.encryptionKey && !this._encryptionKey) {
            // we cannot safely do this assignment if the key is of type kms.Key,
            // and belongs to a stack in a different account or region than the project
            // (that would cause an illegal reference, as KMS keys don't have physical names)
            const keyStack = core_1.Stack.of(options.artifactBucket.encryptionKey);
            const projectStack = core_1.Stack.of(this);
            if (!(options.artifactBucket.encryptionKey instanceof kms.Key &&
                (keyStack.account !== projectStack.account || keyStack.region !== projectStack.region))) {
                this.encryptionKey = options.artifactBucket.encryptionKey;
            }
        }
    }
    /**
     * (experimental) Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @experimental
     * @override true
     */
    validate() {
        const ret = new Array();
        if (this.source.type === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE) {
            if (this._secondarySources.length > 0) {
                ret.push('A Project with a CodePipeline Source cannot have secondary sources. ' +
                    "Use the CodeBuild Pipeline Actions' `extraInputs` property instead");
            }
            if (this._secondaryArtifacts.length > 0) {
                ret.push('A Project with a CodePipeline Source cannot have secondary artifacts. ' +
                    "Use the CodeBuild Pipeline Actions' `outputs` property instead");
            }
        }
        return ret;
    }
    set encryptionKey(encryptionKey) {
        this._encryptionKey = encryptionKey;
        encryptionKey.grantEncryptDecrypt(this);
    }
    createLoggingPermission() {
        const logGroupArn = core_1.Stack.of(this).formatArn({
            service: 'logs',
            resource: 'log-group',
            sep: ':',
            resourceName: `/aws/codebuild/${this.projectName}`,
        });
        const logGroupStarArn = `${logGroupArn}:*`;
        return new iam.PolicyStatement({
            resources: [logGroupArn, logGroupStarArn],
            actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
        });
    }
    renderEnvironment(props, projectVars = {}) {
        var _d, _e, _f, _g;
        const env = (_d = props.environment) !== null && _d !== void 0 ? _d : {};
        const vars = {};
        const containerVars = env.environmentVariables || {};
        // first apply environment variables from the container definition
        for (const name of Object.keys(containerVars)) {
            vars[name] = containerVars[name];
        }
        // now apply project-level vars
        for (const name of Object.keys(projectVars)) {
            vars[name] = projectVars[name];
        }
        const hasEnvironmentVars = Object.keys(vars).length > 0;
        const errors = this.buildImage.validate(env);
        if (errors.length > 0) {
            throw new Error('Invalid CodeBuild environment: ' + errors.join('\n'));
        }
        const imagePullPrincipalType = this.buildImage.imagePullPrincipalType === ImagePullPrincipalType.CODEBUILD
            ? ImagePullPrincipalType.CODEBUILD
            : ImagePullPrincipalType.SERVICE_ROLE;
        if (this.buildImage.repository) {
            if (imagePullPrincipalType === ImagePullPrincipalType.SERVICE_ROLE) {
                this.buildImage.repository.grantPull(this);
            }
            else {
                const statement = new iam.PolicyStatement({
                    principals: [new iam.ServicePrincipal('codebuild.amazonaws.com')],
                    actions: ['ecr:GetDownloadUrlForLayer', 'ecr:BatchGetImage', 'ecr:BatchCheckLayerAvailability'],
                });
                statement.sid = 'CodeBuild';
                this.buildImage.repository.addToResourcePolicy(statement);
            }
        }
        if (imagePullPrincipalType === ImagePullPrincipalType.SERVICE_ROLE) {
            (_e = this.buildImage.secretsManagerCredentials) === null || _e === void 0 ? void 0 : _e.grantRead(this);
        }
        const secret = this.buildImage.secretsManagerCredentials;
        return {
            type: this.buildImage.type,
            image: this.buildImage.imageId,
            imagePullCredentialsType: imagePullPrincipalType,
            registryCredential: secret
                ? {
                    credentialProvider: 'SECRETS_MANAGER',
                    // Secrets must be referenced by either the full ARN (with SecretsManager suffix), or by name.
                    // "Partial" ARNs (without the suffix) will fail a validation regex at deploy-time.
                    credential: (_f = secret.secretFullArn) !== null && _f !== void 0 ? _f : secret.secretName,
                }
                : undefined,
            privilegedMode: env.privileged || false,
            computeType: env.computeType || this.buildImage.defaultComputeType,
            environmentVariables: hasEnvironmentVars
                ? Project.serializeEnvVariables(vars, (_g = props.checkSecretsInPlainTextEnvVariables) !== null && _g !== void 0 ? _g : true, this)
                : undefined,
        };
    }
    renderFileSystemLocations() {
        return this._fileSystemLocations.length === 0
            ? undefined
            : this._fileSystemLocations;
    }
    renderSecondarySources() {
        return this._secondarySources.length === 0
            ? undefined
            : this._secondarySources;
    }
    renderSecondarySourceVersions() {
        return this._secondarySourceVersions.length === 0
            ? undefined
            : this._secondarySourceVersions;
    }
    renderSecondaryArtifacts() {
        return this._secondaryArtifacts.length === 0
            ? undefined
            : this._secondaryArtifacts;
    }
    /**
     * If configured, set up the VPC-related properties
     *
     * Returns the VpcConfig that should be added to the
     * codebuild creation properties.
     */
    configureVpc(props) {
        if ((props.securityGroups || props.allowAllOutbound !== undefined) && !props.vpc) {
            throw new Error('Cannot configure \'securityGroup\' or \'allowAllOutbound\' without configuring a VPC');
        }
        if (!props.vpc) {
            return undefined;
        }
        if ((props.securityGroups && props.securityGroups.length > 0) && props.allowAllOutbound !== undefined) {
            throw new Error('Configure \'allowAllOutbound\' directly on the supplied SecurityGroup.');
        }
        let securityGroups;
        if (props.securityGroups && props.securityGroups.length > 0) {
            securityGroups = props.securityGroups;
        }
        else {
            const securityGroup = new ec2.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
                description: 'Automatic generated security group for CodeBuild ' + core_1.Names.uniqueId(this),
                allowAllOutbound: props.allowAllOutbound,
            });
            securityGroups = [securityGroup];
        }
        this._connections = new ec2.Connections({ securityGroups });
        return {
            vpcId: props.vpc.vpcId,
            subnets: props.vpc.selectSubnets(props.subnetSelection).subnetIds,
            securityGroupIds: this.connections.securityGroups.map(s => s.securityGroupId),
        };
    }
    renderLoggingConfiguration(props) {
        var _d, _e, _f, _g, _h;
        if (props === undefined) {
            return undefined;
        }
        let s3Config = undefined;
        let cloudwatchConfig = undefined;
        if (props.s3) {
            const s3Logs = props.s3;
            s3Config = {
                status: ((_d = s3Logs.enabled) !== null && _d !== void 0 ? _d : true) ? 'ENABLED' : 'DISABLED',
                location: `${s3Logs.bucket.bucketName}` + (s3Logs.prefix ? `/${s3Logs.prefix}` : ''),
                encryptionDisabled: s3Logs.encrypted,
            };
            (_e = s3Logs.bucket) === null || _e === void 0 ? void 0 : _e.grantWrite(this);
        }
        if (props.cloudWatch) {
            const cloudWatchLogs = props.cloudWatch;
            const status = ((_f = cloudWatchLogs.enabled) !== null && _f !== void 0 ? _f : true) ? 'ENABLED' : 'DISABLED';
            if (status === 'ENABLED' && !(cloudWatchLogs.logGroup)) {
                throw new Error('Specifying a LogGroup is required if CloudWatch logging for CodeBuild is enabled');
            }
            (_g = cloudWatchLogs.logGroup) === null || _g === void 0 ? void 0 : _g.grantWrite(this);
            cloudwatchConfig = {
                status,
                groupName: (_h = cloudWatchLogs.logGroup) === null || _h === void 0 ? void 0 : _h.logGroupName,
                streamName: cloudWatchLogs.prefix,
            };
        }
        return {
            s3Logs: s3Config,
            cloudWatchLogs: cloudwatchConfig,
        };
    }
    addVpcRequiredPermissions(props, project) {
        if (!props.vpc || !this.role) {
            return;
        }
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            resources: [`arn:${core_1.Aws.PARTITION}:ec2:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:network-interface/*`],
            actions: ['ec2:CreateNetworkInterfacePermission'],
            conditions: {
                StringEquals: {
                    'ec2:Subnet': props.vpc
                        .selectSubnets(props.subnetSelection).subnetIds
                        .map(si => `arn:${core_1.Aws.PARTITION}:ec2:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:subnet/${si}`),
                    'ec2:AuthorizedService': 'codebuild.amazonaws.com',
                },
            },
        }));
        const policy = new iam.Policy(this, 'PolicyDocument', {
            statements: [
                new iam.PolicyStatement({
                    resources: ['*'],
                    actions: [
                        'ec2:CreateNetworkInterface',
                        'ec2:DescribeNetworkInterfaces',
                        'ec2:DeleteNetworkInterface',
                        'ec2:DescribeSubnets',
                        'ec2:DescribeSecurityGroups',
                        'ec2:DescribeDhcpOptions',
                        'ec2:DescribeVpcs',
                    ],
                }),
            ],
        });
        this.role.attachInlinePolicy(policy);
        // add an explicit dependency between the EC2 Policy and this Project -
        // otherwise, creating the Project fails, as it requires these permissions
        // to be already attached to the Project's Role
        project.node.addDependency(policy);
    }
    validateCodePipelineSettings(artifacts) {
        const sourceType = this.source.type;
        const artifactsType = artifacts.type;
        if ((sourceType === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE ||
            artifactsType === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE) &&
            (sourceType !== artifactsType)) {
            throw new Error('Both source and artifacts must be set to CodePipeline');
        }
    }
}
exports.Project = Project;
_a = JSII_RTTI_SYMBOL_1;
Project[_a] = { fqn: "monocdk.aws_codebuild.Project", version: "1.106.1" };
/**
 * (experimental) Build machine compute type.
 *
 * @experimental
 */
var ComputeType;
(function (ComputeType) {
    ComputeType["SMALL"] = "BUILD_GENERAL1_SMALL";
    ComputeType["MEDIUM"] = "BUILD_GENERAL1_MEDIUM";
    ComputeType["LARGE"] = "BUILD_GENERAL1_LARGE";
    ComputeType["X2_LARGE"] = "BUILD_GENERAL1_2XLARGE";
})(ComputeType = exports.ComputeType || (exports.ComputeType = {}));
/**
 * (experimental) The type of principal CodeBuild will use to pull your build Docker image.
 *
 * @experimental
 */
var ImagePullPrincipalType;
(function (ImagePullPrincipalType) {
    ImagePullPrincipalType["CODEBUILD"] = "CODEBUILD";
    ImagePullPrincipalType["SERVICE_ROLE"] = "SERVICE_ROLE";
})(ImagePullPrincipalType = exports.ImagePullPrincipalType || (exports.ImagePullPrincipalType = {}));
class ArmBuildImage {
    constructor(imageId) {
        this.type = 'ARM_CONTAINER';
        this.defaultComputeType = ComputeType.LARGE;
        this.imagePullPrincipalType = ImagePullPrincipalType.CODEBUILD;
        this.imageId = imageId;
    }
    validate(buildEnvironment) {
        const ret = [];
        if (buildEnvironment.computeType &&
            buildEnvironment.computeType !== ComputeType.LARGE) {
            ret.push(`ARM images only support ComputeType '${ComputeType.LARGE}' - ` +
                `'${buildEnvironment.computeType}' was given`);
        }
        return ret;
    }
    runScriptBuildspec(entrypoint) {
        return run_script_linux_build_spec_1.runScriptLinuxBuildSpec(entrypoint);
    }
}
/**
 * (experimental) A CodeBuild image running Linux.
 *
 * This class has a bunch of public constants that represent the most popular images.
 *
 * You can also specify a custom image using one of the static methods:
 *
 * - LinuxBuildImage.fromDockerRegistry(image[, { secretsManagerCredentials }])
 * - LinuxBuildImage.fromEcrRepository(repo[, tag])
 * - LinuxBuildImage.fromAsset(parent, id, props)
 *
 * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
 * @experimental
 */
class LinuxBuildImage {
    constructor(props) {
        /**
         * (experimental) The type of build environment.
         *
         * @experimental
         */
        this.type = 'LINUX_CONTAINER';
        /**
         * (experimental) The default {@link ComputeType} to use with this image, if one was not specified in {@link BuildEnvironment#computeType} explicitly.
         *
         * @experimental
         */
        this.defaultComputeType = ComputeType.SMALL;
        this.imageId = props.imageId;
        this.imagePullPrincipalType = props.imagePullPrincipalType;
        this.secretsManagerCredentials = props.secretsManagerCredentials;
        this.repository = props.repository;
    }
    /**
     * @returns a Linux build image from a Docker Hub image.
     * @experimental
     */
    static fromDockerRegistry(name, options = {}) {
        return new LinuxBuildImage({
            ...options,
            imageId: name,
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
        });
    }
    /**
     * @param repository The ECR repository.
     * @param tag Image tag (default "latest").
     * @returns A Linux build image from an ECR repository.
     *
     * NOTE: if the repository is external (i.e. imported), then we won't be able to add
     * a resource policy statement for it so CodeBuild can pull the image.
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-ecr.html
     * @experimental
     */
    static fromEcrRepository(repository, tag = 'latest') {
        return new LinuxBuildImage({
            imageId: repository.repositoryUriForTag(tag),
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            repository,
        });
    }
    /**
     * (experimental) Uses an Docker image asset as a Linux build image.
     *
     * @experimental
     */
    static fromAsset(scope, id, props) {
        const asset = new aws_ecr_assets_1.DockerImageAsset(scope, id, props);
        return new LinuxBuildImage({
            imageId: asset.imageUri,
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            repository: asset.repository,
        });
    }
    /**
     * (experimental) Uses a Docker image provided by CodeBuild.
     *
     * @param id The image identifier.
     * @returns A Docker image provided by CodeBuild.
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
     * @experimental
     * @example
     *
     * 'aws/codebuild/standard:4.0'
     */
    static fromCodeBuildImageId(id) {
        return LinuxBuildImage.codeBuildImage(id);
    }
    static codeBuildImage(name) {
        return new LinuxBuildImage({
            imageId: name,
            imagePullPrincipalType: ImagePullPrincipalType.CODEBUILD,
        });
    }
    /**
     * (experimental) Allows the image a chance to validate whether the passed configuration is correct.
     *
     * @experimental
     */
    validate(_) {
        return [];
    }
    /**
     * (experimental) Make a buildspec to run the indicated script.
     *
     * @experimental
     */
    runScriptBuildspec(entrypoint) {
        return run_script_linux_build_spec_1.runScriptLinuxBuildSpec(entrypoint);
    }
}
exports.LinuxBuildImage = LinuxBuildImage;
_b = JSII_RTTI_SYMBOL_1;
LinuxBuildImage[_b] = { fqn: "monocdk.aws_codebuild.LinuxBuildImage", version: "1.106.1" };
/**
 * @experimental
 */
LinuxBuildImage.STANDARD_1_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:1.0');
/**
 * @experimental
 */
LinuxBuildImage.STANDARD_2_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:2.0');
/**
 * @experimental
 */
LinuxBuildImage.STANDARD_3_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:3.0');
/**
 * (experimental) The `aws/codebuild/standard:4.0` build image.
 *
 * @experimental
 */
LinuxBuildImage.STANDARD_4_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:4.0');
/**
 * (experimental) The `aws/codebuild/standard:5.0` build image.
 *
 * @experimental
 */
LinuxBuildImage.STANDARD_5_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:5.0');
/**
 * @experimental
 */
LinuxBuildImage.AMAZON_LINUX_2 = LinuxBuildImage.codeBuildImage('aws/codebuild/amazonlinux2-x86_64-standard:1.0');
/**
 * @experimental
 */
LinuxBuildImage.AMAZON_LINUX_2_2 = LinuxBuildImage.codeBuildImage('aws/codebuild/amazonlinux2-x86_64-standard:2.0');
/**
 * (experimental) The Amazon Linux 2 x86_64 standard image, version `3.0`.
 *
 * @experimental
 */
LinuxBuildImage.AMAZON_LINUX_2_3 = LinuxBuildImage.codeBuildImage('aws/codebuild/amazonlinux2-x86_64-standard:3.0');
/**
 * @experimental
 */
LinuxBuildImage.AMAZON_LINUX_2_ARM = new ArmBuildImage('aws/codebuild/amazonlinux2-aarch64-standard:1.0');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_BASE = LinuxBuildImage.codeBuildImage('aws/codebuild/ubuntu-base:14.04');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_ANDROID_JAVA8_24_4_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/android-java-8:24.4.1');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_ANDROID_JAVA8_26_1_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/android-java-8:26.1.1');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_DOCKER_17_09_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/docker:17.09.0');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_DOCKER_18_09_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/docker:18.09.0');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_GOLANG_1_10 = LinuxBuildImage.codeBuildImage('aws/codebuild/golang:1.10');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_GOLANG_1_11 = LinuxBuildImage.codeBuildImage('aws/codebuild/golang:1.11');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_OPEN_JDK_8 = LinuxBuildImage.codeBuildImage('aws/codebuild/java:openjdk-8');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_OPEN_JDK_9 = LinuxBuildImage.codeBuildImage('aws/codebuild/java:openjdk-9');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_OPEN_JDK_11 = LinuxBuildImage.codeBuildImage('aws/codebuild/java:openjdk-11');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_NODEJS_10_14_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/nodejs:10.14.1');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_NODEJS_10_1_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/nodejs:10.1.0');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_NODEJS_8_11_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/nodejs:8.11.0');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_NODEJS_6_3_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/nodejs:6.3.1');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_PHP_5_6 = LinuxBuildImage.codeBuildImage('aws/codebuild/php:5.6');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_PHP_7_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/php:7.0');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_PHP_7_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/php:7.1');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_7_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.7.1');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_6_5 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.6.5');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_5_2 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.5.2');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_4_5 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.4.5');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_3_6 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.3.6');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_PYTHON_2_7_12 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:2.7.12');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_RUBY_2_5_3 = LinuxBuildImage.codeBuildImage('aws/codebuild/ruby:2.5.3');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_RUBY_2_5_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/ruby:2.5.1');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_RUBY_2_3_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/ruby:2.3.1');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_RUBY_2_2_5 = LinuxBuildImage.codeBuildImage('aws/codebuild/ruby:2.2.5');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_DOTNET_CORE_1_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/dot-net:core-1');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_DOTNET_CORE_2_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/dot-net:core-2.0');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_DOTNET_CORE_2_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/dot-net:core-2.1');
/**
 * (experimental) Environment type for Windows Docker images.
 *
 * @experimental
 */
var WindowsImageType;
(function (WindowsImageType) {
    WindowsImageType["STANDARD"] = "WINDOWS_CONTAINER";
    WindowsImageType["SERVER_2019"] = "WINDOWS_SERVER_2019_CONTAINER";
})(WindowsImageType = exports.WindowsImageType || (exports.WindowsImageType = {}));
/**
 * (experimental) A CodeBuild image running Windows.
 *
 * This class has a bunch of public constants that represent the most popular images.
 *
 * You can also specify a custom image using one of the static methods:
 *
 * - WindowsBuildImage.fromDockerRegistry(image[, { secretsManagerCredentials }, imageType])
 * - WindowsBuildImage.fromEcrRepository(repo[, tag, imageType])
 * - WindowsBuildImage.fromAsset(parent, id, props, [, imageType])
 *
 * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
 * @experimental
 */
class WindowsBuildImage {
    constructor(props) {
        var _d;
        /**
         * (experimental) The default {@link ComputeType} to use with this image, if one was not specified in {@link BuildEnvironment#computeType} explicitly.
         *
         * @experimental
         */
        this.defaultComputeType = ComputeType.MEDIUM;
        this.type = ((_d = props.imageType) !== null && _d !== void 0 ? _d : WindowsImageType.STANDARD).toString();
        this.imageId = props.imageId;
        this.imagePullPrincipalType = props.imagePullPrincipalType;
        this.secretsManagerCredentials = props.secretsManagerCredentials;
        this.repository = props.repository;
    }
    /**
     * @returns a Windows build image from a Docker Hub image.
     * @experimental
     */
    static fromDockerRegistry(name, options = {}, imageType = WindowsImageType.STANDARD) {
        return new WindowsBuildImage({
            ...options,
            imageId: name,
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            imageType,
        });
    }
    /**
     * @param repository The ECR repository.
     * @param tag Image tag (default "latest").
     * @returns A Linux build image from an ECR repository.
     *
     * NOTE: if the repository is external (i.e. imported), then we won't be able to add
     * a resource policy statement for it so CodeBuild can pull the image.
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-ecr.html
     * @experimental
     */
    static fromEcrRepository(repository, tag = 'latest', imageType = WindowsImageType.STANDARD) {
        return new WindowsBuildImage({
            imageId: repository.repositoryUriForTag(tag),
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            imageType,
            repository,
        });
    }
    /**
     * (experimental) Uses an Docker image asset as a Windows build image.
     *
     * @experimental
     */
    static fromAsset(scope, id, props, imageType = WindowsImageType.STANDARD) {
        const asset = new aws_ecr_assets_1.DockerImageAsset(scope, id, props);
        return new WindowsBuildImage({
            imageId: asset.imageUri,
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            imageType,
            repository: asset.repository,
        });
    }
    /**
     * (experimental) Allows the image a chance to validate whether the passed configuration is correct.
     *
     * @experimental
     */
    validate(buildEnvironment) {
        const ret = [];
        if (buildEnvironment.computeType === ComputeType.SMALL) {
            ret.push('Windows images do not support the Small ComputeType');
        }
        return ret;
    }
    /**
     * (experimental) Make a buildspec to run the indicated script.
     *
     * @experimental
     */
    runScriptBuildspec(entrypoint) {
        return build_spec_1.BuildSpec.fromObject({
            version: '0.2',
            phases: {
                pre_build: {
                    // Would love to do downloading here and executing in the next step,
                    // but I don't know how to propagate the value of $TEMPDIR.
                    //
                    // Punting for someone who knows PowerShell well enough.
                    commands: [],
                },
                build: {
                    commands: [
                        'Set-Variable -Name TEMPDIR -Value (New-TemporaryFile).DirectoryName',
                        `aws s3 cp s3://$env:${run_script_linux_build_spec_1.S3_BUCKET_ENV}/$env:${run_script_linux_build_spec_1.S3_KEY_ENV} $TEMPDIR\\scripts.zip`,
                        'New-Item -ItemType Directory -Path $TEMPDIR\\scriptdir',
                        'Expand-Archive -Path $TEMPDIR/scripts.zip -DestinationPath $TEMPDIR\\scriptdir',
                        '$env:SCRIPT_DIR = "$TEMPDIR\\scriptdir"',
                        `& $TEMPDIR\\scriptdir\\${entrypoint}`,
                    ],
                },
            },
        });
    }
}
exports.WindowsBuildImage = WindowsBuildImage;
_c = JSII_RTTI_SYMBOL_1;
WindowsBuildImage[_c] = { fqn: "monocdk.aws_codebuild.WindowsBuildImage", version: "1.106.1" };
/**
 * (deprecated) Corresponds to the standard CodeBuild image `aws/codebuild/windows-base:1.0`.
 *
 * @deprecated `WindowsBuildImage.WINDOWS_BASE_2_0` should be used instead.
 */
WindowsBuildImage.WIN_SERVER_CORE_2016_BASE = new WindowsBuildImage({
    imageId: 'aws/codebuild/windows-base:1.0',
    imagePullPrincipalType: ImagePullPrincipalType.CODEBUILD,
});
/**
 * (experimental) The standard CodeBuild image `aws/codebuild/windows-base:2.0`, which is based off Windows Server Core 2016.
 *
 * @experimental
 */
WindowsBuildImage.WINDOWS_BASE_2_0 = new WindowsBuildImage({
    imageId: 'aws/codebuild/windows-base:2.0',
    imagePullPrincipalType: ImagePullPrincipalType.CODEBUILD,
});
/**
 * (experimental) The standard CodeBuild image `aws/codebuild/windows-base:2019-1.0`, which is based off Windows Server Core 2019.
 *
 * @experimental
 */
WindowsBuildImage.WIN_SERVER_CORE_2019_BASE = new WindowsBuildImage({
    imageId: 'aws/codebuild/windows-base:2019-1.0',
    imagePullPrincipalType: ImagePullPrincipalType.CODEBUILD,
    imageType: WindowsImageType.SERVER_2019,
});
/**
 * @experimental
 */
var BuildEnvironmentVariableType;
(function (BuildEnvironmentVariableType) {
    BuildEnvironmentVariableType["PLAINTEXT"] = "PLAINTEXT";
    BuildEnvironmentVariableType["PARAMETER_STORE"] = "PARAMETER_STORE";
    BuildEnvironmentVariableType["SECRETS_MANAGER"] = "SECRETS_MANAGER";
})(BuildEnvironmentVariableType = exports.BuildEnvironmentVariableType || (exports.BuildEnvironmentVariableType = {}));
//# sourceMappingURL=data:application/json;base64,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