"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.dispatchMetric = exports.metricPeriod = exports.metricKey = void 0;
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const METRICKEY_SYMBOL = Symbol('@aws-cdk/aws-cloudwatch.MetricKey');
/**
 * Return a unique string representation for this metric.
 *
 * Can be used to determine as a hash key to determine if 2 Metric objects
 * represent the same metric. Excludes rendering properties.
 */
function metricKey(metric) {
    // Cache on the object itself. This is safe because Metric objects are immutable.
    if (metric.hasOwnProperty(METRICKEY_SYMBOL)) {
        return metric[METRICKEY_SYMBOL];
    }
    const parts = new Array();
    const conf = metric.toMetricConfig();
    if (conf.mathExpression) {
        parts.push(conf.mathExpression.expression);
        for (const id of Object.keys(conf.mathExpression.usingMetrics).sort()) {
            parts.push(id);
            parts.push(metricKey(conf.mathExpression.usingMetrics[id]));
        }
    }
    if (conf.metricStat) {
        parts.push(conf.metricStat.namespace);
        parts.push(conf.metricStat.metricName);
        for (const dim of conf.metricStat.dimensions || []) {
            parts.push(dim.name);
            parts.push(dim.value);
        }
        if (conf.metricStat.statistic) {
            parts.push(conf.metricStat.statistic);
        }
        if (conf.metricStat.period) {
            parts.push(`${conf.metricStat.period.toSeconds()}`);
        }
        if (conf.metricStat.region) {
            parts.push(conf.metricStat.region);
        }
        if (conf.metricStat.account) {
            parts.push(conf.metricStat.account);
        }
    }
    const ret = parts.join('|');
    Object.defineProperty(metric, METRICKEY_SYMBOL, { value: ret });
    return ret;
}
exports.metricKey = metricKey;
/**
 * Return the period of a metric
 *
 * For a stat metric, return the immediate period.
 *
 * For an expression metric, all metrics used in it have been made to have the
 * same period, so we return the period of the first inner metric.
 */
function metricPeriod(metric) {
    return dispatchMetric(metric, {
        withStat(stat) {
            return stat.period;
        },
        withExpression() {
            return metric.period || core_1.Duration.minutes(5);
        },
    });
}
exports.metricPeriod = metricPeriod;
/**
 * Given a metric object, inspect it and call the correct function for the type of output
 *
 * In addition to the metric object itself, takes a callback object with two
 * methods, to be invoked for the particular type of metric.
 *
 * If the metric represent a metric query (nominally generated through an
 * instantiation of `Metric` but can be generated by any class that implements
 * `IMetric`) a particular field in its `toMetricConfig()` output will be set
 * (to wit, `metricStat`) and the `withStat()` callback will be called with
 * that object.
 *
 * If the metric represents an expression (usually by instantiating `MathExpression`
 * but users can implement `IMetric` arbitrarily) the `mathExpression` field
 * will be set in the object returned from `toMetricConfig` and the callback
 * called `withExpression` will be applied to that object.
 *
 * Will return the values returned by the callbacks.
 *
 * To be used as such:
 *
 * ```ts
 * const ret = dispatchMetric(someMetric, {
 *   withStat(stat) {
 *     // do something with stat
 *     return 1;
 *   },
 *   withExpression(expr) {
 *     // do something with expr
 *     return 2;
 *   },
 * });
 * ```
 *
 * This function encapsulates some type analysis that would otherwise have to be
 * repeated in all places where code needs to make a distinction on the type
 * of metric object that is being passed.
 */
// eslint-disable-next-line max-len
function dispatchMetric(metric, fns) {
    const conf = metric.toMetricConfig();
    if (conf.metricStat && conf.mathExpression) {
        throw new Error('Metric object must not produce both \'metricStat\' and \'mathExpression\'');
    }
    else if (conf.metricStat) {
        return fns.withStat(conf.metricStat, conf);
    }
    else if (conf.mathExpression) {
        return fns.withExpression(conf.mathExpression, conf);
    }
    else {
        throw new Error('Metric object must have either \'metricStat\' or \'mathExpression\'');
    }
}
exports.dispatchMetric = dispatchMetric;
//# sourceMappingURL=data:application/json;base64,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