"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MathExpression = exports.Metric = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const alarm_1 = require("./alarm");
const metric_util_1 = require("./private/metric-util");
const statistic_1 = require("./private/statistic");
/**
 * (experimental) A metric emitted by a service.
 *
 * The metric is a combination of a metric identifier (namespace, name and dimensions)
 * and an aggregation function (statistic, period and unit).
 *
 * It also contains metadata which is used only in graphs, such as color and label.
 * It makes sense to embed this in here, so that compound constructs can attach
 * that metadata to metrics they expose.
 *
 * This class does not represent a resource, so hence is not a construct. Instead,
 * Metric is an abstraction that makes it easy to specify metrics for use in both
 * alarms and graphs.
 *
 * @experimental
 */
class Metric {
    /**
     * @experimental
     */
    constructor(props) {
        this.period = props.period || cdk.Duration.minutes(5);
        const periodSec = this.period.toSeconds();
        if (periodSec !== 1 && periodSec !== 5 && periodSec !== 10 && periodSec !== 30 && periodSec % 60 !== 0) {
            throw new Error(`'period' must be 1, 5, 10, 30, or a multiple of 60 seconds, received ${periodSec}`);
        }
        if (props.dimensions) {
            this.validateDimensions(props.dimensions);
        }
        this.dimensions = props.dimensions;
        this.namespace = props.namespace;
        this.metricName = props.metricName;
        // Try parsing, this will throw if it's not a valid stat
        this.statistic = statistic_1.normalizeStatistic(props.statistic || 'Average');
        this.label = props.label;
        this.color = props.color;
        this.unit = props.unit;
        this.account = props.account;
        this.region = props.region;
    }
    /**
     * (experimental) Grant permissions to the given identity to write metrics.
     *
     * @param grantee The IAM identity to give permissions to.
     * @experimental
     */
    static grantPutMetricData(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['cloudwatch:PutMetricData'],
            resourceArns: ['*'],
        });
    }
    /**
     * (experimental) Return a copy of Metric `with` properties changed.
     *
     * All properties except namespace and metricName can be changed.
     *
     * @param props The set of properties to change.
     * @experimental
     */
    with(props) {
        // Short-circuit creating a new object if there would be no effective change
        if ((props.label === undefined || props.label === this.label)
            && (props.color === undefined || props.color === this.color)
            && (props.statistic === undefined || props.statistic === this.statistic)
            && (props.unit === undefined || props.unit === this.unit)
            && (props.account === undefined || props.account === this.account)
            && (props.region === undefined || props.region === this.region)
            // For these we're not going to do deep equality, misses some opportunity for optimization
            // but that's okay.
            && (props.dimensions === undefined)
            && (props.period === undefined || props.period.toSeconds() === this.period.toSeconds())) {
            return this;
        }
        return new Metric({
            dimensions: ifUndefined(props.dimensions, this.dimensions),
            namespace: this.namespace,
            metricName: this.metricName,
            period: ifUndefined(props.period, this.period),
            statistic: ifUndefined(props.statistic, this.statistic),
            unit: ifUndefined(props.unit, this.unit),
            label: ifUndefined(props.label, this.label),
            color: ifUndefined(props.color, this.color),
            account: ifUndefined(props.account, this.account),
            region: ifUndefined(props.region, this.region),
        });
    }
    /**
     * (experimental) Attach the metric object to the given construct scope.
     *
     * Returns a Metric object that uses the account and region from the Stack
     * the given construct is defined in. If the metric is subsequently used
     * in a Dashboard or Alarm in a different Stack defined in a different
     * account or region, the appropriate 'region' and 'account' fields
     * will be added to it.
     *
     * If the scope we attach to is in an environment-agnostic stack,
     * nothing is done and the same Metric object is returned.
     *
     * @experimental
     */
    attachTo(scope) {
        const stack = cdk.Stack.of(scope);
        return this.with({
            region: cdk.Token.isUnresolved(stack.region) ? undefined : stack.region,
            account: cdk.Token.isUnresolved(stack.account) ? undefined : stack.account,
        });
    }
    /**
     * (experimental) Inspect the details of the metric object.
     *
     * @experimental
     */
    toMetricConfig() {
        const dims = this.dimensionsAsList();
        return {
            metricStat: {
                dimensions: dims.length > 0 ? dims : undefined,
                namespace: this.namespace,
                metricName: this.metricName,
                period: this.period,
                statistic: this.statistic,
                unitFilter: this.unit,
                account: this.account,
                region: this.region,
            },
            renderingProperties: {
                color: this.color,
                label: this.label,
            },
        };
    }
    /**
     * (deprecated) Turn this metric object into an alarm configuration.
     *
     * @deprecated use toMetricConfig()
     */
    toAlarmConfig() {
        const metricConfig = this.toMetricConfig();
        if (metricConfig.metricStat === undefined) {
            throw new Error('Using a math expression is not supported here. Pass a \'Metric\' object instead');
        }
        const stat = statistic_1.parseStatistic(metricConfig.metricStat.statistic);
        return {
            dimensions: metricConfig.metricStat.dimensions,
            namespace: metricConfig.metricStat.namespace,
            metricName: metricConfig.metricStat.metricName,
            period: metricConfig.metricStat.period.toSeconds(),
            statistic: stat.type === 'simple' ? stat.statistic : undefined,
            extendedStatistic: stat.type === 'percentile' ? 'p' + stat.percentile : undefined,
            unit: this.unit,
        };
    }
    /**
     * (deprecated) Turn this metric object into a graph configuration.
     *
     * @deprecated use toMetricConfig()
     */
    toGraphConfig() {
        var _c, _d, _e, _f;
        const metricConfig = this.toMetricConfig();
        if (metricConfig.metricStat === undefined) {
            throw new Error('Using a math expression is not supported here. Pass a \'Metric\' object instead');
        }
        return {
            dimensions: metricConfig.metricStat.dimensions,
            namespace: metricConfig.metricStat.namespace,
            metricName: metricConfig.metricStat.metricName,
            renderingProperties: {
                period: metricConfig.metricStat.period.toSeconds(),
                stat: metricConfig.metricStat.statistic,
                color: asString((_c = metricConfig.renderingProperties) === null || _c === void 0 ? void 0 : _c.color),
                label: asString((_d = metricConfig.renderingProperties) === null || _d === void 0 ? void 0 : _d.label),
            },
            // deprecated properties for backwards compatibility
            period: metricConfig.metricStat.period.toSeconds(),
            statistic: metricConfig.metricStat.statistic,
            color: asString((_e = metricConfig.renderingProperties) === null || _e === void 0 ? void 0 : _e.color),
            label: asString((_f = metricConfig.renderingProperties) === null || _f === void 0 ? void 0 : _f.label),
            unit: this.unit,
        };
    }
    /**
     * (experimental) Make a new Alarm for this metric.
     *
     * Combines both properties that may adjust the metric (aggregation) as well
     * as alarm properties.
     *
     * @experimental
     */
    createAlarm(scope, id, props) {
        return new alarm_1.Alarm(scope, id, {
            metric: this.with({
                statistic: props.statistic,
                period: props.period,
            }),
            alarmName: props.alarmName,
            alarmDescription: props.alarmDescription,
            comparisonOperator: props.comparisonOperator,
            datapointsToAlarm: props.datapointsToAlarm,
            threshold: props.threshold,
            evaluationPeriods: props.evaluationPeriods,
            evaluateLowSampleCountPercentile: props.evaluateLowSampleCountPercentile,
            treatMissingData: props.treatMissingData,
            actionsEnabled: props.actionsEnabled,
        });
    }
    /**
     * (experimental) Returns a string representation of an object.
     *
     * @experimental
     */
    toString() {
        return this.label || this.metricName;
    }
    /**
     * Return the dimensions of this Metric as a list of Dimension.
     */
    dimensionsAsList() {
        const dims = this.dimensions;
        if (dims === undefined) {
            return [];
        }
        const list = Object.keys(dims).sort().map(key => ({ name: key, value: dims[key] }));
        return list;
    }
    validateDimensions(dims) {
        var dimsArray = Object.keys(dims);
        if ((dimsArray === null || dimsArray === void 0 ? void 0 : dimsArray.length) > 10) {
            throw new Error(`The maximum number of dimensions is 10, received ${dimsArray.length}`);
        }
        dimsArray.map(key => {
            if (dims[key] === undefined || dims[key] === null) {
                throw new Error(`Dimension value of '${dims[key]}' is invalid`);
            }
            ;
            if (key.length < 1 || key.length > 255) {
                throw new Error(`Dimension name must be at least 1 and no more than 255 characters; received ${key}`);
            }
            ;
            if (dims[key].length < 1 || dims[key].length > 255) {
                throw new Error(`Dimension value must be at least 1 and no more than 255 characters; received ${dims[key]}`);
            }
            ;
        });
    }
}
exports.Metric = Metric;
_a = JSII_RTTI_SYMBOL_1;
Metric[_a] = { fqn: "monocdk.aws_cloudwatch.Metric", version: "1.106.1" };
function asString(x) {
    if (x === undefined) {
        return undefined;
    }
    if (typeof x !== 'string') {
        throw new Error(`Expected string, got ${x}`);
    }
    return x;
}
/**
 * (experimental) A math expression built with metric(s) emitted by a service.
 *
 * The math expression is a combination of an expression (x+y) and metrics to apply expression on.
 * It also contains metadata which is used only in graphs, such as color and label.
 * It makes sense to embed this in here, so that compound constructs can attach
 * that metadata to metrics they expose.
 *
 * This class does not represent a resource, so hence is not a construct. Instead,
 * MathExpression is an abstraction that makes it easy to specify metrics for use in both
 * alarms and graphs.
 *
 * @experimental
 */
class MathExpression {
    /**
     * @experimental
     */
    constructor(props) {
        this.period = props.period || cdk.Duration.minutes(5);
        this.expression = props.expression;
        this.usingMetrics = changeAllPeriods(props.usingMetrics, this.period);
        this.label = props.label;
        this.color = props.color;
        const invalidVariableNames = Object.keys(props.usingMetrics).filter(x => !validVariableName(x));
        if (invalidVariableNames.length > 0) {
            throw new Error(`Invalid variable names in expression: ${invalidVariableNames}. Must start with lowercase letter and only contain alphanumerics.`);
        }
        this.validateNoIdConflicts();
    }
    /**
     * (experimental) Return a copy of Metric with properties changed.
     *
     * All properties except namespace and metricName can be changed.
     *
     * @param props The set of properties to change.
     * @experimental
     */
    with(props) {
        // Short-circuit creating a new object if there would be no effective change
        if ((props.label === undefined || props.label === this.label)
            && (props.color === undefined || props.color === this.color)
            && (props.period === undefined || props.period.toSeconds() === this.period.toSeconds())) {
            return this;
        }
        return new MathExpression({
            expression: this.expression,
            usingMetrics: this.usingMetrics,
            label: ifUndefined(props.label, this.label),
            color: ifUndefined(props.color, this.color),
            period: ifUndefined(props.period, this.period),
        });
    }
    /**
     * (deprecated) Turn this metric object into an alarm configuration.
     *
     * @deprecated use toMetricConfig()
     */
    toAlarmConfig() {
        throw new Error('Using a math expression is not supported here. Pass a \'Metric\' object instead');
    }
    /**
     * (deprecated) Turn this metric object into a graph configuration.
     *
     * @deprecated use toMetricConfig()
     */
    toGraphConfig() {
        throw new Error('Using a math expression is not supported here. Pass a \'Metric\' object instead');
    }
    /**
     * (experimental) Inspect the details of the metric object.
     *
     * @experimental
     */
    toMetricConfig() {
        return {
            mathExpression: {
                period: this.period.toSeconds(),
                expression: this.expression,
                usingMetrics: this.usingMetrics,
            },
            renderingProperties: {
                label: this.label,
                color: this.color,
            },
        };
    }
    /**
     * (experimental) Make a new Alarm for this metric.
     *
     * Combines both properties that may adjust the metric (aggregation) as well
     * as alarm properties.
     *
     * @experimental
     */
    createAlarm(scope, id, props) {
        return new alarm_1.Alarm(scope, id, {
            metric: this.with({
                period: props.period,
            }),
            alarmName: props.alarmName,
            alarmDescription: props.alarmDescription,
            comparisonOperator: props.comparisonOperator,
            datapointsToAlarm: props.datapointsToAlarm,
            threshold: props.threshold,
            evaluationPeriods: props.evaluationPeriods,
            evaluateLowSampleCountPercentile: props.evaluateLowSampleCountPercentile,
            treatMissingData: props.treatMissingData,
            actionsEnabled: props.actionsEnabled,
        });
    }
    /**
     * (experimental) Returns a string representation of an object.
     *
     * @experimental
     */
    toString() {
        return this.label || this.expression;
    }
    validateNoIdConflicts() {
        const seen = new Map();
        visit(this);
        function visit(metric) {
            metric_util_1.dispatchMetric(metric, {
                withStat() {
                    // Nothing
                },
                withExpression(expr) {
                    for (const [id, subMetric] of Object.entries(expr.usingMetrics)) {
                        const existing = seen.get(id);
                        if (existing && metric_util_1.metricKey(existing) !== metric_util_1.metricKey(subMetric)) {
                            throw new Error(`The ID '${id}' used for two metrics in the expression: '${subMetric}' and '${existing}'. Rename one.`);
                        }
                        seen.set(id, subMetric);
                        visit(subMetric);
                    }
                },
            });
        }
    }
}
exports.MathExpression = MathExpression;
_b = JSII_RTTI_SYMBOL_1;
MathExpression[_b] = { fqn: "monocdk.aws_cloudwatch.MathExpression", version: "1.106.1" };
const VALID_VARIABLE = new RegExp('^[a-z][a-zA-Z0-9_]*$');
function validVariableName(x) {
    return VALID_VARIABLE.test(x);
}
function ifUndefined(x, def) {
    if (x !== undefined) {
        return x;
    }
    return def;
}
/**
 * Change periods of all metrics in the map
 */
function changeAllPeriods(metrics, period) {
    const ret = {};
    for (const [id, metric] of Object.entries(metrics)) {
        ret[id] = changePeriod(metric, period);
    }
    return ret;
}
/**
 * Return a new metric object which is the same type as the input object, but with the period changed
 *
 * Relies on the fact that implementations of `IMetric` are also supposed to have
 * an implementation of `with` that accepts an argument called `period`. See `IModifiableMetric`.
 */
function changePeriod(metric, period) {
    if (isModifiableMetric(metric)) {
        return metric.with({ period });
    }
    throw new Error(`Metric object should also implement 'with': ${metric}`);
}
function isModifiableMetric(m) {
    return typeof m === 'object' && m !== null && !!m.with;
}
//# sourceMappingURL=data:application/json;base64,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