"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CacheBehavior = void 0;
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cache_policy_1 = require("../cache-policy");
const distribution_1 = require("../distribution");
/**
 * Allows configuring a variety of CloudFront functionality for a given URL path pattern.
 *
 * Note: This really should simply by called 'Behavior', but this name is already taken by the legacy
 * CloudFrontWebDistribution implementation.
 */
class CacheBehavior {
    constructor(originId, props) {
        this.props = props;
        this.originId = originId;
        this.validateEdgeLambdas(props.edgeLambdas);
        this.grantEdgeLambdaFunctionExecutionRole(props.edgeLambdas);
    }
    /**
     * Creates and returns the CloudFormation representation of this behavior.
     * This renders as a "CacheBehaviorProperty" regardless of if this is a default
     * cache behavior or not, as the two are identical except that the pathPattern
     * is omitted for the default cache behavior.
     *
     * @internal
     */
    _renderBehavior() {
        var _a, _b, _c, _d, _e, _f, _g, _h;
        return {
            pathPattern: this.props.pathPattern,
            targetOriginId: this.originId,
            allowedMethods: (_a = this.props.allowedMethods) === null || _a === void 0 ? void 0 : _a.methods,
            cachedMethods: (_b = this.props.cachedMethods) === null || _b === void 0 ? void 0 : _b.methods,
            cachePolicyId: ((_c = this.props.cachePolicy) !== null && _c !== void 0 ? _c : cache_policy_1.CachePolicy.CACHING_OPTIMIZED).cachePolicyId,
            compress: (_d = this.props.compress) !== null && _d !== void 0 ? _d : true,
            originRequestPolicyId: (_e = this.props.originRequestPolicy) === null || _e === void 0 ? void 0 : _e.originRequestPolicyId,
            smoothStreaming: this.props.smoothStreaming,
            viewerProtocolPolicy: (_f = this.props.viewerProtocolPolicy) !== null && _f !== void 0 ? _f : distribution_1.ViewerProtocolPolicy.ALLOW_ALL,
            lambdaFunctionAssociations: (_g = this.props.edgeLambdas) === null || _g === void 0 ? void 0 : _g.map(edgeLambda => ({
                lambdaFunctionArn: edgeLambda.functionVersion.edgeArn,
                eventType: edgeLambda.eventType.toString(),
                includeBody: edgeLambda.includeBody,
            })),
            trustedKeyGroups: (_h = this.props.trustedKeyGroups) === null || _h === void 0 ? void 0 : _h.map(keyGroup => keyGroup.keyGroupId),
        };
    }
    validateEdgeLambdas(edgeLambdas) {
        const includeBodyEventTypes = [distribution_1.LambdaEdgeEventType.ORIGIN_REQUEST, distribution_1.LambdaEdgeEventType.VIEWER_REQUEST];
        if (edgeLambdas && edgeLambdas.some(lambda => lambda.includeBody && !includeBodyEventTypes.includes(lambda.eventType))) {
            throw new Error('\'includeBody\' can only be true for ORIGIN_REQUEST or VIEWER_REQUEST event types.');
        }
    }
    grantEdgeLambdaFunctionExecutionRole(edgeLambdas) {
        if (!edgeLambdas || edgeLambdas.length === 0) {
            return;
        }
        edgeLambdas.forEach((edgeLambda) => {
            const role = edgeLambda.functionVersion.role;
            if (role && role instanceof iam.Role && role.assumeRolePolicy) {
                role.assumeRolePolicy.addStatements(new iam.PolicyStatement({
                    actions: ['sts:AssumeRole'],
                    principals: [new iam.ServicePrincipal('edgelambda.amazonaws.com')],
                }));
            }
        });
    }
}
exports.CacheBehavior = CacheBehavior;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2FjaGUtYmVoYXZpb3IuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJjYWNoZS1iZWhhdmlvci50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSx3Q0FBd0MsQ0FBQyxtREFBbUQ7QUFDNUYsa0RBQThDO0FBRTlDLGtEQUE0RztBQVk1Rzs7Ozs7R0FLRztBQUNILE1BQWEsYUFBYTtJQUV0QixZQUFZLFFBQWdCLEVBQW1CLEtBQXlCO1FBQXpCLFVBQUssR0FBTCxLQUFLLENBQW9CO1FBQ3BFLElBQUksQ0FBQyxRQUFRLEdBQUcsUUFBUSxDQUFDO1FBQ3pCLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUFDLENBQUM7UUFDNUMsSUFBSSxDQUFDLG9DQUFvQyxDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUMsQ0FBQztJQUNqRSxDQUFDO0lBQ0Q7Ozs7Ozs7T0FPRztJQUNJLGVBQWU7O1FBQ2xCLE9BQU87WUFDSCxXQUFXLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxXQUFXO1lBQ25DLGNBQWMsRUFBRSxJQUFJLENBQUMsUUFBUTtZQUM3QixjQUFjLFFBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFjLDBDQUFFLE9BQU87WUFDbEQsYUFBYSxRQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsYUFBYSwwQ0FBRSxPQUFPO1lBQ2hELGFBQWEsRUFBRSxPQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsV0FBVyxtQ0FBSSwwQkFBVyxDQUFDLGlCQUFpQixDQUFDLENBQUMsYUFBYTtZQUN0RixRQUFRLFFBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLG1DQUFJLElBQUk7WUFDckMscUJBQXFCLFFBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxtQkFBbUIsMENBQUUscUJBQXFCO1lBQzVFLGVBQWUsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLGVBQWU7WUFDM0Msb0JBQW9CLFFBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxvQkFBb0IsbUNBQUksbUNBQW9CLENBQUMsU0FBUztZQUN2RiwwQkFBMEIsUUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFdBQVcsMENBQUUsR0FBRyxDQUFDLFVBQVUsQ0FBQyxFQUFFLENBQUMsQ0FBQztnQkFDbkUsaUJBQWlCLEVBQUUsVUFBVSxDQUFDLGVBQWUsQ0FBQyxPQUFPO2dCQUNyRCxTQUFTLEVBQUUsVUFBVSxDQUFDLFNBQVMsQ0FBQyxRQUFRLEVBQUU7Z0JBQzFDLFdBQVcsRUFBRSxVQUFVLENBQUMsV0FBVzthQUN0QyxDQUFDLENBQUM7WUFDSCxnQkFBZ0IsUUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLGdCQUFnQiwwQ0FBRSxHQUFHLENBQUMsUUFBUSxDQUFDLEVBQUUsQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDO1NBQ3RGLENBQUM7SUFDTixDQUFDO0lBQ08sbUJBQW1CLENBQUMsV0FBMEI7UUFDbEQsTUFBTSxxQkFBcUIsR0FBRyxDQUFDLGtDQUFtQixDQUFDLGNBQWMsRUFBRSxrQ0FBbUIsQ0FBQyxjQUFjLENBQUMsQ0FBQztRQUN2RyxJQUFJLFdBQVcsSUFBSSxXQUFXLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUFDLFdBQVcsSUFBSSxDQUFDLHFCQUFxQixDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLENBQUMsRUFBRTtZQUNwSCxNQUFNLElBQUksS0FBSyxDQUFDLG9GQUFvRixDQUFDLENBQUM7U0FDekc7SUFDTCxDQUFDO0lBQ08sb0NBQW9DLENBQUMsV0FBMEI7UUFDbkUsSUFBSSxDQUFDLFdBQVcsSUFBSSxXQUFXLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUMxQyxPQUFPO1NBQ1Y7UUFDRCxXQUFXLENBQUMsT0FBTyxDQUFDLENBQUMsVUFBVSxFQUFFLEVBQUU7WUFDL0IsTUFBTSxJQUFJLEdBQUcsVUFBVSxDQUFDLGVBQWUsQ0FBQyxJQUFJLENBQUM7WUFDN0MsSUFBSSxJQUFJLElBQUksSUFBSSxZQUFZLEdBQUcsQ0FBQyxJQUFJLElBQUksSUFBSSxDQUFDLGdCQUFnQixFQUFFO2dCQUMzRCxJQUFJLENBQUMsZ0JBQWdCLENBQUMsYUFBYSxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztvQkFDeEQsT0FBTyxFQUFFLENBQUMsZ0JBQWdCLENBQUM7b0JBQzNCLFVBQVUsRUFBRSxDQUFDLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLDBCQUEwQixDQUFDLENBQUM7aUJBQ3JFLENBQUMsQ0FBQyxDQUFDO2FBQ1A7UUFDTCxDQUFDLENBQUMsQ0FBQztJQUNQLENBQUM7Q0FDSjtBQXRERCxzQ0FzREMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBpYW0gZnJvbSBcIi4uLy4uLy4uL2F3cy1pYW1cIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nXG5pbXBvcnQgeyBDYWNoZVBvbGljeSB9IGZyb20gJy4uL2NhY2hlLXBvbGljeSc7XG5pbXBvcnQgeyBDZm5EaXN0cmlidXRpb24gfSBmcm9tICcuLi9jbG91ZGZyb250LmdlbmVyYXRlZCc7XG5pbXBvcnQgeyBBZGRCZWhhdmlvck9wdGlvbnMsIEVkZ2VMYW1iZGEsIExhbWJkYUVkZ2VFdmVudFR5cGUsIFZpZXdlclByb3RvY29sUG9saWN5IH0gZnJvbSAnLi4vZGlzdHJpYnV0aW9uJztcbi8qKlxuICogUHJvcGVydGllcyBmb3Igc3BlY2lmeWluZyBjdXN0b20gYmVoYXZpb3JzIGZvciBvcmlnaW5zLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIENhY2hlQmVoYXZpb3JQcm9wcyBleHRlbmRzIEFkZEJlaGF2aW9yT3B0aW9ucyB7XG4gICAgLyoqXG4gICAgICogVGhlIHBhdHRlcm4gKGUuZy4sIGBpbWFnZXMvKi5qcGdgKSB0aGF0IHNwZWNpZmllcyB3aGljaCByZXF1ZXN0cyB0byBhcHBseSB0aGUgYmVoYXZpb3IgdG8uXG4gICAgICogVGhlcmUgbXVzdCBiZSBleGFjdGx5IG9uZSBiZWhhdmlvciBhc3NvY2lhdGVkIHdpdGggZWFjaCBgRGlzdHJpYnV0aW9uYCB0aGF0IGhhcyBhIHBhdGggcGF0dGVyblxuICAgICAqIG9mICcqJywgd2hpY2ggYWN0cyBhcyB0aGUgY2F0Y2gtYWxsIGRlZmF1bHQgYmVoYXZpb3IuXG4gICAgICovXG4gICAgcmVhZG9ubHkgcGF0aFBhdHRlcm46IHN0cmluZztcbn1cbi8qKlxuICogQWxsb3dzIGNvbmZpZ3VyaW5nIGEgdmFyaWV0eSBvZiBDbG91ZEZyb250IGZ1bmN0aW9uYWxpdHkgZm9yIGEgZ2l2ZW4gVVJMIHBhdGggcGF0dGVybi5cbiAqXG4gKiBOb3RlOiBUaGlzIHJlYWxseSBzaG91bGQgc2ltcGx5IGJ5IGNhbGxlZCAnQmVoYXZpb3InLCBidXQgdGhpcyBuYW1lIGlzIGFscmVhZHkgdGFrZW4gYnkgdGhlIGxlZ2FjeVxuICogQ2xvdWRGcm9udFdlYkRpc3RyaWJ1dGlvbiBpbXBsZW1lbnRhdGlvbi5cbiAqL1xuZXhwb3J0IGNsYXNzIENhY2hlQmVoYXZpb3Ige1xuICAgIHByaXZhdGUgcmVhZG9ubHkgb3JpZ2luSWQ6IHN0cmluZztcbiAgICBjb25zdHJ1Y3RvcihvcmlnaW5JZDogc3RyaW5nLCBwcml2YXRlIHJlYWRvbmx5IHByb3BzOiBDYWNoZUJlaGF2aW9yUHJvcHMpIHtcbiAgICAgICAgdGhpcy5vcmlnaW5JZCA9IG9yaWdpbklkO1xuICAgICAgICB0aGlzLnZhbGlkYXRlRWRnZUxhbWJkYXMocHJvcHMuZWRnZUxhbWJkYXMpO1xuICAgICAgICB0aGlzLmdyYW50RWRnZUxhbWJkYUZ1bmN0aW9uRXhlY3V0aW9uUm9sZShwcm9wcy5lZGdlTGFtYmRhcyk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIENyZWF0ZXMgYW5kIHJldHVybnMgdGhlIENsb3VkRm9ybWF0aW9uIHJlcHJlc2VudGF0aW9uIG9mIHRoaXMgYmVoYXZpb3IuXG4gICAgICogVGhpcyByZW5kZXJzIGFzIGEgXCJDYWNoZUJlaGF2aW9yUHJvcGVydHlcIiByZWdhcmRsZXNzIG9mIGlmIHRoaXMgaXMgYSBkZWZhdWx0XG4gICAgICogY2FjaGUgYmVoYXZpb3Igb3Igbm90LCBhcyB0aGUgdHdvIGFyZSBpZGVudGljYWwgZXhjZXB0IHRoYXQgdGhlIHBhdGhQYXR0ZXJuXG4gICAgICogaXMgb21pdHRlZCBmb3IgdGhlIGRlZmF1bHQgY2FjaGUgYmVoYXZpb3IuXG4gICAgICpcbiAgICAgKiBAaW50ZXJuYWxcbiAgICAgKi9cbiAgICBwdWJsaWMgX3JlbmRlckJlaGF2aW9yKCk6IENmbkRpc3RyaWJ1dGlvbi5DYWNoZUJlaGF2aW9yUHJvcGVydHkge1xuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgICAgcGF0aFBhdHRlcm46IHRoaXMucHJvcHMucGF0aFBhdHRlcm4sXG4gICAgICAgICAgICB0YXJnZXRPcmlnaW5JZDogdGhpcy5vcmlnaW5JZCxcbiAgICAgICAgICAgIGFsbG93ZWRNZXRob2RzOiB0aGlzLnByb3BzLmFsbG93ZWRNZXRob2RzPy5tZXRob2RzLFxuICAgICAgICAgICAgY2FjaGVkTWV0aG9kczogdGhpcy5wcm9wcy5jYWNoZWRNZXRob2RzPy5tZXRob2RzLFxuICAgICAgICAgICAgY2FjaGVQb2xpY3lJZDogKHRoaXMucHJvcHMuY2FjaGVQb2xpY3kgPz8gQ2FjaGVQb2xpY3kuQ0FDSElOR19PUFRJTUlaRUQpLmNhY2hlUG9saWN5SWQsXG4gICAgICAgICAgICBjb21wcmVzczogdGhpcy5wcm9wcy5jb21wcmVzcyA/PyB0cnVlLFxuICAgICAgICAgICAgb3JpZ2luUmVxdWVzdFBvbGljeUlkOiB0aGlzLnByb3BzLm9yaWdpblJlcXVlc3RQb2xpY3k/Lm9yaWdpblJlcXVlc3RQb2xpY3lJZCxcbiAgICAgICAgICAgIHNtb290aFN0cmVhbWluZzogdGhpcy5wcm9wcy5zbW9vdGhTdHJlYW1pbmcsXG4gICAgICAgICAgICB2aWV3ZXJQcm90b2NvbFBvbGljeTogdGhpcy5wcm9wcy52aWV3ZXJQcm90b2NvbFBvbGljeSA/PyBWaWV3ZXJQcm90b2NvbFBvbGljeS5BTExPV19BTEwsXG4gICAgICAgICAgICBsYW1iZGFGdW5jdGlvbkFzc29jaWF0aW9uczogdGhpcy5wcm9wcy5lZGdlTGFtYmRhcz8ubWFwKGVkZ2VMYW1iZGEgPT4gKHtcbiAgICAgICAgICAgICAgICBsYW1iZGFGdW5jdGlvbkFybjogZWRnZUxhbWJkYS5mdW5jdGlvblZlcnNpb24uZWRnZUFybixcbiAgICAgICAgICAgICAgICBldmVudFR5cGU6IGVkZ2VMYW1iZGEuZXZlbnRUeXBlLnRvU3RyaW5nKCksXG4gICAgICAgICAgICAgICAgaW5jbHVkZUJvZHk6IGVkZ2VMYW1iZGEuaW5jbHVkZUJvZHksXG4gICAgICAgICAgICB9KSksXG4gICAgICAgICAgICB0cnVzdGVkS2V5R3JvdXBzOiB0aGlzLnByb3BzLnRydXN0ZWRLZXlHcm91cHM/Lm1hcChrZXlHcm91cCA9PiBrZXlHcm91cC5rZXlHcm91cElkKSxcbiAgICAgICAgfTtcbiAgICB9XG4gICAgcHJpdmF0ZSB2YWxpZGF0ZUVkZ2VMYW1iZGFzKGVkZ2VMYW1iZGFzPzogRWRnZUxhbWJkYVtdKSB7XG4gICAgICAgIGNvbnN0IGluY2x1ZGVCb2R5RXZlbnRUeXBlcyA9IFtMYW1iZGFFZGdlRXZlbnRUeXBlLk9SSUdJTl9SRVFVRVNULCBMYW1iZGFFZGdlRXZlbnRUeXBlLlZJRVdFUl9SRVFVRVNUXTtcbiAgICAgICAgaWYgKGVkZ2VMYW1iZGFzICYmIGVkZ2VMYW1iZGFzLnNvbWUobGFtYmRhID0+IGxhbWJkYS5pbmNsdWRlQm9keSAmJiAhaW5jbHVkZUJvZHlFdmVudFR5cGVzLmluY2x1ZGVzKGxhbWJkYS5ldmVudFR5cGUpKSkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdcXCdpbmNsdWRlQm9keVxcJyBjYW4gb25seSBiZSB0cnVlIGZvciBPUklHSU5fUkVRVUVTVCBvciBWSUVXRVJfUkVRVUVTVCBldmVudCB0eXBlcy4nKTtcbiAgICAgICAgfVxuICAgIH1cbiAgICBwcml2YXRlIGdyYW50RWRnZUxhbWJkYUZ1bmN0aW9uRXhlY3V0aW9uUm9sZShlZGdlTGFtYmRhcz86IEVkZ2VMYW1iZGFbXSkge1xuICAgICAgICBpZiAoIWVkZ2VMYW1iZGFzIHx8IGVkZ2VMYW1iZGFzLmxlbmd0aCA9PT0gMCkge1xuICAgICAgICAgICAgcmV0dXJuO1xuICAgICAgICB9XG4gICAgICAgIGVkZ2VMYW1iZGFzLmZvckVhY2goKGVkZ2VMYW1iZGEpID0+IHtcbiAgICAgICAgICAgIGNvbnN0IHJvbGUgPSBlZGdlTGFtYmRhLmZ1bmN0aW9uVmVyc2lvbi5yb2xlO1xuICAgICAgICAgICAgaWYgKHJvbGUgJiYgcm9sZSBpbnN0YW5jZW9mIGlhbS5Sb2xlICYmIHJvbGUuYXNzdW1lUm9sZVBvbGljeSkge1xuICAgICAgICAgICAgICAgIHJvbGUuYXNzdW1lUm9sZVBvbGljeS5hZGRTdGF0ZW1lbnRzKG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICAgICAgICAgICAgYWN0aW9uczogWydzdHM6QXNzdW1lUm9sZSddLFxuICAgICAgICAgICAgICAgICAgICBwcmluY2lwYWxzOiBbbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdlZGdlbGFtYmRhLmFtYXpvbmF3cy5jb20nKV0sXG4gICAgICAgICAgICAgICAgfSkpO1xuICAgICAgICAgICAgfVxuICAgICAgICB9KTtcbiAgICB9XG59XG4iXX0=