"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EdgeFunction = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const lambda = require("../../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const ssm = require("../../../aws-ssm"); // Automatically re-written from '@aws-cdk/aws-ssm'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * (experimental) A Lambda@Edge function.
 *
 * Convenience resource for requesting a Lambda function in the 'us-east-1' region for use with Lambda@Edge.
 * Implements several restrictions enforced by Lambda@Edge.
 *
 * Note that this construct requires that the 'us-east-1' region has been bootstrapped.
 * See https://docs.aws.amazon.com/cdk/latest/guide/bootstrapping.html or 'cdk bootstrap --help' for options.
 *
 * @experimental
 * @resource AWS::Lambda::Function
 */
class EdgeFunction extends core_1.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * (experimental) Whether or not this Lambda function was bound to a VPC.
         *
         * If this is is `false`, trying to access the `connections` object will fail.
         *
         * @experimental
         */
        this.isBoundToVpc = false;
        // Create a simple Function if we're already in us-east-1; otherwise create a cross-region stack.
        const regionIsUsEast1 = !core_1.Token.isUnresolved(this.stack.region) && this.stack.region === 'us-east-1';
        const { edgeFunction, edgeArn } = regionIsUsEast1
            ? this.createInRegionFunction(props)
            : this.createCrossRegionFunction(id, props);
        this.edgeArn = edgeArn;
        this.functionArn = edgeArn;
        this._edgeFunction = edgeFunction;
        this.functionName = this._edgeFunction.functionName;
        this.grantPrincipal = this._edgeFunction.role;
        this.permissionsNode = this._edgeFunction.permissionsNode;
        this.version = lambda.extractQualifierFromArn(this.functionArn);
        this.node.defaultChild = this._edgeFunction;
    }
    /**
     * (experimental) The underlying AWS Lambda function.
     *
     * @experimental
     */
    get lambda() {
        return this._edgeFunction;
    }
    /**
     * (experimental) Convenience method to make `EdgeFunction` conform to the same interface as `Function`.
     *
     * @experimental
     */
    get currentVersion() {
        return this;
    }
    /**
     * (experimental) Defines an alias for this version.
     *
     * @experimental
     */
    addAlias(aliasName, options = {}) {
        return new lambda.Alias(this._edgeFunction, `Alias${aliasName}`, {
            aliasName,
            version: this._edgeFunction.currentVersion,
            ...options,
        });
    }
    /**
     * (experimental) Not supported.
     *
     * Connections are only applicable to VPC-enabled functions.
     *
     * @experimental
     */
    get connections() {
        throw new Error('Lambda@Edge does not support connections');
    }
    /**
     * (experimental) The `$LATEST` version of this function.
     *
     * Note that this is reference to a non-specific AWS Lambda version, which
     * means the function this version refers to can return different results in
     * different invocations.
     *
     * To obtain a reference to an explicit version which references the current
     * function configuration, use `lambdaFunction.currentVersion` instead.
     *
     * @experimental
     */
    get latestVersion() {
        throw new Error('$LATEST function version cannot be used for Lambda@Edge');
    }
    /**
     * (experimental) Adds an event source that maps to this AWS Lambda function.
     *
     * @experimental
     */
    addEventSourceMapping(id, options) {
        return this.lambda.addEventSourceMapping(id, options);
    }
    /**
     * (experimental) Adds a permission to the Lambda resource policy.
     *
     * @experimental
     */
    addPermission(id, permission) {
        return this.lambda.addPermission(id, permission);
    }
    /**
     * (experimental) Adds a statement to the IAM role assumed by the instance.
     *
     * @experimental
     */
    addToRolePolicy(statement) {
        return this.lambda.addToRolePolicy(statement);
    }
    /**
     * (experimental) Grant the given identity permissions to invoke this Lambda.
     *
     * @experimental
     */
    grantInvoke(identity) {
        return this.lambda.grantInvoke(identity);
    }
    /**
     * (experimental) Return the given named metric for this Lambda Return the given named metric for this Function.
     *
     * @experimental
     */
    metric(metricName, props) {
        return this.lambda.metric(metricName, { ...props, region: EdgeFunction.EDGE_REGION });
    }
    /**
     * (experimental) Metric for the Duration of this Lambda How long execution of this Lambda takes.
     *
     * Average over 5 minutes
     *
     * @experimental
     */
    metricDuration(props) {
        return this.lambda.metricDuration({ ...props, region: EdgeFunction.EDGE_REGION });
    }
    /**
     * (experimental) How many invocations of this Lambda fail.
     *
     * Sum over 5 minutes
     *
     * @experimental
     */
    metricErrors(props) {
        return this.lambda.metricErrors({ ...props, region: EdgeFunction.EDGE_REGION });
    }
    /**
     * (experimental) Metric for the number of invocations of this Lambda How often this Lambda is invoked.
     *
     * Sum over 5 minutes
     *
     * @experimental
     */
    metricInvocations(props) {
        return this.lambda.metricInvocations({ ...props, region: EdgeFunction.EDGE_REGION });
    }
    /**
     * (experimental) Metric for the number of throttled invocations of this Lambda How often this Lambda is throttled.
     *
     * Sum over 5 minutes
     *
     * @experimental
     */
    metricThrottles(props) {
        return this.lambda.metricThrottles({ ...props, region: EdgeFunction.EDGE_REGION });
    }
    /**
     * (experimental) Adds an event source to this function.
     *
     * @experimental
     */
    addEventSource(source) {
        return this.lambda.addEventSource(source);
    }
    /**
     * (experimental) Configures options for asynchronous invocation.
     *
     * @experimental
     */
    configureAsyncInvoke(options) {
        return this.lambda.configureAsyncInvoke(options);
    }
    /** Create a function in-region */
    createInRegionFunction(props) {
        const edgeFunction = new lambda.Function(this, 'Fn', props);
        addEdgeLambdaToRoleTrustStatement(edgeFunction.role);
        return { edgeFunction, edgeArn: edgeFunction.currentVersion.edgeArn };
    }
    /** Create a support stack and function in us-east-1, and a SSM reader in-region */
    createCrossRegionFunction(id, props) {
        const parameterNamePrefix = '/cdk/EdgeFunctionArn';
        if (core_1.Token.isUnresolved(this.env.region)) {
            throw new Error('stacks which use EdgeFunctions must have an explicitly set region');
        }
        const parameterName = `${parameterNamePrefix}/${this.env.region}/${this.node.path}`;
        const functionStack = this.edgeStack(props.stackId);
        const edgeFunction = new lambda.Function(functionStack, id, props);
        addEdgeLambdaToRoleTrustStatement(edgeFunction.role);
        // Store the current version's ARN to be retrieved by the cross region reader below.
        const version = edgeFunction.currentVersion;
        new ssm.StringParameter(edgeFunction, 'Parameter', {
            parameterName,
            stringValue: version.edgeArn,
        });
        const edgeArn = this.createCrossRegionArnReader(parameterNamePrefix, parameterName, version);
        return { edgeFunction, edgeArn };
    }
    createCrossRegionArnReader(parameterNamePrefix, parameterName, version) {
        // Prefix of the parameter ARN that applies to all EdgeFunctions.
        // This is necessary because the `CustomResourceProvider` is a singleton, and the `policyStatement`
        // must work for multiple EdgeFunctions.
        const parameterArnPrefix = this.stack.formatArn({
            service: 'ssm',
            region: EdgeFunction.EDGE_REGION,
            resource: 'parameter',
            resourceName: parameterNamePrefix + '/*',
            sep: '',
        });
        const resourceType = 'Custom::CrossRegionStringParameterReader';
        const serviceToken = core_1.CustomResourceProvider.getOrCreate(this, resourceType, {
            codeDirectory: path.join(__dirname, 'edge-function'),
            runtime: core_1.CustomResourceProviderRuntime.NODEJS_12_X,
            policyStatements: [{
                    Effect: 'Allow',
                    Resource: parameterArnPrefix,
                    Action: ['ssm:GetParameter'],
                }],
        });
        const resource = new core_1.CustomResource(this, 'ArnReader', {
            resourceType: resourceType,
            serviceToken,
            properties: {
                Region: EdgeFunction.EDGE_REGION,
                ParameterName: parameterName,
                // This is used to determine when the function has changed, to refresh the ARN from the custom resource.
                //
                // Use the logical id of the function version. Whenever a function version changes, the logical id must be
                // changed for it to take effect - a good candidate for RefreshToken.
                RefreshToken: core_1.Lazy.uncachedString({
                    produce: () => {
                        const cfn = version.node.defaultChild;
                        return this.stack.resolve(cfn.logicalId);
                    },
                }),
            },
        });
        return resource.getAttString('FunctionArn');
    }
    edgeStack(stackId) {
        const stage = core_1.Stage.of(this);
        if (!stage) {
            throw new Error('stacks which use EdgeFunctions must be part of a CDK app or stage');
        }
        const edgeStackId = stackId !== null && stackId !== void 0 ? stackId : `edge-lambda-stack-${this.stack.node.addr}`;
        let edgeStack = stage.node.tryFindChild(edgeStackId);
        if (!edgeStack) {
            edgeStack = new core_1.Stack(stage, edgeStackId, {
                env: {
                    region: EdgeFunction.EDGE_REGION,
                    account: core_1.Stack.of(this).account,
                },
            });
        }
        this.stack.addDependency(edgeStack);
        return edgeStack;
    }
}
exports.EdgeFunction = EdgeFunction;
_a = JSII_RTTI_SYMBOL_1;
EdgeFunction[_a] = { fqn: "monocdk.aws_cloudfront.experimental.EdgeFunction", version: "1.106.1" };
EdgeFunction.EDGE_REGION = 'us-east-1';
function addEdgeLambdaToRoleTrustStatement(role) {
    if (role instanceof iam.Role && role.assumeRolePolicy) {
        const statement = new iam.PolicyStatement();
        const edgeLambdaServicePrincipal = new iam.ServicePrincipal('edgelambda.amazonaws.com');
        statement.addPrincipals(edgeLambdaServicePrincipal);
        statement.addActions(edgeLambdaServicePrincipal.assumeRoleAction);
        role.assumeRolePolicy.addStatements(statement);
    }
}
//# sourceMappingURL=data:application/json;base64,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