"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SlackChannelConfiguration = exports.LoggingLevel = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const logs = require("../../aws-logs"); // Automatically re-written from '@aws-cdk/aws-logs'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const chatbot_generated_1 = require("./chatbot.generated");
/**
 * (experimental) Logging levels include ERROR, INFO, or NONE.
 *
 * @experimental
 */
var LoggingLevel;
(function (LoggingLevel) {
    LoggingLevel["ERROR"] = "ERROR";
    LoggingLevel["INFO"] = "INFO";
    LoggingLevel["NONE"] = "NONE";
})(LoggingLevel = exports.LoggingLevel || (exports.LoggingLevel = {}));
/**
 * Either a new or imported Slack channel configuration
 */
class SlackChannelConfigurationBase extends cdk.Resource {
    /**
     * (experimental) Adds extra permission to iam-role of Slack channel configuration.
     *
     * @experimental
     */
    addToRolePolicy(statement) {
        if (!this.role) {
            return;
        }
        this.role.addToPrincipalPolicy(statement);
    }
    /**
     * (experimental) Return the given named metric for this SlackChannelConfiguration.
     *
     * @experimental
     */
    metric(metricName, props) {
        // AWS Chatbot publishes metrics to us-east-1 regardless of stack region
        // https://docs.aws.amazon.com/chatbot/latest/adminguide/monitoring-cloudwatch.html
        return new cloudwatch.Metric({
            namespace: 'AWS/Chatbot',
            region: 'us-east-1',
            dimensions: {
                ConfigurationName: this.slackChannelConfigurationName,
            },
            metricName,
            ...props,
        });
    }
}
/**
 * (experimental) A new Slack channel configuration.
 *
 * @experimental
 */
class SlackChannelConfiguration extends SlackChannelConfigurationBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id, {
            physicalName: props.slackChannelConfigurationName,
        });
        this.role = props.role || new iam.Role(this, 'ConfigurationRole', {
            assumedBy: new iam.ServicePrincipal('chatbot.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        const configuration = new chatbot_generated_1.CfnSlackChannelConfiguration(this, 'Resource', {
            configurationName: props.slackChannelConfigurationName,
            iamRoleArn: this.role.roleArn,
            slackWorkspaceId: props.slackWorkspaceId,
            slackChannelId: props.slackChannelId,
            snsTopicArns: (_b = props.notificationTopics) === null || _b === void 0 ? void 0 : _b.map(topic => topic.topicArn),
            loggingLevel: (_c = props.loggingLevel) === null || _c === void 0 ? void 0 : _c.toString(),
        });
        // Log retention
        // AWS Chatbot publishes logs to us-east-1 regardless of stack region https://docs.aws.amazon.com/chatbot/latest/adminguide/cloudwatch-logs.html
        if (props.logRetention) {
            new logs.LogRetention(this, 'LogRetention', {
                logGroupName: `/aws/chatbot/${props.slackChannelConfigurationName}`,
                retention: props.logRetention,
                role: props.logRetentionRole,
                logGroupRegion: 'us-east-1',
                logRetentionRetryOptions: props.logRetentionRetryOptions,
            });
        }
        this.slackChannelConfigurationArn = configuration.ref;
        this.slackChannelConfigurationName = props.slackChannelConfigurationName;
    }
    /**
     * (experimental) Import an existing Slack channel configuration provided an ARN.
     *
     * @param scope The parent creating construct.
     * @param id The construct's name.
     * @param slackChannelConfigurationArn configuration ARN (i.e. arn:aws:chatbot::1234567890:chat-configuration/slack-channel/my-slack).
     * @returns a reference to the existing Slack channel configuration
     * @experimental
     */
    static fromSlackChannelConfigurationArn(scope, id, slackChannelConfigurationArn) {
        const re = /^slack-channel\//;
        const resourceName = cdk.Stack.of(scope).parseArn(slackChannelConfigurationArn).resourceName;
        if (!re.test(resourceName)) {
            throw new Error('The ARN of a Slack integration must be in the form: arn:aws:chatbot:{region}:{account}:chat-configuration/slack-channel/{slackChannelName}');
        }
        class Import extends SlackChannelConfigurationBase {
            constructor(s, i) {
                super(s, i);
                /**
                 * @attribute
                 */
                this.slackChannelConfigurationArn = slackChannelConfigurationArn;
                this.role = undefined;
                /**
                 * Returns a name of Slack channel configuration
                 *
                 * NOTE:
                 * For example: arn:aws:chatbot::1234567890:chat-configuration/slack-channel/my-slack
                 * The ArnComponents API will return `slack-channel/my-slack`
                 * It need to handle that to gets a correct name.`my-slack`
                 */
                this.slackChannelConfigurationName = resourceName.substring('slack-channel/'.length);
                this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
            }
        }
        return new Import(scope, id);
    }
    /**
     * (experimental) Return the given named metric for All SlackChannelConfigurations.
     *
     * @experimental
     */
    static metricAll(metricName, props) {
        // AWS Chatbot publishes metrics to us-east-1 regardless of stack region
        // https://docs.aws.amazon.com/chatbot/latest/adminguide/monitoring-cloudwatch.html
        return new cloudwatch.Metric({
            namespace: 'AWS/Chatbot',
            region: 'us-east-1',
            metricName,
            ...props,
        });
    }
}
exports.SlackChannelConfiguration = SlackChannelConfiguration;
_a = JSII_RTTI_SYMBOL_1;
SlackChannelConfiguration[_a] = { fqn: "monocdk.aws_chatbot.SlackChannelConfiguration", version: "1.106.1" };
//# sourceMappingURL=data:application/json;base64,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