"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ValidationMethod = exports.Certificate = exports.CertificateValidation = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const certificatemanager_generated_1 = require("./certificatemanager.generated");
const util_1 = require("./util");
/**
 * (experimental) How to validate a certificate.
 *
 * @experimental
 */
class CertificateValidation {
    /** @param props Certification validation properties */
    constructor(props) {
        var _c;
        this.props = props;
        this.method = (_c = props.method) !== null && _c !== void 0 ? _c : ValidationMethod.EMAIL;
    }
    /**
     * (experimental) Validate the certificate with DNS.
     *
     * IMPORTANT: If `hostedZone` is not specified, DNS records must be added
     * manually and the stack will not complete creating until the records are
     * added.
     *
     * @param hostedZone the hosted zone where DNS records must be created.
     * @experimental
     */
    static fromDns(hostedZone) {
        return new CertificateValidation({
            method: ValidationMethod.DNS,
            hostedZone,
        });
    }
    /**
     * (experimental) Validate the certificate with automatically created DNS records in multiple Amazon Route 53 hosted zones.
     *
     * @param hostedZones a map of hosted zones where DNS records must be created for the domains in the certificate.
     * @experimental
     */
    static fromDnsMultiZone(hostedZones) {
        return new CertificateValidation({
            method: ValidationMethod.DNS,
            hostedZones,
        });
    }
    /**
     * (experimental) Validate the certificate with Email.
     *
     * IMPORTANT: if you are creating a certificate as part of your stack, the stack
     * will not complete creating until you read and follow the instructions in the
     * email that you will receive.
     *
     * ACM will send validation emails to the following addresses:
     *
     *   admin@domain.com
     *   administrator@domain.com
     *   hostmaster@domain.com
     *   postmaster@domain.com
     *   webmaster@domain.com
     *
     * For every domain that you register.
     *
     * @param validationDomains a map of validation domains to use for domains in the certificate.
     * @experimental
     */
    static fromEmail(validationDomains) {
        return new CertificateValidation({
            method: ValidationMethod.EMAIL,
            validationDomains,
        });
    }
}
exports.CertificateValidation = CertificateValidation;
_a = JSII_RTTI_SYMBOL_1;
CertificateValidation[_a] = { fqn: "monocdk.aws_certificatemanager.CertificateValidation", version: "1.106.1" };
/**
 * (experimental) A certificate managed by AWS Certificate Manager.
 *
 * @experimental
 */
class Certificate extends core_1.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        let validation;
        if (props.validation) {
            validation = props.validation;
        }
        else { // Deprecated props
            if (props.validationMethod === ValidationMethod.DNS) {
                validation = CertificateValidation.fromDns();
            }
            else {
                validation = CertificateValidation.fromEmail(props.validationDomains);
            }
        }
        const allDomainNames = [props.domainName].concat(props.subjectAlternativeNames || []);
        const cert = new certificatemanager_generated_1.CfnCertificate(this, 'Resource', {
            domainName: props.domainName,
            subjectAlternativeNames: props.subjectAlternativeNames,
            domainValidationOptions: renderDomainValidation(validation, allDomainNames),
            validationMethod: validation.method,
        });
        this.certificateArn = cert.ref;
    }
    /**
     * (experimental) Import a certificate.
     *
     * @experimental
     */
    static fromCertificateArn(scope, id, certificateArn) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.certificateArn = certificateArn;
            }
        }
        return new Import(scope, id);
    }
}
exports.Certificate = Certificate;
_b = JSII_RTTI_SYMBOL_1;
Certificate[_b] = { fqn: "monocdk.aws_certificatemanager.Certificate", version: "1.106.1" };
/**
 * (experimental) Method used to assert ownership of the domain.
 *
 * @experimental
 */
var ValidationMethod;
(function (ValidationMethod) {
    ValidationMethod["EMAIL"] = "EMAIL";
    ValidationMethod["DNS"] = "DNS";
})(ValidationMethod = exports.ValidationMethod || (exports.ValidationMethod = {}));
// eslint-disable-next-line max-len
function renderDomainValidation(validation, domainNames) {
    var _c, _d, _e;
    const domainValidation = [];
    switch (validation.method) {
        case ValidationMethod.DNS:
            for (const domainName of getUniqueDnsDomainNames(domainNames)) {
                const hostedZone = (_d = (_c = validation.props.hostedZones) === null || _c === void 0 ? void 0 : _c[domainName]) !== null && _d !== void 0 ? _d : validation.props.hostedZone;
                if (hostedZone) {
                    domainValidation.push({ domainName, hostedZoneId: hostedZone.hostedZoneId });
                }
            }
            break;
        case ValidationMethod.EMAIL:
            for (const domainName of domainNames) {
                const validationDomain = (_e = validation.props.validationDomains) === null || _e === void 0 ? void 0 : _e[domainName];
                if (!validationDomain && core_1.Token.isUnresolved(domainName)) {
                    throw new Error('When using Tokens for domain names, \'validationDomains\' needs to be supplied');
                }
                domainValidation.push({ domainName, validationDomain: validationDomain !== null && validationDomain !== void 0 ? validationDomain : util_1.apexDomain(domainName) });
            }
            break;
        default:
            throw new Error(`Unknown validation method ${validation.method}`);
    }
    return domainValidation.length !== 0 ? domainValidation : undefined;
}
/**
 * Removes wildcard domains (*.example.com) where the base domain (example.com) is present.
 * This is because the DNS validation treats them as the same thing, and the automated CloudFormation
 * DNS validation errors out with the duplicate records.
 */
function getUniqueDnsDomainNames(domainNames) {
    return domainNames.filter(domain => {
        return core_1.Token.isUnresolved(domain) || !domain.startsWith('*.') || !domainNames.includes(domain.replace('*.', ''));
    });
}
//# sourceMappingURL=data:application/json;base64,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