"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ComputeEnvironment = exports.AllocationStrategy = exports.ComputeResourceType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const batch_generated_1 = require("./batch.generated");
/**
 * (experimental) Property to specify if the compute environment uses On-Demand or SpotFleet compute resources.
 *
 * @experimental
 */
var ComputeResourceType;
(function (ComputeResourceType) {
    ComputeResourceType["ON_DEMAND"] = "EC2";
    ComputeResourceType["SPOT"] = "SPOT";
})(ComputeResourceType = exports.ComputeResourceType || (exports.ComputeResourceType = {}));
/**
 * (experimental) Properties for how to prepare compute resources that are provisioned for a compute environment.
 *
 * @experimental
 */
var AllocationStrategy;
(function (AllocationStrategy) {
    AllocationStrategy["BEST_FIT"] = "BEST_FIT";
    AllocationStrategy["BEST_FIT_PROGRESSIVE"] = "BEST_FIT_PROGRESSIVE";
    AllocationStrategy["SPOT_CAPACITY_OPTIMIZED"] = "SPOT_CAPACITY_OPTIMIZED";
})(AllocationStrategy = exports.AllocationStrategy || (exports.AllocationStrategy = {}));
/**
 * (experimental) Batch Compute Environment.
 *
 * Defines a batch compute environment to run batch jobs on.
 *
 * @experimental
 */
class ComputeEnvironment extends core_1.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props = { enabled: true, managed: true }) {
        var _b, _c;
        super(scope, id, {
            physicalName: props.computeEnvironmentName,
        });
        this.validateProps(props);
        const spotFleetRole = this.getSpotFleetRole(props);
        let computeResources;
        // Only allow compute resources to be set when using MANAGED type
        if (props.computeResources && this.isManaged(props)) {
            computeResources = {
                allocationStrategy: props.computeResources.allocationStrategy
                    || (props.computeResources.type === ComputeResourceType.SPOT
                        ? AllocationStrategy.SPOT_CAPACITY_OPTIMIZED
                        : AllocationStrategy.BEST_FIT),
                bidPercentage: props.computeResources.bidPercentage,
                desiredvCpus: props.computeResources.desiredvCpus,
                ec2KeyPair: props.computeResources.ec2KeyPair,
                imageId: props.computeResources.image && props.computeResources.image.getImage(this).imageId,
                instanceRole: props.computeResources.instanceRole
                    ? props.computeResources.instanceRole
                    : new iam.CfnInstanceProfile(this, 'Instance-Profile', {
                        roles: [new iam.Role(this, 'Ecs-Instance-Role', {
                                assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
                                managedPolicies: [
                                    iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2ContainerServiceforEC2Role'),
                                ],
                            }).roleName],
                    }).attrArn,
                instanceTypes: this.buildInstanceTypes(props.computeResources.instanceTypes),
                launchTemplate: props.computeResources.launchTemplate,
                maxvCpus: props.computeResources.maxvCpus || 256,
                minvCpus: props.computeResources.minvCpus || 0,
                placementGroup: props.computeResources.placementGroup,
                securityGroupIds: this.buildSecurityGroupIds(props.computeResources.vpc, props.computeResources.securityGroups),
                spotIamFleetRole: spotFleetRole === null || spotFleetRole === void 0 ? void 0 : spotFleetRole.roleArn,
                subnets: props.computeResources.vpc.selectSubnets(props.computeResources.vpcSubnets).subnetIds,
                tags: props.computeResources.computeResourcesTags,
                type: props.computeResources.type || ComputeResourceType.ON_DEMAND,
            };
        }
        const computeEnvironment = new batch_generated_1.CfnComputeEnvironment(this, 'Resource', {
            computeEnvironmentName: this.physicalName,
            computeResources,
            serviceRole: (_c = (_b = props.serviceRole) === null || _b === void 0 ? void 0 : _b.roleArn) !== null && _c !== void 0 ? _c : new iam.Role(this, 'Resource-Service-Instance-Role', {
                managedPolicies: [
                    iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSBatchServiceRole'),
                ],
                assumedBy: new iam.ServicePrincipal('batch.amazonaws.com'),
            }).roleArn,
            state: props.enabled === undefined ? 'ENABLED' : (props.enabled ? 'ENABLED' : 'DISABLED'),
            type: this.isManaged(props) ? 'MANAGED' : 'UNMANAGED',
        });
        if (props.computeResources && props.computeResources.vpc) {
            this.node.addDependency(props.computeResources.vpc);
        }
        this.computeEnvironmentArn = this.getResourceArnAttribute(computeEnvironment.ref, {
            service: 'batch',
            resource: 'compute-environment',
            resourceName: this.physicalName,
        });
        this.computeEnvironmentName = this.getResourceNameAttribute(computeEnvironment.ref);
    }
    /**
     * (experimental) Fetches an existing batch compute environment by its amazon resource name.
     *
     * @experimental
     */
    static fromComputeEnvironmentArn(scope, id, computeEnvironmentArn) {
        const stack = core_1.Stack.of(scope);
        const computeEnvironmentName = stack.parseArn(computeEnvironmentArn).resourceName;
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.computeEnvironmentArn = computeEnvironmentArn;
                this.computeEnvironmentName = computeEnvironmentName;
            }
        }
        return new Import(scope, id);
    }
    isManaged(props) {
        var _b;
        return (_b = props.managed) !== null && _b !== void 0 ? _b : true;
    }
    /**
     * Validates the properties provided for a new batch compute environment.
     */
    validateProps(props) {
        if (props === undefined) {
            return;
        }
        if (!this.isManaged(props) && props.computeResources !== undefined) {
            throw new Error('It is not allowed to set computeResources on an AWS unmanaged compute environment');
        }
        if (this.isManaged(props) && props.computeResources === undefined) {
            throw new Error('computeResources is missing but required on a managed compute environment');
        }
        // Setting a bid percentage is only allowed on SPOT resources +
        // Cannot use SPOT_CAPACITY_OPTIMIZED when using ON_DEMAND
        if (props.computeResources) {
            if (props.computeResources.type === ComputeResourceType.ON_DEMAND) {
                // VALIDATE FOR ON_DEMAND
                // Bid percentage is not allowed
                if (props.computeResources.bidPercentage !== undefined) {
                    throw new Error('Setting the bid percentage is only allowed for SPOT type resources on a batch compute environment');
                }
                // SPOT_CAPACITY_OPTIMIZED allocation is not allowed
                if (props.computeResources.allocationStrategy && props.computeResources.allocationStrategy === AllocationStrategy.SPOT_CAPACITY_OPTIMIZED) {
                    throw new Error('The SPOT_CAPACITY_OPTIMIZED allocation strategy is only allowed if the environment is a SPOT type compute environment');
                }
            }
            else {
                // VALIDATE FOR SPOT
                // Bid percentage must be from 0 - 100
                if (props.computeResources.bidPercentage !== undefined &&
                    (props.computeResources.bidPercentage < 0 || props.computeResources.bidPercentage > 100)) {
                    throw new Error('Bid percentage can only be a value between 0 and 100');
                }
            }
            if (props.computeResources.minvCpus) {
                // minvCpus cannot be less than 0
                if (props.computeResources.minvCpus < 0) {
                    throw new Error('Minimum vCpus for a batch compute environment cannot be less than 0');
                }
                // minvCpus cannot exceed max vCpus
                if (props.computeResources.maxvCpus &&
                    props.computeResources.minvCpus > props.computeResources.maxvCpus) {
                    throw new Error('Minimum vCpus cannot be greater than the maximum vCpus');
                }
            }
        }
    }
    buildInstanceTypes(instanceTypes) {
        if (instanceTypes === undefined) {
            return [
                'optimal',
            ];
        }
        return instanceTypes.map((type) => type.toString());
    }
    buildSecurityGroupIds(vpc, securityGroups) {
        if (securityGroups === undefined) {
            return [
                new ec2.SecurityGroup(this, 'Resource-Security-Group', { vpc }).securityGroupId,
            ];
        }
        return securityGroups.map((group) => group.securityGroupId);
    }
    /**
     * Generates an AWS IAM role for provisioning spotfleet resources
     * if the allocation strategy is set to BEST_FIT or not defined.
     *
     * @param props - the compute environment construct properties
     */
    getSpotFleetRole(props) {
        var _b;
        if (((_b = props.computeResources) === null || _b === void 0 ? void 0 : _b.allocationStrategy) && props.computeResources.allocationStrategy !== AllocationStrategy.BEST_FIT) {
            return undefined;
        }
        if (props.computeResources) {
            if (props.computeResources.spotFleetRole) {
                return props.computeResources.spotFleetRole;
            }
            else if (props.computeResources.type === ComputeResourceType.SPOT) {
                return iam.Role.fromRoleArn(this, 'Resource-SpotFleet-Role', `arn:${this.stack.partition}:iam::${this.stack.account}:role/aws-service-role/spotfleet.amazonaws.com/AWSServiceRoleForEC2SpotFleet`);
            }
        }
        return undefined;
    }
}
exports.ComputeEnvironment = ComputeEnvironment;
_a = JSII_RTTI_SYMBOL_1;
ComputeEnvironment[_a] = { fqn: "monocdk.aws_batch.ComputeEnvironment", version: "1.106.1" };
//# sourceMappingURL=data:application/json;base64,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