"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.JsonSchemaMapper = exports.validateInteger = exports.parseAwsApiCall = exports.parseMethodOptionsPath = exports.validateHttpMethod = exports.ALL_METHODS = void 0;
const url_1 = require("url");
const jsonSchema = require("./json-schema");
exports.ALL_METHODS = ['OPTIONS', 'GET', 'PUT', 'POST', 'DELETE', 'PATCH', 'HEAD'];
const ALLOWED_METHODS = ['ANY', ...exports.ALL_METHODS];
function validateHttpMethod(method, messagePrefix = '') {
    if (!ALLOWED_METHODS.includes(method)) {
        throw new Error(`${messagePrefix}Invalid HTTP method "${method}". Allowed methods: ${ALLOWED_METHODS.join(',')}`);
    }
}
exports.validateHttpMethod = validateHttpMethod;
function parseMethodOptionsPath(originalPath) {
    if (!originalPath.startsWith('/')) {
        throw new Error(`Method options path must start with '/': ${originalPath}`);
    }
    const path = originalPath.substr(1); // trim trailing '/'
    const components = path.split('/');
    if (components.length < 2) {
        throw new Error(`Method options path must include at least two components: /{resource}/{method} (i.e. /foo/bar/GET): ${path}`);
    }
    const httpMethod = components.pop().toUpperCase(); // last component is an HTTP method
    if (httpMethod !== '*') {
        validateHttpMethod(httpMethod, `${originalPath}: `);
    }
    let resourcePath = '/~1' + components.join('~1');
    if (components.length === 1 && components[0] === '*') {
        resourcePath = '/*';
    }
    else if (components.length === 1 && components[0] === '') {
        resourcePath = '/';
    }
    return {
        httpMethod,
        resourcePath,
    };
}
exports.parseMethodOptionsPath = parseMethodOptionsPath;
function parseAwsApiCall(path, action, actionParams) {
    if (actionParams && !action) {
        throw new Error('"actionParams" requires that "action" will be set');
    }
    if (path && action) {
        throw new Error(`"path" and "action" are mutually exclusive (path="${path}", action="${action}")`);
    }
    if (path) {
        return {
            apiType: 'path',
            apiValue: path,
        };
    }
    if (action) {
        if (actionParams) {
            action += '&' + url_1.format({ query: actionParams }).substr(1);
        }
        return {
            apiType: 'action',
            apiValue: action,
        };
    }
    throw new Error('Either "path" or "action" are required');
}
exports.parseAwsApiCall = parseAwsApiCall;
function validateInteger(property, messagePrefix) {
    if (property && !Number.isInteger(property)) {
        throw new Error(`${messagePrefix} should be an integer`);
    }
}
exports.validateInteger = validateInteger;
class JsonSchemaMapper {
    /**
     * Transforms naming of some properties to prefix with a $, where needed
     * according to the JSON schema spec
     * @param schema The JsonSchema object to transform for CloudFormation output
     */
    static toCfnJsonSchema(schema) {
        const result = JsonSchemaMapper._toCfnJsonSchema(schema);
        if (!('$schema' in result)) {
            result.$schema = jsonSchema.JsonSchemaVersion.DRAFT4;
        }
        return result;
    }
    static _toCfnJsonSchema(schema, preserveKeys = false) {
        if (schema == null || typeof schema !== 'object') {
            return schema;
        }
        if (Array.isArray(schema)) {
            return schema.map(entry => JsonSchemaMapper._toCfnJsonSchema(entry));
        }
        return Object.assign({}, ...Object.entries(schema).map(([key, value]) => {
            const mapKey = !preserveKeys && (key in JsonSchemaMapper.SchemaPropsWithPrefix);
            const newKey = mapKey ? JsonSchemaMapper.SchemaPropsWithPrefix[key] : key;
            // If keys were preserved, don't consider SchemaPropsWithUserDefinedChildren for those keys (they are user-defined!)
            const newValue = JsonSchemaMapper._toCfnJsonSchema(value, !preserveKeys && JsonSchemaMapper.SchemaPropsWithUserDefinedChildren[key]);
            return { [newKey]: newValue };
        }));
    }
}
exports.JsonSchemaMapper = JsonSchemaMapper;
JsonSchemaMapper.SchemaPropsWithPrefix = {
    schema: '$schema',
    ref: '$ref',
    id: '$id',
};
// The value indicates whether direct children should be key-mapped.
JsonSchemaMapper.SchemaPropsWithUserDefinedChildren = {
    definitions: true,
    properties: true,
    patternProperties: true,
    dependencies: true,
};
//# sourceMappingURL=data:application/json;base64,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