'''
# Amazon ECS Construct Library

<!--BEGIN STABILITY BANNER-->---


![cfn-resources: Stable](https://img.shields.io/badge/cfn--resources-stable-success.svg?style=for-the-badge)

![cdk-constructs: Stable](https://img.shields.io/badge/cdk--constructs-stable-success.svg?style=for-the-badge)

---
<!--END STABILITY BANNER-->

This package contains constructs for working with **Amazon Elastic Container
Service** (Amazon ECS).

Amazon Elastic Container Service (Amazon ECS) is a fully managed container orchestration service.

For further information on Amazon ECS,
see the [Amazon ECS documentation](https://docs.aws.amazon.com/ecs)

The following example creates an Amazon ECS cluster, adds capacity to it, and
runs a service on it:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
from aws_cdk import aws_ecs as ecs


# Create an ECS cluster
cluster = ecs.Cluster(self, "Cluster",
    vpc=vpc
)

# Add capacity to it
cluster.add_capacity("DefaultAutoScalingGroupCapacity",
    instance_type=ec2.InstanceType("t2.xlarge"),
    desired_capacity=3
)

task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")

task_definition.add_container("DefaultContainer",
    image=ecs.ContainerImage.from_registry("amazon/amazon-ecs-sample"),
    memory_limit_mi_b=512
)

# Instantiate an Amazon ECS Service
ecs_service = ecs.Ec2Service(self, "Service",
    cluster=cluster,
    task_definition=task_definition
)
```

For a set of constructs defining common ECS architectural patterns, see the `@aws-cdk/aws-ecs-patterns` package.

## Launch Types: AWS Fargate vs Amazon EC2

There are two sets of constructs in this library; one to run tasks on Amazon EC2 and
one to run tasks on AWS Fargate.

* Use the `Ec2TaskDefinition` and `Ec2Service` constructs to run tasks on Amazon EC2 instances running in your account.
* Use the `FargateTaskDefinition` and `FargateService` constructs to run tasks on
  instances that are managed for you by AWS.

Here are the main differences:

* **Amazon EC2**: instances are under your control. Complete control of task to host
  allocation. Required to specify at least a memory reservation or limit for
  every container. Can use Host, Bridge and AwsVpc networking modes. Can attach
  Classic Load Balancer. Can share volumes between container and host.
* **AWS Fargate**: tasks run on AWS-managed instances, AWS manages task to host
  allocation for you. Requires specification of memory and cpu sizes at the
  taskdefinition level. Only supports AwsVpc networking modes and
  Application/Network Load Balancers. Only the AWS log driver is supported.
  Many host features are not supported such as adding kernel capabilities
  and mounting host devices/volumes inside the container.

For more information on Amazon EC2 vs AWS Fargate and networking see the AWS Documentation:
[AWS Fargate](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/AWS_Fargate.html) and
[Task Networking](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-networking.html).

## Clusters

A `Cluster` defines the infrastructure to run your
tasks on. You can run many tasks on a single cluster.

The following code creates a cluster that can run AWS Fargate tasks:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
cluster = ecs.Cluster(self, "Cluster",
    vpc=vpc
)
```

To use tasks with Amazon EC2 launch-type, you have to add capacity to
the cluster in order for tasks to be scheduled on your instances.  Typically,
you add an AutoScalingGroup with instances running the latest
Amazon ECS-optimized AMI to the cluster. There is a method to build and add such an
AutoScalingGroup automatically, or you can supply a customized AutoScalingGroup
that you construct yourself. It's possible to add multiple AutoScalingGroups
with various instance types.

The following example creates an Amazon ECS cluster and adds capacity to it:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
cluster = ecs.Cluster(self, "Cluster",
    vpc=vpc
)

# Either add default capacity
cluster.add_capacity("DefaultAutoScalingGroupCapacity",
    instance_type=ec2.InstanceType("t2.xlarge"),
    desired_capacity=3
)

# Or add customized capacity. Be sure to start the Amazon ECS-optimized AMI.
auto_scaling_group = autoscaling.AutoScalingGroup(self, "ASG",
    vpc=vpc,
    instance_type=ec2.InstanceType("t2.xlarge"),
    machine_image=EcsOptimizedImage.amazon_linux(),
    # Or use Amazon ECS-Optimized Amazon Linux 2 AMI
    # machineImage: EcsOptimizedImage.amazonLinux2(),
    desired_capacity=3
)

cluster.add_auto_scaling_group(auto_scaling_group)
```

If you omit the property `vpc`, the construct will create a new VPC with two AZs.

### Bottlerocket

[Bottlerocket](https://aws.amazon.com/bottlerocket/) is a Linux-based open source operating system that is
purpose-built by AWS for running containers. You can launch Amazon ECS container instances with the Bottlerocket AMI.

> **NOTICE**: The Bottlerocket AMI is in developer preview release for Amazon ECS and is subject to change.

The following example will create a capacity with self-managed Amazon EC2 capacity of 2 `c5.large` Linux instances running with `Bottlerocket` AMI.

Note that you must specify either a `machineImage` or `machineImageType`, at least one, not both.

The following example adds Bottlerocket capacity to the cluster:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
cluster.add_capacity("bottlerocket-asg",
    min_capacity=2,
    instance_type=ec2.InstanceType("c5.large"),
    machine_image_type=ecs.MachineImageType.BOTTLEROCKET
)
```

### ARM64 (Graviton) Instances

To launch instances with ARM64 hardware, you can use the Amazon ECS-optimized
Amazon Linux 2 (arm64) AMI. Based on Amazon Linux 2, this AMI is recommended
for use when launching your EC2 instances that are powered by Arm-based AWS
Graviton Processors.

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
cluster.add_capacity("graviton-cluster",
    min_capacity=2,
    instance_type=ec2.InstanceType("c6g.large"),
    machine_image=ecs.EcsOptimizedImage.amazon_linux2(ecs.AmiHardwareType.ARM)
)
```

### Spot Instances

To add spot instances into the cluster, you must specify the `spotPrice` in the `ecs.AddCapacityOptions` and optionally enable the `spotInstanceDraining` property.

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
# Add an AutoScalingGroup with spot instances to the existing cluster
cluster.add_capacity("AsgSpot",
    max_capacity=2,
    min_capacity=2,
    desired_capacity=2,
    instance_type=ec2.InstanceType("c5.xlarge"),
    spot_price="0.0735",
    # Enable the Automated Spot Draining support for Amazon ECS
    spot_instance_draining=True
)
```

### SNS Topic Encryption

When the `ecs.AddCapacityOptions` that you provide has a non-zero `taskDrainTime` (the default) then an SNS topic and Lambda are created to ensure that the
cluster's instances have been properly drained of tasks before terminating. The SNS Topic is sent the instance-terminating lifecycle event from the AutoScalingGroup,
and the Lambda acts on that event. If you wish to engage [server-side encryption](https://docs.aws.amazon.com/sns/latest/dg/sns-data-encryption.html) for this SNS Topic
then you may do so by providing a KMS key for the `topicEncryptionKey` property of `ecs.AddCapacityOptions`.

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
# Given
key = kms.Key(...)
# Then, use that key to encrypt the lifecycle-event SNS Topic.
cluster.add_capacity("ASGEncryptedSNS",
    instance_type=ec2.InstanceType("t2.xlarge"),
    desired_capacity=3,
    topic_encryption_key=key
)
```

## Task definitions

A task definition describes what a single copy of a **task** should look like.
A task definition has one or more containers; typically, it has one
main container (the *default container* is the first one that's added
to the task definition, and it is marked *essential*) and optionally
some supporting containers which are used to support the main container,
doings things like upload logs or metrics to monitoring services.

To run a task or service with Amazon EC2 launch type, use the `Ec2TaskDefinition`. For AWS Fargate tasks/services, use the
`FargateTaskDefinition`. These classes provide a simplified API that only contain
properties relevant for that specific launch type.

For a `FargateTaskDefinition`, specify the task size (`memoryLimitMiB` and `cpu`):

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
fargate_task_definition = ecs.FargateTaskDefinition(self, "TaskDef",
    memory_limit_mi_b=512,
    cpu=256
)
```

To add containers to a task definition, call `addContainer()`:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
container = fargate_task_definition.add_container("WebContainer",
    # Use an image from DockerHub
    image=ecs.ContainerImage.from_registry("amazon/amazon-ecs-sample")
)
```

For a `Ec2TaskDefinition`:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
ec2_task_definition = ecs.Ec2TaskDefinition(self, "TaskDef",
    network_mode=NetworkMode.BRIDGE
)

container = ec2_task_definition.add_container("WebContainer",
    # Use an image from DockerHub
    image=ecs.ContainerImage.from_registry("amazon/amazon-ecs-sample"),
    memory_limit_mi_b=1024
)
```

You can specify container properties when you add them to the task definition, or with various methods, e.g.:

To add a port mapping when adding a container to the task definition, specify the `portMappings` option:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
task_definition.add_container("WebContainer",
    image=ecs.ContainerImage.from_registry("amazon/amazon-ecs-sample"),
    memory_limit_mi_b=1024,
    port_mappings=[{"container_port": 3000}]
)
```

To add port mappings directly to a container definition, call `addPortMappings()`:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
container.add_port_mappings(
    container_port=3000
)
```

To add data volumes to a task definition, call `addVolume()`:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
volume = {
    # Use an Elastic FileSystem
    "name": "mydatavolume",
    "efs_volume_configuration": ecs.EfsVolumeConfiguration(
        file_system_id="EFS"
    )
}

container = fargate_task_definition.add_volume("mydatavolume")
```

To use a TaskDefinition that can be used with either Amazon EC2 or
AWS Fargate launch types, use the `TaskDefinition` construct.

When creating a task definition you have to specify what kind of
tasks you intend to run: Amazon EC2, AWS Fargate, or both.
The following example uses both:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
task_definition = ecs.TaskDefinition(self, "TaskDef",
    memory_mi_b="512",
    cpu="256",
    network_mode=NetworkMode.AWS_VPC,
    compatibility=ecs.Compatibility.EC2_AND_FARGATE
)
```

### Images

Images supply the software that runs inside the container. Images can be
obtained from either DockerHub or from ECR repositories, or built directly from a local Dockerfile.

* `ecs.ContainerImage.fromRegistry(imageName)`: use a public image.
* `ecs.ContainerImage.fromRegistry(imageName, { credentials: mySecret })`: use a private image that requires credentials.
* `ecs.ContainerImage.fromEcrRepository(repo, tag)`: use the given ECR repository as the image
  to start. If no tag is provided, "latest" is assumed.
* `ecs.ContainerImage.fromAsset('./image')`: build and upload an
  image directly from a `Dockerfile` in your source directory.
* `ecs.ContainerImage.fromDockerImageAsset(asset)`: uses an existing
  `@aws-cdk/aws-ecr-assets.DockerImageAsset` as a container image.
* `new ecs.TagParameterContainerImage(repository)`: use the given ECR repository as the image
  but a CloudFormation parameter as the tag.

### Environment variables

To pass environment variables to the container, you can use the `environment`, `environmentFiles`, and `secrets` props.

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
task_definition.add_container("container",
    image=ecs.ContainerImage.from_registry("amazon/amazon-ecs-sample"),
    memory_limit_mi_b=1024,
    environment={# clear text, not for sensitive data
        "STAGE": "prod"},
    environment_files=[# list of environment files hosted either on local disk or S3
        ecs.EnvironmentFile.from_asset("./demo-env-file.env"),
        ecs.EnvironmentFile.from_bucket(s3_bucket, "assets/demo-env-file.env")],
    secrets={# Retrieved from AWS Secrets Manager or AWS Systems Manager Parameter Store at container start-up.
        "SECRET": ecs.Secret.from_secrets_manager(secret),
        "DB_PASSWORD": ecs.Secret.from_secrets_manager(db_secret, "password"), # Reference a specific JSON field, (requires platform version 1.4.0 or later for Fargate tasks)
        "PARAMETER": ecs.Secret.from_ssm_parameter(parameter)}
)
```

The task execution role is automatically granted read permissions on the secrets/parameters. Support for environment
files is restricted to the EC2 launch type for files hosted on S3. Further details provided in the AWS documentation
about [specifying environment variables](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/taskdef-envfiles.html).

## Service

A `Service` instantiates a `TaskDefinition` on a `Cluster` a given number of
times, optionally associating them with a load balancer.
If a task fails,
Amazon ECS automatically restarts the task.

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
task_definition =

service = ecs.FargateService(self, "Service",
    cluster=cluster,
    task_definition=task_definition,
    desired_count=5
)
```

`Services` by default will create a security group if not provided.
If you'd like to specify which security groups to use you can override the `securityGroups` property.

### Deployment circuit breaker and rollback

Amazon ECS [deployment circuit breaker](https://aws.amazon.com/tw/blogs/containers/announcing-amazon-ecs-deployment-circuit-breaker/)
automatically rolls back unhealthy service deployments without the need for manual intervention. Use `circuitBreaker` to enable
deployment circuit breaker and optionally enable `rollback` for automatic rollback. See [Using the deployment circuit breaker](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-ecs.html)
for more details.

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
service = ecs.FargateService(stack, "Service",
    cluster=cluster,
    task_definition=task_definition,
    circuit_breaker={"rollback": True}
)
```

### Include an application/network load balancer

`Services` are load balancing targets and can be added to a target group, which will be attached to an application/network load balancers:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
from aws_cdk import aws_elasticloadbalancingv2 as elbv2


service = ecs.FargateService(self, "Service")

lb = elbv2.ApplicationLoadBalancer(self, "LB", vpc=vpc, internet_facing=True)
listener = lb.add_listener("Listener", port=80)
target_group1 = listener.add_targets("ECS1",
    port=80,
    targets=[service]
)
target_group2 = listener.add_targets("ECS2",
    port=80,
    targets=[service.load_balancer_target(
        container_name="MyContainer",
        container_port=8080
    )]
)
```

Note that in the example above, the default `service` only allows you to register the first essential container or the first mapped port on the container as a target and add it to a new target group. To have more control over which container and port to register as targets, you can use `service.loadBalancerTarget()` to return a load balancing target for a specific container and port.

Alternatively, you can also create all load balancer targets to be registered in this service, add them to target groups, and attach target groups to listeners accordingly.

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
from aws_cdk import aws_elasticloadbalancingv2 as elbv2


service = ecs.FargateService(self, "Service")

lb = elbv2.ApplicationLoadBalancer(self, "LB", vpc=vpc, internet_facing=True)
listener = lb.add_listener("Listener", port=80)
service.register_load_balancer_targets(
    container_name="web",
    container_port=80,
    new_target_group_id="ECS",
    listener=ecs.ListenerConfig.application_listener(listener,
        protocol=elbv2.ApplicationProtocol.HTTPS
    )
)
```

### Using a Load Balancer from a different Stack

If you want to put your Load Balancer and the Service it is load balancing to in
different stacks, you may not be able to use the convenience methods
`loadBalancer.addListener()` and `listener.addTargets()`.

The reason is that these methods will create resources in the same Stack as the
object they're called on, which may lead to cyclic references between stacks.
Instead, you will have to create an `ApplicationListener` in the service stack,
or an empty `TargetGroup` in the load balancer stack that you attach your
service to.

See the [ecs/cross-stack-load-balancer example](https://github.com/aws-samples/aws-cdk-examples/tree/master/typescript/ecs/cross-stack-load-balancer/)
for the alternatives.

### Include a classic load balancer

`Services` can also be directly attached to a classic load balancer as targets:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
from aws_cdk import aws_elasticloadbalancing as elb


service = ecs.Ec2Service(self, "Service")

lb = elb.LoadBalancer(stack, "LB", vpc=vpc)
lb.add_listener(external_port=80)
lb.add_target(service)
```

Similarly, if you want to have more control over load balancer targeting:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
from aws_cdk import aws_elasticloadbalancing as elb


service = ecs.Ec2Service(self, "Service")

lb = elb.LoadBalancer(stack, "LB", vpc=vpc)
lb.add_listener(external_port=80)
lb.add_target(service.load_balancer_target(
    container_name="MyContainer",
    container_port=80
))
```

There are two higher-level constructs available which include a load balancer for you that can be found in the aws-ecs-patterns module:

* `LoadBalancedFargateService`
* `LoadBalancedEc2Service`

## Task Auto-Scaling

You can configure the task count of a service to match demand. Task auto-scaling is
configured by calling `autoScaleTaskCount()`:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
scaling = service.auto_scale_task_count(max_capacity=10)
scaling.scale_on_cpu_utilization("CpuScaling",
    target_utilization_percent=50
)

scaling.scale_on_request_count("RequestScaling",
    requests_per_target=10000,
    target_group=target
)
```

Task auto-scaling is powered by *Application Auto-Scaling*.
See that section for details.

## Integration with CloudWatch Events

To start an Amazon ECS task on an Amazon EC2-backed Cluster, instantiate an
`@aws-cdk/aws-events-targets.EcsTask` instead of an `Ec2Service`:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
from aws_cdk import aws_events_targets as targets


# Create a Task Definition for the container to start
task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")
task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_asset(path.resolve(__dirname, "..", "eventhandler-image")),
    memory_limit_mi_b=256,
    logging=ecs.AwsLogDriver(stream_prefix="EventDemo", mode=AwsLogDriverMode.NON_BLOCKING)
)

# An Rule that describes the event trigger (in this case a scheduled run)
rule = events.Rule(self, "Rule",
    schedule=events.Schedule.expression("rate(1 min)")
)

# Pass an environment variable to the container 'TheContainer' in the task
rule.add_target(targets.EcsTask(
    cluster=cluster,
    task_definition=task_definition,
    task_count=1,
    container_overrides=[ContainerOverride(
        container_name="TheContainer",
        environment=[TaskEnvironmentVariable(
            name="I_WAS_TRIGGERED",
            value="From CloudWatch Events"
        )]
    )]
))
```

## Log Drivers

Currently Supported Log Drivers:

* awslogs
* fluentd
* gelf
* journald
* json-file
* splunk
* syslog
* awsfirelens

### awslogs Log Driver

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
# Create a Task Definition for the container to start
task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")
task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_registry("example-image"),
    memory_limit_mi_b=256,
    logging=ecs.LogDrivers.aws_logs(stream_prefix="EventDemo")
)
```

### fluentd Log Driver

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
# Create a Task Definition for the container to start
task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")
task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_registry("example-image"),
    memory_limit_mi_b=256,
    logging=ecs.LogDrivers.fluentd()
)
```

### gelf Log Driver

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
# Create a Task Definition for the container to start
task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")
task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_registry("example-image"),
    memory_limit_mi_b=256,
    logging=ecs.LogDrivers.gelf(address="my-gelf-address")
)
```

### journald Log Driver

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
# Create a Task Definition for the container to start
task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")
task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_registry("example-image"),
    memory_limit_mi_b=256,
    logging=ecs.LogDrivers.journald()
)
```

### json-file Log Driver

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
# Create a Task Definition for the container to start
task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")
task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_registry("example-image"),
    memory_limit_mi_b=256,
    logging=ecs.LogDrivers.json_file()
)
```

### splunk Log Driver

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
# Create a Task Definition for the container to start
task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")
task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_registry("example-image"),
    memory_limit_mi_b=256,
    logging=ecs.LogDrivers.splunk(
        token=cdk.SecretValue.secrets_manager("my-splunk-token"),
        url="my-splunk-url"
    )
)
```

### syslog Log Driver

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
# Create a Task Definition for the container to start
task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")
task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_registry("example-image"),
    memory_limit_mi_b=256,
    logging=ecs.LogDrivers.syslog()
)
```

### firelens Log Driver

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
# Create a Task Definition for the container to start
task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")
task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_registry("example-image"),
    memory_limit_mi_b=256,
    logging=ecs.LogDrivers.firelens(
        options={
            "Name": "firehose",
            "region": "us-west-2",
            "delivery_stream": "my-stream"
        }
    )
)
```

### Generic Log Driver

A generic log driver object exists to provide a lower level abstraction of the log driver configuration.

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
# Create a Task Definition for the container to start
task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")
task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_registry("example-image"),
    memory_limit_mi_b=256,
    logging=ecs.GenericLogDriver(
        log_driver="fluentd",
        options={
            "tag": "example-tag"
        }
    )
)
```

## CloudMap Service Discovery

To register your ECS service with a CloudMap Service Registry, you may add the
`cloudMapOptions` property to your service:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
service = ecs.Ec2Service(stack, "Service",
    cluster=cluster,
    task_definition=task_definition,
    cloud_map_options={
        # Create A records - useful for AWSVPC network mode.
        "dns_record_type": cloudmap.DnsRecordType.A
    }
)
```

With `bridge` or `host` network modes, only `SRV` DNS record types are supported.
By default, `SRV` DNS record types will target the default container and default
port. However, you may target a different container and port on the same ECS task:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
# Add a container to the task definition
specific_container = task_definition.add_container(...)

# Add a port mapping
specific_container.add_port_mappings(
    container_port=7600,
    protocol=ecs.Protocol.TCP
)

ecs.Ec2Service(stack, "Service",
    cluster=cluster,
    task_definition=task_definition,
    cloud_map_options={
        # Create SRV records - useful for bridge networking
        "dns_record_type": cloudmap.DnsRecordType.SRV,
        # Targets port TCP port 7600 `specificContainer`
        "container": specific_container,
        "container_port": 7600
    }
)
```

### Associate With a Specific CloudMap Service

You may associate an ECS service with a specific CloudMap service. To do
this, use the service's `associateCloudMapService` method:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
cloud_map_service = cloudmap.Service(...)
ecs_service = ecs.FargateService(...)

ecs_service.associate_cloud_map_service(
    service=cloud_map_service
)
```

## Capacity Providers

There are two major families of Capacity Providers: [AWS
Fargate](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/fargate-capacity-providers.html)
(including Fargate Spot) and EC2 [Auto Scaling
Group](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/asg-capacity-providers.html)
Capacity Providers. Both are supported.

### Fargate Capacity Providers

To enable Fargate capacity providers, you can either set
`enableFargateCapacityProviders` to `true` when creating your cluster, or by
invoking the `enableFargateCapacityProviders()` method after creating your
cluster. This will add both `FARGATE` and `FARGATE_SPOT` as available capacity
providers on your cluster.

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
cluster = ecs.Cluster(stack, "FargateCPCluster",
    vpc=vpc,
    enable_fargate_capacity_providers=True
)

task_definition = ecs.FargateTaskDefinition(stack, "TaskDef")

task_definition.add_container("web",
    image=ecs.ContainerImage.from_registry("amazon/amazon-ecs-sample")
)

ecs.FargateService(stack, "FargateService",
    cluster=cluster,
    task_definition=task_definition,
    capacity_provider_strategies=[{
        "capacity_provider": "FARGATE_SPOT",
        "weight": 2
    }, {
        "capacity_provider": "FARGATE",
        "weight": 1
    }
    ]
)
```

### Auto Scaling Group Capacity Providers

To add an Auto Scaling Group Capacity Provider, first create an EC2 Auto Scaling
Group. Then, create an `AsgCapacityProvider` and pass the Auto Scaling Group to
it in the constructor. Then add the Capacity Provider to the cluster. Finally,
you can refer to the Provider by its name in your service's or task's Capacity
Provider strategy.

By default, an Auto Scaling Group Capacity Provider will manage the Auto Scaling
Group's size for you. It will also enable managed termination protection, in
order to prevent EC2 Auto Scaling from terminating EC2 instances that have tasks
running on them. If you want to disable this behavior, set both
`enableManagedScaling` to and `enableManagedTerminationProtection` to `false`.

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
cluster = ecs.Cluster(stack, "Cluster",
    vpc=vpc
)

auto_scaling_group = autoscaling.AutoScalingGroup(stack, "ASG",
    vpc=vpc,
    instance_type=ec2.InstanceType("t2.micro"),
    machine_image=ecs.EcsOptimizedImage.amazon_linux2(),
    min_capacity=0,
    max_capacity=100
)

capacity_provider = ecs.AsgCapacityProvider(stack, "AsgCapacityProvider",
    auto_scaling_group=auto_scaling_group
)
cluster.add_asg_capacity_provider(capacity_provider)

task_definition = ecs.Ec2TaskDefinition(stack, "TaskDef")

task_definition.add_container("web",
    image=ecs.ContainerImage.from_registry("amazon/amazon-ecs-sample", memory_reservation_mi_b, 256)
)

ecs.Ec2Service(stack, "EC2Service",
    cluster=cluster,
    task_definition=task_definition,
    capacity_provider_strategies=[{
        "capacity_provider": capacity_provider.capacity_provider_name,
        "weight": 1
    }
    ]
)
```

## Elastic Inference Accelerators

Currently, this feature is only supported for services with EC2 launch types.

To add elastic inference accelerators to your EC2 instance, first add
`inferenceAccelerators` field to the Ec2TaskDefinition and set the `deviceName`
and `deviceType` properties.

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
inference_accelerators = [{
    "device_name": "device1",
    "device_type": "eia2.medium"
}]

task_definition = ecs.Ec2TaskDefinition(stack, "Ec2TaskDef",
    inference_accelerators=inference_accelerators
)
```

To enable using the inference accelerators in the containers, add `inferenceAcceleratorResources`
field and set it to a list of device names used for the inference accelerators. Each value in the
list should match a `DeviceName` for an `InferenceAccelerator` specified in the task definition.

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
inference_accelerator_resources = ["device1"]

task_definition.add_container("cont",
    image=ecs.ContainerImage.from_registry("test"),
    memory_limit_mi_b=1024,
    inference_accelerator_resources=inference_accelerator_resources
)
```
'''
import abc
import builtins
import datetime
import enum
import typing

import jsii
import publication
import typing_extensions

from .._jsii import *

import constructs
from .. import (
    AssetHashType as _AssetHashType_49193809,
    BundlingOptions as _BundlingOptions_ab115a99,
    CfnResource as _CfnResource_e0a482dc,
    CfnTag as _CfnTag_95fbdc29,
    Construct as _Construct_e78e779f,
    Duration as _Duration_070aa057,
    IInspectable as _IInspectable_82c04a63,
    IResolvable as _IResolvable_a771d0ef,
    IResource as _IResource_8c1dbbbd,
    IgnoreMode as _IgnoreMode_31d8bf46,
    Resource as _Resource_abff4495,
    SecretValue as _SecretValue_c18506ef,
    SymlinkFollowMode as _SymlinkFollowMode_abf4527a,
    TagManager as _TagManager_0b7ab120,
    TreeInspector as _TreeInspector_1cd1894e,
)
from ..assets import FollowMode as _FollowMode_98b05cc5
from ..aws_applicationautoscaling import (
    AdjustmentType as _AdjustmentType_c97ebf31,
    BaseScalableAttribute as _BaseScalableAttribute_8cad7a2c,
    BaseScalableAttributeProps as _BaseScalableAttributeProps_1600c742,
    BaseTargetTrackingProps as _BaseTargetTrackingProps_e4402570,
    BasicStepScalingPolicyProps as _BasicStepScalingPolicyProps_2c327255,
    EnableScalingProps as _EnableScalingProps_d2090945,
    MetricAggregationType as _MetricAggregationType_c42ebfd0,
    ScalingInterval as _ScalingInterval_ed8cf944,
    ScalingSchedule as _ScalingSchedule_6c3fc38f,
    Schedule as _Schedule_c2a5a45d,
    ServiceNamespace as _ServiceNamespace_42590eaf,
)
from ..aws_autoscaling import (
    AutoScalingGroup as _AutoScalingGroup_604a934f,
    BlockDevice as _BlockDevice_da8302ba,
    CommonAutoScalingGroupProps as _CommonAutoScalingGroupProps_0c339448,
    GroupMetrics as _GroupMetrics_c42c90fb,
    HealthCheck as _HealthCheck_f6164c37,
    IAutoScalingGroup as _IAutoScalingGroup_352723c6,
    Monitoring as _Monitoring_ab8b25ef,
    NotificationConfiguration as _NotificationConfiguration_14f038b9,
    RollingUpdateConfiguration as _RollingUpdateConfiguration_7b14e30c,
    Signals as _Signals_896b8d51,
    UpdatePolicy as _UpdatePolicy_ffeec124,
    UpdateType as _UpdateType_1d8acce6,
)
from ..aws_cloudwatch import (
    IMetric as _IMetric_5db43d61,
    Metric as _Metric_5b2b8e58,
    MetricOptions as _MetricOptions_1c185ae8,
    Unit as _Unit_113c79f9,
)
from ..aws_ec2 import (
    AmazonLinuxGeneration as _AmazonLinuxGeneration_d9650a6d,
    Connections as _Connections_57ccbda9,
    IMachineImage as _IMachineImage_45d3238d,
    ISecurityGroup as _ISecurityGroup_cdbba9d3,
    IVpc as _IVpc_6d1f76c4,
    InstanceType as _InstanceType_072ad323,
    MachineImageConfig as _MachineImageConfig_963798fe,
    SubnetSelection as _SubnetSelection_1284e62c,
)
from ..aws_ecr import IRepository as _IRepository_8b4d2894
from ..aws_ecr_assets import (
    DockerImageAsset as _DockerImageAsset_3ac823f8,
    DockerImageAssetOptions as _DockerImageAssetOptions_6764a0de,
)
from ..aws_elasticloadbalancing import (
    ILoadBalancerTarget as _ILoadBalancerTarget_ad59785d,
    LoadBalancer as _LoadBalancer_a7de240f,
)
from ..aws_elasticloadbalancingv2 import (
    AddApplicationTargetsProps as _AddApplicationTargetsProps_a85e0c25,
    AddNetworkTargetsProps as _AddNetworkTargetsProps_63ca961a,
    ApplicationListener as _ApplicationListener_835df0e5,
    ApplicationProtocol as _ApplicationProtocol_43133732,
    ApplicationProtocolVersion as _ApplicationProtocolVersion_01dcd421,
    ApplicationTargetGroup as _ApplicationTargetGroup_5b334416,
    HealthCheck as _HealthCheck_46284083,
    IApplicationLoadBalancerTarget as _IApplicationLoadBalancerTarget_db0fdc70,
    IApplicationTargetGroup as _IApplicationTargetGroup_5a474b2b,
    INetworkLoadBalancerTarget as _INetworkLoadBalancerTarget_6691f661,
    INetworkTargetGroup as _INetworkTargetGroup_6938b578,
    ListenerCondition as _ListenerCondition_c2f847ce,
    LoadBalancerTargetProps as _LoadBalancerTargetProps_8b99f223,
    NetworkListener as _NetworkListener_0917bd7b,
    Protocol as _Protocol_5d30b733,
)
from ..aws_iam import (
    Grant as _Grant_bcb5eae7,
    IGrantable as _IGrantable_4c5a91d1,
    IRole as _IRole_59af6f50,
    PolicyStatement as _PolicyStatement_296fe8a3,
)
from ..aws_kms import IKey as _IKey_36930160
from ..aws_logs import (
    ILogGroup as _ILogGroup_846e17a0, RetentionDays as _RetentionDays_6c560d31
)
from ..aws_s3 import IBucket as _IBucket_73486e29, Location as _Location_cce991ca
from ..aws_s3_assets import AssetOptions as _AssetOptions_bd2996da
from ..aws_secretsmanager import ISecret as _ISecret_22fb8757
from ..aws_servicediscovery import (
    DnsRecordType as _DnsRecordType_a28443d0,
    INamespace as _INamespace_eb21e42e,
    IService as _IService_66c1fbd2,
    NamespaceType as _NamespaceType_db55a104,
    Service as _Service_38744a4d,
)
from ..aws_sns import ITopic as _ITopic_465e36b9
from ..aws_ssm import IParameter as _IParameter_ce5fb757


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.AddAutoScalingGroupCapacityOptions",
    jsii_struct_bases=[],
    name_mapping={
        "can_containers_access_instance_role": "canContainersAccessInstanceRole",
        "machine_image_type": "machineImageType",
        "spot_instance_draining": "spotInstanceDraining",
        "task_drain_time": "taskDrainTime",
        "topic_encryption_key": "topicEncryptionKey",
    },
)
class AddAutoScalingGroupCapacityOptions:
    def __init__(
        self,
        *,
        can_containers_access_instance_role: typing.Optional[builtins.bool] = None,
        machine_image_type: typing.Optional["MachineImageType"] = None,
        spot_instance_draining: typing.Optional[builtins.bool] = None,
        task_drain_time: typing.Optional[_Duration_070aa057] = None,
        topic_encryption_key: typing.Optional[_IKey_36930160] = None,
    ) -> None:
        '''(experimental) The properties for adding an AutoScalingGroup.

        :param can_containers_access_instance_role: (experimental) Specifies whether the containers can access the container instance role. Default: false
        :param machine_image_type: (experimental) Specify the machine image type. Default: MachineImageType.AMAZON_LINUX_2
        :param spot_instance_draining: (experimental) Specify whether to enable Automated Draining for Spot Instances running Amazon ECS Services. For more information, see `Using Spot Instances <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container-instance-spot.html>`_. Default: false
        :param task_drain_time: (deprecated) The time period to wait before force terminating an instance that is draining. This creates a Lambda function that is used by a lifecycle hook for the AutoScalingGroup that will delay instance termination until all ECS tasks have drained from the instance. Set to 0 to disable task draining. Set to 0 to disable task draining. Default: Duration.minutes(5)
        :param topic_encryption_key: (experimental) If {@link AddAutoScalingGroupCapacityOptions.taskDrainTime} is non-zero, then the ECS cluster creates an SNS Topic to as part of a system to drain instances of tasks when the instance is being shut down. If this property is provided, then this key will be used to encrypt the contents of that SNS Topic. See `SNS Data Encryption <https://docs.aws.amazon.com/sns/latest/dg/sns-data-encryption.html>`_ for more information. Default: The SNS Topic will not be encrypted.

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if can_containers_access_instance_role is not None:
            self._values["can_containers_access_instance_role"] = can_containers_access_instance_role
        if machine_image_type is not None:
            self._values["machine_image_type"] = machine_image_type
        if spot_instance_draining is not None:
            self._values["spot_instance_draining"] = spot_instance_draining
        if task_drain_time is not None:
            self._values["task_drain_time"] = task_drain_time
        if topic_encryption_key is not None:
            self._values["topic_encryption_key"] = topic_encryption_key

    @builtins.property
    def can_containers_access_instance_role(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specifies whether the containers can access the container instance role.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("can_containers_access_instance_role")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def machine_image_type(self) -> typing.Optional["MachineImageType"]:
        '''(experimental) Specify the machine image type.

        :default: MachineImageType.AMAZON_LINUX_2

        :stability: experimental
        '''
        result = self._values.get("machine_image_type")
        return typing.cast(typing.Optional["MachineImageType"], result)

    @builtins.property
    def spot_instance_draining(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specify whether to enable Automated Draining for Spot Instances running Amazon ECS Services.

        For more information, see `Using Spot Instances <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container-instance-spot.html>`_.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("spot_instance_draining")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def task_drain_time(self) -> typing.Optional[_Duration_070aa057]:
        '''(deprecated) The time period to wait before force terminating an instance that is draining.

        This creates a Lambda function that is used by a lifecycle hook for the
        AutoScalingGroup that will delay instance termination until all ECS tasks
        have drained from the instance. Set to 0 to disable task draining.

        Set to 0 to disable task draining.

        :default: Duration.minutes(5)

        :deprecated: The lifecycle draining hook is not configured if using the EC2 Capacity Provider. Enable managed termination protection instead.

        :stability: deprecated
        '''
        result = self._values.get("task_drain_time")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def topic_encryption_key(self) -> typing.Optional[_IKey_36930160]:
        '''(experimental) If {@link AddAutoScalingGroupCapacityOptions.taskDrainTime} is non-zero, then the ECS cluster creates an SNS Topic to as part of a system to drain instances of tasks when the instance is being shut down. If this property is provided, then this key will be used to encrypt the contents of that SNS Topic. See `SNS Data Encryption <https://docs.aws.amazon.com/sns/latest/dg/sns-data-encryption.html>`_ for more information.

        :default: The SNS Topic will not be encrypted.

        :stability: experimental
        '''
        result = self._values.get("topic_encryption_key")
        return typing.cast(typing.Optional[_IKey_36930160], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AddAutoScalingGroupCapacityOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.AddCapacityOptions",
    jsii_struct_bases=[
        AddAutoScalingGroupCapacityOptions, _CommonAutoScalingGroupProps_0c339448
    ],
    name_mapping={
        "can_containers_access_instance_role": "canContainersAccessInstanceRole",
        "machine_image_type": "machineImageType",
        "spot_instance_draining": "spotInstanceDraining",
        "task_drain_time": "taskDrainTime",
        "topic_encryption_key": "topicEncryptionKey",
        "allow_all_outbound": "allowAllOutbound",
        "associate_public_ip_address": "associatePublicIpAddress",
        "auto_scaling_group_name": "autoScalingGroupName",
        "block_devices": "blockDevices",
        "cooldown": "cooldown",
        "desired_capacity": "desiredCapacity",
        "group_metrics": "groupMetrics",
        "health_check": "healthCheck",
        "ignore_unmodified_size_properties": "ignoreUnmodifiedSizeProperties",
        "instance_monitoring": "instanceMonitoring",
        "key_name": "keyName",
        "max_capacity": "maxCapacity",
        "max_instance_lifetime": "maxInstanceLifetime",
        "min_capacity": "minCapacity",
        "new_instances_protected_from_scale_in": "newInstancesProtectedFromScaleIn",
        "notifications": "notifications",
        "notifications_topic": "notificationsTopic",
        "replacing_update_min_successful_instances_percent": "replacingUpdateMinSuccessfulInstancesPercent",
        "resource_signal_count": "resourceSignalCount",
        "resource_signal_timeout": "resourceSignalTimeout",
        "rolling_update_configuration": "rollingUpdateConfiguration",
        "signals": "signals",
        "spot_price": "spotPrice",
        "update_policy": "updatePolicy",
        "update_type": "updateType",
        "vpc_subnets": "vpcSubnets",
        "instance_type": "instanceType",
        "machine_image": "machineImage",
    },
)
class AddCapacityOptions(
    AddAutoScalingGroupCapacityOptions,
    _CommonAutoScalingGroupProps_0c339448,
):
    def __init__(
        self,
        *,
        can_containers_access_instance_role: typing.Optional[builtins.bool] = None,
        machine_image_type: typing.Optional["MachineImageType"] = None,
        spot_instance_draining: typing.Optional[builtins.bool] = None,
        task_drain_time: typing.Optional[_Duration_070aa057] = None,
        topic_encryption_key: typing.Optional[_IKey_36930160] = None,
        allow_all_outbound: typing.Optional[builtins.bool] = None,
        associate_public_ip_address: typing.Optional[builtins.bool] = None,
        auto_scaling_group_name: typing.Optional[builtins.str] = None,
        block_devices: typing.Optional[typing.Sequence[_BlockDevice_da8302ba]] = None,
        cooldown: typing.Optional[_Duration_070aa057] = None,
        desired_capacity: typing.Optional[jsii.Number] = None,
        group_metrics: typing.Optional[typing.Sequence[_GroupMetrics_c42c90fb]] = None,
        health_check: typing.Optional[_HealthCheck_f6164c37] = None,
        ignore_unmodified_size_properties: typing.Optional[builtins.bool] = None,
        instance_monitoring: typing.Optional[_Monitoring_ab8b25ef] = None,
        key_name: typing.Optional[builtins.str] = None,
        max_capacity: typing.Optional[jsii.Number] = None,
        max_instance_lifetime: typing.Optional[_Duration_070aa057] = None,
        min_capacity: typing.Optional[jsii.Number] = None,
        new_instances_protected_from_scale_in: typing.Optional[builtins.bool] = None,
        notifications: typing.Optional[typing.Sequence[_NotificationConfiguration_14f038b9]] = None,
        notifications_topic: typing.Optional[_ITopic_465e36b9] = None,
        replacing_update_min_successful_instances_percent: typing.Optional[jsii.Number] = None,
        resource_signal_count: typing.Optional[jsii.Number] = None,
        resource_signal_timeout: typing.Optional[_Duration_070aa057] = None,
        rolling_update_configuration: typing.Optional[_RollingUpdateConfiguration_7b14e30c] = None,
        signals: typing.Optional[_Signals_896b8d51] = None,
        spot_price: typing.Optional[builtins.str] = None,
        update_policy: typing.Optional[_UpdatePolicy_ffeec124] = None,
        update_type: typing.Optional[_UpdateType_1d8acce6] = None,
        vpc_subnets: typing.Optional[_SubnetSelection_1284e62c] = None,
        instance_type: _InstanceType_072ad323,
        machine_image: typing.Optional[_IMachineImage_45d3238d] = None,
    ) -> None:
        '''(experimental) The properties for adding instance capacity to an AutoScalingGroup.

        :param can_containers_access_instance_role: (experimental) Specifies whether the containers can access the container instance role. Default: false
        :param machine_image_type: (experimental) Specify the machine image type. Default: MachineImageType.AMAZON_LINUX_2
        :param spot_instance_draining: (experimental) Specify whether to enable Automated Draining for Spot Instances running Amazon ECS Services. For more information, see `Using Spot Instances <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container-instance-spot.html>`_. Default: false
        :param task_drain_time: (deprecated) The time period to wait before force terminating an instance that is draining. This creates a Lambda function that is used by a lifecycle hook for the AutoScalingGroup that will delay instance termination until all ECS tasks have drained from the instance. Set to 0 to disable task draining. Set to 0 to disable task draining. Default: Duration.minutes(5)
        :param topic_encryption_key: (experimental) If {@link AddAutoScalingGroupCapacityOptions.taskDrainTime} is non-zero, then the ECS cluster creates an SNS Topic to as part of a system to drain instances of tasks when the instance is being shut down. If this property is provided, then this key will be used to encrypt the contents of that SNS Topic. See `SNS Data Encryption <https://docs.aws.amazon.com/sns/latest/dg/sns-data-encryption.html>`_ for more information. Default: The SNS Topic will not be encrypted.
        :param allow_all_outbound: (experimental) Whether the instances can initiate connections to anywhere by default. Default: true
        :param associate_public_ip_address: (experimental) Whether instances in the Auto Scaling Group should have public IP addresses associated with them. Default: - Use subnet setting.
        :param auto_scaling_group_name: (experimental) The name of the Auto Scaling group. This name must be unique per Region per account. Default: - Auto generated by CloudFormation
        :param block_devices: (experimental) Specifies how block devices are exposed to the instance. You can specify virtual devices and EBS volumes. Each instance that is launched has an associated root device volume, either an Amazon EBS volume or an instance store volume. You can use block device mappings to specify additional EBS volumes or instance store volumes to attach to an instance when it is launched. Default: - Uses the block device mapping of the AMI
        :param cooldown: (experimental) Default scaling cooldown for this AutoScalingGroup. Default: Duration.minutes(5)
        :param desired_capacity: (experimental) Initial amount of instances in the fleet. If this is set to a number, every deployment will reset the amount of instances to this number. It is recommended to leave this value blank. Default: minCapacity, and leave unchanged during deployment
        :param group_metrics: (experimental) Enable monitoring for group metrics, these metrics describe the group rather than any of its instances. To report all group metrics use ``GroupMetrics.all()`` Group metrics are reported in a granularity of 1 minute at no additional charge. Default: - no group metrics will be reported
        :param health_check: (experimental) Configuration for health checks. Default: - HealthCheck.ec2 with no grace period
        :param ignore_unmodified_size_properties: (experimental) If the ASG has scheduled actions, don't reset unchanged group sizes. Only used if the ASG has scheduled actions (which may scale your ASG up or down regardless of cdk deployments). If true, the size of the group will only be reset if it has been changed in the CDK app. If false, the sizes will always be changed back to what they were in the CDK app on deployment. Default: true
        :param instance_monitoring: (experimental) Controls whether instances in this group are launched with detailed or basic monitoring. When detailed monitoring is enabled, Amazon CloudWatch generates metrics every minute and your account is charged a fee. When you disable detailed monitoring, CloudWatch generates metrics every 5 minutes. Default: - Monitoring.DETAILED
        :param key_name: (experimental) Name of SSH keypair to grant access to instances. Default: - No SSH access will be possible.
        :param max_capacity: (experimental) Maximum number of instances in the fleet. Default: desiredCapacity
        :param max_instance_lifetime: (experimental) The maximum amount of time that an instance can be in service. The maximum duration applies to all current and future instances in the group. As an instance approaches its maximum duration, it is terminated and replaced, and cannot be used again. You must specify a value of at least 604,800 seconds (7 days). To clear a previously set value, leave this property undefined. Default: none
        :param min_capacity: (experimental) Minimum number of instances in the fleet. Default: 1
        :param new_instances_protected_from_scale_in: (experimental) Whether newly-launched instances are protected from termination by Amazon EC2 Auto Scaling when scaling in. By default, Auto Scaling can terminate an instance at any time after launch when scaling in an Auto Scaling Group, subject to the group's termination policy. However, you may wish to protect newly-launched instances from being scaled in if they are going to run critical applications that should not be prematurely terminated. This flag must be enabled if the Auto Scaling Group will be associated with an ECS Capacity Provider with managed termination protection. Default: false
        :param notifications: (experimental) Configure autoscaling group to send notifications about fleet changes to an SNS topic(s). Default: - No fleet change notifications will be sent.
        :param notifications_topic: (deprecated) SNS topic to send notifications about fleet changes. Default: - No fleet change notifications will be sent.
        :param replacing_update_min_successful_instances_percent: (deprecated) Configuration for replacing updates. Only used if updateType == UpdateType.ReplacingUpdate. Specifies how many instances must signal success for the update to succeed. Default: minSuccessfulInstancesPercent
        :param resource_signal_count: (deprecated) How many ResourceSignal calls CloudFormation expects before the resource is considered created. Default: 1 if resourceSignalTimeout is set, 0 otherwise
        :param resource_signal_timeout: (deprecated) The length of time to wait for the resourceSignalCount. The maximum value is 43200 (12 hours). Default: Duration.minutes(5) if resourceSignalCount is set, N/A otherwise
        :param rolling_update_configuration: (deprecated) Configuration for rolling updates. Only used if updateType == UpdateType.RollingUpdate. Default: - RollingUpdateConfiguration with defaults.
        :param signals: (experimental) Configure waiting for signals during deployment. Use this to pause the CloudFormation deployment to wait for the instances in the AutoScalingGroup to report successful startup during creation and updates. The UserData script needs to invoke ``cfn-signal`` with a success or failure code after it is done setting up the instance. Without waiting for signals, the CloudFormation deployment will proceed as soon as the AutoScalingGroup has been created or updated but before the instances in the group have been started. For example, to have instances wait for an Elastic Load Balancing health check before they signal success, add a health-check verification by using the cfn-init helper script. For an example, see the verify_instance_health command in the Auto Scaling rolling updates sample template: https://github.com/awslabs/aws-cloudformation-templates/blob/master/aws/services/AutoScaling/AutoScalingRollingUpdates.yaml Default: - Do not wait for signals
        :param spot_price: (experimental) The maximum hourly price (in USD) to be paid for any Spot Instance launched to fulfill the request. Spot Instances are launched when the price you specify exceeds the current Spot market price. Default: none
        :param update_policy: (experimental) What to do when an AutoScalingGroup's instance configuration is changed. This is applied when any of the settings on the ASG are changed that affect how the instances should be created (VPC, instance type, startup scripts, etc.). It indicates how the existing instances should be replaced with new instances matching the new config. By default, nothing is done and only new instances are launched with the new config. Default: - ``UpdatePolicy.rollingUpdate()`` if using ``init``, ``UpdatePolicy.none()`` otherwise
        :param update_type: (deprecated) What to do when an AutoScalingGroup's instance configuration is changed. This is applied when any of the settings on the ASG are changed that affect how the instances should be created (VPC, instance type, startup scripts, etc.). It indicates how the existing instances should be replaced with new instances matching the new config. By default, nothing is done and only new instances are launched with the new config. Default: UpdateType.None
        :param vpc_subnets: (experimental) Where to place instances within the VPC. Default: - All Private subnets.
        :param instance_type: (experimental) The EC2 instance type to use when launching instances into the AutoScalingGroup.
        :param machine_image: (experimental) The ECS-optimized AMI variant to use. For more information, see `Amazon ECS-optimized AMIs <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html>`_. You must define either ``machineImage`` or ``machineImageType``, not both. Default: - Amazon Linux 2

        :stability: experimental
        '''
        if isinstance(rolling_update_configuration, dict):
            rolling_update_configuration = _RollingUpdateConfiguration_7b14e30c(**rolling_update_configuration)
        if isinstance(vpc_subnets, dict):
            vpc_subnets = _SubnetSelection_1284e62c(**vpc_subnets)
        self._values: typing.Dict[str, typing.Any] = {
            "instance_type": instance_type,
        }
        if can_containers_access_instance_role is not None:
            self._values["can_containers_access_instance_role"] = can_containers_access_instance_role
        if machine_image_type is not None:
            self._values["machine_image_type"] = machine_image_type
        if spot_instance_draining is not None:
            self._values["spot_instance_draining"] = spot_instance_draining
        if task_drain_time is not None:
            self._values["task_drain_time"] = task_drain_time
        if topic_encryption_key is not None:
            self._values["topic_encryption_key"] = topic_encryption_key
        if allow_all_outbound is not None:
            self._values["allow_all_outbound"] = allow_all_outbound
        if associate_public_ip_address is not None:
            self._values["associate_public_ip_address"] = associate_public_ip_address
        if auto_scaling_group_name is not None:
            self._values["auto_scaling_group_name"] = auto_scaling_group_name
        if block_devices is not None:
            self._values["block_devices"] = block_devices
        if cooldown is not None:
            self._values["cooldown"] = cooldown
        if desired_capacity is not None:
            self._values["desired_capacity"] = desired_capacity
        if group_metrics is not None:
            self._values["group_metrics"] = group_metrics
        if health_check is not None:
            self._values["health_check"] = health_check
        if ignore_unmodified_size_properties is not None:
            self._values["ignore_unmodified_size_properties"] = ignore_unmodified_size_properties
        if instance_monitoring is not None:
            self._values["instance_monitoring"] = instance_monitoring
        if key_name is not None:
            self._values["key_name"] = key_name
        if max_capacity is not None:
            self._values["max_capacity"] = max_capacity
        if max_instance_lifetime is not None:
            self._values["max_instance_lifetime"] = max_instance_lifetime
        if min_capacity is not None:
            self._values["min_capacity"] = min_capacity
        if new_instances_protected_from_scale_in is not None:
            self._values["new_instances_protected_from_scale_in"] = new_instances_protected_from_scale_in
        if notifications is not None:
            self._values["notifications"] = notifications
        if notifications_topic is not None:
            self._values["notifications_topic"] = notifications_topic
        if replacing_update_min_successful_instances_percent is not None:
            self._values["replacing_update_min_successful_instances_percent"] = replacing_update_min_successful_instances_percent
        if resource_signal_count is not None:
            self._values["resource_signal_count"] = resource_signal_count
        if resource_signal_timeout is not None:
            self._values["resource_signal_timeout"] = resource_signal_timeout
        if rolling_update_configuration is not None:
            self._values["rolling_update_configuration"] = rolling_update_configuration
        if signals is not None:
            self._values["signals"] = signals
        if spot_price is not None:
            self._values["spot_price"] = spot_price
        if update_policy is not None:
            self._values["update_policy"] = update_policy
        if update_type is not None:
            self._values["update_type"] = update_type
        if vpc_subnets is not None:
            self._values["vpc_subnets"] = vpc_subnets
        if machine_image is not None:
            self._values["machine_image"] = machine_image

    @builtins.property
    def can_containers_access_instance_role(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specifies whether the containers can access the container instance role.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("can_containers_access_instance_role")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def machine_image_type(self) -> typing.Optional["MachineImageType"]:
        '''(experimental) Specify the machine image type.

        :default: MachineImageType.AMAZON_LINUX_2

        :stability: experimental
        '''
        result = self._values.get("machine_image_type")
        return typing.cast(typing.Optional["MachineImageType"], result)

    @builtins.property
    def spot_instance_draining(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specify whether to enable Automated Draining for Spot Instances running Amazon ECS Services.

        For more information, see `Using Spot Instances <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container-instance-spot.html>`_.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("spot_instance_draining")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def task_drain_time(self) -> typing.Optional[_Duration_070aa057]:
        '''(deprecated) The time period to wait before force terminating an instance that is draining.

        This creates a Lambda function that is used by a lifecycle hook for the
        AutoScalingGroup that will delay instance termination until all ECS tasks
        have drained from the instance. Set to 0 to disable task draining.

        Set to 0 to disable task draining.

        :default: Duration.minutes(5)

        :deprecated: The lifecycle draining hook is not configured if using the EC2 Capacity Provider. Enable managed termination protection instead.

        :stability: deprecated
        '''
        result = self._values.get("task_drain_time")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def topic_encryption_key(self) -> typing.Optional[_IKey_36930160]:
        '''(experimental) If {@link AddAutoScalingGroupCapacityOptions.taskDrainTime} is non-zero, then the ECS cluster creates an SNS Topic to as part of a system to drain instances of tasks when the instance is being shut down. If this property is provided, then this key will be used to encrypt the contents of that SNS Topic. See `SNS Data Encryption <https://docs.aws.amazon.com/sns/latest/dg/sns-data-encryption.html>`_ for more information.

        :default: The SNS Topic will not be encrypted.

        :stability: experimental
        '''
        result = self._values.get("topic_encryption_key")
        return typing.cast(typing.Optional[_IKey_36930160], result)

    @builtins.property
    def allow_all_outbound(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether the instances can initiate connections to anywhere by default.

        :default: true

        :stability: experimental
        '''
        result = self._values.get("allow_all_outbound")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def associate_public_ip_address(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether instances in the Auto Scaling Group should have public IP addresses associated with them.

        :default: - Use subnet setting.

        :stability: experimental
        '''
        result = self._values.get("associate_public_ip_address")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def auto_scaling_group_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name of the Auto Scaling group.

        This name must be unique per Region per account.

        :default: - Auto generated by CloudFormation

        :stability: experimental
        '''
        result = self._values.get("auto_scaling_group_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def block_devices(self) -> typing.Optional[typing.List[_BlockDevice_da8302ba]]:
        '''(experimental) Specifies how block devices are exposed to the instance. You can specify virtual devices and EBS volumes.

        Each instance that is launched has an associated root device volume,
        either an Amazon EBS volume or an instance store volume.
        You can use block device mappings to specify additional EBS volumes or
        instance store volumes to attach to an instance when it is launched.

        :default: - Uses the block device mapping of the AMI

        :see: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/block-device-mapping-concepts.html
        :stability: experimental
        '''
        result = self._values.get("block_devices")
        return typing.cast(typing.Optional[typing.List[_BlockDevice_da8302ba]], result)

    @builtins.property
    def cooldown(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Default scaling cooldown for this AutoScalingGroup.

        :default: Duration.minutes(5)

        :stability: experimental
        '''
        result = self._values.get("cooldown")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def desired_capacity(self) -> typing.Optional[jsii.Number]:
        '''(experimental) Initial amount of instances in the fleet.

        If this is set to a number, every deployment will reset the amount of
        instances to this number. It is recommended to leave this value blank.

        :default: minCapacity, and leave unchanged during deployment

        :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-desiredcapacity
        :stability: experimental
        '''
        result = self._values.get("desired_capacity")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def group_metrics(self) -> typing.Optional[typing.List[_GroupMetrics_c42c90fb]]:
        '''(experimental) Enable monitoring for group metrics, these metrics describe the group rather than any of its instances.

        To report all group metrics use ``GroupMetrics.all()``
        Group metrics are reported in a granularity of 1 minute at no additional charge.

        :default: - no group metrics will be reported

        :stability: experimental
        '''
        result = self._values.get("group_metrics")
        return typing.cast(typing.Optional[typing.List[_GroupMetrics_c42c90fb]], result)

    @builtins.property
    def health_check(self) -> typing.Optional[_HealthCheck_f6164c37]:
        '''(experimental) Configuration for health checks.

        :default: - HealthCheck.ec2 with no grace period

        :stability: experimental
        '''
        result = self._values.get("health_check")
        return typing.cast(typing.Optional[_HealthCheck_f6164c37], result)

    @builtins.property
    def ignore_unmodified_size_properties(self) -> typing.Optional[builtins.bool]:
        '''(experimental) If the ASG has scheduled actions, don't reset unchanged group sizes.

        Only used if the ASG has scheduled actions (which may scale your ASG up
        or down regardless of cdk deployments). If true, the size of the group
        will only be reset if it has been changed in the CDK app. If false, the
        sizes will always be changed back to what they were in the CDK app
        on deployment.

        :default: true

        :stability: experimental
        '''
        result = self._values.get("ignore_unmodified_size_properties")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def instance_monitoring(self) -> typing.Optional[_Monitoring_ab8b25ef]:
        '''(experimental) Controls whether instances in this group are launched with detailed or basic monitoring.

        When detailed monitoring is enabled, Amazon CloudWatch generates metrics every minute and your account
        is charged a fee. When you disable detailed monitoring, CloudWatch generates metrics every 5 minutes.

        :default: - Monitoring.DETAILED

        :see: https://docs.aws.amazon.com/autoscaling/latest/userguide/as-instance-monitoring.html#enable-as-instance-metrics
        :stability: experimental
        '''
        result = self._values.get("instance_monitoring")
        return typing.cast(typing.Optional[_Monitoring_ab8b25ef], result)

    @builtins.property
    def key_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) Name of SSH keypair to grant access to instances.

        :default: - No SSH access will be possible.

        :stability: experimental
        '''
        result = self._values.get("key_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def max_capacity(self) -> typing.Optional[jsii.Number]:
        '''(experimental) Maximum number of instances in the fleet.

        :default: desiredCapacity

        :stability: experimental
        '''
        result = self._values.get("max_capacity")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def max_instance_lifetime(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) The maximum amount of time that an instance can be in service.

        The maximum duration applies
        to all current and future instances in the group. As an instance approaches its maximum duration,
        it is terminated and replaced, and cannot be used again.

        You must specify a value of at least 604,800 seconds (7 days). To clear a previously set value,
        leave this property undefined.

        :default: none

        :see: https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-max-instance-lifetime.html
        :stability: experimental
        '''
        result = self._values.get("max_instance_lifetime")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def min_capacity(self) -> typing.Optional[jsii.Number]:
        '''(experimental) Minimum number of instances in the fleet.

        :default: 1

        :stability: experimental
        '''
        result = self._values.get("min_capacity")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def new_instances_protected_from_scale_in(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether newly-launched instances are protected from termination by Amazon EC2 Auto Scaling when scaling in.

        By default, Auto Scaling can terminate an instance at any time after launch
        when scaling in an Auto Scaling Group, subject to the group's termination
        policy. However, you may wish to protect newly-launched instances from
        being scaled in if they are going to run critical applications that should
        not be prematurely terminated.

        This flag must be enabled if the Auto Scaling Group will be associated with
        an ECS Capacity Provider with managed termination protection.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("new_instances_protected_from_scale_in")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def notifications(
        self,
    ) -> typing.Optional[typing.List[_NotificationConfiguration_14f038b9]]:
        '''(experimental) Configure autoscaling group to send notifications about fleet changes to an SNS topic(s).

        :default: - No fleet change notifications will be sent.

        :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-notificationconfigurations
        :stability: experimental
        '''
        result = self._values.get("notifications")
        return typing.cast(typing.Optional[typing.List[_NotificationConfiguration_14f038b9]], result)

    @builtins.property
    def notifications_topic(self) -> typing.Optional[_ITopic_465e36b9]:
        '''(deprecated) SNS topic to send notifications about fleet changes.

        :default: - No fleet change notifications will be sent.

        :deprecated: use ``notifications``

        :stability: deprecated
        '''
        result = self._values.get("notifications_topic")
        return typing.cast(typing.Optional[_ITopic_465e36b9], result)

    @builtins.property
    def replacing_update_min_successful_instances_percent(
        self,
    ) -> typing.Optional[jsii.Number]:
        '''(deprecated) Configuration for replacing updates.

        Only used if updateType == UpdateType.ReplacingUpdate. Specifies how
        many instances must signal success for the update to succeed.

        :default: minSuccessfulInstancesPercent

        :deprecated: Use ``signals`` instead

        :stability: deprecated
        '''
        result = self._values.get("replacing_update_min_successful_instances_percent")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def resource_signal_count(self) -> typing.Optional[jsii.Number]:
        '''(deprecated) How many ResourceSignal calls CloudFormation expects before the resource is considered created.

        :default: 1 if resourceSignalTimeout is set, 0 otherwise

        :deprecated: Use ``signals`` instead.

        :stability: deprecated
        '''
        result = self._values.get("resource_signal_count")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def resource_signal_timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(deprecated) The length of time to wait for the resourceSignalCount.

        The maximum value is 43200 (12 hours).

        :default: Duration.minutes(5) if resourceSignalCount is set, N/A otherwise

        :deprecated: Use ``signals`` instead.

        :stability: deprecated
        '''
        result = self._values.get("resource_signal_timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def rolling_update_configuration(
        self,
    ) -> typing.Optional[_RollingUpdateConfiguration_7b14e30c]:
        '''(deprecated) Configuration for rolling updates.

        Only used if updateType == UpdateType.RollingUpdate.

        :default: - RollingUpdateConfiguration with defaults.

        :deprecated: Use ``updatePolicy`` instead

        :stability: deprecated
        '''
        result = self._values.get("rolling_update_configuration")
        return typing.cast(typing.Optional[_RollingUpdateConfiguration_7b14e30c], result)

    @builtins.property
    def signals(self) -> typing.Optional[_Signals_896b8d51]:
        '''(experimental) Configure waiting for signals during deployment.

        Use this to pause the CloudFormation deployment to wait for the instances
        in the AutoScalingGroup to report successful startup during
        creation and updates. The UserData script needs to invoke ``cfn-signal``
        with a success or failure code after it is done setting up the instance.

        Without waiting for signals, the CloudFormation deployment will proceed as
        soon as the AutoScalingGroup has been created or updated but before the
        instances in the group have been started.

        For example, to have instances wait for an Elastic Load Balancing health check before
        they signal success, add a health-check verification by using the
        cfn-init helper script. For an example, see the verify_instance_health
        command in the Auto Scaling rolling updates sample template:

        https://github.com/awslabs/aws-cloudformation-templates/blob/master/aws/services/AutoScaling/AutoScalingRollingUpdates.yaml

        :default: - Do not wait for signals

        :stability: experimental
        '''
        result = self._values.get("signals")
        return typing.cast(typing.Optional[_Signals_896b8d51], result)

    @builtins.property
    def spot_price(self) -> typing.Optional[builtins.str]:
        '''(experimental) The maximum hourly price (in USD) to be paid for any Spot Instance launched to fulfill the request.

        Spot Instances are
        launched when the price you specify exceeds the current Spot market price.

        :default: none

        :stability: experimental
        '''
        result = self._values.get("spot_price")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def update_policy(self) -> typing.Optional[_UpdatePolicy_ffeec124]:
        '''(experimental) What to do when an AutoScalingGroup's instance configuration is changed.

        This is applied when any of the settings on the ASG are changed that
        affect how the instances should be created (VPC, instance type, startup
        scripts, etc.). It indicates how the existing instances should be
        replaced with new instances matching the new config. By default, nothing
        is done and only new instances are launched with the new config.

        :default: - ``UpdatePolicy.rollingUpdate()`` if using ``init``, ``UpdatePolicy.none()`` otherwise

        :stability: experimental
        '''
        result = self._values.get("update_policy")
        return typing.cast(typing.Optional[_UpdatePolicy_ffeec124], result)

    @builtins.property
    def update_type(self) -> typing.Optional[_UpdateType_1d8acce6]:
        '''(deprecated) What to do when an AutoScalingGroup's instance configuration is changed.

        This is applied when any of the settings on the ASG are changed that
        affect how the instances should be created (VPC, instance type, startup
        scripts, etc.). It indicates how the existing instances should be
        replaced with new instances matching the new config. By default, nothing
        is done and only new instances are launched with the new config.

        :default: UpdateType.None

        :deprecated: Use ``updatePolicy`` instead

        :stability: deprecated
        '''
        result = self._values.get("update_type")
        return typing.cast(typing.Optional[_UpdateType_1d8acce6], result)

    @builtins.property
    def vpc_subnets(self) -> typing.Optional[_SubnetSelection_1284e62c]:
        '''(experimental) Where to place instances within the VPC.

        :default: - All Private subnets.

        :stability: experimental
        '''
        result = self._values.get("vpc_subnets")
        return typing.cast(typing.Optional[_SubnetSelection_1284e62c], result)

    @builtins.property
    def instance_type(self) -> _InstanceType_072ad323:
        '''(experimental) The EC2 instance type to use when launching instances into the AutoScalingGroup.

        :stability: experimental
        '''
        result = self._values.get("instance_type")
        assert result is not None, "Required property 'instance_type' is missing"
        return typing.cast(_InstanceType_072ad323, result)

    @builtins.property
    def machine_image(self) -> typing.Optional[_IMachineImage_45d3238d]:
        '''(experimental) The ECS-optimized AMI variant to use.

        For more information, see
        `Amazon ECS-optimized AMIs <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html>`_.
        You must define either ``machineImage`` or ``machineImageType``, not both.

        :default: - Amazon Linux 2

        :stability: experimental
        '''
        result = self._values.get("machine_image")
        return typing.cast(typing.Optional[_IMachineImage_45d3238d], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AddCapacityOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_ecs.AmiHardwareType")
class AmiHardwareType(enum.Enum):
    '''(experimental) The ECS-optimized AMI variant to use.

    For more information, see
    `Amazon ECS-optimized AMIs <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html>`_.

    :stability: experimental
    '''

    STANDARD = "STANDARD"
    '''(experimental) Use the standard Amazon ECS-optimized AMI.

    :stability: experimental
    '''
    GPU = "GPU"
    '''(experimental) Use the Amazon ECS GPU-optimized AMI.

    :stability: experimental
    '''
    ARM = "ARM"
    '''(experimental) Use the Amazon ECS-optimized Amazon Linux 2 (arm64) AMI.

    :stability: experimental
    '''


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.AppMeshProxyConfigurationConfigProps",
    jsii_struct_bases=[],
    name_mapping={"container_name": "containerName", "properties": "properties"},
)
class AppMeshProxyConfigurationConfigProps:
    def __init__(
        self,
        *,
        container_name: builtins.str,
        properties: "AppMeshProxyConfigurationProps",
    ) -> None:
        '''(experimental) The configuration to use when setting an App Mesh proxy configuration.

        :param container_name: (experimental) The name of the container that will serve as the App Mesh proxy.
        :param properties: (experimental) The set of network configuration parameters to provide the Container Network Interface (CNI) plugin.

        :stability: experimental
        '''
        if isinstance(properties, dict):
            properties = AppMeshProxyConfigurationProps(**properties)
        self._values: typing.Dict[str, typing.Any] = {
            "container_name": container_name,
            "properties": properties,
        }

    @builtins.property
    def container_name(self) -> builtins.str:
        '''(experimental) The name of the container that will serve as the App Mesh proxy.

        :stability: experimental
        '''
        result = self._values.get("container_name")
        assert result is not None, "Required property 'container_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def properties(self) -> "AppMeshProxyConfigurationProps":
        '''(experimental) The set of network configuration parameters to provide the Container Network Interface (CNI) plugin.

        :stability: experimental
        '''
        result = self._values.get("properties")
        assert result is not None, "Required property 'properties' is missing"
        return typing.cast("AppMeshProxyConfigurationProps", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AppMeshProxyConfigurationConfigProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.AppMeshProxyConfigurationProps",
    jsii_struct_bases=[],
    name_mapping={
        "app_ports": "appPorts",
        "proxy_egress_port": "proxyEgressPort",
        "proxy_ingress_port": "proxyIngressPort",
        "egress_ignored_i_ps": "egressIgnoredIPs",
        "egress_ignored_ports": "egressIgnoredPorts",
        "ignored_gid": "ignoredGID",
        "ignored_uid": "ignoredUID",
    },
)
class AppMeshProxyConfigurationProps:
    def __init__(
        self,
        *,
        app_ports: typing.Sequence[jsii.Number],
        proxy_egress_port: jsii.Number,
        proxy_ingress_port: jsii.Number,
        egress_ignored_i_ps: typing.Optional[typing.Sequence[builtins.str]] = None,
        egress_ignored_ports: typing.Optional[typing.Sequence[jsii.Number]] = None,
        ignored_gid: typing.Optional[jsii.Number] = None,
        ignored_uid: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''(experimental) Interface for setting the properties of proxy configuration.

        :param app_ports: (experimental) The list of ports that the application uses. Network traffic to these ports is forwarded to the ProxyIngressPort and ProxyEgressPort.
        :param proxy_egress_port: (experimental) Specifies the port that outgoing traffic from the AppPorts is directed to.
        :param proxy_ingress_port: (experimental) Specifies the port that incoming traffic to the AppPorts is directed to.
        :param egress_ignored_i_ps: (experimental) The egress traffic going to these specified IP addresses is ignored and not redirected to the ProxyEgressPort. It can be an empty list.
        :param egress_ignored_ports: (experimental) The egress traffic going to these specified ports is ignored and not redirected to the ProxyEgressPort. It can be an empty list.
        :param ignored_gid: (experimental) The group ID (GID) of the proxy container as defined by the user parameter in a container definition. This is used to ensure the proxy ignores its own traffic. If IgnoredUID is specified, this field can be empty.
        :param ignored_uid: (experimental) The user ID (UID) of the proxy container as defined by the user parameter in a container definition. This is used to ensure the proxy ignores its own traffic. If IgnoredGID is specified, this field can be empty.

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "app_ports": app_ports,
            "proxy_egress_port": proxy_egress_port,
            "proxy_ingress_port": proxy_ingress_port,
        }
        if egress_ignored_i_ps is not None:
            self._values["egress_ignored_i_ps"] = egress_ignored_i_ps
        if egress_ignored_ports is not None:
            self._values["egress_ignored_ports"] = egress_ignored_ports
        if ignored_gid is not None:
            self._values["ignored_gid"] = ignored_gid
        if ignored_uid is not None:
            self._values["ignored_uid"] = ignored_uid

    @builtins.property
    def app_ports(self) -> typing.List[jsii.Number]:
        '''(experimental) The list of ports that the application uses.

        Network traffic to these ports is forwarded to the ProxyIngressPort and ProxyEgressPort.

        :stability: experimental
        '''
        result = self._values.get("app_ports")
        assert result is not None, "Required property 'app_ports' is missing"
        return typing.cast(typing.List[jsii.Number], result)

    @builtins.property
    def proxy_egress_port(self) -> jsii.Number:
        '''(experimental) Specifies the port that outgoing traffic from the AppPorts is directed to.

        :stability: experimental
        '''
        result = self._values.get("proxy_egress_port")
        assert result is not None, "Required property 'proxy_egress_port' is missing"
        return typing.cast(jsii.Number, result)

    @builtins.property
    def proxy_ingress_port(self) -> jsii.Number:
        '''(experimental) Specifies the port that incoming traffic to the AppPorts is directed to.

        :stability: experimental
        '''
        result = self._values.get("proxy_ingress_port")
        assert result is not None, "Required property 'proxy_ingress_port' is missing"
        return typing.cast(jsii.Number, result)

    @builtins.property
    def egress_ignored_i_ps(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) The egress traffic going to these specified IP addresses is ignored and not redirected to the ProxyEgressPort.

        It can be an empty list.

        :stability: experimental
        '''
        result = self._values.get("egress_ignored_i_ps")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def egress_ignored_ports(self) -> typing.Optional[typing.List[jsii.Number]]:
        '''(experimental) The egress traffic going to these specified ports is ignored and not redirected to the ProxyEgressPort.

        It can be an empty list.

        :stability: experimental
        '''
        result = self._values.get("egress_ignored_ports")
        return typing.cast(typing.Optional[typing.List[jsii.Number]], result)

    @builtins.property
    def ignored_gid(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The group ID (GID) of the proxy container as defined by the user parameter in a container definition.

        This is used to ensure the proxy ignores its own traffic. If IgnoredUID is specified, this field can be empty.

        :stability: experimental
        '''
        result = self._values.get("ignored_gid")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def ignored_uid(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The user ID (UID) of the proxy container as defined by the user parameter in a container definition.

        This is used to ensure the proxy ignores its own traffic. If IgnoredGID is specified, this field can be empty.

        :stability: experimental
        '''
        result = self._values.get("ignored_uid")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AppMeshProxyConfigurationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AsgCapacityProvider(
    _Construct_e78e779f,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_ecs.AsgCapacityProvider",
):
    '''(experimental) An Auto Scaling Group Capacity Provider.

    This allows an ECS cluster to target
    a specific EC2 Auto Scaling Group for the placement of tasks. Optionally (and
    recommended), ECS can manage the number of instances in the ASG to fit the
    tasks, and can ensure that instances are not prematurely terminated while
    there are still tasks running on them.

    :stability: experimental
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        auto_scaling_group: _IAutoScalingGroup_352723c6,
        capacity_provider_name: typing.Optional[builtins.str] = None,
        enable_managed_scaling: typing.Optional[builtins.bool] = None,
        enable_managed_termination_protection: typing.Optional[builtins.bool] = None,
        maximum_scaling_step_size: typing.Optional[jsii.Number] = None,
        minimum_scaling_step_size: typing.Optional[jsii.Number] = None,
        target_capacity_percent: typing.Optional[jsii.Number] = None,
        can_containers_access_instance_role: typing.Optional[builtins.bool] = None,
        machine_image_type: typing.Optional["MachineImageType"] = None,
        spot_instance_draining: typing.Optional[builtins.bool] = None,
        task_drain_time: typing.Optional[_Duration_070aa057] = None,
        topic_encryption_key: typing.Optional[_IKey_36930160] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param auto_scaling_group: (experimental) The autoscaling group to add as a Capacity Provider.
        :param capacity_provider_name: (experimental) The name for the capacity provider. Default: CloudFormation-generated name
        :param enable_managed_scaling: (experimental) Whether to enable managed scaling. Default: true
        :param enable_managed_termination_protection: (experimental) Whether to enable managed termination protection. Default: true
        :param maximum_scaling_step_size: (experimental) Maximum scaling step size. In most cases this should be left alone. Default: 1000
        :param minimum_scaling_step_size: (experimental) Minimum scaling step size. In most cases this should be left alone. Default: 1
        :param target_capacity_percent: (experimental) Target capacity percent. In most cases this should be left alone. Default: 100
        :param can_containers_access_instance_role: (experimental) Specifies whether the containers can access the container instance role. Default: false
        :param machine_image_type: (experimental) Specify the machine image type. Default: MachineImageType.AMAZON_LINUX_2
        :param spot_instance_draining: (experimental) Specify whether to enable Automated Draining for Spot Instances running Amazon ECS Services. For more information, see `Using Spot Instances <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container-instance-spot.html>`_. Default: false
        :param task_drain_time: (deprecated) The time period to wait before force terminating an instance that is draining. This creates a Lambda function that is used by a lifecycle hook for the AutoScalingGroup that will delay instance termination until all ECS tasks have drained from the instance. Set to 0 to disable task draining. Set to 0 to disable task draining. Default: Duration.minutes(5)
        :param topic_encryption_key: (experimental) If {@link AddAutoScalingGroupCapacityOptions.taskDrainTime} is non-zero, then the ECS cluster creates an SNS Topic to as part of a system to drain instances of tasks when the instance is being shut down. If this property is provided, then this key will be used to encrypt the contents of that SNS Topic. See `SNS Data Encryption <https://docs.aws.amazon.com/sns/latest/dg/sns-data-encryption.html>`_ for more information. Default: The SNS Topic will not be encrypted.

        :stability: experimental
        '''
        props = AsgCapacityProviderProps(
            auto_scaling_group=auto_scaling_group,
            capacity_provider_name=capacity_provider_name,
            enable_managed_scaling=enable_managed_scaling,
            enable_managed_termination_protection=enable_managed_termination_protection,
            maximum_scaling_step_size=maximum_scaling_step_size,
            minimum_scaling_step_size=minimum_scaling_step_size,
            target_capacity_percent=target_capacity_percent,
            can_containers_access_instance_role=can_containers_access_instance_role,
            machine_image_type=machine_image_type,
            spot_instance_draining=spot_instance_draining,
            task_drain_time=task_drain_time,
            topic_encryption_key=topic_encryption_key,
        )

        jsii.create(AsgCapacityProvider, self, [scope, id, props])

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="autoScalingGroup")
    def auto_scaling_group(self) -> _AutoScalingGroup_604a934f:
        '''(experimental) Auto Scaling Group.

        :stability: experimental
        '''
        return typing.cast(_AutoScalingGroup_604a934f, jsii.get(self, "autoScalingGroup"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="capacityProviderName")
    def capacity_provider_name(self) -> builtins.str:
        '''(experimental) Capacity provider name.

        :default: Chosen by CloudFormation

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "capacityProviderName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="enableManagedTerminationProtection")
    def enable_managed_termination_protection(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether managed termination protection is enabled.

        :stability: experimental
        '''
        return typing.cast(typing.Optional[builtins.bool], jsii.get(self, "enableManagedTerminationProtection"))


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.AsgCapacityProviderProps",
    jsii_struct_bases=[AddAutoScalingGroupCapacityOptions],
    name_mapping={
        "can_containers_access_instance_role": "canContainersAccessInstanceRole",
        "machine_image_type": "machineImageType",
        "spot_instance_draining": "spotInstanceDraining",
        "task_drain_time": "taskDrainTime",
        "topic_encryption_key": "topicEncryptionKey",
        "auto_scaling_group": "autoScalingGroup",
        "capacity_provider_name": "capacityProviderName",
        "enable_managed_scaling": "enableManagedScaling",
        "enable_managed_termination_protection": "enableManagedTerminationProtection",
        "maximum_scaling_step_size": "maximumScalingStepSize",
        "minimum_scaling_step_size": "minimumScalingStepSize",
        "target_capacity_percent": "targetCapacityPercent",
    },
)
class AsgCapacityProviderProps(AddAutoScalingGroupCapacityOptions):
    def __init__(
        self,
        *,
        can_containers_access_instance_role: typing.Optional[builtins.bool] = None,
        machine_image_type: typing.Optional["MachineImageType"] = None,
        spot_instance_draining: typing.Optional[builtins.bool] = None,
        task_drain_time: typing.Optional[_Duration_070aa057] = None,
        topic_encryption_key: typing.Optional[_IKey_36930160] = None,
        auto_scaling_group: _IAutoScalingGroup_352723c6,
        capacity_provider_name: typing.Optional[builtins.str] = None,
        enable_managed_scaling: typing.Optional[builtins.bool] = None,
        enable_managed_termination_protection: typing.Optional[builtins.bool] = None,
        maximum_scaling_step_size: typing.Optional[jsii.Number] = None,
        minimum_scaling_step_size: typing.Optional[jsii.Number] = None,
        target_capacity_percent: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''(experimental) The options for creating an Auto Scaling Group Capacity Provider.

        :param can_containers_access_instance_role: (experimental) Specifies whether the containers can access the container instance role. Default: false
        :param machine_image_type: (experimental) Specify the machine image type. Default: MachineImageType.AMAZON_LINUX_2
        :param spot_instance_draining: (experimental) Specify whether to enable Automated Draining for Spot Instances running Amazon ECS Services. For more information, see `Using Spot Instances <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container-instance-spot.html>`_. Default: false
        :param task_drain_time: (deprecated) The time period to wait before force terminating an instance that is draining. This creates a Lambda function that is used by a lifecycle hook for the AutoScalingGroup that will delay instance termination until all ECS tasks have drained from the instance. Set to 0 to disable task draining. Set to 0 to disable task draining. Default: Duration.minutes(5)
        :param topic_encryption_key: (experimental) If {@link AddAutoScalingGroupCapacityOptions.taskDrainTime} is non-zero, then the ECS cluster creates an SNS Topic to as part of a system to drain instances of tasks when the instance is being shut down. If this property is provided, then this key will be used to encrypt the contents of that SNS Topic. See `SNS Data Encryption <https://docs.aws.amazon.com/sns/latest/dg/sns-data-encryption.html>`_ for more information. Default: The SNS Topic will not be encrypted.
        :param auto_scaling_group: (experimental) The autoscaling group to add as a Capacity Provider.
        :param capacity_provider_name: (experimental) The name for the capacity provider. Default: CloudFormation-generated name
        :param enable_managed_scaling: (experimental) Whether to enable managed scaling. Default: true
        :param enable_managed_termination_protection: (experimental) Whether to enable managed termination protection. Default: true
        :param maximum_scaling_step_size: (experimental) Maximum scaling step size. In most cases this should be left alone. Default: 1000
        :param minimum_scaling_step_size: (experimental) Minimum scaling step size. In most cases this should be left alone. Default: 1
        :param target_capacity_percent: (experimental) Target capacity percent. In most cases this should be left alone. Default: 100

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "auto_scaling_group": auto_scaling_group,
        }
        if can_containers_access_instance_role is not None:
            self._values["can_containers_access_instance_role"] = can_containers_access_instance_role
        if machine_image_type is not None:
            self._values["machine_image_type"] = machine_image_type
        if spot_instance_draining is not None:
            self._values["spot_instance_draining"] = spot_instance_draining
        if task_drain_time is not None:
            self._values["task_drain_time"] = task_drain_time
        if topic_encryption_key is not None:
            self._values["topic_encryption_key"] = topic_encryption_key
        if capacity_provider_name is not None:
            self._values["capacity_provider_name"] = capacity_provider_name
        if enable_managed_scaling is not None:
            self._values["enable_managed_scaling"] = enable_managed_scaling
        if enable_managed_termination_protection is not None:
            self._values["enable_managed_termination_protection"] = enable_managed_termination_protection
        if maximum_scaling_step_size is not None:
            self._values["maximum_scaling_step_size"] = maximum_scaling_step_size
        if minimum_scaling_step_size is not None:
            self._values["minimum_scaling_step_size"] = minimum_scaling_step_size
        if target_capacity_percent is not None:
            self._values["target_capacity_percent"] = target_capacity_percent

    @builtins.property
    def can_containers_access_instance_role(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specifies whether the containers can access the container instance role.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("can_containers_access_instance_role")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def machine_image_type(self) -> typing.Optional["MachineImageType"]:
        '''(experimental) Specify the machine image type.

        :default: MachineImageType.AMAZON_LINUX_2

        :stability: experimental
        '''
        result = self._values.get("machine_image_type")
        return typing.cast(typing.Optional["MachineImageType"], result)

    @builtins.property
    def spot_instance_draining(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specify whether to enable Automated Draining for Spot Instances running Amazon ECS Services.

        For more information, see `Using Spot Instances <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container-instance-spot.html>`_.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("spot_instance_draining")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def task_drain_time(self) -> typing.Optional[_Duration_070aa057]:
        '''(deprecated) The time period to wait before force terminating an instance that is draining.

        This creates a Lambda function that is used by a lifecycle hook for the
        AutoScalingGroup that will delay instance termination until all ECS tasks
        have drained from the instance. Set to 0 to disable task draining.

        Set to 0 to disable task draining.

        :default: Duration.minutes(5)

        :deprecated: The lifecycle draining hook is not configured if using the EC2 Capacity Provider. Enable managed termination protection instead.

        :stability: deprecated
        '''
        result = self._values.get("task_drain_time")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def topic_encryption_key(self) -> typing.Optional[_IKey_36930160]:
        '''(experimental) If {@link AddAutoScalingGroupCapacityOptions.taskDrainTime} is non-zero, then the ECS cluster creates an SNS Topic to as part of a system to drain instances of tasks when the instance is being shut down. If this property is provided, then this key will be used to encrypt the contents of that SNS Topic. See `SNS Data Encryption <https://docs.aws.amazon.com/sns/latest/dg/sns-data-encryption.html>`_ for more information.

        :default: The SNS Topic will not be encrypted.

        :stability: experimental
        '''
        result = self._values.get("topic_encryption_key")
        return typing.cast(typing.Optional[_IKey_36930160], result)

    @builtins.property
    def auto_scaling_group(self) -> _IAutoScalingGroup_352723c6:
        '''(experimental) The autoscaling group to add as a Capacity Provider.

        :stability: experimental
        '''
        result = self._values.get("auto_scaling_group")
        assert result is not None, "Required property 'auto_scaling_group' is missing"
        return typing.cast(_IAutoScalingGroup_352723c6, result)

    @builtins.property
    def capacity_provider_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name for the capacity provider.

        :default: CloudFormation-generated name

        :stability: experimental
        '''
        result = self._values.get("capacity_provider_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def enable_managed_scaling(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether to enable managed scaling.

        :default: true

        :stability: experimental
        '''
        result = self._values.get("enable_managed_scaling")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def enable_managed_termination_protection(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether to enable managed termination protection.

        :default: true

        :stability: experimental
        '''
        result = self._values.get("enable_managed_termination_protection")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def maximum_scaling_step_size(self) -> typing.Optional[jsii.Number]:
        '''(experimental) Maximum scaling step size.

        In most cases this should be left alone.

        :default: 1000

        :stability: experimental
        '''
        result = self._values.get("maximum_scaling_step_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def minimum_scaling_step_size(self) -> typing.Optional[jsii.Number]:
        '''(experimental) Minimum scaling step size.

        In most cases this should be left alone.

        :default: 1

        :stability: experimental
        '''
        result = self._values.get("minimum_scaling_step_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def target_capacity_percent(self) -> typing.Optional[jsii.Number]:
        '''(experimental) Target capacity percent.

        In most cases this should be left alone.

        :default: 100

        :stability: experimental
        '''
        result = self._values.get("target_capacity_percent")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AsgCapacityProviderProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.AssetImageProps",
    jsii_struct_bases=[_DockerImageAssetOptions_6764a0de],
    name_mapping={
        "exclude": "exclude",
        "follow": "follow",
        "ignore_mode": "ignoreMode",
        "extra_hash": "extraHash",
        "follow_symlinks": "followSymlinks",
        "build_args": "buildArgs",
        "file": "file",
        "repository_name": "repositoryName",
        "target": "target",
    },
)
class AssetImageProps(_DockerImageAssetOptions_6764a0de):
    def __init__(
        self,
        *,
        exclude: typing.Optional[typing.Sequence[builtins.str]] = None,
        follow: typing.Optional[_FollowMode_98b05cc5] = None,
        ignore_mode: typing.Optional[_IgnoreMode_31d8bf46] = None,
        extra_hash: typing.Optional[builtins.str] = None,
        follow_symlinks: typing.Optional[_SymlinkFollowMode_abf4527a] = None,
        build_args: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        file: typing.Optional[builtins.str] = None,
        repository_name: typing.Optional[builtins.str] = None,
        target: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) The properties for building an AssetImage.

        :param exclude: (experimental) Glob patterns to exclude from the copy. Default: - nothing is excluded
        :param follow: (deprecated) A strategy for how to handle symlinks. Default: Never
        :param ignore_mode: (experimental) The ignore behavior to use for exclude patterns. Default: IgnoreMode.GLOB
        :param extra_hash: (experimental) Extra information to encode into the fingerprint (e.g. build instructions and other inputs). Default: - hash is only based on source content
        :param follow_symlinks: (experimental) A strategy for how to handle symlinks. Default: SymlinkFollowMode.NEVER
        :param build_args: (experimental) Build args to pass to the ``docker build`` command. Since Docker build arguments are resolved before deployment, keys and values cannot refer to unresolved tokens (such as ``lambda.functionArn`` or ``queue.queueUrl``). Default: - no build args are passed
        :param file: (experimental) Path to the Dockerfile (relative to the directory). Default: 'Dockerfile'
        :param repository_name: (deprecated) ECR repository name. Specify this property if you need to statically address the image, e.g. from a Kubernetes Pod. Note, this is only the repository name, without the registry and the tag parts. Default: - the default ECR repository for CDK assets
        :param target: (experimental) Docker target to build to. Default: - no target

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if exclude is not None:
            self._values["exclude"] = exclude
        if follow is not None:
            self._values["follow"] = follow
        if ignore_mode is not None:
            self._values["ignore_mode"] = ignore_mode
        if extra_hash is not None:
            self._values["extra_hash"] = extra_hash
        if follow_symlinks is not None:
            self._values["follow_symlinks"] = follow_symlinks
        if build_args is not None:
            self._values["build_args"] = build_args
        if file is not None:
            self._values["file"] = file
        if repository_name is not None:
            self._values["repository_name"] = repository_name
        if target is not None:
            self._values["target"] = target

    @builtins.property
    def exclude(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) Glob patterns to exclude from the copy.

        :default: - nothing is excluded

        :stability: experimental
        '''
        result = self._values.get("exclude")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def follow(self) -> typing.Optional[_FollowMode_98b05cc5]:
        '''(deprecated) A strategy for how to handle symlinks.

        :default: Never

        :deprecated: use ``followSymlinks`` instead

        :stability: deprecated
        '''
        result = self._values.get("follow")
        return typing.cast(typing.Optional[_FollowMode_98b05cc5], result)

    @builtins.property
    def ignore_mode(self) -> typing.Optional[_IgnoreMode_31d8bf46]:
        '''(experimental) The ignore behavior to use for exclude patterns.

        :default: IgnoreMode.GLOB

        :stability: experimental
        '''
        result = self._values.get("ignore_mode")
        return typing.cast(typing.Optional[_IgnoreMode_31d8bf46], result)

    @builtins.property
    def extra_hash(self) -> typing.Optional[builtins.str]:
        '''(experimental) Extra information to encode into the fingerprint (e.g. build instructions and other inputs).

        :default: - hash is only based on source content

        :stability: experimental
        '''
        result = self._values.get("extra_hash")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def follow_symlinks(self) -> typing.Optional[_SymlinkFollowMode_abf4527a]:
        '''(experimental) A strategy for how to handle symlinks.

        :default: SymlinkFollowMode.NEVER

        :stability: experimental
        '''
        result = self._values.get("follow_symlinks")
        return typing.cast(typing.Optional[_SymlinkFollowMode_abf4527a], result)

    @builtins.property
    def build_args(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''(experimental) Build args to pass to the ``docker build`` command.

        Since Docker build arguments are resolved before deployment, keys and
        values cannot refer to unresolved tokens (such as ``lambda.functionArn`` or
        ``queue.queueUrl``).

        :default: - no build args are passed

        :stability: experimental
        '''
        result = self._values.get("build_args")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def file(self) -> typing.Optional[builtins.str]:
        '''(experimental) Path to the Dockerfile (relative to the directory).

        :default: 'Dockerfile'

        :stability: experimental
        '''
        result = self._values.get("file")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def repository_name(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ECR repository name.

        Specify this property if you need to statically address the image, e.g.
        from a Kubernetes Pod. Note, this is only the repository name, without the
        registry and the tag parts.

        :default: - the default ECR repository for CDK assets

        :deprecated:

        to control the location of docker image assets, please override
        ``Stack.addDockerImageAsset``. this feature will be removed in future
        releases.

        :stability: deprecated
        '''
        result = self._values.get("repository_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def target(self) -> typing.Optional[builtins.str]:
        '''(experimental) Docker target to build to.

        :default: - no target

        :stability: experimental
        '''
        result = self._values.get("target")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AssetImageProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.AssociateCloudMapServiceOptions",
    jsii_struct_bases=[],
    name_mapping={
        "service": "service",
        "container": "container",
        "container_port": "containerPort",
    },
)
class AssociateCloudMapServiceOptions:
    def __init__(
        self,
        *,
        service: _IService_66c1fbd2,
        container: typing.Optional["ContainerDefinition"] = None,
        container_port: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''(experimental) The options for using a cloudmap service.

        :param service: (experimental) The cloudmap service to register with.
        :param container: (experimental) The container to point to for a SRV record. Default: - the task definition's default container
        :param container_port: (experimental) The port to point to for a SRV record. Default: - the default port of the task definition's default container

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "service": service,
        }
        if container is not None:
            self._values["container"] = container
        if container_port is not None:
            self._values["container_port"] = container_port

    @builtins.property
    def service(self) -> _IService_66c1fbd2:
        '''(experimental) The cloudmap service to register with.

        :stability: experimental
        '''
        result = self._values.get("service")
        assert result is not None, "Required property 'service' is missing"
        return typing.cast(_IService_66c1fbd2, result)

    @builtins.property
    def container(self) -> typing.Optional["ContainerDefinition"]:
        '''(experimental) The container to point to for a SRV record.

        :default: - the task definition's default container

        :stability: experimental
        '''
        result = self._values.get("container")
        return typing.cast(typing.Optional["ContainerDefinition"], result)

    @builtins.property
    def container_port(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The port to point to for a SRV record.

        :default: - the default port of the task definition's default container

        :stability: experimental
        '''
        result = self._values.get("container_port")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AssociateCloudMapServiceOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.AuthorizationConfig",
    jsii_struct_bases=[],
    name_mapping={"access_point_id": "accessPointId", "iam": "iam"},
)
class AuthorizationConfig:
    def __init__(
        self,
        *,
        access_point_id: typing.Optional[builtins.str] = None,
        iam: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) The authorization configuration details for the Amazon EFS file system.

        :param access_point_id: (experimental) The access point ID to use. If an access point is specified, the root directory value will be relative to the directory set for the access point. If specified, transit encryption must be enabled in the EFSVolumeConfiguration. Default: No id
        :param iam: (experimental) Whether or not to use the Amazon ECS task IAM role defined in a task definition when mounting the Amazon EFS file system. If enabled, transit encryption must be enabled in the EFSVolumeConfiguration. Valid values: ENABLED | DISABLED Default: If this parameter is omitted, the default value of DISABLED is used.

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if access_point_id is not None:
            self._values["access_point_id"] = access_point_id
        if iam is not None:
            self._values["iam"] = iam

    @builtins.property
    def access_point_id(self) -> typing.Optional[builtins.str]:
        '''(experimental) The access point ID to use.

        If an access point is specified, the root directory value will be
        relative to the directory set for the access point.
        If specified, transit encryption must be enabled in the EFSVolumeConfiguration.

        :default: No id

        :stability: experimental
        '''
        result = self._values.get("access_point_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def iam(self) -> typing.Optional[builtins.str]:
        '''(experimental) Whether or not to use the Amazon ECS task IAM role defined in a task definition when mounting the Amazon EFS file system.

        If enabled, transit encryption must be enabled in the EFSVolumeConfiguration.

        Valid values: ENABLED | DISABLED

        :default: If this parameter is omitted, the default value of DISABLED is used.

        :stability: experimental
        '''
        result = self._values.get("iam")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AuthorizationConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_ecs.AwsLogDriverMode")
class AwsLogDriverMode(enum.Enum):
    '''(experimental) awslogs provides two modes for delivering messages from the container to the log driver.

    :stability: experimental
    '''

    BLOCKING = "BLOCKING"
    '''(experimental) (default) direct, blocking delivery from container to driver.

    :stability: experimental
    '''
    NON_BLOCKING = "NON_BLOCKING"
    '''(experimental) The non-blocking message delivery mode prevents applications from blocking due to logging back pressure.

    Applications are likely to fail in unexpected ways when STDERR or STDOUT streams block.

    :stability: experimental
    '''


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.AwsLogDriverProps",
    jsii_struct_bases=[],
    name_mapping={
        "stream_prefix": "streamPrefix",
        "datetime_format": "datetimeFormat",
        "log_group": "logGroup",
        "log_retention": "logRetention",
        "mode": "mode",
        "multiline_pattern": "multilinePattern",
    },
)
class AwsLogDriverProps:
    def __init__(
        self,
        *,
        stream_prefix: builtins.str,
        datetime_format: typing.Optional[builtins.str] = None,
        log_group: typing.Optional[_ILogGroup_846e17a0] = None,
        log_retention: typing.Optional[_RetentionDays_6c560d31] = None,
        mode: typing.Optional[AwsLogDriverMode] = None,
        multiline_pattern: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) Specifies the awslogs log driver configuration options.

        :param stream_prefix: (experimental) Prefix for the log streams. The awslogs-stream-prefix option allows you to associate a log stream with the specified prefix, the container name, and the ID of the Amazon ECS task to which the container belongs. If you specify a prefix with this option, then the log stream takes the following format:: prefix-name/container-name/ecs-task-id
        :param datetime_format: (experimental) This option defines a multiline start pattern in Python strftime format. A log message consists of a line that matches the pattern and any following lines that don’t match the pattern. Thus the matched line is the delimiter between log messages. Default: - No multiline matching.
        :param log_group: (experimental) The log group to log to. Default: - A log group is automatically created.
        :param log_retention: (experimental) The number of days log events are kept in CloudWatch Logs when the log group is automatically created by this construct. Default: - Logs never expire.
        :param mode: (experimental) The delivery mode of log messages from the container to awslogs. Default: - AwsLogDriverMode.BLOCKING
        :param multiline_pattern: (experimental) This option defines a multiline start pattern using a regular expression. A log message consists of a line that matches the pattern and any following lines that don’t match the pattern. Thus the matched line is the delimiter between log messages. This option is ignored if datetimeFormat is also configured. Default: - No multiline matching.

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "stream_prefix": stream_prefix,
        }
        if datetime_format is not None:
            self._values["datetime_format"] = datetime_format
        if log_group is not None:
            self._values["log_group"] = log_group
        if log_retention is not None:
            self._values["log_retention"] = log_retention
        if mode is not None:
            self._values["mode"] = mode
        if multiline_pattern is not None:
            self._values["multiline_pattern"] = multiline_pattern

    @builtins.property
    def stream_prefix(self) -> builtins.str:
        '''(experimental) Prefix for the log streams.

        The awslogs-stream-prefix option allows you to associate a log stream
        with the specified prefix, the container name, and the ID of the Amazon
        ECS task to which the container belongs. If you specify a prefix with
        this option, then the log stream takes the following format::

            prefix-name/container-name/ecs-task-id

        :stability: experimental
        '''
        result = self._values.get("stream_prefix")
        assert result is not None, "Required property 'stream_prefix' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def datetime_format(self) -> typing.Optional[builtins.str]:
        '''(experimental) This option defines a multiline start pattern in Python strftime format.

        A log message consists of a line that matches the pattern and any
        following lines that don’t match the pattern. Thus the matched line is
        the delimiter between log messages.

        :default: - No multiline matching.

        :stability: experimental
        '''
        result = self._values.get("datetime_format")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def log_group(self) -> typing.Optional[_ILogGroup_846e17a0]:
        '''(experimental) The log group to log to.

        :default: - A log group is automatically created.

        :stability: experimental
        '''
        result = self._values.get("log_group")
        return typing.cast(typing.Optional[_ILogGroup_846e17a0], result)

    @builtins.property
    def log_retention(self) -> typing.Optional[_RetentionDays_6c560d31]:
        '''(experimental) The number of days log events are kept in CloudWatch Logs when the log group is automatically created by this construct.

        :default: - Logs never expire.

        :stability: experimental
        '''
        result = self._values.get("log_retention")
        return typing.cast(typing.Optional[_RetentionDays_6c560d31], result)

    @builtins.property
    def mode(self) -> typing.Optional[AwsLogDriverMode]:
        '''(experimental) The delivery mode of log messages from the container to awslogs.

        :default: - AwsLogDriverMode.BLOCKING

        :stability: experimental
        '''
        result = self._values.get("mode")
        return typing.cast(typing.Optional[AwsLogDriverMode], result)

    @builtins.property
    def multiline_pattern(self) -> typing.Optional[builtins.str]:
        '''(experimental) This option defines a multiline start pattern using a regular expression.

        A log message consists of a line that matches the pattern and any
        following lines that don’t match the pattern. Thus the matched line is
        the delimiter between log messages.

        This option is ignored if datetimeFormat is also configured.

        :default: - No multiline matching.

        :stability: experimental
        '''
        result = self._values.get("multiline_pattern")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AwsLogDriverProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.BaseLogDriverProps",
    jsii_struct_bases=[],
    name_mapping={
        "env": "env",
        "env_regex": "envRegex",
        "labels": "labels",
        "tag": "tag",
    },
)
class BaseLogDriverProps:
    def __init__(
        self,
        *,
        env: typing.Optional[typing.Sequence[builtins.str]] = None,
        env_regex: typing.Optional[builtins.str] = None,
        labels: typing.Optional[typing.Sequence[builtins.str]] = None,
        tag: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param env: (experimental) The env option takes an array of keys. If there is collision between label and env keys, the value of the env takes precedence. Adds additional fields to the extra attributes of a logging message. Default: - No env
        :param env_regex: (experimental) The env-regex option is similar to and compatible with env. Its value is a regular expression to match logging-related environment variables. It is used for advanced log tag options. Default: - No envRegex
        :param labels: (experimental) The labels option takes an array of keys. If there is collision between label and env keys, the value of the env takes precedence. Adds additional fields to the extra attributes of a logging message. Default: - No labels
        :param tag: (experimental) By default, Docker uses the first 12 characters of the container ID to tag log messages. Refer to the log tag option documentation for customizing the log tag format. Default: - The first 12 characters of the container ID

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if env is not None:
            self._values["env"] = env
        if env_regex is not None:
            self._values["env_regex"] = env_regex
        if labels is not None:
            self._values["labels"] = labels
        if tag is not None:
            self._values["tag"] = tag

    @builtins.property
    def env(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) The env option takes an array of keys.

        If there is collision between
        label and env keys, the value of the env takes precedence. Adds additional fields
        to the extra attributes of a logging message.

        :default: - No env

        :stability: experimental
        '''
        result = self._values.get("env")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def env_regex(self) -> typing.Optional[builtins.str]:
        '''(experimental) The env-regex option is similar to and compatible with env.

        Its value is a regular
        expression to match logging-related environment variables. It is used for advanced
        log tag options.

        :default: - No envRegex

        :stability: experimental
        '''
        result = self._values.get("env_regex")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def labels(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) The labels option takes an array of keys.

        If there is collision
        between label and env keys, the value of the env takes precedence. Adds additional
        fields to the extra attributes of a logging message.

        :default: - No labels

        :stability: experimental
        '''
        result = self._values.get("labels")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def tag(self) -> typing.Optional[builtins.str]:
        '''(experimental) By default, Docker uses the first 12 characters of the container ID to tag log messages.

        Refer to the log tag option documentation for customizing the
        log tag format.

        :default: - The first 12 characters of the container ID

        :stability: experimental
        '''
        result = self._values.get("tag")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BaseLogDriverProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.BaseServiceOptions",
    jsii_struct_bases=[],
    name_mapping={
        "cluster": "cluster",
        "capacity_provider_strategies": "capacityProviderStrategies",
        "circuit_breaker": "circuitBreaker",
        "cloud_map_options": "cloudMapOptions",
        "deployment_controller": "deploymentController",
        "desired_count": "desiredCount",
        "enable_ecs_managed_tags": "enableECSManagedTags",
        "health_check_grace_period": "healthCheckGracePeriod",
        "max_healthy_percent": "maxHealthyPercent",
        "min_healthy_percent": "minHealthyPercent",
        "propagate_tags": "propagateTags",
        "service_name": "serviceName",
    },
)
class BaseServiceOptions:
    def __init__(
        self,
        *,
        cluster: "ICluster",
        capacity_provider_strategies: typing.Optional[typing.Sequence["CapacityProviderStrategy"]] = None,
        circuit_breaker: typing.Optional["DeploymentCircuitBreaker"] = None,
        cloud_map_options: typing.Optional["CloudMapOptions"] = None,
        deployment_controller: typing.Optional["DeploymentController"] = None,
        desired_count: typing.Optional[jsii.Number] = None,
        enable_ecs_managed_tags: typing.Optional[builtins.bool] = None,
        health_check_grace_period: typing.Optional[_Duration_070aa057] = None,
        max_healthy_percent: typing.Optional[jsii.Number] = None,
        min_healthy_percent: typing.Optional[jsii.Number] = None,
        propagate_tags: typing.Optional["PropagatedTagSource"] = None,
        service_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) The properties for the base Ec2Service or FargateService service.

        :param cluster: (experimental) The name of the cluster that hosts the service.
        :param capacity_provider_strategies: (experimental) A list of Capacity Provider strategies used to place a service. Default: - undefined
        :param circuit_breaker: (experimental) Whether to enable the deployment circuit breaker. If this property is defined, circuit breaker will be implicitly enabled. Default: - disabled
        :param cloud_map_options: (experimental) The options for configuring an Amazon ECS service to use service discovery. Default: - AWS Cloud Map service discovery is not enabled.
        :param deployment_controller: (experimental) Specifies which deployment controller to use for the service. For more information, see `Amazon ECS Deployment Types <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-types.html>`_ Default: - Rolling update (ECS)
        :param desired_count: (experimental) The desired number of instantiations of the task definition to keep running on the service. Default: - When creating the service, default is 1; when updating the service, default uses the current task number.
        :param enable_ecs_managed_tags: (experimental) Specifies whether to enable Amazon ECS managed tags for the tasks within the service. For more information, see `Tagging Your Amazon ECS Resources <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html>`_ Default: false
        :param health_check_grace_period: (experimental) The period of time, in seconds, that the Amazon ECS service scheduler ignores unhealthy Elastic Load Balancing target health checks after a task has first started. Default: - defaults to 60 seconds if at least one load balancer is in-use and it is not already set
        :param max_healthy_percent: (experimental) The maximum number of tasks, specified as a percentage of the Amazon ECS service's DesiredCount value, that can run in a service during a deployment. Default: - 100 if daemon, otherwise 200
        :param min_healthy_percent: (experimental) The minimum number of tasks, specified as a percentage of the Amazon ECS service's DesiredCount value, that must continue to run and remain healthy during a deployment. Default: - 0 if daemon, otherwise 50
        :param propagate_tags: (experimental) Specifies whether to propagate the tags from the task definition or the service to the tasks in the service. Valid values are: PropagatedTagSource.SERVICE, PropagatedTagSource.TASK_DEFINITION or PropagatedTagSource.NONE Default: PropagatedTagSource.NONE
        :param service_name: (experimental) The name of the service. Default: - CloudFormation-generated name.

        :stability: experimental
        '''
        if isinstance(circuit_breaker, dict):
            circuit_breaker = DeploymentCircuitBreaker(**circuit_breaker)
        if isinstance(cloud_map_options, dict):
            cloud_map_options = CloudMapOptions(**cloud_map_options)
        if isinstance(deployment_controller, dict):
            deployment_controller = DeploymentController(**deployment_controller)
        self._values: typing.Dict[str, typing.Any] = {
            "cluster": cluster,
        }
        if capacity_provider_strategies is not None:
            self._values["capacity_provider_strategies"] = capacity_provider_strategies
        if circuit_breaker is not None:
            self._values["circuit_breaker"] = circuit_breaker
        if cloud_map_options is not None:
            self._values["cloud_map_options"] = cloud_map_options
        if deployment_controller is not None:
            self._values["deployment_controller"] = deployment_controller
        if desired_count is not None:
            self._values["desired_count"] = desired_count
        if enable_ecs_managed_tags is not None:
            self._values["enable_ecs_managed_tags"] = enable_ecs_managed_tags
        if health_check_grace_period is not None:
            self._values["health_check_grace_period"] = health_check_grace_period
        if max_healthy_percent is not None:
            self._values["max_healthy_percent"] = max_healthy_percent
        if min_healthy_percent is not None:
            self._values["min_healthy_percent"] = min_healthy_percent
        if propagate_tags is not None:
            self._values["propagate_tags"] = propagate_tags
        if service_name is not None:
            self._values["service_name"] = service_name

    @builtins.property
    def cluster(self) -> "ICluster":
        '''(experimental) The name of the cluster that hosts the service.

        :stability: experimental
        '''
        result = self._values.get("cluster")
        assert result is not None, "Required property 'cluster' is missing"
        return typing.cast("ICluster", result)

    @builtins.property
    def capacity_provider_strategies(
        self,
    ) -> typing.Optional[typing.List["CapacityProviderStrategy"]]:
        '''(experimental) A list of Capacity Provider strategies used to place a service.

        :default: - undefined

        :stability: experimental
        '''
        result = self._values.get("capacity_provider_strategies")
        return typing.cast(typing.Optional[typing.List["CapacityProviderStrategy"]], result)

    @builtins.property
    def circuit_breaker(self) -> typing.Optional["DeploymentCircuitBreaker"]:
        '''(experimental) Whether to enable the deployment circuit breaker.

        If this property is defined, circuit breaker will be implicitly
        enabled.

        :default: - disabled

        :stability: experimental
        '''
        result = self._values.get("circuit_breaker")
        return typing.cast(typing.Optional["DeploymentCircuitBreaker"], result)

    @builtins.property
    def cloud_map_options(self) -> typing.Optional["CloudMapOptions"]:
        '''(experimental) The options for configuring an Amazon ECS service to use service discovery.

        :default: - AWS Cloud Map service discovery is not enabled.

        :stability: experimental
        '''
        result = self._values.get("cloud_map_options")
        return typing.cast(typing.Optional["CloudMapOptions"], result)

    @builtins.property
    def deployment_controller(self) -> typing.Optional["DeploymentController"]:
        '''(experimental) Specifies which deployment controller to use for the service.

        For more information, see
        `Amazon ECS Deployment Types <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-types.html>`_

        :default: - Rolling update (ECS)

        :stability: experimental
        '''
        result = self._values.get("deployment_controller")
        return typing.cast(typing.Optional["DeploymentController"], result)

    @builtins.property
    def desired_count(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The desired number of instantiations of the task definition to keep running on the service.

        :default:

        - When creating the service, default is 1; when updating the service, default uses
        the current task number.

        :stability: experimental
        '''
        result = self._values.get("desired_count")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def enable_ecs_managed_tags(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specifies whether to enable Amazon ECS managed tags for the tasks within the service.

        For more information, see
        `Tagging Your Amazon ECS Resources <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html>`_

        :default: false

        :stability: experimental
        '''
        result = self._values.get("enable_ecs_managed_tags")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def health_check_grace_period(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) The period of time, in seconds, that the Amazon ECS service scheduler ignores unhealthy Elastic Load Balancing target health checks after a task has first started.

        :default: - defaults to 60 seconds if at least one load balancer is in-use and it is not already set

        :stability: experimental
        '''
        result = self._values.get("health_check_grace_period")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def max_healthy_percent(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The maximum number of tasks, specified as a percentage of the Amazon ECS service's DesiredCount value, that can run in a service during a deployment.

        :default: - 100 if daemon, otherwise 200

        :stability: experimental
        '''
        result = self._values.get("max_healthy_percent")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def min_healthy_percent(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The minimum number of tasks, specified as a percentage of the Amazon ECS service's DesiredCount value, that must continue to run and remain healthy during a deployment.

        :default: - 0 if daemon, otherwise 50

        :stability: experimental
        '''
        result = self._values.get("min_healthy_percent")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def propagate_tags(self) -> typing.Optional["PropagatedTagSource"]:
        '''(experimental) Specifies whether to propagate the tags from the task definition or the service to the tasks in the service.

        Valid values are: PropagatedTagSource.SERVICE, PropagatedTagSource.TASK_DEFINITION or PropagatedTagSource.NONE

        :default: PropagatedTagSource.NONE

        :stability: experimental
        '''
        result = self._values.get("propagate_tags")
        return typing.cast(typing.Optional["PropagatedTagSource"], result)

    @builtins.property
    def service_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name of the service.

        :default: - CloudFormation-generated name.

        :stability: experimental
        '''
        result = self._values.get("service_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BaseServiceOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.BaseServiceProps",
    jsii_struct_bases=[BaseServiceOptions],
    name_mapping={
        "cluster": "cluster",
        "capacity_provider_strategies": "capacityProviderStrategies",
        "circuit_breaker": "circuitBreaker",
        "cloud_map_options": "cloudMapOptions",
        "deployment_controller": "deploymentController",
        "desired_count": "desiredCount",
        "enable_ecs_managed_tags": "enableECSManagedTags",
        "health_check_grace_period": "healthCheckGracePeriod",
        "max_healthy_percent": "maxHealthyPercent",
        "min_healthy_percent": "minHealthyPercent",
        "propagate_tags": "propagateTags",
        "service_name": "serviceName",
        "launch_type": "launchType",
    },
)
class BaseServiceProps(BaseServiceOptions):
    def __init__(
        self,
        *,
        cluster: "ICluster",
        capacity_provider_strategies: typing.Optional[typing.Sequence["CapacityProviderStrategy"]] = None,
        circuit_breaker: typing.Optional["DeploymentCircuitBreaker"] = None,
        cloud_map_options: typing.Optional["CloudMapOptions"] = None,
        deployment_controller: typing.Optional["DeploymentController"] = None,
        desired_count: typing.Optional[jsii.Number] = None,
        enable_ecs_managed_tags: typing.Optional[builtins.bool] = None,
        health_check_grace_period: typing.Optional[_Duration_070aa057] = None,
        max_healthy_percent: typing.Optional[jsii.Number] = None,
        min_healthy_percent: typing.Optional[jsii.Number] = None,
        propagate_tags: typing.Optional["PropagatedTagSource"] = None,
        service_name: typing.Optional[builtins.str] = None,
        launch_type: "LaunchType",
    ) -> None:
        '''(experimental) Complete base service properties that are required to be supplied by the implementation of the BaseService class.

        :param cluster: (experimental) The name of the cluster that hosts the service.
        :param capacity_provider_strategies: (experimental) A list of Capacity Provider strategies used to place a service. Default: - undefined
        :param circuit_breaker: (experimental) Whether to enable the deployment circuit breaker. If this property is defined, circuit breaker will be implicitly enabled. Default: - disabled
        :param cloud_map_options: (experimental) The options for configuring an Amazon ECS service to use service discovery. Default: - AWS Cloud Map service discovery is not enabled.
        :param deployment_controller: (experimental) Specifies which deployment controller to use for the service. For more information, see `Amazon ECS Deployment Types <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-types.html>`_ Default: - Rolling update (ECS)
        :param desired_count: (experimental) The desired number of instantiations of the task definition to keep running on the service. Default: - When creating the service, default is 1; when updating the service, default uses the current task number.
        :param enable_ecs_managed_tags: (experimental) Specifies whether to enable Amazon ECS managed tags for the tasks within the service. For more information, see `Tagging Your Amazon ECS Resources <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html>`_ Default: false
        :param health_check_grace_period: (experimental) The period of time, in seconds, that the Amazon ECS service scheduler ignores unhealthy Elastic Load Balancing target health checks after a task has first started. Default: - defaults to 60 seconds if at least one load balancer is in-use and it is not already set
        :param max_healthy_percent: (experimental) The maximum number of tasks, specified as a percentage of the Amazon ECS service's DesiredCount value, that can run in a service during a deployment. Default: - 100 if daemon, otherwise 200
        :param min_healthy_percent: (experimental) The minimum number of tasks, specified as a percentage of the Amazon ECS service's DesiredCount value, that must continue to run and remain healthy during a deployment. Default: - 0 if daemon, otherwise 50
        :param propagate_tags: (experimental) Specifies whether to propagate the tags from the task definition or the service to the tasks in the service. Valid values are: PropagatedTagSource.SERVICE, PropagatedTagSource.TASK_DEFINITION or PropagatedTagSource.NONE Default: PropagatedTagSource.NONE
        :param service_name: (experimental) The name of the service. Default: - CloudFormation-generated name.
        :param launch_type: (experimental) The launch type on which to run your service. LaunchType will be omitted if capacity provider strategies are specified on the service.

        :stability: experimental
        '''
        if isinstance(circuit_breaker, dict):
            circuit_breaker = DeploymentCircuitBreaker(**circuit_breaker)
        if isinstance(cloud_map_options, dict):
            cloud_map_options = CloudMapOptions(**cloud_map_options)
        if isinstance(deployment_controller, dict):
            deployment_controller = DeploymentController(**deployment_controller)
        self._values: typing.Dict[str, typing.Any] = {
            "cluster": cluster,
            "launch_type": launch_type,
        }
        if capacity_provider_strategies is not None:
            self._values["capacity_provider_strategies"] = capacity_provider_strategies
        if circuit_breaker is not None:
            self._values["circuit_breaker"] = circuit_breaker
        if cloud_map_options is not None:
            self._values["cloud_map_options"] = cloud_map_options
        if deployment_controller is not None:
            self._values["deployment_controller"] = deployment_controller
        if desired_count is not None:
            self._values["desired_count"] = desired_count
        if enable_ecs_managed_tags is not None:
            self._values["enable_ecs_managed_tags"] = enable_ecs_managed_tags
        if health_check_grace_period is not None:
            self._values["health_check_grace_period"] = health_check_grace_period
        if max_healthy_percent is not None:
            self._values["max_healthy_percent"] = max_healthy_percent
        if min_healthy_percent is not None:
            self._values["min_healthy_percent"] = min_healthy_percent
        if propagate_tags is not None:
            self._values["propagate_tags"] = propagate_tags
        if service_name is not None:
            self._values["service_name"] = service_name

    @builtins.property
    def cluster(self) -> "ICluster":
        '''(experimental) The name of the cluster that hosts the service.

        :stability: experimental
        '''
        result = self._values.get("cluster")
        assert result is not None, "Required property 'cluster' is missing"
        return typing.cast("ICluster", result)

    @builtins.property
    def capacity_provider_strategies(
        self,
    ) -> typing.Optional[typing.List["CapacityProviderStrategy"]]:
        '''(experimental) A list of Capacity Provider strategies used to place a service.

        :default: - undefined

        :stability: experimental
        '''
        result = self._values.get("capacity_provider_strategies")
        return typing.cast(typing.Optional[typing.List["CapacityProviderStrategy"]], result)

    @builtins.property
    def circuit_breaker(self) -> typing.Optional["DeploymentCircuitBreaker"]:
        '''(experimental) Whether to enable the deployment circuit breaker.

        If this property is defined, circuit breaker will be implicitly
        enabled.

        :default: - disabled

        :stability: experimental
        '''
        result = self._values.get("circuit_breaker")
        return typing.cast(typing.Optional["DeploymentCircuitBreaker"], result)

    @builtins.property
    def cloud_map_options(self) -> typing.Optional["CloudMapOptions"]:
        '''(experimental) The options for configuring an Amazon ECS service to use service discovery.

        :default: - AWS Cloud Map service discovery is not enabled.

        :stability: experimental
        '''
        result = self._values.get("cloud_map_options")
        return typing.cast(typing.Optional["CloudMapOptions"], result)

    @builtins.property
    def deployment_controller(self) -> typing.Optional["DeploymentController"]:
        '''(experimental) Specifies which deployment controller to use for the service.

        For more information, see
        `Amazon ECS Deployment Types <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-types.html>`_

        :default: - Rolling update (ECS)

        :stability: experimental
        '''
        result = self._values.get("deployment_controller")
        return typing.cast(typing.Optional["DeploymentController"], result)

    @builtins.property
    def desired_count(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The desired number of instantiations of the task definition to keep running on the service.

        :default:

        - When creating the service, default is 1; when updating the service, default uses
        the current task number.

        :stability: experimental
        '''
        result = self._values.get("desired_count")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def enable_ecs_managed_tags(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specifies whether to enable Amazon ECS managed tags for the tasks within the service.

        For more information, see
        `Tagging Your Amazon ECS Resources <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html>`_

        :default: false

        :stability: experimental
        '''
        result = self._values.get("enable_ecs_managed_tags")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def health_check_grace_period(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) The period of time, in seconds, that the Amazon ECS service scheduler ignores unhealthy Elastic Load Balancing target health checks after a task has first started.

        :default: - defaults to 60 seconds if at least one load balancer is in-use and it is not already set

        :stability: experimental
        '''
        result = self._values.get("health_check_grace_period")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def max_healthy_percent(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The maximum number of tasks, specified as a percentage of the Amazon ECS service's DesiredCount value, that can run in a service during a deployment.

        :default: - 100 if daemon, otherwise 200

        :stability: experimental
        '''
        result = self._values.get("max_healthy_percent")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def min_healthy_percent(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The minimum number of tasks, specified as a percentage of the Amazon ECS service's DesiredCount value, that must continue to run and remain healthy during a deployment.

        :default: - 0 if daemon, otherwise 50

        :stability: experimental
        '''
        result = self._values.get("min_healthy_percent")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def propagate_tags(self) -> typing.Optional["PropagatedTagSource"]:
        '''(experimental) Specifies whether to propagate the tags from the task definition or the service to the tasks in the service.

        Valid values are: PropagatedTagSource.SERVICE, PropagatedTagSource.TASK_DEFINITION or PropagatedTagSource.NONE

        :default: PropagatedTagSource.NONE

        :stability: experimental
        '''
        result = self._values.get("propagate_tags")
        return typing.cast(typing.Optional["PropagatedTagSource"], result)

    @builtins.property
    def service_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name of the service.

        :default: - CloudFormation-generated name.

        :stability: experimental
        '''
        result = self._values.get("service_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def launch_type(self) -> "LaunchType":
        '''(experimental) The launch type on which to run your service.

        LaunchType will be omitted if capacity provider strategies are specified on the service.

        :see:

        - https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-capacityproviderstrategy

        Valid values are: LaunchType.ECS or LaunchType.FARGATE
        :stability: experimental
        '''
        result = self._values.get("launch_type")
        assert result is not None, "Required property 'launch_type' is missing"
        return typing.cast("LaunchType", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BaseServiceProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_ecs.BinPackResource")
class BinPackResource(enum.Enum):
    '''(experimental) Instance resource used for bin packing.

    :stability: experimental
    '''

    CPU = "CPU"
    '''(experimental) Fill up hosts' CPU allocations first.

    :stability: experimental
    '''
    MEMORY = "MEMORY"
    '''(experimental) Fill up hosts' memory allocations first.

    :stability: experimental
    '''


@jsii.implements(_IMachineImage_45d3238d)
class BottleRocketImage(
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_ecs.BottleRocketImage",
):
    '''(experimental) Construct an Bottlerocket image from the latest AMI published in SSM.

    :stability: experimental
    '''

    def __init__(
        self,
        *,
        variant: typing.Optional["BottlerocketEcsVariant"] = None,
    ) -> None:
        '''(experimental) Constructs a new instance of the BottleRocketImage class.

        :param variant: (experimental) The Amazon ECS variant to use. Only ``aws-ecs-1`` is currently available Default: - BottlerocketEcsVariant.AWS_ECS_1

        :stability: experimental
        '''
        props = BottleRocketImageProps(variant=variant)

        jsii.create(BottleRocketImage, self, [props])

    @jsii.member(jsii_name="getImage")
    def get_image(self, scope: _Construct_e78e779f) -> _MachineImageConfig_963798fe:
        '''(experimental) Return the correct image.

        :param scope: -

        :stability: experimental
        '''
        return typing.cast(_MachineImageConfig_963798fe, jsii.invoke(self, "getImage", [scope]))


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.BottleRocketImageProps",
    jsii_struct_bases=[],
    name_mapping={"variant": "variant"},
)
class BottleRocketImageProps:
    def __init__(
        self,
        *,
        variant: typing.Optional["BottlerocketEcsVariant"] = None,
    ) -> None:
        '''(experimental) Properties for BottleRocketImage.

        :param variant: (experimental) The Amazon ECS variant to use. Only ``aws-ecs-1`` is currently available Default: - BottlerocketEcsVariant.AWS_ECS_1

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if variant is not None:
            self._values["variant"] = variant

    @builtins.property
    def variant(self) -> typing.Optional["BottlerocketEcsVariant"]:
        '''(experimental) The Amazon ECS variant to use.

        Only ``aws-ecs-1`` is currently available

        :default: - BottlerocketEcsVariant.AWS_ECS_1

        :stability: experimental
        '''
        result = self._values.get("variant")
        return typing.cast(typing.Optional["BottlerocketEcsVariant"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BottleRocketImageProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_ecs.BottlerocketEcsVariant")
class BottlerocketEcsVariant(enum.Enum):
    '''(experimental) Amazon ECS variant.

    :stability: experimental
    '''

    AWS_ECS_1 = "AWS_ECS_1"
    '''(experimental) aws-ecs-1 variant.

    :stability: experimental
    '''


class BuiltInAttributes(
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_ecs.BuiltInAttributes",
):
    '''(experimental) The built-in container instance attributes.

    :stability: experimental
    '''

    def __init__(self) -> None:
        '''
        :stability: experimental
        '''
        jsii.create(BuiltInAttributes, self, [])

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="AMI_ID")
    def AMI_ID(cls) -> builtins.str:
        '''(experimental) The AMI id the instance is using.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sget(cls, "AMI_ID"))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="AVAILABILITY_ZONE")
    def AVAILABILITY_ZONE(cls) -> builtins.str:
        '''(experimental) The AvailabilityZone where the instance is running in.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sget(cls, "AVAILABILITY_ZONE"))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="INSTANCE_ID")
    def INSTANCE_ID(cls) -> builtins.str:
        '''(experimental) The id of the instance.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sget(cls, "INSTANCE_ID"))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="INSTANCE_TYPE")
    def INSTANCE_TYPE(cls) -> builtins.str:
        '''(experimental) The EC2 instance type.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sget(cls, "INSTANCE_TYPE"))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="OS_TYPE")
    def OS_TYPE(cls) -> builtins.str:
        '''(experimental) The operating system of the instance.

        Either 'linux' or 'windows'.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sget(cls, "OS_TYPE"))


@jsii.enum(jsii_type="monocdk.aws_ecs.Capability")
class Capability(enum.Enum):
    '''(experimental) A Linux capability.

    :stability: experimental
    '''

    ALL = "ALL"
    '''
    :stability: experimental
    '''
    AUDIT_CONTROL = "AUDIT_CONTROL"
    '''
    :stability: experimental
    '''
    AUDIT_WRITE = "AUDIT_WRITE"
    '''
    :stability: experimental
    '''
    BLOCK_SUSPEND = "BLOCK_SUSPEND"
    '''
    :stability: experimental
    '''
    CHOWN = "CHOWN"
    '''
    :stability: experimental
    '''
    DAC_OVERRIDE = "DAC_OVERRIDE"
    '''
    :stability: experimental
    '''
    DAC_READ_SEARCH = "DAC_READ_SEARCH"
    '''
    :stability: experimental
    '''
    FOWNER = "FOWNER"
    '''
    :stability: experimental
    '''
    FSETID = "FSETID"
    '''
    :stability: experimental
    '''
    IPC_LOCK = "IPC_LOCK"
    '''
    :stability: experimental
    '''
    IPC_OWNER = "IPC_OWNER"
    '''
    :stability: experimental
    '''
    KILL = "KILL"
    '''
    :stability: experimental
    '''
    LEASE = "LEASE"
    '''
    :stability: experimental
    '''
    LINUX_IMMUTABLE = "LINUX_IMMUTABLE"
    '''
    :stability: experimental
    '''
    MAC_ADMIN = "MAC_ADMIN"
    '''
    :stability: experimental
    '''
    MAC_OVERRIDE = "MAC_OVERRIDE"
    '''
    :stability: experimental
    '''
    MKNOD = "MKNOD"
    '''
    :stability: experimental
    '''
    NET_ADMIN = "NET_ADMIN"
    '''
    :stability: experimental
    '''
    NET_BIND_SERVICE = "NET_BIND_SERVICE"
    '''
    :stability: experimental
    '''
    NET_BROADCAST = "NET_BROADCAST"
    '''
    :stability: experimental
    '''
    NET_RAW = "NET_RAW"
    '''
    :stability: experimental
    '''
    SETFCAP = "SETFCAP"
    '''
    :stability: experimental
    '''
    SETGID = "SETGID"
    '''
    :stability: experimental
    '''
    SETPCAP = "SETPCAP"
    '''
    :stability: experimental
    '''
    SETUID = "SETUID"
    '''
    :stability: experimental
    '''
    SYS_ADMIN = "SYS_ADMIN"
    '''
    :stability: experimental
    '''
    SYS_BOOT = "SYS_BOOT"
    '''
    :stability: experimental
    '''
    SYS_CHROOT = "SYS_CHROOT"
    '''
    :stability: experimental
    '''
    SYS_MODULE = "SYS_MODULE"
    '''
    :stability: experimental
    '''
    SYS_NICE = "SYS_NICE"
    '''
    :stability: experimental
    '''
    SYS_PACCT = "SYS_PACCT"
    '''
    :stability: experimental
    '''
    SYS_PTRACE = "SYS_PTRACE"
    '''
    :stability: experimental
    '''
    SYS_RAWIO = "SYS_RAWIO"
    '''
    :stability: experimental
    '''
    SYS_RESOURCE = "SYS_RESOURCE"
    '''
    :stability: experimental
    '''
    SYS_TIME = "SYS_TIME"
    '''
    :stability: experimental
    '''
    SYS_TTY_CONFIG = "SYS_TTY_CONFIG"
    '''
    :stability: experimental
    '''
    SYSLOG = "SYSLOG"
    '''
    :stability: experimental
    '''
    WAKE_ALARM = "WAKE_ALARM"
    '''
    :stability: experimental
    '''


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.CapacityProviderStrategy",
    jsii_struct_bases=[],
    name_mapping={
        "capacity_provider": "capacityProvider",
        "base": "base",
        "weight": "weight",
    },
)
class CapacityProviderStrategy:
    def __init__(
        self,
        *,
        capacity_provider: builtins.str,
        base: typing.Optional[jsii.Number] = None,
        weight: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''(experimental) A Capacity Provider strategy to use for the service.

        NOTE: defaultCapacityProviderStrategy on cluster not currently supported.

        :param capacity_provider: (experimental) The name of the capacity provider.
        :param base: (experimental) The base value designates how many tasks, at a minimum, to run on the specified capacity provider. Only one capacity provider in a capacity provider strategy can have a base defined. If no value is specified, the default value of 0 is used. Default: - none
        :param weight: (experimental) The weight value designates the relative percentage of the total number of tasks launched that should use the specified capacity provider. The weight value is taken into consideration after the base value, if defined, is satisfied. Default: - 0

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "capacity_provider": capacity_provider,
        }
        if base is not None:
            self._values["base"] = base
        if weight is not None:
            self._values["weight"] = weight

    @builtins.property
    def capacity_provider(self) -> builtins.str:
        '''(experimental) The name of the capacity provider.

        :stability: experimental
        '''
        result = self._values.get("capacity_provider")
        assert result is not None, "Required property 'capacity_provider' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def base(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The base value designates how many tasks, at a minimum, to run on the specified capacity provider.

        Only one
        capacity provider in a capacity provider strategy can have a base defined. If no value is specified, the default
        value of 0 is used.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("base")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def weight(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The weight value designates the relative percentage of the total number of tasks launched that should use the specified capacity provider.

        The weight value is taken into consideration after the base value, if defined, is satisfied.

        :default: - 0

        :stability: experimental
        '''
        result = self._values.get("weight")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CapacityProviderStrategy(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnCapacityProvider(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_ecs.CfnCapacityProvider",
):
    '''A CloudFormation ``AWS::ECS::CapacityProvider``.

    :cloudformationResource: AWS::ECS::CapacityProvider
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-capacityprovider.html
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        auto_scaling_group_provider: typing.Union["CfnCapacityProvider.AutoScalingGroupProviderProperty", _IResolvable_a771d0ef],
        name: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[_CfnTag_95fbdc29]] = None,
    ) -> None:
        '''Create a new ``AWS::ECS::CapacityProvider``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param auto_scaling_group_provider: ``AWS::ECS::CapacityProvider.AutoScalingGroupProvider``.
        :param name: ``AWS::ECS::CapacityProvider.Name``.
        :param tags: ``AWS::ECS::CapacityProvider.Tags``.
        '''
        props = CfnCapacityProviderProps(
            auto_scaling_group_provider=auto_scaling_group_provider,
            name=name,
            tags=tags,
        )

        jsii.create(CfnCapacityProvider, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0b7ab120:
        '''``AWS::ECS::CapacityProvider.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-capacityprovider.html#cfn-ecs-capacityprovider-tags
        '''
        return typing.cast(_TagManager_0b7ab120, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="autoScalingGroupProvider")
    def auto_scaling_group_provider(
        self,
    ) -> typing.Union["CfnCapacityProvider.AutoScalingGroupProviderProperty", _IResolvable_a771d0ef]:
        '''``AWS::ECS::CapacityProvider.AutoScalingGroupProvider``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-capacityprovider.html#cfn-ecs-capacityprovider-autoscalinggroupprovider
        '''
        return typing.cast(typing.Union["CfnCapacityProvider.AutoScalingGroupProviderProperty", _IResolvable_a771d0ef], jsii.get(self, "autoScalingGroupProvider"))

    @auto_scaling_group_provider.setter
    def auto_scaling_group_provider(
        self,
        value: typing.Union["CfnCapacityProvider.AutoScalingGroupProviderProperty", _IResolvable_a771d0ef],
    ) -> None:
        jsii.set(self, "autoScalingGroupProvider", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="name")
    def name(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::CapacityProvider.Name``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-capacityprovider.html#cfn-ecs-capacityprovider-name
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "name"))

    @name.setter
    def name(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "name", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnCapacityProvider.AutoScalingGroupProviderProperty",
        jsii_struct_bases=[],
        name_mapping={
            "auto_scaling_group_arn": "autoScalingGroupArn",
            "managed_scaling": "managedScaling",
            "managed_termination_protection": "managedTerminationProtection",
        },
    )
    class AutoScalingGroupProviderProperty:
        def __init__(
            self,
            *,
            auto_scaling_group_arn: builtins.str,
            managed_scaling: typing.Optional[typing.Union["CfnCapacityProvider.ManagedScalingProperty", _IResolvable_a771d0ef]] = None,
            managed_termination_protection: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param auto_scaling_group_arn: ``CfnCapacityProvider.AutoScalingGroupProviderProperty.AutoScalingGroupArn``.
            :param managed_scaling: ``CfnCapacityProvider.AutoScalingGroupProviderProperty.ManagedScaling``.
            :param managed_termination_protection: ``CfnCapacityProvider.AutoScalingGroupProviderProperty.ManagedTerminationProtection``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-capacityprovider-autoscalinggroupprovider.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "auto_scaling_group_arn": auto_scaling_group_arn,
            }
            if managed_scaling is not None:
                self._values["managed_scaling"] = managed_scaling
            if managed_termination_protection is not None:
                self._values["managed_termination_protection"] = managed_termination_protection

        @builtins.property
        def auto_scaling_group_arn(self) -> builtins.str:
            '''``CfnCapacityProvider.AutoScalingGroupProviderProperty.AutoScalingGroupArn``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-capacityprovider-autoscalinggroupprovider.html#cfn-ecs-capacityprovider-autoscalinggroupprovider-autoscalinggrouparn
            '''
            result = self._values.get("auto_scaling_group_arn")
            assert result is not None, "Required property 'auto_scaling_group_arn' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def managed_scaling(
            self,
        ) -> typing.Optional[typing.Union["CfnCapacityProvider.ManagedScalingProperty", _IResolvable_a771d0ef]]:
            '''``CfnCapacityProvider.AutoScalingGroupProviderProperty.ManagedScaling``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-capacityprovider-autoscalinggroupprovider.html#cfn-ecs-capacityprovider-autoscalinggroupprovider-managedscaling
            '''
            result = self._values.get("managed_scaling")
            return typing.cast(typing.Optional[typing.Union["CfnCapacityProvider.ManagedScalingProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def managed_termination_protection(self) -> typing.Optional[builtins.str]:
            '''``CfnCapacityProvider.AutoScalingGroupProviderProperty.ManagedTerminationProtection``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-capacityprovider-autoscalinggroupprovider.html#cfn-ecs-capacityprovider-autoscalinggroupprovider-managedterminationprotection
            '''
            result = self._values.get("managed_termination_protection")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AutoScalingGroupProviderProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnCapacityProvider.ManagedScalingProperty",
        jsii_struct_bases=[],
        name_mapping={
            "instance_warmup_period": "instanceWarmupPeriod",
            "maximum_scaling_step_size": "maximumScalingStepSize",
            "minimum_scaling_step_size": "minimumScalingStepSize",
            "status": "status",
            "target_capacity": "targetCapacity",
        },
    )
    class ManagedScalingProperty:
        def __init__(
            self,
            *,
            instance_warmup_period: typing.Optional[jsii.Number] = None,
            maximum_scaling_step_size: typing.Optional[jsii.Number] = None,
            minimum_scaling_step_size: typing.Optional[jsii.Number] = None,
            status: typing.Optional[builtins.str] = None,
            target_capacity: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''
            :param instance_warmup_period: ``CfnCapacityProvider.ManagedScalingProperty.InstanceWarmupPeriod``.
            :param maximum_scaling_step_size: ``CfnCapacityProvider.ManagedScalingProperty.MaximumScalingStepSize``.
            :param minimum_scaling_step_size: ``CfnCapacityProvider.ManagedScalingProperty.MinimumScalingStepSize``.
            :param status: ``CfnCapacityProvider.ManagedScalingProperty.Status``.
            :param target_capacity: ``CfnCapacityProvider.ManagedScalingProperty.TargetCapacity``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-capacityprovider-managedscaling.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if instance_warmup_period is not None:
                self._values["instance_warmup_period"] = instance_warmup_period
            if maximum_scaling_step_size is not None:
                self._values["maximum_scaling_step_size"] = maximum_scaling_step_size
            if minimum_scaling_step_size is not None:
                self._values["minimum_scaling_step_size"] = minimum_scaling_step_size
            if status is not None:
                self._values["status"] = status
            if target_capacity is not None:
                self._values["target_capacity"] = target_capacity

        @builtins.property
        def instance_warmup_period(self) -> typing.Optional[jsii.Number]:
            '''``CfnCapacityProvider.ManagedScalingProperty.InstanceWarmupPeriod``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-capacityprovider-managedscaling.html#cfn-ecs-capacityprovider-managedscaling-instancewarmupperiod
            '''
            result = self._values.get("instance_warmup_period")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def maximum_scaling_step_size(self) -> typing.Optional[jsii.Number]:
            '''``CfnCapacityProvider.ManagedScalingProperty.MaximumScalingStepSize``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-capacityprovider-managedscaling.html#cfn-ecs-capacityprovider-managedscaling-maximumscalingstepsize
            '''
            result = self._values.get("maximum_scaling_step_size")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def minimum_scaling_step_size(self) -> typing.Optional[jsii.Number]:
            '''``CfnCapacityProvider.ManagedScalingProperty.MinimumScalingStepSize``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-capacityprovider-managedscaling.html#cfn-ecs-capacityprovider-managedscaling-minimumscalingstepsize
            '''
            result = self._values.get("minimum_scaling_step_size")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def status(self) -> typing.Optional[builtins.str]:
            '''``CfnCapacityProvider.ManagedScalingProperty.Status``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-capacityprovider-managedscaling.html#cfn-ecs-capacityprovider-managedscaling-status
            '''
            result = self._values.get("status")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def target_capacity(self) -> typing.Optional[jsii.Number]:
            '''``CfnCapacityProvider.ManagedScalingProperty.TargetCapacity``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-capacityprovider-managedscaling.html#cfn-ecs-capacityprovider-managedscaling-targetcapacity
            '''
            result = self._values.get("target_capacity")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ManagedScalingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.CfnCapacityProviderProps",
    jsii_struct_bases=[],
    name_mapping={
        "auto_scaling_group_provider": "autoScalingGroupProvider",
        "name": "name",
        "tags": "tags",
    },
)
class CfnCapacityProviderProps:
    def __init__(
        self,
        *,
        auto_scaling_group_provider: typing.Union[CfnCapacityProvider.AutoScalingGroupProviderProperty, _IResolvable_a771d0ef],
        name: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[_CfnTag_95fbdc29]] = None,
    ) -> None:
        '''Properties for defining a ``AWS::ECS::CapacityProvider``.

        :param auto_scaling_group_provider: ``AWS::ECS::CapacityProvider.AutoScalingGroupProvider``.
        :param name: ``AWS::ECS::CapacityProvider.Name``.
        :param tags: ``AWS::ECS::CapacityProvider.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-capacityprovider.html
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "auto_scaling_group_provider": auto_scaling_group_provider,
        }
        if name is not None:
            self._values["name"] = name
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def auto_scaling_group_provider(
        self,
    ) -> typing.Union[CfnCapacityProvider.AutoScalingGroupProviderProperty, _IResolvable_a771d0ef]:
        '''``AWS::ECS::CapacityProvider.AutoScalingGroupProvider``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-capacityprovider.html#cfn-ecs-capacityprovider-autoscalinggroupprovider
        '''
        result = self._values.get("auto_scaling_group_provider")
        assert result is not None, "Required property 'auto_scaling_group_provider' is missing"
        return typing.cast(typing.Union[CfnCapacityProvider.AutoScalingGroupProviderProperty, _IResolvable_a771d0ef], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::CapacityProvider.Name``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-capacityprovider.html#cfn-ecs-capacityprovider-name
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_95fbdc29]]:
        '''``AWS::ECS::CapacityProvider.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-capacityprovider.html#cfn-ecs-capacityprovider-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_95fbdc29]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnCapacityProviderProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnCluster(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_ecs.CfnCluster",
):
    '''A CloudFormation ``AWS::ECS::Cluster``.

    :cloudformationResource: AWS::ECS::Cluster
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        capacity_providers: typing.Optional[typing.Sequence[builtins.str]] = None,
        cluster_name: typing.Optional[builtins.str] = None,
        cluster_settings: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnCluster.ClusterSettingsProperty", _IResolvable_a771d0ef]]]] = None,
        configuration: typing.Optional[typing.Union["CfnCluster.ClusterConfigurationProperty", _IResolvable_a771d0ef]] = None,
        default_capacity_provider_strategy: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnCluster.CapacityProviderStrategyItemProperty", _IResolvable_a771d0ef]]]] = None,
        tags: typing.Optional[typing.Sequence[_CfnTag_95fbdc29]] = None,
    ) -> None:
        '''Create a new ``AWS::ECS::Cluster``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param capacity_providers: ``AWS::ECS::Cluster.CapacityProviders``.
        :param cluster_name: ``AWS::ECS::Cluster.ClusterName``.
        :param cluster_settings: ``AWS::ECS::Cluster.ClusterSettings``.
        :param configuration: ``AWS::ECS::Cluster.Configuration``.
        :param default_capacity_provider_strategy: ``AWS::ECS::Cluster.DefaultCapacityProviderStrategy``.
        :param tags: ``AWS::ECS::Cluster.Tags``.
        '''
        props = CfnClusterProps(
            capacity_providers=capacity_providers,
            cluster_name=cluster_name,
            cluster_settings=cluster_settings,
            configuration=configuration,
            default_capacity_provider_strategy=default_capacity_provider_strategy,
            tags=tags,
        )

        jsii.create(CfnCluster, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''
        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0b7ab120:
        '''``AWS::ECS::Cluster.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html#cfn-ecs-cluster-tags
        '''
        return typing.cast(_TagManager_0b7ab120, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="capacityProviders")
    def capacity_providers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''``AWS::ECS::Cluster.CapacityProviders``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html#cfn-ecs-cluster-capacityproviders
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "capacityProviders"))

    @capacity_providers.setter
    def capacity_providers(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        jsii.set(self, "capacityProviders", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="clusterName")
    def cluster_name(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::Cluster.ClusterName``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html#cfn-ecs-cluster-clustername
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "clusterName"))

    @cluster_name.setter
    def cluster_name(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "clusterName", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="clusterSettings")
    def cluster_settings(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnCluster.ClusterSettingsProperty", _IResolvable_a771d0ef]]]]:
        '''``AWS::ECS::Cluster.ClusterSettings``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html#cfn-ecs-cluster-clustersettings
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnCluster.ClusterSettingsProperty", _IResolvable_a771d0ef]]]], jsii.get(self, "clusterSettings"))

    @cluster_settings.setter
    def cluster_settings(
        self,
        value: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnCluster.ClusterSettingsProperty", _IResolvable_a771d0ef]]]],
    ) -> None:
        jsii.set(self, "clusterSettings", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="configuration")
    def configuration(
        self,
    ) -> typing.Optional[typing.Union["CfnCluster.ClusterConfigurationProperty", _IResolvable_a771d0ef]]:
        '''``AWS::ECS::Cluster.Configuration``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html#cfn-ecs-cluster-configuration
        '''
        return typing.cast(typing.Optional[typing.Union["CfnCluster.ClusterConfigurationProperty", _IResolvable_a771d0ef]], jsii.get(self, "configuration"))

    @configuration.setter
    def configuration(
        self,
        value: typing.Optional[typing.Union["CfnCluster.ClusterConfigurationProperty", _IResolvable_a771d0ef]],
    ) -> None:
        jsii.set(self, "configuration", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="defaultCapacityProviderStrategy")
    def default_capacity_provider_strategy(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnCluster.CapacityProviderStrategyItemProperty", _IResolvable_a771d0ef]]]]:
        '''``AWS::ECS::Cluster.DefaultCapacityProviderStrategy``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html#cfn-ecs-cluster-defaultcapacityproviderstrategy
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnCluster.CapacityProviderStrategyItemProperty", _IResolvable_a771d0ef]]]], jsii.get(self, "defaultCapacityProviderStrategy"))

    @default_capacity_provider_strategy.setter
    def default_capacity_provider_strategy(
        self,
        value: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnCluster.CapacityProviderStrategyItemProperty", _IResolvable_a771d0ef]]]],
    ) -> None:
        jsii.set(self, "defaultCapacityProviderStrategy", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnCluster.CapacityProviderStrategyItemProperty",
        jsii_struct_bases=[],
        name_mapping={
            "base": "base",
            "capacity_provider": "capacityProvider",
            "weight": "weight",
        },
    )
    class CapacityProviderStrategyItemProperty:
        def __init__(
            self,
            *,
            base: typing.Optional[jsii.Number] = None,
            capacity_provider: typing.Optional[builtins.str] = None,
            weight: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''
            :param base: ``CfnCluster.CapacityProviderStrategyItemProperty.Base``.
            :param capacity_provider: ``CfnCluster.CapacityProviderStrategyItemProperty.CapacityProvider``.
            :param weight: ``CfnCluster.CapacityProviderStrategyItemProperty.Weight``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-capacityproviderstrategyitem.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if base is not None:
                self._values["base"] = base
            if capacity_provider is not None:
                self._values["capacity_provider"] = capacity_provider
            if weight is not None:
                self._values["weight"] = weight

        @builtins.property
        def base(self) -> typing.Optional[jsii.Number]:
            '''``CfnCluster.CapacityProviderStrategyItemProperty.Base``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-capacityproviderstrategyitem.html#cfn-ecs-cluster-capacityproviderstrategyitem-base
            '''
            result = self._values.get("base")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def capacity_provider(self) -> typing.Optional[builtins.str]:
            '''``CfnCluster.CapacityProviderStrategyItemProperty.CapacityProvider``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-capacityproviderstrategyitem.html#cfn-ecs-cluster-capacityproviderstrategyitem-capacityprovider
            '''
            result = self._values.get("capacity_provider")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def weight(self) -> typing.Optional[jsii.Number]:
            '''``CfnCluster.CapacityProviderStrategyItemProperty.Weight``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-capacityproviderstrategyitem.html#cfn-ecs-cluster-capacityproviderstrategyitem-weight
            '''
            result = self._values.get("weight")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CapacityProviderStrategyItemProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnCluster.ClusterConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"execute_command_configuration": "executeCommandConfiguration"},
    )
    class ClusterConfigurationProperty:
        def __init__(
            self,
            *,
            execute_command_configuration: typing.Optional[typing.Union["CfnCluster.ExecuteCommandConfigurationProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''
            :param execute_command_configuration: ``CfnCluster.ClusterConfigurationProperty.ExecuteCommandConfiguration``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-clusterconfiguration.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if execute_command_configuration is not None:
                self._values["execute_command_configuration"] = execute_command_configuration

        @builtins.property
        def execute_command_configuration(
            self,
        ) -> typing.Optional[typing.Union["CfnCluster.ExecuteCommandConfigurationProperty", _IResolvable_a771d0ef]]:
            '''``CfnCluster.ClusterConfigurationProperty.ExecuteCommandConfiguration``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-clusterconfiguration.html#cfn-ecs-cluster-clusterconfiguration-executecommandconfiguration
            '''
            result = self._values.get("execute_command_configuration")
            return typing.cast(typing.Optional[typing.Union["CfnCluster.ExecuteCommandConfigurationProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ClusterConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnCluster.ClusterSettingsProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "value": "value"},
    )
    class ClusterSettingsProperty:
        def __init__(
            self,
            *,
            name: typing.Optional[builtins.str] = None,
            value: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param name: ``CfnCluster.ClusterSettingsProperty.Name``.
            :param value: ``CfnCluster.ClusterSettingsProperty.Value``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-clustersettings.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if name is not None:
                self._values["name"] = name
            if value is not None:
                self._values["value"] = value

        @builtins.property
        def name(self) -> typing.Optional[builtins.str]:
            '''``CfnCluster.ClusterSettingsProperty.Name``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-clustersettings.html#cfn-ecs-cluster-clustersettings-name
            '''
            result = self._values.get("name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def value(self) -> typing.Optional[builtins.str]:
            '''``CfnCluster.ClusterSettingsProperty.Value``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-clustersettings.html#cfn-ecs-cluster-clustersettings-value
            '''
            result = self._values.get("value")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ClusterSettingsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnCluster.ExecuteCommandConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "kms_key_id": "kmsKeyId",
            "log_configuration": "logConfiguration",
            "logging": "logging",
        },
    )
    class ExecuteCommandConfigurationProperty:
        def __init__(
            self,
            *,
            kms_key_id: typing.Optional[builtins.str] = None,
            log_configuration: typing.Optional[typing.Union["CfnCluster.ExecuteCommandLogConfigurationProperty", _IResolvable_a771d0ef]] = None,
            logging: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param kms_key_id: ``CfnCluster.ExecuteCommandConfigurationProperty.KmsKeyId``.
            :param log_configuration: ``CfnCluster.ExecuteCommandConfigurationProperty.LogConfiguration``.
            :param logging: ``CfnCluster.ExecuteCommandConfigurationProperty.Logging``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-executecommandconfiguration.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if kms_key_id is not None:
                self._values["kms_key_id"] = kms_key_id
            if log_configuration is not None:
                self._values["log_configuration"] = log_configuration
            if logging is not None:
                self._values["logging"] = logging

        @builtins.property
        def kms_key_id(self) -> typing.Optional[builtins.str]:
            '''``CfnCluster.ExecuteCommandConfigurationProperty.KmsKeyId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-executecommandconfiguration.html#cfn-ecs-cluster-executecommandconfiguration-kmskeyid
            '''
            result = self._values.get("kms_key_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def log_configuration(
            self,
        ) -> typing.Optional[typing.Union["CfnCluster.ExecuteCommandLogConfigurationProperty", _IResolvable_a771d0ef]]:
            '''``CfnCluster.ExecuteCommandConfigurationProperty.LogConfiguration``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-executecommandconfiguration.html#cfn-ecs-cluster-executecommandconfiguration-logconfiguration
            '''
            result = self._values.get("log_configuration")
            return typing.cast(typing.Optional[typing.Union["CfnCluster.ExecuteCommandLogConfigurationProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def logging(self) -> typing.Optional[builtins.str]:
            '''``CfnCluster.ExecuteCommandConfigurationProperty.Logging``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-executecommandconfiguration.html#cfn-ecs-cluster-executecommandconfiguration-logging
            '''
            result = self._values.get("logging")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ExecuteCommandConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnCluster.ExecuteCommandLogConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "cloud_watch_encryption_enabled": "cloudWatchEncryptionEnabled",
            "cloud_watch_log_group_name": "cloudWatchLogGroupName",
            "s3_bucket_name": "s3BucketName",
            "s3_encryption_enabled": "s3EncryptionEnabled",
            "s3_key_prefix": "s3KeyPrefix",
        },
    )
    class ExecuteCommandLogConfigurationProperty:
        def __init__(
            self,
            *,
            cloud_watch_encryption_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            cloud_watch_log_group_name: typing.Optional[builtins.str] = None,
            s3_bucket_name: typing.Optional[builtins.str] = None,
            s3_encryption_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            s3_key_prefix: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param cloud_watch_encryption_enabled: ``CfnCluster.ExecuteCommandLogConfigurationProperty.CloudWatchEncryptionEnabled``.
            :param cloud_watch_log_group_name: ``CfnCluster.ExecuteCommandLogConfigurationProperty.CloudWatchLogGroupName``.
            :param s3_bucket_name: ``CfnCluster.ExecuteCommandLogConfigurationProperty.S3BucketName``.
            :param s3_encryption_enabled: ``CfnCluster.ExecuteCommandLogConfigurationProperty.S3EncryptionEnabled``.
            :param s3_key_prefix: ``CfnCluster.ExecuteCommandLogConfigurationProperty.S3KeyPrefix``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-executecommandlogconfiguration.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if cloud_watch_encryption_enabled is not None:
                self._values["cloud_watch_encryption_enabled"] = cloud_watch_encryption_enabled
            if cloud_watch_log_group_name is not None:
                self._values["cloud_watch_log_group_name"] = cloud_watch_log_group_name
            if s3_bucket_name is not None:
                self._values["s3_bucket_name"] = s3_bucket_name
            if s3_encryption_enabled is not None:
                self._values["s3_encryption_enabled"] = s3_encryption_enabled
            if s3_key_prefix is not None:
                self._values["s3_key_prefix"] = s3_key_prefix

        @builtins.property
        def cloud_watch_encryption_enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''``CfnCluster.ExecuteCommandLogConfigurationProperty.CloudWatchEncryptionEnabled``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-executecommandlogconfiguration.html#cfn-ecs-cluster-executecommandlogconfiguration-cloudwatchencryptionenabled
            '''
            result = self._values.get("cloud_watch_encryption_enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def cloud_watch_log_group_name(self) -> typing.Optional[builtins.str]:
            '''``CfnCluster.ExecuteCommandLogConfigurationProperty.CloudWatchLogGroupName``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-executecommandlogconfiguration.html#cfn-ecs-cluster-executecommandlogconfiguration-cloudwatchloggroupname
            '''
            result = self._values.get("cloud_watch_log_group_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def s3_bucket_name(self) -> typing.Optional[builtins.str]:
            '''``CfnCluster.ExecuteCommandLogConfigurationProperty.S3BucketName``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-executecommandlogconfiguration.html#cfn-ecs-cluster-executecommandlogconfiguration-s3bucketname
            '''
            result = self._values.get("s3_bucket_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def s3_encryption_enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''``CfnCluster.ExecuteCommandLogConfigurationProperty.S3EncryptionEnabled``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-executecommandlogconfiguration.html#cfn-ecs-cluster-executecommandlogconfiguration-s3encryptionenabled
            '''
            result = self._values.get("s3_encryption_enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def s3_key_prefix(self) -> typing.Optional[builtins.str]:
            '''``CfnCluster.ExecuteCommandLogConfigurationProperty.S3KeyPrefix``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-executecommandlogconfiguration.html#cfn-ecs-cluster-executecommandlogconfiguration-s3keyprefix
            '''
            result = self._values.get("s3_key_prefix")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ExecuteCommandLogConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.implements(_IInspectable_82c04a63)
class CfnClusterCapacityProviderAssociations(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_ecs.CfnClusterCapacityProviderAssociations",
):
    '''A CloudFormation ``AWS::ECS::ClusterCapacityProviderAssociations``.

    :cloudformationResource: AWS::ECS::ClusterCapacityProviderAssociations
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-clustercapacityproviderassociations.html
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        capacity_providers: typing.Sequence[builtins.str],
        cluster: builtins.str,
        default_capacity_provider_strategy: typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnClusterCapacityProviderAssociations.CapacityProviderStrategyProperty", _IResolvable_a771d0ef]]],
    ) -> None:
        '''Create a new ``AWS::ECS::ClusterCapacityProviderAssociations``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param capacity_providers: ``AWS::ECS::ClusterCapacityProviderAssociations.CapacityProviders``.
        :param cluster: ``AWS::ECS::ClusterCapacityProviderAssociations.Cluster``.
        :param default_capacity_provider_strategy: ``AWS::ECS::ClusterCapacityProviderAssociations.DefaultCapacityProviderStrategy``.
        '''
        props = CfnClusterCapacityProviderAssociationsProps(
            capacity_providers=capacity_providers,
            cluster=cluster,
            default_capacity_provider_strategy=default_capacity_provider_strategy,
        )

        jsii.create(CfnClusterCapacityProviderAssociations, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="capacityProviders")
    def capacity_providers(self) -> typing.List[builtins.str]:
        '''``AWS::ECS::ClusterCapacityProviderAssociations.CapacityProviders``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-clustercapacityproviderassociations.html#cfn-ecs-clustercapacityproviderassociations-capacityproviders
        '''
        return typing.cast(typing.List[builtins.str], jsii.get(self, "capacityProviders"))

    @capacity_providers.setter
    def capacity_providers(self, value: typing.List[builtins.str]) -> None:
        jsii.set(self, "capacityProviders", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cluster")
    def cluster(self) -> builtins.str:
        '''``AWS::ECS::ClusterCapacityProviderAssociations.Cluster``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-clustercapacityproviderassociations.html#cfn-ecs-clustercapacityproviderassociations-cluster
        '''
        return typing.cast(builtins.str, jsii.get(self, "cluster"))

    @cluster.setter
    def cluster(self, value: builtins.str) -> None:
        jsii.set(self, "cluster", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="defaultCapacityProviderStrategy")
    def default_capacity_provider_strategy(
        self,
    ) -> typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnClusterCapacityProviderAssociations.CapacityProviderStrategyProperty", _IResolvable_a771d0ef]]]:
        '''``AWS::ECS::ClusterCapacityProviderAssociations.DefaultCapacityProviderStrategy``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-clustercapacityproviderassociations.html#cfn-ecs-clustercapacityproviderassociations-defaultcapacityproviderstrategy
        '''
        return typing.cast(typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnClusterCapacityProviderAssociations.CapacityProviderStrategyProperty", _IResolvable_a771d0ef]]], jsii.get(self, "defaultCapacityProviderStrategy"))

    @default_capacity_provider_strategy.setter
    def default_capacity_provider_strategy(
        self,
        value: typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnClusterCapacityProviderAssociations.CapacityProviderStrategyProperty", _IResolvable_a771d0ef]]],
    ) -> None:
        jsii.set(self, "defaultCapacityProviderStrategy", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnClusterCapacityProviderAssociations.CapacityProviderStrategyProperty",
        jsii_struct_bases=[],
        name_mapping={
            "capacity_provider": "capacityProvider",
            "base": "base",
            "weight": "weight",
        },
    )
    class CapacityProviderStrategyProperty:
        def __init__(
            self,
            *,
            capacity_provider: builtins.str,
            base: typing.Optional[jsii.Number] = None,
            weight: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''
            :param capacity_provider: ``CfnClusterCapacityProviderAssociations.CapacityProviderStrategyProperty.CapacityProvider``.
            :param base: ``CfnClusterCapacityProviderAssociations.CapacityProviderStrategyProperty.Base``.
            :param weight: ``CfnClusterCapacityProviderAssociations.CapacityProviderStrategyProperty.Weight``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-clustercapacityproviderassociations-capacityproviderstrategy.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "capacity_provider": capacity_provider,
            }
            if base is not None:
                self._values["base"] = base
            if weight is not None:
                self._values["weight"] = weight

        @builtins.property
        def capacity_provider(self) -> builtins.str:
            '''``CfnClusterCapacityProviderAssociations.CapacityProviderStrategyProperty.CapacityProvider``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-clustercapacityproviderassociations-capacityproviderstrategy.html#cfn-ecs-clustercapacityproviderassociations-capacityproviderstrategy-capacityprovider
            '''
            result = self._values.get("capacity_provider")
            assert result is not None, "Required property 'capacity_provider' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def base(self) -> typing.Optional[jsii.Number]:
            '''``CfnClusterCapacityProviderAssociations.CapacityProviderStrategyProperty.Base``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-clustercapacityproviderassociations-capacityproviderstrategy.html#cfn-ecs-clustercapacityproviderassociations-capacityproviderstrategy-base
            '''
            result = self._values.get("base")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def weight(self) -> typing.Optional[jsii.Number]:
            '''``CfnClusterCapacityProviderAssociations.CapacityProviderStrategyProperty.Weight``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-clustercapacityproviderassociations-capacityproviderstrategy.html#cfn-ecs-clustercapacityproviderassociations-capacityproviderstrategy-weight
            '''
            result = self._values.get("weight")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CapacityProviderStrategyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.CfnClusterCapacityProviderAssociationsProps",
    jsii_struct_bases=[],
    name_mapping={
        "capacity_providers": "capacityProviders",
        "cluster": "cluster",
        "default_capacity_provider_strategy": "defaultCapacityProviderStrategy",
    },
)
class CfnClusterCapacityProviderAssociationsProps:
    def __init__(
        self,
        *,
        capacity_providers: typing.Sequence[builtins.str],
        cluster: builtins.str,
        default_capacity_provider_strategy: typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[CfnClusterCapacityProviderAssociations.CapacityProviderStrategyProperty, _IResolvable_a771d0ef]]],
    ) -> None:
        '''Properties for defining a ``AWS::ECS::ClusterCapacityProviderAssociations``.

        :param capacity_providers: ``AWS::ECS::ClusterCapacityProviderAssociations.CapacityProviders``.
        :param cluster: ``AWS::ECS::ClusterCapacityProviderAssociations.Cluster``.
        :param default_capacity_provider_strategy: ``AWS::ECS::ClusterCapacityProviderAssociations.DefaultCapacityProviderStrategy``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-clustercapacityproviderassociations.html
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "capacity_providers": capacity_providers,
            "cluster": cluster,
            "default_capacity_provider_strategy": default_capacity_provider_strategy,
        }

    @builtins.property
    def capacity_providers(self) -> typing.List[builtins.str]:
        '''``AWS::ECS::ClusterCapacityProviderAssociations.CapacityProviders``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-clustercapacityproviderassociations.html#cfn-ecs-clustercapacityproviderassociations-capacityproviders
        '''
        result = self._values.get("capacity_providers")
        assert result is not None, "Required property 'capacity_providers' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def cluster(self) -> builtins.str:
        '''``AWS::ECS::ClusterCapacityProviderAssociations.Cluster``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-clustercapacityproviderassociations.html#cfn-ecs-clustercapacityproviderassociations-cluster
        '''
        result = self._values.get("cluster")
        assert result is not None, "Required property 'cluster' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def default_capacity_provider_strategy(
        self,
    ) -> typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnClusterCapacityProviderAssociations.CapacityProviderStrategyProperty, _IResolvable_a771d0ef]]]:
        '''``AWS::ECS::ClusterCapacityProviderAssociations.DefaultCapacityProviderStrategy``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-clustercapacityproviderassociations.html#cfn-ecs-clustercapacityproviderassociations-defaultcapacityproviderstrategy
        '''
        result = self._values.get("default_capacity_provider_strategy")
        assert result is not None, "Required property 'default_capacity_provider_strategy' is missing"
        return typing.cast(typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnClusterCapacityProviderAssociations.CapacityProviderStrategyProperty, _IResolvable_a771d0ef]]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnClusterCapacityProviderAssociationsProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.CfnClusterProps",
    jsii_struct_bases=[],
    name_mapping={
        "capacity_providers": "capacityProviders",
        "cluster_name": "clusterName",
        "cluster_settings": "clusterSettings",
        "configuration": "configuration",
        "default_capacity_provider_strategy": "defaultCapacityProviderStrategy",
        "tags": "tags",
    },
)
class CfnClusterProps:
    def __init__(
        self,
        *,
        capacity_providers: typing.Optional[typing.Sequence[builtins.str]] = None,
        cluster_name: typing.Optional[builtins.str] = None,
        cluster_settings: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[CfnCluster.ClusterSettingsProperty, _IResolvable_a771d0ef]]]] = None,
        configuration: typing.Optional[typing.Union[CfnCluster.ClusterConfigurationProperty, _IResolvable_a771d0ef]] = None,
        default_capacity_provider_strategy: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[CfnCluster.CapacityProviderStrategyItemProperty, _IResolvable_a771d0ef]]]] = None,
        tags: typing.Optional[typing.Sequence[_CfnTag_95fbdc29]] = None,
    ) -> None:
        '''Properties for defining a ``AWS::ECS::Cluster``.

        :param capacity_providers: ``AWS::ECS::Cluster.CapacityProviders``.
        :param cluster_name: ``AWS::ECS::Cluster.ClusterName``.
        :param cluster_settings: ``AWS::ECS::Cluster.ClusterSettings``.
        :param configuration: ``AWS::ECS::Cluster.Configuration``.
        :param default_capacity_provider_strategy: ``AWS::ECS::Cluster.DefaultCapacityProviderStrategy``.
        :param tags: ``AWS::ECS::Cluster.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if capacity_providers is not None:
            self._values["capacity_providers"] = capacity_providers
        if cluster_name is not None:
            self._values["cluster_name"] = cluster_name
        if cluster_settings is not None:
            self._values["cluster_settings"] = cluster_settings
        if configuration is not None:
            self._values["configuration"] = configuration
        if default_capacity_provider_strategy is not None:
            self._values["default_capacity_provider_strategy"] = default_capacity_provider_strategy
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def capacity_providers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''``AWS::ECS::Cluster.CapacityProviders``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html#cfn-ecs-cluster-capacityproviders
        '''
        result = self._values.get("capacity_providers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def cluster_name(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::Cluster.ClusterName``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html#cfn-ecs-cluster-clustername
        '''
        result = self._values.get("cluster_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def cluster_settings(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnCluster.ClusterSettingsProperty, _IResolvable_a771d0ef]]]]:
        '''``AWS::ECS::Cluster.ClusterSettings``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html#cfn-ecs-cluster-clustersettings
        '''
        result = self._values.get("cluster_settings")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnCluster.ClusterSettingsProperty, _IResolvable_a771d0ef]]]], result)

    @builtins.property
    def configuration(
        self,
    ) -> typing.Optional[typing.Union[CfnCluster.ClusterConfigurationProperty, _IResolvable_a771d0ef]]:
        '''``AWS::ECS::Cluster.Configuration``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html#cfn-ecs-cluster-configuration
        '''
        result = self._values.get("configuration")
        return typing.cast(typing.Optional[typing.Union[CfnCluster.ClusterConfigurationProperty, _IResolvable_a771d0ef]], result)

    @builtins.property
    def default_capacity_provider_strategy(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnCluster.CapacityProviderStrategyItemProperty, _IResolvable_a771d0ef]]]]:
        '''``AWS::ECS::Cluster.DefaultCapacityProviderStrategy``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html#cfn-ecs-cluster-defaultcapacityproviderstrategy
        '''
        result = self._values.get("default_capacity_provider_strategy")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnCluster.CapacityProviderStrategyItemProperty, _IResolvable_a771d0ef]]]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_95fbdc29]]:
        '''``AWS::ECS::Cluster.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html#cfn-ecs-cluster-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_95fbdc29]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnClusterProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnPrimaryTaskSet(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_ecs.CfnPrimaryTaskSet",
):
    '''A CloudFormation ``AWS::ECS::PrimaryTaskSet``.

    :cloudformationResource: AWS::ECS::PrimaryTaskSet
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-primarytaskset.html
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        cluster: builtins.str,
        service: builtins.str,
        task_set_id: builtins.str,
    ) -> None:
        '''Create a new ``AWS::ECS::PrimaryTaskSet``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param cluster: ``AWS::ECS::PrimaryTaskSet.Cluster``.
        :param service: ``AWS::ECS::PrimaryTaskSet.Service``.
        :param task_set_id: ``AWS::ECS::PrimaryTaskSet.TaskSetId``.
        '''
        props = CfnPrimaryTaskSetProps(
            cluster=cluster, service=service, task_set_id=task_set_id
        )

        jsii.create(CfnPrimaryTaskSet, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cluster")
    def cluster(self) -> builtins.str:
        '''``AWS::ECS::PrimaryTaskSet.Cluster``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-primarytaskset.html#cfn-ecs-primarytaskset-cluster
        '''
        return typing.cast(builtins.str, jsii.get(self, "cluster"))

    @cluster.setter
    def cluster(self, value: builtins.str) -> None:
        jsii.set(self, "cluster", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="service")
    def service(self) -> builtins.str:
        '''``AWS::ECS::PrimaryTaskSet.Service``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-primarytaskset.html#cfn-ecs-primarytaskset-service
        '''
        return typing.cast(builtins.str, jsii.get(self, "service"))

    @service.setter
    def service(self, value: builtins.str) -> None:
        jsii.set(self, "service", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskSetId")
    def task_set_id(self) -> builtins.str:
        '''``AWS::ECS::PrimaryTaskSet.TaskSetId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-primarytaskset.html#cfn-ecs-primarytaskset-tasksetid
        '''
        return typing.cast(builtins.str, jsii.get(self, "taskSetId"))

    @task_set_id.setter
    def task_set_id(self, value: builtins.str) -> None:
        jsii.set(self, "taskSetId", value)


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.CfnPrimaryTaskSetProps",
    jsii_struct_bases=[],
    name_mapping={
        "cluster": "cluster",
        "service": "service",
        "task_set_id": "taskSetId",
    },
)
class CfnPrimaryTaskSetProps:
    def __init__(
        self,
        *,
        cluster: builtins.str,
        service: builtins.str,
        task_set_id: builtins.str,
    ) -> None:
        '''Properties for defining a ``AWS::ECS::PrimaryTaskSet``.

        :param cluster: ``AWS::ECS::PrimaryTaskSet.Cluster``.
        :param service: ``AWS::ECS::PrimaryTaskSet.Service``.
        :param task_set_id: ``AWS::ECS::PrimaryTaskSet.TaskSetId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-primarytaskset.html
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "cluster": cluster,
            "service": service,
            "task_set_id": task_set_id,
        }

    @builtins.property
    def cluster(self) -> builtins.str:
        '''``AWS::ECS::PrimaryTaskSet.Cluster``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-primarytaskset.html#cfn-ecs-primarytaskset-cluster
        '''
        result = self._values.get("cluster")
        assert result is not None, "Required property 'cluster' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def service(self) -> builtins.str:
        '''``AWS::ECS::PrimaryTaskSet.Service``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-primarytaskset.html#cfn-ecs-primarytaskset-service
        '''
        result = self._values.get("service")
        assert result is not None, "Required property 'service' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def task_set_id(self) -> builtins.str:
        '''``AWS::ECS::PrimaryTaskSet.TaskSetId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-primarytaskset.html#cfn-ecs-primarytaskset-tasksetid
        '''
        result = self._values.get("task_set_id")
        assert result is not None, "Required property 'task_set_id' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnPrimaryTaskSetProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnService(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_ecs.CfnService",
):
    '''A CloudFormation ``AWS::ECS::Service``.

    :cloudformationResource: AWS::ECS::Service
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        capacity_provider_strategy: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnService.CapacityProviderStrategyItemProperty", _IResolvable_a771d0ef]]]] = None,
        cluster: typing.Optional[builtins.str] = None,
        deployment_configuration: typing.Optional[typing.Union["CfnService.DeploymentConfigurationProperty", _IResolvable_a771d0ef]] = None,
        deployment_controller: typing.Optional[typing.Union["CfnService.DeploymentControllerProperty", _IResolvable_a771d0ef]] = None,
        desired_count: typing.Optional[jsii.Number] = None,
        enable_ecs_managed_tags: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
        enable_execute_command: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
        health_check_grace_period_seconds: typing.Optional[jsii.Number] = None,
        launch_type: typing.Optional[builtins.str] = None,
        load_balancers: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnService.LoadBalancerProperty", _IResolvable_a771d0ef]]]] = None,
        network_configuration: typing.Optional[typing.Union["CfnService.NetworkConfigurationProperty", _IResolvable_a771d0ef]] = None,
        placement_constraints: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnService.PlacementConstraintProperty", _IResolvable_a771d0ef]]]] = None,
        placement_strategies: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnService.PlacementStrategyProperty", _IResolvable_a771d0ef]]]] = None,
        platform_version: typing.Optional[builtins.str] = None,
        propagate_tags: typing.Optional[builtins.str] = None,
        role: typing.Optional[builtins.str] = None,
        scheduling_strategy: typing.Optional[builtins.str] = None,
        service_name: typing.Optional[builtins.str] = None,
        service_registries: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnService.ServiceRegistryProperty", _IResolvable_a771d0ef]]]] = None,
        tags: typing.Optional[typing.Sequence[_CfnTag_95fbdc29]] = None,
        task_definition: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::ECS::Service``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param capacity_provider_strategy: ``AWS::ECS::Service.CapacityProviderStrategy``.
        :param cluster: ``AWS::ECS::Service.Cluster``.
        :param deployment_configuration: ``AWS::ECS::Service.DeploymentConfiguration``.
        :param deployment_controller: ``AWS::ECS::Service.DeploymentController``.
        :param desired_count: ``AWS::ECS::Service.DesiredCount``.
        :param enable_ecs_managed_tags: ``AWS::ECS::Service.EnableECSManagedTags``.
        :param enable_execute_command: ``AWS::ECS::Service.EnableExecuteCommand``.
        :param health_check_grace_period_seconds: ``AWS::ECS::Service.HealthCheckGracePeriodSeconds``.
        :param launch_type: ``AWS::ECS::Service.LaunchType``.
        :param load_balancers: ``AWS::ECS::Service.LoadBalancers``.
        :param network_configuration: ``AWS::ECS::Service.NetworkConfiguration``.
        :param placement_constraints: ``AWS::ECS::Service.PlacementConstraints``.
        :param placement_strategies: ``AWS::ECS::Service.PlacementStrategies``.
        :param platform_version: ``AWS::ECS::Service.PlatformVersion``.
        :param propagate_tags: ``AWS::ECS::Service.PropagateTags``.
        :param role: ``AWS::ECS::Service.Role``.
        :param scheduling_strategy: ``AWS::ECS::Service.SchedulingStrategy``.
        :param service_name: ``AWS::ECS::Service.ServiceName``.
        :param service_registries: ``AWS::ECS::Service.ServiceRegistries``.
        :param tags: ``AWS::ECS::Service.Tags``.
        :param task_definition: ``AWS::ECS::Service.TaskDefinition``.
        '''
        props = CfnServiceProps(
            capacity_provider_strategy=capacity_provider_strategy,
            cluster=cluster,
            deployment_configuration=deployment_configuration,
            deployment_controller=deployment_controller,
            desired_count=desired_count,
            enable_ecs_managed_tags=enable_ecs_managed_tags,
            enable_execute_command=enable_execute_command,
            health_check_grace_period_seconds=health_check_grace_period_seconds,
            launch_type=launch_type,
            load_balancers=load_balancers,
            network_configuration=network_configuration,
            placement_constraints=placement_constraints,
            placement_strategies=placement_strategies,
            platform_version=platform_version,
            propagate_tags=propagate_tags,
            role=role,
            scheduling_strategy=scheduling_strategy,
            service_name=service_name,
            service_registries=service_registries,
            tags=tags,
            task_definition=task_definition,
        )

        jsii.create(CfnService, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrName")
    def attr_name(self) -> builtins.str:
        '''
        :cloudformationAttribute: Name
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrServiceArn")
    def attr_service_arn(self) -> builtins.str:
        '''
        :cloudformationAttribute: ServiceArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrServiceArn"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0b7ab120:
        '''``AWS::ECS::Service.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-tags
        '''
        return typing.cast(_TagManager_0b7ab120, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="capacityProviderStrategy")
    def capacity_provider_strategy(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnService.CapacityProviderStrategyItemProperty", _IResolvable_a771d0ef]]]]:
        '''``AWS::ECS::Service.CapacityProviderStrategy``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-capacityproviderstrategy
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnService.CapacityProviderStrategyItemProperty", _IResolvable_a771d0ef]]]], jsii.get(self, "capacityProviderStrategy"))

    @capacity_provider_strategy.setter
    def capacity_provider_strategy(
        self,
        value: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnService.CapacityProviderStrategyItemProperty", _IResolvable_a771d0ef]]]],
    ) -> None:
        jsii.set(self, "capacityProviderStrategy", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cluster")
    def cluster(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::Service.Cluster``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-cluster
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "cluster"))

    @cluster.setter
    def cluster(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "cluster", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="deploymentConfiguration")
    def deployment_configuration(
        self,
    ) -> typing.Optional[typing.Union["CfnService.DeploymentConfigurationProperty", _IResolvable_a771d0ef]]:
        '''``AWS::ECS::Service.DeploymentConfiguration``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-deploymentconfiguration
        '''
        return typing.cast(typing.Optional[typing.Union["CfnService.DeploymentConfigurationProperty", _IResolvable_a771d0ef]], jsii.get(self, "deploymentConfiguration"))

    @deployment_configuration.setter
    def deployment_configuration(
        self,
        value: typing.Optional[typing.Union["CfnService.DeploymentConfigurationProperty", _IResolvable_a771d0ef]],
    ) -> None:
        jsii.set(self, "deploymentConfiguration", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="deploymentController")
    def deployment_controller(
        self,
    ) -> typing.Optional[typing.Union["CfnService.DeploymentControllerProperty", _IResolvable_a771d0ef]]:
        '''``AWS::ECS::Service.DeploymentController``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-deploymentcontroller
        '''
        return typing.cast(typing.Optional[typing.Union["CfnService.DeploymentControllerProperty", _IResolvable_a771d0ef]], jsii.get(self, "deploymentController"))

    @deployment_controller.setter
    def deployment_controller(
        self,
        value: typing.Optional[typing.Union["CfnService.DeploymentControllerProperty", _IResolvable_a771d0ef]],
    ) -> None:
        jsii.set(self, "deploymentController", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="desiredCount")
    def desired_count(self) -> typing.Optional[jsii.Number]:
        '''``AWS::ECS::Service.DesiredCount``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-desiredcount
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "desiredCount"))

    @desired_count.setter
    def desired_count(self, value: typing.Optional[jsii.Number]) -> None:
        jsii.set(self, "desiredCount", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="enableEcsManagedTags")
    def enable_ecs_managed_tags(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
        '''``AWS::ECS::Service.EnableECSManagedTags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-enableecsmanagedtags
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], jsii.get(self, "enableEcsManagedTags"))

    @enable_ecs_managed_tags.setter
    def enable_ecs_managed_tags(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]],
    ) -> None:
        jsii.set(self, "enableEcsManagedTags", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="enableExecuteCommand")
    def enable_execute_command(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
        '''``AWS::ECS::Service.EnableExecuteCommand``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-enableexecutecommand
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], jsii.get(self, "enableExecuteCommand"))

    @enable_execute_command.setter
    def enable_execute_command(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]],
    ) -> None:
        jsii.set(self, "enableExecuteCommand", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="healthCheckGracePeriodSeconds")
    def health_check_grace_period_seconds(self) -> typing.Optional[jsii.Number]:
        '''``AWS::ECS::Service.HealthCheckGracePeriodSeconds``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-healthcheckgraceperiodseconds
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "healthCheckGracePeriodSeconds"))

    @health_check_grace_period_seconds.setter
    def health_check_grace_period_seconds(
        self,
        value: typing.Optional[jsii.Number],
    ) -> None:
        jsii.set(self, "healthCheckGracePeriodSeconds", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="launchType")
    def launch_type(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::Service.LaunchType``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-launchtype
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "launchType"))

    @launch_type.setter
    def launch_type(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "launchType", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="loadBalancers")
    def load_balancers(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnService.LoadBalancerProperty", _IResolvable_a771d0ef]]]]:
        '''``AWS::ECS::Service.LoadBalancers``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-loadbalancers
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnService.LoadBalancerProperty", _IResolvable_a771d0ef]]]], jsii.get(self, "loadBalancers"))

    @load_balancers.setter
    def load_balancers(
        self,
        value: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnService.LoadBalancerProperty", _IResolvable_a771d0ef]]]],
    ) -> None:
        jsii.set(self, "loadBalancers", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="networkConfiguration")
    def network_configuration(
        self,
    ) -> typing.Optional[typing.Union["CfnService.NetworkConfigurationProperty", _IResolvable_a771d0ef]]:
        '''``AWS::ECS::Service.NetworkConfiguration``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-networkconfiguration
        '''
        return typing.cast(typing.Optional[typing.Union["CfnService.NetworkConfigurationProperty", _IResolvable_a771d0ef]], jsii.get(self, "networkConfiguration"))

    @network_configuration.setter
    def network_configuration(
        self,
        value: typing.Optional[typing.Union["CfnService.NetworkConfigurationProperty", _IResolvable_a771d0ef]],
    ) -> None:
        jsii.set(self, "networkConfiguration", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="placementConstraints")
    def placement_constraints(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnService.PlacementConstraintProperty", _IResolvable_a771d0ef]]]]:
        '''``AWS::ECS::Service.PlacementConstraints``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-placementconstraints
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnService.PlacementConstraintProperty", _IResolvable_a771d0ef]]]], jsii.get(self, "placementConstraints"))

    @placement_constraints.setter
    def placement_constraints(
        self,
        value: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnService.PlacementConstraintProperty", _IResolvable_a771d0ef]]]],
    ) -> None:
        jsii.set(self, "placementConstraints", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="placementStrategies")
    def placement_strategies(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnService.PlacementStrategyProperty", _IResolvable_a771d0ef]]]]:
        '''``AWS::ECS::Service.PlacementStrategies``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-placementstrategies
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnService.PlacementStrategyProperty", _IResolvable_a771d0ef]]]], jsii.get(self, "placementStrategies"))

    @placement_strategies.setter
    def placement_strategies(
        self,
        value: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnService.PlacementStrategyProperty", _IResolvable_a771d0ef]]]],
    ) -> None:
        jsii.set(self, "placementStrategies", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="platformVersion")
    def platform_version(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::Service.PlatformVersion``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-platformversion
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "platformVersion"))

    @platform_version.setter
    def platform_version(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "platformVersion", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="propagateTags")
    def propagate_tags(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::Service.PropagateTags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-propagatetags
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "propagateTags"))

    @propagate_tags.setter
    def propagate_tags(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "propagateTags", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="role")
    def role(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::Service.Role``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-role
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "role"))

    @role.setter
    def role(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "role", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="schedulingStrategy")
    def scheduling_strategy(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::Service.SchedulingStrategy``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-schedulingstrategy
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "schedulingStrategy"))

    @scheduling_strategy.setter
    def scheduling_strategy(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "schedulingStrategy", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="serviceName")
    def service_name(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::Service.ServiceName``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-servicename
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "serviceName"))

    @service_name.setter
    def service_name(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "serviceName", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="serviceRegistries")
    def service_registries(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnService.ServiceRegistryProperty", _IResolvable_a771d0ef]]]]:
        '''``AWS::ECS::Service.ServiceRegistries``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-serviceregistries
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnService.ServiceRegistryProperty", _IResolvable_a771d0ef]]]], jsii.get(self, "serviceRegistries"))

    @service_registries.setter
    def service_registries(
        self,
        value: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnService.ServiceRegistryProperty", _IResolvable_a771d0ef]]]],
    ) -> None:
        jsii.set(self, "serviceRegistries", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskDefinition")
    def task_definition(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::Service.TaskDefinition``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-taskdefinition
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "taskDefinition"))

    @task_definition.setter
    def task_definition(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "taskDefinition", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnService.AwsVpcConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "subnets": "subnets",
            "assign_public_ip": "assignPublicIp",
            "security_groups": "securityGroups",
        },
    )
    class AwsVpcConfigurationProperty:
        def __init__(
            self,
            *,
            subnets: typing.Sequence[builtins.str],
            assign_public_ip: typing.Optional[builtins.str] = None,
            security_groups: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''
            :param subnets: ``CfnService.AwsVpcConfigurationProperty.Subnets``.
            :param assign_public_ip: ``CfnService.AwsVpcConfigurationProperty.AssignPublicIp``.
            :param security_groups: ``CfnService.AwsVpcConfigurationProperty.SecurityGroups``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-awsvpcconfiguration.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "subnets": subnets,
            }
            if assign_public_ip is not None:
                self._values["assign_public_ip"] = assign_public_ip
            if security_groups is not None:
                self._values["security_groups"] = security_groups

        @builtins.property
        def subnets(self) -> typing.List[builtins.str]:
            '''``CfnService.AwsVpcConfigurationProperty.Subnets``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-awsvpcconfiguration.html#cfn-ecs-service-awsvpcconfiguration-subnets
            '''
            result = self._values.get("subnets")
            assert result is not None, "Required property 'subnets' is missing"
            return typing.cast(typing.List[builtins.str], result)

        @builtins.property
        def assign_public_ip(self) -> typing.Optional[builtins.str]:
            '''``CfnService.AwsVpcConfigurationProperty.AssignPublicIp``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-awsvpcconfiguration.html#cfn-ecs-service-awsvpcconfiguration-assignpublicip
            '''
            result = self._values.get("assign_public_ip")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def security_groups(self) -> typing.Optional[typing.List[builtins.str]]:
            '''``CfnService.AwsVpcConfigurationProperty.SecurityGroups``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-awsvpcconfiguration.html#cfn-ecs-service-awsvpcconfiguration-securitygroups
            '''
            result = self._values.get("security_groups")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AwsVpcConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnService.CapacityProviderStrategyItemProperty",
        jsii_struct_bases=[],
        name_mapping={
            "base": "base",
            "capacity_provider": "capacityProvider",
            "weight": "weight",
        },
    )
    class CapacityProviderStrategyItemProperty:
        def __init__(
            self,
            *,
            base: typing.Optional[jsii.Number] = None,
            capacity_provider: typing.Optional[builtins.str] = None,
            weight: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''
            :param base: ``CfnService.CapacityProviderStrategyItemProperty.Base``.
            :param capacity_provider: ``CfnService.CapacityProviderStrategyItemProperty.CapacityProvider``.
            :param weight: ``CfnService.CapacityProviderStrategyItemProperty.Weight``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-capacityproviderstrategyitem.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if base is not None:
                self._values["base"] = base
            if capacity_provider is not None:
                self._values["capacity_provider"] = capacity_provider
            if weight is not None:
                self._values["weight"] = weight

        @builtins.property
        def base(self) -> typing.Optional[jsii.Number]:
            '''``CfnService.CapacityProviderStrategyItemProperty.Base``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-capacityproviderstrategyitem.html#cfn-ecs-service-capacityproviderstrategyitem-base
            '''
            result = self._values.get("base")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def capacity_provider(self) -> typing.Optional[builtins.str]:
            '''``CfnService.CapacityProviderStrategyItemProperty.CapacityProvider``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-capacityproviderstrategyitem.html#cfn-ecs-service-capacityproviderstrategyitem-capacityprovider
            '''
            result = self._values.get("capacity_provider")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def weight(self) -> typing.Optional[jsii.Number]:
            '''``CfnService.CapacityProviderStrategyItemProperty.Weight``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-capacityproviderstrategyitem.html#cfn-ecs-service-capacityproviderstrategyitem-weight
            '''
            result = self._values.get("weight")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CapacityProviderStrategyItemProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnService.DeploymentCircuitBreakerProperty",
        jsii_struct_bases=[],
        name_mapping={"enable": "enable", "rollback": "rollback"},
    )
    class DeploymentCircuitBreakerProperty:
        def __init__(
            self,
            *,
            enable: typing.Union[builtins.bool, _IResolvable_a771d0ef],
            rollback: typing.Union[builtins.bool, _IResolvable_a771d0ef],
        ) -> None:
            '''
            :param enable: ``CfnService.DeploymentCircuitBreakerProperty.Enable``.
            :param rollback: ``CfnService.DeploymentCircuitBreakerProperty.Rollback``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-deploymentcircuitbreaker.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "enable": enable,
                "rollback": rollback,
            }

        @builtins.property
        def enable(self) -> typing.Union[builtins.bool, _IResolvable_a771d0ef]:
            '''``CfnService.DeploymentCircuitBreakerProperty.Enable``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-deploymentcircuitbreaker.html#cfn-ecs-service-deploymentcircuitbreaker-enable
            '''
            result = self._values.get("enable")
            assert result is not None, "Required property 'enable' is missing"
            return typing.cast(typing.Union[builtins.bool, _IResolvable_a771d0ef], result)

        @builtins.property
        def rollback(self) -> typing.Union[builtins.bool, _IResolvable_a771d0ef]:
            '''``CfnService.DeploymentCircuitBreakerProperty.Rollback``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-deploymentcircuitbreaker.html#cfn-ecs-service-deploymentcircuitbreaker-rollback
            '''
            result = self._values.get("rollback")
            assert result is not None, "Required property 'rollback' is missing"
            return typing.cast(typing.Union[builtins.bool, _IResolvable_a771d0ef], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DeploymentCircuitBreakerProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnService.DeploymentConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "deployment_circuit_breaker": "deploymentCircuitBreaker",
            "maximum_percent": "maximumPercent",
            "minimum_healthy_percent": "minimumHealthyPercent",
        },
    )
    class DeploymentConfigurationProperty:
        def __init__(
            self,
            *,
            deployment_circuit_breaker: typing.Optional[typing.Union["CfnService.DeploymentCircuitBreakerProperty", _IResolvable_a771d0ef]] = None,
            maximum_percent: typing.Optional[jsii.Number] = None,
            minimum_healthy_percent: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''
            :param deployment_circuit_breaker: ``CfnService.DeploymentConfigurationProperty.DeploymentCircuitBreaker``.
            :param maximum_percent: ``CfnService.DeploymentConfigurationProperty.MaximumPercent``.
            :param minimum_healthy_percent: ``CfnService.DeploymentConfigurationProperty.MinimumHealthyPercent``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-deploymentconfiguration.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if deployment_circuit_breaker is not None:
                self._values["deployment_circuit_breaker"] = deployment_circuit_breaker
            if maximum_percent is not None:
                self._values["maximum_percent"] = maximum_percent
            if minimum_healthy_percent is not None:
                self._values["minimum_healthy_percent"] = minimum_healthy_percent

        @builtins.property
        def deployment_circuit_breaker(
            self,
        ) -> typing.Optional[typing.Union["CfnService.DeploymentCircuitBreakerProperty", _IResolvable_a771d0ef]]:
            '''``CfnService.DeploymentConfigurationProperty.DeploymentCircuitBreaker``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-deploymentconfiguration.html#cfn-ecs-service-deploymentconfiguration-deploymentcircuitbreaker
            '''
            result = self._values.get("deployment_circuit_breaker")
            return typing.cast(typing.Optional[typing.Union["CfnService.DeploymentCircuitBreakerProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def maximum_percent(self) -> typing.Optional[jsii.Number]:
            '''``CfnService.DeploymentConfigurationProperty.MaximumPercent``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-deploymentconfiguration.html#cfn-ecs-service-deploymentconfiguration-maximumpercent
            '''
            result = self._values.get("maximum_percent")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def minimum_healthy_percent(self) -> typing.Optional[jsii.Number]:
            '''``CfnService.DeploymentConfigurationProperty.MinimumHealthyPercent``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-deploymentconfiguration.html#cfn-ecs-service-deploymentconfiguration-minimumhealthypercent
            '''
            result = self._values.get("minimum_healthy_percent")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DeploymentConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnService.DeploymentControllerProperty",
        jsii_struct_bases=[],
        name_mapping={"type": "type"},
    )
    class DeploymentControllerProperty:
        def __init__(self, *, type: typing.Optional[builtins.str] = None) -> None:
            '''
            :param type: ``CfnService.DeploymentControllerProperty.Type``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-deploymentcontroller.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if type is not None:
                self._values["type"] = type

        @builtins.property
        def type(self) -> typing.Optional[builtins.str]:
            '''``CfnService.DeploymentControllerProperty.Type``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-deploymentcontroller.html#cfn-ecs-service-deploymentcontroller-type
            '''
            result = self._values.get("type")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DeploymentControllerProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnService.LoadBalancerProperty",
        jsii_struct_bases=[],
        name_mapping={
            "container_port": "containerPort",
            "container_name": "containerName",
            "load_balancer_name": "loadBalancerName",
            "target_group_arn": "targetGroupArn",
        },
    )
    class LoadBalancerProperty:
        def __init__(
            self,
            *,
            container_port: jsii.Number,
            container_name: typing.Optional[builtins.str] = None,
            load_balancer_name: typing.Optional[builtins.str] = None,
            target_group_arn: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param container_port: ``CfnService.LoadBalancerProperty.ContainerPort``.
            :param container_name: ``CfnService.LoadBalancerProperty.ContainerName``.
            :param load_balancer_name: ``CfnService.LoadBalancerProperty.LoadBalancerName``.
            :param target_group_arn: ``CfnService.LoadBalancerProperty.TargetGroupArn``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-loadbalancer.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "container_port": container_port,
            }
            if container_name is not None:
                self._values["container_name"] = container_name
            if load_balancer_name is not None:
                self._values["load_balancer_name"] = load_balancer_name
            if target_group_arn is not None:
                self._values["target_group_arn"] = target_group_arn

        @builtins.property
        def container_port(self) -> jsii.Number:
            '''``CfnService.LoadBalancerProperty.ContainerPort``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-loadbalancer.html#cfn-ecs-service-loadbalancer-containerport
            '''
            result = self._values.get("container_port")
            assert result is not None, "Required property 'container_port' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def container_name(self) -> typing.Optional[builtins.str]:
            '''``CfnService.LoadBalancerProperty.ContainerName``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-loadbalancer.html#cfn-ecs-service-loadbalancer-containername
            '''
            result = self._values.get("container_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def load_balancer_name(self) -> typing.Optional[builtins.str]:
            '''``CfnService.LoadBalancerProperty.LoadBalancerName``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-loadbalancer.html#cfn-ecs-service-loadbalancer-loadbalancername
            '''
            result = self._values.get("load_balancer_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def target_group_arn(self) -> typing.Optional[builtins.str]:
            '''``CfnService.LoadBalancerProperty.TargetGroupArn``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-loadbalancer.html#cfn-ecs-service-loadbalancer-targetgrouparn
            '''
            result = self._values.get("target_group_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LoadBalancerProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnService.NetworkConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"awsvpc_configuration": "awsvpcConfiguration"},
    )
    class NetworkConfigurationProperty:
        def __init__(
            self,
            *,
            awsvpc_configuration: typing.Optional[typing.Union["CfnService.AwsVpcConfigurationProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''
            :param awsvpc_configuration: ``CfnService.NetworkConfigurationProperty.AwsvpcConfiguration``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-networkconfiguration.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if awsvpc_configuration is not None:
                self._values["awsvpc_configuration"] = awsvpc_configuration

        @builtins.property
        def awsvpc_configuration(
            self,
        ) -> typing.Optional[typing.Union["CfnService.AwsVpcConfigurationProperty", _IResolvable_a771d0ef]]:
            '''``CfnService.NetworkConfigurationProperty.AwsvpcConfiguration``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-networkconfiguration.html#cfn-ecs-service-networkconfiguration-awsvpcconfiguration
            '''
            result = self._values.get("awsvpc_configuration")
            return typing.cast(typing.Optional[typing.Union["CfnService.AwsVpcConfigurationProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "NetworkConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnService.PlacementConstraintProperty",
        jsii_struct_bases=[],
        name_mapping={"type": "type", "expression": "expression"},
    )
    class PlacementConstraintProperty:
        def __init__(
            self,
            *,
            type: builtins.str,
            expression: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param type: ``CfnService.PlacementConstraintProperty.Type``.
            :param expression: ``CfnService.PlacementConstraintProperty.Expression``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-placementconstraint.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "type": type,
            }
            if expression is not None:
                self._values["expression"] = expression

        @builtins.property
        def type(self) -> builtins.str:
            '''``CfnService.PlacementConstraintProperty.Type``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-placementconstraint.html#cfn-ecs-service-placementconstraint-type
            '''
            result = self._values.get("type")
            assert result is not None, "Required property 'type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def expression(self) -> typing.Optional[builtins.str]:
            '''``CfnService.PlacementConstraintProperty.Expression``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-placementconstraint.html#cfn-ecs-service-placementconstraint-expression
            '''
            result = self._values.get("expression")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PlacementConstraintProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnService.PlacementStrategyProperty",
        jsii_struct_bases=[],
        name_mapping={"type": "type", "field": "field"},
    )
    class PlacementStrategyProperty:
        def __init__(
            self,
            *,
            type: builtins.str,
            field: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param type: ``CfnService.PlacementStrategyProperty.Type``.
            :param field: ``CfnService.PlacementStrategyProperty.Field``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-placementstrategy.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "type": type,
            }
            if field is not None:
                self._values["field"] = field

        @builtins.property
        def type(self) -> builtins.str:
            '''``CfnService.PlacementStrategyProperty.Type``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-placementstrategy.html#cfn-ecs-service-placementstrategy-type
            '''
            result = self._values.get("type")
            assert result is not None, "Required property 'type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def field(self) -> typing.Optional[builtins.str]:
            '''``CfnService.PlacementStrategyProperty.Field``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-placementstrategy.html#cfn-ecs-service-placementstrategy-field
            '''
            result = self._values.get("field")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PlacementStrategyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnService.ServiceRegistryProperty",
        jsii_struct_bases=[],
        name_mapping={
            "container_name": "containerName",
            "container_port": "containerPort",
            "port": "port",
            "registry_arn": "registryArn",
        },
    )
    class ServiceRegistryProperty:
        def __init__(
            self,
            *,
            container_name: typing.Optional[builtins.str] = None,
            container_port: typing.Optional[jsii.Number] = None,
            port: typing.Optional[jsii.Number] = None,
            registry_arn: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param container_name: ``CfnService.ServiceRegistryProperty.ContainerName``.
            :param container_port: ``CfnService.ServiceRegistryProperty.ContainerPort``.
            :param port: ``CfnService.ServiceRegistryProperty.Port``.
            :param registry_arn: ``CfnService.ServiceRegistryProperty.RegistryArn``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-serviceregistry.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if container_name is not None:
                self._values["container_name"] = container_name
            if container_port is not None:
                self._values["container_port"] = container_port
            if port is not None:
                self._values["port"] = port
            if registry_arn is not None:
                self._values["registry_arn"] = registry_arn

        @builtins.property
        def container_name(self) -> typing.Optional[builtins.str]:
            '''``CfnService.ServiceRegistryProperty.ContainerName``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-serviceregistry.html#cfn-ecs-service-serviceregistry-containername
            '''
            result = self._values.get("container_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def container_port(self) -> typing.Optional[jsii.Number]:
            '''``CfnService.ServiceRegistryProperty.ContainerPort``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-serviceregistry.html#cfn-ecs-service-serviceregistry-containerport
            '''
            result = self._values.get("container_port")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def port(self) -> typing.Optional[jsii.Number]:
            '''``CfnService.ServiceRegistryProperty.Port``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-serviceregistry.html#cfn-ecs-service-serviceregistry-port
            '''
            result = self._values.get("port")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def registry_arn(self) -> typing.Optional[builtins.str]:
            '''``CfnService.ServiceRegistryProperty.RegistryArn``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-serviceregistry.html#cfn-ecs-service-serviceregistry-registryarn
            '''
            result = self._values.get("registry_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ServiceRegistryProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.CfnServiceProps",
    jsii_struct_bases=[],
    name_mapping={
        "capacity_provider_strategy": "capacityProviderStrategy",
        "cluster": "cluster",
        "deployment_configuration": "deploymentConfiguration",
        "deployment_controller": "deploymentController",
        "desired_count": "desiredCount",
        "enable_ecs_managed_tags": "enableEcsManagedTags",
        "enable_execute_command": "enableExecuteCommand",
        "health_check_grace_period_seconds": "healthCheckGracePeriodSeconds",
        "launch_type": "launchType",
        "load_balancers": "loadBalancers",
        "network_configuration": "networkConfiguration",
        "placement_constraints": "placementConstraints",
        "placement_strategies": "placementStrategies",
        "platform_version": "platformVersion",
        "propagate_tags": "propagateTags",
        "role": "role",
        "scheduling_strategy": "schedulingStrategy",
        "service_name": "serviceName",
        "service_registries": "serviceRegistries",
        "tags": "tags",
        "task_definition": "taskDefinition",
    },
)
class CfnServiceProps:
    def __init__(
        self,
        *,
        capacity_provider_strategy: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[CfnService.CapacityProviderStrategyItemProperty, _IResolvable_a771d0ef]]]] = None,
        cluster: typing.Optional[builtins.str] = None,
        deployment_configuration: typing.Optional[typing.Union[CfnService.DeploymentConfigurationProperty, _IResolvable_a771d0ef]] = None,
        deployment_controller: typing.Optional[typing.Union[CfnService.DeploymentControllerProperty, _IResolvable_a771d0ef]] = None,
        desired_count: typing.Optional[jsii.Number] = None,
        enable_ecs_managed_tags: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
        enable_execute_command: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
        health_check_grace_period_seconds: typing.Optional[jsii.Number] = None,
        launch_type: typing.Optional[builtins.str] = None,
        load_balancers: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[CfnService.LoadBalancerProperty, _IResolvable_a771d0ef]]]] = None,
        network_configuration: typing.Optional[typing.Union[CfnService.NetworkConfigurationProperty, _IResolvable_a771d0ef]] = None,
        placement_constraints: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[CfnService.PlacementConstraintProperty, _IResolvable_a771d0ef]]]] = None,
        placement_strategies: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[CfnService.PlacementStrategyProperty, _IResolvable_a771d0ef]]]] = None,
        platform_version: typing.Optional[builtins.str] = None,
        propagate_tags: typing.Optional[builtins.str] = None,
        role: typing.Optional[builtins.str] = None,
        scheduling_strategy: typing.Optional[builtins.str] = None,
        service_name: typing.Optional[builtins.str] = None,
        service_registries: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[CfnService.ServiceRegistryProperty, _IResolvable_a771d0ef]]]] = None,
        tags: typing.Optional[typing.Sequence[_CfnTag_95fbdc29]] = None,
        task_definition: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``AWS::ECS::Service``.

        :param capacity_provider_strategy: ``AWS::ECS::Service.CapacityProviderStrategy``.
        :param cluster: ``AWS::ECS::Service.Cluster``.
        :param deployment_configuration: ``AWS::ECS::Service.DeploymentConfiguration``.
        :param deployment_controller: ``AWS::ECS::Service.DeploymentController``.
        :param desired_count: ``AWS::ECS::Service.DesiredCount``.
        :param enable_ecs_managed_tags: ``AWS::ECS::Service.EnableECSManagedTags``.
        :param enable_execute_command: ``AWS::ECS::Service.EnableExecuteCommand``.
        :param health_check_grace_period_seconds: ``AWS::ECS::Service.HealthCheckGracePeriodSeconds``.
        :param launch_type: ``AWS::ECS::Service.LaunchType``.
        :param load_balancers: ``AWS::ECS::Service.LoadBalancers``.
        :param network_configuration: ``AWS::ECS::Service.NetworkConfiguration``.
        :param placement_constraints: ``AWS::ECS::Service.PlacementConstraints``.
        :param placement_strategies: ``AWS::ECS::Service.PlacementStrategies``.
        :param platform_version: ``AWS::ECS::Service.PlatformVersion``.
        :param propagate_tags: ``AWS::ECS::Service.PropagateTags``.
        :param role: ``AWS::ECS::Service.Role``.
        :param scheduling_strategy: ``AWS::ECS::Service.SchedulingStrategy``.
        :param service_name: ``AWS::ECS::Service.ServiceName``.
        :param service_registries: ``AWS::ECS::Service.ServiceRegistries``.
        :param tags: ``AWS::ECS::Service.Tags``.
        :param task_definition: ``AWS::ECS::Service.TaskDefinition``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if capacity_provider_strategy is not None:
            self._values["capacity_provider_strategy"] = capacity_provider_strategy
        if cluster is not None:
            self._values["cluster"] = cluster
        if deployment_configuration is not None:
            self._values["deployment_configuration"] = deployment_configuration
        if deployment_controller is not None:
            self._values["deployment_controller"] = deployment_controller
        if desired_count is not None:
            self._values["desired_count"] = desired_count
        if enable_ecs_managed_tags is not None:
            self._values["enable_ecs_managed_tags"] = enable_ecs_managed_tags
        if enable_execute_command is not None:
            self._values["enable_execute_command"] = enable_execute_command
        if health_check_grace_period_seconds is not None:
            self._values["health_check_grace_period_seconds"] = health_check_grace_period_seconds
        if launch_type is not None:
            self._values["launch_type"] = launch_type
        if load_balancers is not None:
            self._values["load_balancers"] = load_balancers
        if network_configuration is not None:
            self._values["network_configuration"] = network_configuration
        if placement_constraints is not None:
            self._values["placement_constraints"] = placement_constraints
        if placement_strategies is not None:
            self._values["placement_strategies"] = placement_strategies
        if platform_version is not None:
            self._values["platform_version"] = platform_version
        if propagate_tags is not None:
            self._values["propagate_tags"] = propagate_tags
        if role is not None:
            self._values["role"] = role
        if scheduling_strategy is not None:
            self._values["scheduling_strategy"] = scheduling_strategy
        if service_name is not None:
            self._values["service_name"] = service_name
        if service_registries is not None:
            self._values["service_registries"] = service_registries
        if tags is not None:
            self._values["tags"] = tags
        if task_definition is not None:
            self._values["task_definition"] = task_definition

    @builtins.property
    def capacity_provider_strategy(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnService.CapacityProviderStrategyItemProperty, _IResolvable_a771d0ef]]]]:
        '''``AWS::ECS::Service.CapacityProviderStrategy``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-capacityproviderstrategy
        '''
        result = self._values.get("capacity_provider_strategy")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnService.CapacityProviderStrategyItemProperty, _IResolvable_a771d0ef]]]], result)

    @builtins.property
    def cluster(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::Service.Cluster``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-cluster
        '''
        result = self._values.get("cluster")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def deployment_configuration(
        self,
    ) -> typing.Optional[typing.Union[CfnService.DeploymentConfigurationProperty, _IResolvable_a771d0ef]]:
        '''``AWS::ECS::Service.DeploymentConfiguration``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-deploymentconfiguration
        '''
        result = self._values.get("deployment_configuration")
        return typing.cast(typing.Optional[typing.Union[CfnService.DeploymentConfigurationProperty, _IResolvable_a771d0ef]], result)

    @builtins.property
    def deployment_controller(
        self,
    ) -> typing.Optional[typing.Union[CfnService.DeploymentControllerProperty, _IResolvable_a771d0ef]]:
        '''``AWS::ECS::Service.DeploymentController``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-deploymentcontroller
        '''
        result = self._values.get("deployment_controller")
        return typing.cast(typing.Optional[typing.Union[CfnService.DeploymentControllerProperty, _IResolvable_a771d0ef]], result)

    @builtins.property
    def desired_count(self) -> typing.Optional[jsii.Number]:
        '''``AWS::ECS::Service.DesiredCount``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-desiredcount
        '''
        result = self._values.get("desired_count")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def enable_ecs_managed_tags(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
        '''``AWS::ECS::Service.EnableECSManagedTags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-enableecsmanagedtags
        '''
        result = self._values.get("enable_ecs_managed_tags")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

    @builtins.property
    def enable_execute_command(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
        '''``AWS::ECS::Service.EnableExecuteCommand``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-enableexecutecommand
        '''
        result = self._values.get("enable_execute_command")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

    @builtins.property
    def health_check_grace_period_seconds(self) -> typing.Optional[jsii.Number]:
        '''``AWS::ECS::Service.HealthCheckGracePeriodSeconds``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-healthcheckgraceperiodseconds
        '''
        result = self._values.get("health_check_grace_period_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def launch_type(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::Service.LaunchType``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-launchtype
        '''
        result = self._values.get("launch_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def load_balancers(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnService.LoadBalancerProperty, _IResolvable_a771d0ef]]]]:
        '''``AWS::ECS::Service.LoadBalancers``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-loadbalancers
        '''
        result = self._values.get("load_balancers")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnService.LoadBalancerProperty, _IResolvable_a771d0ef]]]], result)

    @builtins.property
    def network_configuration(
        self,
    ) -> typing.Optional[typing.Union[CfnService.NetworkConfigurationProperty, _IResolvable_a771d0ef]]:
        '''``AWS::ECS::Service.NetworkConfiguration``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-networkconfiguration
        '''
        result = self._values.get("network_configuration")
        return typing.cast(typing.Optional[typing.Union[CfnService.NetworkConfigurationProperty, _IResolvable_a771d0ef]], result)

    @builtins.property
    def placement_constraints(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnService.PlacementConstraintProperty, _IResolvable_a771d0ef]]]]:
        '''``AWS::ECS::Service.PlacementConstraints``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-placementconstraints
        '''
        result = self._values.get("placement_constraints")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnService.PlacementConstraintProperty, _IResolvable_a771d0ef]]]], result)

    @builtins.property
    def placement_strategies(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnService.PlacementStrategyProperty, _IResolvable_a771d0ef]]]]:
        '''``AWS::ECS::Service.PlacementStrategies``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-placementstrategies
        '''
        result = self._values.get("placement_strategies")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnService.PlacementStrategyProperty, _IResolvable_a771d0ef]]]], result)

    @builtins.property
    def platform_version(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::Service.PlatformVersion``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-platformversion
        '''
        result = self._values.get("platform_version")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def propagate_tags(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::Service.PropagateTags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-propagatetags
        '''
        result = self._values.get("propagate_tags")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def role(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::Service.Role``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-role
        '''
        result = self._values.get("role")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def scheduling_strategy(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::Service.SchedulingStrategy``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-schedulingstrategy
        '''
        result = self._values.get("scheduling_strategy")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def service_name(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::Service.ServiceName``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-servicename
        '''
        result = self._values.get("service_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def service_registries(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnService.ServiceRegistryProperty, _IResolvable_a771d0ef]]]]:
        '''``AWS::ECS::Service.ServiceRegistries``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-serviceregistries
        '''
        result = self._values.get("service_registries")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnService.ServiceRegistryProperty, _IResolvable_a771d0ef]]]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_95fbdc29]]:
        '''``AWS::ECS::Service.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_95fbdc29]], result)

    @builtins.property
    def task_definition(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::Service.TaskDefinition``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-taskdefinition
        '''
        result = self._values.get("task_definition")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnServiceProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnTaskDefinition(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_ecs.CfnTaskDefinition",
):
    '''A CloudFormation ``AWS::ECS::TaskDefinition``.

    :cloudformationResource: AWS::ECS::TaskDefinition
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        container_definitions: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskDefinition.ContainerDefinitionProperty", _IResolvable_a771d0ef]]]] = None,
        cpu: typing.Optional[builtins.str] = None,
        ephemeral_storage: typing.Optional[typing.Union["CfnTaskDefinition.EphemeralStorageProperty", _IResolvable_a771d0ef]] = None,
        execution_role_arn: typing.Optional[builtins.str] = None,
        family: typing.Optional[builtins.str] = None,
        inference_accelerators: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskDefinition.InferenceAcceleratorProperty", _IResolvable_a771d0ef]]]] = None,
        ipc_mode: typing.Optional[builtins.str] = None,
        memory: typing.Optional[builtins.str] = None,
        network_mode: typing.Optional[builtins.str] = None,
        pid_mode: typing.Optional[builtins.str] = None,
        placement_constraints: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskDefinition.TaskDefinitionPlacementConstraintProperty", _IResolvable_a771d0ef]]]] = None,
        proxy_configuration: typing.Optional[typing.Union["CfnTaskDefinition.ProxyConfigurationProperty", _IResolvable_a771d0ef]] = None,
        requires_compatibilities: typing.Optional[typing.Sequence[builtins.str]] = None,
        tags: typing.Optional[typing.Sequence[_CfnTag_95fbdc29]] = None,
        task_role_arn: typing.Optional[builtins.str] = None,
        volumes: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskDefinition.VolumeProperty", _IResolvable_a771d0ef]]]] = None,
    ) -> None:
        '''Create a new ``AWS::ECS::TaskDefinition``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param container_definitions: ``AWS::ECS::TaskDefinition.ContainerDefinitions``.
        :param cpu: ``AWS::ECS::TaskDefinition.Cpu``.
        :param ephemeral_storage: ``AWS::ECS::TaskDefinition.EphemeralStorage``.
        :param execution_role_arn: ``AWS::ECS::TaskDefinition.ExecutionRoleArn``.
        :param family: ``AWS::ECS::TaskDefinition.Family``.
        :param inference_accelerators: ``AWS::ECS::TaskDefinition.InferenceAccelerators``.
        :param ipc_mode: ``AWS::ECS::TaskDefinition.IpcMode``.
        :param memory: ``AWS::ECS::TaskDefinition.Memory``.
        :param network_mode: ``AWS::ECS::TaskDefinition.NetworkMode``.
        :param pid_mode: ``AWS::ECS::TaskDefinition.PidMode``.
        :param placement_constraints: ``AWS::ECS::TaskDefinition.PlacementConstraints``.
        :param proxy_configuration: ``AWS::ECS::TaskDefinition.ProxyConfiguration``.
        :param requires_compatibilities: ``AWS::ECS::TaskDefinition.RequiresCompatibilities``.
        :param tags: ``AWS::ECS::TaskDefinition.Tags``.
        :param task_role_arn: ``AWS::ECS::TaskDefinition.TaskRoleArn``.
        :param volumes: ``AWS::ECS::TaskDefinition.Volumes``.
        '''
        props = CfnTaskDefinitionProps(
            container_definitions=container_definitions,
            cpu=cpu,
            ephemeral_storage=ephemeral_storage,
            execution_role_arn=execution_role_arn,
            family=family,
            inference_accelerators=inference_accelerators,
            ipc_mode=ipc_mode,
            memory=memory,
            network_mode=network_mode,
            pid_mode=pid_mode,
            placement_constraints=placement_constraints,
            proxy_configuration=proxy_configuration,
            requires_compatibilities=requires_compatibilities,
            tags=tags,
            task_role_arn=task_role_arn,
            volumes=volumes,
        )

        jsii.create(CfnTaskDefinition, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrTaskDefinitionArn")
    def attr_task_definition_arn(self) -> builtins.str:
        '''
        :cloudformationAttribute: TaskDefinitionArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrTaskDefinitionArn"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0b7ab120:
        '''``AWS::ECS::TaskDefinition.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-tags
        '''
        return typing.cast(_TagManager_0b7ab120, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="containerDefinitions")
    def container_definitions(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.ContainerDefinitionProperty", _IResolvable_a771d0ef]]]]:
        '''``AWS::ECS::TaskDefinition.ContainerDefinitions``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-containerdefinitions
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.ContainerDefinitionProperty", _IResolvable_a771d0ef]]]], jsii.get(self, "containerDefinitions"))

    @container_definitions.setter
    def container_definitions(
        self,
        value: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.ContainerDefinitionProperty", _IResolvable_a771d0ef]]]],
    ) -> None:
        jsii.set(self, "containerDefinitions", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cpu")
    def cpu(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::TaskDefinition.Cpu``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-cpu
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "cpu"))

    @cpu.setter
    def cpu(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "cpu", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ephemeralStorage")
    def ephemeral_storage(
        self,
    ) -> typing.Optional[typing.Union["CfnTaskDefinition.EphemeralStorageProperty", _IResolvable_a771d0ef]]:
        '''``AWS::ECS::TaskDefinition.EphemeralStorage``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-ephemeralstorage
        '''
        return typing.cast(typing.Optional[typing.Union["CfnTaskDefinition.EphemeralStorageProperty", _IResolvable_a771d0ef]], jsii.get(self, "ephemeralStorage"))

    @ephemeral_storage.setter
    def ephemeral_storage(
        self,
        value: typing.Optional[typing.Union["CfnTaskDefinition.EphemeralStorageProperty", _IResolvable_a771d0ef]],
    ) -> None:
        jsii.set(self, "ephemeralStorage", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="executionRoleArn")
    def execution_role_arn(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::TaskDefinition.ExecutionRoleArn``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-executionrolearn
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "executionRoleArn"))

    @execution_role_arn.setter
    def execution_role_arn(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "executionRoleArn", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="family")
    def family(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::TaskDefinition.Family``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-family
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "family"))

    @family.setter
    def family(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "family", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="inferenceAccelerators")
    def inference_accelerators(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.InferenceAcceleratorProperty", _IResolvable_a771d0ef]]]]:
        '''``AWS::ECS::TaskDefinition.InferenceAccelerators``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-inferenceaccelerators
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.InferenceAcceleratorProperty", _IResolvable_a771d0ef]]]], jsii.get(self, "inferenceAccelerators"))

    @inference_accelerators.setter
    def inference_accelerators(
        self,
        value: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.InferenceAcceleratorProperty", _IResolvable_a771d0ef]]]],
    ) -> None:
        jsii.set(self, "inferenceAccelerators", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ipcMode")
    def ipc_mode(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::TaskDefinition.IpcMode``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-ipcmode
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "ipcMode"))

    @ipc_mode.setter
    def ipc_mode(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "ipcMode", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="memory")
    def memory(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::TaskDefinition.Memory``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-memory
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "memory"))

    @memory.setter
    def memory(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "memory", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="networkMode")
    def network_mode(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::TaskDefinition.NetworkMode``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-networkmode
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "networkMode"))

    @network_mode.setter
    def network_mode(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "networkMode", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="pidMode")
    def pid_mode(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::TaskDefinition.PidMode``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-pidmode
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "pidMode"))

    @pid_mode.setter
    def pid_mode(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "pidMode", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="placementConstraints")
    def placement_constraints(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.TaskDefinitionPlacementConstraintProperty", _IResolvable_a771d0ef]]]]:
        '''``AWS::ECS::TaskDefinition.PlacementConstraints``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-placementconstraints
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.TaskDefinitionPlacementConstraintProperty", _IResolvable_a771d0ef]]]], jsii.get(self, "placementConstraints"))

    @placement_constraints.setter
    def placement_constraints(
        self,
        value: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.TaskDefinitionPlacementConstraintProperty", _IResolvable_a771d0ef]]]],
    ) -> None:
        jsii.set(self, "placementConstraints", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="proxyConfiguration")
    def proxy_configuration(
        self,
    ) -> typing.Optional[typing.Union["CfnTaskDefinition.ProxyConfigurationProperty", _IResolvable_a771d0ef]]:
        '''``AWS::ECS::TaskDefinition.ProxyConfiguration``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-proxyconfiguration
        '''
        return typing.cast(typing.Optional[typing.Union["CfnTaskDefinition.ProxyConfigurationProperty", _IResolvable_a771d0ef]], jsii.get(self, "proxyConfiguration"))

    @proxy_configuration.setter
    def proxy_configuration(
        self,
        value: typing.Optional[typing.Union["CfnTaskDefinition.ProxyConfigurationProperty", _IResolvable_a771d0ef]],
    ) -> None:
        jsii.set(self, "proxyConfiguration", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="requiresCompatibilities")
    def requires_compatibilities(self) -> typing.Optional[typing.List[builtins.str]]:
        '''``AWS::ECS::TaskDefinition.RequiresCompatibilities``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-requirescompatibilities
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "requiresCompatibilities"))

    @requires_compatibilities.setter
    def requires_compatibilities(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        jsii.set(self, "requiresCompatibilities", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskRoleArn")
    def task_role_arn(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::TaskDefinition.TaskRoleArn``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-taskrolearn
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "taskRoleArn"))

    @task_role_arn.setter
    def task_role_arn(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "taskRoleArn", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="volumes")
    def volumes(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.VolumeProperty", _IResolvable_a771d0ef]]]]:
        '''``AWS::ECS::TaskDefinition.Volumes``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-volumes
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.VolumeProperty", _IResolvable_a771d0ef]]]], jsii.get(self, "volumes"))

    @volumes.setter
    def volumes(
        self,
        value: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.VolumeProperty", _IResolvable_a771d0ef]]]],
    ) -> None:
        jsii.set(self, "volumes", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.AuthorizationConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"access_point_id": "accessPointId", "iam": "iam"},
    )
    class AuthorizationConfigProperty:
        def __init__(
            self,
            *,
            access_point_id: typing.Optional[builtins.str] = None,
            iam: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param access_point_id: ``CfnTaskDefinition.AuthorizationConfigProperty.AccessPointId``.
            :param iam: ``CfnTaskDefinition.AuthorizationConfigProperty.IAM``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-authorizationconfig.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if access_point_id is not None:
                self._values["access_point_id"] = access_point_id
            if iam is not None:
                self._values["iam"] = iam

        @builtins.property
        def access_point_id(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskDefinition.AuthorizationConfigProperty.AccessPointId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-authorizationconfig.html#cfn-ecs-taskdefinition-authorizationconfig-accesspointid
            '''
            result = self._values.get("access_point_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def iam(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskDefinition.AuthorizationConfigProperty.IAM``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-authorizationconfig.html#cfn-ecs-taskdefinition-authorizationconfig-iam
            '''
            result = self._values.get("iam")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AuthorizationConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.ContainerDefinitionProperty",
        jsii_struct_bases=[],
        name_mapping={
            "command": "command",
            "cpu": "cpu",
            "depends_on": "dependsOn",
            "disable_networking": "disableNetworking",
            "dns_search_domains": "dnsSearchDomains",
            "dns_servers": "dnsServers",
            "docker_labels": "dockerLabels",
            "docker_security_options": "dockerSecurityOptions",
            "entry_point": "entryPoint",
            "environment": "environment",
            "environment_files": "environmentFiles",
            "essential": "essential",
            "extra_hosts": "extraHosts",
            "firelens_configuration": "firelensConfiguration",
            "health_check": "healthCheck",
            "hostname": "hostname",
            "image": "image",
            "interactive": "interactive",
            "links": "links",
            "linux_parameters": "linuxParameters",
            "log_configuration": "logConfiguration",
            "memory": "memory",
            "memory_reservation": "memoryReservation",
            "mount_points": "mountPoints",
            "name": "name",
            "port_mappings": "portMappings",
            "privileged": "privileged",
            "pseudo_terminal": "pseudoTerminal",
            "readonly_root_filesystem": "readonlyRootFilesystem",
            "repository_credentials": "repositoryCredentials",
            "resource_requirements": "resourceRequirements",
            "secrets": "secrets",
            "start_timeout": "startTimeout",
            "stop_timeout": "stopTimeout",
            "system_controls": "systemControls",
            "ulimits": "ulimits",
            "user": "user",
            "volumes_from": "volumesFrom",
            "working_directory": "workingDirectory",
        },
    )
    class ContainerDefinitionProperty:
        def __init__(
            self,
            *,
            command: typing.Optional[typing.Sequence[builtins.str]] = None,
            cpu: typing.Optional[jsii.Number] = None,
            depends_on: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskDefinition.ContainerDependencyProperty", _IResolvable_a771d0ef]]]] = None,
            disable_networking: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            dns_search_domains: typing.Optional[typing.Sequence[builtins.str]] = None,
            dns_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
            docker_labels: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]] = None,
            docker_security_options: typing.Optional[typing.Sequence[builtins.str]] = None,
            entry_point: typing.Optional[typing.Sequence[builtins.str]] = None,
            environment: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskDefinition.KeyValuePairProperty", _IResolvable_a771d0ef]]]] = None,
            environment_files: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskDefinition.EnvironmentFileProperty", _IResolvable_a771d0ef]]]] = None,
            essential: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            extra_hosts: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskDefinition.HostEntryProperty", _IResolvable_a771d0ef]]]] = None,
            firelens_configuration: typing.Optional[typing.Union["CfnTaskDefinition.FirelensConfigurationProperty", _IResolvable_a771d0ef]] = None,
            health_check: typing.Optional[typing.Union["CfnTaskDefinition.HealthCheckProperty", _IResolvable_a771d0ef]] = None,
            hostname: typing.Optional[builtins.str] = None,
            image: typing.Optional[builtins.str] = None,
            interactive: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            links: typing.Optional[typing.Sequence[builtins.str]] = None,
            linux_parameters: typing.Optional[typing.Union["CfnTaskDefinition.LinuxParametersProperty", _IResolvable_a771d0ef]] = None,
            log_configuration: typing.Optional[typing.Union["CfnTaskDefinition.LogConfigurationProperty", _IResolvable_a771d0ef]] = None,
            memory: typing.Optional[jsii.Number] = None,
            memory_reservation: typing.Optional[jsii.Number] = None,
            mount_points: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskDefinition.MountPointProperty", _IResolvable_a771d0ef]]]] = None,
            name: typing.Optional[builtins.str] = None,
            port_mappings: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskDefinition.PortMappingProperty", _IResolvable_a771d0ef]]]] = None,
            privileged: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            pseudo_terminal: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            readonly_root_filesystem: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            repository_credentials: typing.Optional[typing.Union["CfnTaskDefinition.RepositoryCredentialsProperty", _IResolvable_a771d0ef]] = None,
            resource_requirements: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskDefinition.ResourceRequirementProperty", _IResolvable_a771d0ef]]]] = None,
            secrets: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskDefinition.SecretProperty", _IResolvable_a771d0ef]]]] = None,
            start_timeout: typing.Optional[jsii.Number] = None,
            stop_timeout: typing.Optional[jsii.Number] = None,
            system_controls: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskDefinition.SystemControlProperty", _IResolvable_a771d0ef]]]] = None,
            ulimits: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskDefinition.UlimitProperty", _IResolvable_a771d0ef]]]] = None,
            user: typing.Optional[builtins.str] = None,
            volumes_from: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskDefinition.VolumeFromProperty", _IResolvable_a771d0ef]]]] = None,
            working_directory: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param command: ``CfnTaskDefinition.ContainerDefinitionProperty.Command``.
            :param cpu: ``CfnTaskDefinition.ContainerDefinitionProperty.Cpu``.
            :param depends_on: ``CfnTaskDefinition.ContainerDefinitionProperty.DependsOn``.
            :param disable_networking: ``CfnTaskDefinition.ContainerDefinitionProperty.DisableNetworking``.
            :param dns_search_domains: ``CfnTaskDefinition.ContainerDefinitionProperty.DnsSearchDomains``.
            :param dns_servers: ``CfnTaskDefinition.ContainerDefinitionProperty.DnsServers``.
            :param docker_labels: ``CfnTaskDefinition.ContainerDefinitionProperty.DockerLabels``.
            :param docker_security_options: ``CfnTaskDefinition.ContainerDefinitionProperty.DockerSecurityOptions``.
            :param entry_point: ``CfnTaskDefinition.ContainerDefinitionProperty.EntryPoint``.
            :param environment: ``CfnTaskDefinition.ContainerDefinitionProperty.Environment``.
            :param environment_files: ``CfnTaskDefinition.ContainerDefinitionProperty.EnvironmentFiles``.
            :param essential: ``CfnTaskDefinition.ContainerDefinitionProperty.Essential``.
            :param extra_hosts: ``CfnTaskDefinition.ContainerDefinitionProperty.ExtraHosts``.
            :param firelens_configuration: ``CfnTaskDefinition.ContainerDefinitionProperty.FirelensConfiguration``.
            :param health_check: ``CfnTaskDefinition.ContainerDefinitionProperty.HealthCheck``.
            :param hostname: ``CfnTaskDefinition.ContainerDefinitionProperty.Hostname``.
            :param image: ``CfnTaskDefinition.ContainerDefinitionProperty.Image``.
            :param interactive: ``CfnTaskDefinition.ContainerDefinitionProperty.Interactive``.
            :param links: ``CfnTaskDefinition.ContainerDefinitionProperty.Links``.
            :param linux_parameters: ``CfnTaskDefinition.ContainerDefinitionProperty.LinuxParameters``.
            :param log_configuration: ``CfnTaskDefinition.ContainerDefinitionProperty.LogConfiguration``.
            :param memory: ``CfnTaskDefinition.ContainerDefinitionProperty.Memory``.
            :param memory_reservation: ``CfnTaskDefinition.ContainerDefinitionProperty.MemoryReservation``.
            :param mount_points: ``CfnTaskDefinition.ContainerDefinitionProperty.MountPoints``.
            :param name: ``CfnTaskDefinition.ContainerDefinitionProperty.Name``.
            :param port_mappings: ``CfnTaskDefinition.ContainerDefinitionProperty.PortMappings``.
            :param privileged: ``CfnTaskDefinition.ContainerDefinitionProperty.Privileged``.
            :param pseudo_terminal: ``CfnTaskDefinition.ContainerDefinitionProperty.PseudoTerminal``.
            :param readonly_root_filesystem: ``CfnTaskDefinition.ContainerDefinitionProperty.ReadonlyRootFilesystem``.
            :param repository_credentials: ``CfnTaskDefinition.ContainerDefinitionProperty.RepositoryCredentials``.
            :param resource_requirements: ``CfnTaskDefinition.ContainerDefinitionProperty.ResourceRequirements``.
            :param secrets: ``CfnTaskDefinition.ContainerDefinitionProperty.Secrets``.
            :param start_timeout: ``CfnTaskDefinition.ContainerDefinitionProperty.StartTimeout``.
            :param stop_timeout: ``CfnTaskDefinition.ContainerDefinitionProperty.StopTimeout``.
            :param system_controls: ``CfnTaskDefinition.ContainerDefinitionProperty.SystemControls``.
            :param ulimits: ``CfnTaskDefinition.ContainerDefinitionProperty.Ulimits``.
            :param user: ``CfnTaskDefinition.ContainerDefinitionProperty.User``.
            :param volumes_from: ``CfnTaskDefinition.ContainerDefinitionProperty.VolumesFrom``.
            :param working_directory: ``CfnTaskDefinition.ContainerDefinitionProperty.WorkingDirectory``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if command is not None:
                self._values["command"] = command
            if cpu is not None:
                self._values["cpu"] = cpu
            if depends_on is not None:
                self._values["depends_on"] = depends_on
            if disable_networking is not None:
                self._values["disable_networking"] = disable_networking
            if dns_search_domains is not None:
                self._values["dns_search_domains"] = dns_search_domains
            if dns_servers is not None:
                self._values["dns_servers"] = dns_servers
            if docker_labels is not None:
                self._values["docker_labels"] = docker_labels
            if docker_security_options is not None:
                self._values["docker_security_options"] = docker_security_options
            if entry_point is not None:
                self._values["entry_point"] = entry_point
            if environment is not None:
                self._values["environment"] = environment
            if environment_files is not None:
                self._values["environment_files"] = environment_files
            if essential is not None:
                self._values["essential"] = essential
            if extra_hosts is not None:
                self._values["extra_hosts"] = extra_hosts
            if firelens_configuration is not None:
                self._values["firelens_configuration"] = firelens_configuration
            if health_check is not None:
                self._values["health_check"] = health_check
            if hostname is not None:
                self._values["hostname"] = hostname
            if image is not None:
                self._values["image"] = image
            if interactive is not None:
                self._values["interactive"] = interactive
            if links is not None:
                self._values["links"] = links
            if linux_parameters is not None:
                self._values["linux_parameters"] = linux_parameters
            if log_configuration is not None:
                self._values["log_configuration"] = log_configuration
            if memory is not None:
                self._values["memory"] = memory
            if memory_reservation is not None:
                self._values["memory_reservation"] = memory_reservation
            if mount_points is not None:
                self._values["mount_points"] = mount_points
            if name is not None:
                self._values["name"] = name
            if port_mappings is not None:
                self._values["port_mappings"] = port_mappings
            if privileged is not None:
                self._values["privileged"] = privileged
            if pseudo_terminal is not None:
                self._values["pseudo_terminal"] = pseudo_terminal
            if readonly_root_filesystem is not None:
                self._values["readonly_root_filesystem"] = readonly_root_filesystem
            if repository_credentials is not None:
                self._values["repository_credentials"] = repository_credentials
            if resource_requirements is not None:
                self._values["resource_requirements"] = resource_requirements
            if secrets is not None:
                self._values["secrets"] = secrets
            if start_timeout is not None:
                self._values["start_timeout"] = start_timeout
            if stop_timeout is not None:
                self._values["stop_timeout"] = stop_timeout
            if system_controls is not None:
                self._values["system_controls"] = system_controls
            if ulimits is not None:
                self._values["ulimits"] = ulimits
            if user is not None:
                self._values["user"] = user
            if volumes_from is not None:
                self._values["volumes_from"] = volumes_from
            if working_directory is not None:
                self._values["working_directory"] = working_directory

        @builtins.property
        def command(self) -> typing.Optional[typing.List[builtins.str]]:
            '''``CfnTaskDefinition.ContainerDefinitionProperty.Command``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-command
            '''
            result = self._values.get("command")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def cpu(self) -> typing.Optional[jsii.Number]:
            '''``CfnTaskDefinition.ContainerDefinitionProperty.Cpu``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-cpu
            '''
            result = self._values.get("cpu")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def depends_on(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.ContainerDependencyProperty", _IResolvable_a771d0ef]]]]:
            '''``CfnTaskDefinition.ContainerDefinitionProperty.DependsOn``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-dependson
            '''
            result = self._values.get("depends_on")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.ContainerDependencyProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def disable_networking(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''``CfnTaskDefinition.ContainerDefinitionProperty.DisableNetworking``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-disablenetworking
            '''
            result = self._values.get("disable_networking")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def dns_search_domains(self) -> typing.Optional[typing.List[builtins.str]]:
            '''``CfnTaskDefinition.ContainerDefinitionProperty.DnsSearchDomains``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-dnssearchdomains
            '''
            result = self._values.get("dns_search_domains")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def dns_servers(self) -> typing.Optional[typing.List[builtins.str]]:
            '''``CfnTaskDefinition.ContainerDefinitionProperty.DnsServers``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-dnsservers
            '''
            result = self._values.get("dns_servers")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def docker_labels(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]]:
            '''``CfnTaskDefinition.ContainerDefinitionProperty.DockerLabels``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-dockerlabels
            '''
            result = self._values.get("docker_labels")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]], result)

        @builtins.property
        def docker_security_options(self) -> typing.Optional[typing.List[builtins.str]]:
            '''``CfnTaskDefinition.ContainerDefinitionProperty.DockerSecurityOptions``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-dockersecurityoptions
            '''
            result = self._values.get("docker_security_options")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def entry_point(self) -> typing.Optional[typing.List[builtins.str]]:
            '''``CfnTaskDefinition.ContainerDefinitionProperty.EntryPoint``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-entrypoint
            '''
            result = self._values.get("entry_point")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def environment(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.KeyValuePairProperty", _IResolvable_a771d0ef]]]]:
            '''``CfnTaskDefinition.ContainerDefinitionProperty.Environment``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-environment
            '''
            result = self._values.get("environment")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.KeyValuePairProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def environment_files(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.EnvironmentFileProperty", _IResolvable_a771d0ef]]]]:
            '''``CfnTaskDefinition.ContainerDefinitionProperty.EnvironmentFiles``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-environmentfiles
            '''
            result = self._values.get("environment_files")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.EnvironmentFileProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def essential(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''``CfnTaskDefinition.ContainerDefinitionProperty.Essential``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-essential
            '''
            result = self._values.get("essential")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def extra_hosts(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.HostEntryProperty", _IResolvable_a771d0ef]]]]:
            '''``CfnTaskDefinition.ContainerDefinitionProperty.ExtraHosts``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-extrahosts
            '''
            result = self._values.get("extra_hosts")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.HostEntryProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def firelens_configuration(
            self,
        ) -> typing.Optional[typing.Union["CfnTaskDefinition.FirelensConfigurationProperty", _IResolvable_a771d0ef]]:
            '''``CfnTaskDefinition.ContainerDefinitionProperty.FirelensConfiguration``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-firelensconfiguration
            '''
            result = self._values.get("firelens_configuration")
            return typing.cast(typing.Optional[typing.Union["CfnTaskDefinition.FirelensConfigurationProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def health_check(
            self,
        ) -> typing.Optional[typing.Union["CfnTaskDefinition.HealthCheckProperty", _IResolvable_a771d0ef]]:
            '''``CfnTaskDefinition.ContainerDefinitionProperty.HealthCheck``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-healthcheck
            '''
            result = self._values.get("health_check")
            return typing.cast(typing.Optional[typing.Union["CfnTaskDefinition.HealthCheckProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def hostname(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskDefinition.ContainerDefinitionProperty.Hostname``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-hostname
            '''
            result = self._values.get("hostname")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def image(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskDefinition.ContainerDefinitionProperty.Image``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-image
            '''
            result = self._values.get("image")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def interactive(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''``CfnTaskDefinition.ContainerDefinitionProperty.Interactive``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-interactive
            '''
            result = self._values.get("interactive")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def links(self) -> typing.Optional[typing.List[builtins.str]]:
            '''``CfnTaskDefinition.ContainerDefinitionProperty.Links``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-links
            '''
            result = self._values.get("links")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def linux_parameters(
            self,
        ) -> typing.Optional[typing.Union["CfnTaskDefinition.LinuxParametersProperty", _IResolvable_a771d0ef]]:
            '''``CfnTaskDefinition.ContainerDefinitionProperty.LinuxParameters``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-linuxparameters
            '''
            result = self._values.get("linux_parameters")
            return typing.cast(typing.Optional[typing.Union["CfnTaskDefinition.LinuxParametersProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def log_configuration(
            self,
        ) -> typing.Optional[typing.Union["CfnTaskDefinition.LogConfigurationProperty", _IResolvable_a771d0ef]]:
            '''``CfnTaskDefinition.ContainerDefinitionProperty.LogConfiguration``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-logconfiguration
            '''
            result = self._values.get("log_configuration")
            return typing.cast(typing.Optional[typing.Union["CfnTaskDefinition.LogConfigurationProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def memory(self) -> typing.Optional[jsii.Number]:
            '''``CfnTaskDefinition.ContainerDefinitionProperty.Memory``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-memory
            '''
            result = self._values.get("memory")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def memory_reservation(self) -> typing.Optional[jsii.Number]:
            '''``CfnTaskDefinition.ContainerDefinitionProperty.MemoryReservation``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-memoryreservation
            '''
            result = self._values.get("memory_reservation")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def mount_points(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.MountPointProperty", _IResolvable_a771d0ef]]]]:
            '''``CfnTaskDefinition.ContainerDefinitionProperty.MountPoints``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-mountpoints
            '''
            result = self._values.get("mount_points")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.MountPointProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def name(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskDefinition.ContainerDefinitionProperty.Name``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-name
            '''
            result = self._values.get("name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def port_mappings(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.PortMappingProperty", _IResolvable_a771d0ef]]]]:
            '''``CfnTaskDefinition.ContainerDefinitionProperty.PortMappings``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-portmappings
            '''
            result = self._values.get("port_mappings")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.PortMappingProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def privileged(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''``CfnTaskDefinition.ContainerDefinitionProperty.Privileged``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-privileged
            '''
            result = self._values.get("privileged")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def pseudo_terminal(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''``CfnTaskDefinition.ContainerDefinitionProperty.PseudoTerminal``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-pseudoterminal
            '''
            result = self._values.get("pseudo_terminal")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def readonly_root_filesystem(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''``CfnTaskDefinition.ContainerDefinitionProperty.ReadonlyRootFilesystem``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-readonlyrootfilesystem
            '''
            result = self._values.get("readonly_root_filesystem")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def repository_credentials(
            self,
        ) -> typing.Optional[typing.Union["CfnTaskDefinition.RepositoryCredentialsProperty", _IResolvable_a771d0ef]]:
            '''``CfnTaskDefinition.ContainerDefinitionProperty.RepositoryCredentials``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-repositorycredentials
            '''
            result = self._values.get("repository_credentials")
            return typing.cast(typing.Optional[typing.Union["CfnTaskDefinition.RepositoryCredentialsProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def resource_requirements(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.ResourceRequirementProperty", _IResolvable_a771d0ef]]]]:
            '''``CfnTaskDefinition.ContainerDefinitionProperty.ResourceRequirements``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-resourcerequirements
            '''
            result = self._values.get("resource_requirements")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.ResourceRequirementProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def secrets(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.SecretProperty", _IResolvable_a771d0ef]]]]:
            '''``CfnTaskDefinition.ContainerDefinitionProperty.Secrets``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-secrets
            '''
            result = self._values.get("secrets")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.SecretProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def start_timeout(self) -> typing.Optional[jsii.Number]:
            '''``CfnTaskDefinition.ContainerDefinitionProperty.StartTimeout``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-starttimeout
            '''
            result = self._values.get("start_timeout")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def stop_timeout(self) -> typing.Optional[jsii.Number]:
            '''``CfnTaskDefinition.ContainerDefinitionProperty.StopTimeout``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-stoptimeout
            '''
            result = self._values.get("stop_timeout")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def system_controls(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.SystemControlProperty", _IResolvable_a771d0ef]]]]:
            '''``CfnTaskDefinition.ContainerDefinitionProperty.SystemControls``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-systemcontrols
            '''
            result = self._values.get("system_controls")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.SystemControlProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def ulimits(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.UlimitProperty", _IResolvable_a771d0ef]]]]:
            '''``CfnTaskDefinition.ContainerDefinitionProperty.Ulimits``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-ulimits
            '''
            result = self._values.get("ulimits")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.UlimitProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def user(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskDefinition.ContainerDefinitionProperty.User``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-user
            '''
            result = self._values.get("user")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def volumes_from(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.VolumeFromProperty", _IResolvable_a771d0ef]]]]:
            '''``CfnTaskDefinition.ContainerDefinitionProperty.VolumesFrom``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-volumesfrom
            '''
            result = self._values.get("volumes_from")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.VolumeFromProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def working_directory(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskDefinition.ContainerDefinitionProperty.WorkingDirectory``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-workingdirectory
            '''
            result = self._values.get("working_directory")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ContainerDefinitionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.ContainerDependencyProperty",
        jsii_struct_bases=[],
        name_mapping={"condition": "condition", "container_name": "containerName"},
    )
    class ContainerDependencyProperty:
        def __init__(
            self,
            *,
            condition: typing.Optional[builtins.str] = None,
            container_name: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param condition: ``CfnTaskDefinition.ContainerDependencyProperty.Condition``.
            :param container_name: ``CfnTaskDefinition.ContainerDependencyProperty.ContainerName``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdependency.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if condition is not None:
                self._values["condition"] = condition
            if container_name is not None:
                self._values["container_name"] = container_name

        @builtins.property
        def condition(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskDefinition.ContainerDependencyProperty.Condition``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdependency.html#cfn-ecs-taskdefinition-containerdependency-condition
            '''
            result = self._values.get("condition")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def container_name(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskDefinition.ContainerDependencyProperty.ContainerName``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdependency.html#cfn-ecs-taskdefinition-containerdependency-containername
            '''
            result = self._values.get("container_name")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ContainerDependencyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.DeviceProperty",
        jsii_struct_bases=[],
        name_mapping={
            "container_path": "containerPath",
            "host_path": "hostPath",
            "permissions": "permissions",
        },
    )
    class DeviceProperty:
        def __init__(
            self,
            *,
            container_path: typing.Optional[builtins.str] = None,
            host_path: typing.Optional[builtins.str] = None,
            permissions: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''
            :param container_path: ``CfnTaskDefinition.DeviceProperty.ContainerPath``.
            :param host_path: ``CfnTaskDefinition.DeviceProperty.HostPath``.
            :param permissions: ``CfnTaskDefinition.DeviceProperty.Permissions``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-device.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if container_path is not None:
                self._values["container_path"] = container_path
            if host_path is not None:
                self._values["host_path"] = host_path
            if permissions is not None:
                self._values["permissions"] = permissions

        @builtins.property
        def container_path(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskDefinition.DeviceProperty.ContainerPath``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-device.html#cfn-ecs-taskdefinition-device-containerpath
            '''
            result = self._values.get("container_path")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def host_path(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskDefinition.DeviceProperty.HostPath``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-device.html#cfn-ecs-taskdefinition-device-hostpath
            '''
            result = self._values.get("host_path")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def permissions(self) -> typing.Optional[typing.List[builtins.str]]:
            '''``CfnTaskDefinition.DeviceProperty.Permissions``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-device.html#cfn-ecs-taskdefinition-device-permissions
            '''
            result = self._values.get("permissions")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DeviceProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.DockerVolumeConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "autoprovision": "autoprovision",
            "driver": "driver",
            "driver_opts": "driverOpts",
            "labels": "labels",
            "scope": "scope",
        },
    )
    class DockerVolumeConfigurationProperty:
        def __init__(
            self,
            *,
            autoprovision: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            driver: typing.Optional[builtins.str] = None,
            driver_opts: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]] = None,
            labels: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]] = None,
            scope: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param autoprovision: ``CfnTaskDefinition.DockerVolumeConfigurationProperty.Autoprovision``.
            :param driver: ``CfnTaskDefinition.DockerVolumeConfigurationProperty.Driver``.
            :param driver_opts: ``CfnTaskDefinition.DockerVolumeConfigurationProperty.DriverOpts``.
            :param labels: ``CfnTaskDefinition.DockerVolumeConfigurationProperty.Labels``.
            :param scope: ``CfnTaskDefinition.DockerVolumeConfigurationProperty.Scope``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-dockervolumeconfiguration.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if autoprovision is not None:
                self._values["autoprovision"] = autoprovision
            if driver is not None:
                self._values["driver"] = driver
            if driver_opts is not None:
                self._values["driver_opts"] = driver_opts
            if labels is not None:
                self._values["labels"] = labels
            if scope is not None:
                self._values["scope"] = scope

        @builtins.property
        def autoprovision(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''``CfnTaskDefinition.DockerVolumeConfigurationProperty.Autoprovision``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-dockervolumeconfiguration.html#cfn-ecs-taskdefinition-dockervolumeconfiguration-autoprovision
            '''
            result = self._values.get("autoprovision")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def driver(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskDefinition.DockerVolumeConfigurationProperty.Driver``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-dockervolumeconfiguration.html#cfn-ecs-taskdefinition-dockervolumeconfiguration-driver
            '''
            result = self._values.get("driver")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def driver_opts(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]]:
            '''``CfnTaskDefinition.DockerVolumeConfigurationProperty.DriverOpts``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-dockervolumeconfiguration.html#cfn-ecs-taskdefinition-dockervolumeconfiguration-driveropts
            '''
            result = self._values.get("driver_opts")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]], result)

        @builtins.property
        def labels(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]]:
            '''``CfnTaskDefinition.DockerVolumeConfigurationProperty.Labels``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-dockervolumeconfiguration.html#cfn-ecs-taskdefinition-dockervolumeconfiguration-labels
            '''
            result = self._values.get("labels")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]], result)

        @builtins.property
        def scope(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskDefinition.DockerVolumeConfigurationProperty.Scope``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-dockervolumeconfiguration.html#cfn-ecs-taskdefinition-dockervolumeconfiguration-scope
            '''
            result = self._values.get("scope")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DockerVolumeConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.EfsVolumeConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "file_system_id": "fileSystemId",
            "authorization_config": "authorizationConfig",
            "root_directory": "rootDirectory",
            "transit_encryption": "transitEncryption",
            "transit_encryption_port": "transitEncryptionPort",
        },
    )
    class EfsVolumeConfigurationProperty:
        def __init__(
            self,
            *,
            file_system_id: builtins.str,
            authorization_config: typing.Optional[typing.Union["CfnTaskDefinition.AuthorizationConfigProperty", _IResolvable_a771d0ef]] = None,
            root_directory: typing.Optional[builtins.str] = None,
            transit_encryption: typing.Optional[builtins.str] = None,
            transit_encryption_port: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''
            :param file_system_id: ``CfnTaskDefinition.EfsVolumeConfigurationProperty.FileSystemId``.
            :param authorization_config: ``CfnTaskDefinition.EfsVolumeConfigurationProperty.AuthorizationConfig``.
            :param root_directory: ``CfnTaskDefinition.EfsVolumeConfigurationProperty.RootDirectory``.
            :param transit_encryption: ``CfnTaskDefinition.EfsVolumeConfigurationProperty.TransitEncryption``.
            :param transit_encryption_port: ``CfnTaskDefinition.EfsVolumeConfigurationProperty.TransitEncryptionPort``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-efsvolumeconfiguration.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "file_system_id": file_system_id,
            }
            if authorization_config is not None:
                self._values["authorization_config"] = authorization_config
            if root_directory is not None:
                self._values["root_directory"] = root_directory
            if transit_encryption is not None:
                self._values["transit_encryption"] = transit_encryption
            if transit_encryption_port is not None:
                self._values["transit_encryption_port"] = transit_encryption_port

        @builtins.property
        def file_system_id(self) -> builtins.str:
            '''``CfnTaskDefinition.EfsVolumeConfigurationProperty.FileSystemId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-efsvolumeconfiguration.html#cfn-ecs-taskdefinition-efsvolumeconfiguration-filesystemid
            '''
            result = self._values.get("file_system_id")
            assert result is not None, "Required property 'file_system_id' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def authorization_config(
            self,
        ) -> typing.Optional[typing.Union["CfnTaskDefinition.AuthorizationConfigProperty", _IResolvable_a771d0ef]]:
            '''``CfnTaskDefinition.EfsVolumeConfigurationProperty.AuthorizationConfig``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-efsvolumeconfiguration.html#cfn-ecs-taskdefinition-efsvolumeconfiguration-authorizationconfig
            '''
            result = self._values.get("authorization_config")
            return typing.cast(typing.Optional[typing.Union["CfnTaskDefinition.AuthorizationConfigProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def root_directory(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskDefinition.EfsVolumeConfigurationProperty.RootDirectory``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-efsvolumeconfiguration.html#cfn-ecs-taskdefinition-efsvolumeconfiguration-rootdirectory
            '''
            result = self._values.get("root_directory")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def transit_encryption(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskDefinition.EfsVolumeConfigurationProperty.TransitEncryption``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-efsvolumeconfiguration.html#cfn-ecs-taskdefinition-efsvolumeconfiguration-transitencryption
            '''
            result = self._values.get("transit_encryption")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def transit_encryption_port(self) -> typing.Optional[jsii.Number]:
            '''``CfnTaskDefinition.EfsVolumeConfigurationProperty.TransitEncryptionPort``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-efsvolumeconfiguration.html#cfn-ecs-taskdefinition-efsvolumeconfiguration-transitencryptionport
            '''
            result = self._values.get("transit_encryption_port")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EfsVolumeConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.EnvironmentFileProperty",
        jsii_struct_bases=[],
        name_mapping={"type": "type", "value": "value"},
    )
    class EnvironmentFileProperty:
        def __init__(
            self,
            *,
            type: typing.Optional[builtins.str] = None,
            value: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param type: ``CfnTaskDefinition.EnvironmentFileProperty.Type``.
            :param value: ``CfnTaskDefinition.EnvironmentFileProperty.Value``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-environmentfile.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if type is not None:
                self._values["type"] = type
            if value is not None:
                self._values["value"] = value

        @builtins.property
        def type(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskDefinition.EnvironmentFileProperty.Type``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-environmentfile.html#cfn-ecs-taskdefinition-environmentfile-type
            '''
            result = self._values.get("type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def value(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskDefinition.EnvironmentFileProperty.Value``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-environmentfile.html#cfn-ecs-taskdefinition-environmentfile-value
            '''
            result = self._values.get("value")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EnvironmentFileProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.EphemeralStorageProperty",
        jsii_struct_bases=[],
        name_mapping={"size_in_gib": "sizeInGiB"},
    )
    class EphemeralStorageProperty:
        def __init__(self, *, size_in_gib: typing.Optional[jsii.Number] = None) -> None:
            '''
            :param size_in_gib: ``CfnTaskDefinition.EphemeralStorageProperty.SizeInGiB``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-ephemeralstorage.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if size_in_gib is not None:
                self._values["size_in_gib"] = size_in_gib

        @builtins.property
        def size_in_gib(self) -> typing.Optional[jsii.Number]:
            '''``CfnTaskDefinition.EphemeralStorageProperty.SizeInGiB``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-ephemeralstorage.html#cfn-ecs-taskdefinition-ephemeralstorage-sizeingib
            '''
            result = self._values.get("size_in_gib")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EphemeralStorageProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.FirelensConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"options": "options", "type": "type"},
    )
    class FirelensConfigurationProperty:
        def __init__(
            self,
            *,
            options: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]] = None,
            type: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param options: ``CfnTaskDefinition.FirelensConfigurationProperty.Options``.
            :param type: ``CfnTaskDefinition.FirelensConfigurationProperty.Type``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-firelensconfiguration.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if options is not None:
                self._values["options"] = options
            if type is not None:
                self._values["type"] = type

        @builtins.property
        def options(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]]:
            '''``CfnTaskDefinition.FirelensConfigurationProperty.Options``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-firelensconfiguration.html#cfn-ecs-taskdefinition-firelensconfiguration-options
            '''
            result = self._values.get("options")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]], result)

        @builtins.property
        def type(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskDefinition.FirelensConfigurationProperty.Type``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-firelensconfiguration.html#cfn-ecs-taskdefinition-firelensconfiguration-type
            '''
            result = self._values.get("type")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FirelensConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.HealthCheckProperty",
        jsii_struct_bases=[],
        name_mapping={
            "command": "command",
            "interval": "interval",
            "retries": "retries",
            "start_period": "startPeriod",
            "timeout": "timeout",
        },
    )
    class HealthCheckProperty:
        def __init__(
            self,
            *,
            command: typing.Optional[typing.Sequence[builtins.str]] = None,
            interval: typing.Optional[jsii.Number] = None,
            retries: typing.Optional[jsii.Number] = None,
            start_period: typing.Optional[jsii.Number] = None,
            timeout: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''
            :param command: ``CfnTaskDefinition.HealthCheckProperty.Command``.
            :param interval: ``CfnTaskDefinition.HealthCheckProperty.Interval``.
            :param retries: ``CfnTaskDefinition.HealthCheckProperty.Retries``.
            :param start_period: ``CfnTaskDefinition.HealthCheckProperty.StartPeriod``.
            :param timeout: ``CfnTaskDefinition.HealthCheckProperty.Timeout``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-healthcheck.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if command is not None:
                self._values["command"] = command
            if interval is not None:
                self._values["interval"] = interval
            if retries is not None:
                self._values["retries"] = retries
            if start_period is not None:
                self._values["start_period"] = start_period
            if timeout is not None:
                self._values["timeout"] = timeout

        @builtins.property
        def command(self) -> typing.Optional[typing.List[builtins.str]]:
            '''``CfnTaskDefinition.HealthCheckProperty.Command``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-healthcheck.html#cfn-ecs-taskdefinition-healthcheck-command
            '''
            result = self._values.get("command")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def interval(self) -> typing.Optional[jsii.Number]:
            '''``CfnTaskDefinition.HealthCheckProperty.Interval``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-healthcheck.html#cfn-ecs-taskdefinition-healthcheck-interval
            '''
            result = self._values.get("interval")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def retries(self) -> typing.Optional[jsii.Number]:
            '''``CfnTaskDefinition.HealthCheckProperty.Retries``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-healthcheck.html#cfn-ecs-taskdefinition-healthcheck-retries
            '''
            result = self._values.get("retries")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def start_period(self) -> typing.Optional[jsii.Number]:
            '''``CfnTaskDefinition.HealthCheckProperty.StartPeriod``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-healthcheck.html#cfn-ecs-taskdefinition-healthcheck-startperiod
            '''
            result = self._values.get("start_period")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def timeout(self) -> typing.Optional[jsii.Number]:
            '''``CfnTaskDefinition.HealthCheckProperty.Timeout``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-healthcheck.html#cfn-ecs-taskdefinition-healthcheck-timeout
            '''
            result = self._values.get("timeout")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HealthCheckProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.HostEntryProperty",
        jsii_struct_bases=[],
        name_mapping={"hostname": "hostname", "ip_address": "ipAddress"},
    )
    class HostEntryProperty:
        def __init__(
            self,
            *,
            hostname: typing.Optional[builtins.str] = None,
            ip_address: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param hostname: ``CfnTaskDefinition.HostEntryProperty.Hostname``.
            :param ip_address: ``CfnTaskDefinition.HostEntryProperty.IpAddress``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-hostentry.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if hostname is not None:
                self._values["hostname"] = hostname
            if ip_address is not None:
                self._values["ip_address"] = ip_address

        @builtins.property
        def hostname(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskDefinition.HostEntryProperty.Hostname``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-hostentry.html#cfn-ecs-taskdefinition-containerdefinition-hostentry-hostname
            '''
            result = self._values.get("hostname")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def ip_address(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskDefinition.HostEntryProperty.IpAddress``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-hostentry.html#cfn-ecs-taskdefinition-containerdefinition-hostentry-ipaddress
            '''
            result = self._values.get("ip_address")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HostEntryProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.HostVolumePropertiesProperty",
        jsii_struct_bases=[],
        name_mapping={"source_path": "sourcePath"},
    )
    class HostVolumePropertiesProperty:
        def __init__(
            self,
            *,
            source_path: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param source_path: ``CfnTaskDefinition.HostVolumePropertiesProperty.SourcePath``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-volumes-host.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if source_path is not None:
                self._values["source_path"] = source_path

        @builtins.property
        def source_path(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskDefinition.HostVolumePropertiesProperty.SourcePath``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-volumes-host.html#cfn-ecs-taskdefinition-volumes-host-sourcepath
            '''
            result = self._values.get("source_path")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HostVolumePropertiesProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.InferenceAcceleratorProperty",
        jsii_struct_bases=[],
        name_mapping={"device_name": "deviceName", "device_type": "deviceType"},
    )
    class InferenceAcceleratorProperty:
        def __init__(
            self,
            *,
            device_name: typing.Optional[builtins.str] = None,
            device_type: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param device_name: ``CfnTaskDefinition.InferenceAcceleratorProperty.DeviceName``.
            :param device_type: ``CfnTaskDefinition.InferenceAcceleratorProperty.DeviceType``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-inferenceaccelerator.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if device_name is not None:
                self._values["device_name"] = device_name
            if device_type is not None:
                self._values["device_type"] = device_type

        @builtins.property
        def device_name(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskDefinition.InferenceAcceleratorProperty.DeviceName``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-inferenceaccelerator.html#cfn-ecs-taskdefinition-inferenceaccelerator-devicename
            '''
            result = self._values.get("device_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def device_type(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskDefinition.InferenceAcceleratorProperty.DeviceType``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-inferenceaccelerator.html#cfn-ecs-taskdefinition-inferenceaccelerator-devicetype
            '''
            result = self._values.get("device_type")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "InferenceAcceleratorProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.KernelCapabilitiesProperty",
        jsii_struct_bases=[],
        name_mapping={"add": "add", "drop": "drop"},
    )
    class KernelCapabilitiesProperty:
        def __init__(
            self,
            *,
            add: typing.Optional[typing.Sequence[builtins.str]] = None,
            drop: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''
            :param add: ``CfnTaskDefinition.KernelCapabilitiesProperty.Add``.
            :param drop: ``CfnTaskDefinition.KernelCapabilitiesProperty.Drop``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-kernelcapabilities.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if add is not None:
                self._values["add"] = add
            if drop is not None:
                self._values["drop"] = drop

        @builtins.property
        def add(self) -> typing.Optional[typing.List[builtins.str]]:
            '''``CfnTaskDefinition.KernelCapabilitiesProperty.Add``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-kernelcapabilities.html#cfn-ecs-taskdefinition-kernelcapabilities-add
            '''
            result = self._values.get("add")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def drop(self) -> typing.Optional[typing.List[builtins.str]]:
            '''``CfnTaskDefinition.KernelCapabilitiesProperty.Drop``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-kernelcapabilities.html#cfn-ecs-taskdefinition-kernelcapabilities-drop
            '''
            result = self._values.get("drop")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "KernelCapabilitiesProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.KeyValuePairProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "value": "value"},
    )
    class KeyValuePairProperty:
        def __init__(
            self,
            *,
            name: typing.Optional[builtins.str] = None,
            value: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param name: ``CfnTaskDefinition.KeyValuePairProperty.Name``.
            :param value: ``CfnTaskDefinition.KeyValuePairProperty.Value``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-environment.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if name is not None:
                self._values["name"] = name
            if value is not None:
                self._values["value"] = value

        @builtins.property
        def name(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskDefinition.KeyValuePairProperty.Name``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-environment.html#cfn-ecs-taskdefinition-containerdefinition-environment-name
            '''
            result = self._values.get("name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def value(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskDefinition.KeyValuePairProperty.Value``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-environment.html#cfn-ecs-taskdefinition-containerdefinition-environment-value
            '''
            result = self._values.get("value")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "KeyValuePairProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.LinuxParametersProperty",
        jsii_struct_bases=[],
        name_mapping={
            "capabilities": "capabilities",
            "devices": "devices",
            "init_process_enabled": "initProcessEnabled",
            "max_swap": "maxSwap",
            "shared_memory_size": "sharedMemorySize",
            "swappiness": "swappiness",
            "tmpfs": "tmpfs",
        },
    )
    class LinuxParametersProperty:
        def __init__(
            self,
            *,
            capabilities: typing.Optional[typing.Union["CfnTaskDefinition.KernelCapabilitiesProperty", _IResolvable_a771d0ef]] = None,
            devices: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskDefinition.DeviceProperty", _IResolvable_a771d0ef]]]] = None,
            init_process_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            max_swap: typing.Optional[jsii.Number] = None,
            shared_memory_size: typing.Optional[jsii.Number] = None,
            swappiness: typing.Optional[jsii.Number] = None,
            tmpfs: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskDefinition.TmpfsProperty", _IResolvable_a771d0ef]]]] = None,
        ) -> None:
            '''
            :param capabilities: ``CfnTaskDefinition.LinuxParametersProperty.Capabilities``.
            :param devices: ``CfnTaskDefinition.LinuxParametersProperty.Devices``.
            :param init_process_enabled: ``CfnTaskDefinition.LinuxParametersProperty.InitProcessEnabled``.
            :param max_swap: ``CfnTaskDefinition.LinuxParametersProperty.MaxSwap``.
            :param shared_memory_size: ``CfnTaskDefinition.LinuxParametersProperty.SharedMemorySize``.
            :param swappiness: ``CfnTaskDefinition.LinuxParametersProperty.Swappiness``.
            :param tmpfs: ``CfnTaskDefinition.LinuxParametersProperty.Tmpfs``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-linuxparameters.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if capabilities is not None:
                self._values["capabilities"] = capabilities
            if devices is not None:
                self._values["devices"] = devices
            if init_process_enabled is not None:
                self._values["init_process_enabled"] = init_process_enabled
            if max_swap is not None:
                self._values["max_swap"] = max_swap
            if shared_memory_size is not None:
                self._values["shared_memory_size"] = shared_memory_size
            if swappiness is not None:
                self._values["swappiness"] = swappiness
            if tmpfs is not None:
                self._values["tmpfs"] = tmpfs

        @builtins.property
        def capabilities(
            self,
        ) -> typing.Optional[typing.Union["CfnTaskDefinition.KernelCapabilitiesProperty", _IResolvable_a771d0ef]]:
            '''``CfnTaskDefinition.LinuxParametersProperty.Capabilities``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-linuxparameters.html#cfn-ecs-taskdefinition-linuxparameters-capabilities
            '''
            result = self._values.get("capabilities")
            return typing.cast(typing.Optional[typing.Union["CfnTaskDefinition.KernelCapabilitiesProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def devices(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.DeviceProperty", _IResolvable_a771d0ef]]]]:
            '''``CfnTaskDefinition.LinuxParametersProperty.Devices``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-linuxparameters.html#cfn-ecs-taskdefinition-linuxparameters-devices
            '''
            result = self._values.get("devices")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.DeviceProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def init_process_enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''``CfnTaskDefinition.LinuxParametersProperty.InitProcessEnabled``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-linuxparameters.html#cfn-ecs-taskdefinition-linuxparameters-initprocessenabled
            '''
            result = self._values.get("init_process_enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def max_swap(self) -> typing.Optional[jsii.Number]:
            '''``CfnTaskDefinition.LinuxParametersProperty.MaxSwap``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-linuxparameters.html#cfn-ecs-taskdefinition-linuxparameters-maxswap
            '''
            result = self._values.get("max_swap")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def shared_memory_size(self) -> typing.Optional[jsii.Number]:
            '''``CfnTaskDefinition.LinuxParametersProperty.SharedMemorySize``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-linuxparameters.html#cfn-ecs-taskdefinition-linuxparameters-sharedmemorysize
            '''
            result = self._values.get("shared_memory_size")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def swappiness(self) -> typing.Optional[jsii.Number]:
            '''``CfnTaskDefinition.LinuxParametersProperty.Swappiness``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-linuxparameters.html#cfn-ecs-taskdefinition-linuxparameters-swappiness
            '''
            result = self._values.get("swappiness")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def tmpfs(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.TmpfsProperty", _IResolvable_a771d0ef]]]]:
            '''``CfnTaskDefinition.LinuxParametersProperty.Tmpfs``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-linuxparameters.html#cfn-ecs-taskdefinition-linuxparameters-tmpfs
            '''
            result = self._values.get("tmpfs")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.TmpfsProperty", _IResolvable_a771d0ef]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LinuxParametersProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.LogConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "log_driver": "logDriver",
            "options": "options",
            "secret_options": "secretOptions",
        },
    )
    class LogConfigurationProperty:
        def __init__(
            self,
            *,
            log_driver: builtins.str,
            options: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]] = None,
            secret_options: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskDefinition.SecretProperty", _IResolvable_a771d0ef]]]] = None,
        ) -> None:
            '''
            :param log_driver: ``CfnTaskDefinition.LogConfigurationProperty.LogDriver``.
            :param options: ``CfnTaskDefinition.LogConfigurationProperty.Options``.
            :param secret_options: ``CfnTaskDefinition.LogConfigurationProperty.SecretOptions``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-logconfiguration.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "log_driver": log_driver,
            }
            if options is not None:
                self._values["options"] = options
            if secret_options is not None:
                self._values["secret_options"] = secret_options

        @builtins.property
        def log_driver(self) -> builtins.str:
            '''``CfnTaskDefinition.LogConfigurationProperty.LogDriver``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-logconfiguration.html#cfn-ecs-taskdefinition-containerdefinition-logconfiguration-logdriver
            '''
            result = self._values.get("log_driver")
            assert result is not None, "Required property 'log_driver' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def options(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]]:
            '''``CfnTaskDefinition.LogConfigurationProperty.Options``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-logconfiguration.html#cfn-ecs-taskdefinition-containerdefinition-logconfiguration-options
            '''
            result = self._values.get("options")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]], result)

        @builtins.property
        def secret_options(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.SecretProperty", _IResolvable_a771d0ef]]]]:
            '''``CfnTaskDefinition.LogConfigurationProperty.SecretOptions``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-logconfiguration.html#cfn-ecs-taskdefinition-logconfiguration-secretoptions
            '''
            result = self._values.get("secret_options")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.SecretProperty", _IResolvable_a771d0ef]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LogConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.MountPointProperty",
        jsii_struct_bases=[],
        name_mapping={
            "container_path": "containerPath",
            "read_only": "readOnly",
            "source_volume": "sourceVolume",
        },
    )
    class MountPointProperty:
        def __init__(
            self,
            *,
            container_path: typing.Optional[builtins.str] = None,
            read_only: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            source_volume: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param container_path: ``CfnTaskDefinition.MountPointProperty.ContainerPath``.
            :param read_only: ``CfnTaskDefinition.MountPointProperty.ReadOnly``.
            :param source_volume: ``CfnTaskDefinition.MountPointProperty.SourceVolume``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-mountpoints.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if container_path is not None:
                self._values["container_path"] = container_path
            if read_only is not None:
                self._values["read_only"] = read_only
            if source_volume is not None:
                self._values["source_volume"] = source_volume

        @builtins.property
        def container_path(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskDefinition.MountPointProperty.ContainerPath``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-mountpoints.html#cfn-ecs-taskdefinition-containerdefinition-mountpoints-containerpath
            '''
            result = self._values.get("container_path")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def read_only(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''``CfnTaskDefinition.MountPointProperty.ReadOnly``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-mountpoints.html#cfn-ecs-taskdefinition-containerdefinition-mountpoints-readonly
            '''
            result = self._values.get("read_only")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def source_volume(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskDefinition.MountPointProperty.SourceVolume``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-mountpoints.html#cfn-ecs-taskdefinition-containerdefinition-mountpoints-sourcevolume
            '''
            result = self._values.get("source_volume")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MountPointProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.PortMappingProperty",
        jsii_struct_bases=[],
        name_mapping={
            "container_port": "containerPort",
            "host_port": "hostPort",
            "protocol": "protocol",
        },
    )
    class PortMappingProperty:
        def __init__(
            self,
            *,
            container_port: typing.Optional[jsii.Number] = None,
            host_port: typing.Optional[jsii.Number] = None,
            protocol: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param container_port: ``CfnTaskDefinition.PortMappingProperty.ContainerPort``.
            :param host_port: ``CfnTaskDefinition.PortMappingProperty.HostPort``.
            :param protocol: ``CfnTaskDefinition.PortMappingProperty.Protocol``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-portmappings.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if container_port is not None:
                self._values["container_port"] = container_port
            if host_port is not None:
                self._values["host_port"] = host_port
            if protocol is not None:
                self._values["protocol"] = protocol

        @builtins.property
        def container_port(self) -> typing.Optional[jsii.Number]:
            '''``CfnTaskDefinition.PortMappingProperty.ContainerPort``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-portmappings.html#cfn-ecs-taskdefinition-containerdefinition-portmappings-containerport
            '''
            result = self._values.get("container_port")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def host_port(self) -> typing.Optional[jsii.Number]:
            '''``CfnTaskDefinition.PortMappingProperty.HostPort``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-portmappings.html#cfn-ecs-taskdefinition-containerdefinition-portmappings-readonly
            '''
            result = self._values.get("host_port")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def protocol(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskDefinition.PortMappingProperty.Protocol``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-portmappings.html#cfn-ecs-taskdefinition-containerdefinition-portmappings-sourcevolume
            '''
            result = self._values.get("protocol")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PortMappingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.ProxyConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "container_name": "containerName",
            "proxy_configuration_properties": "proxyConfigurationProperties",
            "type": "type",
        },
    )
    class ProxyConfigurationProperty:
        def __init__(
            self,
            *,
            container_name: builtins.str,
            proxy_configuration_properties: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskDefinition.KeyValuePairProperty", _IResolvable_a771d0ef]]]] = None,
            type: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param container_name: ``CfnTaskDefinition.ProxyConfigurationProperty.ContainerName``.
            :param proxy_configuration_properties: ``CfnTaskDefinition.ProxyConfigurationProperty.ProxyConfigurationProperties``.
            :param type: ``CfnTaskDefinition.ProxyConfigurationProperty.Type``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-proxyconfiguration.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "container_name": container_name,
            }
            if proxy_configuration_properties is not None:
                self._values["proxy_configuration_properties"] = proxy_configuration_properties
            if type is not None:
                self._values["type"] = type

        @builtins.property
        def container_name(self) -> builtins.str:
            '''``CfnTaskDefinition.ProxyConfigurationProperty.ContainerName``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-proxyconfiguration.html#cfn-ecs-taskdefinition-proxyconfiguration-containername
            '''
            result = self._values.get("container_name")
            assert result is not None, "Required property 'container_name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def proxy_configuration_properties(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.KeyValuePairProperty", _IResolvable_a771d0ef]]]]:
            '''``CfnTaskDefinition.ProxyConfigurationProperty.ProxyConfigurationProperties``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-proxyconfiguration.html#cfn-ecs-taskdefinition-proxyconfiguration-proxyconfigurationproperties
            '''
            result = self._values.get("proxy_configuration_properties")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.KeyValuePairProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def type(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskDefinition.ProxyConfigurationProperty.Type``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-proxyconfiguration.html#cfn-ecs-taskdefinition-proxyconfiguration-type
            '''
            result = self._values.get("type")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ProxyConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.RepositoryCredentialsProperty",
        jsii_struct_bases=[],
        name_mapping={"credentials_parameter": "credentialsParameter"},
    )
    class RepositoryCredentialsProperty:
        def __init__(
            self,
            *,
            credentials_parameter: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param credentials_parameter: ``CfnTaskDefinition.RepositoryCredentialsProperty.CredentialsParameter``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-repositorycredentials.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if credentials_parameter is not None:
                self._values["credentials_parameter"] = credentials_parameter

        @builtins.property
        def credentials_parameter(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskDefinition.RepositoryCredentialsProperty.CredentialsParameter``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-repositorycredentials.html#cfn-ecs-taskdefinition-repositorycredentials-credentialsparameter
            '''
            result = self._values.get("credentials_parameter")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "RepositoryCredentialsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.ResourceRequirementProperty",
        jsii_struct_bases=[],
        name_mapping={"type": "type", "value": "value"},
    )
    class ResourceRequirementProperty:
        def __init__(self, *, type: builtins.str, value: builtins.str) -> None:
            '''
            :param type: ``CfnTaskDefinition.ResourceRequirementProperty.Type``.
            :param value: ``CfnTaskDefinition.ResourceRequirementProperty.Value``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-resourcerequirement.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "type": type,
                "value": value,
            }

        @builtins.property
        def type(self) -> builtins.str:
            '''``CfnTaskDefinition.ResourceRequirementProperty.Type``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-resourcerequirement.html#cfn-ecs-taskdefinition-resourcerequirement-type
            '''
            result = self._values.get("type")
            assert result is not None, "Required property 'type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def value(self) -> builtins.str:
            '''``CfnTaskDefinition.ResourceRequirementProperty.Value``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-resourcerequirement.html#cfn-ecs-taskdefinition-resourcerequirement-value
            '''
            result = self._values.get("value")
            assert result is not None, "Required property 'value' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ResourceRequirementProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.SecretProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "value_from": "valueFrom"},
    )
    class SecretProperty:
        def __init__(self, *, name: builtins.str, value_from: builtins.str) -> None:
            '''
            :param name: ``CfnTaskDefinition.SecretProperty.Name``.
            :param value_from: ``CfnTaskDefinition.SecretProperty.ValueFrom``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-secret.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "name": name,
                "value_from": value_from,
            }

        @builtins.property
        def name(self) -> builtins.str:
            '''``CfnTaskDefinition.SecretProperty.Name``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-secret.html#cfn-ecs-taskdefinition-secret-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def value_from(self) -> builtins.str:
            '''``CfnTaskDefinition.SecretProperty.ValueFrom``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-secret.html#cfn-ecs-taskdefinition-secret-valuefrom
            '''
            result = self._values.get("value_from")
            assert result is not None, "Required property 'value_from' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SecretProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.SystemControlProperty",
        jsii_struct_bases=[],
        name_mapping={"namespace": "namespace", "value": "value"},
    )
    class SystemControlProperty:
        def __init__(
            self,
            *,
            namespace: typing.Optional[builtins.str] = None,
            value: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param namespace: ``CfnTaskDefinition.SystemControlProperty.Namespace``.
            :param value: ``CfnTaskDefinition.SystemControlProperty.Value``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-systemcontrol.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if namespace is not None:
                self._values["namespace"] = namespace
            if value is not None:
                self._values["value"] = value

        @builtins.property
        def namespace(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskDefinition.SystemControlProperty.Namespace``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-systemcontrol.html#cfn-ecs-taskdefinition-systemcontrol-namespace
            '''
            result = self._values.get("namespace")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def value(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskDefinition.SystemControlProperty.Value``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-systemcontrol.html#cfn-ecs-taskdefinition-systemcontrol-value
            '''
            result = self._values.get("value")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SystemControlProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.TaskDefinitionPlacementConstraintProperty",
        jsii_struct_bases=[],
        name_mapping={"type": "type", "expression": "expression"},
    )
    class TaskDefinitionPlacementConstraintProperty:
        def __init__(
            self,
            *,
            type: builtins.str,
            expression: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param type: ``CfnTaskDefinition.TaskDefinitionPlacementConstraintProperty.Type``.
            :param expression: ``CfnTaskDefinition.TaskDefinitionPlacementConstraintProperty.Expression``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-taskdefinitionplacementconstraint.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "type": type,
            }
            if expression is not None:
                self._values["expression"] = expression

        @builtins.property
        def type(self) -> builtins.str:
            '''``CfnTaskDefinition.TaskDefinitionPlacementConstraintProperty.Type``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-taskdefinitionplacementconstraint.html#cfn-ecs-taskdefinition-taskdefinitionplacementconstraint-type
            '''
            result = self._values.get("type")
            assert result is not None, "Required property 'type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def expression(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskDefinition.TaskDefinitionPlacementConstraintProperty.Expression``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-taskdefinitionplacementconstraint.html#cfn-ecs-taskdefinition-taskdefinitionplacementconstraint-expression
            '''
            result = self._values.get("expression")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TaskDefinitionPlacementConstraintProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.TmpfsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "size": "size",
            "container_path": "containerPath",
            "mount_options": "mountOptions",
        },
    )
    class TmpfsProperty:
        def __init__(
            self,
            *,
            size: jsii.Number,
            container_path: typing.Optional[builtins.str] = None,
            mount_options: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''
            :param size: ``CfnTaskDefinition.TmpfsProperty.Size``.
            :param container_path: ``CfnTaskDefinition.TmpfsProperty.ContainerPath``.
            :param mount_options: ``CfnTaskDefinition.TmpfsProperty.MountOptions``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-tmpfs.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "size": size,
            }
            if container_path is not None:
                self._values["container_path"] = container_path
            if mount_options is not None:
                self._values["mount_options"] = mount_options

        @builtins.property
        def size(self) -> jsii.Number:
            '''``CfnTaskDefinition.TmpfsProperty.Size``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-tmpfs.html#cfn-ecs-taskdefinition-tmpfs-size
            '''
            result = self._values.get("size")
            assert result is not None, "Required property 'size' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def container_path(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskDefinition.TmpfsProperty.ContainerPath``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-tmpfs.html#cfn-ecs-taskdefinition-tmpfs-containerpath
            '''
            result = self._values.get("container_path")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def mount_options(self) -> typing.Optional[typing.List[builtins.str]]:
            '''``CfnTaskDefinition.TmpfsProperty.MountOptions``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-tmpfs.html#cfn-ecs-taskdefinition-tmpfs-mountoptions
            '''
            result = self._values.get("mount_options")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TmpfsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.UlimitProperty",
        jsii_struct_bases=[],
        name_mapping={
            "hard_limit": "hardLimit",
            "name": "name",
            "soft_limit": "softLimit",
        },
    )
    class UlimitProperty:
        def __init__(
            self,
            *,
            hard_limit: jsii.Number,
            name: builtins.str,
            soft_limit: jsii.Number,
        ) -> None:
            '''
            :param hard_limit: ``CfnTaskDefinition.UlimitProperty.HardLimit``.
            :param name: ``CfnTaskDefinition.UlimitProperty.Name``.
            :param soft_limit: ``CfnTaskDefinition.UlimitProperty.SoftLimit``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-ulimit.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "hard_limit": hard_limit,
                "name": name,
                "soft_limit": soft_limit,
            }

        @builtins.property
        def hard_limit(self) -> jsii.Number:
            '''``CfnTaskDefinition.UlimitProperty.HardLimit``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-ulimit.html#cfn-ecs-taskdefinition-containerdefinition-ulimit-hardlimit
            '''
            result = self._values.get("hard_limit")
            assert result is not None, "Required property 'hard_limit' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def name(self) -> builtins.str:
            '''``CfnTaskDefinition.UlimitProperty.Name``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-ulimit.html#cfn-ecs-taskdefinition-containerdefinition-ulimit-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def soft_limit(self) -> jsii.Number:
            '''``CfnTaskDefinition.UlimitProperty.SoftLimit``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-ulimit.html#cfn-ecs-taskdefinition-containerdefinition-ulimit-softlimit
            '''
            result = self._values.get("soft_limit")
            assert result is not None, "Required property 'soft_limit' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "UlimitProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.VolumeFromProperty",
        jsii_struct_bases=[],
        name_mapping={"read_only": "readOnly", "source_container": "sourceContainer"},
    )
    class VolumeFromProperty:
        def __init__(
            self,
            *,
            read_only: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            source_container: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param read_only: ``CfnTaskDefinition.VolumeFromProperty.ReadOnly``.
            :param source_container: ``CfnTaskDefinition.VolumeFromProperty.SourceContainer``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-volumesfrom.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if read_only is not None:
                self._values["read_only"] = read_only
            if source_container is not None:
                self._values["source_container"] = source_container

        @builtins.property
        def read_only(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''``CfnTaskDefinition.VolumeFromProperty.ReadOnly``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-volumesfrom.html#cfn-ecs-taskdefinition-containerdefinition-volumesfrom-readonly
            '''
            result = self._values.get("read_only")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def source_container(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskDefinition.VolumeFromProperty.SourceContainer``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-volumesfrom.html#cfn-ecs-taskdefinition-containerdefinition-volumesfrom-sourcecontainer
            '''
            result = self._values.get("source_container")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VolumeFromProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.VolumeProperty",
        jsii_struct_bases=[],
        name_mapping={
            "docker_volume_configuration": "dockerVolumeConfiguration",
            "efs_volume_configuration": "efsVolumeConfiguration",
            "host": "host",
            "name": "name",
        },
    )
    class VolumeProperty:
        def __init__(
            self,
            *,
            docker_volume_configuration: typing.Optional[typing.Union["CfnTaskDefinition.DockerVolumeConfigurationProperty", _IResolvable_a771d0ef]] = None,
            efs_volume_configuration: typing.Optional[typing.Union["CfnTaskDefinition.EfsVolumeConfigurationProperty", _IResolvable_a771d0ef]] = None,
            host: typing.Optional[typing.Union["CfnTaskDefinition.HostVolumePropertiesProperty", _IResolvable_a771d0ef]] = None,
            name: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param docker_volume_configuration: ``CfnTaskDefinition.VolumeProperty.DockerVolumeConfiguration``.
            :param efs_volume_configuration: ``CfnTaskDefinition.VolumeProperty.EfsVolumeConfiguration``.
            :param host: ``CfnTaskDefinition.VolumeProperty.Host``.
            :param name: ``CfnTaskDefinition.VolumeProperty.Name``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-volumes.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if docker_volume_configuration is not None:
                self._values["docker_volume_configuration"] = docker_volume_configuration
            if efs_volume_configuration is not None:
                self._values["efs_volume_configuration"] = efs_volume_configuration
            if host is not None:
                self._values["host"] = host
            if name is not None:
                self._values["name"] = name

        @builtins.property
        def docker_volume_configuration(
            self,
        ) -> typing.Optional[typing.Union["CfnTaskDefinition.DockerVolumeConfigurationProperty", _IResolvable_a771d0ef]]:
            '''``CfnTaskDefinition.VolumeProperty.DockerVolumeConfiguration``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-volumes.html#cfn-ecs-taskdefinition-volume-dockervolumeconfiguration
            '''
            result = self._values.get("docker_volume_configuration")
            return typing.cast(typing.Optional[typing.Union["CfnTaskDefinition.DockerVolumeConfigurationProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def efs_volume_configuration(
            self,
        ) -> typing.Optional[typing.Union["CfnTaskDefinition.EfsVolumeConfigurationProperty", _IResolvable_a771d0ef]]:
            '''``CfnTaskDefinition.VolumeProperty.EfsVolumeConfiguration``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-volumes.html#cfn-ecs-taskdefinition-volume-efsvolumeconfiguration
            '''
            result = self._values.get("efs_volume_configuration")
            return typing.cast(typing.Optional[typing.Union["CfnTaskDefinition.EfsVolumeConfigurationProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def host(
            self,
        ) -> typing.Optional[typing.Union["CfnTaskDefinition.HostVolumePropertiesProperty", _IResolvable_a771d0ef]]:
            '''``CfnTaskDefinition.VolumeProperty.Host``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-volumes.html#cfn-ecs-taskdefinition-volumes-host
            '''
            result = self._values.get("host")
            return typing.cast(typing.Optional[typing.Union["CfnTaskDefinition.HostVolumePropertiesProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def name(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskDefinition.VolumeProperty.Name``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-volumes.html#cfn-ecs-taskdefinition-volumes-name
            '''
            result = self._values.get("name")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VolumeProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.CfnTaskDefinitionProps",
    jsii_struct_bases=[],
    name_mapping={
        "container_definitions": "containerDefinitions",
        "cpu": "cpu",
        "ephemeral_storage": "ephemeralStorage",
        "execution_role_arn": "executionRoleArn",
        "family": "family",
        "inference_accelerators": "inferenceAccelerators",
        "ipc_mode": "ipcMode",
        "memory": "memory",
        "network_mode": "networkMode",
        "pid_mode": "pidMode",
        "placement_constraints": "placementConstraints",
        "proxy_configuration": "proxyConfiguration",
        "requires_compatibilities": "requiresCompatibilities",
        "tags": "tags",
        "task_role_arn": "taskRoleArn",
        "volumes": "volumes",
    },
)
class CfnTaskDefinitionProps:
    def __init__(
        self,
        *,
        container_definitions: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[CfnTaskDefinition.ContainerDefinitionProperty, _IResolvable_a771d0ef]]]] = None,
        cpu: typing.Optional[builtins.str] = None,
        ephemeral_storage: typing.Optional[typing.Union[CfnTaskDefinition.EphemeralStorageProperty, _IResolvable_a771d0ef]] = None,
        execution_role_arn: typing.Optional[builtins.str] = None,
        family: typing.Optional[builtins.str] = None,
        inference_accelerators: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[CfnTaskDefinition.InferenceAcceleratorProperty, _IResolvable_a771d0ef]]]] = None,
        ipc_mode: typing.Optional[builtins.str] = None,
        memory: typing.Optional[builtins.str] = None,
        network_mode: typing.Optional[builtins.str] = None,
        pid_mode: typing.Optional[builtins.str] = None,
        placement_constraints: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[CfnTaskDefinition.TaskDefinitionPlacementConstraintProperty, _IResolvable_a771d0ef]]]] = None,
        proxy_configuration: typing.Optional[typing.Union[CfnTaskDefinition.ProxyConfigurationProperty, _IResolvable_a771d0ef]] = None,
        requires_compatibilities: typing.Optional[typing.Sequence[builtins.str]] = None,
        tags: typing.Optional[typing.Sequence[_CfnTag_95fbdc29]] = None,
        task_role_arn: typing.Optional[builtins.str] = None,
        volumes: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[CfnTaskDefinition.VolumeProperty, _IResolvable_a771d0ef]]]] = None,
    ) -> None:
        '''Properties for defining a ``AWS::ECS::TaskDefinition``.

        :param container_definitions: ``AWS::ECS::TaskDefinition.ContainerDefinitions``.
        :param cpu: ``AWS::ECS::TaskDefinition.Cpu``.
        :param ephemeral_storage: ``AWS::ECS::TaskDefinition.EphemeralStorage``.
        :param execution_role_arn: ``AWS::ECS::TaskDefinition.ExecutionRoleArn``.
        :param family: ``AWS::ECS::TaskDefinition.Family``.
        :param inference_accelerators: ``AWS::ECS::TaskDefinition.InferenceAccelerators``.
        :param ipc_mode: ``AWS::ECS::TaskDefinition.IpcMode``.
        :param memory: ``AWS::ECS::TaskDefinition.Memory``.
        :param network_mode: ``AWS::ECS::TaskDefinition.NetworkMode``.
        :param pid_mode: ``AWS::ECS::TaskDefinition.PidMode``.
        :param placement_constraints: ``AWS::ECS::TaskDefinition.PlacementConstraints``.
        :param proxy_configuration: ``AWS::ECS::TaskDefinition.ProxyConfiguration``.
        :param requires_compatibilities: ``AWS::ECS::TaskDefinition.RequiresCompatibilities``.
        :param tags: ``AWS::ECS::TaskDefinition.Tags``.
        :param task_role_arn: ``AWS::ECS::TaskDefinition.TaskRoleArn``.
        :param volumes: ``AWS::ECS::TaskDefinition.Volumes``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if container_definitions is not None:
            self._values["container_definitions"] = container_definitions
        if cpu is not None:
            self._values["cpu"] = cpu
        if ephemeral_storage is not None:
            self._values["ephemeral_storage"] = ephemeral_storage
        if execution_role_arn is not None:
            self._values["execution_role_arn"] = execution_role_arn
        if family is not None:
            self._values["family"] = family
        if inference_accelerators is not None:
            self._values["inference_accelerators"] = inference_accelerators
        if ipc_mode is not None:
            self._values["ipc_mode"] = ipc_mode
        if memory is not None:
            self._values["memory"] = memory
        if network_mode is not None:
            self._values["network_mode"] = network_mode
        if pid_mode is not None:
            self._values["pid_mode"] = pid_mode
        if placement_constraints is not None:
            self._values["placement_constraints"] = placement_constraints
        if proxy_configuration is not None:
            self._values["proxy_configuration"] = proxy_configuration
        if requires_compatibilities is not None:
            self._values["requires_compatibilities"] = requires_compatibilities
        if tags is not None:
            self._values["tags"] = tags
        if task_role_arn is not None:
            self._values["task_role_arn"] = task_role_arn
        if volumes is not None:
            self._values["volumes"] = volumes

    @builtins.property
    def container_definitions(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnTaskDefinition.ContainerDefinitionProperty, _IResolvable_a771d0ef]]]]:
        '''``AWS::ECS::TaskDefinition.ContainerDefinitions``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-containerdefinitions
        '''
        result = self._values.get("container_definitions")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnTaskDefinition.ContainerDefinitionProperty, _IResolvable_a771d0ef]]]], result)

    @builtins.property
    def cpu(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::TaskDefinition.Cpu``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-cpu
        '''
        result = self._values.get("cpu")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def ephemeral_storage(
        self,
    ) -> typing.Optional[typing.Union[CfnTaskDefinition.EphemeralStorageProperty, _IResolvable_a771d0ef]]:
        '''``AWS::ECS::TaskDefinition.EphemeralStorage``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-ephemeralstorage
        '''
        result = self._values.get("ephemeral_storage")
        return typing.cast(typing.Optional[typing.Union[CfnTaskDefinition.EphemeralStorageProperty, _IResolvable_a771d0ef]], result)

    @builtins.property
    def execution_role_arn(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::TaskDefinition.ExecutionRoleArn``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-executionrolearn
        '''
        result = self._values.get("execution_role_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def family(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::TaskDefinition.Family``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-family
        '''
        result = self._values.get("family")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def inference_accelerators(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnTaskDefinition.InferenceAcceleratorProperty, _IResolvable_a771d0ef]]]]:
        '''``AWS::ECS::TaskDefinition.InferenceAccelerators``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-inferenceaccelerators
        '''
        result = self._values.get("inference_accelerators")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnTaskDefinition.InferenceAcceleratorProperty, _IResolvable_a771d0ef]]]], result)

    @builtins.property
    def ipc_mode(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::TaskDefinition.IpcMode``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-ipcmode
        '''
        result = self._values.get("ipc_mode")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def memory(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::TaskDefinition.Memory``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-memory
        '''
        result = self._values.get("memory")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def network_mode(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::TaskDefinition.NetworkMode``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-networkmode
        '''
        result = self._values.get("network_mode")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def pid_mode(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::TaskDefinition.PidMode``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-pidmode
        '''
        result = self._values.get("pid_mode")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def placement_constraints(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnTaskDefinition.TaskDefinitionPlacementConstraintProperty, _IResolvable_a771d0ef]]]]:
        '''``AWS::ECS::TaskDefinition.PlacementConstraints``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-placementconstraints
        '''
        result = self._values.get("placement_constraints")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnTaskDefinition.TaskDefinitionPlacementConstraintProperty, _IResolvable_a771d0ef]]]], result)

    @builtins.property
    def proxy_configuration(
        self,
    ) -> typing.Optional[typing.Union[CfnTaskDefinition.ProxyConfigurationProperty, _IResolvable_a771d0ef]]:
        '''``AWS::ECS::TaskDefinition.ProxyConfiguration``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-proxyconfiguration
        '''
        result = self._values.get("proxy_configuration")
        return typing.cast(typing.Optional[typing.Union[CfnTaskDefinition.ProxyConfigurationProperty, _IResolvable_a771d0ef]], result)

    @builtins.property
    def requires_compatibilities(self) -> typing.Optional[typing.List[builtins.str]]:
        '''``AWS::ECS::TaskDefinition.RequiresCompatibilities``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-requirescompatibilities
        '''
        result = self._values.get("requires_compatibilities")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_95fbdc29]]:
        '''``AWS::ECS::TaskDefinition.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_95fbdc29]], result)

    @builtins.property
    def task_role_arn(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::TaskDefinition.TaskRoleArn``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-taskrolearn
        '''
        result = self._values.get("task_role_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def volumes(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnTaskDefinition.VolumeProperty, _IResolvable_a771d0ef]]]]:
        '''``AWS::ECS::TaskDefinition.Volumes``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-volumes
        '''
        result = self._values.get("volumes")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnTaskDefinition.VolumeProperty, _IResolvable_a771d0ef]]]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnTaskDefinitionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnTaskSet(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_ecs.CfnTaskSet",
):
    '''A CloudFormation ``AWS::ECS::TaskSet``.

    :cloudformationResource: AWS::ECS::TaskSet
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        cluster: builtins.str,
        service: builtins.str,
        task_definition: builtins.str,
        external_id: typing.Optional[builtins.str] = None,
        launch_type: typing.Optional[builtins.str] = None,
        load_balancers: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskSet.LoadBalancerProperty", _IResolvable_a771d0ef]]]] = None,
        network_configuration: typing.Optional[typing.Union["CfnTaskSet.NetworkConfigurationProperty", _IResolvable_a771d0ef]] = None,
        platform_version: typing.Optional[builtins.str] = None,
        scale: typing.Optional[typing.Union["CfnTaskSet.ScaleProperty", _IResolvable_a771d0ef]] = None,
        service_registries: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskSet.ServiceRegistryProperty", _IResolvable_a771d0ef]]]] = None,
    ) -> None:
        '''Create a new ``AWS::ECS::TaskSet``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param cluster: ``AWS::ECS::TaskSet.Cluster``.
        :param service: ``AWS::ECS::TaskSet.Service``.
        :param task_definition: ``AWS::ECS::TaskSet.TaskDefinition``.
        :param external_id: ``AWS::ECS::TaskSet.ExternalId``.
        :param launch_type: ``AWS::ECS::TaskSet.LaunchType``.
        :param load_balancers: ``AWS::ECS::TaskSet.LoadBalancers``.
        :param network_configuration: ``AWS::ECS::TaskSet.NetworkConfiguration``.
        :param platform_version: ``AWS::ECS::TaskSet.PlatformVersion``.
        :param scale: ``AWS::ECS::TaskSet.Scale``.
        :param service_registries: ``AWS::ECS::TaskSet.ServiceRegistries``.
        '''
        props = CfnTaskSetProps(
            cluster=cluster,
            service=service,
            task_definition=task_definition,
            external_id=external_id,
            launch_type=launch_type,
            load_balancers=load_balancers,
            network_configuration=network_configuration,
            platform_version=platform_version,
            scale=scale,
            service_registries=service_registries,
        )

        jsii.create(CfnTaskSet, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''
        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cluster")
    def cluster(self) -> builtins.str:
        '''``AWS::ECS::TaskSet.Cluster``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-cluster
        '''
        return typing.cast(builtins.str, jsii.get(self, "cluster"))

    @cluster.setter
    def cluster(self, value: builtins.str) -> None:
        jsii.set(self, "cluster", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="service")
    def service(self) -> builtins.str:
        '''``AWS::ECS::TaskSet.Service``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-service
        '''
        return typing.cast(builtins.str, jsii.get(self, "service"))

    @service.setter
    def service(self, value: builtins.str) -> None:
        jsii.set(self, "service", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskDefinition")
    def task_definition(self) -> builtins.str:
        '''``AWS::ECS::TaskSet.TaskDefinition``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-taskdefinition
        '''
        return typing.cast(builtins.str, jsii.get(self, "taskDefinition"))

    @task_definition.setter
    def task_definition(self, value: builtins.str) -> None:
        jsii.set(self, "taskDefinition", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="externalId")
    def external_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::TaskSet.ExternalId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-externalid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "externalId"))

    @external_id.setter
    def external_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "externalId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="launchType")
    def launch_type(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::TaskSet.LaunchType``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-launchtype
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "launchType"))

    @launch_type.setter
    def launch_type(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "launchType", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="loadBalancers")
    def load_balancers(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskSet.LoadBalancerProperty", _IResolvable_a771d0ef]]]]:
        '''``AWS::ECS::TaskSet.LoadBalancers``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-loadbalancers
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskSet.LoadBalancerProperty", _IResolvable_a771d0ef]]]], jsii.get(self, "loadBalancers"))

    @load_balancers.setter
    def load_balancers(
        self,
        value: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskSet.LoadBalancerProperty", _IResolvable_a771d0ef]]]],
    ) -> None:
        jsii.set(self, "loadBalancers", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="networkConfiguration")
    def network_configuration(
        self,
    ) -> typing.Optional[typing.Union["CfnTaskSet.NetworkConfigurationProperty", _IResolvable_a771d0ef]]:
        '''``AWS::ECS::TaskSet.NetworkConfiguration``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-networkconfiguration
        '''
        return typing.cast(typing.Optional[typing.Union["CfnTaskSet.NetworkConfigurationProperty", _IResolvable_a771d0ef]], jsii.get(self, "networkConfiguration"))

    @network_configuration.setter
    def network_configuration(
        self,
        value: typing.Optional[typing.Union["CfnTaskSet.NetworkConfigurationProperty", _IResolvable_a771d0ef]],
    ) -> None:
        jsii.set(self, "networkConfiguration", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="platformVersion")
    def platform_version(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::TaskSet.PlatformVersion``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-platformversion
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "platformVersion"))

    @platform_version.setter
    def platform_version(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "platformVersion", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="scale")
    def scale(
        self,
    ) -> typing.Optional[typing.Union["CfnTaskSet.ScaleProperty", _IResolvable_a771d0ef]]:
        '''``AWS::ECS::TaskSet.Scale``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-scale
        '''
        return typing.cast(typing.Optional[typing.Union["CfnTaskSet.ScaleProperty", _IResolvable_a771d0ef]], jsii.get(self, "scale"))

    @scale.setter
    def scale(
        self,
        value: typing.Optional[typing.Union["CfnTaskSet.ScaleProperty", _IResolvable_a771d0ef]],
    ) -> None:
        jsii.set(self, "scale", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="serviceRegistries")
    def service_registries(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskSet.ServiceRegistryProperty", _IResolvable_a771d0ef]]]]:
        '''``AWS::ECS::TaskSet.ServiceRegistries``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-serviceregistries
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskSet.ServiceRegistryProperty", _IResolvable_a771d0ef]]]], jsii.get(self, "serviceRegistries"))

    @service_registries.setter
    def service_registries(
        self,
        value: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskSet.ServiceRegistryProperty", _IResolvable_a771d0ef]]]],
    ) -> None:
        jsii.set(self, "serviceRegistries", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskSet.AwsVpcConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "subnets": "subnets",
            "assign_public_ip": "assignPublicIp",
            "security_groups": "securityGroups",
        },
    )
    class AwsVpcConfigurationProperty:
        def __init__(
            self,
            *,
            subnets: typing.Sequence[builtins.str],
            assign_public_ip: typing.Optional[builtins.str] = None,
            security_groups: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''
            :param subnets: ``CfnTaskSet.AwsVpcConfigurationProperty.Subnets``.
            :param assign_public_ip: ``CfnTaskSet.AwsVpcConfigurationProperty.AssignPublicIp``.
            :param security_groups: ``CfnTaskSet.AwsVpcConfigurationProperty.SecurityGroups``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskset-awsvpcconfiguration.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "subnets": subnets,
            }
            if assign_public_ip is not None:
                self._values["assign_public_ip"] = assign_public_ip
            if security_groups is not None:
                self._values["security_groups"] = security_groups

        @builtins.property
        def subnets(self) -> typing.List[builtins.str]:
            '''``CfnTaskSet.AwsVpcConfigurationProperty.Subnets``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskset-awsvpcconfiguration.html#cfn-ecs-taskset-awsvpcconfiguration-subnets
            '''
            result = self._values.get("subnets")
            assert result is not None, "Required property 'subnets' is missing"
            return typing.cast(typing.List[builtins.str], result)

        @builtins.property
        def assign_public_ip(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskSet.AwsVpcConfigurationProperty.AssignPublicIp``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskset-awsvpcconfiguration.html#cfn-ecs-taskset-awsvpcconfiguration-assignpublicip
            '''
            result = self._values.get("assign_public_ip")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def security_groups(self) -> typing.Optional[typing.List[builtins.str]]:
            '''``CfnTaskSet.AwsVpcConfigurationProperty.SecurityGroups``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskset-awsvpcconfiguration.html#cfn-ecs-taskset-awsvpcconfiguration-securitygroups
            '''
            result = self._values.get("security_groups")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AwsVpcConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskSet.LoadBalancerProperty",
        jsii_struct_bases=[],
        name_mapping={
            "container_name": "containerName",
            "container_port": "containerPort",
            "load_balancer_name": "loadBalancerName",
            "target_group_arn": "targetGroupArn",
        },
    )
    class LoadBalancerProperty:
        def __init__(
            self,
            *,
            container_name: typing.Optional[builtins.str] = None,
            container_port: typing.Optional[jsii.Number] = None,
            load_balancer_name: typing.Optional[builtins.str] = None,
            target_group_arn: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param container_name: ``CfnTaskSet.LoadBalancerProperty.ContainerName``.
            :param container_port: ``CfnTaskSet.LoadBalancerProperty.ContainerPort``.
            :param load_balancer_name: ``CfnTaskSet.LoadBalancerProperty.LoadBalancerName``.
            :param target_group_arn: ``CfnTaskSet.LoadBalancerProperty.TargetGroupArn``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskset-loadbalancer.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if container_name is not None:
                self._values["container_name"] = container_name
            if container_port is not None:
                self._values["container_port"] = container_port
            if load_balancer_name is not None:
                self._values["load_balancer_name"] = load_balancer_name
            if target_group_arn is not None:
                self._values["target_group_arn"] = target_group_arn

        @builtins.property
        def container_name(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskSet.LoadBalancerProperty.ContainerName``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskset-loadbalancer.html#cfn-ecs-taskset-loadbalancer-containername
            '''
            result = self._values.get("container_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def container_port(self) -> typing.Optional[jsii.Number]:
            '''``CfnTaskSet.LoadBalancerProperty.ContainerPort``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskset-loadbalancer.html#cfn-ecs-taskset-loadbalancer-containerport
            '''
            result = self._values.get("container_port")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def load_balancer_name(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskSet.LoadBalancerProperty.LoadBalancerName``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskset-loadbalancer.html#cfn-ecs-taskset-loadbalancer-loadbalancername
            '''
            result = self._values.get("load_balancer_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def target_group_arn(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskSet.LoadBalancerProperty.TargetGroupArn``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskset-loadbalancer.html#cfn-ecs-taskset-loadbalancer-targetgrouparn
            '''
            result = self._values.get("target_group_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LoadBalancerProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskSet.NetworkConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"aws_vpc_configuration": "awsVpcConfiguration"},
    )
    class NetworkConfigurationProperty:
        def __init__(
            self,
            *,
            aws_vpc_configuration: typing.Optional[typing.Union["CfnTaskSet.AwsVpcConfigurationProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''
            :param aws_vpc_configuration: ``CfnTaskSet.NetworkConfigurationProperty.AwsVpcConfiguration``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskset-networkconfiguration.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if aws_vpc_configuration is not None:
                self._values["aws_vpc_configuration"] = aws_vpc_configuration

        @builtins.property
        def aws_vpc_configuration(
            self,
        ) -> typing.Optional[typing.Union["CfnTaskSet.AwsVpcConfigurationProperty", _IResolvable_a771d0ef]]:
            '''``CfnTaskSet.NetworkConfigurationProperty.AwsVpcConfiguration``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskset-networkconfiguration.html#cfn-ecs-taskset-networkconfiguration-awsvpcconfiguration
            '''
            result = self._values.get("aws_vpc_configuration")
            return typing.cast(typing.Optional[typing.Union["CfnTaskSet.AwsVpcConfigurationProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "NetworkConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskSet.ScaleProperty",
        jsii_struct_bases=[],
        name_mapping={"unit": "unit", "value": "value"},
    )
    class ScaleProperty:
        def __init__(
            self,
            *,
            unit: typing.Optional[builtins.str] = None,
            value: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''
            :param unit: ``CfnTaskSet.ScaleProperty.Unit``.
            :param value: ``CfnTaskSet.ScaleProperty.Value``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskset-scale.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if unit is not None:
                self._values["unit"] = unit
            if value is not None:
                self._values["value"] = value

        @builtins.property
        def unit(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskSet.ScaleProperty.Unit``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskset-scale.html#cfn-ecs-taskset-scale-unit
            '''
            result = self._values.get("unit")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def value(self) -> typing.Optional[jsii.Number]:
            '''``CfnTaskSet.ScaleProperty.Value``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskset-scale.html#cfn-ecs-taskset-scale-value
            '''
            result = self._values.get("value")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ScaleProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskSet.ServiceRegistryProperty",
        jsii_struct_bases=[],
        name_mapping={
            "container_name": "containerName",
            "container_port": "containerPort",
            "port": "port",
            "registry_arn": "registryArn",
        },
    )
    class ServiceRegistryProperty:
        def __init__(
            self,
            *,
            container_name: typing.Optional[builtins.str] = None,
            container_port: typing.Optional[jsii.Number] = None,
            port: typing.Optional[jsii.Number] = None,
            registry_arn: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param container_name: ``CfnTaskSet.ServiceRegistryProperty.ContainerName``.
            :param container_port: ``CfnTaskSet.ServiceRegistryProperty.ContainerPort``.
            :param port: ``CfnTaskSet.ServiceRegistryProperty.Port``.
            :param registry_arn: ``CfnTaskSet.ServiceRegistryProperty.RegistryArn``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskset-serviceregistry.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if container_name is not None:
                self._values["container_name"] = container_name
            if container_port is not None:
                self._values["container_port"] = container_port
            if port is not None:
                self._values["port"] = port
            if registry_arn is not None:
                self._values["registry_arn"] = registry_arn

        @builtins.property
        def container_name(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskSet.ServiceRegistryProperty.ContainerName``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskset-serviceregistry.html#cfn-ecs-taskset-serviceregistry-containername
            '''
            result = self._values.get("container_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def container_port(self) -> typing.Optional[jsii.Number]:
            '''``CfnTaskSet.ServiceRegistryProperty.ContainerPort``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskset-serviceregistry.html#cfn-ecs-taskset-serviceregistry-containerport
            '''
            result = self._values.get("container_port")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def port(self) -> typing.Optional[jsii.Number]:
            '''``CfnTaskSet.ServiceRegistryProperty.Port``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskset-serviceregistry.html#cfn-ecs-taskset-serviceregistry-port
            '''
            result = self._values.get("port")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def registry_arn(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskSet.ServiceRegistryProperty.RegistryArn``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskset-serviceregistry.html#cfn-ecs-taskset-serviceregistry-registryarn
            '''
            result = self._values.get("registry_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ServiceRegistryProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.CfnTaskSetProps",
    jsii_struct_bases=[],
    name_mapping={
        "cluster": "cluster",
        "service": "service",
        "task_definition": "taskDefinition",
        "external_id": "externalId",
        "launch_type": "launchType",
        "load_balancers": "loadBalancers",
        "network_configuration": "networkConfiguration",
        "platform_version": "platformVersion",
        "scale": "scale",
        "service_registries": "serviceRegistries",
    },
)
class CfnTaskSetProps:
    def __init__(
        self,
        *,
        cluster: builtins.str,
        service: builtins.str,
        task_definition: builtins.str,
        external_id: typing.Optional[builtins.str] = None,
        launch_type: typing.Optional[builtins.str] = None,
        load_balancers: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[CfnTaskSet.LoadBalancerProperty, _IResolvable_a771d0ef]]]] = None,
        network_configuration: typing.Optional[typing.Union[CfnTaskSet.NetworkConfigurationProperty, _IResolvable_a771d0ef]] = None,
        platform_version: typing.Optional[builtins.str] = None,
        scale: typing.Optional[typing.Union[CfnTaskSet.ScaleProperty, _IResolvable_a771d0ef]] = None,
        service_registries: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[CfnTaskSet.ServiceRegistryProperty, _IResolvable_a771d0ef]]]] = None,
    ) -> None:
        '''Properties for defining a ``AWS::ECS::TaskSet``.

        :param cluster: ``AWS::ECS::TaskSet.Cluster``.
        :param service: ``AWS::ECS::TaskSet.Service``.
        :param task_definition: ``AWS::ECS::TaskSet.TaskDefinition``.
        :param external_id: ``AWS::ECS::TaskSet.ExternalId``.
        :param launch_type: ``AWS::ECS::TaskSet.LaunchType``.
        :param load_balancers: ``AWS::ECS::TaskSet.LoadBalancers``.
        :param network_configuration: ``AWS::ECS::TaskSet.NetworkConfiguration``.
        :param platform_version: ``AWS::ECS::TaskSet.PlatformVersion``.
        :param scale: ``AWS::ECS::TaskSet.Scale``.
        :param service_registries: ``AWS::ECS::TaskSet.ServiceRegistries``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "cluster": cluster,
            "service": service,
            "task_definition": task_definition,
        }
        if external_id is not None:
            self._values["external_id"] = external_id
        if launch_type is not None:
            self._values["launch_type"] = launch_type
        if load_balancers is not None:
            self._values["load_balancers"] = load_balancers
        if network_configuration is not None:
            self._values["network_configuration"] = network_configuration
        if platform_version is not None:
            self._values["platform_version"] = platform_version
        if scale is not None:
            self._values["scale"] = scale
        if service_registries is not None:
            self._values["service_registries"] = service_registries

    @builtins.property
    def cluster(self) -> builtins.str:
        '''``AWS::ECS::TaskSet.Cluster``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-cluster
        '''
        result = self._values.get("cluster")
        assert result is not None, "Required property 'cluster' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def service(self) -> builtins.str:
        '''``AWS::ECS::TaskSet.Service``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-service
        '''
        result = self._values.get("service")
        assert result is not None, "Required property 'service' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def task_definition(self) -> builtins.str:
        '''``AWS::ECS::TaskSet.TaskDefinition``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-taskdefinition
        '''
        result = self._values.get("task_definition")
        assert result is not None, "Required property 'task_definition' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def external_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::TaskSet.ExternalId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-externalid
        '''
        result = self._values.get("external_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def launch_type(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::TaskSet.LaunchType``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-launchtype
        '''
        result = self._values.get("launch_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def load_balancers(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnTaskSet.LoadBalancerProperty, _IResolvable_a771d0ef]]]]:
        '''``AWS::ECS::TaskSet.LoadBalancers``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-loadbalancers
        '''
        result = self._values.get("load_balancers")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnTaskSet.LoadBalancerProperty, _IResolvable_a771d0ef]]]], result)

    @builtins.property
    def network_configuration(
        self,
    ) -> typing.Optional[typing.Union[CfnTaskSet.NetworkConfigurationProperty, _IResolvable_a771d0ef]]:
        '''``AWS::ECS::TaskSet.NetworkConfiguration``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-networkconfiguration
        '''
        result = self._values.get("network_configuration")
        return typing.cast(typing.Optional[typing.Union[CfnTaskSet.NetworkConfigurationProperty, _IResolvable_a771d0ef]], result)

    @builtins.property
    def platform_version(self) -> typing.Optional[builtins.str]:
        '''``AWS::ECS::TaskSet.PlatformVersion``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-platformversion
        '''
        result = self._values.get("platform_version")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def scale(
        self,
    ) -> typing.Optional[typing.Union[CfnTaskSet.ScaleProperty, _IResolvable_a771d0ef]]:
        '''``AWS::ECS::TaskSet.Scale``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-scale
        '''
        result = self._values.get("scale")
        return typing.cast(typing.Optional[typing.Union[CfnTaskSet.ScaleProperty, _IResolvable_a771d0ef]], result)

    @builtins.property
    def service_registries(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnTaskSet.ServiceRegistryProperty, _IResolvable_a771d0ef]]]]:
        '''``AWS::ECS::TaskSet.ServiceRegistries``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-serviceregistries
        '''
        result = self._values.get("service_registries")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnTaskSet.ServiceRegistryProperty, _IResolvable_a771d0ef]]]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnTaskSetProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.CloudMapNamespaceOptions",
    jsii_struct_bases=[],
    name_mapping={"name": "name", "type": "type", "vpc": "vpc"},
)
class CloudMapNamespaceOptions:
    def __init__(
        self,
        *,
        name: builtins.str,
        type: typing.Optional[_NamespaceType_db55a104] = None,
        vpc: typing.Optional[_IVpc_6d1f76c4] = None,
    ) -> None:
        '''(experimental) The options for creating an AWS Cloud Map namespace.

        :param name: (experimental) The name of the namespace, such as example.com.
        :param type: (experimental) The type of CloudMap Namespace to create. Default: PrivateDns
        :param vpc: (experimental) The VPC to associate the namespace with. This property is required for private DNS namespaces. Default: VPC of the cluster for Private DNS Namespace, otherwise none

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "name": name,
        }
        if type is not None:
            self._values["type"] = type
        if vpc is not None:
            self._values["vpc"] = vpc

    @builtins.property
    def name(self) -> builtins.str:
        '''(experimental) The name of the namespace, such as example.com.

        :stability: experimental
        '''
        result = self._values.get("name")
        assert result is not None, "Required property 'name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def type(self) -> typing.Optional[_NamespaceType_db55a104]:
        '''(experimental) The type of CloudMap Namespace to create.

        :default: PrivateDns

        :stability: experimental
        '''
        result = self._values.get("type")
        return typing.cast(typing.Optional[_NamespaceType_db55a104], result)

    @builtins.property
    def vpc(self) -> typing.Optional[_IVpc_6d1f76c4]:
        '''(experimental) The VPC to associate the namespace with.

        This property is required for private DNS namespaces.

        :default: VPC of the cluster for Private DNS Namespace, otherwise none

        :stability: experimental
        '''
        result = self._values.get("vpc")
        return typing.cast(typing.Optional[_IVpc_6d1f76c4], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CloudMapNamespaceOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.CloudMapOptions",
    jsii_struct_bases=[],
    name_mapping={
        "cloud_map_namespace": "cloudMapNamespace",
        "container": "container",
        "container_port": "containerPort",
        "dns_record_type": "dnsRecordType",
        "dns_ttl": "dnsTtl",
        "failure_threshold": "failureThreshold",
        "name": "name",
    },
)
class CloudMapOptions:
    def __init__(
        self,
        *,
        cloud_map_namespace: typing.Optional[_INamespace_eb21e42e] = None,
        container: typing.Optional["ContainerDefinition"] = None,
        container_port: typing.Optional[jsii.Number] = None,
        dns_record_type: typing.Optional[_DnsRecordType_a28443d0] = None,
        dns_ttl: typing.Optional[_Duration_070aa057] = None,
        failure_threshold: typing.Optional[jsii.Number] = None,
        name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) The options to enabling AWS Cloud Map for an Amazon ECS service.

        :param cloud_map_namespace: (experimental) The service discovery namespace for the Cloud Map service to attach to the ECS service. Default: - the defaultCloudMapNamespace associated to the cluster
        :param container: (experimental) The container to point to for a SRV record. Default: - the task definition's default container
        :param container_port: (experimental) The port to point to for a SRV record. Default: - the default port of the task definition's default container
        :param dns_record_type: (experimental) The DNS record type that you want AWS Cloud Map to create. The supported record types are A or SRV. Default: - DnsRecordType.A if TaskDefinition.networkMode = AWS_VPC, otherwise DnsRecordType.SRV
        :param dns_ttl: (experimental) The amount of time that you want DNS resolvers to cache the settings for this record. Default: 60
        :param failure_threshold: (experimental) The number of 30-second intervals that you want Cloud Map to wait after receiving an UpdateInstanceCustomHealthStatus request before it changes the health status of a service instance. NOTE: This is used for HealthCheckCustomConfig
        :param name: (experimental) The name of the Cloud Map service to attach to the ECS service. Default: CloudFormation-generated name

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if cloud_map_namespace is not None:
            self._values["cloud_map_namespace"] = cloud_map_namespace
        if container is not None:
            self._values["container"] = container
        if container_port is not None:
            self._values["container_port"] = container_port
        if dns_record_type is not None:
            self._values["dns_record_type"] = dns_record_type
        if dns_ttl is not None:
            self._values["dns_ttl"] = dns_ttl
        if failure_threshold is not None:
            self._values["failure_threshold"] = failure_threshold
        if name is not None:
            self._values["name"] = name

    @builtins.property
    def cloud_map_namespace(self) -> typing.Optional[_INamespace_eb21e42e]:
        '''(experimental) The service discovery namespace for the Cloud Map service to attach to the ECS service.

        :default: - the defaultCloudMapNamespace associated to the cluster

        :stability: experimental
        '''
        result = self._values.get("cloud_map_namespace")
        return typing.cast(typing.Optional[_INamespace_eb21e42e], result)

    @builtins.property
    def container(self) -> typing.Optional["ContainerDefinition"]:
        '''(experimental) The container to point to for a SRV record.

        :default: - the task definition's default container

        :stability: experimental
        '''
        result = self._values.get("container")
        return typing.cast(typing.Optional["ContainerDefinition"], result)

    @builtins.property
    def container_port(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The port to point to for a SRV record.

        :default: - the default port of the task definition's default container

        :stability: experimental
        '''
        result = self._values.get("container_port")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def dns_record_type(self) -> typing.Optional[_DnsRecordType_a28443d0]:
        '''(experimental) The DNS record type that you want AWS Cloud Map to create.

        The supported record types are A or SRV.

        :default: - DnsRecordType.A if TaskDefinition.networkMode = AWS_VPC, otherwise DnsRecordType.SRV

        :stability: experimental
        '''
        result = self._values.get("dns_record_type")
        return typing.cast(typing.Optional[_DnsRecordType_a28443d0], result)

    @builtins.property
    def dns_ttl(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) The amount of time that you want DNS resolvers to cache the settings for this record.

        :default: 60

        :stability: experimental
        '''
        result = self._values.get("dns_ttl")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def failure_threshold(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The number of 30-second intervals that you want Cloud Map to wait after receiving an UpdateInstanceCustomHealthStatus request before it changes the health status of a service instance.

        NOTE: This is used for HealthCheckCustomConfig

        :stability: experimental
        '''
        result = self._values.get("failure_threshold")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name of the Cloud Map service to attach to the ECS service.

        :default: CloudFormation-generated name

        :stability: experimental
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CloudMapOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.ClusterAttributes",
    jsii_struct_bases=[],
    name_mapping={
        "cluster_name": "clusterName",
        "security_groups": "securityGroups",
        "vpc": "vpc",
        "autoscaling_group": "autoscalingGroup",
        "cluster_arn": "clusterArn",
        "default_cloud_map_namespace": "defaultCloudMapNamespace",
        "has_ec2_capacity": "hasEc2Capacity",
    },
)
class ClusterAttributes:
    def __init__(
        self,
        *,
        cluster_name: builtins.str,
        security_groups: typing.Sequence[_ISecurityGroup_cdbba9d3],
        vpc: _IVpc_6d1f76c4,
        autoscaling_group: typing.Optional[_IAutoScalingGroup_352723c6] = None,
        cluster_arn: typing.Optional[builtins.str] = None,
        default_cloud_map_namespace: typing.Optional[_INamespace_eb21e42e] = None,
        has_ec2_capacity: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''(experimental) The properties to import from the ECS cluster.

        :param cluster_name: (experimental) The name of the cluster.
        :param security_groups: (experimental) The security groups associated with the container instances registered to the cluster.
        :param vpc: (experimental) The VPC associated with the cluster.
        :param autoscaling_group: (experimental) Autoscaling group added to the cluster if capacity is added. Default: - No default autoscaling group
        :param cluster_arn: (experimental) The Amazon Resource Name (ARN) that identifies the cluster. Default: Derived from clusterName
        :param default_cloud_map_namespace: (experimental) The AWS Cloud Map namespace to associate with the cluster. Default: - No default namespace
        :param has_ec2_capacity: (experimental) Specifies whether the cluster has EC2 instance capacity. Default: true

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "cluster_name": cluster_name,
            "security_groups": security_groups,
            "vpc": vpc,
        }
        if autoscaling_group is not None:
            self._values["autoscaling_group"] = autoscaling_group
        if cluster_arn is not None:
            self._values["cluster_arn"] = cluster_arn
        if default_cloud_map_namespace is not None:
            self._values["default_cloud_map_namespace"] = default_cloud_map_namespace
        if has_ec2_capacity is not None:
            self._values["has_ec2_capacity"] = has_ec2_capacity

    @builtins.property
    def cluster_name(self) -> builtins.str:
        '''(experimental) The name of the cluster.

        :stability: experimental
        '''
        result = self._values.get("cluster_name")
        assert result is not None, "Required property 'cluster_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def security_groups(self) -> typing.List[_ISecurityGroup_cdbba9d3]:
        '''(experimental) The security groups associated with the container instances registered to the cluster.

        :stability: experimental
        '''
        result = self._values.get("security_groups")
        assert result is not None, "Required property 'security_groups' is missing"
        return typing.cast(typing.List[_ISecurityGroup_cdbba9d3], result)

    @builtins.property
    def vpc(self) -> _IVpc_6d1f76c4:
        '''(experimental) The VPC associated with the cluster.

        :stability: experimental
        '''
        result = self._values.get("vpc")
        assert result is not None, "Required property 'vpc' is missing"
        return typing.cast(_IVpc_6d1f76c4, result)

    @builtins.property
    def autoscaling_group(self) -> typing.Optional[_IAutoScalingGroup_352723c6]:
        '''(experimental) Autoscaling group added to the cluster if capacity is added.

        :default: - No default autoscaling group

        :stability: experimental
        '''
        result = self._values.get("autoscaling_group")
        return typing.cast(typing.Optional[_IAutoScalingGroup_352723c6], result)

    @builtins.property
    def cluster_arn(self) -> typing.Optional[builtins.str]:
        '''(experimental) The Amazon Resource Name (ARN) that identifies the cluster.

        :default: Derived from clusterName

        :stability: experimental
        '''
        result = self._values.get("cluster_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def default_cloud_map_namespace(self) -> typing.Optional[_INamespace_eb21e42e]:
        '''(experimental) The AWS Cloud Map namespace to associate with the cluster.

        :default: - No default namespace

        :stability: experimental
        '''
        result = self._values.get("default_cloud_map_namespace")
        return typing.cast(typing.Optional[_INamespace_eb21e42e], result)

    @builtins.property
    def has_ec2_capacity(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specifies whether the cluster has EC2 instance capacity.

        :default: true

        :stability: experimental
        '''
        result = self._values.get("has_ec2_capacity")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ClusterAttributes(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.ClusterProps",
    jsii_struct_bases=[],
    name_mapping={
        "capacity": "capacity",
        "capacity_providers": "capacityProviders",
        "cluster_name": "clusterName",
        "container_insights": "containerInsights",
        "default_cloud_map_namespace": "defaultCloudMapNamespace",
        "enable_fargate_capacity_providers": "enableFargateCapacityProviders",
        "vpc": "vpc",
    },
)
class ClusterProps:
    def __init__(
        self,
        *,
        capacity: typing.Optional[AddCapacityOptions] = None,
        capacity_providers: typing.Optional[typing.Sequence[builtins.str]] = None,
        cluster_name: typing.Optional[builtins.str] = None,
        container_insights: typing.Optional[builtins.bool] = None,
        default_cloud_map_namespace: typing.Optional[CloudMapNamespaceOptions] = None,
        enable_fargate_capacity_providers: typing.Optional[builtins.bool] = None,
        vpc: typing.Optional[_IVpc_6d1f76c4] = None,
    ) -> None:
        '''(experimental) The properties used to define an ECS cluster.

        :param capacity: (experimental) The ec2 capacity to add to the cluster. Default: - no EC2 capacity will be added, you can use ``addCapacity`` to add capacity later.
        :param capacity_providers: (deprecated) The capacity providers to add to the cluster. Default: - None. Currently only FARGATE and FARGATE_SPOT are supported.
        :param cluster_name: (experimental) The name for the cluster. Default: CloudFormation-generated name
        :param container_insights: (experimental) If true CloudWatch Container Insights will be enabled for the cluster. Default: - Container Insights will be disabled for this cluser.
        :param default_cloud_map_namespace: (experimental) The service discovery namespace created in this cluster. Default: - no service discovery namespace created, you can use ``addDefaultCloudMapNamespace`` to add a default service discovery namespace later.
        :param enable_fargate_capacity_providers: (experimental) Whether to enable Fargate Capacity Providers. Default: false
        :param vpc: (experimental) The VPC where your ECS instances will be running or your ENIs will be deployed. Default: - creates a new VPC with two AZs

        :stability: experimental
        '''
        if isinstance(capacity, dict):
            capacity = AddCapacityOptions(**capacity)
        if isinstance(default_cloud_map_namespace, dict):
            default_cloud_map_namespace = CloudMapNamespaceOptions(**default_cloud_map_namespace)
        self._values: typing.Dict[str, typing.Any] = {}
        if capacity is not None:
            self._values["capacity"] = capacity
        if capacity_providers is not None:
            self._values["capacity_providers"] = capacity_providers
        if cluster_name is not None:
            self._values["cluster_name"] = cluster_name
        if container_insights is not None:
            self._values["container_insights"] = container_insights
        if default_cloud_map_namespace is not None:
            self._values["default_cloud_map_namespace"] = default_cloud_map_namespace
        if enable_fargate_capacity_providers is not None:
            self._values["enable_fargate_capacity_providers"] = enable_fargate_capacity_providers
        if vpc is not None:
            self._values["vpc"] = vpc

    @builtins.property
    def capacity(self) -> typing.Optional[AddCapacityOptions]:
        '''(experimental) The ec2 capacity to add to the cluster.

        :default: - no EC2 capacity will be added, you can use ``addCapacity`` to add capacity later.

        :stability: experimental
        '''
        result = self._values.get("capacity")
        return typing.cast(typing.Optional[AddCapacityOptions], result)

    @builtins.property
    def capacity_providers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(deprecated) The capacity providers to add to the cluster.

        :default: - None. Currently only FARGATE and FARGATE_SPOT are supported.

        :deprecated: Use {@link ClusterProps.enableFargateCapacityProviders} instead.

        :stability: deprecated
        '''
        result = self._values.get("capacity_providers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def cluster_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name for the cluster.

        :default: CloudFormation-generated name

        :stability: experimental
        '''
        result = self._values.get("cluster_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def container_insights(self) -> typing.Optional[builtins.bool]:
        '''(experimental) If true CloudWatch Container Insights will be enabled for the cluster.

        :default: - Container Insights will be disabled for this cluser.

        :stability: experimental
        '''
        result = self._values.get("container_insights")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def default_cloud_map_namespace(self) -> typing.Optional[CloudMapNamespaceOptions]:
        '''(experimental) The service discovery namespace created in this cluster.

        :default:

        - no service discovery namespace created, you can use ``addDefaultCloudMapNamespace`` to add a
        default service discovery namespace later.

        :stability: experimental
        '''
        result = self._values.get("default_cloud_map_namespace")
        return typing.cast(typing.Optional[CloudMapNamespaceOptions], result)

    @builtins.property
    def enable_fargate_capacity_providers(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether to enable Fargate Capacity Providers.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("enable_fargate_capacity_providers")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def vpc(self) -> typing.Optional[_IVpc_6d1f76c4]:
        '''(experimental) The VPC where your ECS instances will be running or your ENIs will be deployed.

        :default: - creates a new VPC with two AZs

        :stability: experimental
        '''
        result = self._values.get("vpc")
        return typing.cast(typing.Optional[_IVpc_6d1f76c4], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ClusterProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.CommonTaskDefinitionAttributes",
    jsii_struct_bases=[],
    name_mapping={
        "task_definition_arn": "taskDefinitionArn",
        "network_mode": "networkMode",
        "task_role": "taskRole",
    },
)
class CommonTaskDefinitionAttributes:
    def __init__(
        self,
        *,
        task_definition_arn: builtins.str,
        network_mode: typing.Optional["NetworkMode"] = None,
        task_role: typing.Optional[_IRole_59af6f50] = None,
    ) -> None:
        '''(experimental) The common task definition attributes used across all types of task definitions.

        :param task_definition_arn: (experimental) The arn of the task definition.
        :param network_mode: (experimental) The networking mode to use for the containers in the task. Default: Network mode cannot be provided to the imported task.
        :param task_role: (experimental) The name of the IAM role that grants containers in the task permission to call AWS APIs on your behalf. Default: Permissions cannot be granted to the imported task.

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "task_definition_arn": task_definition_arn,
        }
        if network_mode is not None:
            self._values["network_mode"] = network_mode
        if task_role is not None:
            self._values["task_role"] = task_role

    @builtins.property
    def task_definition_arn(self) -> builtins.str:
        '''(experimental) The arn of the task definition.

        :stability: experimental
        '''
        result = self._values.get("task_definition_arn")
        assert result is not None, "Required property 'task_definition_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def network_mode(self) -> typing.Optional["NetworkMode"]:
        '''(experimental) The networking mode to use for the containers in the task.

        :default: Network mode cannot be provided to the imported task.

        :stability: experimental
        '''
        result = self._values.get("network_mode")
        return typing.cast(typing.Optional["NetworkMode"], result)

    @builtins.property
    def task_role(self) -> typing.Optional[_IRole_59af6f50]:
        '''(experimental) The name of the IAM role that grants containers in the task permission to call AWS APIs on your behalf.

        :default: Permissions cannot be granted to the imported task.

        :stability: experimental
        '''
        result = self._values.get("task_role")
        return typing.cast(typing.Optional[_IRole_59af6f50], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CommonTaskDefinitionAttributes(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.CommonTaskDefinitionProps",
    jsii_struct_bases=[],
    name_mapping={
        "execution_role": "executionRole",
        "family": "family",
        "proxy_configuration": "proxyConfiguration",
        "task_role": "taskRole",
        "volumes": "volumes",
    },
)
class CommonTaskDefinitionProps:
    def __init__(
        self,
        *,
        execution_role: typing.Optional[_IRole_59af6f50] = None,
        family: typing.Optional[builtins.str] = None,
        proxy_configuration: typing.Optional["ProxyConfiguration"] = None,
        task_role: typing.Optional[_IRole_59af6f50] = None,
        volumes: typing.Optional[typing.Sequence["Volume"]] = None,
    ) -> None:
        '''(experimental) The common properties for all task definitions.

        For more information, see
        `Task Definition Parameters <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_definition_parameters.html>`_.

        :param execution_role: (experimental) The name of the IAM task execution role that grants the ECS agent to call AWS APIs on your behalf. The role will be used to retrieve container images from ECR and create CloudWatch log groups. Default: - An execution role will be automatically created if you use ECR images in your task definition.
        :param family: (experimental) The name of a family that this task definition is registered to. A family groups multiple versions of a task definition. Default: - Automatically generated name.
        :param proxy_configuration: (experimental) The configuration details for the App Mesh proxy. Default: - No proxy configuration.
        :param task_role: (experimental) The name of the IAM role that grants containers in the task permission to call AWS APIs on your behalf. Default: - A task role is automatically created for you.
        :param volumes: (experimental) The list of volume definitions for the task. For more information, see `Task Definition Parameter Volumes <https://docs.aws.amazon.com/AmazonECS/latest/developerguide//task_definition_parameters.html#volumes>`_. Default: - No volumes are passed to the Docker daemon on a container instance.

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if execution_role is not None:
            self._values["execution_role"] = execution_role
        if family is not None:
            self._values["family"] = family
        if proxy_configuration is not None:
            self._values["proxy_configuration"] = proxy_configuration
        if task_role is not None:
            self._values["task_role"] = task_role
        if volumes is not None:
            self._values["volumes"] = volumes

    @builtins.property
    def execution_role(self) -> typing.Optional[_IRole_59af6f50]:
        '''(experimental) The name of the IAM task execution role that grants the ECS agent to call AWS APIs on your behalf.

        The role will be used to retrieve container images from ECR and create CloudWatch log groups.

        :default: - An execution role will be automatically created if you use ECR images in your task definition.

        :stability: experimental
        '''
        result = self._values.get("execution_role")
        return typing.cast(typing.Optional[_IRole_59af6f50], result)

    @builtins.property
    def family(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name of a family that this task definition is registered to.

        A family groups multiple versions of a task definition.

        :default: - Automatically generated name.

        :stability: experimental
        '''
        result = self._values.get("family")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def proxy_configuration(self) -> typing.Optional["ProxyConfiguration"]:
        '''(experimental) The configuration details for the App Mesh proxy.

        :default: - No proxy configuration.

        :stability: experimental
        '''
        result = self._values.get("proxy_configuration")
        return typing.cast(typing.Optional["ProxyConfiguration"], result)

    @builtins.property
    def task_role(self) -> typing.Optional[_IRole_59af6f50]:
        '''(experimental) The name of the IAM role that grants containers in the task permission to call AWS APIs on your behalf.

        :default: - A task role is automatically created for you.

        :stability: experimental
        '''
        result = self._values.get("task_role")
        return typing.cast(typing.Optional[_IRole_59af6f50], result)

    @builtins.property
    def volumes(self) -> typing.Optional[typing.List["Volume"]]:
        '''(experimental) The list of volume definitions for the task.

        For more information, see
        `Task Definition Parameter Volumes <https://docs.aws.amazon.com/AmazonECS/latest/developerguide//task_definition_parameters.html#volumes>`_.

        :default: - No volumes are passed to the Docker daemon on a container instance.

        :stability: experimental
        '''
        result = self._values.get("volumes")
        return typing.cast(typing.Optional[typing.List["Volume"]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CommonTaskDefinitionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_ecs.Compatibility")
class Compatibility(enum.Enum):
    '''(experimental) The task launch type compatibility requirement.

    :stability: experimental
    '''

    EC2 = "EC2"
    '''(experimental) The task should specify the EC2 launch type.

    :stability: experimental
    '''
    FARGATE = "FARGATE"
    '''(experimental) The task should specify the Fargate launch type.

    :stability: experimental
    '''
    EC2_AND_FARGATE = "EC2_AND_FARGATE"
    '''(experimental) The task can specify either the EC2 or Fargate launch types.

    :stability: experimental
    '''


class ContainerDefinition(
    _Construct_e78e779f,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_ecs.ContainerDefinition",
):
    '''(experimental) A container definition is used in a task definition to describe the containers that are launched as part of a task.

    :stability: experimental
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        task_definition: "TaskDefinition",
        image: "ContainerImage",
        command: typing.Optional[typing.Sequence[builtins.str]] = None,
        container_name: typing.Optional[builtins.str] = None,
        cpu: typing.Optional[jsii.Number] = None,
        disable_networking: typing.Optional[builtins.bool] = None,
        dns_search_domains: typing.Optional[typing.Sequence[builtins.str]] = None,
        dns_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        docker_labels: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        docker_security_options: typing.Optional[typing.Sequence[builtins.str]] = None,
        entry_point: typing.Optional[typing.Sequence[builtins.str]] = None,
        environment: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        environment_files: typing.Optional[typing.Sequence["EnvironmentFile"]] = None,
        essential: typing.Optional[builtins.bool] = None,
        extra_hosts: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        gpu_count: typing.Optional[jsii.Number] = None,
        health_check: typing.Optional["HealthCheck"] = None,
        hostname: typing.Optional[builtins.str] = None,
        inference_accelerator_resources: typing.Optional[typing.Sequence[builtins.str]] = None,
        linux_parameters: typing.Optional["LinuxParameters"] = None,
        logging: typing.Optional["LogDriver"] = None,
        memory_limit_mib: typing.Optional[jsii.Number] = None,
        memory_reservation_mib: typing.Optional[jsii.Number] = None,
        port_mappings: typing.Optional[typing.Sequence["PortMapping"]] = None,
        privileged: typing.Optional[builtins.bool] = None,
        readonly_root_filesystem: typing.Optional[builtins.bool] = None,
        secrets: typing.Optional[typing.Mapping[builtins.str, "Secret"]] = None,
        start_timeout: typing.Optional[_Duration_070aa057] = None,
        stop_timeout: typing.Optional[_Duration_070aa057] = None,
        user: typing.Optional[builtins.str] = None,
        working_directory: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) Constructs a new instance of the ContainerDefinition class.

        :param scope: -
        :param id: -
        :param task_definition: (experimental) The name of the task definition that includes this container definition. [disable-awslint:ref-via-interface]
        :param image: (experimental) The image used to start a container. This string is passed directly to the Docker daemon. Images in the Docker Hub registry are available by default. Other repositories are specified with either repository-url/image:tag or repository-url/image@digest. TODO: Update these to specify using classes of IContainerImage
        :param command: (experimental) The command that is passed to the container. If you provide a shell command as a single string, you have to quote command-line arguments. Default: - CMD value built into container image.
        :param container_name: (experimental) The name of the container. Default: - id of node associated with ContainerDefinition.
        :param cpu: (experimental) The minimum number of CPU units to reserve for the container. Default: - No minimum CPU units reserved.
        :param disable_networking: (experimental) Specifies whether networking is disabled within the container. When this parameter is true, networking is disabled within the container. Default: false
        :param dns_search_domains: (experimental) A list of DNS search domains that are presented to the container. Default: - No search domains.
        :param dns_servers: (experimental) A list of DNS servers that are presented to the container. Default: - Default DNS servers.
        :param docker_labels: (experimental) A key/value map of labels to add to the container. Default: - No labels.
        :param docker_security_options: (experimental) A list of strings to provide custom labels for SELinux and AppArmor multi-level security systems. Default: - No security labels.
        :param entry_point: (experimental) The ENTRYPOINT value to pass to the container. Default: - Entry point configured in container.
        :param environment: (experimental) The environment variables to pass to the container. Default: - No environment variables.
        :param environment_files: (experimental) The environment files to pass to the container. Default: - No environment files.
        :param essential: (experimental) Specifies whether the container is marked essential. If the essential parameter of a container is marked as true, and that container fails or stops for any reason, all other containers that are part of the task are stopped. If the essential parameter of a container is marked as false, then its failure does not affect the rest of the containers in a task. All tasks must have at least one essential container. If this parameter is omitted, a container is assumed to be essential. Default: true
        :param extra_hosts: (experimental) A list of hostnames and IP address mappings to append to the /etc/hosts file on the container. Default: - No extra hosts.
        :param gpu_count: (experimental) The number of GPUs assigned to the container. Default: - No GPUs assigned.
        :param health_check: (experimental) The health check command and associated configuration parameters for the container. Default: - Health check configuration from container.
        :param hostname: (experimental) The hostname to use for your container. Default: - Automatic hostname.
        :param inference_accelerator_resources: (experimental) The inference accelerators referenced by the container. Default: - No inference accelerators assigned.
        :param linux_parameters: (experimental) Linux-specific modifications that are applied to the container, such as Linux kernel capabilities. For more information see `KernelCapabilities <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_KernelCapabilities.html>`_. Default: - No Linux parameters.
        :param logging: (experimental) The log configuration specification for the container. Default: - Containers use the same logging driver that the Docker daemon uses.
        :param memory_limit_mib: (experimental) The amount (in MiB) of memory to present to the container. If your container attempts to exceed the allocated memory, the container is terminated. At least one of memoryLimitMiB and memoryReservationMiB is required for non-Fargate services. Default: - No memory limit.
        :param memory_reservation_mib: (experimental) The soft limit (in MiB) of memory to reserve for the container. When system memory is under heavy contention, Docker attempts to keep the container memory to this soft limit. However, your container can consume more memory when it needs to, up to either the hard limit specified with the memory parameter (if applicable), or all of the available memory on the container instance, whichever comes first. At least one of memoryLimitMiB and memoryReservationMiB is required for non-Fargate services. Default: - No memory reserved.
        :param port_mappings: (experimental) The port mappings to add to the container definition. Default: - No ports are mapped.
        :param privileged: (experimental) Specifies whether the container is marked as privileged. When this parameter is true, the container is given elevated privileges on the host container instance (similar to the root user). Default: false
        :param readonly_root_filesystem: (experimental) When this parameter is true, the container is given read-only access to its root file system. Default: false
        :param secrets: (experimental) The secret environment variables to pass to the container. Default: - No secret environment variables.
        :param start_timeout: (experimental) Time duration (in seconds) to wait before giving up on resolving dependencies for a container. Default: - none
        :param stop_timeout: (experimental) Time duration (in seconds) to wait before the container is forcefully killed if it doesn't exit normally on its own. Default: - none
        :param user: (experimental) The user name to use inside the container. Default: root
        :param working_directory: (experimental) The working directory in which to run commands inside the container. Default: /

        :stability: experimental
        '''
        props = ContainerDefinitionProps(
            task_definition=task_definition,
            image=image,
            command=command,
            container_name=container_name,
            cpu=cpu,
            disable_networking=disable_networking,
            dns_search_domains=dns_search_domains,
            dns_servers=dns_servers,
            docker_labels=docker_labels,
            docker_security_options=docker_security_options,
            entry_point=entry_point,
            environment=environment,
            environment_files=environment_files,
            essential=essential,
            extra_hosts=extra_hosts,
            gpu_count=gpu_count,
            health_check=health_check,
            hostname=hostname,
            inference_accelerator_resources=inference_accelerator_resources,
            linux_parameters=linux_parameters,
            logging=logging,
            memory_limit_mib=memory_limit_mib,
            memory_reservation_mib=memory_reservation_mib,
            port_mappings=port_mappings,
            privileged=privileged,
            readonly_root_filesystem=readonly_root_filesystem,
            secrets=secrets,
            start_timeout=start_timeout,
            stop_timeout=stop_timeout,
            user=user,
            working_directory=working_directory,
        )

        jsii.create(ContainerDefinition, self, [scope, id, props])

    @jsii.member(jsii_name="addContainerDependencies")
    def add_container_dependencies(
        self,
        *container_dependencies: "ContainerDependency",
    ) -> None:
        '''(experimental) This method adds one or more container dependencies to the container.

        :param container_dependencies: -

        :stability: experimental
        '''
        return typing.cast(None, jsii.invoke(self, "addContainerDependencies", [*container_dependencies]))

    @jsii.member(jsii_name="addInferenceAcceleratorResource")
    def add_inference_accelerator_resource(
        self,
        *inference_accelerator_resources: builtins.str,
    ) -> None:
        '''(experimental) This method adds one or more resources to the container.

        :param inference_accelerator_resources: -

        :stability: experimental
        '''
        return typing.cast(None, jsii.invoke(self, "addInferenceAcceleratorResource", [*inference_accelerator_resources]))

    @jsii.member(jsii_name="addLink")
    def add_link(
        self,
        container: "ContainerDefinition",
        alias: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) This method adds a link which allows containers to communicate with each other without the need for port mappings.

        This parameter is only supported if the task definition is using the bridge network mode.
        Warning: The --link flag is a legacy feature of Docker. It may eventually be removed.

        :param container: -
        :param alias: -

        :stability: experimental
        '''
        return typing.cast(None, jsii.invoke(self, "addLink", [container, alias]))

    @jsii.member(jsii_name="addMountPoints")
    def add_mount_points(self, *mount_points: "MountPoint") -> None:
        '''(experimental) This method adds one or more mount points for data volumes to the container.

        :param mount_points: -

        :stability: experimental
        '''
        return typing.cast(None, jsii.invoke(self, "addMountPoints", [*mount_points]))

    @jsii.member(jsii_name="addPortMappings")
    def add_port_mappings(self, *port_mappings: "PortMapping") -> None:
        '''(experimental) This method adds one or more port mappings to the container.

        :param port_mappings: -

        :stability: experimental
        '''
        return typing.cast(None, jsii.invoke(self, "addPortMappings", [*port_mappings]))

    @jsii.member(jsii_name="addScratch")
    def add_scratch(
        self,
        *,
        container_path: builtins.str,
        name: builtins.str,
        read_only: builtins.bool,
        source_path: builtins.str,
    ) -> None:
        '''(experimental) This method mounts temporary disk space to the container.

        This adds the correct container mountPoint and task definition volume.

        :param container_path: (experimental) The path on the container to mount the scratch volume at.
        :param name: (experimental) The name of the scratch volume to mount. Must be a volume name referenced in the name parameter of task definition volume.
        :param read_only: (experimental) Specifies whether to give the container read-only access to the scratch volume. If this value is true, the container has read-only access to the scratch volume. If this value is false, then the container can write to the scratch volume.
        :param source_path: 

        :stability: experimental
        '''
        scratch = ScratchSpace(
            container_path=container_path,
            name=name,
            read_only=read_only,
            source_path=source_path,
        )

        return typing.cast(None, jsii.invoke(self, "addScratch", [scratch]))

    @jsii.member(jsii_name="addToExecutionPolicy")
    def add_to_execution_policy(self, statement: _PolicyStatement_296fe8a3) -> None:
        '''(experimental) This method adds the specified statement to the IAM task execution policy in the task definition.

        :param statement: -

        :stability: experimental
        '''
        return typing.cast(None, jsii.invoke(self, "addToExecutionPolicy", [statement]))

    @jsii.member(jsii_name="addUlimits")
    def add_ulimits(self, *ulimits: "Ulimit") -> None:
        '''(experimental) This method adds one or more ulimits to the container.

        :param ulimits: -

        :stability: experimental
        '''
        return typing.cast(None, jsii.invoke(self, "addUlimits", [*ulimits]))

    @jsii.member(jsii_name="addVolumesFrom")
    def add_volumes_from(self, *volumes_from: "VolumeFrom") -> None:
        '''(experimental) This method adds one or more volumes to the container.

        :param volumes_from: -

        :stability: experimental
        '''
        return typing.cast(None, jsii.invoke(self, "addVolumesFrom", [*volumes_from]))

    @jsii.member(jsii_name="findPortMapping")
    def find_port_mapping(
        self,
        container_port: jsii.Number,
        protocol: "Protocol",
    ) -> typing.Optional["PortMapping"]:
        '''(experimental) Returns the host port for the requested container port if it exists.

        :param container_port: -
        :param protocol: -

        :stability: experimental
        '''
        return typing.cast(typing.Optional["PortMapping"], jsii.invoke(self, "findPortMapping", [container_port, protocol]))

    @jsii.member(jsii_name="renderContainerDefinition")
    def render_container_definition(
        self,
        _task_definition: typing.Optional["TaskDefinition"] = None,
    ) -> CfnTaskDefinition.ContainerDefinitionProperty:
        '''(experimental) Render this container definition to a CloudFormation object.

        :param _task_definition: [disable-awslint:ref-via-interface] (unused but kept to avoid breaking change).

        :stability: experimental
        '''
        return typing.cast(CfnTaskDefinition.ContainerDefinitionProperty, jsii.invoke(self, "renderContainerDefinition", [_task_definition]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="containerDependencies")
    def container_dependencies(self) -> typing.List["ContainerDependency"]:
        '''(experimental) An array dependencies defined for container startup and shutdown.

        :stability: experimental
        '''
        return typing.cast(typing.List["ContainerDependency"], jsii.get(self, "containerDependencies"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="containerName")
    def container_name(self) -> builtins.str:
        '''(experimental) The name of this container.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "containerName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="containerPort")
    def container_port(self) -> jsii.Number:
        '''(experimental) The port the container will listen on.

        :stability: experimental
        '''
        return typing.cast(jsii.Number, jsii.get(self, "containerPort"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="essential")
    def essential(self) -> builtins.bool:
        '''(experimental) Specifies whether the container will be marked essential.

        If the essential parameter of a container is marked as true, and that container
        fails or stops for any reason, all other containers that are part of the task are
        stopped. If the essential parameter of a container is marked as false, then its
        failure does not affect the rest of the containers in a task.

        If this parameter is omitted, a container is assumed to be essential.

        :stability: experimental
        '''
        return typing.cast(builtins.bool, jsii.get(self, "essential"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ingressPort")
    def ingress_port(self) -> jsii.Number:
        '''(experimental) The inbound rules associated with the security group the task or service will use.

        This property is only used for tasks that use the awsvpc network mode.

        :stability: experimental
        '''
        return typing.cast(jsii.Number, jsii.get(self, "ingressPort"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="memoryLimitSpecified")
    def memory_limit_specified(self) -> builtins.bool:
        '''(experimental) Whether there was at least one memory limit specified in this definition.

        :stability: experimental
        '''
        return typing.cast(builtins.bool, jsii.get(self, "memoryLimitSpecified"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="mountPoints")
    def mount_points(self) -> typing.List["MountPoint"]:
        '''(experimental) The mount points for data volumes in your container.

        :stability: experimental
        '''
        return typing.cast(typing.List["MountPoint"], jsii.get(self, "mountPoints"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="portMappings")
    def port_mappings(self) -> typing.List["PortMapping"]:
        '''(experimental) The list of port mappings for the container.

        Port mappings allow containers to access ports
        on the host container instance to send or receive traffic.

        :stability: experimental
        '''
        return typing.cast(typing.List["PortMapping"], jsii.get(self, "portMappings"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskDefinition")
    def task_definition(self) -> "TaskDefinition":
        '''(experimental) The name of the task definition that includes this container definition.

        :stability: experimental
        '''
        return typing.cast("TaskDefinition", jsii.get(self, "taskDefinition"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ulimits")
    def ulimits(self) -> typing.List["Ulimit"]:
        '''(experimental) An array of ulimits to set in the container.

        :stability: experimental
        '''
        return typing.cast(typing.List["Ulimit"], jsii.get(self, "ulimits"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="volumesFrom")
    def volumes_from(self) -> typing.List["VolumeFrom"]:
        '''(experimental) The data volumes to mount from another container in the same task definition.

        :stability: experimental
        '''
        return typing.cast(typing.List["VolumeFrom"], jsii.get(self, "volumesFrom"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="environmentFiles")
    def environment_files(
        self,
    ) -> typing.Optional[typing.List["EnvironmentFileConfig"]]:
        '''(experimental) The environment files for this container.

        :stability: experimental
        '''
        return typing.cast(typing.Optional[typing.List["EnvironmentFileConfig"]], jsii.get(self, "environmentFiles"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="linuxParameters")
    def linux_parameters(self) -> typing.Optional["LinuxParameters"]:
        '''(experimental) The Linux-specific modifications that are applied to the container, such as Linux kernel capabilities.

        :stability: experimental
        '''
        return typing.cast(typing.Optional["LinuxParameters"], jsii.get(self, "linuxParameters"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="logDriverConfig")
    def log_driver_config(self) -> typing.Optional["LogDriverConfig"]:
        '''(experimental) The log configuration specification for the container.

        :stability: experimental
        '''
        return typing.cast(typing.Optional["LogDriverConfig"], jsii.get(self, "logDriverConfig"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="referencesSecretJsonField")
    def references_secret_json_field(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether this container definition references a specific JSON field of a secret stored in Secrets Manager.

        :stability: experimental
        '''
        return typing.cast(typing.Optional[builtins.bool], jsii.get(self, "referencesSecretJsonField"))


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.ContainerDefinitionOptions",
    jsii_struct_bases=[],
    name_mapping={
        "image": "image",
        "command": "command",
        "container_name": "containerName",
        "cpu": "cpu",
        "disable_networking": "disableNetworking",
        "dns_search_domains": "dnsSearchDomains",
        "dns_servers": "dnsServers",
        "docker_labels": "dockerLabels",
        "docker_security_options": "dockerSecurityOptions",
        "entry_point": "entryPoint",
        "environment": "environment",
        "environment_files": "environmentFiles",
        "essential": "essential",
        "extra_hosts": "extraHosts",
        "gpu_count": "gpuCount",
        "health_check": "healthCheck",
        "hostname": "hostname",
        "inference_accelerator_resources": "inferenceAcceleratorResources",
        "linux_parameters": "linuxParameters",
        "logging": "logging",
        "memory_limit_mib": "memoryLimitMiB",
        "memory_reservation_mib": "memoryReservationMiB",
        "port_mappings": "portMappings",
        "privileged": "privileged",
        "readonly_root_filesystem": "readonlyRootFilesystem",
        "secrets": "secrets",
        "start_timeout": "startTimeout",
        "stop_timeout": "stopTimeout",
        "user": "user",
        "working_directory": "workingDirectory",
    },
)
class ContainerDefinitionOptions:
    def __init__(
        self,
        *,
        image: "ContainerImage",
        command: typing.Optional[typing.Sequence[builtins.str]] = None,
        container_name: typing.Optional[builtins.str] = None,
        cpu: typing.Optional[jsii.Number] = None,
        disable_networking: typing.Optional[builtins.bool] = None,
        dns_search_domains: typing.Optional[typing.Sequence[builtins.str]] = None,
        dns_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        docker_labels: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        docker_security_options: typing.Optional[typing.Sequence[builtins.str]] = None,
        entry_point: typing.Optional[typing.Sequence[builtins.str]] = None,
        environment: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        environment_files: typing.Optional[typing.Sequence["EnvironmentFile"]] = None,
        essential: typing.Optional[builtins.bool] = None,
        extra_hosts: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        gpu_count: typing.Optional[jsii.Number] = None,
        health_check: typing.Optional["HealthCheck"] = None,
        hostname: typing.Optional[builtins.str] = None,
        inference_accelerator_resources: typing.Optional[typing.Sequence[builtins.str]] = None,
        linux_parameters: typing.Optional["LinuxParameters"] = None,
        logging: typing.Optional["LogDriver"] = None,
        memory_limit_mib: typing.Optional[jsii.Number] = None,
        memory_reservation_mib: typing.Optional[jsii.Number] = None,
        port_mappings: typing.Optional[typing.Sequence["PortMapping"]] = None,
        privileged: typing.Optional[builtins.bool] = None,
        readonly_root_filesystem: typing.Optional[builtins.bool] = None,
        secrets: typing.Optional[typing.Mapping[builtins.str, "Secret"]] = None,
        start_timeout: typing.Optional[_Duration_070aa057] = None,
        stop_timeout: typing.Optional[_Duration_070aa057] = None,
        user: typing.Optional[builtins.str] = None,
        working_directory: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param image: (experimental) The image used to start a container. This string is passed directly to the Docker daemon. Images in the Docker Hub registry are available by default. Other repositories are specified with either repository-url/image:tag or repository-url/image@digest. TODO: Update these to specify using classes of IContainerImage
        :param command: (experimental) The command that is passed to the container. If you provide a shell command as a single string, you have to quote command-line arguments. Default: - CMD value built into container image.
        :param container_name: (experimental) The name of the container. Default: - id of node associated with ContainerDefinition.
        :param cpu: (experimental) The minimum number of CPU units to reserve for the container. Default: - No minimum CPU units reserved.
        :param disable_networking: (experimental) Specifies whether networking is disabled within the container. When this parameter is true, networking is disabled within the container. Default: false
        :param dns_search_domains: (experimental) A list of DNS search domains that are presented to the container. Default: - No search domains.
        :param dns_servers: (experimental) A list of DNS servers that are presented to the container. Default: - Default DNS servers.
        :param docker_labels: (experimental) A key/value map of labels to add to the container. Default: - No labels.
        :param docker_security_options: (experimental) A list of strings to provide custom labels for SELinux and AppArmor multi-level security systems. Default: - No security labels.
        :param entry_point: (experimental) The ENTRYPOINT value to pass to the container. Default: - Entry point configured in container.
        :param environment: (experimental) The environment variables to pass to the container. Default: - No environment variables.
        :param environment_files: (experimental) The environment files to pass to the container. Default: - No environment files.
        :param essential: (experimental) Specifies whether the container is marked essential. If the essential parameter of a container is marked as true, and that container fails or stops for any reason, all other containers that are part of the task are stopped. If the essential parameter of a container is marked as false, then its failure does not affect the rest of the containers in a task. All tasks must have at least one essential container. If this parameter is omitted, a container is assumed to be essential. Default: true
        :param extra_hosts: (experimental) A list of hostnames and IP address mappings to append to the /etc/hosts file on the container. Default: - No extra hosts.
        :param gpu_count: (experimental) The number of GPUs assigned to the container. Default: - No GPUs assigned.
        :param health_check: (experimental) The health check command and associated configuration parameters for the container. Default: - Health check configuration from container.
        :param hostname: (experimental) The hostname to use for your container. Default: - Automatic hostname.
        :param inference_accelerator_resources: (experimental) The inference accelerators referenced by the container. Default: - No inference accelerators assigned.
        :param linux_parameters: (experimental) Linux-specific modifications that are applied to the container, such as Linux kernel capabilities. For more information see `KernelCapabilities <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_KernelCapabilities.html>`_. Default: - No Linux parameters.
        :param logging: (experimental) The log configuration specification for the container. Default: - Containers use the same logging driver that the Docker daemon uses.
        :param memory_limit_mib: (experimental) The amount (in MiB) of memory to present to the container. If your container attempts to exceed the allocated memory, the container is terminated. At least one of memoryLimitMiB and memoryReservationMiB is required for non-Fargate services. Default: - No memory limit.
        :param memory_reservation_mib: (experimental) The soft limit (in MiB) of memory to reserve for the container. When system memory is under heavy contention, Docker attempts to keep the container memory to this soft limit. However, your container can consume more memory when it needs to, up to either the hard limit specified with the memory parameter (if applicable), or all of the available memory on the container instance, whichever comes first. At least one of memoryLimitMiB and memoryReservationMiB is required for non-Fargate services. Default: - No memory reserved.
        :param port_mappings: (experimental) The port mappings to add to the container definition. Default: - No ports are mapped.
        :param privileged: (experimental) Specifies whether the container is marked as privileged. When this parameter is true, the container is given elevated privileges on the host container instance (similar to the root user). Default: false
        :param readonly_root_filesystem: (experimental) When this parameter is true, the container is given read-only access to its root file system. Default: false
        :param secrets: (experimental) The secret environment variables to pass to the container. Default: - No secret environment variables.
        :param start_timeout: (experimental) Time duration (in seconds) to wait before giving up on resolving dependencies for a container. Default: - none
        :param stop_timeout: (experimental) Time duration (in seconds) to wait before the container is forcefully killed if it doesn't exit normally on its own. Default: - none
        :param user: (experimental) The user name to use inside the container. Default: root
        :param working_directory: (experimental) The working directory in which to run commands inside the container. Default: /

        :stability: experimental
        '''
        if isinstance(health_check, dict):
            health_check = HealthCheck(**health_check)
        self._values: typing.Dict[str, typing.Any] = {
            "image": image,
        }
        if command is not None:
            self._values["command"] = command
        if container_name is not None:
            self._values["container_name"] = container_name
        if cpu is not None:
            self._values["cpu"] = cpu
        if disable_networking is not None:
            self._values["disable_networking"] = disable_networking
        if dns_search_domains is not None:
            self._values["dns_search_domains"] = dns_search_domains
        if dns_servers is not None:
            self._values["dns_servers"] = dns_servers
        if docker_labels is not None:
            self._values["docker_labels"] = docker_labels
        if docker_security_options is not None:
            self._values["docker_security_options"] = docker_security_options
        if entry_point is not None:
            self._values["entry_point"] = entry_point
        if environment is not None:
            self._values["environment"] = environment
        if environment_files is not None:
            self._values["environment_files"] = environment_files
        if essential is not None:
            self._values["essential"] = essential
        if extra_hosts is not None:
            self._values["extra_hosts"] = extra_hosts
        if gpu_count is not None:
            self._values["gpu_count"] = gpu_count
        if health_check is not None:
            self._values["health_check"] = health_check
        if hostname is not None:
            self._values["hostname"] = hostname
        if inference_accelerator_resources is not None:
            self._values["inference_accelerator_resources"] = inference_accelerator_resources
        if linux_parameters is not None:
            self._values["linux_parameters"] = linux_parameters
        if logging is not None:
            self._values["logging"] = logging
        if memory_limit_mib is not None:
            self._values["memory_limit_mib"] = memory_limit_mib
        if memory_reservation_mib is not None:
            self._values["memory_reservation_mib"] = memory_reservation_mib
        if port_mappings is not None:
            self._values["port_mappings"] = port_mappings
        if privileged is not None:
            self._values["privileged"] = privileged
        if readonly_root_filesystem is not None:
            self._values["readonly_root_filesystem"] = readonly_root_filesystem
        if secrets is not None:
            self._values["secrets"] = secrets
        if start_timeout is not None:
            self._values["start_timeout"] = start_timeout
        if stop_timeout is not None:
            self._values["stop_timeout"] = stop_timeout
        if user is not None:
            self._values["user"] = user
        if working_directory is not None:
            self._values["working_directory"] = working_directory

    @builtins.property
    def image(self) -> "ContainerImage":
        '''(experimental) The image used to start a container.

        This string is passed directly to the Docker daemon.
        Images in the Docker Hub registry are available by default.
        Other repositories are specified with either repository-url/image:tag or repository-url/image@digest.
        TODO: Update these to specify using classes of IContainerImage

        :stability: experimental
        '''
        result = self._values.get("image")
        assert result is not None, "Required property 'image' is missing"
        return typing.cast("ContainerImage", result)

    @builtins.property
    def command(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) The command that is passed to the container.

        If you provide a shell command as a single string, you have to quote command-line arguments.

        :default: - CMD value built into container image.

        :stability: experimental
        '''
        result = self._values.get("command")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def container_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name of the container.

        :default: - id of node associated with ContainerDefinition.

        :stability: experimental
        '''
        result = self._values.get("container_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def cpu(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The minimum number of CPU units to reserve for the container.

        :default: - No minimum CPU units reserved.

        :stability: experimental
        '''
        result = self._values.get("cpu")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def disable_networking(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specifies whether networking is disabled within the container.

        When this parameter is true, networking is disabled within the container.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("disable_networking")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def dns_search_domains(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) A list of DNS search domains that are presented to the container.

        :default: - No search domains.

        :stability: experimental
        '''
        result = self._values.get("dns_search_domains")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def dns_servers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) A list of DNS servers that are presented to the container.

        :default: - Default DNS servers.

        :stability: experimental
        '''
        result = self._values.get("dns_servers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def docker_labels(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''(experimental) A key/value map of labels to add to the container.

        :default: - No labels.

        :stability: experimental
        '''
        result = self._values.get("docker_labels")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def docker_security_options(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) A list of strings to provide custom labels for SELinux and AppArmor multi-level security systems.

        :default: - No security labels.

        :stability: experimental
        '''
        result = self._values.get("docker_security_options")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def entry_point(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) The ENTRYPOINT value to pass to the container.

        :default: - Entry point configured in container.

        :see: https://docs.docker.com/engine/reference/builder/#entrypoint
        :stability: experimental
        '''
        result = self._values.get("entry_point")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def environment(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''(experimental) The environment variables to pass to the container.

        :default: - No environment variables.

        :stability: experimental
        '''
        result = self._values.get("environment")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def environment_files(self) -> typing.Optional[typing.List["EnvironmentFile"]]:
        '''(experimental) The environment files to pass to the container.

        :default: - No environment files.

        :see: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/taskdef-envfiles.html
        :stability: experimental
        '''
        result = self._values.get("environment_files")
        return typing.cast(typing.Optional[typing.List["EnvironmentFile"]], result)

    @builtins.property
    def essential(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specifies whether the container is marked essential.

        If the essential parameter of a container is marked as true, and that container fails
        or stops for any reason, all other containers that are part of the task are stopped.
        If the essential parameter of a container is marked as false, then its failure does not
        affect the rest of the containers in a task. All tasks must have at least one essential container.

        If this parameter is omitted, a container is assumed to be essential.

        :default: true

        :stability: experimental
        '''
        result = self._values.get("essential")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def extra_hosts(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''(experimental) A list of hostnames and IP address mappings to append to the /etc/hosts file on the container.

        :default: - No extra hosts.

        :stability: experimental
        '''
        result = self._values.get("extra_hosts")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def gpu_count(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The number of GPUs assigned to the container.

        :default: - No GPUs assigned.

        :stability: experimental
        '''
        result = self._values.get("gpu_count")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def health_check(self) -> typing.Optional["HealthCheck"]:
        '''(experimental) The health check command and associated configuration parameters for the container.

        :default: - Health check configuration from container.

        :stability: experimental
        '''
        result = self._values.get("health_check")
        return typing.cast(typing.Optional["HealthCheck"], result)

    @builtins.property
    def hostname(self) -> typing.Optional[builtins.str]:
        '''(experimental) The hostname to use for your container.

        :default: - Automatic hostname.

        :stability: experimental
        '''
        result = self._values.get("hostname")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def inference_accelerator_resources(
        self,
    ) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) The inference accelerators referenced by the container.

        :default: - No inference accelerators assigned.

        :stability: experimental
        '''
        result = self._values.get("inference_accelerator_resources")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def linux_parameters(self) -> typing.Optional["LinuxParameters"]:
        '''(experimental) Linux-specific modifications that are applied to the container, such as Linux kernel capabilities.

        For more information see `KernelCapabilities <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_KernelCapabilities.html>`_.

        :default: - No Linux parameters.

        :stability: experimental
        '''
        result = self._values.get("linux_parameters")
        return typing.cast(typing.Optional["LinuxParameters"], result)

    @builtins.property
    def logging(self) -> typing.Optional["LogDriver"]:
        '''(experimental) The log configuration specification for the container.

        :default: - Containers use the same logging driver that the Docker daemon uses.

        :stability: experimental
        '''
        result = self._values.get("logging")
        return typing.cast(typing.Optional["LogDriver"], result)

    @builtins.property
    def memory_limit_mib(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The amount (in MiB) of memory to present to the container.

        If your container attempts to exceed the allocated memory, the container
        is terminated.

        At least one of memoryLimitMiB and memoryReservationMiB is required for non-Fargate services.

        :default: - No memory limit.

        :stability: experimental
        '''
        result = self._values.get("memory_limit_mib")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def memory_reservation_mib(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The soft limit (in MiB) of memory to reserve for the container.

        When system memory is under heavy contention, Docker attempts to keep the
        container memory to this soft limit. However, your container can consume more
        memory when it needs to, up to either the hard limit specified with the memory
        parameter (if applicable), or all of the available memory on the container
        instance, whichever comes first.

        At least one of memoryLimitMiB and memoryReservationMiB is required for non-Fargate services.

        :default: - No memory reserved.

        :stability: experimental
        '''
        result = self._values.get("memory_reservation_mib")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def port_mappings(self) -> typing.Optional[typing.List["PortMapping"]]:
        '''(experimental) The port mappings to add to the container definition.

        :default: - No ports are mapped.

        :stability: experimental
        '''
        result = self._values.get("port_mappings")
        return typing.cast(typing.Optional[typing.List["PortMapping"]], result)

    @builtins.property
    def privileged(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specifies whether the container is marked as privileged.

        When this parameter is true, the container is given elevated privileges on the host container instance (similar to the root user).

        :default: false

        :stability: experimental
        '''
        result = self._values.get("privileged")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def readonly_root_filesystem(self) -> typing.Optional[builtins.bool]:
        '''(experimental) When this parameter is true, the container is given read-only access to its root file system.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("readonly_root_filesystem")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def secrets(self) -> typing.Optional[typing.Mapping[builtins.str, "Secret"]]:
        '''(experimental) The secret environment variables to pass to the container.

        :default: - No secret environment variables.

        :stability: experimental
        '''
        result = self._values.get("secrets")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, "Secret"]], result)

    @builtins.property
    def start_timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Time duration (in seconds) to wait before giving up on resolving dependencies for a container.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("start_timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def stop_timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Time duration (in seconds) to wait before the container is forcefully killed if it doesn't exit normally on its own.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("stop_timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def user(self) -> typing.Optional[builtins.str]:
        '''(experimental) The user name to use inside the container.

        :default: root

        :stability: experimental
        '''
        result = self._values.get("user")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def working_directory(self) -> typing.Optional[builtins.str]:
        '''(experimental) The working directory in which to run commands inside the container.

        :default: /

        :stability: experimental
        '''
        result = self._values.get("working_directory")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ContainerDefinitionOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.ContainerDefinitionProps",
    jsii_struct_bases=[ContainerDefinitionOptions],
    name_mapping={
        "image": "image",
        "command": "command",
        "container_name": "containerName",
        "cpu": "cpu",
        "disable_networking": "disableNetworking",
        "dns_search_domains": "dnsSearchDomains",
        "dns_servers": "dnsServers",
        "docker_labels": "dockerLabels",
        "docker_security_options": "dockerSecurityOptions",
        "entry_point": "entryPoint",
        "environment": "environment",
        "environment_files": "environmentFiles",
        "essential": "essential",
        "extra_hosts": "extraHosts",
        "gpu_count": "gpuCount",
        "health_check": "healthCheck",
        "hostname": "hostname",
        "inference_accelerator_resources": "inferenceAcceleratorResources",
        "linux_parameters": "linuxParameters",
        "logging": "logging",
        "memory_limit_mib": "memoryLimitMiB",
        "memory_reservation_mib": "memoryReservationMiB",
        "port_mappings": "portMappings",
        "privileged": "privileged",
        "readonly_root_filesystem": "readonlyRootFilesystem",
        "secrets": "secrets",
        "start_timeout": "startTimeout",
        "stop_timeout": "stopTimeout",
        "user": "user",
        "working_directory": "workingDirectory",
        "task_definition": "taskDefinition",
    },
)
class ContainerDefinitionProps(ContainerDefinitionOptions):
    def __init__(
        self,
        *,
        image: "ContainerImage",
        command: typing.Optional[typing.Sequence[builtins.str]] = None,
        container_name: typing.Optional[builtins.str] = None,
        cpu: typing.Optional[jsii.Number] = None,
        disable_networking: typing.Optional[builtins.bool] = None,
        dns_search_domains: typing.Optional[typing.Sequence[builtins.str]] = None,
        dns_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        docker_labels: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        docker_security_options: typing.Optional[typing.Sequence[builtins.str]] = None,
        entry_point: typing.Optional[typing.Sequence[builtins.str]] = None,
        environment: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        environment_files: typing.Optional[typing.Sequence["EnvironmentFile"]] = None,
        essential: typing.Optional[builtins.bool] = None,
        extra_hosts: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        gpu_count: typing.Optional[jsii.Number] = None,
        health_check: typing.Optional["HealthCheck"] = None,
        hostname: typing.Optional[builtins.str] = None,
        inference_accelerator_resources: typing.Optional[typing.Sequence[builtins.str]] = None,
        linux_parameters: typing.Optional["LinuxParameters"] = None,
        logging: typing.Optional["LogDriver"] = None,
        memory_limit_mib: typing.Optional[jsii.Number] = None,
        memory_reservation_mib: typing.Optional[jsii.Number] = None,
        port_mappings: typing.Optional[typing.Sequence["PortMapping"]] = None,
        privileged: typing.Optional[builtins.bool] = None,
        readonly_root_filesystem: typing.Optional[builtins.bool] = None,
        secrets: typing.Optional[typing.Mapping[builtins.str, "Secret"]] = None,
        start_timeout: typing.Optional[_Duration_070aa057] = None,
        stop_timeout: typing.Optional[_Duration_070aa057] = None,
        user: typing.Optional[builtins.str] = None,
        working_directory: typing.Optional[builtins.str] = None,
        task_definition: "TaskDefinition",
    ) -> None:
        '''(experimental) The properties in a container definition.

        :param image: (experimental) The image used to start a container. This string is passed directly to the Docker daemon. Images in the Docker Hub registry are available by default. Other repositories are specified with either repository-url/image:tag or repository-url/image@digest. TODO: Update these to specify using classes of IContainerImage
        :param command: (experimental) The command that is passed to the container. If you provide a shell command as a single string, you have to quote command-line arguments. Default: - CMD value built into container image.
        :param container_name: (experimental) The name of the container. Default: - id of node associated with ContainerDefinition.
        :param cpu: (experimental) The minimum number of CPU units to reserve for the container. Default: - No minimum CPU units reserved.
        :param disable_networking: (experimental) Specifies whether networking is disabled within the container. When this parameter is true, networking is disabled within the container. Default: false
        :param dns_search_domains: (experimental) A list of DNS search domains that are presented to the container. Default: - No search domains.
        :param dns_servers: (experimental) A list of DNS servers that are presented to the container. Default: - Default DNS servers.
        :param docker_labels: (experimental) A key/value map of labels to add to the container. Default: - No labels.
        :param docker_security_options: (experimental) A list of strings to provide custom labels for SELinux and AppArmor multi-level security systems. Default: - No security labels.
        :param entry_point: (experimental) The ENTRYPOINT value to pass to the container. Default: - Entry point configured in container.
        :param environment: (experimental) The environment variables to pass to the container. Default: - No environment variables.
        :param environment_files: (experimental) The environment files to pass to the container. Default: - No environment files.
        :param essential: (experimental) Specifies whether the container is marked essential. If the essential parameter of a container is marked as true, and that container fails or stops for any reason, all other containers that are part of the task are stopped. If the essential parameter of a container is marked as false, then its failure does not affect the rest of the containers in a task. All tasks must have at least one essential container. If this parameter is omitted, a container is assumed to be essential. Default: true
        :param extra_hosts: (experimental) A list of hostnames and IP address mappings to append to the /etc/hosts file on the container. Default: - No extra hosts.
        :param gpu_count: (experimental) The number of GPUs assigned to the container. Default: - No GPUs assigned.
        :param health_check: (experimental) The health check command and associated configuration parameters for the container. Default: - Health check configuration from container.
        :param hostname: (experimental) The hostname to use for your container. Default: - Automatic hostname.
        :param inference_accelerator_resources: (experimental) The inference accelerators referenced by the container. Default: - No inference accelerators assigned.
        :param linux_parameters: (experimental) Linux-specific modifications that are applied to the container, such as Linux kernel capabilities. For more information see `KernelCapabilities <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_KernelCapabilities.html>`_. Default: - No Linux parameters.
        :param logging: (experimental) The log configuration specification for the container. Default: - Containers use the same logging driver that the Docker daemon uses.
        :param memory_limit_mib: (experimental) The amount (in MiB) of memory to present to the container. If your container attempts to exceed the allocated memory, the container is terminated. At least one of memoryLimitMiB and memoryReservationMiB is required for non-Fargate services. Default: - No memory limit.
        :param memory_reservation_mib: (experimental) The soft limit (in MiB) of memory to reserve for the container. When system memory is under heavy contention, Docker attempts to keep the container memory to this soft limit. However, your container can consume more memory when it needs to, up to either the hard limit specified with the memory parameter (if applicable), or all of the available memory on the container instance, whichever comes first. At least one of memoryLimitMiB and memoryReservationMiB is required for non-Fargate services. Default: - No memory reserved.
        :param port_mappings: (experimental) The port mappings to add to the container definition. Default: - No ports are mapped.
        :param privileged: (experimental) Specifies whether the container is marked as privileged. When this parameter is true, the container is given elevated privileges on the host container instance (similar to the root user). Default: false
        :param readonly_root_filesystem: (experimental) When this parameter is true, the container is given read-only access to its root file system. Default: false
        :param secrets: (experimental) The secret environment variables to pass to the container. Default: - No secret environment variables.
        :param start_timeout: (experimental) Time duration (in seconds) to wait before giving up on resolving dependencies for a container. Default: - none
        :param stop_timeout: (experimental) Time duration (in seconds) to wait before the container is forcefully killed if it doesn't exit normally on its own. Default: - none
        :param user: (experimental) The user name to use inside the container. Default: root
        :param working_directory: (experimental) The working directory in which to run commands inside the container. Default: /
        :param task_definition: (experimental) The name of the task definition that includes this container definition. [disable-awslint:ref-via-interface]

        :stability: experimental
        '''
        if isinstance(health_check, dict):
            health_check = HealthCheck(**health_check)
        self._values: typing.Dict[str, typing.Any] = {
            "image": image,
            "task_definition": task_definition,
        }
        if command is not None:
            self._values["command"] = command
        if container_name is not None:
            self._values["container_name"] = container_name
        if cpu is not None:
            self._values["cpu"] = cpu
        if disable_networking is not None:
            self._values["disable_networking"] = disable_networking
        if dns_search_domains is not None:
            self._values["dns_search_domains"] = dns_search_domains
        if dns_servers is not None:
            self._values["dns_servers"] = dns_servers
        if docker_labels is not None:
            self._values["docker_labels"] = docker_labels
        if docker_security_options is not None:
            self._values["docker_security_options"] = docker_security_options
        if entry_point is not None:
            self._values["entry_point"] = entry_point
        if environment is not None:
            self._values["environment"] = environment
        if environment_files is not None:
            self._values["environment_files"] = environment_files
        if essential is not None:
            self._values["essential"] = essential
        if extra_hosts is not None:
            self._values["extra_hosts"] = extra_hosts
        if gpu_count is not None:
            self._values["gpu_count"] = gpu_count
        if health_check is not None:
            self._values["health_check"] = health_check
        if hostname is not None:
            self._values["hostname"] = hostname
        if inference_accelerator_resources is not None:
            self._values["inference_accelerator_resources"] = inference_accelerator_resources
        if linux_parameters is not None:
            self._values["linux_parameters"] = linux_parameters
        if logging is not None:
            self._values["logging"] = logging
        if memory_limit_mib is not None:
            self._values["memory_limit_mib"] = memory_limit_mib
        if memory_reservation_mib is not None:
            self._values["memory_reservation_mib"] = memory_reservation_mib
        if port_mappings is not None:
            self._values["port_mappings"] = port_mappings
        if privileged is not None:
            self._values["privileged"] = privileged
        if readonly_root_filesystem is not None:
            self._values["readonly_root_filesystem"] = readonly_root_filesystem
        if secrets is not None:
            self._values["secrets"] = secrets
        if start_timeout is not None:
            self._values["start_timeout"] = start_timeout
        if stop_timeout is not None:
            self._values["stop_timeout"] = stop_timeout
        if user is not None:
            self._values["user"] = user
        if working_directory is not None:
            self._values["working_directory"] = working_directory

    @builtins.property
    def image(self) -> "ContainerImage":
        '''(experimental) The image used to start a container.

        This string is passed directly to the Docker daemon.
        Images in the Docker Hub registry are available by default.
        Other repositories are specified with either repository-url/image:tag or repository-url/image@digest.
        TODO: Update these to specify using classes of IContainerImage

        :stability: experimental
        '''
        result = self._values.get("image")
        assert result is not None, "Required property 'image' is missing"
        return typing.cast("ContainerImage", result)

    @builtins.property
    def command(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) The command that is passed to the container.

        If you provide a shell command as a single string, you have to quote command-line arguments.

        :default: - CMD value built into container image.

        :stability: experimental
        '''
        result = self._values.get("command")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def container_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name of the container.

        :default: - id of node associated with ContainerDefinition.

        :stability: experimental
        '''
        result = self._values.get("container_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def cpu(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The minimum number of CPU units to reserve for the container.

        :default: - No minimum CPU units reserved.

        :stability: experimental
        '''
        result = self._values.get("cpu")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def disable_networking(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specifies whether networking is disabled within the container.

        When this parameter is true, networking is disabled within the container.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("disable_networking")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def dns_search_domains(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) A list of DNS search domains that are presented to the container.

        :default: - No search domains.

        :stability: experimental
        '''
        result = self._values.get("dns_search_domains")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def dns_servers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) A list of DNS servers that are presented to the container.

        :default: - Default DNS servers.

        :stability: experimental
        '''
        result = self._values.get("dns_servers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def docker_labels(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''(experimental) A key/value map of labels to add to the container.

        :default: - No labels.

        :stability: experimental
        '''
        result = self._values.get("docker_labels")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def docker_security_options(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) A list of strings to provide custom labels for SELinux and AppArmor multi-level security systems.

        :default: - No security labels.

        :stability: experimental
        '''
        result = self._values.get("docker_security_options")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def entry_point(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) The ENTRYPOINT value to pass to the container.

        :default: - Entry point configured in container.

        :see: https://docs.docker.com/engine/reference/builder/#entrypoint
        :stability: experimental
        '''
        result = self._values.get("entry_point")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def environment(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''(experimental) The environment variables to pass to the container.

        :default: - No environment variables.

        :stability: experimental
        '''
        result = self._values.get("environment")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def environment_files(self) -> typing.Optional[typing.List["EnvironmentFile"]]:
        '''(experimental) The environment files to pass to the container.

        :default: - No environment files.

        :see: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/taskdef-envfiles.html
        :stability: experimental
        '''
        result = self._values.get("environment_files")
        return typing.cast(typing.Optional[typing.List["EnvironmentFile"]], result)

    @builtins.property
    def essential(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specifies whether the container is marked essential.

        If the essential parameter of a container is marked as true, and that container fails
        or stops for any reason, all other containers that are part of the task are stopped.
        If the essential parameter of a container is marked as false, then its failure does not
        affect the rest of the containers in a task. All tasks must have at least one essential container.

        If this parameter is omitted, a container is assumed to be essential.

        :default: true

        :stability: experimental
        '''
        result = self._values.get("essential")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def extra_hosts(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''(experimental) A list of hostnames and IP address mappings to append to the /etc/hosts file on the container.

        :default: - No extra hosts.

        :stability: experimental
        '''
        result = self._values.get("extra_hosts")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def gpu_count(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The number of GPUs assigned to the container.

        :default: - No GPUs assigned.

        :stability: experimental
        '''
        result = self._values.get("gpu_count")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def health_check(self) -> typing.Optional["HealthCheck"]:
        '''(experimental) The health check command and associated configuration parameters for the container.

        :default: - Health check configuration from container.

        :stability: experimental
        '''
        result = self._values.get("health_check")
        return typing.cast(typing.Optional["HealthCheck"], result)

    @builtins.property
    def hostname(self) -> typing.Optional[builtins.str]:
        '''(experimental) The hostname to use for your container.

        :default: - Automatic hostname.

        :stability: experimental
        '''
        result = self._values.get("hostname")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def inference_accelerator_resources(
        self,
    ) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) The inference accelerators referenced by the container.

        :default: - No inference accelerators assigned.

        :stability: experimental
        '''
        result = self._values.get("inference_accelerator_resources")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def linux_parameters(self) -> typing.Optional["LinuxParameters"]:
        '''(experimental) Linux-specific modifications that are applied to the container, such as Linux kernel capabilities.

        For more information see `KernelCapabilities <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_KernelCapabilities.html>`_.

        :default: - No Linux parameters.

        :stability: experimental
        '''
        result = self._values.get("linux_parameters")
        return typing.cast(typing.Optional["LinuxParameters"], result)

    @builtins.property
    def logging(self) -> typing.Optional["LogDriver"]:
        '''(experimental) The log configuration specification for the container.

        :default: - Containers use the same logging driver that the Docker daemon uses.

        :stability: experimental
        '''
        result = self._values.get("logging")
        return typing.cast(typing.Optional["LogDriver"], result)

    @builtins.property
    def memory_limit_mib(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The amount (in MiB) of memory to present to the container.

        If your container attempts to exceed the allocated memory, the container
        is terminated.

        At least one of memoryLimitMiB and memoryReservationMiB is required for non-Fargate services.

        :default: - No memory limit.

        :stability: experimental
        '''
        result = self._values.get("memory_limit_mib")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def memory_reservation_mib(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The soft limit (in MiB) of memory to reserve for the container.

        When system memory is under heavy contention, Docker attempts to keep the
        container memory to this soft limit. However, your container can consume more
        memory when it needs to, up to either the hard limit specified with the memory
        parameter (if applicable), or all of the available memory on the container
        instance, whichever comes first.

        At least one of memoryLimitMiB and memoryReservationMiB is required for non-Fargate services.

        :default: - No memory reserved.

        :stability: experimental
        '''
        result = self._values.get("memory_reservation_mib")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def port_mappings(self) -> typing.Optional[typing.List["PortMapping"]]:
        '''(experimental) The port mappings to add to the container definition.

        :default: - No ports are mapped.

        :stability: experimental
        '''
        result = self._values.get("port_mappings")
        return typing.cast(typing.Optional[typing.List["PortMapping"]], result)

    @builtins.property
    def privileged(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specifies whether the container is marked as privileged.

        When this parameter is true, the container is given elevated privileges on the host container instance (similar to the root user).

        :default: false

        :stability: experimental
        '''
        result = self._values.get("privileged")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def readonly_root_filesystem(self) -> typing.Optional[builtins.bool]:
        '''(experimental) When this parameter is true, the container is given read-only access to its root file system.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("readonly_root_filesystem")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def secrets(self) -> typing.Optional[typing.Mapping[builtins.str, "Secret"]]:
        '''(experimental) The secret environment variables to pass to the container.

        :default: - No secret environment variables.

        :stability: experimental
        '''
        result = self._values.get("secrets")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, "Secret"]], result)

    @builtins.property
    def start_timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Time duration (in seconds) to wait before giving up on resolving dependencies for a container.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("start_timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def stop_timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Time duration (in seconds) to wait before the container is forcefully killed if it doesn't exit normally on its own.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("stop_timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def user(self) -> typing.Optional[builtins.str]:
        '''(experimental) The user name to use inside the container.

        :default: root

        :stability: experimental
        '''
        result = self._values.get("user")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def working_directory(self) -> typing.Optional[builtins.str]:
        '''(experimental) The working directory in which to run commands inside the container.

        :default: /

        :stability: experimental
        '''
        result = self._values.get("working_directory")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def task_definition(self) -> "TaskDefinition":
        '''(experimental) The name of the task definition that includes this container definition.

        [disable-awslint:ref-via-interface]

        :stability: experimental
        '''
        result = self._values.get("task_definition")
        assert result is not None, "Required property 'task_definition' is missing"
        return typing.cast("TaskDefinition", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ContainerDefinitionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.ContainerDependency",
    jsii_struct_bases=[],
    name_mapping={"container": "container", "condition": "condition"},
)
class ContainerDependency:
    def __init__(
        self,
        *,
        container: ContainerDefinition,
        condition: typing.Optional["ContainerDependencyCondition"] = None,
    ) -> None:
        '''(experimental) The details of a dependency on another container in the task definition.

        :param container: (experimental) The container to depend on.
        :param condition: (experimental) The state the container needs to be in to satisfy the dependency and proceed with startup. Valid values are ContainerDependencyCondition.START, ContainerDependencyCondition.COMPLETE, ContainerDependencyCondition.SUCCESS and ContainerDependencyCondition.HEALTHY. Default: ContainerDependencyCondition.HEALTHY

        :see: https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDependency.html
        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "container": container,
        }
        if condition is not None:
            self._values["condition"] = condition

    @builtins.property
    def container(self) -> ContainerDefinition:
        '''(experimental) The container to depend on.

        :stability: experimental
        '''
        result = self._values.get("container")
        assert result is not None, "Required property 'container' is missing"
        return typing.cast(ContainerDefinition, result)

    @builtins.property
    def condition(self) -> typing.Optional["ContainerDependencyCondition"]:
        '''(experimental) The state the container needs to be in to satisfy the dependency and proceed with startup.

        Valid values are ContainerDependencyCondition.START, ContainerDependencyCondition.COMPLETE,
        ContainerDependencyCondition.SUCCESS and ContainerDependencyCondition.HEALTHY.

        :default: ContainerDependencyCondition.HEALTHY

        :stability: experimental
        '''
        result = self._values.get("condition")
        return typing.cast(typing.Optional["ContainerDependencyCondition"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ContainerDependency(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_ecs.ContainerDependencyCondition")
class ContainerDependencyCondition(enum.Enum):
    '''
    :stability: experimental
    '''

    START = "START"
    '''(experimental) This condition emulates the behavior of links and volumes today.

    It validates that a dependent container is started before permitting other containers to start.

    :stability: experimental
    '''
    COMPLETE = "COMPLETE"
    '''(experimental) This condition validates that a dependent container runs to completion (exits) before permitting other containers to start.

    This can be useful for nonessential containers that run a script and then exit.

    :stability: experimental
    '''
    SUCCESS = "SUCCESS"
    '''(experimental) This condition is the same as COMPLETE, but it also requires that the container exits with a zero status.

    :stability: experimental
    '''
    HEALTHY = "HEALTHY"
    '''(experimental) This condition validates that the dependent container passes its Docker health check before permitting other containers to start.

    This requires that the dependent container has health checks configured. This condition is confirmed only at task startup.

    :stability: experimental
    '''


class ContainerImage(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="monocdk.aws_ecs.ContainerImage",
):
    '''(experimental) Constructs for types of container images.

    :stability: experimental
    '''

    @builtins.staticmethod
    def __jsii_proxy_class__() -> typing.Type["_ContainerImageProxy"]:
        return _ContainerImageProxy

    def __init__(self) -> None:
        '''
        :stability: experimental
        '''
        jsii.create(ContainerImage, self, [])

    @jsii.member(jsii_name="fromAsset") # type: ignore[misc]
    @builtins.classmethod
    def from_asset(
        cls,
        directory: builtins.str,
        *,
        build_args: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        file: typing.Optional[builtins.str] = None,
        repository_name: typing.Optional[builtins.str] = None,
        target: typing.Optional[builtins.str] = None,
        extra_hash: typing.Optional[builtins.str] = None,
        exclude: typing.Optional[typing.Sequence[builtins.str]] = None,
        follow: typing.Optional[_FollowMode_98b05cc5] = None,
        ignore_mode: typing.Optional[_IgnoreMode_31d8bf46] = None,
        follow_symlinks: typing.Optional[_SymlinkFollowMode_abf4527a] = None,
    ) -> "AssetImage":
        '''(experimental) Reference an image that's constructed directly from sources on disk.

        If you already have a ``DockerImageAsset`` instance, you can use the
        ``ContainerImage.fromDockerImageAsset`` method instead.

        :param directory: The directory containing the Dockerfile.
        :param build_args: (experimental) Build args to pass to the ``docker build`` command. Since Docker build arguments are resolved before deployment, keys and values cannot refer to unresolved tokens (such as ``lambda.functionArn`` or ``queue.queueUrl``). Default: - no build args are passed
        :param file: (experimental) Path to the Dockerfile (relative to the directory). Default: 'Dockerfile'
        :param repository_name: (deprecated) ECR repository name. Specify this property if you need to statically address the image, e.g. from a Kubernetes Pod. Note, this is only the repository name, without the registry and the tag parts. Default: - the default ECR repository for CDK assets
        :param target: (experimental) Docker target to build to. Default: - no target
        :param extra_hash: (deprecated) Extra information to encode into the fingerprint (e.g. build instructions and other inputs). Default: - hash is only based on source content
        :param exclude: (deprecated) Glob patterns to exclude from the copy. Default: nothing is excluded
        :param follow: (deprecated) A strategy for how to handle symlinks. Default: Never
        :param ignore_mode: (deprecated) The ignore behavior to use for exclude patterns. Default: - GLOB for file assets, DOCKER or GLOB for docker assets depending on whether the '
        :param follow_symlinks: (experimental) A strategy for how to handle symlinks. Default: SymlinkFollowMode.NEVER

        :stability: experimental
        '''
        props = AssetImageProps(
            build_args=build_args,
            file=file,
            repository_name=repository_name,
            target=target,
            extra_hash=extra_hash,
            exclude=exclude,
            follow=follow,
            ignore_mode=ignore_mode,
            follow_symlinks=follow_symlinks,
        )

        return typing.cast("AssetImage", jsii.sinvoke(cls, "fromAsset", [directory, props]))

    @jsii.member(jsii_name="fromDockerImageAsset") # type: ignore[misc]
    @builtins.classmethod
    def from_docker_image_asset(
        cls,
        asset: _DockerImageAsset_3ac823f8,
    ) -> "ContainerImage":
        '''(experimental) Use an existing ``DockerImageAsset`` for this container image.

        :param asset: The ``DockerImageAsset`` to use for this container definition.

        :stability: experimental
        '''
        return typing.cast("ContainerImage", jsii.sinvoke(cls, "fromDockerImageAsset", [asset]))

    @jsii.member(jsii_name="fromEcrRepository") # type: ignore[misc]
    @builtins.classmethod
    def from_ecr_repository(
        cls,
        repository: _IRepository_8b4d2894,
        tag: typing.Optional[builtins.str] = None,
    ) -> "EcrImage":
        '''(experimental) Reference an image in an ECR repository.

        :param repository: -
        :param tag: -

        :stability: experimental
        '''
        return typing.cast("EcrImage", jsii.sinvoke(cls, "fromEcrRepository", [repository, tag]))

    @jsii.member(jsii_name="fromRegistry") # type: ignore[misc]
    @builtins.classmethod
    def from_registry(
        cls,
        name: builtins.str,
        *,
        credentials: typing.Optional[_ISecret_22fb8757] = None,
    ) -> "RepositoryImage":
        '''(experimental) Reference an image on DockerHub or another online registry.

        :param name: -
        :param credentials: (experimental) The secret to expose to the container that contains the credentials for the image repository. The supported value is the full ARN of an AWS Secrets Manager secret.

        :stability: experimental
        '''
        props = RepositoryImageProps(credentials=credentials)

        return typing.cast("RepositoryImage", jsii.sinvoke(cls, "fromRegistry", [name, props]))

    @jsii.member(jsii_name="bind") # type: ignore[misc]
    @abc.abstractmethod
    def bind(
        self,
        scope: _Construct_e78e779f,
        container_definition: ContainerDefinition,
    ) -> "ContainerImageConfig":
        '''(experimental) Called when the image is used by a ContainerDefinition.

        :param scope: -
        :param container_definition: -

        :stability: experimental
        '''
        ...


class _ContainerImageProxy(ContainerImage):
    @jsii.member(jsii_name="bind")
    def bind(
        self,
        scope: _Construct_e78e779f,
        container_definition: ContainerDefinition,
    ) -> "ContainerImageConfig":
        '''(experimental) Called when the image is used by a ContainerDefinition.

        :param scope: -
        :param container_definition: -

        :stability: experimental
        '''
        return typing.cast("ContainerImageConfig", jsii.invoke(self, "bind", [scope, container_definition]))


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.ContainerImageConfig",
    jsii_struct_bases=[],
    name_mapping={
        "image_name": "imageName",
        "repository_credentials": "repositoryCredentials",
    },
)
class ContainerImageConfig:
    def __init__(
        self,
        *,
        image_name: builtins.str,
        repository_credentials: typing.Optional[CfnTaskDefinition.RepositoryCredentialsProperty] = None,
    ) -> None:
        '''(experimental) The configuration for creating a container image.

        :param image_name: (experimental) Specifies the name of the container image.
        :param repository_credentials: (experimental) Specifies the credentials used to access the image repository.

        :stability: experimental
        '''
        if isinstance(repository_credentials, dict):
            repository_credentials = CfnTaskDefinition.RepositoryCredentialsProperty(**repository_credentials)
        self._values: typing.Dict[str, typing.Any] = {
            "image_name": image_name,
        }
        if repository_credentials is not None:
            self._values["repository_credentials"] = repository_credentials

    @builtins.property
    def image_name(self) -> builtins.str:
        '''(experimental) Specifies the name of the container image.

        :stability: experimental
        '''
        result = self._values.get("image_name")
        assert result is not None, "Required property 'image_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def repository_credentials(
        self,
    ) -> typing.Optional[CfnTaskDefinition.RepositoryCredentialsProperty]:
        '''(experimental) Specifies the credentials used to access the image repository.

        :stability: experimental
        '''
        result = self._values.get("repository_credentials")
        return typing.cast(typing.Optional[CfnTaskDefinition.RepositoryCredentialsProperty], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ContainerImageConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.CpuUtilizationScalingProps",
    jsii_struct_bases=[_BaseTargetTrackingProps_e4402570],
    name_mapping={
        "disable_scale_in": "disableScaleIn",
        "policy_name": "policyName",
        "scale_in_cooldown": "scaleInCooldown",
        "scale_out_cooldown": "scaleOutCooldown",
        "target_utilization_percent": "targetUtilizationPercent",
    },
)
class CpuUtilizationScalingProps(_BaseTargetTrackingProps_e4402570):
    def __init__(
        self,
        *,
        disable_scale_in: typing.Optional[builtins.bool] = None,
        policy_name: typing.Optional[builtins.str] = None,
        scale_in_cooldown: typing.Optional[_Duration_070aa057] = None,
        scale_out_cooldown: typing.Optional[_Duration_070aa057] = None,
        target_utilization_percent: jsii.Number,
    ) -> None:
        '''(experimental) The properties for enabling scaling based on CPU utilization.

        :param disable_scale_in: (experimental) Indicates whether scale in by the target tracking policy is disabled. If the value is true, scale in is disabled and the target tracking policy won't remove capacity from the scalable resource. Otherwise, scale in is enabled and the target tracking policy can remove capacity from the scalable resource. Default: false
        :param policy_name: (experimental) A name for the scaling policy. Default: - Automatically generated name.
        :param scale_in_cooldown: (experimental) Period after a scale in activity completes before another scale in activity can start. Default: Duration.seconds(300) for the following scalable targets: ECS services, Spot Fleet requests, EMR clusters, AppStream 2.0 fleets, Aurora DB clusters, Amazon SageMaker endpoint variants, Custom resources. For all other scalable targets, the default value is Duration.seconds(0): DynamoDB tables, DynamoDB global secondary indexes, Amazon Comprehend document classification endpoints, Lambda provisioned concurrency
        :param scale_out_cooldown: (experimental) Period after a scale out activity completes before another scale out activity can start. Default: Duration.seconds(300) for the following scalable targets: ECS services, Spot Fleet requests, EMR clusters, AppStream 2.0 fleets, Aurora DB clusters, Amazon SageMaker endpoint variants, Custom resources. For all other scalable targets, the default value is Duration.seconds(0): DynamoDB tables, DynamoDB global secondary indexes, Amazon Comprehend document classification endpoints, Lambda provisioned concurrency
        :param target_utilization_percent: (experimental) The target value for CPU utilization across all tasks in the service.

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "target_utilization_percent": target_utilization_percent,
        }
        if disable_scale_in is not None:
            self._values["disable_scale_in"] = disable_scale_in
        if policy_name is not None:
            self._values["policy_name"] = policy_name
        if scale_in_cooldown is not None:
            self._values["scale_in_cooldown"] = scale_in_cooldown
        if scale_out_cooldown is not None:
            self._values["scale_out_cooldown"] = scale_out_cooldown

    @builtins.property
    def disable_scale_in(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Indicates whether scale in by the target tracking policy is disabled.

        If the value is true, scale in is disabled and the target tracking policy
        won't remove capacity from the scalable resource. Otherwise, scale in is
        enabled and the target tracking policy can remove capacity from the
        scalable resource.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("disable_scale_in")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def policy_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) A name for the scaling policy.

        :default: - Automatically generated name.

        :stability: experimental
        '''
        result = self._values.get("policy_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def scale_in_cooldown(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Period after a scale in activity completes before another scale in activity can start.

        :default:

        Duration.seconds(300) for the following scalable targets: ECS services,
        Spot Fleet requests, EMR clusters, AppStream 2.0 fleets, Aurora DB clusters,
        Amazon SageMaker endpoint variants, Custom resources. For all other scalable
        targets, the default value is Duration.seconds(0): DynamoDB tables, DynamoDB
        global secondary indexes, Amazon Comprehend document classification endpoints,
        Lambda provisioned concurrency

        :stability: experimental
        '''
        result = self._values.get("scale_in_cooldown")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def scale_out_cooldown(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Period after a scale out activity completes before another scale out activity can start.

        :default:

        Duration.seconds(300) for the following scalable targets: ECS services,
        Spot Fleet requests, EMR clusters, AppStream 2.0 fleets, Aurora DB clusters,
        Amazon SageMaker endpoint variants, Custom resources. For all other scalable
        targets, the default value is Duration.seconds(0): DynamoDB tables, DynamoDB
        global secondary indexes, Amazon Comprehend document classification endpoints,
        Lambda provisioned concurrency

        :stability: experimental
        '''
        result = self._values.get("scale_out_cooldown")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def target_utilization_percent(self) -> jsii.Number:
        '''(experimental) The target value for CPU utilization across all tasks in the service.

        :stability: experimental
        '''
        result = self._values.get("target_utilization_percent")
        assert result is not None, "Required property 'target_utilization_percent' is missing"
        return typing.cast(jsii.Number, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CpuUtilizationScalingProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.DeploymentCircuitBreaker",
    jsii_struct_bases=[],
    name_mapping={"rollback": "rollback"},
)
class DeploymentCircuitBreaker:
    def __init__(self, *, rollback: typing.Optional[builtins.bool] = None) -> None:
        '''(experimental) The deployment circuit breaker to use for the service.

        :param rollback: (experimental) Whether to enable rollback on deployment failure. Default: false

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if rollback is not None:
            self._values["rollback"] = rollback

    @builtins.property
    def rollback(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether to enable rollback on deployment failure.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("rollback")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DeploymentCircuitBreaker(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.DeploymentController",
    jsii_struct_bases=[],
    name_mapping={"type": "type"},
)
class DeploymentController:
    def __init__(
        self,
        *,
        type: typing.Optional["DeploymentControllerType"] = None,
    ) -> None:
        '''(experimental) The deployment controller to use for the service.

        :param type: (experimental) The deployment controller type to use. Default: DeploymentControllerType.ECS

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if type is not None:
            self._values["type"] = type

    @builtins.property
    def type(self) -> typing.Optional["DeploymentControllerType"]:
        '''(experimental) The deployment controller type to use.

        :default: DeploymentControllerType.ECS

        :stability: experimental
        '''
        result = self._values.get("type")
        return typing.cast(typing.Optional["DeploymentControllerType"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DeploymentController(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_ecs.DeploymentControllerType")
class DeploymentControllerType(enum.Enum):
    '''(experimental) The deployment controller type to use for the service.

    :stability: experimental
    '''

    ECS = "ECS"
    '''(experimental) The rolling update (ECS) deployment type involves replacing the current running version of the container with the latest version.

    :stability: experimental
    '''
    CODE_DEPLOY = "CODE_DEPLOY"
    '''(experimental) The blue/green (CODE_DEPLOY) deployment type uses the blue/green deployment model powered by AWS CodeDeploy.

    :stability: experimental
    '''
    EXTERNAL = "EXTERNAL"
    '''(experimental) The external (EXTERNAL) deployment type enables you to use any third-party deployment controller.

    :stability: experimental
    '''


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.Device",
    jsii_struct_bases=[],
    name_mapping={
        "host_path": "hostPath",
        "container_path": "containerPath",
        "permissions": "permissions",
    },
)
class Device:
    def __init__(
        self,
        *,
        host_path: builtins.str,
        container_path: typing.Optional[builtins.str] = None,
        permissions: typing.Optional[typing.Sequence["DevicePermission"]] = None,
    ) -> None:
        '''(experimental) A container instance host device.

        :param host_path: (experimental) The path for the device on the host container instance.
        :param container_path: (experimental) The path inside the container at which to expose the host device. Default: Same path as the host
        :param permissions: (experimental) The explicit permissions to provide to the container for the device. By default, the container has permissions for read, write, and mknod for the device. Default: Readonly

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "host_path": host_path,
        }
        if container_path is not None:
            self._values["container_path"] = container_path
        if permissions is not None:
            self._values["permissions"] = permissions

    @builtins.property
    def host_path(self) -> builtins.str:
        '''(experimental) The path for the device on the host container instance.

        :stability: experimental
        '''
        result = self._values.get("host_path")
        assert result is not None, "Required property 'host_path' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def container_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) The path inside the container at which to expose the host device.

        :default: Same path as the host

        :stability: experimental
        '''
        result = self._values.get("container_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def permissions(self) -> typing.Optional[typing.List["DevicePermission"]]:
        '''(experimental) The explicit permissions to provide to the container for the device.

        By default, the container has permissions for read, write, and mknod for the device.

        :default: Readonly

        :stability: experimental
        '''
        result = self._values.get("permissions")
        return typing.cast(typing.Optional[typing.List["DevicePermission"]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "Device(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_ecs.DevicePermission")
class DevicePermission(enum.Enum):
    '''(experimental) Permissions for device access.

    :stability: experimental
    '''

    READ = "READ"
    '''(experimental) Read.

    :stability: experimental
    '''
    WRITE = "WRITE"
    '''(experimental) Write.

    :stability: experimental
    '''
    MKNOD = "MKNOD"
    '''(experimental) Make a node.

    :stability: experimental
    '''


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.DockerVolumeConfiguration",
    jsii_struct_bases=[],
    name_mapping={
        "driver": "driver",
        "scope": "scope",
        "autoprovision": "autoprovision",
        "driver_opts": "driverOpts",
        "labels": "labels",
    },
)
class DockerVolumeConfiguration:
    def __init__(
        self,
        *,
        driver: builtins.str,
        scope: "Scope",
        autoprovision: typing.Optional[builtins.bool] = None,
        driver_opts: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        labels: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
    ) -> None:
        '''(experimental) The configuration for a Docker volume.

        Docker volumes are only supported when you are using the EC2 launch type.

        :param driver: (experimental) The Docker volume driver to use.
        :param scope: (experimental) The scope for the Docker volume that determines its lifecycle.
        :param autoprovision: (experimental) Specifies whether the Docker volume should be created if it does not already exist. If true is specified, the Docker volume will be created for you. Default: false
        :param driver_opts: (experimental) A map of Docker driver-specific options passed through. Default: No options
        :param labels: (experimental) Custom metadata to add to your Docker volume. Default: No labels

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "driver": driver,
            "scope": scope,
        }
        if autoprovision is not None:
            self._values["autoprovision"] = autoprovision
        if driver_opts is not None:
            self._values["driver_opts"] = driver_opts
        if labels is not None:
            self._values["labels"] = labels

    @builtins.property
    def driver(self) -> builtins.str:
        '''(experimental) The Docker volume driver to use.

        :stability: experimental
        '''
        result = self._values.get("driver")
        assert result is not None, "Required property 'driver' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def scope(self) -> "Scope":
        '''(experimental) The scope for the Docker volume that determines its lifecycle.

        :stability: experimental
        '''
        result = self._values.get("scope")
        assert result is not None, "Required property 'scope' is missing"
        return typing.cast("Scope", result)

    @builtins.property
    def autoprovision(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specifies whether the Docker volume should be created if it does not already exist.

        If true is specified, the Docker volume will be created for you.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("autoprovision")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def driver_opts(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''(experimental) A map of Docker driver-specific options passed through.

        :default: No options

        :stability: experimental
        '''
        result = self._values.get("driver_opts")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def labels(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''(experimental) Custom metadata to add to your Docker volume.

        :default: No labels

        :stability: experimental
        '''
        result = self._values.get("labels")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DockerVolumeConfiguration(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.Ec2ServiceAttributes",
    jsii_struct_bases=[],
    name_mapping={
        "cluster": "cluster",
        "service_arn": "serviceArn",
        "service_name": "serviceName",
    },
)
class Ec2ServiceAttributes:
    def __init__(
        self,
        *,
        cluster: "ICluster",
        service_arn: typing.Optional[builtins.str] = None,
        service_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) The properties to import from the service using the EC2 launch type.

        :param cluster: (experimental) The cluster that hosts the service.
        :param service_arn: (experimental) The service ARN. Default: - either this, or {@link serviceName}, is required
        :param service_name: (experimental) The name of the service. Default: - either this, or {@link serviceArn}, is required

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "cluster": cluster,
        }
        if service_arn is not None:
            self._values["service_arn"] = service_arn
        if service_name is not None:
            self._values["service_name"] = service_name

    @builtins.property
    def cluster(self) -> "ICluster":
        '''(experimental) The cluster that hosts the service.

        :stability: experimental
        '''
        result = self._values.get("cluster")
        assert result is not None, "Required property 'cluster' is missing"
        return typing.cast("ICluster", result)

    @builtins.property
    def service_arn(self) -> typing.Optional[builtins.str]:
        '''(experimental) The service ARN.

        :default: - either this, or {@link serviceName}, is required

        :stability: experimental
        '''
        result = self._values.get("service_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def service_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name of the service.

        :default: - either this, or {@link serviceArn}, is required

        :stability: experimental
        '''
        result = self._values.get("service_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "Ec2ServiceAttributes(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.Ec2ServiceProps",
    jsii_struct_bases=[BaseServiceOptions],
    name_mapping={
        "cluster": "cluster",
        "capacity_provider_strategies": "capacityProviderStrategies",
        "circuit_breaker": "circuitBreaker",
        "cloud_map_options": "cloudMapOptions",
        "deployment_controller": "deploymentController",
        "desired_count": "desiredCount",
        "enable_ecs_managed_tags": "enableECSManagedTags",
        "health_check_grace_period": "healthCheckGracePeriod",
        "max_healthy_percent": "maxHealthyPercent",
        "min_healthy_percent": "minHealthyPercent",
        "propagate_tags": "propagateTags",
        "service_name": "serviceName",
        "task_definition": "taskDefinition",
        "assign_public_ip": "assignPublicIp",
        "daemon": "daemon",
        "placement_constraints": "placementConstraints",
        "placement_strategies": "placementStrategies",
        "propagate_task_tags_from": "propagateTaskTagsFrom",
        "security_group": "securityGroup",
        "security_groups": "securityGroups",
        "vpc_subnets": "vpcSubnets",
    },
)
class Ec2ServiceProps(BaseServiceOptions):
    def __init__(
        self,
        *,
        cluster: "ICluster",
        capacity_provider_strategies: typing.Optional[typing.Sequence[CapacityProviderStrategy]] = None,
        circuit_breaker: typing.Optional[DeploymentCircuitBreaker] = None,
        cloud_map_options: typing.Optional[CloudMapOptions] = None,
        deployment_controller: typing.Optional[DeploymentController] = None,
        desired_count: typing.Optional[jsii.Number] = None,
        enable_ecs_managed_tags: typing.Optional[builtins.bool] = None,
        health_check_grace_period: typing.Optional[_Duration_070aa057] = None,
        max_healthy_percent: typing.Optional[jsii.Number] = None,
        min_healthy_percent: typing.Optional[jsii.Number] = None,
        propagate_tags: typing.Optional["PropagatedTagSource"] = None,
        service_name: typing.Optional[builtins.str] = None,
        task_definition: "TaskDefinition",
        assign_public_ip: typing.Optional[builtins.bool] = None,
        daemon: typing.Optional[builtins.bool] = None,
        placement_constraints: typing.Optional[typing.Sequence["PlacementConstraint"]] = None,
        placement_strategies: typing.Optional[typing.Sequence["PlacementStrategy"]] = None,
        propagate_task_tags_from: typing.Optional["PropagatedTagSource"] = None,
        security_group: typing.Optional[_ISecurityGroup_cdbba9d3] = None,
        security_groups: typing.Optional[typing.Sequence[_ISecurityGroup_cdbba9d3]] = None,
        vpc_subnets: typing.Optional[_SubnetSelection_1284e62c] = None,
    ) -> None:
        '''(experimental) The properties for defining a service using the EC2 launch type.

        :param cluster: (experimental) The name of the cluster that hosts the service.
        :param capacity_provider_strategies: (experimental) A list of Capacity Provider strategies used to place a service. Default: - undefined
        :param circuit_breaker: (experimental) Whether to enable the deployment circuit breaker. If this property is defined, circuit breaker will be implicitly enabled. Default: - disabled
        :param cloud_map_options: (experimental) The options for configuring an Amazon ECS service to use service discovery. Default: - AWS Cloud Map service discovery is not enabled.
        :param deployment_controller: (experimental) Specifies which deployment controller to use for the service. For more information, see `Amazon ECS Deployment Types <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-types.html>`_ Default: - Rolling update (ECS)
        :param desired_count: (experimental) The desired number of instantiations of the task definition to keep running on the service. Default: - When creating the service, default is 1; when updating the service, default uses the current task number.
        :param enable_ecs_managed_tags: (experimental) Specifies whether to enable Amazon ECS managed tags for the tasks within the service. For more information, see `Tagging Your Amazon ECS Resources <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html>`_ Default: false
        :param health_check_grace_period: (experimental) The period of time, in seconds, that the Amazon ECS service scheduler ignores unhealthy Elastic Load Balancing target health checks after a task has first started. Default: - defaults to 60 seconds if at least one load balancer is in-use and it is not already set
        :param max_healthy_percent: (experimental) The maximum number of tasks, specified as a percentage of the Amazon ECS service's DesiredCount value, that can run in a service during a deployment. Default: - 100 if daemon, otherwise 200
        :param min_healthy_percent: (experimental) The minimum number of tasks, specified as a percentage of the Amazon ECS service's DesiredCount value, that must continue to run and remain healthy during a deployment. Default: - 0 if daemon, otherwise 50
        :param propagate_tags: (experimental) Specifies whether to propagate the tags from the task definition or the service to the tasks in the service. Valid values are: PropagatedTagSource.SERVICE, PropagatedTagSource.TASK_DEFINITION or PropagatedTagSource.NONE Default: PropagatedTagSource.NONE
        :param service_name: (experimental) The name of the service. Default: - CloudFormation-generated name.
        :param task_definition: (experimental) The task definition to use for tasks in the service. [disable-awslint:ref-via-interface]
        :param assign_public_ip: (experimental) Specifies whether the task's elastic network interface receives a public IP address. If true, each task will receive a public IP address. This property is only used for tasks that use the awsvpc network mode. Default: false
        :param daemon: (experimental) Specifies whether the service will use the daemon scheduling strategy. If true, the service scheduler deploys exactly one task on each container instance in your cluster. When you are using this strategy, do not specify a desired number of tasks orany task placement strategies. Default: false
        :param placement_constraints: (experimental) The placement constraints to use for tasks in the service. For more information, see `Amazon ECS Task Placement Constraints <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-constraints.html>`_. Default: - No constraints.
        :param placement_strategies: (experimental) The placement strategies to use for tasks in the service. For more information, see `Amazon ECS Task Placement Strategies <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-strategies.html>`_. Default: - No strategies.
        :param propagate_task_tags_from: (deprecated) Specifies whether to propagate the tags from the task definition or the service to the tasks in the service. Tags can only be propagated to the tasks within the service during service creation. Default: PropagatedTagSource.NONE
        :param security_group: (deprecated) The security groups to associate with the service. If you do not specify a security group, the default security group for the VPC is used. This property is only used for tasks that use the awsvpc network mode. Default: - A new security group is created.
        :param security_groups: (experimental) The security groups to associate with the service. If you do not specify a security group, the default security group for the VPC is used. This property is only used for tasks that use the awsvpc network mode. Default: - A new security group is created.
        :param vpc_subnets: (experimental) The subnets to associate with the service. This property is only used for tasks that use the awsvpc network mode. Default: - Public subnets if ``assignPublicIp`` is set, otherwise the first available one of Private, Isolated, Public, in that order.

        :stability: experimental
        '''
        if isinstance(circuit_breaker, dict):
            circuit_breaker = DeploymentCircuitBreaker(**circuit_breaker)
        if isinstance(cloud_map_options, dict):
            cloud_map_options = CloudMapOptions(**cloud_map_options)
        if isinstance(deployment_controller, dict):
            deployment_controller = DeploymentController(**deployment_controller)
        if isinstance(vpc_subnets, dict):
            vpc_subnets = _SubnetSelection_1284e62c(**vpc_subnets)
        self._values: typing.Dict[str, typing.Any] = {
            "cluster": cluster,
            "task_definition": task_definition,
        }
        if capacity_provider_strategies is not None:
            self._values["capacity_provider_strategies"] = capacity_provider_strategies
        if circuit_breaker is not None:
            self._values["circuit_breaker"] = circuit_breaker
        if cloud_map_options is not None:
            self._values["cloud_map_options"] = cloud_map_options
        if deployment_controller is not None:
            self._values["deployment_controller"] = deployment_controller
        if desired_count is not None:
            self._values["desired_count"] = desired_count
        if enable_ecs_managed_tags is not None:
            self._values["enable_ecs_managed_tags"] = enable_ecs_managed_tags
        if health_check_grace_period is not None:
            self._values["health_check_grace_period"] = health_check_grace_period
        if max_healthy_percent is not None:
            self._values["max_healthy_percent"] = max_healthy_percent
        if min_healthy_percent is not None:
            self._values["min_healthy_percent"] = min_healthy_percent
        if propagate_tags is not None:
            self._values["propagate_tags"] = propagate_tags
        if service_name is not None:
            self._values["service_name"] = service_name
        if assign_public_ip is not None:
            self._values["assign_public_ip"] = assign_public_ip
        if daemon is not None:
            self._values["daemon"] = daemon
        if placement_constraints is not None:
            self._values["placement_constraints"] = placement_constraints
        if placement_strategies is not None:
            self._values["placement_strategies"] = placement_strategies
        if propagate_task_tags_from is not None:
            self._values["propagate_task_tags_from"] = propagate_task_tags_from
        if security_group is not None:
            self._values["security_group"] = security_group
        if security_groups is not None:
            self._values["security_groups"] = security_groups
        if vpc_subnets is not None:
            self._values["vpc_subnets"] = vpc_subnets

    @builtins.property
    def cluster(self) -> "ICluster":
        '''(experimental) The name of the cluster that hosts the service.

        :stability: experimental
        '''
        result = self._values.get("cluster")
        assert result is not None, "Required property 'cluster' is missing"
        return typing.cast("ICluster", result)

    @builtins.property
    def capacity_provider_strategies(
        self,
    ) -> typing.Optional[typing.List[CapacityProviderStrategy]]:
        '''(experimental) A list of Capacity Provider strategies used to place a service.

        :default: - undefined

        :stability: experimental
        '''
        result = self._values.get("capacity_provider_strategies")
        return typing.cast(typing.Optional[typing.List[CapacityProviderStrategy]], result)

    @builtins.property
    def circuit_breaker(self) -> typing.Optional[DeploymentCircuitBreaker]:
        '''(experimental) Whether to enable the deployment circuit breaker.

        If this property is defined, circuit breaker will be implicitly
        enabled.

        :default: - disabled

        :stability: experimental
        '''
        result = self._values.get("circuit_breaker")
        return typing.cast(typing.Optional[DeploymentCircuitBreaker], result)

    @builtins.property
    def cloud_map_options(self) -> typing.Optional[CloudMapOptions]:
        '''(experimental) The options for configuring an Amazon ECS service to use service discovery.

        :default: - AWS Cloud Map service discovery is not enabled.

        :stability: experimental
        '''
        result = self._values.get("cloud_map_options")
        return typing.cast(typing.Optional[CloudMapOptions], result)

    @builtins.property
    def deployment_controller(self) -> typing.Optional[DeploymentController]:
        '''(experimental) Specifies which deployment controller to use for the service.

        For more information, see
        `Amazon ECS Deployment Types <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-types.html>`_

        :default: - Rolling update (ECS)

        :stability: experimental
        '''
        result = self._values.get("deployment_controller")
        return typing.cast(typing.Optional[DeploymentController], result)

    @builtins.property
    def desired_count(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The desired number of instantiations of the task definition to keep running on the service.

        :default:

        - When creating the service, default is 1; when updating the service, default uses
        the current task number.

        :stability: experimental
        '''
        result = self._values.get("desired_count")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def enable_ecs_managed_tags(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specifies whether to enable Amazon ECS managed tags for the tasks within the service.

        For more information, see
        `Tagging Your Amazon ECS Resources <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html>`_

        :default: false

        :stability: experimental
        '''
        result = self._values.get("enable_ecs_managed_tags")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def health_check_grace_period(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) The period of time, in seconds, that the Amazon ECS service scheduler ignores unhealthy Elastic Load Balancing target health checks after a task has first started.

        :default: - defaults to 60 seconds if at least one load balancer is in-use and it is not already set

        :stability: experimental
        '''
        result = self._values.get("health_check_grace_period")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def max_healthy_percent(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The maximum number of tasks, specified as a percentage of the Amazon ECS service's DesiredCount value, that can run in a service during a deployment.

        :default: - 100 if daemon, otherwise 200

        :stability: experimental
        '''
        result = self._values.get("max_healthy_percent")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def min_healthy_percent(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The minimum number of tasks, specified as a percentage of the Amazon ECS service's DesiredCount value, that must continue to run and remain healthy during a deployment.

        :default: - 0 if daemon, otherwise 50

        :stability: experimental
        '''
        result = self._values.get("min_healthy_percent")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def propagate_tags(self) -> typing.Optional["PropagatedTagSource"]:
        '''(experimental) Specifies whether to propagate the tags from the task definition or the service to the tasks in the service.

        Valid values are: PropagatedTagSource.SERVICE, PropagatedTagSource.TASK_DEFINITION or PropagatedTagSource.NONE

        :default: PropagatedTagSource.NONE

        :stability: experimental
        '''
        result = self._values.get("propagate_tags")
        return typing.cast(typing.Optional["PropagatedTagSource"], result)

    @builtins.property
    def service_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name of the service.

        :default: - CloudFormation-generated name.

        :stability: experimental
        '''
        result = self._values.get("service_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def task_definition(self) -> "TaskDefinition":
        '''(experimental) The task definition to use for tasks in the service.

        [disable-awslint:ref-via-interface]

        :stability: experimental
        '''
        result = self._values.get("task_definition")
        assert result is not None, "Required property 'task_definition' is missing"
        return typing.cast("TaskDefinition", result)

    @builtins.property
    def assign_public_ip(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specifies whether the task's elastic network interface receives a public IP address.

        If true, each task will receive a public IP address.

        This property is only used for tasks that use the awsvpc network mode.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("assign_public_ip")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def daemon(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specifies whether the service will use the daemon scheduling strategy.

        If true, the service scheduler deploys exactly one task on each container instance in your cluster.

        When you are using this strategy, do not specify a desired number of tasks orany task placement strategies.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("daemon")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def placement_constraints(
        self,
    ) -> typing.Optional[typing.List["PlacementConstraint"]]:
        '''(experimental) The placement constraints to use for tasks in the service.

        For more information, see
        `Amazon ECS Task Placement Constraints <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-constraints.html>`_.

        :default: - No constraints.

        :stability: experimental
        '''
        result = self._values.get("placement_constraints")
        return typing.cast(typing.Optional[typing.List["PlacementConstraint"]], result)

    @builtins.property
    def placement_strategies(self) -> typing.Optional[typing.List["PlacementStrategy"]]:
        '''(experimental) The placement strategies to use for tasks in the service.

        For more information, see
        `Amazon ECS Task Placement Strategies <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-strategies.html>`_.

        :default: - No strategies.

        :stability: experimental
        '''
        result = self._values.get("placement_strategies")
        return typing.cast(typing.Optional[typing.List["PlacementStrategy"]], result)

    @builtins.property
    def propagate_task_tags_from(self) -> typing.Optional["PropagatedTagSource"]:
        '''(deprecated) Specifies whether to propagate the tags from the task definition or the service to the tasks in the service.

        Tags can only be propagated to the tasks within the service during service creation.

        :default: PropagatedTagSource.NONE

        :deprecated: Use ``propagateTags`` instead.

        :stability: deprecated
        '''
        result = self._values.get("propagate_task_tags_from")
        return typing.cast(typing.Optional["PropagatedTagSource"], result)

    @builtins.property
    def security_group(self) -> typing.Optional[_ISecurityGroup_cdbba9d3]:
        '''(deprecated) The security groups to associate with the service.

        If you do not specify a security group, the default security group for the VPC is used.

        This property is only used for tasks that use the awsvpc network mode.

        :default: - A new security group is created.

        :deprecated: use securityGroups instead.

        :stability: deprecated
        '''
        result = self._values.get("security_group")
        return typing.cast(typing.Optional[_ISecurityGroup_cdbba9d3], result)

    @builtins.property
    def security_groups(self) -> typing.Optional[typing.List[_ISecurityGroup_cdbba9d3]]:
        '''(experimental) The security groups to associate with the service.

        If you do not specify a security group, the default security group for the VPC is used.

        This property is only used for tasks that use the awsvpc network mode.

        :default: - A new security group is created.

        :stability: experimental
        '''
        result = self._values.get("security_groups")
        return typing.cast(typing.Optional[typing.List[_ISecurityGroup_cdbba9d3]], result)

    @builtins.property
    def vpc_subnets(self) -> typing.Optional[_SubnetSelection_1284e62c]:
        '''(experimental) The subnets to associate with the service.

        This property is only used for tasks that use the awsvpc network mode.

        :default: - Public subnets if ``assignPublicIp`` is set, otherwise the first available one of Private, Isolated, Public, in that order.

        :stability: experimental
        '''
        result = self._values.get("vpc_subnets")
        return typing.cast(typing.Optional[_SubnetSelection_1284e62c], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "Ec2ServiceProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.Ec2TaskDefinitionAttributes",
    jsii_struct_bases=[CommonTaskDefinitionAttributes],
    name_mapping={
        "task_definition_arn": "taskDefinitionArn",
        "network_mode": "networkMode",
        "task_role": "taskRole",
    },
)
class Ec2TaskDefinitionAttributes(CommonTaskDefinitionAttributes):
    def __init__(
        self,
        *,
        task_definition_arn: builtins.str,
        network_mode: typing.Optional["NetworkMode"] = None,
        task_role: typing.Optional[_IRole_59af6f50] = None,
    ) -> None:
        '''(experimental) Attributes used to import an existing EC2 task definition.

        :param task_definition_arn: (experimental) The arn of the task definition.
        :param network_mode: (experimental) The networking mode to use for the containers in the task. Default: Network mode cannot be provided to the imported task.
        :param task_role: (experimental) The name of the IAM role that grants containers in the task permission to call AWS APIs on your behalf. Default: Permissions cannot be granted to the imported task.

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "task_definition_arn": task_definition_arn,
        }
        if network_mode is not None:
            self._values["network_mode"] = network_mode
        if task_role is not None:
            self._values["task_role"] = task_role

    @builtins.property
    def task_definition_arn(self) -> builtins.str:
        '''(experimental) The arn of the task definition.

        :stability: experimental
        '''
        result = self._values.get("task_definition_arn")
        assert result is not None, "Required property 'task_definition_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def network_mode(self) -> typing.Optional["NetworkMode"]:
        '''(experimental) The networking mode to use for the containers in the task.

        :default: Network mode cannot be provided to the imported task.

        :stability: experimental
        '''
        result = self._values.get("network_mode")
        return typing.cast(typing.Optional["NetworkMode"], result)

    @builtins.property
    def task_role(self) -> typing.Optional[_IRole_59af6f50]:
        '''(experimental) The name of the IAM role that grants containers in the task permission to call AWS APIs on your behalf.

        :default: Permissions cannot be granted to the imported task.

        :stability: experimental
        '''
        result = self._values.get("task_role")
        return typing.cast(typing.Optional[_IRole_59af6f50], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "Ec2TaskDefinitionAttributes(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.Ec2TaskDefinitionProps",
    jsii_struct_bases=[CommonTaskDefinitionProps],
    name_mapping={
        "execution_role": "executionRole",
        "family": "family",
        "proxy_configuration": "proxyConfiguration",
        "task_role": "taskRole",
        "volumes": "volumes",
        "inference_accelerators": "inferenceAccelerators",
        "ipc_mode": "ipcMode",
        "network_mode": "networkMode",
        "pid_mode": "pidMode",
        "placement_constraints": "placementConstraints",
    },
)
class Ec2TaskDefinitionProps(CommonTaskDefinitionProps):
    def __init__(
        self,
        *,
        execution_role: typing.Optional[_IRole_59af6f50] = None,
        family: typing.Optional[builtins.str] = None,
        proxy_configuration: typing.Optional["ProxyConfiguration"] = None,
        task_role: typing.Optional[_IRole_59af6f50] = None,
        volumes: typing.Optional[typing.Sequence["Volume"]] = None,
        inference_accelerators: typing.Optional[typing.Sequence["InferenceAccelerator"]] = None,
        ipc_mode: typing.Optional["IpcMode"] = None,
        network_mode: typing.Optional["NetworkMode"] = None,
        pid_mode: typing.Optional["PidMode"] = None,
        placement_constraints: typing.Optional[typing.Sequence["PlacementConstraint"]] = None,
    ) -> None:
        '''(experimental) The properties for a task definition run on an EC2 cluster.

        :param execution_role: (experimental) The name of the IAM task execution role that grants the ECS agent to call AWS APIs on your behalf. The role will be used to retrieve container images from ECR and create CloudWatch log groups. Default: - An execution role will be automatically created if you use ECR images in your task definition.
        :param family: (experimental) The name of a family that this task definition is registered to. A family groups multiple versions of a task definition. Default: - Automatically generated name.
        :param proxy_configuration: (experimental) The configuration details for the App Mesh proxy. Default: - No proxy configuration.
        :param task_role: (experimental) The name of the IAM role that grants containers in the task permission to call AWS APIs on your behalf. Default: - A task role is automatically created for you.
        :param volumes: (experimental) The list of volume definitions for the task. For more information, see `Task Definition Parameter Volumes <https://docs.aws.amazon.com/AmazonECS/latest/developerguide//task_definition_parameters.html#volumes>`_. Default: - No volumes are passed to the Docker daemon on a container instance.
        :param inference_accelerators: (experimental) The inference accelerators to use for the containers in the task. Not supported in Fargate. Default: - No inference accelerators.
        :param ipc_mode: (experimental) The IPC resource namespace to use for the containers in the task. Not supported in Fargate and Windows containers. Default: - IpcMode used by the task is not specified
        :param network_mode: (experimental) The Docker networking mode to use for the containers in the task. The valid values are none, bridge, awsvpc, and host. Default: - NetworkMode.Bridge for EC2 tasks, AwsVpc for Fargate tasks.
        :param pid_mode: (experimental) The process namespace to use for the containers in the task. Not supported in Fargate and Windows containers. Default: - PidMode used by the task is not specified
        :param placement_constraints: (experimental) An array of placement constraint objects to use for the task. You can specify a maximum of 10 constraints per task (this limit includes constraints in the task definition and those specified at run time). Default: - No placement constraints.

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if execution_role is not None:
            self._values["execution_role"] = execution_role
        if family is not None:
            self._values["family"] = family
        if proxy_configuration is not None:
            self._values["proxy_configuration"] = proxy_configuration
        if task_role is not None:
            self._values["task_role"] = task_role
        if volumes is not None:
            self._values["volumes"] = volumes
        if inference_accelerators is not None:
            self._values["inference_accelerators"] = inference_accelerators
        if ipc_mode is not None:
            self._values["ipc_mode"] = ipc_mode
        if network_mode is not None:
            self._values["network_mode"] = network_mode
        if pid_mode is not None:
            self._values["pid_mode"] = pid_mode
        if placement_constraints is not None:
            self._values["placement_constraints"] = placement_constraints

    @builtins.property
    def execution_role(self) -> typing.Optional[_IRole_59af6f50]:
        '''(experimental) The name of the IAM task execution role that grants the ECS agent to call AWS APIs on your behalf.

        The role will be used to retrieve container images from ECR and create CloudWatch log groups.

        :default: - An execution role will be automatically created if you use ECR images in your task definition.

        :stability: experimental
        '''
        result = self._values.get("execution_role")
        return typing.cast(typing.Optional[_IRole_59af6f50], result)

    @builtins.property
    def family(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name of a family that this task definition is registered to.

        A family groups multiple versions of a task definition.

        :default: - Automatically generated name.

        :stability: experimental
        '''
        result = self._values.get("family")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def proxy_configuration(self) -> typing.Optional["ProxyConfiguration"]:
        '''(experimental) The configuration details for the App Mesh proxy.

        :default: - No proxy configuration.

        :stability: experimental
        '''
        result = self._values.get("proxy_configuration")
        return typing.cast(typing.Optional["ProxyConfiguration"], result)

    @builtins.property
    def task_role(self) -> typing.Optional[_IRole_59af6f50]:
        '''(experimental) The name of the IAM role that grants containers in the task permission to call AWS APIs on your behalf.

        :default: - A task role is automatically created for you.

        :stability: experimental
        '''
        result = self._values.get("task_role")
        return typing.cast(typing.Optional[_IRole_59af6f50], result)

    @builtins.property
    def volumes(self) -> typing.Optional[typing.List["Volume"]]:
        '''(experimental) The list of volume definitions for the task.

        For more information, see
        `Task Definition Parameter Volumes <https://docs.aws.amazon.com/AmazonECS/latest/developerguide//task_definition_parameters.html#volumes>`_.

        :default: - No volumes are passed to the Docker daemon on a container instance.

        :stability: experimental
        '''
        result = self._values.get("volumes")
        return typing.cast(typing.Optional[typing.List["Volume"]], result)

    @builtins.property
    def inference_accelerators(
        self,
    ) -> typing.Optional[typing.List["InferenceAccelerator"]]:
        '''(experimental) The inference accelerators to use for the containers in the task.

        Not supported in Fargate.

        :default: - No inference accelerators.

        :stability: experimental
        '''
        result = self._values.get("inference_accelerators")
        return typing.cast(typing.Optional[typing.List["InferenceAccelerator"]], result)

    @builtins.property
    def ipc_mode(self) -> typing.Optional["IpcMode"]:
        '''(experimental) The IPC resource namespace to use for the containers in the task.

        Not supported in Fargate and Windows containers.

        :default: - IpcMode used by the task is not specified

        :stability: experimental
        '''
        result = self._values.get("ipc_mode")
        return typing.cast(typing.Optional["IpcMode"], result)

    @builtins.property
    def network_mode(self) -> typing.Optional["NetworkMode"]:
        '''(experimental) The Docker networking mode to use for the containers in the task.

        The valid values are none, bridge, awsvpc, and host.

        :default: - NetworkMode.Bridge for EC2 tasks, AwsVpc for Fargate tasks.

        :stability: experimental
        '''
        result = self._values.get("network_mode")
        return typing.cast(typing.Optional["NetworkMode"], result)

    @builtins.property
    def pid_mode(self) -> typing.Optional["PidMode"]:
        '''(experimental) The process namespace to use for the containers in the task.

        Not supported in Fargate and Windows containers.

        :default: - PidMode used by the task is not specified

        :stability: experimental
        '''
        result = self._values.get("pid_mode")
        return typing.cast(typing.Optional["PidMode"], result)

    @builtins.property
    def placement_constraints(
        self,
    ) -> typing.Optional[typing.List["PlacementConstraint"]]:
        '''(experimental) An array of placement constraint objects to use for the task.

        You can
        specify a maximum of 10 constraints per task (this limit includes
        constraints in the task definition and those specified at run time).

        :default: - No placement constraints.

        :stability: experimental
        '''
        result = self._values.get("placement_constraints")
        return typing.cast(typing.Optional[typing.List["PlacementConstraint"]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "Ec2TaskDefinitionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class EcrImage(
    ContainerImage,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_ecs.EcrImage",
):
    '''(experimental) An image from an Amazon ECR repository.

    :stability: experimental
    '''

    def __init__(
        self,
        repository: _IRepository_8b4d2894,
        tag_or_digest: builtins.str,
    ) -> None:
        '''(experimental) Constructs a new instance of the EcrImage class.

        :param repository: -
        :param tag_or_digest: -

        :stability: experimental
        '''
        jsii.create(EcrImage, self, [repository, tag_or_digest])

    @jsii.member(jsii_name="bind")
    def bind(
        self,
        _scope: _Construct_e78e779f,
        container_definition: ContainerDefinition,
    ) -> ContainerImageConfig:
        '''(experimental) Called when the image is used by a ContainerDefinition.

        :param _scope: -
        :param container_definition: -

        :stability: experimental
        '''
        return typing.cast(ContainerImageConfig, jsii.invoke(self, "bind", [_scope, container_definition]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="imageName")
    def image_name(self) -> builtins.str:
        '''(experimental) The image name. Images in Amazon ECR repositories can be specified by either using the full registry/repository:tag or registry/repository@digest.

        For example, 012345678910.dkr.ecr..amazonaws.com/:latest or
        012345678910.dkr.ecr..amazonaws.com/@sha256:94afd1f2e64d908bc90dbca0035a5b567EXAMPLE.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "imageName"))


@jsii.implements(_IMachineImage_45d3238d)
class EcsOptimizedAmi(
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_ecs.EcsOptimizedAmi",
):
    '''(deprecated) Construct a Linux or Windows machine image from the latest ECS Optimized AMI published in SSM.

    :deprecated: see {@link EcsOptimizedImage#amazonLinux}, {@link EcsOptimizedImage#amazonLinux} and {@link EcsOptimizedImage#windows}

    :stability: deprecated
    '''

    def __init__(
        self,
        *,
        generation: typing.Optional[_AmazonLinuxGeneration_d9650a6d] = None,
        hardware_type: typing.Optional[AmiHardwareType] = None,
        windows_version: typing.Optional["WindowsOptimizedVersion"] = None,
    ) -> None:
        '''(deprecated) Constructs a new instance of the EcsOptimizedAmi class.

        :param generation: (deprecated) The Amazon Linux generation to use. Default: AmazonLinuxGeneration.AmazonLinux2
        :param hardware_type: (deprecated) The ECS-optimized AMI variant to use. Default: AmiHardwareType.Standard
        :param windows_version: (deprecated) The Windows Server version to use. Default: none, uses Linux generation

        :stability: deprecated
        '''
        props = EcsOptimizedAmiProps(
            generation=generation,
            hardware_type=hardware_type,
            windows_version=windows_version,
        )

        jsii.create(EcsOptimizedAmi, self, [props])

    @jsii.member(jsii_name="getImage")
    def get_image(self, scope: _Construct_e78e779f) -> _MachineImageConfig_963798fe:
        '''(deprecated) Return the correct image.

        :param scope: -

        :stability: deprecated
        '''
        return typing.cast(_MachineImageConfig_963798fe, jsii.invoke(self, "getImage", [scope]))


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.EcsOptimizedAmiProps",
    jsii_struct_bases=[],
    name_mapping={
        "generation": "generation",
        "hardware_type": "hardwareType",
        "windows_version": "windowsVersion",
    },
)
class EcsOptimizedAmiProps:
    def __init__(
        self,
        *,
        generation: typing.Optional[_AmazonLinuxGeneration_d9650a6d] = None,
        hardware_type: typing.Optional[AmiHardwareType] = None,
        windows_version: typing.Optional["WindowsOptimizedVersion"] = None,
    ) -> None:
        '''(deprecated) The properties that define which ECS-optimized AMI is used.

        :param generation: (deprecated) The Amazon Linux generation to use. Default: AmazonLinuxGeneration.AmazonLinux2
        :param hardware_type: (deprecated) The ECS-optimized AMI variant to use. Default: AmiHardwareType.Standard
        :param windows_version: (deprecated) The Windows Server version to use. Default: none, uses Linux generation

        :deprecated: see {@link EcsOptimizedImage}

        :stability: deprecated
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if generation is not None:
            self._values["generation"] = generation
        if hardware_type is not None:
            self._values["hardware_type"] = hardware_type
        if windows_version is not None:
            self._values["windows_version"] = windows_version

    @builtins.property
    def generation(self) -> typing.Optional[_AmazonLinuxGeneration_d9650a6d]:
        '''(deprecated) The Amazon Linux generation to use.

        :default: AmazonLinuxGeneration.AmazonLinux2

        :stability: deprecated
        '''
        result = self._values.get("generation")
        return typing.cast(typing.Optional[_AmazonLinuxGeneration_d9650a6d], result)

    @builtins.property
    def hardware_type(self) -> typing.Optional[AmiHardwareType]:
        '''(deprecated) The ECS-optimized AMI variant to use.

        :default: AmiHardwareType.Standard

        :stability: deprecated
        '''
        result = self._values.get("hardware_type")
        return typing.cast(typing.Optional[AmiHardwareType], result)

    @builtins.property
    def windows_version(self) -> typing.Optional["WindowsOptimizedVersion"]:
        '''(deprecated) The Windows Server version to use.

        :default: none, uses Linux generation

        :stability: deprecated
        '''
        result = self._values.get("windows_version")
        return typing.cast(typing.Optional["WindowsOptimizedVersion"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EcsOptimizedAmiProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IMachineImage_45d3238d)
class EcsOptimizedImage(
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_ecs.EcsOptimizedImage",
):
    '''(experimental) Construct a Linux or Windows machine image from the latest ECS Optimized AMI published in SSM.

    :stability: experimental
    '''

    @jsii.member(jsii_name="amazonLinux") # type: ignore[misc]
    @builtins.classmethod
    def amazon_linux(cls) -> "EcsOptimizedImage":
        '''(experimental) Construct an Amazon Linux AMI image from the latest ECS Optimized AMI published in SSM.

        :stability: experimental
        '''
        return typing.cast("EcsOptimizedImage", jsii.sinvoke(cls, "amazonLinux", []))

    @jsii.member(jsii_name="amazonLinux2") # type: ignore[misc]
    @builtins.classmethod
    def amazon_linux2(
        cls,
        hardware_type: typing.Optional[AmiHardwareType] = None,
    ) -> "EcsOptimizedImage":
        '''(experimental) Construct an Amazon Linux 2 image from the latest ECS Optimized AMI published in SSM.

        :param hardware_type: ECS-optimized AMI variant to use.

        :stability: experimental
        '''
        return typing.cast("EcsOptimizedImage", jsii.sinvoke(cls, "amazonLinux2", [hardware_type]))

    @jsii.member(jsii_name="windows") # type: ignore[misc]
    @builtins.classmethod
    def windows(cls, windows_version: "WindowsOptimizedVersion") -> "EcsOptimizedImage":
        '''(experimental) Construct a Windows image from the latest ECS Optimized AMI published in SSM.

        :param windows_version: Windows Version to use.

        :stability: experimental
        '''
        return typing.cast("EcsOptimizedImage", jsii.sinvoke(cls, "windows", [windows_version]))

    @jsii.member(jsii_name="getImage")
    def get_image(self, scope: _Construct_e78e779f) -> _MachineImageConfig_963798fe:
        '''(experimental) Return the correct image.

        :param scope: -

        :stability: experimental
        '''
        return typing.cast(_MachineImageConfig_963798fe, jsii.invoke(self, "getImage", [scope]))


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.EcsTarget",
    jsii_struct_bases=[],
    name_mapping={
        "container_name": "containerName",
        "listener": "listener",
        "new_target_group_id": "newTargetGroupId",
        "container_port": "containerPort",
        "protocol": "protocol",
    },
)
class EcsTarget:
    def __init__(
        self,
        *,
        container_name: builtins.str,
        listener: "ListenerConfig",
        new_target_group_id: builtins.str,
        container_port: typing.Optional[jsii.Number] = None,
        protocol: typing.Optional["Protocol"] = None,
    ) -> None:
        '''
        :param container_name: (experimental) The name of the container.
        :param listener: (experimental) Listener and properties for adding target group to the listener.
        :param new_target_group_id: (experimental) ID for a target group to be created.
        :param container_port: (experimental) The port number of the container. Only applicable when using application/network load balancers. Default: - Container port of the first added port mapping.
        :param protocol: (experimental) The protocol used for the port mapping. Only applicable when using application load balancers. Default: Protocol.TCP

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "container_name": container_name,
            "listener": listener,
            "new_target_group_id": new_target_group_id,
        }
        if container_port is not None:
            self._values["container_port"] = container_port
        if protocol is not None:
            self._values["protocol"] = protocol

    @builtins.property
    def container_name(self) -> builtins.str:
        '''(experimental) The name of the container.

        :stability: experimental
        '''
        result = self._values.get("container_name")
        assert result is not None, "Required property 'container_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def listener(self) -> "ListenerConfig":
        '''(experimental) Listener and properties for adding target group to the listener.

        :stability: experimental
        '''
        result = self._values.get("listener")
        assert result is not None, "Required property 'listener' is missing"
        return typing.cast("ListenerConfig", result)

    @builtins.property
    def new_target_group_id(self) -> builtins.str:
        '''(experimental) ID for a target group to be created.

        :stability: experimental
        '''
        result = self._values.get("new_target_group_id")
        assert result is not None, "Required property 'new_target_group_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def container_port(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The port number of the container.

        Only applicable when using application/network load balancers.

        :default: - Container port of the first added port mapping.

        :stability: experimental
        '''
        result = self._values.get("container_port")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def protocol(self) -> typing.Optional["Protocol"]:
        '''(experimental) The protocol used for the port mapping.

        Only applicable when using application load balancers.

        :default: Protocol.TCP

        :stability: experimental
        '''
        result = self._values.get("protocol")
        return typing.cast(typing.Optional["Protocol"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EcsTarget(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.EfsVolumeConfiguration",
    jsii_struct_bases=[],
    name_mapping={
        "file_system_id": "fileSystemId",
        "authorization_config": "authorizationConfig",
        "root_directory": "rootDirectory",
        "transit_encryption": "transitEncryption",
        "transit_encryption_port": "transitEncryptionPort",
    },
)
class EfsVolumeConfiguration:
    def __init__(
        self,
        *,
        file_system_id: builtins.str,
        authorization_config: typing.Optional[AuthorizationConfig] = None,
        root_directory: typing.Optional[builtins.str] = None,
        transit_encryption: typing.Optional[builtins.str] = None,
        transit_encryption_port: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''(experimental) The configuration for an Elastic FileSystem volume.

        :param file_system_id: (experimental) The Amazon EFS file system ID to use.
        :param authorization_config: (experimental) The authorization configuration details for the Amazon EFS file system. Default: No configuration.
        :param root_directory: (experimental) The directory within the Amazon EFS file system to mount as the root directory inside the host. Specifying / will have the same effect as omitting this parameter. Default: The root of the Amazon EFS volume
        :param transit_encryption: (experimental) Whether or not to enable encryption for Amazon EFS data in transit between the Amazon ECS host and the Amazon EFS server. Transit encryption must be enabled if Amazon EFS IAM authorization is used. Valid values: ENABLED | DISABLED Default: DISABLED
        :param transit_encryption_port: (experimental) The port to use when sending encrypted data between the Amazon ECS host and the Amazon EFS server. EFS mount helper uses. Default: Port selection strategy that the Amazon EFS mount helper uses.

        :stability: experimental
        '''
        if isinstance(authorization_config, dict):
            authorization_config = AuthorizationConfig(**authorization_config)
        self._values: typing.Dict[str, typing.Any] = {
            "file_system_id": file_system_id,
        }
        if authorization_config is not None:
            self._values["authorization_config"] = authorization_config
        if root_directory is not None:
            self._values["root_directory"] = root_directory
        if transit_encryption is not None:
            self._values["transit_encryption"] = transit_encryption
        if transit_encryption_port is not None:
            self._values["transit_encryption_port"] = transit_encryption_port

    @builtins.property
    def file_system_id(self) -> builtins.str:
        '''(experimental) The Amazon EFS file system ID to use.

        :stability: experimental
        '''
        result = self._values.get("file_system_id")
        assert result is not None, "Required property 'file_system_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def authorization_config(self) -> typing.Optional[AuthorizationConfig]:
        '''(experimental) The authorization configuration details for the Amazon EFS file system.

        :default: No configuration.

        :stability: experimental
        '''
        result = self._values.get("authorization_config")
        return typing.cast(typing.Optional[AuthorizationConfig], result)

    @builtins.property
    def root_directory(self) -> typing.Optional[builtins.str]:
        '''(experimental) The directory within the Amazon EFS file system to mount as the root directory inside the host.

        Specifying / will have the same effect as omitting this parameter.

        :default: The root of the Amazon EFS volume

        :stability: experimental
        '''
        result = self._values.get("root_directory")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def transit_encryption(self) -> typing.Optional[builtins.str]:
        '''(experimental) Whether or not to enable encryption for Amazon EFS data in transit between the Amazon ECS host and the Amazon EFS server.

        Transit encryption must be enabled if Amazon EFS IAM authorization is used.

        Valid values: ENABLED | DISABLED

        :default: DISABLED

        :stability: experimental
        '''
        result = self._values.get("transit_encryption")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def transit_encryption_port(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The port to use when sending encrypted data between the Amazon ECS host and the Amazon EFS server.

        EFS mount helper uses.

        :default: Port selection strategy that the Amazon EFS mount helper uses.

        :stability: experimental
        '''
        result = self._values.get("transit_encryption_port")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EfsVolumeConfiguration(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class EnvironmentFile(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="monocdk.aws_ecs.EnvironmentFile",
):
    '''(experimental) Constructs for types of environment files.

    :stability: experimental
    '''

    @builtins.staticmethod
    def __jsii_proxy_class__() -> typing.Type["_EnvironmentFileProxy"]:
        return _EnvironmentFileProxy

    def __init__(self) -> None:
        '''
        :stability: experimental
        '''
        jsii.create(EnvironmentFile, self, [])

    @jsii.member(jsii_name="fromAsset") # type: ignore[misc]
    @builtins.classmethod
    def from_asset(
        cls,
        path: builtins.str,
        *,
        readers: typing.Optional[typing.Sequence[_IGrantable_4c5a91d1]] = None,
        source_hash: typing.Optional[builtins.str] = None,
        exclude: typing.Optional[typing.Sequence[builtins.str]] = None,
        follow: typing.Optional[_FollowMode_98b05cc5] = None,
        ignore_mode: typing.Optional[_IgnoreMode_31d8bf46] = None,
        follow_symlinks: typing.Optional[_SymlinkFollowMode_abf4527a] = None,
        asset_hash: typing.Optional[builtins.str] = None,
        asset_hash_type: typing.Optional[_AssetHashType_49193809] = None,
        bundling: typing.Optional[_BundlingOptions_ab115a99] = None,
    ) -> "AssetEnvironmentFile":
        '''(experimental) Loads the environment file from a local disk path.

        :param path: Local disk path.
        :param readers: (experimental) A list of principals that should be able to read this asset from S3. You can use ``asset.grantRead(principal)`` to grant read permissions later. Default: - No principals that can read file asset.
        :param source_hash: (deprecated) Custom hash to use when identifying the specific version of the asset. For consistency, this custom hash will be SHA256 hashed and encoded as hex. The resulting hash will be the asset hash. NOTE: the source hash is used in order to identify a specific revision of the asset, and used for optimizing and caching deployment activities related to this asset such as packaging, uploading to Amazon S3, etc. If you chose to customize the source hash, you will need to make sure it is updated every time the source changes, or otherwise it is possible that some deployments will not be invalidated. Default: - automatically calculate source hash based on the contents of the source file or directory.
        :param exclude: (deprecated) Glob patterns to exclude from the copy. Default: nothing is excluded
        :param follow: (deprecated) A strategy for how to handle symlinks. Default: Never
        :param ignore_mode: (deprecated) The ignore behavior to use for exclude patterns. Default: - GLOB for file assets, DOCKER or GLOB for docker assets depending on whether the '
        :param follow_symlinks: (experimental) A strategy for how to handle symlinks. Default: SymlinkFollowMode.NEVER
        :param asset_hash: (experimental) Specify a custom hash for this asset. If ``assetHashType`` is set it must be set to ``AssetHashType.CUSTOM``. For consistency, this custom hash will be SHA256 hashed and encoded as hex. The resulting hash will be the asset hash. NOTE: the hash is used in order to identify a specific revision of the asset, and used for optimizing and caching deployment activities related to this asset such as packaging, uploading to Amazon S3, etc. If you chose to customize the hash, you will need to make sure it is updated every time the asset changes, or otherwise it is possible that some deployments will not be invalidated. Default: - based on ``assetHashType``
        :param asset_hash_type: (experimental) Specifies the type of hash to calculate for this asset. If ``assetHash`` is configured, this option must be ``undefined`` or ``AssetHashType.CUSTOM``. Default: - the default is ``AssetHashType.SOURCE``, but if ``assetHash`` is explicitly specified this value defaults to ``AssetHashType.CUSTOM``.
        :param bundling: (experimental) Bundle the asset by executing a command in a Docker container. The asset path will be mounted at ``/asset-input``. The Docker container is responsible for putting content at ``/asset-output``. The content at ``/asset-output`` will be zipped and used as the final asset. Default: - uploaded as-is to S3 if the asset is a regular file or a .zip file, archived into a .zip file and uploaded to S3 otherwise

        :stability: experimental
        '''
        options = _AssetOptions_bd2996da(
            readers=readers,
            source_hash=source_hash,
            exclude=exclude,
            follow=follow,
            ignore_mode=ignore_mode,
            follow_symlinks=follow_symlinks,
            asset_hash=asset_hash,
            asset_hash_type=asset_hash_type,
            bundling=bundling,
        )

        return typing.cast("AssetEnvironmentFile", jsii.sinvoke(cls, "fromAsset", [path, options]))

    @jsii.member(jsii_name="fromBucket") # type: ignore[misc]
    @builtins.classmethod
    def from_bucket(
        cls,
        bucket: _IBucket_73486e29,
        key: builtins.str,
        object_version: typing.Optional[builtins.str] = None,
    ) -> "S3EnvironmentFile":
        '''(experimental) Loads the environment file from an S3 bucket.

        :param bucket: The S3 bucket.
        :param key: The object key.
        :param object_version: Optional S3 object version.

        :return: ``S3EnvironmentFile`` associated with the specified S3 object.

        :stability: experimental
        '''
        return typing.cast("S3EnvironmentFile", jsii.sinvoke(cls, "fromBucket", [bucket, key, object_version]))

    @jsii.member(jsii_name="bind") # type: ignore[misc]
    @abc.abstractmethod
    def bind(self, scope: _Construct_e78e779f) -> "EnvironmentFileConfig":
        '''(experimental) Called when the container is initialized to allow this object to bind to the stack.

        :param scope: The binding scope.

        :stability: experimental
        '''
        ...


class _EnvironmentFileProxy(EnvironmentFile):
    @jsii.member(jsii_name="bind")
    def bind(self, scope: _Construct_e78e779f) -> "EnvironmentFileConfig":
        '''(experimental) Called when the container is initialized to allow this object to bind to the stack.

        :param scope: The binding scope.

        :stability: experimental
        '''
        return typing.cast("EnvironmentFileConfig", jsii.invoke(self, "bind", [scope]))


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.EnvironmentFileConfig",
    jsii_struct_bases=[],
    name_mapping={"file_type": "fileType", "s3_location": "s3Location"},
)
class EnvironmentFileConfig:
    def __init__(
        self,
        *,
        file_type: "EnvironmentFileType",
        s3_location: _Location_cce991ca,
    ) -> None:
        '''(experimental) Configuration for the environment file.

        :param file_type: (experimental) The type of environment file.
        :param s3_location: (experimental) The location of the environment file in S3.

        :stability: experimental
        '''
        if isinstance(s3_location, dict):
            s3_location = _Location_cce991ca(**s3_location)
        self._values: typing.Dict[str, typing.Any] = {
            "file_type": file_type,
            "s3_location": s3_location,
        }

    @builtins.property
    def file_type(self) -> "EnvironmentFileType":
        '''(experimental) The type of environment file.

        :stability: experimental
        '''
        result = self._values.get("file_type")
        assert result is not None, "Required property 'file_type' is missing"
        return typing.cast("EnvironmentFileType", result)

    @builtins.property
    def s3_location(self) -> _Location_cce991ca:
        '''(experimental) The location of the environment file in S3.

        :stability: experimental
        '''
        result = self._values.get("s3_location")
        assert result is not None, "Required property 's3_location' is missing"
        return typing.cast(_Location_cce991ca, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EnvironmentFileConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_ecs.EnvironmentFileType")
class EnvironmentFileType(enum.Enum):
    '''(experimental) Type of environment file to be included in the container definition.

    :stability: experimental
    '''

    S3 = "S3"
    '''(experimental) Environment file hosted on S3, referenced by object ARN.

    :stability: experimental
    '''


@jsii.enum(jsii_type="monocdk.aws_ecs.FargatePlatformVersion")
class FargatePlatformVersion(enum.Enum):
    '''(experimental) The platform version on which to run your service.

    :see: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html
    :stability: experimental
    '''

    LATEST = "LATEST"
    '''(experimental) The latest, recommended platform version.

    :stability: experimental
    '''
    VERSION1_4 = "VERSION1_4"
    '''(experimental) Version 1.4.0.

    Supports EFS endpoints, CAP_SYS_PTRACE Linux capability,
    network performance metrics in CloudWatch Container Insights,
    consolidated 20 GB ephemeral volume.

    :stability: experimental
    '''
    VERSION1_3 = "VERSION1_3"
    '''(experimental) Version 1.3.0.

    Supports secrets, task recycling.

    :stability: experimental
    '''
    VERSION1_2 = "VERSION1_2"
    '''(experimental) Version 1.2.0.

    Supports private registries.

    :stability: experimental
    '''
    VERSION1_1 = "VERSION1_1"
    '''(experimental) Version 1.1.0.

    Supports task metadata, health checks, service discovery.

    :stability: experimental
    '''
    VERSION1_0 = "VERSION1_0"
    '''(experimental) Initial release.

    Based on Amazon Linux 2017.09.

    :stability: experimental
    '''


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.FargateServiceAttributes",
    jsii_struct_bases=[],
    name_mapping={
        "cluster": "cluster",
        "service_arn": "serviceArn",
        "service_name": "serviceName",
    },
)
class FargateServiceAttributes:
    def __init__(
        self,
        *,
        cluster: "ICluster",
        service_arn: typing.Optional[builtins.str] = None,
        service_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) The properties to import from the service using the Fargate launch type.

        :param cluster: (experimental) The cluster that hosts the service.
        :param service_arn: (experimental) The service ARN. Default: - either this, or {@link serviceName}, is required
        :param service_name: (experimental) The name of the service. Default: - either this, or {@link serviceArn}, is required

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "cluster": cluster,
        }
        if service_arn is not None:
            self._values["service_arn"] = service_arn
        if service_name is not None:
            self._values["service_name"] = service_name

    @builtins.property
    def cluster(self) -> "ICluster":
        '''(experimental) The cluster that hosts the service.

        :stability: experimental
        '''
        result = self._values.get("cluster")
        assert result is not None, "Required property 'cluster' is missing"
        return typing.cast("ICluster", result)

    @builtins.property
    def service_arn(self) -> typing.Optional[builtins.str]:
        '''(experimental) The service ARN.

        :default: - either this, or {@link serviceName}, is required

        :stability: experimental
        '''
        result = self._values.get("service_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def service_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name of the service.

        :default: - either this, or {@link serviceArn}, is required

        :stability: experimental
        '''
        result = self._values.get("service_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "FargateServiceAttributes(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.FargateServiceProps",
    jsii_struct_bases=[BaseServiceOptions],
    name_mapping={
        "cluster": "cluster",
        "capacity_provider_strategies": "capacityProviderStrategies",
        "circuit_breaker": "circuitBreaker",
        "cloud_map_options": "cloudMapOptions",
        "deployment_controller": "deploymentController",
        "desired_count": "desiredCount",
        "enable_ecs_managed_tags": "enableECSManagedTags",
        "health_check_grace_period": "healthCheckGracePeriod",
        "max_healthy_percent": "maxHealthyPercent",
        "min_healthy_percent": "minHealthyPercent",
        "propagate_tags": "propagateTags",
        "service_name": "serviceName",
        "task_definition": "taskDefinition",
        "assign_public_ip": "assignPublicIp",
        "platform_version": "platformVersion",
        "propagate_task_tags_from": "propagateTaskTagsFrom",
        "security_group": "securityGroup",
        "security_groups": "securityGroups",
        "vpc_subnets": "vpcSubnets",
    },
)
class FargateServiceProps(BaseServiceOptions):
    def __init__(
        self,
        *,
        cluster: "ICluster",
        capacity_provider_strategies: typing.Optional[typing.Sequence[CapacityProviderStrategy]] = None,
        circuit_breaker: typing.Optional[DeploymentCircuitBreaker] = None,
        cloud_map_options: typing.Optional[CloudMapOptions] = None,
        deployment_controller: typing.Optional[DeploymentController] = None,
        desired_count: typing.Optional[jsii.Number] = None,
        enable_ecs_managed_tags: typing.Optional[builtins.bool] = None,
        health_check_grace_period: typing.Optional[_Duration_070aa057] = None,
        max_healthy_percent: typing.Optional[jsii.Number] = None,
        min_healthy_percent: typing.Optional[jsii.Number] = None,
        propagate_tags: typing.Optional["PropagatedTagSource"] = None,
        service_name: typing.Optional[builtins.str] = None,
        task_definition: "TaskDefinition",
        assign_public_ip: typing.Optional[builtins.bool] = None,
        platform_version: typing.Optional[FargatePlatformVersion] = None,
        propagate_task_tags_from: typing.Optional["PropagatedTagSource"] = None,
        security_group: typing.Optional[_ISecurityGroup_cdbba9d3] = None,
        security_groups: typing.Optional[typing.Sequence[_ISecurityGroup_cdbba9d3]] = None,
        vpc_subnets: typing.Optional[_SubnetSelection_1284e62c] = None,
    ) -> None:
        '''(experimental) The properties for defining a service using the Fargate launch type.

        :param cluster: (experimental) The name of the cluster that hosts the service.
        :param capacity_provider_strategies: (experimental) A list of Capacity Provider strategies used to place a service. Default: - undefined
        :param circuit_breaker: (experimental) Whether to enable the deployment circuit breaker. If this property is defined, circuit breaker will be implicitly enabled. Default: - disabled
        :param cloud_map_options: (experimental) The options for configuring an Amazon ECS service to use service discovery. Default: - AWS Cloud Map service discovery is not enabled.
        :param deployment_controller: (experimental) Specifies which deployment controller to use for the service. For more information, see `Amazon ECS Deployment Types <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-types.html>`_ Default: - Rolling update (ECS)
        :param desired_count: (experimental) The desired number of instantiations of the task definition to keep running on the service. Default: - When creating the service, default is 1; when updating the service, default uses the current task number.
        :param enable_ecs_managed_tags: (experimental) Specifies whether to enable Amazon ECS managed tags for the tasks within the service. For more information, see `Tagging Your Amazon ECS Resources <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html>`_ Default: false
        :param health_check_grace_period: (experimental) The period of time, in seconds, that the Amazon ECS service scheduler ignores unhealthy Elastic Load Balancing target health checks after a task has first started. Default: - defaults to 60 seconds if at least one load balancer is in-use and it is not already set
        :param max_healthy_percent: (experimental) The maximum number of tasks, specified as a percentage of the Amazon ECS service's DesiredCount value, that can run in a service during a deployment. Default: - 100 if daemon, otherwise 200
        :param min_healthy_percent: (experimental) The minimum number of tasks, specified as a percentage of the Amazon ECS service's DesiredCount value, that must continue to run and remain healthy during a deployment. Default: - 0 if daemon, otherwise 50
        :param propagate_tags: (experimental) Specifies whether to propagate the tags from the task definition or the service to the tasks in the service. Valid values are: PropagatedTagSource.SERVICE, PropagatedTagSource.TASK_DEFINITION or PropagatedTagSource.NONE Default: PropagatedTagSource.NONE
        :param service_name: (experimental) The name of the service. Default: - CloudFormation-generated name.
        :param task_definition: (experimental) The task definition to use for tasks in the service. [disable-awslint:ref-via-interface]
        :param assign_public_ip: (experimental) Specifies whether the task's elastic network interface receives a public IP address. If true, each task will receive a public IP address. Default: false
        :param platform_version: (experimental) The platform version on which to run your service. If one is not specified, the LATEST platform version is used by default. For more information, see `AWS Fargate Platform Versions <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html>`_ in the Amazon Elastic Container Service Developer Guide. Default: Latest
        :param propagate_task_tags_from: (deprecated) Specifies whether to propagate the tags from the task definition or the service to the tasks in the service. Tags can only be propagated to the tasks within the service during service creation. Default: PropagatedTagSource.NONE
        :param security_group: (deprecated) The security groups to associate with the service. If you do not specify a security group, the default security group for the VPC is used. Default: - A new security group is created.
        :param security_groups: (experimental) The security groups to associate with the service. If you do not specify a security group, the default security group for the VPC is used. Default: - A new security group is created.
        :param vpc_subnets: (experimental) The subnets to associate with the service. Default: - Public subnets if ``assignPublicIp`` is set, otherwise the first available one of Private, Isolated, Public, in that order.

        :stability: experimental
        '''
        if isinstance(circuit_breaker, dict):
            circuit_breaker = DeploymentCircuitBreaker(**circuit_breaker)
        if isinstance(cloud_map_options, dict):
            cloud_map_options = CloudMapOptions(**cloud_map_options)
        if isinstance(deployment_controller, dict):
            deployment_controller = DeploymentController(**deployment_controller)
        if isinstance(vpc_subnets, dict):
            vpc_subnets = _SubnetSelection_1284e62c(**vpc_subnets)
        self._values: typing.Dict[str, typing.Any] = {
            "cluster": cluster,
            "task_definition": task_definition,
        }
        if capacity_provider_strategies is not None:
            self._values["capacity_provider_strategies"] = capacity_provider_strategies
        if circuit_breaker is not None:
            self._values["circuit_breaker"] = circuit_breaker
        if cloud_map_options is not None:
            self._values["cloud_map_options"] = cloud_map_options
        if deployment_controller is not None:
            self._values["deployment_controller"] = deployment_controller
        if desired_count is not None:
            self._values["desired_count"] = desired_count
        if enable_ecs_managed_tags is not None:
            self._values["enable_ecs_managed_tags"] = enable_ecs_managed_tags
        if health_check_grace_period is not None:
            self._values["health_check_grace_period"] = health_check_grace_period
        if max_healthy_percent is not None:
            self._values["max_healthy_percent"] = max_healthy_percent
        if min_healthy_percent is not None:
            self._values["min_healthy_percent"] = min_healthy_percent
        if propagate_tags is not None:
            self._values["propagate_tags"] = propagate_tags
        if service_name is not None:
            self._values["service_name"] = service_name
        if assign_public_ip is not None:
            self._values["assign_public_ip"] = assign_public_ip
        if platform_version is not None:
            self._values["platform_version"] = platform_version
        if propagate_task_tags_from is not None:
            self._values["propagate_task_tags_from"] = propagate_task_tags_from
        if security_group is not None:
            self._values["security_group"] = security_group
        if security_groups is not None:
            self._values["security_groups"] = security_groups
        if vpc_subnets is not None:
            self._values["vpc_subnets"] = vpc_subnets

    @builtins.property
    def cluster(self) -> "ICluster":
        '''(experimental) The name of the cluster that hosts the service.

        :stability: experimental
        '''
        result = self._values.get("cluster")
        assert result is not None, "Required property 'cluster' is missing"
        return typing.cast("ICluster", result)

    @builtins.property
    def capacity_provider_strategies(
        self,
    ) -> typing.Optional[typing.List[CapacityProviderStrategy]]:
        '''(experimental) A list of Capacity Provider strategies used to place a service.

        :default: - undefined

        :stability: experimental
        '''
        result = self._values.get("capacity_provider_strategies")
        return typing.cast(typing.Optional[typing.List[CapacityProviderStrategy]], result)

    @builtins.property
    def circuit_breaker(self) -> typing.Optional[DeploymentCircuitBreaker]:
        '''(experimental) Whether to enable the deployment circuit breaker.

        If this property is defined, circuit breaker will be implicitly
        enabled.

        :default: - disabled

        :stability: experimental
        '''
        result = self._values.get("circuit_breaker")
        return typing.cast(typing.Optional[DeploymentCircuitBreaker], result)

    @builtins.property
    def cloud_map_options(self) -> typing.Optional[CloudMapOptions]:
        '''(experimental) The options for configuring an Amazon ECS service to use service discovery.

        :default: - AWS Cloud Map service discovery is not enabled.

        :stability: experimental
        '''
        result = self._values.get("cloud_map_options")
        return typing.cast(typing.Optional[CloudMapOptions], result)

    @builtins.property
    def deployment_controller(self) -> typing.Optional[DeploymentController]:
        '''(experimental) Specifies which deployment controller to use for the service.

        For more information, see
        `Amazon ECS Deployment Types <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-types.html>`_

        :default: - Rolling update (ECS)

        :stability: experimental
        '''
        result = self._values.get("deployment_controller")
        return typing.cast(typing.Optional[DeploymentController], result)

    @builtins.property
    def desired_count(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The desired number of instantiations of the task definition to keep running on the service.

        :default:

        - When creating the service, default is 1; when updating the service, default uses
        the current task number.

        :stability: experimental
        '''
        result = self._values.get("desired_count")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def enable_ecs_managed_tags(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specifies whether to enable Amazon ECS managed tags for the tasks within the service.

        For more information, see
        `Tagging Your Amazon ECS Resources <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html>`_

        :default: false

        :stability: experimental
        '''
        result = self._values.get("enable_ecs_managed_tags")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def health_check_grace_period(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) The period of time, in seconds, that the Amazon ECS service scheduler ignores unhealthy Elastic Load Balancing target health checks after a task has first started.

        :default: - defaults to 60 seconds if at least one load balancer is in-use and it is not already set

        :stability: experimental
        '''
        result = self._values.get("health_check_grace_period")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def max_healthy_percent(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The maximum number of tasks, specified as a percentage of the Amazon ECS service's DesiredCount value, that can run in a service during a deployment.

        :default: - 100 if daemon, otherwise 200

        :stability: experimental
        '''
        result = self._values.get("max_healthy_percent")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def min_healthy_percent(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The minimum number of tasks, specified as a percentage of the Amazon ECS service's DesiredCount value, that must continue to run and remain healthy during a deployment.

        :default: - 0 if daemon, otherwise 50

        :stability: experimental
        '''
        result = self._values.get("min_healthy_percent")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def propagate_tags(self) -> typing.Optional["PropagatedTagSource"]:
        '''(experimental) Specifies whether to propagate the tags from the task definition or the service to the tasks in the service.

        Valid values are: PropagatedTagSource.SERVICE, PropagatedTagSource.TASK_DEFINITION or PropagatedTagSource.NONE

        :default: PropagatedTagSource.NONE

        :stability: experimental
        '''
        result = self._values.get("propagate_tags")
        return typing.cast(typing.Optional["PropagatedTagSource"], result)

    @builtins.property
    def service_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name of the service.

        :default: - CloudFormation-generated name.

        :stability: experimental
        '''
        result = self._values.get("service_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def task_definition(self) -> "TaskDefinition":
        '''(experimental) The task definition to use for tasks in the service.

        [disable-awslint:ref-via-interface]

        :stability: experimental
        '''
        result = self._values.get("task_definition")
        assert result is not None, "Required property 'task_definition' is missing"
        return typing.cast("TaskDefinition", result)

    @builtins.property
    def assign_public_ip(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specifies whether the task's elastic network interface receives a public IP address.

        If true, each task will receive a public IP address.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("assign_public_ip")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def platform_version(self) -> typing.Optional[FargatePlatformVersion]:
        '''(experimental) The platform version on which to run your service.

        If one is not specified, the LATEST platform version is used by default. For more information, see
        `AWS Fargate Platform Versions <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html>`_
        in the Amazon Elastic Container Service Developer Guide.

        :default: Latest

        :stability: experimental
        '''
        result = self._values.get("platform_version")
        return typing.cast(typing.Optional[FargatePlatformVersion], result)

    @builtins.property
    def propagate_task_tags_from(self) -> typing.Optional["PropagatedTagSource"]:
        '''(deprecated) Specifies whether to propagate the tags from the task definition or the service to the tasks in the service.

        Tags can only be propagated to the tasks within the service during service creation.

        :default: PropagatedTagSource.NONE

        :deprecated: Use ``propagateTags`` instead.

        :stability: deprecated
        '''
        result = self._values.get("propagate_task_tags_from")
        return typing.cast(typing.Optional["PropagatedTagSource"], result)

    @builtins.property
    def security_group(self) -> typing.Optional[_ISecurityGroup_cdbba9d3]:
        '''(deprecated) The security groups to associate with the service.

        If you do not specify a security group, the default security group for the VPC is used.

        :default: - A new security group is created.

        :deprecated: use securityGroups instead.

        :stability: deprecated
        '''
        result = self._values.get("security_group")
        return typing.cast(typing.Optional[_ISecurityGroup_cdbba9d3], result)

    @builtins.property
    def security_groups(self) -> typing.Optional[typing.List[_ISecurityGroup_cdbba9d3]]:
        '''(experimental) The security groups to associate with the service.

        If you do not specify a security group, the default security group for the VPC is used.

        :default: - A new security group is created.

        :stability: experimental
        '''
        result = self._values.get("security_groups")
        return typing.cast(typing.Optional[typing.List[_ISecurityGroup_cdbba9d3]], result)

    @builtins.property
    def vpc_subnets(self) -> typing.Optional[_SubnetSelection_1284e62c]:
        '''(experimental) The subnets to associate with the service.

        :default: - Public subnets if ``assignPublicIp`` is set, otherwise the first available one of Private, Isolated, Public, in that order.

        :stability: experimental
        '''
        result = self._values.get("vpc_subnets")
        return typing.cast(typing.Optional[_SubnetSelection_1284e62c], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "FargateServiceProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.FargateTaskDefinitionAttributes",
    jsii_struct_bases=[CommonTaskDefinitionAttributes],
    name_mapping={
        "task_definition_arn": "taskDefinitionArn",
        "network_mode": "networkMode",
        "task_role": "taskRole",
    },
)
class FargateTaskDefinitionAttributes(CommonTaskDefinitionAttributes):
    def __init__(
        self,
        *,
        task_definition_arn: builtins.str,
        network_mode: typing.Optional["NetworkMode"] = None,
        task_role: typing.Optional[_IRole_59af6f50] = None,
    ) -> None:
        '''(experimental) Attributes used to import an existing Fargate task definition.

        :param task_definition_arn: (experimental) The arn of the task definition.
        :param network_mode: (experimental) The networking mode to use for the containers in the task. Default: Network mode cannot be provided to the imported task.
        :param task_role: (experimental) The name of the IAM role that grants containers in the task permission to call AWS APIs on your behalf. Default: Permissions cannot be granted to the imported task.

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "task_definition_arn": task_definition_arn,
        }
        if network_mode is not None:
            self._values["network_mode"] = network_mode
        if task_role is not None:
            self._values["task_role"] = task_role

    @builtins.property
    def task_definition_arn(self) -> builtins.str:
        '''(experimental) The arn of the task definition.

        :stability: experimental
        '''
        result = self._values.get("task_definition_arn")
        assert result is not None, "Required property 'task_definition_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def network_mode(self) -> typing.Optional["NetworkMode"]:
        '''(experimental) The networking mode to use for the containers in the task.

        :default: Network mode cannot be provided to the imported task.

        :stability: experimental
        '''
        result = self._values.get("network_mode")
        return typing.cast(typing.Optional["NetworkMode"], result)

    @builtins.property
    def task_role(self) -> typing.Optional[_IRole_59af6f50]:
        '''(experimental) The name of the IAM role that grants containers in the task permission to call AWS APIs on your behalf.

        :default: Permissions cannot be granted to the imported task.

        :stability: experimental
        '''
        result = self._values.get("task_role")
        return typing.cast(typing.Optional[_IRole_59af6f50], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "FargateTaskDefinitionAttributes(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.FargateTaskDefinitionProps",
    jsii_struct_bases=[CommonTaskDefinitionProps],
    name_mapping={
        "execution_role": "executionRole",
        "family": "family",
        "proxy_configuration": "proxyConfiguration",
        "task_role": "taskRole",
        "volumes": "volumes",
        "cpu": "cpu",
        "memory_limit_mib": "memoryLimitMiB",
    },
)
class FargateTaskDefinitionProps(CommonTaskDefinitionProps):
    def __init__(
        self,
        *,
        execution_role: typing.Optional[_IRole_59af6f50] = None,
        family: typing.Optional[builtins.str] = None,
        proxy_configuration: typing.Optional["ProxyConfiguration"] = None,
        task_role: typing.Optional[_IRole_59af6f50] = None,
        volumes: typing.Optional[typing.Sequence["Volume"]] = None,
        cpu: typing.Optional[jsii.Number] = None,
        memory_limit_mib: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''(experimental) The properties for a task definition.

        :param execution_role: (experimental) The name of the IAM task execution role that grants the ECS agent to call AWS APIs on your behalf. The role will be used to retrieve container images from ECR and create CloudWatch log groups. Default: - An execution role will be automatically created if you use ECR images in your task definition.
        :param family: (experimental) The name of a family that this task definition is registered to. A family groups multiple versions of a task definition. Default: - Automatically generated name.
        :param proxy_configuration: (experimental) The configuration details for the App Mesh proxy. Default: - No proxy configuration.
        :param task_role: (experimental) The name of the IAM role that grants containers in the task permission to call AWS APIs on your behalf. Default: - A task role is automatically created for you.
        :param volumes: (experimental) The list of volume definitions for the task. For more information, see `Task Definition Parameter Volumes <https://docs.aws.amazon.com/AmazonECS/latest/developerguide//task_definition_parameters.html#volumes>`_. Default: - No volumes are passed to the Docker daemon on a container instance.
        :param cpu: (experimental) The number of cpu units used by the task. For tasks using the Fargate launch type, this field is required and you must use one of the following values, which determines your range of valid values for the memory parameter: 256 (.25 vCPU) - Available memory values: 512 (0.5 GB), 1024 (1 GB), 2048 (2 GB) 512 (.5 vCPU) - Available memory values: 1024 (1 GB), 2048 (2 GB), 3072 (3 GB), 4096 (4 GB) 1024 (1 vCPU) - Available memory values: 2048 (2 GB), 3072 (3 GB), 4096 (4 GB), 5120 (5 GB), 6144 (6 GB), 7168 (7 GB), 8192 (8 GB) 2048 (2 vCPU) - Available memory values: Between 4096 (4 GB) and 16384 (16 GB) in increments of 1024 (1 GB) 4096 (4 vCPU) - Available memory values: Between 8192 (8 GB) and 30720 (30 GB) in increments of 1024 (1 GB) Default: 256
        :param memory_limit_mib: (experimental) The amount (in MiB) of memory used by the task. For tasks using the Fargate launch type, this field is required and you must use one of the following values, which determines your range of valid values for the cpu parameter: 512 (0.5 GB), 1024 (1 GB), 2048 (2 GB) - Available cpu values: 256 (.25 vCPU) 1024 (1 GB), 2048 (2 GB), 3072 (3 GB), 4096 (4 GB) - Available cpu values: 512 (.5 vCPU) 2048 (2 GB), 3072 (3 GB), 4096 (4 GB), 5120 (5 GB), 6144 (6 GB), 7168 (7 GB), 8192 (8 GB) - Available cpu values: 1024 (1 vCPU) Between 4096 (4 GB) and 16384 (16 GB) in increments of 1024 (1 GB) - Available cpu values: 2048 (2 vCPU) Between 8192 (8 GB) and 30720 (30 GB) in increments of 1024 (1 GB) - Available cpu values: 4096 (4 vCPU) Default: 512

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if execution_role is not None:
            self._values["execution_role"] = execution_role
        if family is not None:
            self._values["family"] = family
        if proxy_configuration is not None:
            self._values["proxy_configuration"] = proxy_configuration
        if task_role is not None:
            self._values["task_role"] = task_role
        if volumes is not None:
            self._values["volumes"] = volumes
        if cpu is not None:
            self._values["cpu"] = cpu
        if memory_limit_mib is not None:
            self._values["memory_limit_mib"] = memory_limit_mib

    @builtins.property
    def execution_role(self) -> typing.Optional[_IRole_59af6f50]:
        '''(experimental) The name of the IAM task execution role that grants the ECS agent to call AWS APIs on your behalf.

        The role will be used to retrieve container images from ECR and create CloudWatch log groups.

        :default: - An execution role will be automatically created if you use ECR images in your task definition.

        :stability: experimental
        '''
        result = self._values.get("execution_role")
        return typing.cast(typing.Optional[_IRole_59af6f50], result)

    @builtins.property
    def family(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name of a family that this task definition is registered to.

        A family groups multiple versions of a task definition.

        :default: - Automatically generated name.

        :stability: experimental
        '''
        result = self._values.get("family")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def proxy_configuration(self) -> typing.Optional["ProxyConfiguration"]:
        '''(experimental) The configuration details for the App Mesh proxy.

        :default: - No proxy configuration.

        :stability: experimental
        '''
        result = self._values.get("proxy_configuration")
        return typing.cast(typing.Optional["ProxyConfiguration"], result)

    @builtins.property
    def task_role(self) -> typing.Optional[_IRole_59af6f50]:
        '''(experimental) The name of the IAM role that grants containers in the task permission to call AWS APIs on your behalf.

        :default: - A task role is automatically created for you.

        :stability: experimental
        '''
        result = self._values.get("task_role")
        return typing.cast(typing.Optional[_IRole_59af6f50], result)

    @builtins.property
    def volumes(self) -> typing.Optional[typing.List["Volume"]]:
        '''(experimental) The list of volume definitions for the task.

        For more information, see
        `Task Definition Parameter Volumes <https://docs.aws.amazon.com/AmazonECS/latest/developerguide//task_definition_parameters.html#volumes>`_.

        :default: - No volumes are passed to the Docker daemon on a container instance.

        :stability: experimental
        '''
        result = self._values.get("volumes")
        return typing.cast(typing.Optional[typing.List["Volume"]], result)

    @builtins.property
    def cpu(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The number of cpu units used by the task.

        For tasks using the Fargate launch type,
        this field is required and you must use one of the following values,
        which determines your range of valid values for the memory parameter:

        256 (.25 vCPU) - Available memory values: 512 (0.5 GB), 1024 (1 GB), 2048 (2 GB)

        512 (.5 vCPU) - Available memory values: 1024 (1 GB), 2048 (2 GB), 3072 (3 GB), 4096 (4 GB)

        1024 (1 vCPU) - Available memory values: 2048 (2 GB), 3072 (3 GB), 4096 (4 GB), 5120 (5 GB), 6144 (6 GB), 7168 (7 GB), 8192 (8 GB)

        2048 (2 vCPU) - Available memory values: Between 4096 (4 GB) and 16384 (16 GB) in increments of 1024 (1 GB)

        4096 (4 vCPU) - Available memory values: Between 8192 (8 GB) and 30720 (30 GB) in increments of 1024 (1 GB)

        :default: 256

        :stability: experimental
        '''
        result = self._values.get("cpu")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def memory_limit_mib(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The amount (in MiB) of memory used by the task.

        For tasks using the Fargate launch type,
        this field is required and you must use one of the following values, which determines your range of valid values for the cpu parameter:

        512 (0.5 GB), 1024 (1 GB), 2048 (2 GB) - Available cpu values: 256 (.25 vCPU)

        1024 (1 GB), 2048 (2 GB), 3072 (3 GB), 4096 (4 GB) - Available cpu values: 512 (.5 vCPU)

        2048 (2 GB), 3072 (3 GB), 4096 (4 GB), 5120 (5 GB), 6144 (6 GB), 7168 (7 GB), 8192 (8 GB) - Available cpu values: 1024 (1 vCPU)

        Between 4096 (4 GB) and 16384 (16 GB) in increments of 1024 (1 GB) - Available cpu values: 2048 (2 vCPU)

        Between 8192 (8 GB) and 30720 (30 GB) in increments of 1024 (1 GB) - Available cpu values: 4096 (4 vCPU)

        :default: 512

        :stability: experimental
        '''
        result = self._values.get("memory_limit_mib")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "FargateTaskDefinitionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.FireLensLogDriverProps",
    jsii_struct_bases=[BaseLogDriverProps],
    name_mapping={
        "env": "env",
        "env_regex": "envRegex",
        "labels": "labels",
        "tag": "tag",
        "options": "options",
    },
)
class FireLensLogDriverProps(BaseLogDriverProps):
    def __init__(
        self,
        *,
        env: typing.Optional[typing.Sequence[builtins.str]] = None,
        env_regex: typing.Optional[builtins.str] = None,
        labels: typing.Optional[typing.Sequence[builtins.str]] = None,
        tag: typing.Optional[builtins.str] = None,
        options: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
    ) -> None:
        '''(experimental) Specifies the firelens log driver configuration options.

        :param env: (experimental) The env option takes an array of keys. If there is collision between label and env keys, the value of the env takes precedence. Adds additional fields to the extra attributes of a logging message. Default: - No env
        :param env_regex: (experimental) The env-regex option is similar to and compatible with env. Its value is a regular expression to match logging-related environment variables. It is used for advanced log tag options. Default: - No envRegex
        :param labels: (experimental) The labels option takes an array of keys. If there is collision between label and env keys, the value of the env takes precedence. Adds additional fields to the extra attributes of a logging message. Default: - No labels
        :param tag: (experimental) By default, Docker uses the first 12 characters of the container ID to tag log messages. Refer to the log tag option documentation for customizing the log tag format. Default: - The first 12 characters of the container ID
        :param options: (experimental) The configuration options to send to the log driver. Default: - the log driver options

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if env is not None:
            self._values["env"] = env
        if env_regex is not None:
            self._values["env_regex"] = env_regex
        if labels is not None:
            self._values["labels"] = labels
        if tag is not None:
            self._values["tag"] = tag
        if options is not None:
            self._values["options"] = options

    @builtins.property
    def env(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) The env option takes an array of keys.

        If there is collision between
        label and env keys, the value of the env takes precedence. Adds additional fields
        to the extra attributes of a logging message.

        :default: - No env

        :stability: experimental
        '''
        result = self._values.get("env")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def env_regex(self) -> typing.Optional[builtins.str]:
        '''(experimental) The env-regex option is similar to and compatible with env.

        Its value is a regular
        expression to match logging-related environment variables. It is used for advanced
        log tag options.

        :default: - No envRegex

        :stability: experimental
        '''
        result = self._values.get("env_regex")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def labels(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) The labels option takes an array of keys.

        If there is collision
        between label and env keys, the value of the env takes precedence. Adds additional
        fields to the extra attributes of a logging message.

        :default: - No labels

        :stability: experimental
        '''
        result = self._values.get("labels")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def tag(self) -> typing.Optional[builtins.str]:
        '''(experimental) By default, Docker uses the first 12 characters of the container ID to tag log messages.

        Refer to the log tag option documentation for customizing the
        log tag format.

        :default: - The first 12 characters of the container ID

        :stability: experimental
        '''
        result = self._values.get("tag")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def options(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''(experimental) The configuration options to send to the log driver.

        :default: - the log driver options

        :stability: experimental
        '''
        result = self._values.get("options")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "FireLensLogDriverProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.FirelensConfig",
    jsii_struct_bases=[],
    name_mapping={"type": "type", "options": "options"},
)
class FirelensConfig:
    def __init__(
        self,
        *,
        type: "FirelensLogRouterType",
        options: typing.Optional["FirelensOptions"] = None,
    ) -> None:
        '''(experimental) Firelens Configuration https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html#firelens-taskdef.

        :param type: (experimental) The log router to use. Default: - fluentbit
        :param options: (experimental) Firelens options. Default: - no additional options

        :stability: experimental
        '''
        if isinstance(options, dict):
            options = FirelensOptions(**options)
        self._values: typing.Dict[str, typing.Any] = {
            "type": type,
        }
        if options is not None:
            self._values["options"] = options

    @builtins.property
    def type(self) -> "FirelensLogRouterType":
        '''(experimental) The log router to use.

        :default: - fluentbit

        :stability: experimental
        '''
        result = self._values.get("type")
        assert result is not None, "Required property 'type' is missing"
        return typing.cast("FirelensLogRouterType", result)

    @builtins.property
    def options(self) -> typing.Optional["FirelensOptions"]:
        '''(experimental) Firelens options.

        :default: - no additional options

        :stability: experimental
        '''
        result = self._values.get("options")
        return typing.cast(typing.Optional["FirelensOptions"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "FirelensConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_ecs.FirelensConfigFileType")
class FirelensConfigFileType(enum.Enum):
    '''(experimental) Firelens configuration file type, s3 or file path.

    https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html#firelens-taskdef-customconfig

    :stability: experimental
    '''

    S3 = "S3"
    '''(experimental) s3.

    :stability: experimental
    '''
    FILE = "FILE"
    '''(experimental) fluentd.

    :stability: experimental
    '''


class FirelensLogRouter(
    ContainerDefinition,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_ecs.FirelensLogRouter",
):
    '''(experimental) Firelens log router.

    :stability: experimental
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        firelens_config: FirelensConfig,
        task_definition: "TaskDefinition",
        image: ContainerImage,
        command: typing.Optional[typing.Sequence[builtins.str]] = None,
        container_name: typing.Optional[builtins.str] = None,
        cpu: typing.Optional[jsii.Number] = None,
        disable_networking: typing.Optional[builtins.bool] = None,
        dns_search_domains: typing.Optional[typing.Sequence[builtins.str]] = None,
        dns_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        docker_labels: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        docker_security_options: typing.Optional[typing.Sequence[builtins.str]] = None,
        entry_point: typing.Optional[typing.Sequence[builtins.str]] = None,
        environment: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        environment_files: typing.Optional[typing.Sequence[EnvironmentFile]] = None,
        essential: typing.Optional[builtins.bool] = None,
        extra_hosts: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        gpu_count: typing.Optional[jsii.Number] = None,
        health_check: typing.Optional["HealthCheck"] = None,
        hostname: typing.Optional[builtins.str] = None,
        inference_accelerator_resources: typing.Optional[typing.Sequence[builtins.str]] = None,
        linux_parameters: typing.Optional["LinuxParameters"] = None,
        logging: typing.Optional["LogDriver"] = None,
        memory_limit_mib: typing.Optional[jsii.Number] = None,
        memory_reservation_mib: typing.Optional[jsii.Number] = None,
        port_mappings: typing.Optional[typing.Sequence["PortMapping"]] = None,
        privileged: typing.Optional[builtins.bool] = None,
        readonly_root_filesystem: typing.Optional[builtins.bool] = None,
        secrets: typing.Optional[typing.Mapping[builtins.str, "Secret"]] = None,
        start_timeout: typing.Optional[_Duration_070aa057] = None,
        stop_timeout: typing.Optional[_Duration_070aa057] = None,
        user: typing.Optional[builtins.str] = None,
        working_directory: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) Constructs a new instance of the FirelensLogRouter class.

        :param scope: -
        :param id: -
        :param firelens_config: (experimental) Firelens configuration.
        :param task_definition: (experimental) The name of the task definition that includes this container definition. [disable-awslint:ref-via-interface]
        :param image: (experimental) The image used to start a container. This string is passed directly to the Docker daemon. Images in the Docker Hub registry are available by default. Other repositories are specified with either repository-url/image:tag or repository-url/image@digest. TODO: Update these to specify using classes of IContainerImage
        :param command: (experimental) The command that is passed to the container. If you provide a shell command as a single string, you have to quote command-line arguments. Default: - CMD value built into container image.
        :param container_name: (experimental) The name of the container. Default: - id of node associated with ContainerDefinition.
        :param cpu: (experimental) The minimum number of CPU units to reserve for the container. Default: - No minimum CPU units reserved.
        :param disable_networking: (experimental) Specifies whether networking is disabled within the container. When this parameter is true, networking is disabled within the container. Default: false
        :param dns_search_domains: (experimental) A list of DNS search domains that are presented to the container. Default: - No search domains.
        :param dns_servers: (experimental) A list of DNS servers that are presented to the container. Default: - Default DNS servers.
        :param docker_labels: (experimental) A key/value map of labels to add to the container. Default: - No labels.
        :param docker_security_options: (experimental) A list of strings to provide custom labels for SELinux and AppArmor multi-level security systems. Default: - No security labels.
        :param entry_point: (experimental) The ENTRYPOINT value to pass to the container. Default: - Entry point configured in container.
        :param environment: (experimental) The environment variables to pass to the container. Default: - No environment variables.
        :param environment_files: (experimental) The environment files to pass to the container. Default: - No environment files.
        :param essential: (experimental) Specifies whether the container is marked essential. If the essential parameter of a container is marked as true, and that container fails or stops for any reason, all other containers that are part of the task are stopped. If the essential parameter of a container is marked as false, then its failure does not affect the rest of the containers in a task. All tasks must have at least one essential container. If this parameter is omitted, a container is assumed to be essential. Default: true
        :param extra_hosts: (experimental) A list of hostnames and IP address mappings to append to the /etc/hosts file on the container. Default: - No extra hosts.
        :param gpu_count: (experimental) The number of GPUs assigned to the container. Default: - No GPUs assigned.
        :param health_check: (experimental) The health check command and associated configuration parameters for the container. Default: - Health check configuration from container.
        :param hostname: (experimental) The hostname to use for your container. Default: - Automatic hostname.
        :param inference_accelerator_resources: (experimental) The inference accelerators referenced by the container. Default: - No inference accelerators assigned.
        :param linux_parameters: (experimental) Linux-specific modifications that are applied to the container, such as Linux kernel capabilities. For more information see `KernelCapabilities <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_KernelCapabilities.html>`_. Default: - No Linux parameters.
        :param logging: (experimental) The log configuration specification for the container. Default: - Containers use the same logging driver that the Docker daemon uses.
        :param memory_limit_mib: (experimental) The amount (in MiB) of memory to present to the container. If your container attempts to exceed the allocated memory, the container is terminated. At least one of memoryLimitMiB and memoryReservationMiB is required for non-Fargate services. Default: - No memory limit.
        :param memory_reservation_mib: (experimental) The soft limit (in MiB) of memory to reserve for the container. When system memory is under heavy contention, Docker attempts to keep the container memory to this soft limit. However, your container can consume more memory when it needs to, up to either the hard limit specified with the memory parameter (if applicable), or all of the available memory on the container instance, whichever comes first. At least one of memoryLimitMiB and memoryReservationMiB is required for non-Fargate services. Default: - No memory reserved.
        :param port_mappings: (experimental) The port mappings to add to the container definition. Default: - No ports are mapped.
        :param privileged: (experimental) Specifies whether the container is marked as privileged. When this parameter is true, the container is given elevated privileges on the host container instance (similar to the root user). Default: false
        :param readonly_root_filesystem: (experimental) When this parameter is true, the container is given read-only access to its root file system. Default: false
        :param secrets: (experimental) The secret environment variables to pass to the container. Default: - No secret environment variables.
        :param start_timeout: (experimental) Time duration (in seconds) to wait before giving up on resolving dependencies for a container. Default: - none
        :param stop_timeout: (experimental) Time duration (in seconds) to wait before the container is forcefully killed if it doesn't exit normally on its own. Default: - none
        :param user: (experimental) The user name to use inside the container. Default: root
        :param working_directory: (experimental) The working directory in which to run commands inside the container. Default: /

        :stability: experimental
        '''
        props = FirelensLogRouterProps(
            firelens_config=firelens_config,
            task_definition=task_definition,
            image=image,
            command=command,
            container_name=container_name,
            cpu=cpu,
            disable_networking=disable_networking,
            dns_search_domains=dns_search_domains,
            dns_servers=dns_servers,
            docker_labels=docker_labels,
            docker_security_options=docker_security_options,
            entry_point=entry_point,
            environment=environment,
            environment_files=environment_files,
            essential=essential,
            extra_hosts=extra_hosts,
            gpu_count=gpu_count,
            health_check=health_check,
            hostname=hostname,
            inference_accelerator_resources=inference_accelerator_resources,
            linux_parameters=linux_parameters,
            logging=logging,
            memory_limit_mib=memory_limit_mib,
            memory_reservation_mib=memory_reservation_mib,
            port_mappings=port_mappings,
            privileged=privileged,
            readonly_root_filesystem=readonly_root_filesystem,
            secrets=secrets,
            start_timeout=start_timeout,
            stop_timeout=stop_timeout,
            user=user,
            working_directory=working_directory,
        )

        jsii.create(FirelensLogRouter, self, [scope, id, props])

    @jsii.member(jsii_name="renderContainerDefinition")
    def render_container_definition(
        self,
        _task_definition: typing.Optional["TaskDefinition"] = None,
    ) -> CfnTaskDefinition.ContainerDefinitionProperty:
        '''(experimental) Render this container definition to a CloudFormation object.

        :param _task_definition: -

        :stability: experimental
        '''
        return typing.cast(CfnTaskDefinition.ContainerDefinitionProperty, jsii.invoke(self, "renderContainerDefinition", [_task_definition]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="firelensConfig")
    def firelens_config(self) -> FirelensConfig:
        '''(experimental) Firelens configuration.

        :stability: experimental
        '''
        return typing.cast(FirelensConfig, jsii.get(self, "firelensConfig"))


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.FirelensLogRouterDefinitionOptions",
    jsii_struct_bases=[ContainerDefinitionOptions],
    name_mapping={
        "image": "image",
        "command": "command",
        "container_name": "containerName",
        "cpu": "cpu",
        "disable_networking": "disableNetworking",
        "dns_search_domains": "dnsSearchDomains",
        "dns_servers": "dnsServers",
        "docker_labels": "dockerLabels",
        "docker_security_options": "dockerSecurityOptions",
        "entry_point": "entryPoint",
        "environment": "environment",
        "environment_files": "environmentFiles",
        "essential": "essential",
        "extra_hosts": "extraHosts",
        "gpu_count": "gpuCount",
        "health_check": "healthCheck",
        "hostname": "hostname",
        "inference_accelerator_resources": "inferenceAcceleratorResources",
        "linux_parameters": "linuxParameters",
        "logging": "logging",
        "memory_limit_mib": "memoryLimitMiB",
        "memory_reservation_mib": "memoryReservationMiB",
        "port_mappings": "portMappings",
        "privileged": "privileged",
        "readonly_root_filesystem": "readonlyRootFilesystem",
        "secrets": "secrets",
        "start_timeout": "startTimeout",
        "stop_timeout": "stopTimeout",
        "user": "user",
        "working_directory": "workingDirectory",
        "firelens_config": "firelensConfig",
    },
)
class FirelensLogRouterDefinitionOptions(ContainerDefinitionOptions):
    def __init__(
        self,
        *,
        image: ContainerImage,
        command: typing.Optional[typing.Sequence[builtins.str]] = None,
        container_name: typing.Optional[builtins.str] = None,
        cpu: typing.Optional[jsii.Number] = None,
        disable_networking: typing.Optional[builtins.bool] = None,
        dns_search_domains: typing.Optional[typing.Sequence[builtins.str]] = None,
        dns_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        docker_labels: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        docker_security_options: typing.Optional[typing.Sequence[builtins.str]] = None,
        entry_point: typing.Optional[typing.Sequence[builtins.str]] = None,
        environment: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        environment_files: typing.Optional[typing.Sequence[EnvironmentFile]] = None,
        essential: typing.Optional[builtins.bool] = None,
        extra_hosts: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        gpu_count: typing.Optional[jsii.Number] = None,
        health_check: typing.Optional["HealthCheck"] = None,
        hostname: typing.Optional[builtins.str] = None,
        inference_accelerator_resources: typing.Optional[typing.Sequence[builtins.str]] = None,
        linux_parameters: typing.Optional["LinuxParameters"] = None,
        logging: typing.Optional["LogDriver"] = None,
        memory_limit_mib: typing.Optional[jsii.Number] = None,
        memory_reservation_mib: typing.Optional[jsii.Number] = None,
        port_mappings: typing.Optional[typing.Sequence["PortMapping"]] = None,
        privileged: typing.Optional[builtins.bool] = None,
        readonly_root_filesystem: typing.Optional[builtins.bool] = None,
        secrets: typing.Optional[typing.Mapping[builtins.str, "Secret"]] = None,
        start_timeout: typing.Optional[_Duration_070aa057] = None,
        stop_timeout: typing.Optional[_Duration_070aa057] = None,
        user: typing.Optional[builtins.str] = None,
        working_directory: typing.Optional[builtins.str] = None,
        firelens_config: FirelensConfig,
    ) -> None:
        '''(experimental) The options for creating a firelens log router.

        :param image: (experimental) The image used to start a container. This string is passed directly to the Docker daemon. Images in the Docker Hub registry are available by default. Other repositories are specified with either repository-url/image:tag or repository-url/image@digest. TODO: Update these to specify using classes of IContainerImage
        :param command: (experimental) The command that is passed to the container. If you provide a shell command as a single string, you have to quote command-line arguments. Default: - CMD value built into container image.
        :param container_name: (experimental) The name of the container. Default: - id of node associated with ContainerDefinition.
        :param cpu: (experimental) The minimum number of CPU units to reserve for the container. Default: - No minimum CPU units reserved.
        :param disable_networking: (experimental) Specifies whether networking is disabled within the container. When this parameter is true, networking is disabled within the container. Default: false
        :param dns_search_domains: (experimental) A list of DNS search domains that are presented to the container. Default: - No search domains.
        :param dns_servers: (experimental) A list of DNS servers that are presented to the container. Default: - Default DNS servers.
        :param docker_labels: (experimental) A key/value map of labels to add to the container. Default: - No labels.
        :param docker_security_options: (experimental) A list of strings to provide custom labels for SELinux and AppArmor multi-level security systems. Default: - No security labels.
        :param entry_point: (experimental) The ENTRYPOINT value to pass to the container. Default: - Entry point configured in container.
        :param environment: (experimental) The environment variables to pass to the container. Default: - No environment variables.
        :param environment_files: (experimental) The environment files to pass to the container. Default: - No environment files.
        :param essential: (experimental) Specifies whether the container is marked essential. If the essential parameter of a container is marked as true, and that container fails or stops for any reason, all other containers that are part of the task are stopped. If the essential parameter of a container is marked as false, then its failure does not affect the rest of the containers in a task. All tasks must have at least one essential container. If this parameter is omitted, a container is assumed to be essential. Default: true
        :param extra_hosts: (experimental) A list of hostnames and IP address mappings to append to the /etc/hosts file on the container. Default: - No extra hosts.
        :param gpu_count: (experimental) The number of GPUs assigned to the container. Default: - No GPUs assigned.
        :param health_check: (experimental) The health check command and associated configuration parameters for the container. Default: - Health check configuration from container.
        :param hostname: (experimental) The hostname to use for your container. Default: - Automatic hostname.
        :param inference_accelerator_resources: (experimental) The inference accelerators referenced by the container. Default: - No inference accelerators assigned.
        :param linux_parameters: (experimental) Linux-specific modifications that are applied to the container, such as Linux kernel capabilities. For more information see `KernelCapabilities <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_KernelCapabilities.html>`_. Default: - No Linux parameters.
        :param logging: (experimental) The log configuration specification for the container. Default: - Containers use the same logging driver that the Docker daemon uses.
        :param memory_limit_mib: (experimental) The amount (in MiB) of memory to present to the container. If your container attempts to exceed the allocated memory, the container is terminated. At least one of memoryLimitMiB and memoryReservationMiB is required for non-Fargate services. Default: - No memory limit.
        :param memory_reservation_mib: (experimental) The soft limit (in MiB) of memory to reserve for the container. When system memory is under heavy contention, Docker attempts to keep the container memory to this soft limit. However, your container can consume more memory when it needs to, up to either the hard limit specified with the memory parameter (if applicable), or all of the available memory on the container instance, whichever comes first. At least one of memoryLimitMiB and memoryReservationMiB is required for non-Fargate services. Default: - No memory reserved.
        :param port_mappings: (experimental) The port mappings to add to the container definition. Default: - No ports are mapped.
        :param privileged: (experimental) Specifies whether the container is marked as privileged. When this parameter is true, the container is given elevated privileges on the host container instance (similar to the root user). Default: false
        :param readonly_root_filesystem: (experimental) When this parameter is true, the container is given read-only access to its root file system. Default: false
        :param secrets: (experimental) The secret environment variables to pass to the container. Default: - No secret environment variables.
        :param start_timeout: (experimental) Time duration (in seconds) to wait before giving up on resolving dependencies for a container. Default: - none
        :param stop_timeout: (experimental) Time duration (in seconds) to wait before the container is forcefully killed if it doesn't exit normally on its own. Default: - none
        :param user: (experimental) The user name to use inside the container. Default: root
        :param working_directory: (experimental) The working directory in which to run commands inside the container. Default: /
        :param firelens_config: (experimental) Firelens configuration.

        :stability: experimental
        '''
        if isinstance(health_check, dict):
            health_check = HealthCheck(**health_check)
        if isinstance(firelens_config, dict):
            firelens_config = FirelensConfig(**firelens_config)
        self._values: typing.Dict[str, typing.Any] = {
            "image": image,
            "firelens_config": firelens_config,
        }
        if command is not None:
            self._values["command"] = command
        if container_name is not None:
            self._values["container_name"] = container_name
        if cpu is not None:
            self._values["cpu"] = cpu
        if disable_networking is not None:
            self._values["disable_networking"] = disable_networking
        if dns_search_domains is not None:
            self._values["dns_search_domains"] = dns_search_domains
        if dns_servers is not None:
            self._values["dns_servers"] = dns_servers
        if docker_labels is not None:
            self._values["docker_labels"] = docker_labels
        if docker_security_options is not None:
            self._values["docker_security_options"] = docker_security_options
        if entry_point is not None:
            self._values["entry_point"] = entry_point
        if environment is not None:
            self._values["environment"] = environment
        if environment_files is not None:
            self._values["environment_files"] = environment_files
        if essential is not None:
            self._values["essential"] = essential
        if extra_hosts is not None:
            self._values["extra_hosts"] = extra_hosts
        if gpu_count is not None:
            self._values["gpu_count"] = gpu_count
        if health_check is not None:
            self._values["health_check"] = health_check
        if hostname is not None:
            self._values["hostname"] = hostname
        if inference_accelerator_resources is not None:
            self._values["inference_accelerator_resources"] = inference_accelerator_resources
        if linux_parameters is not None:
            self._values["linux_parameters"] = linux_parameters
        if logging is not None:
            self._values["logging"] = logging
        if memory_limit_mib is not None:
            self._values["memory_limit_mib"] = memory_limit_mib
        if memory_reservation_mib is not None:
            self._values["memory_reservation_mib"] = memory_reservation_mib
        if port_mappings is not None:
            self._values["port_mappings"] = port_mappings
        if privileged is not None:
            self._values["privileged"] = privileged
        if readonly_root_filesystem is not None:
            self._values["readonly_root_filesystem"] = readonly_root_filesystem
        if secrets is not None:
            self._values["secrets"] = secrets
        if start_timeout is not None:
            self._values["start_timeout"] = start_timeout
        if stop_timeout is not None:
            self._values["stop_timeout"] = stop_timeout
        if user is not None:
            self._values["user"] = user
        if working_directory is not None:
            self._values["working_directory"] = working_directory

    @builtins.property
    def image(self) -> ContainerImage:
        '''(experimental) The image used to start a container.

        This string is passed directly to the Docker daemon.
        Images in the Docker Hub registry are available by default.
        Other repositories are specified with either repository-url/image:tag or repository-url/image@digest.
        TODO: Update these to specify using classes of IContainerImage

        :stability: experimental
        '''
        result = self._values.get("image")
        assert result is not None, "Required property 'image' is missing"
        return typing.cast(ContainerImage, result)

    @builtins.property
    def command(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) The command that is passed to the container.

        If you provide a shell command as a single string, you have to quote command-line arguments.

        :default: - CMD value built into container image.

        :stability: experimental
        '''
        result = self._values.get("command")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def container_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name of the container.

        :default: - id of node associated with ContainerDefinition.

        :stability: experimental
        '''
        result = self._values.get("container_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def cpu(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The minimum number of CPU units to reserve for the container.

        :default: - No minimum CPU units reserved.

        :stability: experimental
        '''
        result = self._values.get("cpu")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def disable_networking(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specifies whether networking is disabled within the container.

        When this parameter is true, networking is disabled within the container.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("disable_networking")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def dns_search_domains(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) A list of DNS search domains that are presented to the container.

        :default: - No search domains.

        :stability: experimental
        '''
        result = self._values.get("dns_search_domains")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def dns_servers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) A list of DNS servers that are presented to the container.

        :default: - Default DNS servers.

        :stability: experimental
        '''
        result = self._values.get("dns_servers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def docker_labels(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''(experimental) A key/value map of labels to add to the container.

        :default: - No labels.

        :stability: experimental
        '''
        result = self._values.get("docker_labels")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def docker_security_options(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) A list of strings to provide custom labels for SELinux and AppArmor multi-level security systems.

        :default: - No security labels.

        :stability: experimental
        '''
        result = self._values.get("docker_security_options")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def entry_point(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) The ENTRYPOINT value to pass to the container.

        :default: - Entry point configured in container.

        :see: https://docs.docker.com/engine/reference/builder/#entrypoint
        :stability: experimental
        '''
        result = self._values.get("entry_point")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def environment(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''(experimental) The environment variables to pass to the container.

        :default: - No environment variables.

        :stability: experimental
        '''
        result = self._values.get("environment")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def environment_files(self) -> typing.Optional[typing.List[EnvironmentFile]]:
        '''(experimental) The environment files to pass to the container.

        :default: - No environment files.

        :see: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/taskdef-envfiles.html
        :stability: experimental
        '''
        result = self._values.get("environment_files")
        return typing.cast(typing.Optional[typing.List[EnvironmentFile]], result)

    @builtins.property
    def essential(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specifies whether the container is marked essential.

        If the essential parameter of a container is marked as true, and that container fails
        or stops for any reason, all other containers that are part of the task are stopped.
        If the essential parameter of a container is marked as false, then its failure does not
        affect the rest of the containers in a task. All tasks must have at least one essential container.

        If this parameter is omitted, a container is assumed to be essential.

        :default: true

        :stability: experimental
        '''
        result = self._values.get("essential")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def extra_hosts(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''(experimental) A list of hostnames and IP address mappings to append to the /etc/hosts file on the container.

        :default: - No extra hosts.

        :stability: experimental
        '''
        result = self._values.get("extra_hosts")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def gpu_count(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The number of GPUs assigned to the container.

        :default: - No GPUs assigned.

        :stability: experimental
        '''
        result = self._values.get("gpu_count")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def health_check(self) -> typing.Optional["HealthCheck"]:
        '''(experimental) The health check command and associated configuration parameters for the container.

        :default: - Health check configuration from container.

        :stability: experimental
        '''
        result = self._values.get("health_check")
        return typing.cast(typing.Optional["HealthCheck"], result)

    @builtins.property
    def hostname(self) -> typing.Optional[builtins.str]:
        '''(experimental) The hostname to use for your container.

        :default: - Automatic hostname.

        :stability: experimental
        '''
        result = self._values.get("hostname")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def inference_accelerator_resources(
        self,
    ) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) The inference accelerators referenced by the container.

        :default: - No inference accelerators assigned.

        :stability: experimental
        '''
        result = self._values.get("inference_accelerator_resources")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def linux_parameters(self) -> typing.Optional["LinuxParameters"]:
        '''(experimental) Linux-specific modifications that are applied to the container, such as Linux kernel capabilities.

        For more information see `KernelCapabilities <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_KernelCapabilities.html>`_.

        :default: - No Linux parameters.

        :stability: experimental
        '''
        result = self._values.get("linux_parameters")
        return typing.cast(typing.Optional["LinuxParameters"], result)

    @builtins.property
    def logging(self) -> typing.Optional["LogDriver"]:
        '''(experimental) The log configuration specification for the container.

        :default: - Containers use the same logging driver that the Docker daemon uses.

        :stability: experimental
        '''
        result = self._values.get("logging")
        return typing.cast(typing.Optional["LogDriver"], result)

    @builtins.property
    def memory_limit_mib(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The amount (in MiB) of memory to present to the container.

        If your container attempts to exceed the allocated memory, the container
        is terminated.

        At least one of memoryLimitMiB and memoryReservationMiB is required for non-Fargate services.

        :default: - No memory limit.

        :stability: experimental
        '''
        result = self._values.get("memory_limit_mib")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def memory_reservation_mib(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The soft limit (in MiB) of memory to reserve for the container.

        When system memory is under heavy contention, Docker attempts to keep the
        container memory to this soft limit. However, your container can consume more
        memory when it needs to, up to either the hard limit specified with the memory
        parameter (if applicable), or all of the available memory on the container
        instance, whichever comes first.

        At least one of memoryLimitMiB and memoryReservationMiB is required for non-Fargate services.

        :default: - No memory reserved.

        :stability: experimental
        '''
        result = self._values.get("memory_reservation_mib")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def port_mappings(self) -> typing.Optional[typing.List["PortMapping"]]:
        '''(experimental) The port mappings to add to the container definition.

        :default: - No ports are mapped.

        :stability: experimental
        '''
        result = self._values.get("port_mappings")
        return typing.cast(typing.Optional[typing.List["PortMapping"]], result)

    @builtins.property
    def privileged(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specifies whether the container is marked as privileged.

        When this parameter is true, the container is given elevated privileges on the host container instance (similar to the root user).

        :default: false

        :stability: experimental
        '''
        result = self._values.get("privileged")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def readonly_root_filesystem(self) -> typing.Optional[builtins.bool]:
        '''(experimental) When this parameter is true, the container is given read-only access to its root file system.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("readonly_root_filesystem")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def secrets(self) -> typing.Optional[typing.Mapping[builtins.str, "Secret"]]:
        '''(experimental) The secret environment variables to pass to the container.

        :default: - No secret environment variables.

        :stability: experimental
        '''
        result = self._values.get("secrets")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, "Secret"]], result)

    @builtins.property
    def start_timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Time duration (in seconds) to wait before giving up on resolving dependencies for a container.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("start_timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def stop_timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Time duration (in seconds) to wait before the container is forcefully killed if it doesn't exit normally on its own.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("stop_timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def user(self) -> typing.Optional[builtins.str]:
        '''(experimental) The user name to use inside the container.

        :default: root

        :stability: experimental
        '''
        result = self._values.get("user")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def working_directory(self) -> typing.Optional[builtins.str]:
        '''(experimental) The working directory in which to run commands inside the container.

        :default: /

        :stability: experimental
        '''
        result = self._values.get("working_directory")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def firelens_config(self) -> FirelensConfig:
        '''(experimental) Firelens configuration.

        :stability: experimental
        '''
        result = self._values.get("firelens_config")
        assert result is not None, "Required property 'firelens_config' is missing"
        return typing.cast(FirelensConfig, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "FirelensLogRouterDefinitionOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.FirelensLogRouterProps",
    jsii_struct_bases=[ContainerDefinitionProps],
    name_mapping={
        "image": "image",
        "command": "command",
        "container_name": "containerName",
        "cpu": "cpu",
        "disable_networking": "disableNetworking",
        "dns_search_domains": "dnsSearchDomains",
        "dns_servers": "dnsServers",
        "docker_labels": "dockerLabels",
        "docker_security_options": "dockerSecurityOptions",
        "entry_point": "entryPoint",
        "environment": "environment",
        "environment_files": "environmentFiles",
        "essential": "essential",
        "extra_hosts": "extraHosts",
        "gpu_count": "gpuCount",
        "health_check": "healthCheck",
        "hostname": "hostname",
        "inference_accelerator_resources": "inferenceAcceleratorResources",
        "linux_parameters": "linuxParameters",
        "logging": "logging",
        "memory_limit_mib": "memoryLimitMiB",
        "memory_reservation_mib": "memoryReservationMiB",
        "port_mappings": "portMappings",
        "privileged": "privileged",
        "readonly_root_filesystem": "readonlyRootFilesystem",
        "secrets": "secrets",
        "start_timeout": "startTimeout",
        "stop_timeout": "stopTimeout",
        "user": "user",
        "working_directory": "workingDirectory",
        "task_definition": "taskDefinition",
        "firelens_config": "firelensConfig",
    },
)
class FirelensLogRouterProps(ContainerDefinitionProps):
    def __init__(
        self,
        *,
        image: ContainerImage,
        command: typing.Optional[typing.Sequence[builtins.str]] = None,
        container_name: typing.Optional[builtins.str] = None,
        cpu: typing.Optional[jsii.Number] = None,
        disable_networking: typing.Optional[builtins.bool] = None,
        dns_search_domains: typing.Optional[typing.Sequence[builtins.str]] = None,
        dns_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        docker_labels: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        docker_security_options: typing.Optional[typing.Sequence[builtins.str]] = None,
        entry_point: typing.Optional[typing.Sequence[builtins.str]] = None,
        environment: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        environment_files: typing.Optional[typing.Sequence[EnvironmentFile]] = None,
        essential: typing.Optional[builtins.bool] = None,
        extra_hosts: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        gpu_count: typing.Optional[jsii.Number] = None,
        health_check: typing.Optional["HealthCheck"] = None,
        hostname: typing.Optional[builtins.str] = None,
        inference_accelerator_resources: typing.Optional[typing.Sequence[builtins.str]] = None,
        linux_parameters: typing.Optional["LinuxParameters"] = None,
        logging: typing.Optional["LogDriver"] = None,
        memory_limit_mib: typing.Optional[jsii.Number] = None,
        memory_reservation_mib: typing.Optional[jsii.Number] = None,
        port_mappings: typing.Optional[typing.Sequence["PortMapping"]] = None,
        privileged: typing.Optional[builtins.bool] = None,
        readonly_root_filesystem: typing.Optional[builtins.bool] = None,
        secrets: typing.Optional[typing.Mapping[builtins.str, "Secret"]] = None,
        start_timeout: typing.Optional[_Duration_070aa057] = None,
        stop_timeout: typing.Optional[_Duration_070aa057] = None,
        user: typing.Optional[builtins.str] = None,
        working_directory: typing.Optional[builtins.str] = None,
        task_definition: "TaskDefinition",
        firelens_config: FirelensConfig,
    ) -> None:
        '''(experimental) The properties in a firelens log router.

        :param image: (experimental) The image used to start a container. This string is passed directly to the Docker daemon. Images in the Docker Hub registry are available by default. Other repositories are specified with either repository-url/image:tag or repository-url/image@digest. TODO: Update these to specify using classes of IContainerImage
        :param command: (experimental) The command that is passed to the container. If you provide a shell command as a single string, you have to quote command-line arguments. Default: - CMD value built into container image.
        :param container_name: (experimental) The name of the container. Default: - id of node associated with ContainerDefinition.
        :param cpu: (experimental) The minimum number of CPU units to reserve for the container. Default: - No minimum CPU units reserved.
        :param disable_networking: (experimental) Specifies whether networking is disabled within the container. When this parameter is true, networking is disabled within the container. Default: false
        :param dns_search_domains: (experimental) A list of DNS search domains that are presented to the container. Default: - No search domains.
        :param dns_servers: (experimental) A list of DNS servers that are presented to the container. Default: - Default DNS servers.
        :param docker_labels: (experimental) A key/value map of labels to add to the container. Default: - No labels.
        :param docker_security_options: (experimental) A list of strings to provide custom labels for SELinux and AppArmor multi-level security systems. Default: - No security labels.
        :param entry_point: (experimental) The ENTRYPOINT value to pass to the container. Default: - Entry point configured in container.
        :param environment: (experimental) The environment variables to pass to the container. Default: - No environment variables.
        :param environment_files: (experimental) The environment files to pass to the container. Default: - No environment files.
        :param essential: (experimental) Specifies whether the container is marked essential. If the essential parameter of a container is marked as true, and that container fails or stops for any reason, all other containers that are part of the task are stopped. If the essential parameter of a container is marked as false, then its failure does not affect the rest of the containers in a task. All tasks must have at least one essential container. If this parameter is omitted, a container is assumed to be essential. Default: true
        :param extra_hosts: (experimental) A list of hostnames and IP address mappings to append to the /etc/hosts file on the container. Default: - No extra hosts.
        :param gpu_count: (experimental) The number of GPUs assigned to the container. Default: - No GPUs assigned.
        :param health_check: (experimental) The health check command and associated configuration parameters for the container. Default: - Health check configuration from container.
        :param hostname: (experimental) The hostname to use for your container. Default: - Automatic hostname.
        :param inference_accelerator_resources: (experimental) The inference accelerators referenced by the container. Default: - No inference accelerators assigned.
        :param linux_parameters: (experimental) Linux-specific modifications that are applied to the container, such as Linux kernel capabilities. For more information see `KernelCapabilities <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_KernelCapabilities.html>`_. Default: - No Linux parameters.
        :param logging: (experimental) The log configuration specification for the container. Default: - Containers use the same logging driver that the Docker daemon uses.
        :param memory_limit_mib: (experimental) The amount (in MiB) of memory to present to the container. If your container attempts to exceed the allocated memory, the container is terminated. At least one of memoryLimitMiB and memoryReservationMiB is required for non-Fargate services. Default: - No memory limit.
        :param memory_reservation_mib: (experimental) The soft limit (in MiB) of memory to reserve for the container. When system memory is under heavy contention, Docker attempts to keep the container memory to this soft limit. However, your container can consume more memory when it needs to, up to either the hard limit specified with the memory parameter (if applicable), or all of the available memory on the container instance, whichever comes first. At least one of memoryLimitMiB and memoryReservationMiB is required for non-Fargate services. Default: - No memory reserved.
        :param port_mappings: (experimental) The port mappings to add to the container definition. Default: - No ports are mapped.
        :param privileged: (experimental) Specifies whether the container is marked as privileged. When this parameter is true, the container is given elevated privileges on the host container instance (similar to the root user). Default: false
        :param readonly_root_filesystem: (experimental) When this parameter is true, the container is given read-only access to its root file system. Default: false
        :param secrets: (experimental) The secret environment variables to pass to the container. Default: - No secret environment variables.
        :param start_timeout: (experimental) Time duration (in seconds) to wait before giving up on resolving dependencies for a container. Default: - none
        :param stop_timeout: (experimental) Time duration (in seconds) to wait before the container is forcefully killed if it doesn't exit normally on its own. Default: - none
        :param user: (experimental) The user name to use inside the container. Default: root
        :param working_directory: (experimental) The working directory in which to run commands inside the container. Default: /
        :param task_definition: (experimental) The name of the task definition that includes this container definition. [disable-awslint:ref-via-interface]
        :param firelens_config: (experimental) Firelens configuration.

        :stability: experimental
        '''
        if isinstance(health_check, dict):
            health_check = HealthCheck(**health_check)
        if isinstance(firelens_config, dict):
            firelens_config = FirelensConfig(**firelens_config)
        self._values: typing.Dict[str, typing.Any] = {
            "image": image,
            "task_definition": task_definition,
            "firelens_config": firelens_config,
        }
        if command is not None:
            self._values["command"] = command
        if container_name is not None:
            self._values["container_name"] = container_name
        if cpu is not None:
            self._values["cpu"] = cpu
        if disable_networking is not None:
            self._values["disable_networking"] = disable_networking
        if dns_search_domains is not None:
            self._values["dns_search_domains"] = dns_search_domains
        if dns_servers is not None:
            self._values["dns_servers"] = dns_servers
        if docker_labels is not None:
            self._values["docker_labels"] = docker_labels
        if docker_security_options is not None:
            self._values["docker_security_options"] = docker_security_options
        if entry_point is not None:
            self._values["entry_point"] = entry_point
        if environment is not None:
            self._values["environment"] = environment
        if environment_files is not None:
            self._values["environment_files"] = environment_files
        if essential is not None:
            self._values["essential"] = essential
        if extra_hosts is not None:
            self._values["extra_hosts"] = extra_hosts
        if gpu_count is not None:
            self._values["gpu_count"] = gpu_count
        if health_check is not None:
            self._values["health_check"] = health_check
        if hostname is not None:
            self._values["hostname"] = hostname
        if inference_accelerator_resources is not None:
            self._values["inference_accelerator_resources"] = inference_accelerator_resources
        if linux_parameters is not None:
            self._values["linux_parameters"] = linux_parameters
        if logging is not None:
            self._values["logging"] = logging
        if memory_limit_mib is not None:
            self._values["memory_limit_mib"] = memory_limit_mib
        if memory_reservation_mib is not None:
            self._values["memory_reservation_mib"] = memory_reservation_mib
        if port_mappings is not None:
            self._values["port_mappings"] = port_mappings
        if privileged is not None:
            self._values["privileged"] = privileged
        if readonly_root_filesystem is not None:
            self._values["readonly_root_filesystem"] = readonly_root_filesystem
        if secrets is not None:
            self._values["secrets"] = secrets
        if start_timeout is not None:
            self._values["start_timeout"] = start_timeout
        if stop_timeout is not None:
            self._values["stop_timeout"] = stop_timeout
        if user is not None:
            self._values["user"] = user
        if working_directory is not None:
            self._values["working_directory"] = working_directory

    @builtins.property
    def image(self) -> ContainerImage:
        '''(experimental) The image used to start a container.

        This string is passed directly to the Docker daemon.
        Images in the Docker Hub registry are available by default.
        Other repositories are specified with either repository-url/image:tag or repository-url/image@digest.
        TODO: Update these to specify using classes of IContainerImage

        :stability: experimental
        '''
        result = self._values.get("image")
        assert result is not None, "Required property 'image' is missing"
        return typing.cast(ContainerImage, result)

    @builtins.property
    def command(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) The command that is passed to the container.

        If you provide a shell command as a single string, you have to quote command-line arguments.

        :default: - CMD value built into container image.

        :stability: experimental
        '''
        result = self._values.get("command")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def container_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name of the container.

        :default: - id of node associated with ContainerDefinition.

        :stability: experimental
        '''
        result = self._values.get("container_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def cpu(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The minimum number of CPU units to reserve for the container.

        :default: - No minimum CPU units reserved.

        :stability: experimental
        '''
        result = self._values.get("cpu")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def disable_networking(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specifies whether networking is disabled within the container.

        When this parameter is true, networking is disabled within the container.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("disable_networking")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def dns_search_domains(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) A list of DNS search domains that are presented to the container.

        :default: - No search domains.

        :stability: experimental
        '''
        result = self._values.get("dns_search_domains")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def dns_servers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) A list of DNS servers that are presented to the container.

        :default: - Default DNS servers.

        :stability: experimental
        '''
        result = self._values.get("dns_servers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def docker_labels(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''(experimental) A key/value map of labels to add to the container.

        :default: - No labels.

        :stability: experimental
        '''
        result = self._values.get("docker_labels")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def docker_security_options(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) A list of strings to provide custom labels for SELinux and AppArmor multi-level security systems.

        :default: - No security labels.

        :stability: experimental
        '''
        result = self._values.get("docker_security_options")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def entry_point(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) The ENTRYPOINT value to pass to the container.

        :default: - Entry point configured in container.

        :see: https://docs.docker.com/engine/reference/builder/#entrypoint
        :stability: experimental
        '''
        result = self._values.get("entry_point")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def environment(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''(experimental) The environment variables to pass to the container.

        :default: - No environment variables.

        :stability: experimental
        '''
        result = self._values.get("environment")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def environment_files(self) -> typing.Optional[typing.List[EnvironmentFile]]:
        '''(experimental) The environment files to pass to the container.

        :default: - No environment files.

        :see: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/taskdef-envfiles.html
        :stability: experimental
        '''
        result = self._values.get("environment_files")
        return typing.cast(typing.Optional[typing.List[EnvironmentFile]], result)

    @builtins.property
    def essential(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specifies whether the container is marked essential.

        If the essential parameter of a container is marked as true, and that container fails
        or stops for any reason, all other containers that are part of the task are stopped.
        If the essential parameter of a container is marked as false, then its failure does not
        affect the rest of the containers in a task. All tasks must have at least one essential container.

        If this parameter is omitted, a container is assumed to be essential.

        :default: true

        :stability: experimental
        '''
        result = self._values.get("essential")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def extra_hosts(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''(experimental) A list of hostnames and IP address mappings to append to the /etc/hosts file on the container.

        :default: - No extra hosts.

        :stability: experimental
        '''
        result = self._values.get("extra_hosts")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def gpu_count(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The number of GPUs assigned to the container.

        :default: - No GPUs assigned.

        :stability: experimental
        '''
        result = self._values.get("gpu_count")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def health_check(self) -> typing.Optional["HealthCheck"]:
        '''(experimental) The health check command and associated configuration parameters for the container.

        :default: - Health check configuration from container.

        :stability: experimental
        '''
        result = self._values.get("health_check")
        return typing.cast(typing.Optional["HealthCheck"], result)

    @builtins.property
    def hostname(self) -> typing.Optional[builtins.str]:
        '''(experimental) The hostname to use for your container.

        :default: - Automatic hostname.

        :stability: experimental
        '''
        result = self._values.get("hostname")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def inference_accelerator_resources(
        self,
    ) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) The inference accelerators referenced by the container.

        :default: - No inference accelerators assigned.

        :stability: experimental
        '''
        result = self._values.get("inference_accelerator_resources")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def linux_parameters(self) -> typing.Optional["LinuxParameters"]:
        '''(experimental) Linux-specific modifications that are applied to the container, such as Linux kernel capabilities.

        For more information see `KernelCapabilities <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_KernelCapabilities.html>`_.

        :default: - No Linux parameters.

        :stability: experimental
        '''
        result = self._values.get("linux_parameters")
        return typing.cast(typing.Optional["LinuxParameters"], result)

    @builtins.property
    def logging(self) -> typing.Optional["LogDriver"]:
        '''(experimental) The log configuration specification for the container.

        :default: - Containers use the same logging driver that the Docker daemon uses.

        :stability: experimental
        '''
        result = self._values.get("logging")
        return typing.cast(typing.Optional["LogDriver"], result)

    @builtins.property
    def memory_limit_mib(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The amount (in MiB) of memory to present to the container.

        If your container attempts to exceed the allocated memory, the container
        is terminated.

        At least one of memoryLimitMiB and memoryReservationMiB is required for non-Fargate services.

        :default: - No memory limit.

        :stability: experimental
        '''
        result = self._values.get("memory_limit_mib")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def memory_reservation_mib(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The soft limit (in MiB) of memory to reserve for the container.

        When system memory is under heavy contention, Docker attempts to keep the
        container memory to this soft limit. However, your container can consume more
        memory when it needs to, up to either the hard limit specified with the memory
        parameter (if applicable), or all of the available memory on the container
        instance, whichever comes first.

        At least one of memoryLimitMiB and memoryReservationMiB is required for non-Fargate services.

        :default: - No memory reserved.

        :stability: experimental
        '''
        result = self._values.get("memory_reservation_mib")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def port_mappings(self) -> typing.Optional[typing.List["PortMapping"]]:
        '''(experimental) The port mappings to add to the container definition.

        :default: - No ports are mapped.

        :stability: experimental
        '''
        result = self._values.get("port_mappings")
        return typing.cast(typing.Optional[typing.List["PortMapping"]], result)

    @builtins.property
    def privileged(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specifies whether the container is marked as privileged.

        When this parameter is true, the container is given elevated privileges on the host container instance (similar to the root user).

        :default: false

        :stability: experimental
        '''
        result = self._values.get("privileged")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def readonly_root_filesystem(self) -> typing.Optional[builtins.bool]:
        '''(experimental) When this parameter is true, the container is given read-only access to its root file system.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("readonly_root_filesystem")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def secrets(self) -> typing.Optional[typing.Mapping[builtins.str, "Secret"]]:
        '''(experimental) The secret environment variables to pass to the container.

        :default: - No secret environment variables.

        :stability: experimental
        '''
        result = self._values.get("secrets")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, "Secret"]], result)

    @builtins.property
    def start_timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Time duration (in seconds) to wait before giving up on resolving dependencies for a container.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("start_timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def stop_timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Time duration (in seconds) to wait before the container is forcefully killed if it doesn't exit normally on its own.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("stop_timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def user(self) -> typing.Optional[builtins.str]:
        '''(experimental) The user name to use inside the container.

        :default: root

        :stability: experimental
        '''
        result = self._values.get("user")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def working_directory(self) -> typing.Optional[builtins.str]:
        '''(experimental) The working directory in which to run commands inside the container.

        :default: /

        :stability: experimental
        '''
        result = self._values.get("working_directory")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def task_definition(self) -> "TaskDefinition":
        '''(experimental) The name of the task definition that includes this container definition.

        [disable-awslint:ref-via-interface]

        :stability: experimental
        '''
        result = self._values.get("task_definition")
        assert result is not None, "Required property 'task_definition' is missing"
        return typing.cast("TaskDefinition", result)

    @builtins.property
    def firelens_config(self) -> FirelensConfig:
        '''(experimental) Firelens configuration.

        :stability: experimental
        '''
        result = self._values.get("firelens_config")
        assert result is not None, "Required property 'firelens_config' is missing"
        return typing.cast(FirelensConfig, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "FirelensLogRouterProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_ecs.FirelensLogRouterType")
class FirelensLogRouterType(enum.Enum):
    '''(experimental) Firelens log router type, fluentbit or fluentd.

    https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html

    :stability: experimental
    '''

    FLUENTBIT = "FLUENTBIT"
    '''(experimental) fluentbit.

    :stability: experimental
    '''
    FLUENTD = "FLUENTD"
    '''(experimental) fluentd.

    :stability: experimental
    '''


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.FirelensOptions",
    jsii_struct_bases=[],
    name_mapping={
        "config_file_value": "configFileValue",
        "config_file_type": "configFileType",
        "enable_ecs_log_metadata": "enableECSLogMetadata",
    },
)
class FirelensOptions:
    def __init__(
        self,
        *,
        config_file_value: builtins.str,
        config_file_type: typing.Optional[FirelensConfigFileType] = None,
        enable_ecs_log_metadata: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''(experimental) The options for firelens log router https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html#firelens-taskdef-customconfig.

        :param config_file_value: (experimental) Custom configuration file, S3 ARN or a file path.
        :param config_file_type: (experimental) Custom configuration file, s3 or file. Default: - determined by checking configFileValue with S3 ARN.
        :param enable_ecs_log_metadata: (experimental) By default, Amazon ECS adds additional fields in your log entries that help identify the source of the logs. You can disable this action by setting enable-ecs-log-metadata to false. Default: - true

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "config_file_value": config_file_value,
        }
        if config_file_type is not None:
            self._values["config_file_type"] = config_file_type
        if enable_ecs_log_metadata is not None:
            self._values["enable_ecs_log_metadata"] = enable_ecs_log_metadata

    @builtins.property
    def config_file_value(self) -> builtins.str:
        '''(experimental) Custom configuration file, S3 ARN or a file path.

        :stability: experimental
        '''
        result = self._values.get("config_file_value")
        assert result is not None, "Required property 'config_file_value' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def config_file_type(self) -> typing.Optional[FirelensConfigFileType]:
        '''(experimental) Custom configuration file, s3 or file.

        :default: - determined by checking configFileValue with S3 ARN.

        :stability: experimental
        '''
        result = self._values.get("config_file_type")
        return typing.cast(typing.Optional[FirelensConfigFileType], result)

    @builtins.property
    def enable_ecs_log_metadata(self) -> typing.Optional[builtins.bool]:
        '''(experimental) By default, Amazon ECS adds additional fields in your log entries that help identify the source of the logs.

        You can disable this action by setting enable-ecs-log-metadata to false.

        :default: - true

        :stability: experimental
        '''
        result = self._values.get("enable_ecs_log_metadata")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "FirelensOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.FluentdLogDriverProps",
    jsii_struct_bases=[BaseLogDriverProps],
    name_mapping={
        "env": "env",
        "env_regex": "envRegex",
        "labels": "labels",
        "tag": "tag",
        "address": "address",
        "async_connect": "asyncConnect",
        "buffer_limit": "bufferLimit",
        "max_retries": "maxRetries",
        "retry_wait": "retryWait",
        "sub_second_precision": "subSecondPrecision",
    },
)
class FluentdLogDriverProps(BaseLogDriverProps):
    def __init__(
        self,
        *,
        env: typing.Optional[typing.Sequence[builtins.str]] = None,
        env_regex: typing.Optional[builtins.str] = None,
        labels: typing.Optional[typing.Sequence[builtins.str]] = None,
        tag: typing.Optional[builtins.str] = None,
        address: typing.Optional[builtins.str] = None,
        async_connect: typing.Optional[builtins.bool] = None,
        buffer_limit: typing.Optional[jsii.Number] = None,
        max_retries: typing.Optional[jsii.Number] = None,
        retry_wait: typing.Optional[_Duration_070aa057] = None,
        sub_second_precision: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''(experimental) Specifies the fluentd log driver configuration options.

        `Source <https://docs.docker.com/config/containers/logging/fluentd/>`_

        :param env: (experimental) The env option takes an array of keys. If there is collision between label and env keys, the value of the env takes precedence. Adds additional fields to the extra attributes of a logging message. Default: - No env
        :param env_regex: (experimental) The env-regex option is similar to and compatible with env. Its value is a regular expression to match logging-related environment variables. It is used for advanced log tag options. Default: - No envRegex
        :param labels: (experimental) The labels option takes an array of keys. If there is collision between label and env keys, the value of the env takes precedence. Adds additional fields to the extra attributes of a logging message. Default: - No labels
        :param tag: (experimental) By default, Docker uses the first 12 characters of the container ID to tag log messages. Refer to the log tag option documentation for customizing the log tag format. Default: - The first 12 characters of the container ID
        :param address: (experimental) By default, the logging driver connects to localhost:24224. Supply the address option to connect to a different address. tcp(default) and unix sockets are supported. Default: - address not set.
        :param async_connect: (experimental) Docker connects to Fluentd in the background. Messages are buffered until the connection is established. Default: - false
        :param buffer_limit: (experimental) The amount of data to buffer before flushing to disk. Default: - The amount of RAM available to the container.
        :param max_retries: (experimental) The maximum number of retries. Default: - 4294967295 (2**32 - 1).
        :param retry_wait: (experimental) How long to wait between retries. Default: - 1 second
        :param sub_second_precision: (experimental) Generates event logs in nanosecond resolution. Default: - false

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if env is not None:
            self._values["env"] = env
        if env_regex is not None:
            self._values["env_regex"] = env_regex
        if labels is not None:
            self._values["labels"] = labels
        if tag is not None:
            self._values["tag"] = tag
        if address is not None:
            self._values["address"] = address
        if async_connect is not None:
            self._values["async_connect"] = async_connect
        if buffer_limit is not None:
            self._values["buffer_limit"] = buffer_limit
        if max_retries is not None:
            self._values["max_retries"] = max_retries
        if retry_wait is not None:
            self._values["retry_wait"] = retry_wait
        if sub_second_precision is not None:
            self._values["sub_second_precision"] = sub_second_precision

    @builtins.property
    def env(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) The env option takes an array of keys.

        If there is collision between
        label and env keys, the value of the env takes precedence. Adds additional fields
        to the extra attributes of a logging message.

        :default: - No env

        :stability: experimental
        '''
        result = self._values.get("env")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def env_regex(self) -> typing.Optional[builtins.str]:
        '''(experimental) The env-regex option is similar to and compatible with env.

        Its value is a regular
        expression to match logging-related environment variables. It is used for advanced
        log tag options.

        :default: - No envRegex

        :stability: experimental
        '''
        result = self._values.get("env_regex")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def labels(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) The labels option takes an array of keys.

        If there is collision
        between label and env keys, the value of the env takes precedence. Adds additional
        fields to the extra attributes of a logging message.

        :default: - No labels

        :stability: experimental
        '''
        result = self._values.get("labels")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def tag(self) -> typing.Optional[builtins.str]:
        '''(experimental) By default, Docker uses the first 12 characters of the container ID to tag log messages.

        Refer to the log tag option documentation for customizing the
        log tag format.

        :default: - The first 12 characters of the container ID

        :stability: experimental
        '''
        result = self._values.get("tag")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def address(self) -> typing.Optional[builtins.str]:
        '''(experimental) By default, the logging driver connects to localhost:24224.

        Supply the
        address option to connect to a different address. tcp(default) and unix
        sockets are supported.

        :default: - address not set.

        :stability: experimental
        '''
        result = self._values.get("address")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def async_connect(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Docker connects to Fluentd in the background.

        Messages are buffered until
        the connection is established.

        :default: - false

        :stability: experimental
        '''
        result = self._values.get("async_connect")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def buffer_limit(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The amount of data to buffer before flushing to disk.

        :default: - The amount of RAM available to the container.

        :stability: experimental
        '''
        result = self._values.get("buffer_limit")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def max_retries(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The maximum number of retries.

        :default: - 4294967295 (2**32 - 1).

        :stability: experimental
        '''
        result = self._values.get("max_retries")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def retry_wait(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) How long to wait between retries.

        :default: - 1 second

        :stability: experimental
        '''
        result = self._values.get("retry_wait")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def sub_second_precision(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Generates event logs in nanosecond resolution.

        :default: - false

        :stability: experimental
        '''
        result = self._values.get("sub_second_precision")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "FluentdLogDriverProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_ecs.GelfCompressionType")
class GelfCompressionType(enum.Enum):
    '''(experimental) The type of compression the GELF driver uses to compress each log message.

    :stability: experimental
    '''

    GZIP = "GZIP"
    '''
    :stability: experimental
    '''
    ZLIB = "ZLIB"
    '''
    :stability: experimental
    '''
    NONE = "NONE"
    '''
    :stability: experimental
    '''


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.GelfLogDriverProps",
    jsii_struct_bases=[BaseLogDriverProps],
    name_mapping={
        "env": "env",
        "env_regex": "envRegex",
        "labels": "labels",
        "tag": "tag",
        "address": "address",
        "compression_level": "compressionLevel",
        "compression_type": "compressionType",
        "tcp_max_reconnect": "tcpMaxReconnect",
        "tcp_reconnect_delay": "tcpReconnectDelay",
    },
)
class GelfLogDriverProps(BaseLogDriverProps):
    def __init__(
        self,
        *,
        env: typing.Optional[typing.Sequence[builtins.str]] = None,
        env_regex: typing.Optional[builtins.str] = None,
        labels: typing.Optional[typing.Sequence[builtins.str]] = None,
        tag: typing.Optional[builtins.str] = None,
        address: builtins.str,
        compression_level: typing.Optional[jsii.Number] = None,
        compression_type: typing.Optional[GelfCompressionType] = None,
        tcp_max_reconnect: typing.Optional[jsii.Number] = None,
        tcp_reconnect_delay: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''(experimental) Specifies the journald log driver configuration options.

        `Source <https://docs.docker.com/config/containers/logging/gelf/>`_

        :param env: (experimental) The env option takes an array of keys. If there is collision between label and env keys, the value of the env takes precedence. Adds additional fields to the extra attributes of a logging message. Default: - No env
        :param env_regex: (experimental) The env-regex option is similar to and compatible with env. Its value is a regular expression to match logging-related environment variables. It is used for advanced log tag options. Default: - No envRegex
        :param labels: (experimental) The labels option takes an array of keys. If there is collision between label and env keys, the value of the env takes precedence. Adds additional fields to the extra attributes of a logging message. Default: - No labels
        :param tag: (experimental) By default, Docker uses the first 12 characters of the container ID to tag log messages. Refer to the log tag option documentation for customizing the log tag format. Default: - The first 12 characters of the container ID
        :param address: (experimental) The address of the GELF server. tcp and udp are the only supported URI specifier and you must specify the port.
        :param compression_level: (experimental) UDP Only The level of compression when gzip or zlib is the gelf-compression-type. An integer in the range of -1 to 9 (BestCompression). Higher levels provide more compression at lower speed. Either -1 or 0 disables compression. Default: - 1
        :param compression_type: (experimental) UDP Only The type of compression the GELF driver uses to compress each log message. Allowed values are gzip, zlib and none. Default: - gzip
        :param tcp_max_reconnect: (experimental) TCP Only The maximum number of reconnection attempts when the connection drop. A positive integer. Default: - 3
        :param tcp_reconnect_delay: (experimental) TCP Only The number of seconds to wait between reconnection attempts. A positive integer. Default: - 1

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "address": address,
        }
        if env is not None:
            self._values["env"] = env
        if env_regex is not None:
            self._values["env_regex"] = env_regex
        if labels is not None:
            self._values["labels"] = labels
        if tag is not None:
            self._values["tag"] = tag
        if compression_level is not None:
            self._values["compression_level"] = compression_level
        if compression_type is not None:
            self._values["compression_type"] = compression_type
        if tcp_max_reconnect is not None:
            self._values["tcp_max_reconnect"] = tcp_max_reconnect
        if tcp_reconnect_delay is not None:
            self._values["tcp_reconnect_delay"] = tcp_reconnect_delay

    @builtins.property
    def env(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) The env option takes an array of keys.

        If there is collision between
        label and env keys, the value of the env takes precedence. Adds additional fields
        to the extra attributes of a logging message.

        :default: - No env

        :stability: experimental
        '''
        result = self._values.get("env")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def env_regex(self) -> typing.Optional[builtins.str]:
        '''(experimental) The env-regex option is similar to and compatible with env.

        Its value is a regular
        expression to match logging-related environment variables. It is used for advanced
        log tag options.

        :default: - No envRegex

        :stability: experimental
        '''
        result = self._values.get("env_regex")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def labels(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) The labels option takes an array of keys.

        If there is collision
        between label and env keys, the value of the env takes precedence. Adds additional
        fields to the extra attributes of a logging message.

        :default: - No labels

        :stability: experimental
        '''
        result = self._values.get("labels")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def tag(self) -> typing.Optional[builtins.str]:
        '''(experimental) By default, Docker uses the first 12 characters of the container ID to tag log messages.

        Refer to the log tag option documentation for customizing the
        log tag format.

        :default: - The first 12 characters of the container ID

        :stability: experimental
        '''
        result = self._values.get("tag")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def address(self) -> builtins.str:
        '''(experimental) The address of the GELF server.

        tcp and udp are the only supported URI
        specifier and you must specify the port.

        :stability: experimental
        '''
        result = self._values.get("address")
        assert result is not None, "Required property 'address' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def compression_level(self) -> typing.Optional[jsii.Number]:
        '''(experimental) UDP Only The level of compression when gzip or zlib is the gelf-compression-type.

        An integer in the range of -1 to 9 (BestCompression). Higher levels provide more
        compression at lower speed. Either -1 or 0 disables compression.

        :default: - 1

        :stability: experimental
        '''
        result = self._values.get("compression_level")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def compression_type(self) -> typing.Optional[GelfCompressionType]:
        '''(experimental) UDP Only The type of compression the GELF driver uses to compress each log message.

        Allowed values are gzip, zlib and none.

        :default: - gzip

        :stability: experimental
        '''
        result = self._values.get("compression_type")
        return typing.cast(typing.Optional[GelfCompressionType], result)

    @builtins.property
    def tcp_max_reconnect(self) -> typing.Optional[jsii.Number]:
        '''(experimental) TCP Only The maximum number of reconnection attempts when the connection drop.

        A positive integer.

        :default: - 3

        :stability: experimental
        '''
        result = self._values.get("tcp_max_reconnect")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def tcp_reconnect_delay(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) TCP Only The number of seconds to wait between reconnection attempts.

        A positive integer.

        :default: - 1

        :stability: experimental
        '''
        result = self._values.get("tcp_reconnect_delay")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "GelfLogDriverProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.HealthCheck",
    jsii_struct_bases=[],
    name_mapping={
        "command": "command",
        "interval": "interval",
        "retries": "retries",
        "start_period": "startPeriod",
        "timeout": "timeout",
    },
)
class HealthCheck:
    def __init__(
        self,
        *,
        command: typing.Sequence[builtins.str],
        interval: typing.Optional[_Duration_070aa057] = None,
        retries: typing.Optional[jsii.Number] = None,
        start_period: typing.Optional[_Duration_070aa057] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''(experimental) The health check command and associated configuration parameters for the container.

        :param command: (experimental) A string array representing the command that the container runs to determine if it is healthy. The string array must start with CMD to execute the command arguments directly, or CMD-SHELL to run the command with the container's default shell. For example: [ "CMD-SHELL", "curl -f http://localhost/ || exit 1" ]
        :param interval: (experimental) The time period in seconds between each health check execution. You may specify between 5 and 300 seconds. Default: Duration.seconds(30)
        :param retries: (experimental) The number of times to retry a failed health check before the container is considered unhealthy. You may specify between 1 and 10 retries. Default: 3
        :param start_period: (experimental) The optional grace period within which to provide containers time to bootstrap before failed health checks count towards the maximum number of retries. You may specify between 0 and 300 seconds. Default: No start period
        :param timeout: (experimental) The time period in seconds to wait for a health check to succeed before it is considered a failure. You may specify between 2 and 60 seconds. Default: Duration.seconds(5)

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "command": command,
        }
        if interval is not None:
            self._values["interval"] = interval
        if retries is not None:
            self._values["retries"] = retries
        if start_period is not None:
            self._values["start_period"] = start_period
        if timeout is not None:
            self._values["timeout"] = timeout

    @builtins.property
    def command(self) -> typing.List[builtins.str]:
        '''(experimental) A string array representing the command that the container runs to determine if it is healthy.

        The string array must start with CMD to execute the command arguments directly, or
        CMD-SHELL to run the command with the container's default shell.

        For example: [ "CMD-SHELL", "curl -f http://localhost/ || exit 1" ]

        :stability: experimental
        '''
        result = self._values.get("command")
        assert result is not None, "Required property 'command' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def interval(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) The time period in seconds between each health check execution.

        You may specify between 5 and 300 seconds.

        :default: Duration.seconds(30)

        :stability: experimental
        '''
        result = self._values.get("interval")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def retries(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The number of times to retry a failed health check before the container is considered unhealthy.

        You may specify between 1 and 10 retries.

        :default: 3

        :stability: experimental
        '''
        result = self._values.get("retries")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def start_period(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) The optional grace period within which to provide containers time to bootstrap before failed health checks count towards the maximum number of retries.

        You may specify between 0 and 300 seconds.

        :default: No start period

        :stability: experimental
        '''
        result = self._values.get("start_period")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) The time period in seconds to wait for a health check to succeed before it is considered a failure.

        You may specify between 2 and 60 seconds.

        :default: Duration.seconds(5)

        :stability: experimental
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "HealthCheck(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.Host",
    jsii_struct_bases=[],
    name_mapping={"source_path": "sourcePath"},
)
class Host:
    def __init__(self, *, source_path: typing.Optional[builtins.str] = None) -> None:
        '''(experimental) The details on a container instance bind mount host volume.

        :param source_path: (experimental) Specifies the path on the host container instance that is presented to the container. If the sourcePath value does not exist on the host container instance, the Docker daemon creates it. If the location does exist, the contents of the source path folder are exported. This property is not supported for tasks that use the Fargate launch type.

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if source_path is not None:
            self._values["source_path"] = source_path

    @builtins.property
    def source_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) Specifies the path on the host container instance that is presented to the container.

        If the sourcePath value does not exist on the host container instance, the Docker daemon creates it.
        If the location does exist, the contents of the source path folder are exported.

        This property is not supported for tasks that use the Fargate launch type.

        :stability: experimental
        '''
        result = self._values.get("source_path")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "Host(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.interface(jsii_type="monocdk.aws_ecs.ICluster")
class ICluster(_IResource_8c1dbbbd, typing_extensions.Protocol):
    '''(experimental) A regional grouping of one or more container instances on which you can run tasks and services.

    :stability: experimental
    '''

    @builtins.staticmethod
    def __jsii_proxy_class__() -> typing.Type["_IClusterProxy"]:
        return _IClusterProxy

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="clusterArn")
    def cluster_arn(self) -> builtins.str:
        '''(experimental) The Amazon Resource Name (ARN) that identifies the cluster.

        :stability: experimental
        :attribute: true
        '''
        ...

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="clusterName")
    def cluster_name(self) -> builtins.str:
        '''(experimental) The name of the cluster.

        :stability: experimental
        :attribute: true
        '''
        ...

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="connections")
    def connections(self) -> _Connections_57ccbda9:
        '''(experimental) Manage the allowed network connections for the cluster with Security Groups.

        :stability: experimental
        '''
        ...

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="hasEc2Capacity")
    def has_ec2_capacity(self) -> builtins.bool:
        '''(experimental) Specifies whether the cluster has EC2 instance capacity.

        :stability: experimental
        '''
        ...

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="vpc")
    def vpc(self) -> _IVpc_6d1f76c4:
        '''(experimental) The VPC associated with the cluster.

        :stability: experimental
        '''
        ...

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="autoscalingGroup")
    def autoscaling_group(self) -> typing.Optional[_IAutoScalingGroup_352723c6]:
        '''(experimental) The autoscaling group added to the cluster if capacity is associated to the cluster.

        :stability: experimental
        '''
        ...

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="defaultCloudMapNamespace")
    def default_cloud_map_namespace(self) -> typing.Optional[_INamespace_eb21e42e]:
        '''(experimental) The AWS Cloud Map namespace to associate with the cluster.

        :stability: experimental
        '''
        ...


class _IClusterProxy(
    jsii.proxy_for(_IResource_8c1dbbbd) # type: ignore[misc]
):
    '''(experimental) A regional grouping of one or more container instances on which you can run tasks and services.

    :stability: experimental
    '''

    __jsii_type__: typing.ClassVar[str] = "monocdk.aws_ecs.ICluster"

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="clusterArn")
    def cluster_arn(self) -> builtins.str:
        '''(experimental) The Amazon Resource Name (ARN) that identifies the cluster.

        :stability: experimental
        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "clusterArn"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="clusterName")
    def cluster_name(self) -> builtins.str:
        '''(experimental) The name of the cluster.

        :stability: experimental
        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "clusterName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="connections")
    def connections(self) -> _Connections_57ccbda9:
        '''(experimental) Manage the allowed network connections for the cluster with Security Groups.

        :stability: experimental
        '''
        return typing.cast(_Connections_57ccbda9, jsii.get(self, "connections"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="hasEc2Capacity")
    def has_ec2_capacity(self) -> builtins.bool:
        '''(experimental) Specifies whether the cluster has EC2 instance capacity.

        :stability: experimental
        '''
        return typing.cast(builtins.bool, jsii.get(self, "hasEc2Capacity"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="vpc")
    def vpc(self) -> _IVpc_6d1f76c4:
        '''(experimental) The VPC associated with the cluster.

        :stability: experimental
        '''
        return typing.cast(_IVpc_6d1f76c4, jsii.get(self, "vpc"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="autoscalingGroup")
    def autoscaling_group(self) -> typing.Optional[_IAutoScalingGroup_352723c6]:
        '''(experimental) The autoscaling group added to the cluster if capacity is associated to the cluster.

        :stability: experimental
        '''
        return typing.cast(typing.Optional[_IAutoScalingGroup_352723c6], jsii.get(self, "autoscalingGroup"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="defaultCloudMapNamespace")
    def default_cloud_map_namespace(self) -> typing.Optional[_INamespace_eb21e42e]:
        '''(experimental) The AWS Cloud Map namespace to associate with the cluster.

        :stability: experimental
        '''
        return typing.cast(typing.Optional[_INamespace_eb21e42e], jsii.get(self, "defaultCloudMapNamespace"))


@jsii.interface(jsii_type="monocdk.aws_ecs.IEcsLoadBalancerTarget")
class IEcsLoadBalancerTarget(
    _IApplicationLoadBalancerTarget_db0fdc70,
    _INetworkLoadBalancerTarget_6691f661,
    _ILoadBalancerTarget_ad59785d,
    typing_extensions.Protocol,
):
    '''(experimental) Interface for ECS load balancer target.

    :stability: experimental
    '''

    @builtins.staticmethod
    def __jsii_proxy_class__() -> typing.Type["_IEcsLoadBalancerTargetProxy"]:
        return _IEcsLoadBalancerTargetProxy


class _IEcsLoadBalancerTargetProxy(
    jsii.proxy_for(_IApplicationLoadBalancerTarget_db0fdc70), # type: ignore[misc]
    jsii.proxy_for(_INetworkLoadBalancerTarget_6691f661), # type: ignore[misc]
    jsii.proxy_for(_ILoadBalancerTarget_ad59785d), # type: ignore[misc]
):
    '''(experimental) Interface for ECS load balancer target.

    :stability: experimental
    '''

    __jsii_type__: typing.ClassVar[str] = "monocdk.aws_ecs.IEcsLoadBalancerTarget"
    pass


@jsii.interface(jsii_type="monocdk.aws_ecs.IService")
class IService(_IResource_8c1dbbbd, typing_extensions.Protocol):
    '''(experimental) The interface for a service.

    :stability: experimental
    '''

    @builtins.staticmethod
    def __jsii_proxy_class__() -> typing.Type["_IServiceProxy"]:
        return _IServiceProxy

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="serviceArn")
    def service_arn(self) -> builtins.str:
        '''(experimental) The Amazon Resource Name (ARN) of the service.

        :stability: experimental
        :attribute: true
        '''
        ...

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="serviceName")
    def service_name(self) -> builtins.str:
        '''(experimental) The name of the service.

        :stability: experimental
        :attribute: true
        '''
        ...


class _IServiceProxy(
    jsii.proxy_for(_IResource_8c1dbbbd) # type: ignore[misc]
):
    '''(experimental) The interface for a service.

    :stability: experimental
    '''

    __jsii_type__: typing.ClassVar[str] = "monocdk.aws_ecs.IService"

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="serviceArn")
    def service_arn(self) -> builtins.str:
        '''(experimental) The Amazon Resource Name (ARN) of the service.

        :stability: experimental
        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "serviceArn"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="serviceName")
    def service_name(self) -> builtins.str:
        '''(experimental) The name of the service.

        :stability: experimental
        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "serviceName"))


@jsii.interface(jsii_type="monocdk.aws_ecs.ITaskDefinition")
class ITaskDefinition(_IResource_8c1dbbbd, typing_extensions.Protocol):
    '''(experimental) The interface for all task definitions.

    :stability: experimental
    '''

    @builtins.staticmethod
    def __jsii_proxy_class__() -> typing.Type["_ITaskDefinitionProxy"]:
        return _ITaskDefinitionProxy

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="compatibility")
    def compatibility(self) -> Compatibility:
        '''(experimental) What launch types this task definition should be compatible with.

        :stability: experimental
        '''
        ...

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="isEc2Compatible")
    def is_ec2_compatible(self) -> builtins.bool:
        '''(experimental) Return true if the task definition can be run on an EC2 cluster.

        :stability: experimental
        '''
        ...

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="isFargateCompatible")
    def is_fargate_compatible(self) -> builtins.bool:
        '''(experimental) Return true if the task definition can be run on a Fargate cluster.

        :stability: experimental
        '''
        ...

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="networkMode")
    def network_mode(self) -> "NetworkMode":
        '''(experimental) The networking mode to use for the containers in the task.

        :stability: experimental
        '''
        ...

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskDefinitionArn")
    def task_definition_arn(self) -> builtins.str:
        '''(experimental) ARN of this task definition.

        :stability: experimental
        :attribute: true
        '''
        ...

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskRole")
    def task_role(self) -> _IRole_59af6f50:
        '''(experimental) The name of the IAM role that grants containers in the task permission to call AWS APIs on your behalf.

        :stability: experimental
        '''
        ...

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="executionRole")
    def execution_role(self) -> typing.Optional[_IRole_59af6f50]:
        '''(experimental) Execution role for this task definition.

        :stability: experimental
        '''
        ...


class _ITaskDefinitionProxy(
    jsii.proxy_for(_IResource_8c1dbbbd) # type: ignore[misc]
):
    '''(experimental) The interface for all task definitions.

    :stability: experimental
    '''

    __jsii_type__: typing.ClassVar[str] = "monocdk.aws_ecs.ITaskDefinition"

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="compatibility")
    def compatibility(self) -> Compatibility:
        '''(experimental) What launch types this task definition should be compatible with.

        :stability: experimental
        '''
        return typing.cast(Compatibility, jsii.get(self, "compatibility"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="isEc2Compatible")
    def is_ec2_compatible(self) -> builtins.bool:
        '''(experimental) Return true if the task definition can be run on an EC2 cluster.

        :stability: experimental
        '''
        return typing.cast(builtins.bool, jsii.get(self, "isEc2Compatible"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="isFargateCompatible")
    def is_fargate_compatible(self) -> builtins.bool:
        '''(experimental) Return true if the task definition can be run on a Fargate cluster.

        :stability: experimental
        '''
        return typing.cast(builtins.bool, jsii.get(self, "isFargateCompatible"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="networkMode")
    def network_mode(self) -> "NetworkMode":
        '''(experimental) The networking mode to use for the containers in the task.

        :stability: experimental
        '''
        return typing.cast("NetworkMode", jsii.get(self, "networkMode"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskDefinitionArn")
    def task_definition_arn(self) -> builtins.str:
        '''(experimental) ARN of this task definition.

        :stability: experimental
        :attribute: true
        '''
        return typing.cast(builtins.str, jsii.get(self, "taskDefinitionArn"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskRole")
    def task_role(self) -> _IRole_59af6f50:
        '''(experimental) The name of the IAM role that grants containers in the task permission to call AWS APIs on your behalf.

        :stability: experimental
        '''
        return typing.cast(_IRole_59af6f50, jsii.get(self, "taskRole"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="executionRole")
    def execution_role(self) -> typing.Optional[_IRole_59af6f50]:
        '''(experimental) Execution role for this task definition.

        :stability: experimental
        '''
        return typing.cast(typing.Optional[_IRole_59af6f50], jsii.get(self, "executionRole"))


@jsii.interface(jsii_type="monocdk.aws_ecs.ITaskDefinitionExtension")
class ITaskDefinitionExtension(typing_extensions.Protocol):
    '''(experimental) An extension for Task Definitions.

    Classes that want to make changes to a TaskDefinition (such as
    adding helper containers) can implement this interface, and can
    then be "added" to a TaskDefinition like so::

       taskDefinition.addExtension(new MyExtension("some_parameter"));

    :stability: experimental
    '''

    @builtins.staticmethod
    def __jsii_proxy_class__() -> typing.Type["_ITaskDefinitionExtensionProxy"]:
        return _ITaskDefinitionExtensionProxy

    @jsii.member(jsii_name="extend")
    def extend(self, task_definition: "TaskDefinition") -> None:
        '''(experimental) Apply the extension to the given TaskDefinition.

        :param task_definition: [disable-awslint:ref-via-interface].

        :stability: experimental
        '''
        ...


class _ITaskDefinitionExtensionProxy:
    '''(experimental) An extension for Task Definitions.

    Classes that want to make changes to a TaskDefinition (such as
    adding helper containers) can implement this interface, and can
    then be "added" to a TaskDefinition like so::

       taskDefinition.addExtension(new MyExtension("some_parameter"));

    :stability: experimental
    '''

    __jsii_type__: typing.ClassVar[str] = "monocdk.aws_ecs.ITaskDefinitionExtension"

    @jsii.member(jsii_name="extend")
    def extend(self, task_definition: "TaskDefinition") -> None:
        '''(experimental) Apply the extension to the given TaskDefinition.

        :param task_definition: [disable-awslint:ref-via-interface].

        :stability: experimental
        '''
        return typing.cast(None, jsii.invoke(self, "extend", [task_definition]))


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.InferenceAccelerator",
    jsii_struct_bases=[],
    name_mapping={"device_name": "deviceName", "device_type": "deviceType"},
)
class InferenceAccelerator:
    def __init__(
        self,
        *,
        device_name: typing.Optional[builtins.str] = None,
        device_type: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) Elastic Inference Accelerator.

        For more information, see `Elastic Inference Basics <https://docs.aws.amazon.com/elastic-inference/latest/developerguide/basics.html>`_

        :param device_name: (experimental) The Elastic Inference accelerator device name. Default: - empty
        :param device_type: (experimental) The Elastic Inference accelerator type to use. The allowed values are: eia2.medium, eia2.large and eia2.xlarge. Default: - empty

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if device_name is not None:
            self._values["device_name"] = device_name
        if device_type is not None:
            self._values["device_type"] = device_type

    @builtins.property
    def device_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) The Elastic Inference accelerator device name.

        :default: - empty

        :stability: experimental
        '''
        result = self._values.get("device_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def device_type(self) -> typing.Optional[builtins.str]:
        '''(experimental) The Elastic Inference accelerator type to use.

        The allowed values are: eia2.medium, eia2.large and eia2.xlarge.

        :default: - empty

        :stability: experimental
        '''
        result = self._values.get("device_type")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "InferenceAccelerator(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_ecs.IpcMode")
class IpcMode(enum.Enum):
    '''(experimental) The IPC resource namespace to use for the containers in the task.

    :stability: experimental
    '''

    NONE = "NONE"
    '''(experimental) If none is specified, then IPC resources within the containers of a task are private and not shared with other containers in a task or on the container instance.

    :stability: experimental
    '''
    HOST = "HOST"
    '''(experimental) If host is specified, then all containers within the tasks that specified the host IPC mode on the same container instance share the same IPC resources with the host Amazon EC2 instance.

    :stability: experimental
    '''
    TASK = "TASK"
    '''(experimental) If task is specified, all containers within the specified task share the same IPC resources.

    :stability: experimental
    '''


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.JournaldLogDriverProps",
    jsii_struct_bases=[BaseLogDriverProps],
    name_mapping={
        "env": "env",
        "env_regex": "envRegex",
        "labels": "labels",
        "tag": "tag",
    },
)
class JournaldLogDriverProps(BaseLogDriverProps):
    def __init__(
        self,
        *,
        env: typing.Optional[typing.Sequence[builtins.str]] = None,
        env_regex: typing.Optional[builtins.str] = None,
        labels: typing.Optional[typing.Sequence[builtins.str]] = None,
        tag: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) Specifies the journald log driver configuration options.

        `Source <https://docs.docker.com/config/containers/logging/journald/>`_

        :param env: (experimental) The env option takes an array of keys. If there is collision between label and env keys, the value of the env takes precedence. Adds additional fields to the extra attributes of a logging message. Default: - No env
        :param env_regex: (experimental) The env-regex option is similar to and compatible with env. Its value is a regular expression to match logging-related environment variables. It is used for advanced log tag options. Default: - No envRegex
        :param labels: (experimental) The labels option takes an array of keys. If there is collision between label and env keys, the value of the env takes precedence. Adds additional fields to the extra attributes of a logging message. Default: - No labels
        :param tag: (experimental) By default, Docker uses the first 12 characters of the container ID to tag log messages. Refer to the log tag option documentation for customizing the log tag format. Default: - The first 12 characters of the container ID

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if env is not None:
            self._values["env"] = env
        if env_regex is not None:
            self._values["env_regex"] = env_regex
        if labels is not None:
            self._values["labels"] = labels
        if tag is not None:
            self._values["tag"] = tag

    @builtins.property
    def env(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) The env option takes an array of keys.

        If there is collision between
        label and env keys, the value of the env takes precedence. Adds additional fields
        to the extra attributes of a logging message.

        :default: - No env

        :stability: experimental
        '''
        result = self._values.get("env")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def env_regex(self) -> typing.Optional[builtins.str]:
        '''(experimental) The env-regex option is similar to and compatible with env.

        Its value is a regular
        expression to match logging-related environment variables. It is used for advanced
        log tag options.

        :default: - No envRegex

        :stability: experimental
        '''
        result = self._values.get("env_regex")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def labels(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) The labels option takes an array of keys.

        If there is collision
        between label and env keys, the value of the env takes precedence. Adds additional
        fields to the extra attributes of a logging message.

        :default: - No labels

        :stability: experimental
        '''
        result = self._values.get("labels")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def tag(self) -> typing.Optional[builtins.str]:
        '''(experimental) By default, Docker uses the first 12 characters of the container ID to tag log messages.

        Refer to the log tag option documentation for customizing the
        log tag format.

        :default: - The first 12 characters of the container ID

        :stability: experimental
        '''
        result = self._values.get("tag")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "JournaldLogDriverProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.JsonFileLogDriverProps",
    jsii_struct_bases=[BaseLogDriverProps],
    name_mapping={
        "env": "env",
        "env_regex": "envRegex",
        "labels": "labels",
        "tag": "tag",
        "compress": "compress",
        "max_file": "maxFile",
        "max_size": "maxSize",
    },
)
class JsonFileLogDriverProps(BaseLogDriverProps):
    def __init__(
        self,
        *,
        env: typing.Optional[typing.Sequence[builtins.str]] = None,
        env_regex: typing.Optional[builtins.str] = None,
        labels: typing.Optional[typing.Sequence[builtins.str]] = None,
        tag: typing.Optional[builtins.str] = None,
        compress: typing.Optional[builtins.bool] = None,
        max_file: typing.Optional[jsii.Number] = None,
        max_size: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) Specifies the json-file log driver configuration options.

        `Source <https://docs.docker.com/config/containers/logging/json-file/>`_

        :param env: (experimental) The env option takes an array of keys. If there is collision between label and env keys, the value of the env takes precedence. Adds additional fields to the extra attributes of a logging message. Default: - No env
        :param env_regex: (experimental) The env-regex option is similar to and compatible with env. Its value is a regular expression to match logging-related environment variables. It is used for advanced log tag options. Default: - No envRegex
        :param labels: (experimental) The labels option takes an array of keys. If there is collision between label and env keys, the value of the env takes precedence. Adds additional fields to the extra attributes of a logging message. Default: - No labels
        :param tag: (experimental) By default, Docker uses the first 12 characters of the container ID to tag log messages. Refer to the log tag option documentation for customizing the log tag format. Default: - The first 12 characters of the container ID
        :param compress: (experimental) Toggles compression for rotated logs. Default: - false
        :param max_file: (experimental) The maximum number of log files that can be present. If rolling the logs creates excess files, the oldest file is removed. Only effective when max-size is also set. A positive integer. Default: - 1
        :param max_size: (experimental) The maximum size of the log before it is rolled. A positive integer plus a modifier representing the unit of measure (k, m, or g). Default: - -1 (unlimited)

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if env is not None:
            self._values["env"] = env
        if env_regex is not None:
            self._values["env_regex"] = env_regex
        if labels is not None:
            self._values["labels"] = labels
        if tag is not None:
            self._values["tag"] = tag
        if compress is not None:
            self._values["compress"] = compress
        if max_file is not None:
            self._values["max_file"] = max_file
        if max_size is not None:
            self._values["max_size"] = max_size

    @builtins.property
    def env(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) The env option takes an array of keys.

        If there is collision between
        label and env keys, the value of the env takes precedence. Adds additional fields
        to the extra attributes of a logging message.

        :default: - No env

        :stability: experimental
        '''
        result = self._values.get("env")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def env_regex(self) -> typing.Optional[builtins.str]:
        '''(experimental) The env-regex option is similar to and compatible with env.

        Its value is a regular
        expression to match logging-related environment variables. It is used for advanced
        log tag options.

        :default: - No envRegex

        :stability: experimental
        '''
        result = self._values.get("env_regex")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def labels(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) The labels option takes an array of keys.

        If there is collision
        between label and env keys, the value of the env takes precedence. Adds additional
        fields to the extra attributes of a logging message.

        :default: - No labels

        :stability: experimental
        '''
        result = self._values.get("labels")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def tag(self) -> typing.Optional[builtins.str]:
        '''(experimental) By default, Docker uses the first 12 characters of the container ID to tag log messages.

        Refer to the log tag option documentation for customizing the
        log tag format.

        :default: - The first 12 characters of the container ID

        :stability: experimental
        '''
        result = self._values.get("tag")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def compress(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Toggles compression for rotated logs.

        :default: - false

        :stability: experimental
        '''
        result = self._values.get("compress")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def max_file(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The maximum number of log files that can be present.

        If rolling the logs creates
        excess files, the oldest file is removed. Only effective when max-size is also set.
        A positive integer.

        :default: - 1

        :stability: experimental
        '''
        result = self._values.get("max_file")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def max_size(self) -> typing.Optional[builtins.str]:
        '''(experimental) The maximum size of the log before it is rolled.

        A positive integer plus a modifier
        representing the unit of measure (k, m, or g).

        :default: - -1 (unlimited)

        :stability: experimental
        '''
        result = self._values.get("max_size")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "JsonFileLogDriverProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_ecs.LaunchType")
class LaunchType(enum.Enum):
    '''(experimental) The launch type of an ECS service.

    :stability: experimental
    '''

    EC2 = "EC2"
    '''(experimental) The service will be launched using the EC2 launch type.

    :stability: experimental
    '''
    FARGATE = "FARGATE"
    '''(experimental) The service will be launched using the FARGATE launch type.

    :stability: experimental
    '''


class LinuxParameters(
    _Construct_e78e779f,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_ecs.LinuxParameters",
):
    '''(experimental) Linux-specific options that are applied to the container.

    :stability: experimental
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        init_process_enabled: typing.Optional[builtins.bool] = None,
        shared_memory_size: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''(experimental) Constructs a new instance of the LinuxParameters class.

        :param scope: -
        :param id: -
        :param init_process_enabled: (experimental) Specifies whether to run an init process inside the container that forwards signals and reaps processes. Default: false
        :param shared_memory_size: (experimental) The value for the size (in MiB) of the /dev/shm volume. Default: No shared memory.

        :stability: experimental
        '''
        props = LinuxParametersProps(
            init_process_enabled=init_process_enabled,
            shared_memory_size=shared_memory_size,
        )

        jsii.create(LinuxParameters, self, [scope, id, props])

    @jsii.member(jsii_name="addCapabilities")
    def add_capabilities(self, *cap: Capability) -> None:
        '''(experimental) Adds one or more Linux capabilities to the Docker configuration of a container.

        Only works with EC2 launch type.

        :param cap: -

        :stability: experimental
        '''
        return typing.cast(None, jsii.invoke(self, "addCapabilities", [*cap]))

    @jsii.member(jsii_name="addDevices")
    def add_devices(self, *device: Device) -> None:
        '''(experimental) Adds one or more host devices to a container.

        :param device: -

        :stability: experimental
        '''
        return typing.cast(None, jsii.invoke(self, "addDevices", [*device]))

    @jsii.member(jsii_name="addTmpfs")
    def add_tmpfs(self, *tmpfs: "Tmpfs") -> None:
        '''(experimental) Specifies the container path, mount options, and size (in MiB) of the tmpfs mount for a container.

        Only works with EC2 launch type.

        :param tmpfs: -

        :stability: experimental
        '''
        return typing.cast(None, jsii.invoke(self, "addTmpfs", [*tmpfs]))

    @jsii.member(jsii_name="dropCapabilities")
    def drop_capabilities(self, *cap: Capability) -> None:
        '''(experimental) Removes one or more Linux capabilities to the Docker configuration of a container.

        Only works with EC2 launch type.

        :param cap: -

        :stability: experimental
        '''
        return typing.cast(None, jsii.invoke(self, "dropCapabilities", [*cap]))

    @jsii.member(jsii_name="renderLinuxParameters")
    def render_linux_parameters(self) -> CfnTaskDefinition.LinuxParametersProperty:
        '''(experimental) Renders the Linux parameters to a CloudFormation object.

        :stability: experimental
        '''
        return typing.cast(CfnTaskDefinition.LinuxParametersProperty, jsii.invoke(self, "renderLinuxParameters", []))


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.LinuxParametersProps",
    jsii_struct_bases=[],
    name_mapping={
        "init_process_enabled": "initProcessEnabled",
        "shared_memory_size": "sharedMemorySize",
    },
)
class LinuxParametersProps:
    def __init__(
        self,
        *,
        init_process_enabled: typing.Optional[builtins.bool] = None,
        shared_memory_size: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''(experimental) The properties for defining Linux-specific options that are applied to the container.

        :param init_process_enabled: (experimental) Specifies whether to run an init process inside the container that forwards signals and reaps processes. Default: false
        :param shared_memory_size: (experimental) The value for the size (in MiB) of the /dev/shm volume. Default: No shared memory.

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if init_process_enabled is not None:
            self._values["init_process_enabled"] = init_process_enabled
        if shared_memory_size is not None:
            self._values["shared_memory_size"] = shared_memory_size

    @builtins.property
    def init_process_enabled(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specifies whether to run an init process inside the container that forwards signals and reaps processes.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("init_process_enabled")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def shared_memory_size(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The value for the size (in MiB) of the /dev/shm volume.

        :default: No shared memory.

        :stability: experimental
        '''
        result = self._values.get("shared_memory_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "LinuxParametersProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class ListenerConfig(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="monocdk.aws_ecs.ListenerConfig",
):
    '''(experimental) Base class for configuring listener when registering targets.

    :stability: experimental
    '''

    @builtins.staticmethod
    def __jsii_proxy_class__() -> typing.Type["_ListenerConfigProxy"]:
        return _ListenerConfigProxy

    def __init__(self) -> None:
        '''
        :stability: experimental
        '''
        jsii.create(ListenerConfig, self, [])

    @jsii.member(jsii_name="applicationListener") # type: ignore[misc]
    @builtins.classmethod
    def application_listener(
        cls,
        listener: _ApplicationListener_835df0e5,
        *,
        deregistration_delay: typing.Optional[_Duration_070aa057] = None,
        health_check: typing.Optional[_HealthCheck_46284083] = None,
        port: typing.Optional[jsii.Number] = None,
        protocol: typing.Optional[_ApplicationProtocol_43133732] = None,
        protocol_version: typing.Optional[_ApplicationProtocolVersion_01dcd421] = None,
        slow_start: typing.Optional[_Duration_070aa057] = None,
        stickiness_cookie_duration: typing.Optional[_Duration_070aa057] = None,
        stickiness_cookie_name: typing.Optional[builtins.str] = None,
        target_group_name: typing.Optional[builtins.str] = None,
        targets: typing.Optional[typing.Sequence[_IApplicationLoadBalancerTarget_db0fdc70]] = None,
        conditions: typing.Optional[typing.Sequence[_ListenerCondition_c2f847ce]] = None,
        host_header: typing.Optional[builtins.str] = None,
        path_pattern: typing.Optional[builtins.str] = None,
        path_patterns: typing.Optional[typing.Sequence[builtins.str]] = None,
        priority: typing.Optional[jsii.Number] = None,
    ) -> "ListenerConfig":
        '''(experimental) Create a config for adding target group to ALB listener.

        :param listener: -
        :param deregistration_delay: (experimental) The amount of time for Elastic Load Balancing to wait before deregistering a target. The range is 0-3600 seconds. Default: Duration.minutes(5)
        :param health_check: (experimental) Health check configuration. Default: No health check
        :param port: (experimental) The port on which the listener listens for requests. Default: Determined from protocol if known
        :param protocol: (experimental) The protocol to use. Default: Determined from port if known
        :param protocol_version: (experimental) The protocol version to use. Default: ApplicationProtocolVersion.HTTP1
        :param slow_start: (experimental) The time period during which the load balancer sends a newly registered target a linearly increasing share of the traffic to the target group. The range is 30-900 seconds (15 minutes). Default: 0
        :param stickiness_cookie_duration: (experimental) The stickiness cookie expiration period. Setting this value enables load balancer stickiness. After this period, the cookie is considered stale. The minimum value is 1 second and the maximum value is 7 days (604800 seconds). Default: Stickiness disabled
        :param stickiness_cookie_name: (experimental) The name of an application-based stickiness cookie. Names that start with the following prefixes are not allowed: AWSALB, AWSALBAPP, and AWSALBTG; they're reserved for use by the load balancer. Note: ``stickinessCookieName`` parameter depends on the presence of ``stickinessCookieDuration`` parameter. If ``stickinessCookieDuration`` is not set, ``stickinessCookieName`` will be omitted. Default: - If ``stickinessCookieDuration`` is set, a load-balancer generated cookie is used. Otherwise, no stickiness is defined.
        :param target_group_name: (experimental) The name of the target group. This name must be unique per region per account, can have a maximum of 32 characters, must contain only alphanumeric characters or hyphens, and must not begin or end with a hyphen. Default: Automatically generated
        :param targets: (experimental) The targets to add to this target group. Can be ``Instance``, ``IPAddress``, or any self-registering load balancing target. All target must be of the same type.
        :param conditions: (experimental) Rule applies if matches the conditions. Default: - No conditions.
        :param host_header: (deprecated) Rule applies if the requested host matches the indicated host. May contain up to three '*' wildcards. Requires that priority is set. Default: No host condition
        :param path_pattern: (deprecated) Rule applies if the requested path matches the given path pattern. May contain up to three '*' wildcards. Requires that priority is set. Default: No path condition
        :param path_patterns: (deprecated) Rule applies if the requested path matches any of the given patterns. May contain up to three '*' wildcards. Requires that priority is set. Default: - No path condition.
        :param priority: (experimental) Priority of this target group. The rule with the lowest priority will be used for every request. If priority is not given, these target groups will be added as defaults, and must not have conditions. Priorities must be unique. Default: Target groups are used as defaults

        :stability: experimental
        '''
        props = _AddApplicationTargetsProps_a85e0c25(
            deregistration_delay=deregistration_delay,
            health_check=health_check,
            port=port,
            protocol=protocol,
            protocol_version=protocol_version,
            slow_start=slow_start,
            stickiness_cookie_duration=stickiness_cookie_duration,
            stickiness_cookie_name=stickiness_cookie_name,
            target_group_name=target_group_name,
            targets=targets,
            conditions=conditions,
            host_header=host_header,
            path_pattern=path_pattern,
            path_patterns=path_patterns,
            priority=priority,
        )

        return typing.cast("ListenerConfig", jsii.sinvoke(cls, "applicationListener", [listener, props]))

    @jsii.member(jsii_name="networkListener") # type: ignore[misc]
    @builtins.classmethod
    def network_listener(
        cls,
        listener: _NetworkListener_0917bd7b,
        *,
        port: jsii.Number,
        deregistration_delay: typing.Optional[_Duration_070aa057] = None,
        health_check: typing.Optional[_HealthCheck_46284083] = None,
        preserve_client_ip: typing.Optional[builtins.bool] = None,
        protocol: typing.Optional[_Protocol_5d30b733] = None,
        proxy_protocol_v2: typing.Optional[builtins.bool] = None,
        target_group_name: typing.Optional[builtins.str] = None,
        targets: typing.Optional[typing.Sequence[_INetworkLoadBalancerTarget_6691f661]] = None,
    ) -> "ListenerConfig":
        '''(experimental) Create a config for adding target group to NLB listener.

        :param listener: -
        :param port: (experimental) The port on which the listener listens for requests. Default: Determined from protocol if known
        :param deregistration_delay: (experimental) The amount of time for Elastic Load Balancing to wait before deregistering a target. The range is 0-3600 seconds. Default: Duration.minutes(5)
        :param health_check: (experimental) Health check configuration. Default: No health check
        :param preserve_client_ip: (experimental) Indicates whether client IP preservation is enabled. Default: false if the target group type is IP address and the target group protocol is TCP or TLS. Otherwise, true.
        :param protocol: (experimental) Protocol for target group, expects TCP, TLS, UDP, or TCP_UDP. Default: - inherits the protocol of the listener
        :param proxy_protocol_v2: (experimental) Indicates whether Proxy Protocol version 2 is enabled. Default: false
        :param target_group_name: (experimental) The name of the target group. This name must be unique per region per account, can have a maximum of 32 characters, must contain only alphanumeric characters or hyphens, and must not begin or end with a hyphen. Default: Automatically generated
        :param targets: (experimental) The targets to add to this target group. Can be ``Instance``, ``IPAddress``, or any self-registering load balancing target. If you use either ``Instance`` or ``IPAddress`` as targets, all target must be of the same type.

        :stability: experimental
        '''
        props = _AddNetworkTargetsProps_63ca961a(
            port=port,
            deregistration_delay=deregistration_delay,
            health_check=health_check,
            preserve_client_ip=preserve_client_ip,
            protocol=protocol,
            proxy_protocol_v2=proxy_protocol_v2,
            target_group_name=target_group_name,
            targets=targets,
        )

        return typing.cast("ListenerConfig", jsii.sinvoke(cls, "networkListener", [listener, props]))

    @jsii.member(jsii_name="addTargets") # type: ignore[misc]
    @abc.abstractmethod
    def add_targets(
        self,
        id: builtins.str,
        target: "LoadBalancerTargetOptions",
        service: "BaseService",
    ) -> None:
        '''(experimental) Create and attach a target group to listener.

        :param id: -
        :param target: -
        :param service: -

        :stability: experimental
        '''
        ...


class _ListenerConfigProxy(ListenerConfig):
    @jsii.member(jsii_name="addTargets")
    def add_targets(
        self,
        id: builtins.str,
        target: "LoadBalancerTargetOptions",
        service: "BaseService",
    ) -> None:
        '''(experimental) Create and attach a target group to listener.

        :param id: -
        :param target: -
        :param service: -

        :stability: experimental
        '''
        return typing.cast(None, jsii.invoke(self, "addTargets", [id, target, service]))


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.LoadBalancerTargetOptions",
    jsii_struct_bases=[],
    name_mapping={
        "container_name": "containerName",
        "container_port": "containerPort",
        "protocol": "protocol",
    },
)
class LoadBalancerTargetOptions:
    def __init__(
        self,
        *,
        container_name: builtins.str,
        container_port: typing.Optional[jsii.Number] = None,
        protocol: typing.Optional["Protocol"] = None,
    ) -> None:
        '''(experimental) Properties for defining an ECS target.

        The port mapping for it must already have been created through addPortMapping().

        :param container_name: (experimental) The name of the container.
        :param container_port: (experimental) The port number of the container. Only applicable when using application/network load balancers. Default: - Container port of the first added port mapping.
        :param protocol: (experimental) The protocol used for the port mapping. Only applicable when using application load balancers. Default: Protocol.TCP

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "container_name": container_name,
        }
        if container_port is not None:
            self._values["container_port"] = container_port
        if protocol is not None:
            self._values["protocol"] = protocol

    @builtins.property
    def container_name(self) -> builtins.str:
        '''(experimental) The name of the container.

        :stability: experimental
        '''
        result = self._values.get("container_name")
        assert result is not None, "Required property 'container_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def container_port(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The port number of the container.

        Only applicable when using application/network load balancers.

        :default: - Container port of the first added port mapping.

        :stability: experimental
        '''
        result = self._values.get("container_port")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def protocol(self) -> typing.Optional["Protocol"]:
        '''(experimental) The protocol used for the port mapping.

        Only applicable when using application load balancers.

        :default: Protocol.TCP

        :stability: experimental
        '''
        result = self._values.get("protocol")
        return typing.cast(typing.Optional["Protocol"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "LoadBalancerTargetOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class LogDriver(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="monocdk.aws_ecs.LogDriver",
):
    '''(experimental) The base class for log drivers.

    :stability: experimental
    '''

    @builtins.staticmethod
    def __jsii_proxy_class__() -> typing.Type["_LogDriverProxy"]:
        return _LogDriverProxy

    def __init__(self) -> None:
        '''
        :stability: experimental
        '''
        jsii.create(LogDriver, self, [])

    @jsii.member(jsii_name="awsLogs") # type: ignore[misc]
    @builtins.classmethod
    def aws_logs(
        cls,
        *,
        stream_prefix: builtins.str,
        datetime_format: typing.Optional[builtins.str] = None,
        log_group: typing.Optional[_ILogGroup_846e17a0] = None,
        log_retention: typing.Optional[_RetentionDays_6c560d31] = None,
        mode: typing.Optional[AwsLogDriverMode] = None,
        multiline_pattern: typing.Optional[builtins.str] = None,
    ) -> "LogDriver":
        '''(experimental) Creates a log driver configuration that sends log information to CloudWatch Logs.

        :param stream_prefix: (experimental) Prefix for the log streams. The awslogs-stream-prefix option allows you to associate a log stream with the specified prefix, the container name, and the ID of the Amazon ECS task to which the container belongs. If you specify a prefix with this option, then the log stream takes the following format:: prefix-name/container-name/ecs-task-id
        :param datetime_format: (experimental) This option defines a multiline start pattern in Python strftime format. A log message consists of a line that matches the pattern and any following lines that don’t match the pattern. Thus the matched line is the delimiter between log messages. Default: - No multiline matching.
        :param log_group: (experimental) The log group to log to. Default: - A log group is automatically created.
        :param log_retention: (experimental) The number of days log events are kept in CloudWatch Logs when the log group is automatically created by this construct. Default: - Logs never expire.
        :param mode: (experimental) The delivery mode of log messages from the container to awslogs. Default: - AwsLogDriverMode.BLOCKING
        :param multiline_pattern: (experimental) This option defines a multiline start pattern using a regular expression. A log message consists of a line that matches the pattern and any following lines that don’t match the pattern. Thus the matched line is the delimiter between log messages. This option is ignored if datetimeFormat is also configured. Default: - No multiline matching.

        :stability: experimental
        '''
        props = AwsLogDriverProps(
            stream_prefix=stream_prefix,
            datetime_format=datetime_format,
            log_group=log_group,
            log_retention=log_retention,
            mode=mode,
            multiline_pattern=multiline_pattern,
        )

        return typing.cast("LogDriver", jsii.sinvoke(cls, "awsLogs", [props]))

    @jsii.member(jsii_name="bind") # type: ignore[misc]
    @abc.abstractmethod
    def bind(
        self,
        scope: _Construct_e78e779f,
        container_definition: ContainerDefinition,
    ) -> "LogDriverConfig":
        '''(experimental) Called when the log driver is configured on a container.

        :param scope: -
        :param container_definition: -

        :stability: experimental
        '''
        ...


class _LogDriverProxy(LogDriver):
    @jsii.member(jsii_name="bind")
    def bind(
        self,
        scope: _Construct_e78e779f,
        container_definition: ContainerDefinition,
    ) -> "LogDriverConfig":
        '''(experimental) Called when the log driver is configured on a container.

        :param scope: -
        :param container_definition: -

        :stability: experimental
        '''
        return typing.cast("LogDriverConfig", jsii.invoke(self, "bind", [scope, container_definition]))


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.LogDriverConfig",
    jsii_struct_bases=[],
    name_mapping={"log_driver": "logDriver", "options": "options"},
)
class LogDriverConfig:
    def __init__(
        self,
        *,
        log_driver: builtins.str,
        options: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
    ) -> None:
        '''(experimental) The configuration to use when creating a log driver.

        :param log_driver: (experimental) The log driver to use for the container. The valid values listed for this parameter are log drivers that the Amazon ECS container agent can communicate with by default. For tasks using the Fargate launch type, the supported log drivers are awslogs, splunk, and awsfirelens. For tasks using the EC2 launch type, the supported log drivers are awslogs, fluentd, gelf, json-file, journald, logentries,syslog, splunk, and awsfirelens. For more information about using the awslogs log driver, see `Using the awslogs Log Driver <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_awslogs.html>`_ in the Amazon Elastic Container Service Developer Guide. For more information about using the awsfirelens log driver, see `Custom Log Routing <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html>`_ in the Amazon Elastic Container Service Developer Guide.
        :param options: (experimental) The configuration options to send to the log driver.

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "log_driver": log_driver,
        }
        if options is not None:
            self._values["options"] = options

    @builtins.property
    def log_driver(self) -> builtins.str:
        '''(experimental) The log driver to use for the container.

        The valid values listed for this parameter are log drivers
        that the Amazon ECS container agent can communicate with by default.

        For tasks using the Fargate launch type, the supported log drivers are awslogs, splunk, and awsfirelens.
        For tasks using the EC2 launch type, the supported log drivers are awslogs, fluentd, gelf, json-file, journald,
        logentries,syslog, splunk, and awsfirelens.

        For more information about using the awslogs log driver, see
        `Using the awslogs Log Driver <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_awslogs.html>`_
        in the Amazon Elastic Container Service Developer Guide.

        For more information about using the awsfirelens log driver, see
        `Custom Log Routing <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html>`_
        in the Amazon Elastic Container Service Developer Guide.

        :stability: experimental
        '''
        result = self._values.get("log_driver")
        assert result is not None, "Required property 'log_driver' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def options(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''(experimental) The configuration options to send to the log driver.

        :stability: experimental
        '''
        result = self._values.get("options")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "LogDriverConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class LogDrivers(metaclass=jsii.JSIIMeta, jsii_type="monocdk.aws_ecs.LogDrivers"):
    '''(experimental) The base class for log drivers.

    :stability: experimental
    '''

    def __init__(self) -> None:
        '''
        :stability: experimental
        '''
        jsii.create(LogDrivers, self, [])

    @jsii.member(jsii_name="awsLogs") # type: ignore[misc]
    @builtins.classmethod
    def aws_logs(
        cls,
        *,
        stream_prefix: builtins.str,
        datetime_format: typing.Optional[builtins.str] = None,
        log_group: typing.Optional[_ILogGroup_846e17a0] = None,
        log_retention: typing.Optional[_RetentionDays_6c560d31] = None,
        mode: typing.Optional[AwsLogDriverMode] = None,
        multiline_pattern: typing.Optional[builtins.str] = None,
    ) -> LogDriver:
        '''(experimental) Creates a log driver configuration that sends log information to CloudWatch Logs.

        :param stream_prefix: (experimental) Prefix for the log streams. The awslogs-stream-prefix option allows you to associate a log stream with the specified prefix, the container name, and the ID of the Amazon ECS task to which the container belongs. If you specify a prefix with this option, then the log stream takes the following format:: prefix-name/container-name/ecs-task-id
        :param datetime_format: (experimental) This option defines a multiline start pattern in Python strftime format. A log message consists of a line that matches the pattern and any following lines that don’t match the pattern. Thus the matched line is the delimiter between log messages. Default: - No multiline matching.
        :param log_group: (experimental) The log group to log to. Default: - A log group is automatically created.
        :param log_retention: (experimental) The number of days log events are kept in CloudWatch Logs when the log group is automatically created by this construct. Default: - Logs never expire.
        :param mode: (experimental) The delivery mode of log messages from the container to awslogs. Default: - AwsLogDriverMode.BLOCKING
        :param multiline_pattern: (experimental) This option defines a multiline start pattern using a regular expression. A log message consists of a line that matches the pattern and any following lines that don’t match the pattern. Thus the matched line is the delimiter between log messages. This option is ignored if datetimeFormat is also configured. Default: - No multiline matching.

        :stability: experimental
        '''
        props = AwsLogDriverProps(
            stream_prefix=stream_prefix,
            datetime_format=datetime_format,
            log_group=log_group,
            log_retention=log_retention,
            mode=mode,
            multiline_pattern=multiline_pattern,
        )

        return typing.cast(LogDriver, jsii.sinvoke(cls, "awsLogs", [props]))

    @jsii.member(jsii_name="firelens") # type: ignore[misc]
    @builtins.classmethod
    def firelens(
        cls,
        *,
        options: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        env: typing.Optional[typing.Sequence[builtins.str]] = None,
        env_regex: typing.Optional[builtins.str] = None,
        labels: typing.Optional[typing.Sequence[builtins.str]] = None,
        tag: typing.Optional[builtins.str] = None,
    ) -> LogDriver:
        '''(experimental) Creates a log driver configuration that sends log information to firelens log router.

        For detail configurations, please refer to Amazon ECS FireLens Examples:
        https://github.com/aws-samples/amazon-ecs-firelens-examples

        :param options: (experimental) The configuration options to send to the log driver. Default: - the log driver options
        :param env: (experimental) The env option takes an array of keys. If there is collision between label and env keys, the value of the env takes precedence. Adds additional fields to the extra attributes of a logging message. Default: - No env
        :param env_regex: (experimental) The env-regex option is similar to and compatible with env. Its value is a regular expression to match logging-related environment variables. It is used for advanced log tag options. Default: - No envRegex
        :param labels: (experimental) The labels option takes an array of keys. If there is collision between label and env keys, the value of the env takes precedence. Adds additional fields to the extra attributes of a logging message. Default: - No labels
        :param tag: (experimental) By default, Docker uses the first 12 characters of the container ID to tag log messages. Refer to the log tag option documentation for customizing the log tag format. Default: - The first 12 characters of the container ID

        :stability: experimental
        '''
        props = FireLensLogDriverProps(
            options=options, env=env, env_regex=env_regex, labels=labels, tag=tag
        )

        return typing.cast(LogDriver, jsii.sinvoke(cls, "firelens", [props]))

    @jsii.member(jsii_name="fluentd") # type: ignore[misc]
    @builtins.classmethod
    def fluentd(
        cls,
        *,
        address: typing.Optional[builtins.str] = None,
        async_connect: typing.Optional[builtins.bool] = None,
        buffer_limit: typing.Optional[jsii.Number] = None,
        max_retries: typing.Optional[jsii.Number] = None,
        retry_wait: typing.Optional[_Duration_070aa057] = None,
        sub_second_precision: typing.Optional[builtins.bool] = None,
        env: typing.Optional[typing.Sequence[builtins.str]] = None,
        env_regex: typing.Optional[builtins.str] = None,
        labels: typing.Optional[typing.Sequence[builtins.str]] = None,
        tag: typing.Optional[builtins.str] = None,
    ) -> LogDriver:
        '''(experimental) Creates a log driver configuration that sends log information to fluentd Logs.

        :param address: (experimental) By default, the logging driver connects to localhost:24224. Supply the address option to connect to a different address. tcp(default) and unix sockets are supported. Default: - address not set.
        :param async_connect: (experimental) Docker connects to Fluentd in the background. Messages are buffered until the connection is established. Default: - false
        :param buffer_limit: (experimental) The amount of data to buffer before flushing to disk. Default: - The amount of RAM available to the container.
        :param max_retries: (experimental) The maximum number of retries. Default: - 4294967295 (2**32 - 1).
        :param retry_wait: (experimental) How long to wait between retries. Default: - 1 second
        :param sub_second_precision: (experimental) Generates event logs in nanosecond resolution. Default: - false
        :param env: (experimental) The env option takes an array of keys. If there is collision between label and env keys, the value of the env takes precedence. Adds additional fields to the extra attributes of a logging message. Default: - No env
        :param env_regex: (experimental) The env-regex option is similar to and compatible with env. Its value is a regular expression to match logging-related environment variables. It is used for advanced log tag options. Default: - No envRegex
        :param labels: (experimental) The labels option takes an array of keys. If there is collision between label and env keys, the value of the env takes precedence. Adds additional fields to the extra attributes of a logging message. Default: - No labels
        :param tag: (experimental) By default, Docker uses the first 12 characters of the container ID to tag log messages. Refer to the log tag option documentation for customizing the log tag format. Default: - The first 12 characters of the container ID

        :stability: experimental
        '''
        props = FluentdLogDriverProps(
            address=address,
            async_connect=async_connect,
            buffer_limit=buffer_limit,
            max_retries=max_retries,
            retry_wait=retry_wait,
            sub_second_precision=sub_second_precision,
            env=env,
            env_regex=env_regex,
            labels=labels,
            tag=tag,
        )

        return typing.cast(LogDriver, jsii.sinvoke(cls, "fluentd", [props]))

    @jsii.member(jsii_name="gelf") # type: ignore[misc]
    @builtins.classmethod
    def gelf(
        cls,
        *,
        address: builtins.str,
        compression_level: typing.Optional[jsii.Number] = None,
        compression_type: typing.Optional[GelfCompressionType] = None,
        tcp_max_reconnect: typing.Optional[jsii.Number] = None,
        tcp_reconnect_delay: typing.Optional[_Duration_070aa057] = None,
        env: typing.Optional[typing.Sequence[builtins.str]] = None,
        env_regex: typing.Optional[builtins.str] = None,
        labels: typing.Optional[typing.Sequence[builtins.str]] = None,
        tag: typing.Optional[builtins.str] = None,
    ) -> LogDriver:
        '''(experimental) Creates a log driver configuration that sends log information to gelf Logs.

        :param address: (experimental) The address of the GELF server. tcp and udp are the only supported URI specifier and you must specify the port.
        :param compression_level: (experimental) UDP Only The level of compression when gzip or zlib is the gelf-compression-type. An integer in the range of -1 to 9 (BestCompression). Higher levels provide more compression at lower speed. Either -1 or 0 disables compression. Default: - 1
        :param compression_type: (experimental) UDP Only The type of compression the GELF driver uses to compress each log message. Allowed values are gzip, zlib and none. Default: - gzip
        :param tcp_max_reconnect: (experimental) TCP Only The maximum number of reconnection attempts when the connection drop. A positive integer. Default: - 3
        :param tcp_reconnect_delay: (experimental) TCP Only The number of seconds to wait between reconnection attempts. A positive integer. Default: - 1
        :param env: (experimental) The env option takes an array of keys. If there is collision between label and env keys, the value of the env takes precedence. Adds additional fields to the extra attributes of a logging message. Default: - No env
        :param env_regex: (experimental) The env-regex option is similar to and compatible with env. Its value is a regular expression to match logging-related environment variables. It is used for advanced log tag options. Default: - No envRegex
        :param labels: (experimental) The labels option takes an array of keys. If there is collision between label and env keys, the value of the env takes precedence. Adds additional fields to the extra attributes of a logging message. Default: - No labels
        :param tag: (experimental) By default, Docker uses the first 12 characters of the container ID to tag log messages. Refer to the log tag option documentation for customizing the log tag format. Default: - The first 12 characters of the container ID

        :stability: experimental
        '''
        props = GelfLogDriverProps(
            address=address,
            compression_level=compression_level,
            compression_type=compression_type,
            tcp_max_reconnect=tcp_max_reconnect,
            tcp_reconnect_delay=tcp_reconnect_delay,
            env=env,
            env_regex=env_regex,
            labels=labels,
            tag=tag,
        )

        return typing.cast(LogDriver, jsii.sinvoke(cls, "gelf", [props]))

    @jsii.member(jsii_name="journald") # type: ignore[misc]
    @builtins.classmethod
    def journald(
        cls,
        *,
        env: typing.Optional[typing.Sequence[builtins.str]] = None,
        env_regex: typing.Optional[builtins.str] = None,
        labels: typing.Optional[typing.Sequence[builtins.str]] = None,
        tag: typing.Optional[builtins.str] = None,
    ) -> LogDriver:
        '''(experimental) Creates a log driver configuration that sends log information to journald Logs.

        :param env: (experimental) The env option takes an array of keys. If there is collision between label and env keys, the value of the env takes precedence. Adds additional fields to the extra attributes of a logging message. Default: - No env
        :param env_regex: (experimental) The env-regex option is similar to and compatible with env. Its value is a regular expression to match logging-related environment variables. It is used for advanced log tag options. Default: - No envRegex
        :param labels: (experimental) The labels option takes an array of keys. If there is collision between label and env keys, the value of the env takes precedence. Adds additional fields to the extra attributes of a logging message. Default: - No labels
        :param tag: (experimental) By default, Docker uses the first 12 characters of the container ID to tag log messages. Refer to the log tag option documentation for customizing the log tag format. Default: - The first 12 characters of the container ID

        :stability: experimental
        '''
        props = JournaldLogDriverProps(
            env=env, env_regex=env_regex, labels=labels, tag=tag
        )

        return typing.cast(LogDriver, jsii.sinvoke(cls, "journald", [props]))

    @jsii.member(jsii_name="jsonFile") # type: ignore[misc]
    @builtins.classmethod
    def json_file(
        cls,
        *,
        compress: typing.Optional[builtins.bool] = None,
        max_file: typing.Optional[jsii.Number] = None,
        max_size: typing.Optional[builtins.str] = None,
        env: typing.Optional[typing.Sequence[builtins.str]] = None,
        env_regex: typing.Optional[builtins.str] = None,
        labels: typing.Optional[typing.Sequence[builtins.str]] = None,
        tag: typing.Optional[builtins.str] = None,
    ) -> LogDriver:
        '''(experimental) Creates a log driver configuration that sends log information to json-file Logs.

        :param compress: (experimental) Toggles compression for rotated logs. Default: - false
        :param max_file: (experimental) The maximum number of log files that can be present. If rolling the logs creates excess files, the oldest file is removed. Only effective when max-size is also set. A positive integer. Default: - 1
        :param max_size: (experimental) The maximum size of the log before it is rolled. A positive integer plus a modifier representing the unit of measure (k, m, or g). Default: - -1 (unlimited)
        :param env: (experimental) The env option takes an array of keys. If there is collision between label and env keys, the value of the env takes precedence. Adds additional fields to the extra attributes of a logging message. Default: - No env
        :param env_regex: (experimental) The env-regex option is similar to and compatible with env. Its value is a regular expression to match logging-related environment variables. It is used for advanced log tag options. Default: - No envRegex
        :param labels: (experimental) The labels option takes an array of keys. If there is collision between label and env keys, the value of the env takes precedence. Adds additional fields to the extra attributes of a logging message. Default: - No labels
        :param tag: (experimental) By default, Docker uses the first 12 characters of the container ID to tag log messages. Refer to the log tag option documentation for customizing the log tag format. Default: - The first 12 characters of the container ID

        :stability: experimental
        '''
        props = JsonFileLogDriverProps(
            compress=compress,
            max_file=max_file,
            max_size=max_size,
            env=env,
            env_regex=env_regex,
            labels=labels,
            tag=tag,
        )

        return typing.cast(LogDriver, jsii.sinvoke(cls, "jsonFile", [props]))

    @jsii.member(jsii_name="splunk") # type: ignore[misc]
    @builtins.classmethod
    def splunk(
        cls,
        *,
        token: _SecretValue_c18506ef,
        url: builtins.str,
        ca_name: typing.Optional[builtins.str] = None,
        ca_path: typing.Optional[builtins.str] = None,
        format: typing.Optional["SplunkLogFormat"] = None,
        gzip: typing.Optional[builtins.bool] = None,
        gzip_level: typing.Optional[jsii.Number] = None,
        index: typing.Optional[builtins.str] = None,
        insecure_skip_verify: typing.Optional[builtins.str] = None,
        source: typing.Optional[builtins.str] = None,
        source_type: typing.Optional[builtins.str] = None,
        verify_connection: typing.Optional[builtins.bool] = None,
        env: typing.Optional[typing.Sequence[builtins.str]] = None,
        env_regex: typing.Optional[builtins.str] = None,
        labels: typing.Optional[typing.Sequence[builtins.str]] = None,
        tag: typing.Optional[builtins.str] = None,
    ) -> LogDriver:
        '''(experimental) Creates a log driver configuration that sends log information to splunk Logs.

        :param token: (experimental) Splunk HTTP Event Collector token.
        :param url: (experimental) Path to your Splunk Enterprise, self-service Splunk Cloud instance, or Splunk Cloud managed cluster (including port and scheme used by HTTP Event Collector) in one of the following formats: https://your_splunk_instance:8088 or https://input-prd-p-XXXXXXX.cloud.splunk.com:8088 or https://http-inputs-XXXXXXXX.splunkcloud.com.
        :param ca_name: (experimental) Name to use for validating server certificate. Default: - The hostname of the splunk-url
        :param ca_path: (experimental) Path to root certificate. Default: - caPath not set.
        :param format: (experimental) Message format. Can be inline, json or raw. Default: - inline
        :param gzip: (experimental) Enable/disable gzip compression to send events to Splunk Enterprise or Splunk Cloud instance. Default: - false
        :param gzip_level: (experimental) Set compression level for gzip. Valid values are -1 (default), 0 (no compression), 1 (best speed) ... 9 (best compression). Default: - -1 (Default Compression)
        :param index: (experimental) Event index. Default: - index not set.
        :param insecure_skip_verify: (experimental) Ignore server certificate validation. Default: - insecureSkipVerify not set.
        :param source: (experimental) Event source. Default: - source not set.
        :param source_type: (experimental) Event source type. Default: - sourceType not set.
        :param verify_connection: (experimental) Verify on start, that docker can connect to Splunk server. Default: - true
        :param env: (experimental) The env option takes an array of keys. If there is collision between label and env keys, the value of the env takes precedence. Adds additional fields to the extra attributes of a logging message. Default: - No env
        :param env_regex: (experimental) The env-regex option is similar to and compatible with env. Its value is a regular expression to match logging-related environment variables. It is used for advanced log tag options. Default: - No envRegex
        :param labels: (experimental) The labels option takes an array of keys. If there is collision between label and env keys, the value of the env takes precedence. Adds additional fields to the extra attributes of a logging message. Default: - No labels
        :param tag: (experimental) By default, Docker uses the first 12 characters of the container ID to tag log messages. Refer to the log tag option documentation for customizing the log tag format. Default: - The first 12 characters of the container ID

        :stability: experimental
        '''
        props = SplunkLogDriverProps(
            token=token,
            url=url,
            ca_name=ca_name,
            ca_path=ca_path,
            format=format,
            gzip=gzip,
            gzip_level=gzip_level,
            index=index,
            insecure_skip_verify=insecure_skip_verify,
            source=source,
            source_type=source_type,
            verify_connection=verify_connection,
            env=env,
            env_regex=env_regex,
            labels=labels,
            tag=tag,
        )

        return typing.cast(LogDriver, jsii.sinvoke(cls, "splunk", [props]))

    @jsii.member(jsii_name="syslog") # type: ignore[misc]
    @builtins.classmethod
    def syslog(
        cls,
        *,
        address: typing.Optional[builtins.str] = None,
        facility: typing.Optional[builtins.str] = None,
        format: typing.Optional[builtins.str] = None,
        tls_ca_cert: typing.Optional[builtins.str] = None,
        tls_cert: typing.Optional[builtins.str] = None,
        tls_key: typing.Optional[builtins.str] = None,
        tls_skip_verify: typing.Optional[builtins.bool] = None,
        env: typing.Optional[typing.Sequence[builtins.str]] = None,
        env_regex: typing.Optional[builtins.str] = None,
        labels: typing.Optional[typing.Sequence[builtins.str]] = None,
        tag: typing.Optional[builtins.str] = None,
    ) -> LogDriver:
        '''(experimental) Creates a log driver configuration that sends log information to syslog Logs.

        :param address: (experimental) The address of an external syslog server. The URI specifier may be [tcp|udp|tcp+tls]://host:port, unix://path, or unixgram://path. Default: - If the transport is tcp, udp, or tcp+tls, the default port is 514.
        :param facility: (experimental) The syslog facility to use. Can be the number or name for any valid syslog facility. See the syslog documentation: https://tools.ietf.org/html/rfc5424#section-6.2.1. Default: - facility not set
        :param format: (experimental) The syslog message format to use. If not specified the local UNIX syslog format is used, without a specified hostname. Specify rfc3164 for the RFC-3164 compatible format, rfc5424 for RFC-5424 compatible format, or rfc5424micro for RFC-5424 compatible format with microsecond timestamp resolution. Default: - format not set
        :param tls_ca_cert: (experimental) The absolute path to the trust certificates signed by the CA. Ignored if the address protocol is not tcp+tls. Default: - tlsCaCert not set
        :param tls_cert: (experimental) The absolute path to the TLS certificate file. Ignored if the address protocol is not tcp+tls. Default: - tlsCert not set
        :param tls_key: (experimental) The absolute path to the TLS key file. Ignored if the address protocol is not tcp+tls. Default: - tlsKey not set
        :param tls_skip_verify: (experimental) If set to true, TLS verification is skipped when connecting to the syslog daemon. Ignored if the address protocol is not tcp+tls. Default: - false
        :param env: (experimental) The env option takes an array of keys. If there is collision between label and env keys, the value of the env takes precedence. Adds additional fields to the extra attributes of a logging message. Default: - No env
        :param env_regex: (experimental) The env-regex option is similar to and compatible with env. Its value is a regular expression to match logging-related environment variables. It is used for advanced log tag options. Default: - No envRegex
        :param labels: (experimental) The labels option takes an array of keys. If there is collision between label and env keys, the value of the env takes precedence. Adds additional fields to the extra attributes of a logging message. Default: - No labels
        :param tag: (experimental) By default, Docker uses the first 12 characters of the container ID to tag log messages. Refer to the log tag option documentation for customizing the log tag format. Default: - The first 12 characters of the container ID

        :stability: experimental
        '''
        props = SyslogLogDriverProps(
            address=address,
            facility=facility,
            format=format,
            tls_ca_cert=tls_ca_cert,
            tls_cert=tls_cert,
            tls_key=tls_key,
            tls_skip_verify=tls_skip_verify,
            env=env,
            env_regex=env_regex,
            labels=labels,
            tag=tag,
        )

        return typing.cast(LogDriver, jsii.sinvoke(cls, "syslog", [props]))


@jsii.enum(jsii_type="monocdk.aws_ecs.MachineImageType")
class MachineImageType(enum.Enum):
    '''(experimental) The machine image type.

    :stability: experimental
    '''

    AMAZON_LINUX_2 = "AMAZON_LINUX_2"
    '''(experimental) Amazon ECS-optimized Amazon Linux 2 AMI.

    :stability: experimental
    '''
    BOTTLEROCKET = "BOTTLEROCKET"
    '''(experimental) Bottlerocket AMI.

    :stability: experimental
    '''


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.MemoryUtilizationScalingProps",
    jsii_struct_bases=[_BaseTargetTrackingProps_e4402570],
    name_mapping={
        "disable_scale_in": "disableScaleIn",
        "policy_name": "policyName",
        "scale_in_cooldown": "scaleInCooldown",
        "scale_out_cooldown": "scaleOutCooldown",
        "target_utilization_percent": "targetUtilizationPercent",
    },
)
class MemoryUtilizationScalingProps(_BaseTargetTrackingProps_e4402570):
    def __init__(
        self,
        *,
        disable_scale_in: typing.Optional[builtins.bool] = None,
        policy_name: typing.Optional[builtins.str] = None,
        scale_in_cooldown: typing.Optional[_Duration_070aa057] = None,
        scale_out_cooldown: typing.Optional[_Duration_070aa057] = None,
        target_utilization_percent: jsii.Number,
    ) -> None:
        '''(experimental) The properties for enabling scaling based on memory utilization.

        :param disable_scale_in: (experimental) Indicates whether scale in by the target tracking policy is disabled. If the value is true, scale in is disabled and the target tracking policy won't remove capacity from the scalable resource. Otherwise, scale in is enabled and the target tracking policy can remove capacity from the scalable resource. Default: false
        :param policy_name: (experimental) A name for the scaling policy. Default: - Automatically generated name.
        :param scale_in_cooldown: (experimental) Period after a scale in activity completes before another scale in activity can start. Default: Duration.seconds(300) for the following scalable targets: ECS services, Spot Fleet requests, EMR clusters, AppStream 2.0 fleets, Aurora DB clusters, Amazon SageMaker endpoint variants, Custom resources. For all other scalable targets, the default value is Duration.seconds(0): DynamoDB tables, DynamoDB global secondary indexes, Amazon Comprehend document classification endpoints, Lambda provisioned concurrency
        :param scale_out_cooldown: (experimental) Period after a scale out activity completes before another scale out activity can start. Default: Duration.seconds(300) for the following scalable targets: ECS services, Spot Fleet requests, EMR clusters, AppStream 2.0 fleets, Aurora DB clusters, Amazon SageMaker endpoint variants, Custom resources. For all other scalable targets, the default value is Duration.seconds(0): DynamoDB tables, DynamoDB global secondary indexes, Amazon Comprehend document classification endpoints, Lambda provisioned concurrency
        :param target_utilization_percent: (experimental) The target value for memory utilization across all tasks in the service.

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "target_utilization_percent": target_utilization_percent,
        }
        if disable_scale_in is not None:
            self._values["disable_scale_in"] = disable_scale_in
        if policy_name is not None:
            self._values["policy_name"] = policy_name
        if scale_in_cooldown is not None:
            self._values["scale_in_cooldown"] = scale_in_cooldown
        if scale_out_cooldown is not None:
            self._values["scale_out_cooldown"] = scale_out_cooldown

    @builtins.property
    def disable_scale_in(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Indicates whether scale in by the target tracking policy is disabled.

        If the value is true, scale in is disabled and the target tracking policy
        won't remove capacity from the scalable resource. Otherwise, scale in is
        enabled and the target tracking policy can remove capacity from the
        scalable resource.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("disable_scale_in")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def policy_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) A name for the scaling policy.

        :default: - Automatically generated name.

        :stability: experimental
        '''
        result = self._values.get("policy_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def scale_in_cooldown(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Period after a scale in activity completes before another scale in activity can start.

        :default:

        Duration.seconds(300) for the following scalable targets: ECS services,
        Spot Fleet requests, EMR clusters, AppStream 2.0 fleets, Aurora DB clusters,
        Amazon SageMaker endpoint variants, Custom resources. For all other scalable
        targets, the default value is Duration.seconds(0): DynamoDB tables, DynamoDB
        global secondary indexes, Amazon Comprehend document classification endpoints,
        Lambda provisioned concurrency

        :stability: experimental
        '''
        result = self._values.get("scale_in_cooldown")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def scale_out_cooldown(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Period after a scale out activity completes before another scale out activity can start.

        :default:

        Duration.seconds(300) for the following scalable targets: ECS services,
        Spot Fleet requests, EMR clusters, AppStream 2.0 fleets, Aurora DB clusters,
        Amazon SageMaker endpoint variants, Custom resources. For all other scalable
        targets, the default value is Duration.seconds(0): DynamoDB tables, DynamoDB
        global secondary indexes, Amazon Comprehend document classification endpoints,
        Lambda provisioned concurrency

        :stability: experimental
        '''
        result = self._values.get("scale_out_cooldown")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def target_utilization_percent(self) -> jsii.Number:
        '''(experimental) The target value for memory utilization across all tasks in the service.

        :stability: experimental
        '''
        result = self._values.get("target_utilization_percent")
        assert result is not None, "Required property 'target_utilization_percent' is missing"
        return typing.cast(jsii.Number, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "MemoryUtilizationScalingProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.MountPoint",
    jsii_struct_bases=[],
    name_mapping={
        "container_path": "containerPath",
        "read_only": "readOnly",
        "source_volume": "sourceVolume",
    },
)
class MountPoint:
    def __init__(
        self,
        *,
        container_path: builtins.str,
        read_only: builtins.bool,
        source_volume: builtins.str,
    ) -> None:
        '''(experimental) The details of data volume mount points for a container.

        :param container_path: (experimental) The path on the container to mount the host volume at.
        :param read_only: (experimental) Specifies whether to give the container read-only access to the volume. If this value is true, the container has read-only access to the volume. If this value is false, then the container can write to the volume.
        :param source_volume: (experimental) The name of the volume to mount. Must be a volume name referenced in the name parameter of task definition volume.

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "container_path": container_path,
            "read_only": read_only,
            "source_volume": source_volume,
        }

    @builtins.property
    def container_path(self) -> builtins.str:
        '''(experimental) The path on the container to mount the host volume at.

        :stability: experimental
        '''
        result = self._values.get("container_path")
        assert result is not None, "Required property 'container_path' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def read_only(self) -> builtins.bool:
        '''(experimental) Specifies whether to give the container read-only access to the volume.

        If this value is true, the container has read-only access to the volume.
        If this value is false, then the container can write to the volume.

        :stability: experimental
        '''
        result = self._values.get("read_only")
        assert result is not None, "Required property 'read_only' is missing"
        return typing.cast(builtins.bool, result)

    @builtins.property
    def source_volume(self) -> builtins.str:
        '''(experimental) The name of the volume to mount.

        Must be a volume name referenced in the name parameter of task definition volume.

        :stability: experimental
        '''
        result = self._values.get("source_volume")
        assert result is not None, "Required property 'source_volume' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "MountPoint(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_ecs.NetworkMode")
class NetworkMode(enum.Enum):
    '''(experimental) The networking mode to use for the containers in the task.

    :stability: experimental
    '''

    NONE = "NONE"
    '''(experimental) The task's containers do not have external connectivity and port mappings can't be specified in the container definition.

    :stability: experimental
    '''
    BRIDGE = "BRIDGE"
    '''(experimental) The task utilizes Docker's built-in virtual network which runs inside each container instance.

    :stability: experimental
    '''
    AWS_VPC = "AWS_VPC"
    '''(experimental) The task is allocated an elastic network interface.

    :stability: experimental
    '''
    HOST = "HOST"
    '''(experimental) The task bypasses Docker's built-in virtual network and maps container ports directly to the EC2 instance's network interface directly.

    In this mode, you can't run multiple instantiations of the same task on a
    single container instance when port mappings are used.

    :stability: experimental
    '''
    NAT = "NAT"
    '''(experimental) The task utilizes NAT network mode required by Windows containers.

    This is the only supported network mode for Windows containers. For more information, see
    `Task Definition Parameters <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_definition_parameters.html#network_mode>`_.

    :stability: experimental
    '''


@jsii.enum(jsii_type="monocdk.aws_ecs.PidMode")
class PidMode(enum.Enum):
    '''(experimental) The process namespace to use for the containers in the task.

    :stability: experimental
    '''

    HOST = "HOST"
    '''(experimental) If host is specified, then all containers within the tasks that specified the host PID mode on the same container instance share the same process namespace with the host Amazon EC2 instance.

    :stability: experimental
    '''
    TASK = "TASK"
    '''(experimental) If task is specified, all containers within the specified task share the same process namespace.

    :stability: experimental
    '''


class PlacementConstraint(
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_ecs.PlacementConstraint",
):
    '''(experimental) The placement constraints to use for tasks in the service. For more information, see `Amazon ECS Task Placement Constraints <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-constraints.html>`_.

    Tasks will only be placed on instances that match these rules.

    :stability: experimental
    '''

    @jsii.member(jsii_name="distinctInstances") # type: ignore[misc]
    @builtins.classmethod
    def distinct_instances(cls) -> "PlacementConstraint":
        '''(experimental) Use distinctInstance to ensure that each task in a particular group is running on a different container instance.

        :stability: experimental
        '''
        return typing.cast("PlacementConstraint", jsii.sinvoke(cls, "distinctInstances", []))

    @jsii.member(jsii_name="memberOf") # type: ignore[misc]
    @builtins.classmethod
    def member_of(cls, *expressions: builtins.str) -> "PlacementConstraint":
        '''(experimental) Use memberOf to restrict the selection to a group of valid candidates specified by a query expression.

        Multiple expressions can be specified. For more information, see
        `Cluster Query Language <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/cluster-query-language.html>`_.

        You can specify multiple expressions in one call. The tasks will only be placed on instances matching all expressions.

        :param expressions: -

        :see: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/cluster-query-language.html
        :stability: experimental
        '''
        return typing.cast("PlacementConstraint", jsii.sinvoke(cls, "memberOf", [*expressions]))

    @jsii.member(jsii_name="toJson")
    def to_json(self) -> typing.List[CfnService.PlacementConstraintProperty]:
        '''(experimental) Return the placement JSON.

        :stability: experimental
        '''
        return typing.cast(typing.List[CfnService.PlacementConstraintProperty], jsii.invoke(self, "toJson", []))


class PlacementStrategy(
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_ecs.PlacementStrategy",
):
    '''(experimental) The placement strategies to use for tasks in the service. For more information, see `Amazon ECS Task Placement Strategies <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-strategies.html>`_.

    Tasks will preferentially be placed on instances that match these rules.

    :stability: experimental
    '''

    @jsii.member(jsii_name="packedBy") # type: ignore[misc]
    @builtins.classmethod
    def packed_by(cls, resource: BinPackResource) -> "PlacementStrategy":
        '''(experimental) Places tasks on the container instances with the least available capacity of the specified resource.

        :param resource: -

        :stability: experimental
        '''
        return typing.cast("PlacementStrategy", jsii.sinvoke(cls, "packedBy", [resource]))

    @jsii.member(jsii_name="packedByCpu") # type: ignore[misc]
    @builtins.classmethod
    def packed_by_cpu(cls) -> "PlacementStrategy":
        '''(experimental) Places tasks on container instances with the least available amount of CPU capacity.

        This minimizes the number of instances in use.

        :stability: experimental
        '''
        return typing.cast("PlacementStrategy", jsii.sinvoke(cls, "packedByCpu", []))

    @jsii.member(jsii_name="packedByMemory") # type: ignore[misc]
    @builtins.classmethod
    def packed_by_memory(cls) -> "PlacementStrategy":
        '''(experimental) Places tasks on container instances with the least available amount of memory capacity.

        This minimizes the number of instances in use.

        :stability: experimental
        '''
        return typing.cast("PlacementStrategy", jsii.sinvoke(cls, "packedByMemory", []))

    @jsii.member(jsii_name="randomly") # type: ignore[misc]
    @builtins.classmethod
    def randomly(cls) -> "PlacementStrategy":
        '''(experimental) Places tasks randomly.

        :stability: experimental
        '''
        return typing.cast("PlacementStrategy", jsii.sinvoke(cls, "randomly", []))

    @jsii.member(jsii_name="spreadAcross") # type: ignore[misc]
    @builtins.classmethod
    def spread_across(cls, *fields: builtins.str) -> "PlacementStrategy":
        '''(experimental) Places tasks evenly based on the specified value.

        You can use one of the built-in attributes found on ``BuiltInAttributes``
        or supply your own custom instance attributes. If more than one attribute
        is supplied, spreading is done in order.

        :param fields: -

        :default: attributes instanceId

        :stability: experimental
        '''
        return typing.cast("PlacementStrategy", jsii.sinvoke(cls, "spreadAcross", [*fields]))

    @jsii.member(jsii_name="spreadAcrossInstances") # type: ignore[misc]
    @builtins.classmethod
    def spread_across_instances(cls) -> "PlacementStrategy":
        '''(experimental) Places tasks evenly across all container instances in the cluster.

        :stability: experimental
        '''
        return typing.cast("PlacementStrategy", jsii.sinvoke(cls, "spreadAcrossInstances", []))

    @jsii.member(jsii_name="toJson")
    def to_json(self) -> typing.List[CfnService.PlacementStrategyProperty]:
        '''(experimental) Return the placement JSON.

        :stability: experimental
        '''
        return typing.cast(typing.List[CfnService.PlacementStrategyProperty], jsii.invoke(self, "toJson", []))


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.PortMapping",
    jsii_struct_bases=[],
    name_mapping={
        "container_port": "containerPort",
        "host_port": "hostPort",
        "protocol": "protocol",
    },
)
class PortMapping:
    def __init__(
        self,
        *,
        container_port: jsii.Number,
        host_port: typing.Optional[jsii.Number] = None,
        protocol: typing.Optional["Protocol"] = None,
    ) -> None:
        '''(experimental) Port mappings allow containers to access ports on the host container instance to send or receive traffic.

        :param container_port: (experimental) The port number on the container that is bound to the user-specified or automatically assigned host port. If you are using containers in a task with the awsvpc or host network mode, exposed ports should be specified using containerPort. If you are using containers in a task with the bridge network mode and you specify a container port and not a host port, your container automatically receives a host port in the ephemeral port range. For more information, see hostPort. Port mappings that are automatically assigned in this way do not count toward the 100 reserved ports limit of a container instance.
        :param host_port: (experimental) The port number on the container instance to reserve for your container. If you are using containers in a task with the awsvpc or host network mode, the hostPort can either be left blank or set to the same value as the containerPort. If you are using containers in a task with the bridge network mode, you can specify a non-reserved host port for your container port mapping, or you can omit the hostPort (or set it to 0) while specifying a containerPort and your container automatically receives a port in the ephemeral port range for your container instance operating system and Docker version.
        :param protocol: (experimental) The protocol used for the port mapping. Valid values are Protocol.TCP and Protocol.UDP. Default: TCP

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "container_port": container_port,
        }
        if host_port is not None:
            self._values["host_port"] = host_port
        if protocol is not None:
            self._values["protocol"] = protocol

    @builtins.property
    def container_port(self) -> jsii.Number:
        '''(experimental) The port number on the container that is bound to the user-specified or automatically assigned host port.

        If you are using containers in a task with the awsvpc or host network mode, exposed ports should be specified using containerPort.
        If you are using containers in a task with the bridge network mode and you specify a container port and not a host port,
        your container automatically receives a host port in the ephemeral port range.

        For more information, see hostPort.
        Port mappings that are automatically assigned in this way do not count toward the 100 reserved ports limit of a container instance.

        :stability: experimental
        '''
        result = self._values.get("container_port")
        assert result is not None, "Required property 'container_port' is missing"
        return typing.cast(jsii.Number, result)

    @builtins.property
    def host_port(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The port number on the container instance to reserve for your container.

        If you are using containers in a task with the awsvpc or host network mode,
        the hostPort can either be left blank or set to the same value as the containerPort.

        If you are using containers in a task with the bridge network mode,
        you can specify a non-reserved host port for your container port mapping, or
        you can omit the hostPort (or set it to 0) while specifying a containerPort and
        your container automatically receives a port in the ephemeral port range for
        your container instance operating system and Docker version.

        :stability: experimental
        '''
        result = self._values.get("host_port")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def protocol(self) -> typing.Optional["Protocol"]:
        '''(experimental) The protocol used for the port mapping.

        Valid values are Protocol.TCP and Protocol.UDP.

        :default: TCP

        :stability: experimental
        '''
        result = self._values.get("protocol")
        return typing.cast(typing.Optional["Protocol"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "PortMapping(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_ecs.PropagatedTagSource")
class PropagatedTagSource(enum.Enum):
    '''(experimental) Propagate tags from either service or task definition.

    :stability: experimental
    '''

    SERVICE = "SERVICE"
    '''(experimental) Propagate tags from service.

    :stability: experimental
    '''
    TASK_DEFINITION = "TASK_DEFINITION"
    '''(experimental) Propagate tags from task definition.

    :stability: experimental
    '''
    NONE = "NONE"
    '''(experimental) Do not propagate.

    :stability: experimental
    '''


@jsii.enum(jsii_type="monocdk.aws_ecs.Protocol")
class Protocol(enum.Enum):
    '''(experimental) Network protocol.

    :stability: experimental
    '''

    TCP = "TCP"
    '''(experimental) TCP.

    :stability: experimental
    '''
    UDP = "UDP"
    '''(experimental) UDP.

    :stability: experimental
    '''


class ProxyConfiguration(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="monocdk.aws_ecs.ProxyConfiguration",
):
    '''(experimental) The base class for proxy configurations.

    :stability: experimental
    '''

    @builtins.staticmethod
    def __jsii_proxy_class__() -> typing.Type["_ProxyConfigurationProxy"]:
        return _ProxyConfigurationProxy

    def __init__(self) -> None:
        '''
        :stability: experimental
        '''
        jsii.create(ProxyConfiguration, self, [])

    @jsii.member(jsii_name="bind") # type: ignore[misc]
    @abc.abstractmethod
    def bind(
        self,
        _scope: _Construct_e78e779f,
        _task_definition: "TaskDefinition",
    ) -> CfnTaskDefinition.ProxyConfigurationProperty:
        '''(experimental) Called when the proxy configuration is configured on a task definition.

        :param _scope: -
        :param _task_definition: -

        :stability: experimental
        '''
        ...


class _ProxyConfigurationProxy(ProxyConfiguration):
    @jsii.member(jsii_name="bind")
    def bind(
        self,
        _scope: _Construct_e78e779f,
        _task_definition: "TaskDefinition",
    ) -> CfnTaskDefinition.ProxyConfigurationProperty:
        '''(experimental) Called when the proxy configuration is configured on a task definition.

        :param _scope: -
        :param _task_definition: -

        :stability: experimental
        '''
        return typing.cast(CfnTaskDefinition.ProxyConfigurationProperty, jsii.invoke(self, "bind", [_scope, _task_definition]))


class ProxyConfigurations(
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_ecs.ProxyConfigurations",
):
    '''(experimental) The base class for proxy configurations.

    :stability: experimental
    '''

    def __init__(self) -> None:
        '''
        :stability: experimental
        '''
        jsii.create(ProxyConfigurations, self, [])

    @jsii.member(jsii_name="appMeshProxyConfiguration") # type: ignore[misc]
    @builtins.classmethod
    def app_mesh_proxy_configuration(
        cls,
        *,
        container_name: builtins.str,
        properties: AppMeshProxyConfigurationProps,
    ) -> ProxyConfiguration:
        '''(experimental) Constructs a new instance of the ProxyConfiguration class.

        :param container_name: (experimental) The name of the container that will serve as the App Mesh proxy.
        :param properties: (experimental) The set of network configuration parameters to provide the Container Network Interface (CNI) plugin.

        :stability: experimental
        '''
        props = AppMeshProxyConfigurationConfigProps(
            container_name=container_name, properties=properties
        )

        return typing.cast(ProxyConfiguration, jsii.sinvoke(cls, "appMeshProxyConfiguration", [props]))


class RepositoryImage(
    ContainerImage,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_ecs.RepositoryImage",
):
    '''(experimental) An image hosted in a public or private repository.

    For images hosted in Amazon ECR, see
    `EcrImage <https://docs.aws.amazon.com/AmazonECR/latest/userguide/images.html>`_.

    :stability: experimental
    '''

    def __init__(
        self,
        image_name: builtins.str,
        *,
        credentials: typing.Optional[_ISecret_22fb8757] = None,
    ) -> None:
        '''(experimental) Constructs a new instance of the RepositoryImage class.

        :param image_name: -
        :param credentials: (experimental) The secret to expose to the container that contains the credentials for the image repository. The supported value is the full ARN of an AWS Secrets Manager secret.

        :stability: experimental
        '''
        props = RepositoryImageProps(credentials=credentials)

        jsii.create(RepositoryImage, self, [image_name, props])

    @jsii.member(jsii_name="bind")
    def bind(
        self,
        scope: _Construct_e78e779f,
        container_definition: ContainerDefinition,
    ) -> ContainerImageConfig:
        '''(experimental) Called when the image is used by a ContainerDefinition.

        :param scope: -
        :param container_definition: -

        :stability: experimental
        '''
        return typing.cast(ContainerImageConfig, jsii.invoke(self, "bind", [scope, container_definition]))


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.RepositoryImageProps",
    jsii_struct_bases=[],
    name_mapping={"credentials": "credentials"},
)
class RepositoryImageProps:
    def __init__(
        self,
        *,
        credentials: typing.Optional[_ISecret_22fb8757] = None,
    ) -> None:
        '''(experimental) The properties for an image hosted in a public or private repository.

        :param credentials: (experimental) The secret to expose to the container that contains the credentials for the image repository. The supported value is the full ARN of an AWS Secrets Manager secret.

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if credentials is not None:
            self._values["credentials"] = credentials

    @builtins.property
    def credentials(self) -> typing.Optional[_ISecret_22fb8757]:
        '''(experimental) The secret to expose to the container that contains the credentials for the image repository.

        The supported value is the full ARN of an AWS Secrets Manager secret.

        :stability: experimental
        '''
        result = self._values.get("credentials")
        return typing.cast(typing.Optional[_ISecret_22fb8757], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "RepositoryImageProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.RequestCountScalingProps",
    jsii_struct_bases=[_BaseTargetTrackingProps_e4402570],
    name_mapping={
        "disable_scale_in": "disableScaleIn",
        "policy_name": "policyName",
        "scale_in_cooldown": "scaleInCooldown",
        "scale_out_cooldown": "scaleOutCooldown",
        "requests_per_target": "requestsPerTarget",
        "target_group": "targetGroup",
    },
)
class RequestCountScalingProps(_BaseTargetTrackingProps_e4402570):
    def __init__(
        self,
        *,
        disable_scale_in: typing.Optional[builtins.bool] = None,
        policy_name: typing.Optional[builtins.str] = None,
        scale_in_cooldown: typing.Optional[_Duration_070aa057] = None,
        scale_out_cooldown: typing.Optional[_Duration_070aa057] = None,
        requests_per_target: jsii.Number,
        target_group: _ApplicationTargetGroup_5b334416,
    ) -> None:
        '''(experimental) The properties for enabling scaling based on Application Load Balancer (ALB) request counts.

        :param disable_scale_in: (experimental) Indicates whether scale in by the target tracking policy is disabled. If the value is true, scale in is disabled and the target tracking policy won't remove capacity from the scalable resource. Otherwise, scale in is enabled and the target tracking policy can remove capacity from the scalable resource. Default: false
        :param policy_name: (experimental) A name for the scaling policy. Default: - Automatically generated name.
        :param scale_in_cooldown: (experimental) Period after a scale in activity completes before another scale in activity can start. Default: Duration.seconds(300) for the following scalable targets: ECS services, Spot Fleet requests, EMR clusters, AppStream 2.0 fleets, Aurora DB clusters, Amazon SageMaker endpoint variants, Custom resources. For all other scalable targets, the default value is Duration.seconds(0): DynamoDB tables, DynamoDB global secondary indexes, Amazon Comprehend document classification endpoints, Lambda provisioned concurrency
        :param scale_out_cooldown: (experimental) Period after a scale out activity completes before another scale out activity can start. Default: Duration.seconds(300) for the following scalable targets: ECS services, Spot Fleet requests, EMR clusters, AppStream 2.0 fleets, Aurora DB clusters, Amazon SageMaker endpoint variants, Custom resources. For all other scalable targets, the default value is Duration.seconds(0): DynamoDB tables, DynamoDB global secondary indexes, Amazon Comprehend document classification endpoints, Lambda provisioned concurrency
        :param requests_per_target: (experimental) The number of ALB requests per target.
        :param target_group: (experimental) The ALB target group name.

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "requests_per_target": requests_per_target,
            "target_group": target_group,
        }
        if disable_scale_in is not None:
            self._values["disable_scale_in"] = disable_scale_in
        if policy_name is not None:
            self._values["policy_name"] = policy_name
        if scale_in_cooldown is not None:
            self._values["scale_in_cooldown"] = scale_in_cooldown
        if scale_out_cooldown is not None:
            self._values["scale_out_cooldown"] = scale_out_cooldown

    @builtins.property
    def disable_scale_in(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Indicates whether scale in by the target tracking policy is disabled.

        If the value is true, scale in is disabled and the target tracking policy
        won't remove capacity from the scalable resource. Otherwise, scale in is
        enabled and the target tracking policy can remove capacity from the
        scalable resource.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("disable_scale_in")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def policy_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) A name for the scaling policy.

        :default: - Automatically generated name.

        :stability: experimental
        '''
        result = self._values.get("policy_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def scale_in_cooldown(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Period after a scale in activity completes before another scale in activity can start.

        :default:

        Duration.seconds(300) for the following scalable targets: ECS services,
        Spot Fleet requests, EMR clusters, AppStream 2.0 fleets, Aurora DB clusters,
        Amazon SageMaker endpoint variants, Custom resources. For all other scalable
        targets, the default value is Duration.seconds(0): DynamoDB tables, DynamoDB
        global secondary indexes, Amazon Comprehend document classification endpoints,
        Lambda provisioned concurrency

        :stability: experimental
        '''
        result = self._values.get("scale_in_cooldown")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def scale_out_cooldown(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Period after a scale out activity completes before another scale out activity can start.

        :default:

        Duration.seconds(300) for the following scalable targets: ECS services,
        Spot Fleet requests, EMR clusters, AppStream 2.0 fleets, Aurora DB clusters,
        Amazon SageMaker endpoint variants, Custom resources. For all other scalable
        targets, the default value is Duration.seconds(0): DynamoDB tables, DynamoDB
        global secondary indexes, Amazon Comprehend document classification endpoints,
        Lambda provisioned concurrency

        :stability: experimental
        '''
        result = self._values.get("scale_out_cooldown")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def requests_per_target(self) -> jsii.Number:
        '''(experimental) The number of ALB requests per target.

        :stability: experimental
        '''
        result = self._values.get("requests_per_target")
        assert result is not None, "Required property 'requests_per_target' is missing"
        return typing.cast(jsii.Number, result)

    @builtins.property
    def target_group(self) -> _ApplicationTargetGroup_5b334416:
        '''(experimental) The ALB target group name.

        :stability: experimental
        '''
        result = self._values.get("target_group")
        assert result is not None, "Required property 'target_group' is missing"
        return typing.cast(_ApplicationTargetGroup_5b334416, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "RequestCountScalingProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class S3EnvironmentFile(
    EnvironmentFile,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_ecs.S3EnvironmentFile",
):
    '''(experimental) Environment file from S3.

    :stability: experimental
    '''

    def __init__(
        self,
        bucket: _IBucket_73486e29,
        key: builtins.str,
        object_version: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param bucket: -
        :param key: -
        :param object_version: -

        :stability: experimental
        '''
        jsii.create(S3EnvironmentFile, self, [bucket, key, object_version])

    @jsii.member(jsii_name="bind")
    def bind(self, _scope: _Construct_e78e779f) -> EnvironmentFileConfig:
        '''(experimental) Called when the container is initialized to allow this object to bind to the stack.

        :param _scope: -

        :stability: experimental
        '''
        return typing.cast(EnvironmentFileConfig, jsii.invoke(self, "bind", [_scope]))


class ScalableTaskCount(
    _BaseScalableAttribute_8cad7a2c,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_ecs.ScalableTaskCount",
):
    '''(experimental) The scalable attribute representing task count.

    :stability: experimental
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        dimension: builtins.str,
        resource_id: builtins.str,
        role: _IRole_59af6f50,
        service_namespace: _ServiceNamespace_42590eaf,
        max_capacity: jsii.Number,
        min_capacity: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''(experimental) Constructs a new instance of the ScalableTaskCount class.

        :param scope: -
        :param id: -
        :param dimension: (experimental) Scalable dimension of the attribute.
        :param resource_id: (experimental) Resource ID of the attribute.
        :param role: (experimental) Role to use for scaling.
        :param service_namespace: (experimental) Service namespace of the scalable attribute.
        :param max_capacity: (experimental) Maximum capacity to scale to.
        :param min_capacity: (experimental) Minimum capacity to scale to. Default: 1

        :stability: experimental
        '''
        props = ScalableTaskCountProps(
            dimension=dimension,
            resource_id=resource_id,
            role=role,
            service_namespace=service_namespace,
            max_capacity=max_capacity,
            min_capacity=min_capacity,
        )

        jsii.create(ScalableTaskCount, self, [scope, id, props])

    @jsii.member(jsii_name="scaleOnCpuUtilization")
    def scale_on_cpu_utilization(
        self,
        id: builtins.str,
        *,
        target_utilization_percent: jsii.Number,
        disable_scale_in: typing.Optional[builtins.bool] = None,
        policy_name: typing.Optional[builtins.str] = None,
        scale_in_cooldown: typing.Optional[_Duration_070aa057] = None,
        scale_out_cooldown: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''(experimental) Scales in or out to achieve a target CPU utilization.

        :param id: -
        :param target_utilization_percent: (experimental) The target value for CPU utilization across all tasks in the service.
        :param disable_scale_in: (experimental) Indicates whether scale in by the target tracking policy is disabled. If the value is true, scale in is disabled and the target tracking policy won't remove capacity from the scalable resource. Otherwise, scale in is enabled and the target tracking policy can remove capacity from the scalable resource. Default: false
        :param policy_name: (experimental) A name for the scaling policy. Default: - Automatically generated name.
        :param scale_in_cooldown: (experimental) Period after a scale in activity completes before another scale in activity can start. Default: Duration.seconds(300) for the following scalable targets: ECS services, Spot Fleet requests, EMR clusters, AppStream 2.0 fleets, Aurora DB clusters, Amazon SageMaker endpoint variants, Custom resources. For all other scalable targets, the default value is Duration.seconds(0): DynamoDB tables, DynamoDB global secondary indexes, Amazon Comprehend document classification endpoints, Lambda provisioned concurrency
        :param scale_out_cooldown: (experimental) Period after a scale out activity completes before another scale out activity can start. Default: Duration.seconds(300) for the following scalable targets: ECS services, Spot Fleet requests, EMR clusters, AppStream 2.0 fleets, Aurora DB clusters, Amazon SageMaker endpoint variants, Custom resources. For all other scalable targets, the default value is Duration.seconds(0): DynamoDB tables, DynamoDB global secondary indexes, Amazon Comprehend document classification endpoints, Lambda provisioned concurrency

        :stability: experimental
        '''
        props = CpuUtilizationScalingProps(
            target_utilization_percent=target_utilization_percent,
            disable_scale_in=disable_scale_in,
            policy_name=policy_name,
            scale_in_cooldown=scale_in_cooldown,
            scale_out_cooldown=scale_out_cooldown,
        )

        return typing.cast(None, jsii.invoke(self, "scaleOnCpuUtilization", [id, props]))

    @jsii.member(jsii_name="scaleOnMemoryUtilization")
    def scale_on_memory_utilization(
        self,
        id: builtins.str,
        *,
        target_utilization_percent: jsii.Number,
        disable_scale_in: typing.Optional[builtins.bool] = None,
        policy_name: typing.Optional[builtins.str] = None,
        scale_in_cooldown: typing.Optional[_Duration_070aa057] = None,
        scale_out_cooldown: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''(experimental) Scales in or out to achieve a target memory utilization.

        :param id: -
        :param target_utilization_percent: (experimental) The target value for memory utilization across all tasks in the service.
        :param disable_scale_in: (experimental) Indicates whether scale in by the target tracking policy is disabled. If the value is true, scale in is disabled and the target tracking policy won't remove capacity from the scalable resource. Otherwise, scale in is enabled and the target tracking policy can remove capacity from the scalable resource. Default: false
        :param policy_name: (experimental) A name for the scaling policy. Default: - Automatically generated name.
        :param scale_in_cooldown: (experimental) Period after a scale in activity completes before another scale in activity can start. Default: Duration.seconds(300) for the following scalable targets: ECS services, Spot Fleet requests, EMR clusters, AppStream 2.0 fleets, Aurora DB clusters, Amazon SageMaker endpoint variants, Custom resources. For all other scalable targets, the default value is Duration.seconds(0): DynamoDB tables, DynamoDB global secondary indexes, Amazon Comprehend document classification endpoints, Lambda provisioned concurrency
        :param scale_out_cooldown: (experimental) Period after a scale out activity completes before another scale out activity can start. Default: Duration.seconds(300) for the following scalable targets: ECS services, Spot Fleet requests, EMR clusters, AppStream 2.0 fleets, Aurora DB clusters, Amazon SageMaker endpoint variants, Custom resources. For all other scalable targets, the default value is Duration.seconds(0): DynamoDB tables, DynamoDB global secondary indexes, Amazon Comprehend document classification endpoints, Lambda provisioned concurrency

        :stability: experimental
        '''
        props = MemoryUtilizationScalingProps(
            target_utilization_percent=target_utilization_percent,
            disable_scale_in=disable_scale_in,
            policy_name=policy_name,
            scale_in_cooldown=scale_in_cooldown,
            scale_out_cooldown=scale_out_cooldown,
        )

        return typing.cast(None, jsii.invoke(self, "scaleOnMemoryUtilization", [id, props]))

    @jsii.member(jsii_name="scaleOnMetric")
    def scale_on_metric(
        self,
        id: builtins.str,
        *,
        metric: _IMetric_5db43d61,
        scaling_steps: typing.Sequence[_ScalingInterval_ed8cf944],
        adjustment_type: typing.Optional[_AdjustmentType_c97ebf31] = None,
        cooldown: typing.Optional[_Duration_070aa057] = None,
        evaluation_periods: typing.Optional[jsii.Number] = None,
        metric_aggregation_type: typing.Optional[_MetricAggregationType_c42ebfd0] = None,
        min_adjustment_magnitude: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''(experimental) Scales in or out based on a specified metric value.

        :param id: -
        :param metric: (experimental) Metric to scale on.
        :param scaling_steps: (experimental) The intervals for scaling. Maps a range of metric values to a particular scaling behavior.
        :param adjustment_type: (experimental) How the adjustment numbers inside 'intervals' are interpreted. Default: ChangeInCapacity
        :param cooldown: (experimental) Grace period after scaling activity. Subsequent scale outs during the cooldown period are squashed so that only the biggest scale out happens. Subsequent scale ins during the cooldown period are ignored. Default: No cooldown period
        :param evaluation_periods: (experimental) How many evaluation periods of the metric to wait before triggering a scaling action. Raising this value can be used to smooth out the metric, at the expense of slower response times. Default: 1
        :param metric_aggregation_type: (experimental) Aggregation to apply to all data points over the evaluation periods. Only has meaning if ``evaluationPeriods != 1``. Default: - The statistic from the metric if applicable (MIN, MAX, AVERAGE), otherwise AVERAGE.
        :param min_adjustment_magnitude: (experimental) Minimum absolute number to adjust capacity with as result of percentage scaling. Only when using AdjustmentType = PercentChangeInCapacity, this number controls the minimum absolute effect size. Default: No minimum scaling effect

        :stability: experimental
        '''
        props = _BasicStepScalingPolicyProps_2c327255(
            metric=metric,
            scaling_steps=scaling_steps,
            adjustment_type=adjustment_type,
            cooldown=cooldown,
            evaluation_periods=evaluation_periods,
            metric_aggregation_type=metric_aggregation_type,
            min_adjustment_magnitude=min_adjustment_magnitude,
        )

        return typing.cast(None, jsii.invoke(self, "scaleOnMetric", [id, props]))

    @jsii.member(jsii_name="scaleOnRequestCount")
    def scale_on_request_count(
        self,
        id: builtins.str,
        *,
        requests_per_target: jsii.Number,
        target_group: _ApplicationTargetGroup_5b334416,
        disable_scale_in: typing.Optional[builtins.bool] = None,
        policy_name: typing.Optional[builtins.str] = None,
        scale_in_cooldown: typing.Optional[_Duration_070aa057] = None,
        scale_out_cooldown: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''(experimental) Scales in or out to achieve a target Application Load Balancer request count per target.

        :param id: -
        :param requests_per_target: (experimental) The number of ALB requests per target.
        :param target_group: (experimental) The ALB target group name.
        :param disable_scale_in: (experimental) Indicates whether scale in by the target tracking policy is disabled. If the value is true, scale in is disabled and the target tracking policy won't remove capacity from the scalable resource. Otherwise, scale in is enabled and the target tracking policy can remove capacity from the scalable resource. Default: false
        :param policy_name: (experimental) A name for the scaling policy. Default: - Automatically generated name.
        :param scale_in_cooldown: (experimental) Period after a scale in activity completes before another scale in activity can start. Default: Duration.seconds(300) for the following scalable targets: ECS services, Spot Fleet requests, EMR clusters, AppStream 2.0 fleets, Aurora DB clusters, Amazon SageMaker endpoint variants, Custom resources. For all other scalable targets, the default value is Duration.seconds(0): DynamoDB tables, DynamoDB global secondary indexes, Amazon Comprehend document classification endpoints, Lambda provisioned concurrency
        :param scale_out_cooldown: (experimental) Period after a scale out activity completes before another scale out activity can start. Default: Duration.seconds(300) for the following scalable targets: ECS services, Spot Fleet requests, EMR clusters, AppStream 2.0 fleets, Aurora DB clusters, Amazon SageMaker endpoint variants, Custom resources. For all other scalable targets, the default value is Duration.seconds(0): DynamoDB tables, DynamoDB global secondary indexes, Amazon Comprehend document classification endpoints, Lambda provisioned concurrency

        :stability: experimental
        '''
        props = RequestCountScalingProps(
            requests_per_target=requests_per_target,
            target_group=target_group,
            disable_scale_in=disable_scale_in,
            policy_name=policy_name,
            scale_in_cooldown=scale_in_cooldown,
            scale_out_cooldown=scale_out_cooldown,
        )

        return typing.cast(None, jsii.invoke(self, "scaleOnRequestCount", [id, props]))

    @jsii.member(jsii_name="scaleOnSchedule")
    def scale_on_schedule(
        self,
        id: builtins.str,
        *,
        schedule: _Schedule_c2a5a45d,
        end_time: typing.Optional[datetime.datetime] = None,
        max_capacity: typing.Optional[jsii.Number] = None,
        min_capacity: typing.Optional[jsii.Number] = None,
        start_time: typing.Optional[datetime.datetime] = None,
    ) -> None:
        '''(experimental) Scales in or out based on a specified scheduled time.

        :param id: -
        :param schedule: (experimental) When to perform this action.
        :param end_time: (experimental) When this scheduled action expires. Default: The rule never expires.
        :param max_capacity: (experimental) The new maximum capacity. During the scheduled time, the current capacity is above the maximum capacity, Application Auto Scaling scales in to the maximum capacity. At least one of maxCapacity and minCapacity must be supplied. Default: No new maximum capacity
        :param min_capacity: (experimental) The new minimum capacity. During the scheduled time, if the current capacity is below the minimum capacity, Application Auto Scaling scales out to the minimum capacity. At least one of maxCapacity and minCapacity must be supplied. Default: No new minimum capacity
        :param start_time: (experimental) When this scheduled action becomes active. Default: The rule is activate immediately

        :stability: experimental
        '''
        props = _ScalingSchedule_6c3fc38f(
            schedule=schedule,
            end_time=end_time,
            max_capacity=max_capacity,
            min_capacity=min_capacity,
            start_time=start_time,
        )

        return typing.cast(None, jsii.invoke(self, "scaleOnSchedule", [id, props]))

    @jsii.member(jsii_name="scaleToTrackCustomMetric")
    def scale_to_track_custom_metric(
        self,
        id: builtins.str,
        *,
        metric: _IMetric_5db43d61,
        target_value: jsii.Number,
        disable_scale_in: typing.Optional[builtins.bool] = None,
        policy_name: typing.Optional[builtins.str] = None,
        scale_in_cooldown: typing.Optional[_Duration_070aa057] = None,
        scale_out_cooldown: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''(experimental) Scales in or out to achieve a target on a custom metric.

        :param id: -
        :param metric: (experimental) The custom CloudWatch metric to track. The metric must represent utilization; that is, you will always get the following behavior: - metric > targetValue => scale out - metric < targetValue => scale in
        :param target_value: (experimental) The target value for the custom CloudWatch metric.
        :param disable_scale_in: (experimental) Indicates whether scale in by the target tracking policy is disabled. If the value is true, scale in is disabled and the target tracking policy won't remove capacity from the scalable resource. Otherwise, scale in is enabled and the target tracking policy can remove capacity from the scalable resource. Default: false
        :param policy_name: (experimental) A name for the scaling policy. Default: - Automatically generated name.
        :param scale_in_cooldown: (experimental) Period after a scale in activity completes before another scale in activity can start. Default: Duration.seconds(300) for the following scalable targets: ECS services, Spot Fleet requests, EMR clusters, AppStream 2.0 fleets, Aurora DB clusters, Amazon SageMaker endpoint variants, Custom resources. For all other scalable targets, the default value is Duration.seconds(0): DynamoDB tables, DynamoDB global secondary indexes, Amazon Comprehend document classification endpoints, Lambda provisioned concurrency
        :param scale_out_cooldown: (experimental) Period after a scale out activity completes before another scale out activity can start. Default: Duration.seconds(300) for the following scalable targets: ECS services, Spot Fleet requests, EMR clusters, AppStream 2.0 fleets, Aurora DB clusters, Amazon SageMaker endpoint variants, Custom resources. For all other scalable targets, the default value is Duration.seconds(0): DynamoDB tables, DynamoDB global secondary indexes, Amazon Comprehend document classification endpoints, Lambda provisioned concurrency

        :stability: experimental
        '''
        props = TrackCustomMetricProps(
            metric=metric,
            target_value=target_value,
            disable_scale_in=disable_scale_in,
            policy_name=policy_name,
            scale_in_cooldown=scale_in_cooldown,
            scale_out_cooldown=scale_out_cooldown,
        )

        return typing.cast(None, jsii.invoke(self, "scaleToTrackCustomMetric", [id, props]))


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.ScalableTaskCountProps",
    jsii_struct_bases=[_BaseScalableAttributeProps_1600c742],
    name_mapping={
        "max_capacity": "maxCapacity",
        "min_capacity": "minCapacity",
        "dimension": "dimension",
        "resource_id": "resourceId",
        "role": "role",
        "service_namespace": "serviceNamespace",
    },
)
class ScalableTaskCountProps(_BaseScalableAttributeProps_1600c742):
    def __init__(
        self,
        *,
        max_capacity: jsii.Number,
        min_capacity: typing.Optional[jsii.Number] = None,
        dimension: builtins.str,
        resource_id: builtins.str,
        role: _IRole_59af6f50,
        service_namespace: _ServiceNamespace_42590eaf,
    ) -> None:
        '''(experimental) The properties of a scalable attribute representing task count.

        :param max_capacity: (experimental) Maximum capacity to scale to.
        :param min_capacity: (experimental) Minimum capacity to scale to. Default: 1
        :param dimension: (experimental) Scalable dimension of the attribute.
        :param resource_id: (experimental) Resource ID of the attribute.
        :param role: (experimental) Role to use for scaling.
        :param service_namespace: (experimental) Service namespace of the scalable attribute.

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "max_capacity": max_capacity,
            "dimension": dimension,
            "resource_id": resource_id,
            "role": role,
            "service_namespace": service_namespace,
        }
        if min_capacity is not None:
            self._values["min_capacity"] = min_capacity

    @builtins.property
    def max_capacity(self) -> jsii.Number:
        '''(experimental) Maximum capacity to scale to.

        :stability: experimental
        '''
        result = self._values.get("max_capacity")
        assert result is not None, "Required property 'max_capacity' is missing"
        return typing.cast(jsii.Number, result)

    @builtins.property
    def min_capacity(self) -> typing.Optional[jsii.Number]:
        '''(experimental) Minimum capacity to scale to.

        :default: 1

        :stability: experimental
        '''
        result = self._values.get("min_capacity")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def dimension(self) -> builtins.str:
        '''(experimental) Scalable dimension of the attribute.

        :stability: experimental
        '''
        result = self._values.get("dimension")
        assert result is not None, "Required property 'dimension' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def resource_id(self) -> builtins.str:
        '''(experimental) Resource ID of the attribute.

        :stability: experimental
        '''
        result = self._values.get("resource_id")
        assert result is not None, "Required property 'resource_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def role(self) -> _IRole_59af6f50:
        '''(experimental) Role to use for scaling.

        :stability: experimental
        '''
        result = self._values.get("role")
        assert result is not None, "Required property 'role' is missing"
        return typing.cast(_IRole_59af6f50, result)

    @builtins.property
    def service_namespace(self) -> _ServiceNamespace_42590eaf:
        '''(experimental) Service namespace of the scalable attribute.

        :stability: experimental
        '''
        result = self._values.get("service_namespace")
        assert result is not None, "Required property 'service_namespace' is missing"
        return typing.cast(_ServiceNamespace_42590eaf, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ScalableTaskCountProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_ecs.Scope")
class Scope(enum.Enum):
    '''(experimental) The scope for the Docker volume that determines its lifecycle.

    Docker volumes that are scoped to a task are automatically provisioned when the task starts and destroyed when the task stops.
    Docker volumes that are scoped as shared persist after the task stops.

    :stability: experimental
    '''

    TASK = "TASK"
    '''(experimental) Docker volumes that are scoped to a task are automatically provisioned when the task starts and destroyed when the task stops.

    :stability: experimental
    '''
    SHARED = "SHARED"
    '''(experimental) Docker volumes that are scoped as shared persist after the task stops.

    :stability: experimental
    '''


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.ScratchSpace",
    jsii_struct_bases=[],
    name_mapping={
        "container_path": "containerPath",
        "name": "name",
        "read_only": "readOnly",
        "source_path": "sourcePath",
    },
)
class ScratchSpace:
    def __init__(
        self,
        *,
        container_path: builtins.str,
        name: builtins.str,
        read_only: builtins.bool,
        source_path: builtins.str,
    ) -> None:
        '''(experimental) The temporary disk space mounted to the container.

        :param container_path: (experimental) The path on the container to mount the scratch volume at.
        :param name: (experimental) The name of the scratch volume to mount. Must be a volume name referenced in the name parameter of task definition volume.
        :param read_only: (experimental) Specifies whether to give the container read-only access to the scratch volume. If this value is true, the container has read-only access to the scratch volume. If this value is false, then the container can write to the scratch volume.
        :param source_path: 

        :stability: experimental
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "container_path": container_path,
            "name": name,
            "read_only": read_only,
            "source_path": source_path,
        }

    @builtins.property
    def container_path(self) -> builtins.str:
        '''(experimental) The path on the container to mount the scratch volume at.

        :stability: experimental
        '''
        result = self._values.get("container_path")
        assert result is not None, "Required property 'container_path' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def name(self) -> builtins.str:
        '''(experimental) The name of the scratch volume to mount.

        Must be a volume name referenced in the name parameter of task definition volume.

        :stability: experimental
        '''
        result = self._values.get("name")
        assert result is not None, "Required property 'name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def read_only(self) -> builtins.bool:
        '''(experimental) Specifies whether to give the container read-only access to the scratch volume.

        If this value is true, the container has read-only access to the scratch volume.
        If this value is false, then the container can write to the scratch volume.

        :stability: experimental
        '''
        result = self._values.get("read_only")
        assert result is not None, "Required property 'read_only' is missing"
        return typing.cast(builtins.bool, result)

    @builtins.property
    def source_path(self) -> builtins.str:
        '''
        :stability: experimental
        '''
        result = self._values.get("source_path")
        assert result is not None, "Required property 'source_path' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ScratchSpace(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class Secret(metaclass=jsii.JSIIAbstractClass, jsii_type="monocdk.aws_ecs.Secret"):
    '''(experimental) A secret environment variable.

    :stability: experimental
    '''

    @builtins.staticmethod
    def __jsii_proxy_class__() -> typing.Type["_SecretProxy"]:
        return _SecretProxy

    def __init__(self) -> None:
        '''
        :stability: experimental
        '''
        jsii.create(Secret, self, [])

    @jsii.member(jsii_name="fromSecretsManager") # type: ignore[misc]
    @builtins.classmethod
    def from_secrets_manager(
        cls,
        secret: _ISecret_22fb8757,
        field: typing.Optional[builtins.str] = None,
    ) -> "Secret":
        '''(experimental) Creates a environment variable value from a secret stored in AWS Secrets Manager.

        :param secret: the secret stored in AWS Secrets Manager.
        :param field: the name of the field with the value that you want to set as the environment variable value. Only values in JSON format are supported. If you do not specify a JSON field, then the full content of the secret is used.

        :stability: experimental
        '''
        return typing.cast("Secret", jsii.sinvoke(cls, "fromSecretsManager", [secret, field]))

    @jsii.member(jsii_name="fromSsmParameter") # type: ignore[misc]
    @builtins.classmethod
    def from_ssm_parameter(cls, parameter: _IParameter_ce5fb757) -> "Secret":
        '''(experimental) Creates an environment variable value from a parameter stored in AWS Systems Manager Parameter Store.

        :param parameter: -

        :stability: experimental
        '''
        return typing.cast("Secret", jsii.sinvoke(cls, "fromSsmParameter", [parameter]))

    @jsii.member(jsii_name="grantRead") # type: ignore[misc]
    @abc.abstractmethod
    def grant_read(self, grantee: _IGrantable_4c5a91d1) -> _Grant_bcb5eae7:
        '''(experimental) Grants reading the secret to a principal.

        :param grantee: -

        :stability: experimental
        '''
        ...

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="arn")
    @abc.abstractmethod
    def arn(self) -> builtins.str:
        '''(experimental) The ARN of the secret.

        :stability: experimental
        '''
        ...

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="hasField")
    @abc.abstractmethod
    def has_field(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether this secret uses a specific JSON field.

        :stability: experimental
        '''
        ...


class _SecretProxy(Secret):
    @jsii.member(jsii_name="grantRead")
    def grant_read(self, grantee: _IGrantable_4c5a91d1) -> _Grant_bcb5eae7:
        '''(experimental) Grants reading the secret to a principal.

        :param grantee: -

        :stability: experimental
        '''
        return typing.cast(_Grant_bcb5eae7, jsii.invoke(self, "grantRead", [grantee]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="arn")
    def arn(self) -> builtins.str:
        '''(experimental) The ARN of the secret.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "arn"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="hasField")
    def has_field(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether this secret uses a specific JSON field.

        :stability: experimental
        '''
        return typing.cast(typing.Optional[builtins.bool], jsii.get(self, "hasField"))


class SplunkLogDriver(
    LogDriver,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_ecs.SplunkLogDriver",
):
    '''(experimental) A log driver that sends log information 