"""Includes the supermodels decorator and namespace methods."""
import json
from collections import deque
from dataclasses import asdict, dataclass, field, fields, is_dataclass, make_dataclass
from datetime import datetime
from typing import Any, Optional

from supermodels.exceptions import ValidationError
from supermodels.serdes import (
    DataModelDeserializer,
    DataModelSerializer,
    FieldDeserializer,
)
from supermodels.util import get_timestamp, get_uuid, is_supermodel

masked_field = lambda: field(metadata=dict(mask=True))
uuid_field = lambda: field(default_factory=get_uuid)
timestamp_field = lambda: field(default_factory=get_timestamp, repr=False, kw_only=True)
version_field = lambda: field(default_factory=lambda: 1, repr=False, kw_only=True)


def __post_init__(self):
    """Validates the supermodel instance data against its rules."""
    fails = []

    for own_field in fields(self):
        value = getattr(self, own_field.name)
        rules = own_field.metadata.get("rules", None)

        if rules:
            rule_fails = rules.validate(value)
            fails.extend(
                [
                    f"{own_field.name}_{rule_name}_{rule_value}"
                    for rule_name, rule_value in rule_fails
                ]
            )

    if not fails:
        return

    msg = "Failing rules: " + "; ".join(fails)
    self.raise_validation_error(msg)


def demodel_data(
    cls_or_self, data: dict[str, Any], repr_only: Optional[bool] = False
) -> dict[str, Any]:
    """Returns a dict with all nested models converted to dicts."""
    result = {}

    for own_field in fields(cls_or_self):
        if repr_only and not own_field.repr:
            continue

        attr = own_field.name
        value = data.get(attr)

        if value is None:
            result[attr] = value
            continue

        iter_types = (list, tuple, set, frozenset, deque)
        is_iter = isinstance(value, iter_types)
        values = list(value) if is_iter else [value]

        for i, value in enumerate(values):
            if is_supermodel(value):
                values[i] = value.to_dict(repr_only=repr_only)
            elif is_dataclass(value):
                values[i] = asdict(value)

        result[attr] = values if is_iter else values[0]

    return result


def copy_data(cls, data: dict[str, Any]) -> dict[str, Any]:
    """Returns a copy of a data dict via the custom json decoder."""
    data = demodel_data(cls, data)
    payload = json.dumps(data, cls=DataModelSerializer)
    return DataModelDeserializer.deserialize(payload)


def from_dict(cls, data: dict[str, Any]):
    """Returns a model instance with the provided attributes."""
    data = copy_data(cls, data)

    for own_field in fields(cls):
        key = own_field.name

        if key not in data:
            continue

        value = data[key]
        deser = FieldDeserializer(own_field.type, value)
        data[key] = deser.get_deserialized_value()

    timestamp_attrs = ("_created", "_updated")
    meta_attrs = timestamp_attrs + ("_version",)

    if any(not data.get(attr) for attr in meta_attrs):
        # If any of the meta attrs is missing,
        # all the meta needs to be regenerated
        now = get_timestamp()

        for attr in timestamp_attrs:
            data[attr] = now

        data["_version"] = 1

    return cls(**data)


def from_json(cls, payload: str):
    """Returns a model instance with data from a json payload."""
    data = DataModelDeserializer.deserialize(payload)
    return cls.from_dict(data)


def to_dict(self, repr_only: Optional[bool] = False) -> str:
    """Returns a dictionary with model attributes and values."""
    data = asdict(self)

    ignored_field_names = [
        own_field.name
        for own_field in fields(self)
        if (
            own_field.metadata.get("mask") is True or (repr_only and not own_field.repr)
        )
    ]

    for key in ignored_field_names:
        del data[key]

    return data


def to_json(self, repr_only: Optional[bool] = False) -> str:
    """Returns a json representation of the model."""
    data = self.to_dict(repr_only=repr_only)
    data = demodel_data(self, data, repr_only=repr_only)

    return DataModelSerializer.serialize(data)


def raise_validation_error(self, msg: str) -> None:
    """Raises a validation error for the model."""
    data = self.to_json(repr_only=True)

    msg = f"{msg.strip('!. ')} - {data}"
    raise ValidationError(self, msg)


def update(self, **kwargs):
    """Returns an copy of the model with updated attributes."""
    data = {field.name: getattr(self, field.name) for field in fields(self)}

    data.update(kwargs)

    _version = self._version + 1
    data.update(dict(_updated=get_timestamp(), _version=_version))

    return type(self)(**data)


def supermodel(cls=None, **kwargs):
    """Provides a default decorator for dataclasses."""

    def set_fields(cls):
        field_names = [field.name for field in fields(cls)]

        attrs = []
        for attr in ("_created", "_updated"):
            if attr not in field_names:
                attrs.append((attr, datetime, timestamp_field()))

        if "_version" not in field_names:
            attrs.append(("_version", int, version_field()))

        attrs.extend([(field.name, field.type, field) for field in fields(cls)])

        return attrs

    def wrapper(cls):
        defaults = dict(frozen=True)
        kwargs.update(defaults)
        dc = dataclass(cls, **kwargs)

        bases = tuple(c for c in cls.__mro__ if is_dataclass(c) and c != cls)

        namespace = dict(
            __post_init__=__post_init__,
            update=update,
            from_dict=classmethod(from_dict),
            from_json=classmethod(from_json),
            to_dict=to_dict,
            to_json=to_json,
            raise_validation_error=raise_validation_error,
            _SUPERMODEL=True,
        )

        return make_dataclass(
            dc.__name__,
            bases=bases,
            fields=set_fields(cls),
            namespace=namespace,
            **kwargs,
        )

    if cls is None:
        return wrapper

    return wrapper(cls)
