"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.WebApp = void 0;
const path = require("path");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const origins = require("@aws-cdk/aws-cloudfront-origins");
const r53 = require("@aws-cdk/aws-route53");
const r53targets = require("@aws-cdk/aws-route53-targets");
const s3 = require("@aws-cdk/aws-s3");
const s3deploy = require("@aws-cdk/aws-s3-deployment");
const core_1 = require("@aws-cdk/core");
const constants_1 = require("../backend/shared/constants");
const caching_1 = require("../caching");
const monitored_certificate_1 = require("../monitored-certificate");
const storage_1 = require("../s3/storage");
const config_1 = require("./config");
const response_function_1 = require("./response-function");
class WebApp extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        const storageFactory = storage_1.S3StorageFactory.getOrCreate(this);
        this.bucket = storageFactory.newBucket(this, 'WebsiteBucket', {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            enforceSSL: true,
        });
        // generate a stable unique id for the cloudfront function and use it
        // both for the function name and the logical id of the function so if
        // it is changed the function will be recreated.
        // see https://github.com/aws/aws-cdk/issues/15523
        const functionId = `AddHeadersFunction${this.node.addr}`;
        const behaviorOptions = {
            compress: true,
            cachePolicy: cloudfront.CachePolicy.CACHING_OPTIMIZED,
            functionAssociations: [{
                    function: new response_function_1.ResponseFunction(this, functionId, {
                        functionName: functionId,
                    }),
                    eventType: cloudfront.FunctionEventType.VIEWER_RESPONSE,
                }],
        };
        this.distribution = new cloudfront.Distribution(this, 'Distribution', {
            defaultBehavior: { origin: new origins.S3Origin(this.bucket), ...behaviorOptions },
            domainNames: props.domain ? [props.domain.zone.zoneName] : undefined,
            certificate: props.domain ? props.domain.cert : undefined,
            defaultRootObject: 'index.html',
            errorResponses: [404, 403].map(httpStatus => ({
                httpStatus,
                responseHttpStatus: 200,
                responsePagePath: '/index.html',
            })),
            minimumProtocolVersion: cloudfront.SecurityPolicyProtocol.TLS_V1_2_2018,
        });
        // The base URL is currently the custom DNS if any was used, or the distribution domain name.
        // This needs changing in case, for example, we add support for a custom URL prefix.
        this.baseUrl = `https://${props.domain ? props.domain.zone.zoneName : this.distribution.distributionDomainName}`;
        const jsiiObjOrigin = new origins.S3Origin(props.packageData);
        this.distribution.addBehavior('/data/*', jsiiObjOrigin, behaviorOptions);
        this.distribution.addBehavior(`/${constants_1.CATALOG_KEY}`, jsiiObjOrigin, behaviorOptions);
        this.distribution.addBehavior(`/${constants_1.VERSION_TRACKER_KEY}`, jsiiObjOrigin, behaviorOptions);
        if (props.packageStats) {
            this.distribution.addBehavior(`/${props.packageStats.statsKey}`, jsiiObjOrigin, behaviorOptions);
        }
        // if we use a domain, and A records with a CloudFront alias
        if (props.domain) {
            // IPv4
            new r53.ARecord(this, 'ARecord', {
                zone: props.domain.zone,
                target: r53.RecordTarget.fromAlias(new r53targets.CloudFrontTarget(this.distribution)),
                comment: 'Created by the AWS CDK',
            });
            // IPv6
            new r53.AaaaRecord(this, 'AaaaRecord', {
                zone: props.domain.zone,
                target: r53.RecordTarget.fromAlias(new r53targets.CloudFrontTarget(this.distribution)),
                comment: 'Created by the AWS CDK',
            });
            // Monitor certificate expiration
            if ((_a = props.domain.monitorCertificateExpiration) !== null && _a !== void 0 ? _a : true) {
                const monitored = new monitored_certificate_1.MonitoredCertificate(this, 'ExpirationMonitor', {
                    certificate: props.domain.cert,
                    domainName: props.domain.zone.zoneName,
                });
                props.monitoring.addHighSeverityAlarm('ACM Certificate Expiry', monitored.alarmAcmCertificateExpiresSoon);
                props.monitoring.addHighSeverityAlarm('Endpoint Certificate Expiry', monitored.alarmEndpointCertificateExpiresSoon);
            }
        }
        // "website" contains the static react app
        const webappDir = path.join(__dirname, '..', '..', 'website');
        new s3deploy.BucketDeployment(this, 'DeployWebsite', {
            destinationBucket: this.bucket,
            distribution: this.distribution,
            prune: false,
            sources: [s3deploy.Source.asset(webappDir)],
            cacheControl: caching_1.CacheStrategy.default().toArray(),
        });
        // Generate config.json to customize frontend behavior
        const config = new config_1.WebappConfig({
            packageLinks: props.packageLinks,
            packageTags: props.packageTags,
            featuredPackages: props.featuredPackages,
            showPackageStats: (_b = props.showPackageStats) !== null && _b !== void 0 ? _b : props.packageStats !== undefined,
            featureFlags: props.featureFlags,
            categories: props.categories,
        });
        new s3deploy.BucketDeployment(this, 'DeployWebsiteConfig', {
            sources: [s3deploy.Source.asset(config.file.dir)],
            destinationBucket: this.bucket,
            distribution: this.distribution,
            prune: false,
            cacheControl: caching_1.CacheStrategy.default().toArray(),
        });
        new core_1.CfnOutput(this, 'DomainName', {
            value: this.distribution.domainName,
            exportName: 'ConstructHubDomainName',
        });
        // add a canary that pings our home page and alarms if it returns errors.
        props.monitoring.addWebCanary('Home Page', `https://${this.distribution.domainName}`);
    }
}
exports.WebApp = WebApp;
//# sourceMappingURL=data:application/json;base64,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