"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const console = require("console");
const zlib_1 = require("zlib");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const aws_xray_sdk_core_1 = require("aws-xray-sdk-core");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const client_lambda_shared_1 = require("../../backend/deny-list/client.lambda-shared");
const client_lambda_shared_2 = require("../../backend/license-list/client.lambda-shared");
const aws = require("../../backend/shared/aws.lambda-shared");
const env_lambda_shared_1 = require("../../backend/shared/env.lambda-shared");
const constants_lambda_shared_1 = require("./constants.lambda-shared");
const couch_changes_lambda_shared_1 = require("./couch-changes.lambda-shared");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const normalizeNPMMetadata = require('normalize-registry-metadata');
const CONSTRUCT_KEYWORDS = new Set(['cdk', 'aws-cdk', 'awscdk', 'cdk8s', 'cdktf']);
const NPM_REPLICA_REGISTRY_URL = 'https://replicate.npmjs.com/';
/**
 * The release date of `aws-cdk@0.8.0`. Anything earlier than this basically is
 * not a relevant package, as it cannot possibly be a constructs-based package.
 * This is used to fast-forward over boring stuff when the sequence number is
 * reset.
 */
const DAWN_OF_CONSTRUCTS = new Date('2018-07-31T13:43:04.615Z');
// Configure embedded metrics format
aws_embedded_metrics_1.Configuration.namespace = constants_lambda_shared_1.METRICS_NAMESPACE;
// Make sure X-Ray traces will include HTTP(s) calls.
// eslint-disable-next-line @typescript-eslint/no-require-imports
aws_xray_sdk_core_1.captureHTTPsGlobal(require('https'));
// eslint-disable-next-line @typescript-eslint/no-require-imports
aws_xray_sdk_core_1.captureHTTPsGlobal(require('http'));
/**
 * This function triggers on a fixed schedule and reads a stream of changes from npmjs couchdb _changes endpoint.
 * Upon invocation the function starts reading from a sequence stored in an s3 object - the `marker`.
 * If the marker fails to load (or do not exist), the stream will start from `now` - the latest change.
 * For each change:
 *  - the package version tarball will be copied from the npm registry to a stating bucket.
 *  - a message will be sent to an sqs queue
 * npm registry API docs: https://github.com/npm/registry/blob/master/docs/REGISTRY-API.md
 * @param context a Lambda execution context
 */
async function handler(event, context) {
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    const stagingBucket = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const stagingFunction = env_lambda_shared_1.requireEnv('FUNCTION_NAME');
    const denyList = await client_lambda_shared_1.DenyListClient.newClient();
    const licenseList = await client_lambda_shared_2.LicenseListClient.newClient();
    const npm = new couch_changes_lambda_shared_1.CouchChanges(NPM_REPLICA_REGISTRY_URL, 'registry');
    const { marker: initialMarker, knownVersions } = await loadLastTransactionMarker(stagingBucket, npm);
    // The last written marker seq id.
    let updatedMarker = initialMarker;
    // The slowest batch processing time so far (starts at 30 seconds). This is how much time should
    // be left before timeout if a new batch is to be fetched.
    let maxBatchProcessingTime = 30000;
    // Whether we should continue reading more items or not... This is set to false when the current
    // latest change is reached (i.e: next page of changes is empty).
    let shouldContinue = true;
    do {
        await aws_embedded_metrics_1.metricScope((metrics) => async () => {
            var _a;
            const changes = await npm.changes(updatedMarker);
            // Clear automatically set dimensions - we don't need them (see https://github.com/awslabs/aws-embedded-metrics-node/issues/73)
            metrics.setDimensions();
            // Recording current seq range and updating the `updatedMarker`.
            metrics.setProperty('StartSeq', updatedMarker);
            updatedMarker = changes.last_seq;
            metrics.setProperty('EndSeq', updatedMarker);
            const startTime = Date.now();
            try {
                const batch = changes.results;
                // The most recent "modified" timestamp observed in the batch.
                let lastModified;
                // Emit npm.js replication lag
                for (const { doc } of batch) {
                    if ((_a = doc === null || doc === void 0 ? void 0 : doc.time) === null || _a === void 0 ? void 0 : _a.modified) {
                        const modified = new Date(doc.time.modified);
                        metrics.putMetric("NpmJsChangeAge" /* NPMJS_CHANGE_AGE */, startTime - modified.getTime(), aws_embedded_metrics_1.Unit.Milliseconds);
                        if (lastModified == null || lastModified < modified) {
                            lastModified = modified;
                        }
                    }
                }
                console.log(`Received a batch of ${batch.length} element(s)`);
                metrics.putMetric("ChangeCount" /* CHANGE_COUNT */, batch.length, aws_embedded_metrics_1.Unit.Count);
                if (lastModified && lastModified < DAWN_OF_CONSTRUCTS) {
                    console.log(`Skipping batch as the latest modification is ${lastModified}, which is pre-Constructs`);
                }
                else if (batch.length === 0) {
                    console.log('Received 0 changes, caught up to "now", exiting...');
                    shouldContinue = false;
                }
                else {
                    // Obtain the modified package version from the update event, and filter
                    // out packages that are not of interest to us (not construct libraries).
                    const versionInfos = getRelevantVersionInfos(batch, metrics, denyList, licenseList, knownVersions);
                    console.log(`Identified ${versionInfos.length} relevant package version update(s)`);
                    metrics.putMetric("RelevantPackageVersions" /* RELEVANT_PACKAGE_VERSIONS */, versionInfos.length, aws_embedded_metrics_1.Unit.Count);
                    // Process all remaining updates
                    await Promise.all(versionInfos.map(async ({ infos, modified, seq }) => {
                        const invokeArgs = {
                            integrity: infos.dist.shasum,
                            modified: modified.toISOString(),
                            name: infos.name,
                            seq: seq === null || seq === void 0 ? void 0 : seq.toString(),
                            tarballUrl: infos.dist.tarball,
                            version: infos.version,
                        };
                        // "Fire-and-forget" invocation here.
                        await aws.lambda().invokeAsync({
                            FunctionName: stagingFunction,
                            InvokeArgs: JSON.stringify(invokeArgs, null, 2),
                        }).promise();
                        // Record that this is now a "known" version (no need to re-discover)
                        knownVersions.set(`${infos.name}@${infos.version}`, modified);
                    }));
                }
                // Updating the S3 stored marker with the new seq id as communicated by nano.
                await saveLastTransactionMarker(context, stagingBucket, updatedMarker, knownVersions);
            }
            finally {
                // Markers may not always be numeric (but in practice they are now), so we protect against that...
                if (typeof updatedMarker === 'number' || /^\d+$/.test(updatedMarker)) {
                    metrics.putMetric("LastSeq" /* LAST_SEQ */, typeof updatedMarker === 'number' ? updatedMarker : parseInt(updatedMarker), aws_embedded_metrics_1.Unit.None);
                }
                metrics.putMetric("BatchProcessingTime" /* BATCH_PROCESSING_TIME */, Date.now() - startTime, aws_embedded_metrics_1.Unit.Milliseconds);
                metrics.putMetric("RemainingTime" /* REMAINING_TIME */, context.getRemainingTimeInMillis(), aws_embedded_metrics_1.Unit.Milliseconds);
            }
        })();
    } while (shouldContinue && context.getRemainingTimeInMillis() >= maxBatchProcessingTime);
    console.log('All done here, we have success!');
    return { initialMarker, updatedMarker };
}
exports.handler = handler;
//#region Last transaction marker
/**
 * Loads the last transaction marker from S3.
 *
 * @param registry a Nano database corresponding to the Npmjs.com CouchDB instance.
 *
 * @returns the value of the last transaction marker and the map of package names + versions to the last modification
 *          of that package version that was processed.
 */
async function loadLastTransactionMarker(stagingBucket, registry) {
    try {
        const response = await aws.s3().getObject({
            Bucket: stagingBucket,
            Key: constants_lambda_shared_1.MARKER_FILE_NAME,
        }).promise();
        if (response.ContentEncoding === 'gzip') {
            response.Body = zlib_1.gunzipSync(Buffer.from(response.Body));
        }
        let data = JSON.parse(response.Body.toString('utf-8'), (key, value) => {
            if (key !== 'knownVersions') {
                return value;
            }
            const map = new Map();
            for (const [pkgVersion, iso] of Object.entries(value)) {
                if (typeof iso === 'string' || typeof iso === 'number') {
                    map.set(pkgVersion, new Date(iso));
                }
                else {
                    console.error(`Ignoring invalid entry: ${pkgVersion} => ${iso}`);
                }
            }
            return map;
        });
        if (typeof data === 'number') {
            data = { marker: data.toFixed(), knownVersions: new Map() };
        }
        console.log(`Read last transaction marker: ${data.marker}`);
        const dbUpdateSeq = (await registry.info()).update_seq;
        if (dbUpdateSeq < data.marker) {
            console.warn(`Current DB update_seq (${dbUpdateSeq}) is lower than marker (CouchDB instance was likely replaced), resetting to 0!`);
            return { marker: '0', knownVersions: data.knownVersion };
        }
        return data;
    }
    catch (error) {
        if (error.code !== 'NoSuchKey') {
            throw error;
        }
        console.warn(`Marker object (s3://${stagingBucket}/${constants_lambda_shared_1.MARKER_FILE_NAME}) does not exist, starting from scratch`);
        return { marker: '0', knownVersions: new Map() };
    }
}
/**
 * Updates the last transaction marker in S3.
 *
 * @param marker the last transaction marker value
 * @param knownVersions the map of package name + version to last modified timestamp of packages that have been processed.
 */
async function saveLastTransactionMarker(context, stagingBucket, marker, knownVersions) {
    console.log(`Updating last transaction marker to ${marker}`);
    return putObject(context, stagingBucket, constants_lambda_shared_1.MARKER_FILE_NAME, zlib_1.gzipSync(JSON.stringify({ marker, knownVersions }, (_, value) => {
        if (value instanceof Date) {
            return value.toISOString();
        }
        else if (value instanceof Map) {
            return Object.fromEntries(value);
        }
        else {
            return value;
        }
    }, 2), { level: 9 }), {
        ContentType: 'application/json',
        ContentEncoding: 'gzip',
    });
}
//#endregion
//#region Asynchronous Primitives
/**
 * Puts an object in the staging bucket, with standardized object metadata.
 *
 * @param key  the key for the object to be put.
 * @param body the body of the object to be put.
 * @param opts any other options to use when sending the S3 request.
 *
 * @returns the result of the S3 request.
 */
function putObject(context, bucket, key, body, opts = {}) {
    return aws.s3().putObject({
        Bucket: bucket,
        Key: key,
        Body: body,
        Metadata: {
            'Lambda-Log-Group': context.logGroupName,
            'Lambda-Log-Stream': context.logStreamName,
            'Lambda-Run-Id': context.awsRequestId,
            ...opts.Metadata,
        },
        ...opts,
    }).promise();
}
//#endregion
/**
 * Obtains the `VersionInfo` corresponding to the modified version(s) in the
 * provided `Change` objects, ensures they are relevant (construct libraries),
 * and returns those only.
 *
 * @param changes the changes to be processed.
 * @param metrics the metrics logger to use.
 * @param denyList deny list client
 *
 * @returns a list of `VersionInfo` objects
 */
function getRelevantVersionInfos(changes, metrics, denyList, licenseList, knownVersions) {
    var _a, _b;
    const result = new Array();
    for (const change of changes) {
        // Filter out all elements that don't have a "name" in the document, as
        // these are schemas, which are not relevant to our business here.
        if (change.doc.name === undefined) {
            console.error(`[${change.seq}] Changed document contains no 'name': ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // The normalize function change the object in place, if the doc object is invalid it will return undefined
        if (normalizeNPMMetadata(change.doc) === undefined) {
            console.error(`[${change.seq}] Changed document invalid, npm normalize returned undefined: ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // Sometimes, there are no versions in the document. We skip those.
        if (change.doc.versions == null) {
            console.error(`[${change.seq}] Changed document contains no 'versions': ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // Sometimes, there is no 'time' entry in the document. We skip those.
        if (change.doc.time == null) {
            console.error(`[${change.seq}] Changed document contains no 'time': ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // Get the last modification date from the change
        const packageVersionUpdates = Object.entries(change.doc.time)
            // Ignore the "created" and "modified" keys here
            .filter(([key]) => key !== 'created' && key !== 'modified')
            // Parse all the dates to ensure they are comparable
            .map(([version, isoDate]) => [version, new Date(isoDate)]);
        metrics.putMetric("PackageVersionCount" /* PACKAGE_VERSION_COUNT */, packageVersionUpdates.length, aws_embedded_metrics_1.Unit.Count);
        for (const [version, modified] of packageVersionUpdates) {
            const knownKey = `${change.doc.name}@${version}`;
            const known = knownVersions.get(knownKey);
            if (known == null || known < modified) {
                const infos = change.doc.versions[version];
                if (infos == null) {
                    // Could be the version in question was un-published.
                    console.log(`[${change.seq}] Could not find info for "${change.doc.name}@${version}". Was it un-published?`);
                }
                else if (isConstructLibrary(infos)) {
                    // skip if this package is denied
                    const denied = denyList.lookup(infos.name, infos.version);
                    if (denied) {
                        console.log(`[${change.seq}] Package denied: ${JSON.stringify(denied)}`);
                        knownVersions.set(knownKey, modified);
                        metrics.putMetric("DenyListedCount" /* DENY_LISTED_COUNT */, 1, aws_embedded_metrics_1.Unit.Count);
                        continue;
                    }
                    metrics.putMetric("PackageVersionAge" /* PACKAGE_VERSION_AGE */, Date.now() - modified.getTime(), aws_embedded_metrics_1.Unit.Milliseconds);
                    const isEligible = licenseList.lookup((_a = infos.license) !== null && _a !== void 0 ? _a : 'UNLICENSED') != null;
                    metrics.putMetric("IneligibleLicense" /* INELIGIBLE_LICENSE */, isEligible ? 0 : 1, aws_embedded_metrics_1.Unit.Count);
                    if (isEligible) {
                        result.push({ infos, modified, seq: change.seq });
                    }
                    else {
                        console.log(`[${change.seq}] Package "${change.doc.name}@${version}" does not use allow-listed license: ${(_b = infos.license) !== null && _b !== void 0 ? _b : 'UNLICENSED'}`);
                        knownVersions.set(knownKey, modified);
                    }
                }
                // Else this is not a construct library, so we'll just ignore it...
            }
        }
    }
    return result;
    /**
     * This determines whether a package is "interesting" to ConstructHub or not. This is related but
     * not necessarily identical to the logic in the ingestion process that annotates package metadata
     * with a construct framework name + version (those could ultimately be re-factored to share more
     * of the logic/heuristics, though).
     *
     * Concretely, it checks for a list of known "official" packages for various construct frameworks,
     * and packages that have a dependency on such a package. It also has a keywords allow-list as a
     * fall-back (the current dependency-based logic does not consider transitive dependencies and
     * might hence miss certain rare use-cases, which keywords would rescue).
     */
    function isConstructLibrary(infos) {
        var _a, _b, _c, _d;
        if (infos.jsii == null) {
            return false;
        }
        // The "constructs" package is a sign of a constructs library
        return isConstructFrameworkPackage(infos.name)
            // Recursively apply on dependencies
            || Object.keys((_a = infos.dependencies) !== null && _a !== void 0 ? _a : {}).some(isConstructFrameworkPackage)
            || Object.keys((_b = infos.devDependencies) !== null && _b !== void 0 ? _b : {}).some(isConstructFrameworkPackage)
            || Object.keys((_c = infos.peerDependencies) !== null && _c !== void 0 ? _c : {}).some(isConstructFrameworkPackage)
            // Keyword-based fallback
            || ((_d = infos.keywords) === null || _d === void 0 ? void 0 : _d.some((kw) => CONSTRUCT_KEYWORDS.has(kw)));
    }
    /**
     * Package is one of the known construct framework's first party packages:
     * - @aws-cdk/*
     * - @cdktf/*
     * - cdk8s or cdk8s-plus
     */
    function isConstructFrameworkPackage(name) {
        // IMPORTANT NOTE: Prefix matching should only be used for @scope/ names.
        // The low-level constructs package
        return name === 'constructs'
            // AWS CDK Packages
            || name === 'aws-cdk-lib'
            || name === 'monocdk'
            || name.startsWith('@aws-cdk/')
            // CDK8s packages
            || name === 'cdk8s'
            || /^cdk8s-plus(?:-(?:17|20|21|22))?$/.test(name)
            // CDKTf packages
            || name === 'cdktf'
            || name.startsWith('@cdktf/');
    }
}
//# sourceMappingURL=data:application/json;base64,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